# -*- coding: utf-8 -*-
"""
/***************************************************************************
 UldkGugik
                                 A QGIS plugin
 Wtyczka pozwala na pobieranie geometrii granic działek katastralnych, obrębów, gmin, powiatów i województw.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-05-31
        git sha              : $Format:%H$
        copyright            : (C) 2019 by EnviroSolutions Sp. z o.o.
        email                : office@envirosolutions.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, qVersion, QCoreApplication, QVariant, Qt
from qgis.PyQt.QtGui import QIcon, QPixmap, QKeySequence
from qgis.PyQt.QtWidgets import QAction, QToolBar, QShortcut, QWidget, QLabel, QDialog
from qgis.gui import QgsMessageBar, QgsMapToolEmitPoint, QgsDockWidget
from qgis.core import (Qgis, QgsVectorLayer, QgsGeometry, QgsFeature, QgsProject, QgsField,
                       QgsCoordinateReferenceSystem, QgsPoint, QgsCoordinateTransform, QgsMessageLog,
                       QgsSettings)

import requests

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .uldk_gugik_dialog import UldkGugikDialog
from .uldk_gugik_dialog_parcel import UldkGugikDialogParcel
from .qgis_feed import QgisFeedDialog
import os.path
from . import utils, uldk_api, uldk_xy, uldk_parcel
from .constants import DEFAULT_SRID


"""Wersja wtyczki"""
plugin_version = '1.4.2'
plugin_name = 'ULDK GUGiK'  

class UldkGugik:
    """QGIS Plugin Implementation."""
    nazwy_warstw = {1: "dzialki_ew_uldk", 2: "obreby_ew_uldk", 3: "gminy_uldk", 4: "powiaty_uldk", 5: "wojewodztwa_uldk", 6: "budynki_uldk"}

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.settings = QgsSettings() 

        if Qgis.QGIS_VERSION_INT >= 31000:
            from .qgis_feed import QgisFeed

            #qgis feed
            self.selected_industry = self.settings.value("selected_industry", None)
            show_dialog = self.settings.value("showDialog", True, type=bool)
            
            if self.selected_industry is None and show_dialog:
                self.showBranchSelectionDialog()
        
            select_indust_session = self.settings.value('selected_industry')
            
            self.feed = QgisFeed(selected_industry=select_indust_session, plugin_name=plugin_name)
            self.feed.initFeed()

        #DialogOnTop

        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'UldkGugik_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&EnviroSolutions')

        #toolbar
        self.toolbar = self.iface.mainWindow().findChild(QToolBar, 'EnviroSolutions')
        if not self.toolbar:
            self.toolbar = self.iface.addToolBar(u'EnviroSolutions')
            self.toolbar.setObjectName(u'EnviroSolutions')

        self.shortcut = None
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        self.canvas = self.iface.mapCanvas()
        
        # out click tool will emit a QgsPoint on every click
        self.clickTool = QgsMapToolEmitPoint(self.canvas)
        self.clickTool.canvasClicked.connect(self.canvasClicked)

        self.dlg = UldkGugikDialog()
        self.region_name = None
        self.project = QgsProject.instance()
        

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('UldkGugik', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            #self.iface.addToolBarIcon(action)
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""


        icon_path = ':/plugins/uldk_gugik/images/uldk.svg'
        self.add_action(
            icon_path,
            text=self.tr(u'Usługa Lokalizacji Działek Katastralnych (ULDK)'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True
        #self.dock = UldkGugikDialog()
        #self.iface.addDockWidget(Qt.RigthDockWidgetArea, self.dock)


        # Inicjacja grafik
        self.dlg.img_tab2.setPixmap(QPixmap(':/plugins/uldk_gugik/images/coords.png'))

        # rozmiar okna
        self.dlg.setFixedSize(self.dlg.size())

        # informacje o wersji
        self.dlg.setWindowTitle('%s %s' % (plugin_name, plugin_version))
        self.dlg.lbl_pluginVersion.setText('%s %s' % (plugin_name, plugin_version))

        self.dlg.btn_download_tab1.clicked.connect(self.btn_download_tab1_clicked)
        self.dlg.btn_download_tab2.clicked.connect(self.btn_download_tab2_clicked)
        self.dlg.btn_download_tab3.clicked.connect(self.btn_download_tab3_clicked)
        self.dlg.btn_search_tab3.clicked.connect(self.btn_search_tab3_clicked)
        
        self.dlg.btn_frommap.clicked.connect(self.btn_frommap_clicked)
        self.dlg.btn_frommap.setToolTip("skrót: ALT + F")


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&EnviroSolutions'),
                action)
            # self.iface.removeToolBarIcon(action)
            self.toolbar.removeAction(action)

    def run(self):
        """Otwarcie okna wtyczki"""
        try:
            srid = self.project.crs().postgisSrid()
        except IndexError:
            self.iface.messageBar().pushMessage("Projekt QGIS nie posiada zdefiniowanego układu współrzędnych.", 
                                                "W celu dalszej pracy zdefiniuj układ współrzędnych dla projektu",
                                                level=Qgis.Warning, duration=10)
            return 1
        if self.first_start == True:
            self.first_start = False
            if Qgis.QGIS_VERSION_INT >= 30000:
                # skrot klawiszowy
                self.shortcut = QShortcut(self.iface.mainWindow())
                self.shortcut.setKey(QKeySequence(Qt.ALT + Qt.Key_F))
                self.shortcut.activated.connect(self.shortcut_activated)
            else:
                self.shortcut = QShortcut(QKeySequence(Qt.ALT + Qt.Key_F), self.iface.mainWindow())
                self.shortcut.setContext(Qt.ApplicationShortcut)
                self.shortcut.activated.connect(self.shortcut_activated)
        
        try:
            odpowiedz = requests.get('https://uldk.gugik.gov.pl', verify=False)

            if odpowiedz.status_code == 200:
                self.setup_dialog()
                # self.disable_button_download()
            else:
                self.iface.messageBar().pushMessage("Ostrzeżenie:", 
                                                'Serwer ULDK nie odpowiada. Spróbuj ponownie później',
                                                level=Qgis.Warning, duration=10)

        except requests.exceptions.ConnectionError:
            self.iface.messageBar().pushMessage("Ostrzeżenie:", 
                                                'Brak połączenia z internetem',
                                                level=Qgis.Warning, duration=10)
        self.dlg.projectionWidget.setCrs(
            QgsCoordinateReferenceSystem(srid, QgsCoordinateReferenceSystem.EpsgCrsId))


    def showBranchSelectionDialog(self):
        self.qgisfeed_dialog = QgisFeedDialog()

        if self.qgisfeed_dialog.exec_() == QDialog.Accepted:
            self.selected_branch = self.qgisfeed_dialog.comboBox.currentText()
            
            #Zapis w QGIS3.ini
            self.settings.setValue("selected_industry", self.selected_branch)  
            self.settings.setValue("showDialog", False) 

    def setup_dialog(self):
        if self.dlg.regionFetch is None:
            self.dlg = UldkGugikDialog()
        self.dlg.show()

    def btn_download_tab1_clicked(self):
        """kliknięcie klawisza pobierania po numerze TERYT w oknie wtyczki"""

        teryt = self.dlg.edit_id.text().strip()

        if not teryt:
            self.iface.messageBar().pushMessage("Błąd formularza:",
                                                'musisz wpisać identyfikator',
                                                level=Qgis.Warning, duration=10)
        elif utils.isInternetConnected():
            try:
                self.performRequestTeryt(teryt=teryt)
            except:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    'API nie zwróciło odpowiedzi dla żądanego zapytania',
                    level=Qgis.Critical,
                    duration=10
                )


        else:
            self.iface.messageBar().pushMessage("Nie udało się pobrać obiektu:",
                                                'brak połączenia z internetem',
                                                level=Qgis.Critical, duration=10)

    def btn_download_tab2_clicked(self):
        """kliknięcie klawisza pobierania według X i Y wpisanych w oknie wtyczki"""
        srid = self.dlg.projectionWidget.crs().authid().split(":")[1]
        self.downloadByXY(srid, type="form", zoomToFeature=False)

    def btn_search_tab3_clicked(self):
        arkusze_numery = set()
        arkusze_numery_posortowane = set()
        if str(self.dlg.obrcomboBox.currentText().strip()):
            objRegion = str(self.dlg.obrcomboBox.currentText().strip())
            objectType = self.checkedFeatureType()
        else:
            objRegion = str(self.dlg.gmicomboBox.currentText().strip())
            objectType = self.checkedFeatureType()

        if objectType != 1:
            return
        self.dlg.parcel_lineedit.setReadOnly(True)
        current_idx = self.dlg.gmicomboBox.currentIndex()
        teryt = self.dlg.gmicomboBox.itemData(current_idx)
        objParcel = self.dlg.parcel_lineedit.text().strip() # nr działki

        if not objRegion:
            self.iface.messageBar().pushMessage("Błąd formularza:",
                                                'musisz wpisać obręb',
                                                level=Qgis.Warning, duration=10)

        if not objParcel:
            self.iface.messageBar().pushMessage("Błąd formularza:",
                                                'musisz wpisać numer działki',
                                                level=Qgis.Warning, duration=10)

        elif utils.isInternetConnected():
            self.dlg.arkcomboBox.clear()

            obr_idx = self.dlg.obrcomboBox.currentIndex()
            teryt = self.dlg.obrcomboBox.itemData(obr_idx)

            result_obreb = uldk_parcel.GetRegionById(id=teryt)
            result_obreb = list(result_obreb)

            #sprawdzanie obrebow po usunieciu niepotrzebnych numerow
            for obreb in result_obreb:
                if len(obreb) < 3:
                    result_obreb.remove(obreb)
                else:
                    pass

            #sprawdzanie czy obręb leży w tym województwie, w którym powinien
            for obreb in result_obreb:
                if obreb.split("|")[-1].rstrip() == self.dlg.wojcomboBox.currentText():
                    pass
                else:
                    result_obreb.remove(obreb)

            #sprawdzanie czy obręb leży w tym powiecie, w którym powinien
            for obreb in result_obreb:
                if obreb.split("|")[-2] == self.dlg.powcomboBox.currentText():
                    pass
                else:
                    result_obreb.remove(obreb)

            #sprawdzanie czy obręb leży w tej gminie, w której powinien
            for obreb in result_obreb:
                if obreb.split("|")[-3] == self.dlg.gmicomboBox.currentText():
                    pass
                else:
                    result_obreb.remove(obreb)

            try:
                self.region_name = result_obreb[0].split("|")[0]
                name = self.region_name + '.' + objParcel

                result = uldk_parcel.getParcelById2(name)
                result = list(result)

                for rezultat in result:
                    if rezultat.find("-1 brak wyników") >= 1 or rezultat.find("usługa nie zwróciła odpowiedzi") >= 1 or rezultat.find("błędny format odpowiedzi XML, usługa zwróciła odpowiedź") >= 1 or rezultat.find("XML") >= 1 or rezultat.find("błędny format") >= 1:
                        response = False
                    else:
                        response = True

                if response == True:
                    for i in range(len(result)):
                        if len(result[i]) < 3:
                            pass
                        elif result[i].find(";") > -1:
                            if result[i].split(";")[1].split("|")[1].split(".")[-2].find("AR") > -1:
                                arkusze_numery.add(result[i].split(";")[1].split("|")[1].split(".")[-2].strip())
                            else:
                                pass
                        else:
                            if result[i].split("|")[1].split(".")[-2].find("AR") > -1:
                                arkusze_numery.add(result[i].split("|")[1].split(".")[-2].strip())
                            else:
                                pass

                    arkusze_numery = list(arkusze_numery)
                    if len(arkusze_numery) >= 1:
                        for arkusz in arkusze_numery:
                            arkusze_numery_posortowane.add(arkusz.split("AR_")[-1])
                        for sort_ark in sorted(arkusze_numery_posortowane,key=int):
                            self.dlg.arkcomboBox.addItem(f"AR_{sort_ark}")
                    else:
                        pass
                    self.successDownload(arkusze_numery)
                else:
                    self.iface.messageBar().pushMessage("Ostrzeżenie:",
                                                        'Nie zwrócono żadnej działki dla podanych parametrów',
                                                             level=Qgis.Warning, duration=10)
            except IndexError:
                self.iface.messageBar().pushMessage("Ostrzeżenie:",
                                                    'Nie zwrócono żadnej działki dla podanych parametrów',
                                                    level=Qgis.Warning, duration=10)
        self.dlg.parcel_lineedit.setReadOnly(False)

    def successDownload(self, arkusze_numery):
        if len(arkusze_numery) >= 1:
            self.iface.messageBar().pushMessage(
                'Informacja:',
                'Znaleziono działkę/i dla podanych parametrów, wybierz numer arkusza.',
                level=Qgis.Info,
                duration=10
            )
        else:
            self.iface.messageBar().pushMessage(
                'Informacja:',
                'Znaleziono działkę dla podanych parametrów. Aby pobrać działkę, kliknij przycisk Pobierz.',
                level=Qgis.Info,
                duration=10
            )
        self.dlg.btn_download_tab3.setEnabled(True)

    def btn_download_tab3_clicked(self):
        if self.region_name:
            objRegion = self.region_name
        else:
            objRegion = str(self.dlg.gmicomboBox.currentText().strip())

        objectType = self.checkedFeatureType()
        if objectType == 1:
            current_idx = self.dlg.gmicomboBox.currentIndex()
            teryt = self.dlg.gmicomboBox.itemData(current_idx)
            objParcel = self.dlg.parcel_lineedit.text().strip() # nr działki
            if not objRegion:
                self.iface.messageBar().pushMessage("Błąd formularza:",
                                                    'musisz wpisać obręb',
                                                    level=Qgis.Warning, duration=10)

            if not objParcel:
                self.iface.messageBar().pushMessage("Błąd formularza:",
                                                    'musisz wpisać numer działki',
                                                    level=Qgis.Warning, duration=10)

            elif utils.isInternetConnected():
                self.performRequestParcel(region=objRegion, parcel=objParcel, teryt=None, zoomToFeature=True)

            else:
                self.iface.messageBar().pushMessage("Nie udało się pobrać obiektu:",
                                                    'brak połączenia z internetem',
                                                    level=Qgis.Critical, duration=10)

            return
        elif objectType == 2:  # obręb
            obr_idx = self.dlg.obrcomboBox.currentIndex()
            teryt = self.dlg.obrcomboBox.itemData(obr_idx)

            resp = uldk_api.getRegionById(teryt, objectType=2, obreb=True)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obrębu:",
                    f'API nie zwróciło obrębu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obrębu:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]

        elif objectType == 3:  # gmina
            current_idx = self.dlg.gmicomboBox.currentIndex()
            teryt = self.dlg.gmicomboBox.itemData(current_idx)
            resp = uldk_api.getCommuneById(teryt, objectType=3)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać gminy:",
                    f'API nie zwróciło obiektu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać gminy:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = res[2]
            county = res[3]
            voivodeship = res[4]

        elif objectType == 4:
            current_idx = self.dlg.powcomboBox.currentIndex()
            teryt = self.dlg.powcomboBox.itemData(current_idx)
            resp = uldk_api.getCountyById(teryt, objectType=4)

            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać powiatu:",
                    f'API nie zwróciło obiektu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać powiatu:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = res[2]
            voivodeship = res[3]

        elif objectType == 5: # województwo
            current_idx = self.dlg.wojcomboBox.currentIndex()
            teryt = self.dlg.wojcomboBox.itemData(current_idx)
            resp = uldk_api.getVoivodeshipById(teryt, objectType=5)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać województwa:",
                    f'API nie zwróciło obiektu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać województwa:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = None
            voivodeship = res[2]

        self.addResultsToLayer(
            objectType=objectType,
            wkt=wkt,
            teryt=teryt,
            parcel=parcel,
            region=region,
            commune=commune,
            county=county,
            voivodeship=voivodeship,
            zoomToFeature=True)

        object ={
            1: "działkę o nr teryt: %s",
            2: "obręb ewidencyjny",
            3: "gminę",
            4: "powiat",
            5: "województwo",
        }

        success_message = f"Pobrano {object[objectType]}"  % teryt if objectType == 1 else f"Pobrano {object[objectType]}"

        self.iface.messageBar().pushMessage("Sukces:",
                                            success_message,
                                            level=Qgis.Success, duration=10)

    def downloadByXY(self, srid, type, zoomToFeature=False):
        """pobranie według X i Y i SRID"""

        objX = self.dlg.doubleSpinBoxX.text().strip()
        objY = self.dlg.doubleSpinBoxY.text().strip()

        if type == "form" and srid in ['2180', '4326', '3857', '2176', '2177', '2178', '2179']:
            objX = self.dlg.doubleSpinBoxY.text().strip()
            objY = self.dlg.doubleSpinBoxX.text().strip()

        if not objX:
            self.iface.messageBar().pushMessage("Błąd formularza:",
                                                'musisz wpisać współrzędną X',
                                                level=Qgis.Warning, duration=10)

        if not objY:
            self.iface.messageBar().pushMessage("Błąd formularza:",
                                                'musisz wpisać współrzędną Y',
                                                level=Qgis.Warning, duration=10)

        elif utils.isInternetConnected():
            try:
                self.performRequestXY(x=objX, y=objY, srid=srid, zoomToFeature=zoomToFeature)
                # self.dlg.hide()  # jeżeli wtyczka ma zostawiać włączone okno, zamiast hide wpisz show
            except:
                self.iface.messageBar().pushMessage("Nie udało się pobrać obiektu:",
                                                    'API nie zwróciło obiektu dla wybranego zapytania',
                                                    level=Qgis.Critical, duration=10)


        else:
            self.iface.messageBar().pushMessage("Nie udało się pobrać obiektu:",
                                                'brak połączenia z internetem',
                                                level=Qgis.Critical, duration=10)

    def shortcut_activated(self):
        """zdarzenie aktywowania klawisza skrótu wskazania działki na mapie"""
        self.canvas.setMapTool(self.clickTool)

    def btn_frommap_clicked(self):
        """zdarzenie wciśnięcia w oknie wtyczki klawisza wskazania działki na mapie"""
        self.canvas.setMapTool(self.clickTool)
        self.dlg.hide()

    def canvasClicked(self, point):
        """kliknięcie na mapie"""
        self.canvas.unsetMapTool(self.clickTool)
        self.dlg.doubleSpinBoxX.setValue(point.x())
        self.dlg.doubleSpinBoxY.setValue(point.y())
        coords = "{}, {}".format(point.x(), point.y())
        QgsMessageLog.logMessage(str(coords), 'ULDK')
        srid = self.project.crs().postgisSrid()
        self.downloadByXY(srid, type="click", zoomToFeature=False)

    def performRequestParcel(self, region, parcel, teryt, zoomToFeature=True):
        objectType = self.checkedFeatureType()
        try:
            if self.dlg.arkcomboBox.currentText() != '':
                name = region + '.' + self.dlg.arkcomboBox.currentText() + '.' + parcel
            else:
                name = region + '.' + parcel
            result = uldk_parcel.getParcelById(name, objectType=1)

            if result is None:
                self.iface.messageBar().pushMessage("Nie udało się pobrać obiektu:",
                                                    'API nie zwróciło obiektu dla id %s' % name,
                                                    level=Qgis.Critical, duration=10)
                return

            res = result.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage("Nie udało się pobrać obiektu:",
                                                    'API nie zwróciło geometrii dla id %s' % name,
                                                    level=Qgis.Critical, duration=10)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = res[2]
            region = res[3]
            commune = res[4]
            county = res[5]
            voivodeship = res[6]

            # layer
            nazwa = self.nazwy_warstw[objectType]

            layers = self.project.mapLayersByName(nazwa)

            if layers:
                # jezeli istnieje to dodaj obiekt do warstwy
                layer = layers[0]
            else:
                # jezeli nie istnieje to stworz warstwe
                epsg = f"Polygon?crs=EPSG:{DEFAULT_SRID}"
                layer = QgsVectorLayer(epsg, nazwa, "memory")
                self.project.addMapLayer(layer)
            
            provider = layer.dataProvider()
            layer.updateExtents()

            if not layers:
                voivField = QgsField('województwo', QVariant.String, len=30)
                provider.addAttributes([voivField])

                conField = QgsField('powiat', QVariant.String, len=30)
                provider.addAttributes([conField])

                comField = QgsField('gmina', QVariant.String, len=30)
                provider.addAttributes([comField])

                regField = QgsField('obręb', QVariant.String, len=30)
                provider.addAttributes([regField])

                parField = QgsField('numer', QVariant.String, len=30)
                provider.addAttributes([parField])
                
                idField = QgsField('teryt', QVariant.String, len=40)
                provider.addAttributes([idField])

                layer.updateFields()

            feat = QgsFeature(provider.fields())
            feat.setGeometry(QgsGeometry().fromWkt(wkt))

            fields_mapping = {
                'teryt': teryt,
                'numer': parcel,
                'obręb': region,
                'gmina': commune,
                'powiat': county,
            }

            feat.setAttribute('województwo', voivodeship)
            for field_name, attr in fields_mapping.items():
                if field_name in feat.fields().names():
                    feat.setAttribute(field_name, attr or None)

            provider.addFeature(feat)

            self.iface.messageBar().pushMessage("Sukces:",
                                                'Pobrano działkę dla obiektu: %s' % (name),
                                                level=Qgis.Success, duration=10)

            if zoomToFeature:
                project_crs = self.project.crs().postgisSrid()
                if project_crs != DEFAULT_SRID:
                    source_crs = QgsCoordinateReferenceSystem.fromEpsgId(DEFAULT_SRID)
                    dest_crs = QgsCoordinateReferenceSystem.fromEpsgId(project_crs)
                    tr = QgsCoordinateTransform(source_crs, dest_crs, self.project)
                    box = tr.transform(feat.geometry().boundingBox())
                else:
                    box = feat.geometry().boundingBox()

                self.canvas.setExtent(box)
                self.canvas.refresh()
            else:
                layer.triggerRepaint()

        except IndexError:
            self.iface.messageBar().pushMessage(
                'Ostrzeżenie:',
                'Nie pobrano żadnej działki dla podanych parametrów',
                level=Qgis.Warning,
                duration=10
            )

    def performRequestTeryt(self, teryt, zoomToFeature=True):
        """wykonanie zapytania pobierającego obiekt na podstawie kodu TERYT"""
        object_type = self.checkedFeatureType()

        if object_type == 1:
            resp = uldk_api.getParcelById(teryt, objectType=1)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            if teryt != res[1] and "AR" in res[1]:
                self.iface.messageBar().pushMessage("Informacja:",
                                                    'W wybranym obrębie znaleziono więcej działek o identyfikatorze TERYT: %s. \r\nDodaj numer arkusza w celu odnalezienia właściwej działki' % teryt,
                                                    level=Qgis.Info, duration=10)
            wkt = res[0]
            teryt = res[1]
            parcel = res[2]
            region = res[3]
            commune = res[4]
            county = res[5]
            voivodeship = res[6]


        elif object_type == 2:
            resp = uldk_api.getRegionById(teryt, objectType=2)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]


        elif object_type == 3:
            resp = uldk_api.getCommuneById(teryt, objectType=3)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = res[2]
            county = res[3]
            voivodeship = res[4]


        elif object_type == 4:
            resp = uldk_api.getCountyById(teryt, objectType=4)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = res[2]
            voivodeship = res[3]

        elif object_type == 5:
            resp = uldk_api.getVoivodeshipById(teryt, objectType=5)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = None
            voivodeship = res[2]

        elif object_type == 6:
            resp = uldk_api.getBuildingById(teryt, object_type=6)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            if res[0] == '':
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło geometrii dla id {teryt}',
                    level=Qgis.Critical,
                    duration=10
                )
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]

        self.addResultsToLayer(
            objectType=object_type,
            wkt=wkt,
            teryt=teryt,
            parcel=parcel,
            region=region,
            commune=commune,
            county=county,
            voivodeship=voivodeship,
            zoomToFeature=zoomToFeature)

        object ={
            1: "działkę o nr teryt: %s",
            2: "obręb ewidencyjny",
            3: "gminę",
            4: "powiat",
            5: "województwo",
            6: "budynek",
        }

        success_message = f"Pobrano {object[object_type]}"  % teryt if object_type == 1 else f"Pobrano {object[object_type]}"

        self.iface.messageBar().pushMessage("Sukces:",
                                            success_message,
                                            level=Qgis.Success, duration=10)

    def performRequestXY(self, x, y, srid, zoomToFeature):
        """wykonanie zapytania pobierającego obiekt na podstawie współrzędnych"""

        objectType = self.checkedFeatureType()

        #Sprawdzenie, czy współrzędne są z przecinkiem czy kropką
        if "," in x or "," in y:
            x = float(x.replace(",", "."))
            y = float(y.replace(",", "."))
        else:
            x = float(x)
            y = float(y)

        requestPoint = QgsPoint(x, y)
        QgsMessageLog.logMessage(str(srid), 'ULDK')
        if str(srid) != DEFAULT_SRID:
            sourceCrs = QgsCoordinateReferenceSystem.fromEpsgId(int(srid))
            destCrs = QgsCoordinateReferenceSystem.fromEpsgId(DEFAULT_SRID)
            tr = QgsCoordinateTransform(sourceCrs, destCrs, self.project)
            requestPoint.transform(tr)
        pid = f"{str(requestPoint.x())},{str(requestPoint.y())}"

        if objectType == 1:# działka
            resp = uldk_xy.getParcelByXY(xy=pid, objectType=1)

            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla współrzędnych {pid}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = res[2]
            region = res[3]
            commune = res[4]
            county = res[5]
            voivodeship = res[6]

        elif objectType == 2:
            resp = uldk_xy.getRegionByXY(xy=pid, objectType=2)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla współrzędnych {pid}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]


        elif objectType == 3:
            resp = uldk_xy.getCommuneByXY(xy=pid, objectType=3)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla współrzędnych {pid}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = res[2]
            county = res[3]
            voivodeship = res[4]


        elif objectType == 4:
            resp = uldk_xy.getCountyByXY(xy=pid, objectType=4)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla współrzędnych {pid}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = res[2]
            voivodeship = res[3]


        elif objectType == 5:
            resp = uldk_xy.getVoivodeshipByXY(xy=pid, objectType=5)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla współrzędnych {pid}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return

            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = None
            voivodeship = res[2]

        elif objectType == 6:
            resp = uldk_xy.GetBuildingByXY(xy=pid, object_type=6)
            if not resp:
                self.iface.messageBar().pushMessage(
                    "Nie udało się pobrać obiektu:",
                    f'API nie zwróciło obiektu dla współrzędnych {pid}',
                    level=Qgis.Critical,
                    duration=10,
                )
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]

        self.addResultsToLayer(
            objectType=objectType,
            wkt=wkt,
            teryt=teryt,
            parcel=parcel,
            region=region,
            commune=commune,
            county=county,
            voivodeship=voivodeship,
            zoomToFeature=zoomToFeature)

        object ={
            1: "działkę o nr teryt: %s",
            2: "obręb ewidencyjny",
            3: "gminę",
            4: "powiat",
            5: "województwo",
            6: "budynek"
        }
        success_message = f"Pobrano {object[objectType]}"  % teryt if objectType == 1 else f"Pobrano {object[objectType]}"

        self.iface.messageBar().pushMessage("Sukces:",
                                            success_message,
                                            level=Qgis.Success, duration=10)

    def addResultsToLayer(self, objectType, wkt, teryt, parcel, region, commune, county, voivodeship, zoomToFeature):
        """dodaje wyniki (odpowiedź z serwera) do mapy jako warstwę z atrybutami i geometrią"""

        # layer
        nazwa = self.nazwy_warstw[objectType]
        layers = self.project.mapLayersByName(nazwa)

        # usuwanie pustych warstw z projektu
        for layer in layers:
            if layer.featureCount() == 0:
                self.project.removeMapLayer(layer)
                layers.remove(layer)


        if layers:
            # jezeli istnieje to dodaj obiekt do warstwy
            layer = layers[0]
            provider = layer.dataProvider()

        else:
            # jezeli nie istnieje to stworz warstwe
            layer = QgsVectorLayer(f"Polygon?crs=EPSG:{DEFAULT_SRID}", nazwa, "memory")
            self.project.addMapLayer(layer)

            provider = layer.dataProvider()

            voivField = QgsField('województwo', QVariant.String, len=30)
            provider.addAttributes([voivField])

            if objectType==6 or objectType == 4 or objectType == 3 or objectType == 2 or objectType == 1:
                conField = QgsField('powiat', QVariant.String, len=30)
                provider.addAttributes([conField])

            if objectType==6 or objectType == 3 or objectType == 2 or objectType == 1:
                comField = QgsField('gmina', QVariant.String, len=30)
                provider.addAttributes([comField])

            if objectType==6 or objectType == 2 or objectType == 1:
                regField = QgsField('obręb', QVariant.String, len=30)
                provider.addAttributes([regField])

            if objectType == 1:
                parField = QgsField('numer', QVariant.String, len=30)
                provider.addAttributes([parField])

            idField = QgsField('teryt', QVariant.String, len=40)
            provider.addAttributes([idField])

            layer.updateFields()

        feat = QgsFeature(provider.fields())
        feat.setGeometry(QgsGeometry().fromWkt(wkt))

        fields_mapping = {
            'teryt': teryt,
            'numer': parcel,
            'obręb': region,
            'gmina': commune,
            'powiat': county,
        }

        if voivodeship != None:
            feat.setAttribute('województwo', voivodeship)

        for field_name, attr in fields_mapping.items():
            if field_name in feat.fields().names():
                feat.setAttribute(field_name, attr or None)
        provider.addFeature(feat)

        if zoomToFeature:
            project_crs = self.project.crs().postgisSrid()

            if project_crs != DEFAULT_SRID:
                source_crs = QgsCoordinateReferenceSystem.fromEpsgId(DEFAULT_SRID)
                dest_crs = QgsCoordinateReferenceSystem.fromEpsgId(project_crs)
                tr = QgsCoordinateTransform(source_crs, dest_crs, self.project)
                box = tr.transform(feat.geometry().boundingBox())
            else:
                box = feat.geometry().boundingBox()

            self.canvas.setExtent(box)
            self.canvas.refresh()
        else:
            layer.triggerRepaint()

        # Funkcja odświeża wszystkie elementy jakie są w warstwie
        self.iface.mapCanvas().refreshAllLayers()


    def checkedFeatureType(self):
        """
        Funkcja pomocnicza sprawdzająca jaki typ obiektu jest zaznaczony w oknie wtyczki
        @returns:
        1 - działka ewidencyjna
        2 - obręb ewidencyjny
        3 - gmina
        4 - powiat
        5 - województwo
        6 - budynek
        0 - niezdefiniowany
        """
        dlg = self.dlg
        if dlg.rdb_dz.isChecked():
            return 1
        elif dlg.rdb_ob.isChecked():
            return 2
        elif dlg.rdb_gm.isChecked():
            return 3
        elif dlg.rdb_pw.isChecked():
            return 4
        elif dlg.rdb_wo.isChecked():
            return 5
        elif dlg.rdb_bu.isChecked():
            return 6
        else:
            return 0
