import os
import math
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant, Qt
from qgis.PyQt.QtGui import QIcon, QPixmap
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.core import (QgsProject, QgsVectorLayer, QgsFeature, 
                       QgsGeometry, QgsPointXY, QgsWkbTypes, QgsField)

from .resources import *
from .UA_slope_topo_builder_dialog import UA_slope_topo_builderDialog

# --- ГЛОБАЛЬНІ КОНСТАНТИ ---
BUFFER_BOTTOM = 0.15
POINT_OFFSET = 0.5
POINT_KOD_VALUE = 44411000
REGION_KOD_VALUE = 92000000
SEARCH_LEN = 500.0
SMOOTH_WINDOW = 5

# --- ГЕОМЕТРИЧНІ ФУНКЦІЇ ---
def fix_crossings_and_smooth(dists_on_bottom, is_reversed):
    if not dists_on_bottom: return []
    clean_dists = []
    last_valid = 0.0
    for d in dists_on_bottom:
        if d is not None:
            last_valid = d
            clean_dists.append(d)
        else:
            clean_dists.append(last_valid)
    fixed = list(clean_dists)
    step_push = 0.05 
    if not is_reversed:
        for i in range(1, len(fixed)):
            if fixed[i] <= fixed[i-1]: fixed[i] = fixed[i-1] + step_push
    else:
        for i in range(1, len(fixed)):
            if fixed[i] >= fixed[i-1]: fixed[i] = fixed[i-1] - step_push
    if len(fixed) < SMOOTH_WINDOW: return fixed
    smoothed = []
    half = SMOOTH_WINDOW // 2
    for i in range(len(fixed)):
        start, end = max(0, i - half), min(len(fixed), i + half + 1)
        chunk = fixed[start:end]
        smoothed.append(sum(chunk) / len(chunk))
    return smoothed

def is_line_reversed(geom_top, geom_bot):
    if geom_top.isEmpty() or geom_bot.isEmpty(): return False
    top_start = geom_top.interpolate(0).asPoint()
    bot_start = geom_bot.interpolate(0).asPoint()
    bot_end = geom_bot.interpolate(geom_bot.length()).asPoint()
    d_ss = math.hypot(top_start.x() - bot_start.x(), top_start.y() - bot_start.y())
    d_se = math.hypot(top_start.x() - bot_end.x(), top_start.y() - bot_end.y())
    return d_se < d_ss

def get_intersection_dist_on_bottom(pt_top, nx, ny, target_geom):
    p1 = QgsPointXY(pt_top.x() + nx * SEARCH_LEN, pt_top.y() + ny * SEARCH_LEN)
    p2 = QgsPointXY(pt_top.x() - nx * SEARCH_LEN, pt_top.y() - ny * SEARCH_LEN)
    ray_1 = QgsGeometry.fromPolylineXY([pt_top, p1])
    ray_2 = QgsGeometry.fromPolylineXY([pt_top, p2])
    def find_closest_dist(ray):
        res = ray.intersection(target_geom)
        if res.isEmpty(): return None
        pts = [QgsPointXY(v) for v in res.vertices()]
        if not pts: return None
        closest_pt, min_d = None, float('inf')
        for p in pts:
            dist = math.hypot(p.x() - pt_top.x(), p.y() - pt_top.y())
            if 0.001 < dist < min_d: closest_pt, min_d = p, dist
        return target_geom.lineLocatePoint(QgsGeometry.fromPointXY(closest_pt)) if closest_pt else None
    d1 = find_closest_dist(ray_1)
    return d1 if d1 is not None else find_closest_dist(ray_2)

# --- ОСНОВНИЙ КЛАС ПЛАГІНА ---
class UA_slope_topo_builder:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = self.tr(u'&Генератор Укосів')
        self.first_start = None

    def tr(self, message):
        return QCoreApplication.translate('UA_slope_topo_builder', message)

    def add_action(self, icon_path, text, callback, enabled_flag=True, add_to_menu=True, add_to_toolbar=True, status_tip=None, parent=None):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)
        if status_tip: action.setStatusTip(status_tip)
        if add_to_toolbar: self.iface.addToolBarIcon(action)
        if add_to_menu: self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)
        return action

    def initGui(self):
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        self.add_action(icon_path, text=self.tr(u'Генератор Укосів'), callback=self.run, parent=self.iface.mainWindow())
        self.first_start = True

    def unload(self):
        for action in self.actions:
            self.iface.removePluginMenu(self.tr(u'&Генератор Укосів'), action)
            self.iface.removeToolBarIcon(action)

    def update_preview_image(self, text):
        image_name = "ukos.png"
        if text == "Обриви": image_name = "obryv.png"
        elif text == "Укоси укріплені": image_name = "ukripleni.png"
        elif text == "Набережні похилі": image_name = "naberezhni.png"
        elif text == "Підпірні стінки похилі": image_name = "pidpirni.png"
            
        img_path = os.path.join(self.plugin_dir, image_name)
        pixmap = QPixmap(img_path)
        
        if hasattr(self.dlg, 'lblPreview'):
            if not pixmap.isNull():
                scaled_pixmap = pixmap.scaled(self.dlg.lblPreview.size(), Qt.KeepAspectRatio, Qt.SmoothTransformation)
                self.dlg.lblPreview.setPixmap(scaled_pixmap)
            else:
                self.dlg.lblPreview.setText(f"Картинка\n{image_name}\nне знайдена")

    def run(self):
        if self.first_start == True:
            self.first_start = False
            self.dlg = UA_slope_topo_builderDialog()
            
            if hasattr(self.dlg, 'cmbSlopeType') and hasattr(self.dlg, 'lblPreview'):
                self.dlg.cmbSlopeType.currentTextChanged.connect(self.update_preview_image)
            
            if hasattr(self.dlg, 'btnGenerate'):
                self.dlg.btnGenerate.clicked.connect(self.dlg.accept)
        
        if hasattr(self.dlg, 'cmbSlopeType'):
            self.update_preview_image(self.dlg.cmbSlopeType.currentText())

        self.dlg.show()
        
        if self.dlg.exec_():
            try:
                layer_top = self.dlg.cmbLayerTop.currentLayer()
                layer_bot = self.dlg.cmbLayerBot.currentLayer()
                slope_type = self.dlg.cmbSlopeType.currentText()
                step = self.dlg.spinStep.value()

                if not layer_top or not layer_bot:
                    QMessageBox.warning(self.dlg, "Увага", "Оберіть обидва шари!")
                    return
                self.process_slopes(layer_top, layer_bot, slope_type, step)
            except Exception:
                import traceback
                QMessageBox.critical(None, "Помилка", f"Сталася помилка:\n{traceback.format_exc()}")

    def process_slopes(self, layer_top, layer_bot, slope_type, step):
        try:
            use_points, point_layer = (slope_type in ["Укоси укріплені", "Набережні похилі"]), None
            if use_points:
                layers = QgsProject.instance().mapLayersByName("92_point")
                if layers: point_layer = layers[0]
                else:
                    point_layer = QgsVectorLayer("Point?crs=" + layer_top.crs().authid(), "92_point", "memory")
                    point_layer.dataProvider().addAttributes([QgsField("KOD", QVariant.Int)])
                    point_layer.updateFields()
                    QgsProject.instance().addMapLayer(point_layer)
                point_layer.startEditing()

            use_regions = (slope_type == "Підпірні стінки похилі")
            region_layer = None
            if use_regions:
                layers = QgsProject.instance().mapLayersByName("92_region")
                if layers: region_layer = layers[0]
                else:
                    # СТВОРЮЄМО ШАР ЯК MULTIPOLYGON
                    region_layer = QgsVectorLayer("MultiPolygon?crs=" + layer_top.crs().authid(), "92_region", "memory")
                    region_layer.dataProvider().addAttributes([QgsField("KOD", QVariant.Int)])
                    region_layer.updateFields()
                    QgsProject.instance().addMapLayer(region_layer)
                region_layer.startEditing()

            layer_top.startEditing()
            target_geoms = [f.geometry() for f in layer_bot.getFeatures() if not f.geometry().isEmpty()]
            if not target_geoms: return QMessageBox.critical(self.dlg, "Помилка", "Шар низу порожній!")

            features = layer_top.selectedFeatures() if layer_top.selectedFeatureCount() > 0 else layer_top.getFeatures()
            count_features = count_points = count_polygons = 0

            for feature in features:
                top_geom = feature.geometry()
                if top_geom.isEmpty(): continue
                closest_bot_geom = min(target_geoms, key=lambda tg: top_geom.distance(tg))
                is_reversed = is_line_reversed(top_geom, closest_bot_geom)
                
                points_top, d_values_top, d, top_len = [], [], 0.0, top_geom.length()
                
                while d < top_len - 0.001:
                    g = top_geom.interpolate(d)
                    if not g.isEmpty():
                        points_top.append(g.asPoint())
                        d_values_top.append(d)
                    d += step
                
                g_end = top_geom.interpolate(top_len)
                if g_end.isEmpty(): g_end = top_geom.interpolate(top_len - 0.001)
                
                if not g_end.isEmpty():
                    points_top.append(g_end.asPoint())
                    d_values_top.append(top_len)
                
                dists_on_bottom = []
                for i, pt_top in enumerate(points_top):
                    d_val = d_values_top[i]
                    d_p, d_n = max(0.0, d_val - 0.1), min(top_len, d_val + 0.1)
                    if d_p == d_n: d_p, d_n = 0.0, top_len
                    
                    g_p = top_geom.interpolate(d_p)
                    g_n = top_geom.interpolate(d_n)
                    if g_p.isEmpty(): g_p = top_geom.interpolate(max(0.0, d_p - 0.001))
                    if g_n.isEmpty(): g_n = top_geom.interpolate(min(top_len, d_n - 0.001))
                    
                    if g_p.isEmpty() or g_n.isEmpty():
                        dists_on_bottom.append(None)
                        continue
                        
                    p_p, p_n = g_p.asPoint(), g_n.asPoint()
                    dx, dy = p_n.x() - p_p.x(), p_n.y() - p_p.y()
                    len_v = math.hypot(dx, dy)
                    dist_int = get_intersection_dist_on_bottom(pt_top, -dy/len_v, dx/len_v, closest_bot_geom) if len_v > 0 else None
                    if dist_int is None:
                        if i == 0: dist_int = 0.0 if not is_reversed else closest_bot_geom.length()
                        elif i == len(points_top)-1: dist_int = closest_bot_geom.length() if not is_reversed else 0.0
                    dists_on_bottom.append(dist_int)
                
                f_bot_dists = fix_crossings_and_smooth(dists_on_bottom, is_reversed)
                
                parts = [] # Для лінійних штрихів
                polys = [] # СПИСОК ДЛЯ ЗБОРУ ТРИКУТНИКІВ У МУЛЬТИПОЛІГОН
                
                for i, pt_top in enumerate(points_top):
                    if f_bot_dists[i] is None: continue
                    
                    dist_b = max(0, min(closest_bot_geom.length(), f_bot_dists[i]))
                    g_bot = closest_bot_geom.interpolate(dist_b)
                    if g_bot.isEmpty(): g_bot = closest_bot_geom.interpolate(max(0, dist_b - 0.001))
                    if g_bot.isEmpty(): continue
                    
                    pt_bot = g_bot.asPoint()
                    vx, vy = pt_bot.x() - pt_top.x(), pt_bot.y() - pt_top.y()
                    full_d = math.hypot(vx, vy)
                    
                    if full_d > BUFFER_BOTTOM + 0.05:
                        is_s = False
                        
                        if slope_type in ["Обриви", "Набережні похилі", "Підпірні стінки похилі"]: d_len = full_d - BUFFER_BOTTOM
                        else:
                            if i % 2 == 0: d_len = full_d - BUFFER_BOTTOM
                            else: d_len, is_s = full_d / 3.0, True
                        r = d_len / full_d
                        pt_end = QgsPointXY(pt_top.x()+vx*r, pt_top.y()+vy*r)
                        
                        if slope_type == "Підпірні стінки похилі":
                            nx, ny = -vy / full_d, vx / full_d
                            hw = step / 8.0 
                            p1 = QgsPointXY(pt_top.x() + nx * hw, pt_top.y() + ny * hw)
                            p2 = QgsPointXY(pt_top.x() - nx * hw, pt_top.y() - ny * hw)
                            # ДОДАЄМО ТРИКУТНИК ДО СПИСКУ
                            polys.append([[p1, pt_end, p2, p1]])
                        else:
                            parts.append([pt_top, pt_end])
                            
                        if slope_type == "Укоси укріплені" and is_s:
                            r_p = (d_len + POINT_OFFSET) / full_d
                            f_p = QgsFeature(point_layer.fields())
                            f_p.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(pt_top.x()+vx*r_p, pt_top.y()+vy*r_p)))
                            f_p.setAttribute("KOD", POINT_KOD_VALUE)
                            point_layer.addFeature(f_p); count_points += 1
                        
                        if slope_type == "Набережні похилі" and i < len(points_top) - 1:
                            if f_bot_dists[i+1] is None: continue
                            pt_top_next = points_top[i+1]
                            dist_b_next = max(0, min(closest_bot_geom.length(), f_bot_dists[i+1]))
                            
                            g_bot_next = closest_bot_geom.interpolate(dist_b_next)
                            if g_bot_next.isEmpty(): g_bot_next = closest_bot_geom.interpolate(max(0, dist_b_next - 0.001))
                            if g_bot_next.isEmpty(): continue
                            
                            pt_bot_next = g_bot_next.asPoint()
                            mx, my = (pt_top.x()+pt_top_next.x())/2, (pt_top.y()+pt_top_next.y())/2
                            mbx, mby = (pt_bot.x()+pt_bot_next.x())/2, (pt_bot.y()+pt_bot_next.y())/2
                            vmx, vmy = mbx - mx, mby - my
                            dm = math.hypot(vmx, vmy)
                            if dm > 0:
                                for r_val in [0.5, 1.0]:
                                    if dm >= r_val:
                                        r_m = r_val / dm
                                        pt_f = QgsFeature(point_layer.fields())
                                        pt_f.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(mx + vmx*r_m, my + vmy*r_m)))
                                        pt_f.setAttribute("KOD", POINT_KOD_VALUE)
                                        point_layer.addFeature(pt_f)
                                        count_points += 1

                # ЗБЕРЕЖЕННЯ ЛІНІЙ (ЯКЩО ВОНИ Є)
                if parts: 
                    f_out = QgsFeature(feature); f_out.setGeometry(QgsGeometry.fromMultiPolylineXY(parts))
                    layer_top.addFeature(f_out); count_features += 1
                
                # СТВОРЕННЯ ЄДИНОГО МУЛЬТИПОЛІГОНА (ДЛЯ ПІДПІРНИХ СТІНОК)
                if polys:
                    poly_geom = QgsGeometry.fromMultiPolygonXY(polys)
                    f_poly = QgsFeature(region_layer.fields())
                    f_poly.setGeometry(poly_geom)
                    f_poly.setAttribute("KOD", REGION_KOD_VALUE)
                    region_layer.addFeature(f_poly)
                    count_polygons += 1

            layer_top.commitChanges()
            if point_layer: point_layer.commitChanges()
            if region_layer: region_layer.commitChanges()
            
            msg = f"Готово!\nЛіній: {count_features}\nТочок: {count_points}"
            if count_polygons > 0:
                msg += f"\nМультиполігонів (Стінок): {count_polygons}"
                
            QMessageBox.information(self.dlg, "Успіх", msg)
            
        except Exception:
            import traceback
            QMessageBox.critical(self.dlg, "Помилка", f"Помилка розрахунків:\n{traceback.format_exc()}")