# -*- coding: utf-8 -*-
"""
/***************************************************************************
 TransparencySetter
                                 A QGIS plugin
 Apply the specified transparency value to both vector and raster layers, as well as layers within the selected groups in the Layer Panel
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-05-10
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Idrostudi Srl
        email                : martinis@idrostudi.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.core import QgsVectorLayer, QgsRasterLayer
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from PyQt5.QtWidgets import QWidgetAction, QSlider, QLabel, QSpinBox, QStyleOptionSlider, QStyle, QCheckBox
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QIntValidator, QColor
from PyQt5.QtCore import Qt, QPoint
import os.path

# Initialize Qt resources from file resources.py
from .resources import *


class TransparencySetter:
    """
    QGIS Plugin Implementation for setting transparency to layers and groups.
    """

    def __init__(self, iface):
        """
        Constructor.

        :param iface: An interface instance that provides the hook to manipulate the QGIS application.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'TransparencySetter_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Transparency Setter')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    def tr(self, message):
        """
        Get the translation for a string using Qt translation API.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        return QCoreApplication.translate('TransparencySetter', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None
    ):
        """
        Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource path or a file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """
        Create the menu entries and toolbar icons inside the QGIS GUI.
        """

        # will be set False in run()
        self.first_start = True

        # Create a slider widget
        self.slider = SteppedSlider(Qt.Horizontal)
        self.slider.setMaximumWidth(100)  # Set slider maximum width
        self.slider.setRange(0, 100)  # Range from 0 to 100
        self.slider.setValue(0)  # Initial value
        self.slider.setTickInterval(10)  # Set slider tick interval
        self.slider.setPageStep(10)  # Set slider page step
        self.slider.setSingleStep(10)  # Set slider single step
        self.slider.setTracking(True)
        self.slider.valueChanged.connect(self.slider_changed)

        # Create a checkbox widget to disable functionality
        self.disable_checkbox = QCheckBox()
        self.disable_checkbox.setToolTip("Disable plugin")
        self.disable_checkbox.setChecked(False)  # Initial state
        self.disable_checkbox.stateChanged.connect(self.checkbox_state_changed)

        # Create a label widget
        self.label = QLabel("Transparency:")

        # Create a QSpinBox widget
        self.transparency_input = QSpinBox()
        self.transparency_input.setRange(0, 100)  # Range from 0 to 100
        self.transparency_input.setValue(0)  # Initial value
        self.transparency_input.setSingleStep(10)  # Set step size to 10
        self.transparency_input.valueChanged.connect(self.spin_box_changed)

        # Add widgets to the toolbar
        self.toolbar = self.iface.addToolBar("Layer Transparency Toolbar")
        self.toolbar.setObjectName("Layer Transparency Toolbar")

        self.toolbar.addWidget(self.disable_checkbox)
        self.toolbar.addWidget(self.label)
        self.toolbar.addWidget(self.transparency_input)
        self.toolbar.addWidget(self.slider)

        self.iface.layerTreeView().selectionModel().selectionChanged.connect(self.update_slider_from_selected_layer)

    def unload(self):
        """
        Removes the plugin menu item and icon from QGIS GUI.
        """
        for action in self.actions:
            self.iface.removePluginMenu(self.tr(u'&Transparency Setter'), action)
            self.iface.removeToolBarIcon(action)

    def checkbox_state_changed(self, state):
        """
        Handle checkbox state changes.

        :param state: The new checkbox state.
        :type state: int
        """
        disabled = state == Qt.Checked
        # Make other widgets gray and unreachable when checkbox is checked
        self.slider.setEnabled(not disabled)
        self.label.setEnabled(not disabled)
        self.transparency_input.setEnabled(not disabled)

    def apply_transparency(self, transparency):
        """
        Apply the specified transparency value to selected layers or groups.

        :param transparency: The transparency value to apply.
        :type transparency: int
        """
        if self.disable_checkbox.isChecked():
            return

        layer_tree_view = self.iface.layerTreeView()
        selected_nodes = layer_tree_view.selectedNodes()

        if selected_nodes:
            for selected_node in selected_nodes:
                if selected_node.nodeType() == 0:
                    self.apply_transparency_to_group(selected_node, transparency)
                elif selected_node.nodeType() == 1:
                    self.apply_transparency_to_layer(selected_node.layer(), transparency)
        else:
            print("Please select one or more groups and/or layers in the Layer Panel.")

    def apply_transparency_to_group(self, group_node, transparency):
        """
        Apply the specified transparency value to layers within the given group.

        :param group_node: The group node to apply transparency to.
        :type group_node: QgsLayerTreeNode

        :param transparency: The transparency value to apply.
        :type transparency: int
        """
        for child_node in group_node.children():
            if child_node.nodeType() == 0:  # Group node
                self.apply_transparency_to_group(child_node, transparency)
            elif child_node.nodeType() == 1:  # Layer node
                self.apply_transparency_to_layer(child_node.layer(), transparency)

    def apply_transparency_to_layer(self, layer, transparency):
        """
        Apply the specified transparency value to the given layer.

        :param layer: The layer to apply transparency to.
        :type layer: QgsMapLayer

        :param transparency: The transparency value to apply.
        :type transparency: int
        """
        opacity = (100 - transparency) / 100.0
        if isinstance(layer, QgsVectorLayer):
            layer.setOpacity(opacity)
        elif isinstance(layer, QgsRasterLayer):
            layer.renderer().setOpacity(opacity)
        layer.triggerRepaint()

    def update_slider_from_selected_layer(self):
        """
        Update the slider value based on the currently selected layer or group in the layer panel.
        """
        if self.disable_checkbox.isChecked():
            return

        layer_tree_view = self.iface.layerTreeView()
        selected_nodes = layer_tree_view.selectedNodes()

        selected_layer_node = None
        selected_group_node = None

        for selected_node in selected_nodes:
            if selected_node.nodeType() == 1:  # Layer node
                selected_layer_node = selected_node
            elif selected_node.nodeType() == 0:  # Group node
                selected_group_node = selected_node

        if selected_layer_node:
            self.update_slider_from_layer_node(selected_layer_node)
        elif selected_group_node:
            self.update_slider_from_group_node(selected_group_node)

    def update_slider_from_layer_node(self, layer_node):
        """
        Update the slider value based on the transparency of the selected layer.

        :param layer_node: The layer node to update the slider for.
        :type layer_node: QgsLayerTreeNode
        """
        if self.disable_checkbox.isChecked():
            return

        layer = layer_node.layer()
        if isinstance(layer, QgsRasterLayer):
            transparency = (1 - layer.renderer().opacity()) * 100
        elif isinstance(layer, QgsVectorLayer):
            transparency = (1 - layer.opacity()) * 100
        rounded_transparency = round(transparency)
        self.transparency_input.setStyleSheet("color: black;")
        self.slider.setValue(int(rounded_transparency))
        self.transparency_input.setValue(int(rounded_transparency))

    def update_slider_from_group_node(self, group_node):
        """
        Update the slider value based on the transparency of the selected group.

        :param group_node: The group node to update the slider for.
        :type group_node: QgsLayerTreeNode
        """
        if self.disable_checkbox.isChecked():
            return

        transparency_values = []
        self.get_transparency_values_from_group(group_node, transparency_values)

        if len(set(transparency_values)) == 1:  # All layers have the same transparency
            transparency = transparency_values[0]
            self.transparency_input.setStyleSheet("color: black;")
        else:  # Layers have different transparency settings
            transparency = self.slider.value()  # Keep the current slider value
            self.transparency_input.setStyleSheet("color: red;")

        self.slider.setValue(int(transparency))
        self.transparency_input.setValue(int(transparency))

    def get_transparency_values_from_group(self, group_node, transparency_values):
        """
        Get the transparency values of all layers within the given group.

        :param group_node: The group node to get transparency values from.
        :type group_node: QgsLayerTreeNode

        :param transparency_values: List to store the transparency values.
        :type transparency_values: list[float]
        """
        layer_nodes = group_node.children()
        for layer_node in layer_nodes:
            if layer_node.nodeType() == 0:  # Child node is a group
                self.get_transparency_values_from_group(layer_node, transparency_values)
            else:  # Child node is a layer
                layer = layer_node.layer()
                if isinstance(layer, QgsRasterLayer):
                    transparency = (1.0 - layer.renderer().opacity()) * 100.0
                elif isinstance(layer, QgsVectorLayer):
                    transparency = (1.0 - layer.opacity()) * 100.0
                transparency_values.append(transparency)

    def slider_changed(self, value):
        """
        Handle slider value changes and update the transparency.

        :param value: The new slider value.
        :type value: int
        """
        if self.disable_checkbox.isChecked():
            return

        self.label.setText(f"Transparency:")
        self.transparency_input.setValue(int(value))
        self.apply_transparency(value)

    def spin_box_changed(self, value):
        """
        Handle spin box value changes and update the transparency.

        :param value: The new spin box value.
        :type value: int
        """
        if self.disable_checkbox.isChecked():
            return

        self.slider.setValue(int(value))
        self.apply_transparency(value)

    def run(self):
        """
        This method is no longer needed as the slider directly handles the transparency.
        """
        pass


class SteppedSlider(QSlider):
    """
    Custom QSlider class that supports stepping slider values.
    """

    def __init__(self, orientation, parent=None):
        """
        Constructor.

        :param orientation: The orientation of the slider.
        :type orientation: Qt.Orientation

        :param parent: Parent widget for the slider. Defaults to None.
        :type parent: QWidget
        """
        super(SteppedSlider, self).__init__(orientation, parent)

    def mouseMoveEvent(self, e):
        """
        Override the mouse move event to snap the slider value to the nearest step.

        :param e: The mouse event.
        :type e: QMouseEvent
        """
        opt = QStyleOptionSlider()
        self.initStyleOption(opt)
        sr = self.style().subControlRect(QStyle.CC_Slider, opt, QStyle.SC_SliderHandle, self)

        if e.buttons() & Qt.LeftButton:
            if sr.contains(e.pos()) == False:
                new_value = self.style().sliderValueFromPosition(self.minimum(), self.maximum(), e.x(), sr.width())
                stepped_value = round(new_value / 10) * 10
                self.setValue(int(stepped_value))
                e.accept()
        super(SteppedSlider, self).mouseMoveEvent(e)
