# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PublicTransitAnalysis
                                 A QGIS plugin
 Using OpenTripPlanner to calculate public transport reachability from a
    starting point to all stops in a GTFS feed.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-05-14
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Julek Weck
        email                : j.weck@tu-braunschweig.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import math

# class imports
from .stop import Stop
from .station import Station
from .route import Route
from .referencePoint import ReferencePoint
# python packages
import requests, json
import geopandas as gpd
import pandas as pd
from datetime import time, date, datetime
import sys
import os

sys.path.append('C:\\OSGeo4W64\\apps\\qgis\\python')
sys.path.append('C:\\OSGeo4W64\\apps\\qgis\\python\\plugins')

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .transit_reachability_analyser_dialog import TransitReachabilityAnalyserDialog
import os.path

from qgis.gui import *
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant
# This is needed to create own graduated symbol renderer
from qgis.PyQt import QtGui
from qgis.PyQt.QtGui import QIcon, QColor
from qgis.PyQt.QtWidgets import QAction, QFileDialog

from qgis.core import * #this would be enough, but then there are red marks in the code
from qgis.core import QgsStyle, QgsColorRamp, QgsColorRampShader #for a color ramp
from qgis.core import QgsLayerTreeLayer, QgsLayerTreeGroup #to load all layers, also those in groups
from qgis.core import QgsWkbTypes, QgsFillSymbol, QgsSimpleLineSymbolLayer # to set the polygon line to zero
# downloaded for symbology, https://docs.qgis.org/testing/en/docs/pyqgis_developer_cookbook/vector.html#appearance-symbology-of-vector-layers
from qgis.core import (
   QgsApplication,
   QgsProject,
   QgsGraduatedSymbolRenderer,
   QgsMarkerSymbol,
   QgsRendererRange,
   QgsSymbol,
   QgsVectorLayer
 )

class TransitReachabilityAnalyser:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'TransitReachabilityAnalyser_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Transit Reachability Analyser')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('TransitReachabilityAnalyser', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = f'{self.plugin_dir}/icon.png'#':/plugins/transit_reachability_analyser/icon.png' Forum: https://gis.stackexchange.com/a/457606
        self.add_action(
            icon_path,
            text=self.tr(u'Transit Reachability Analyser'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Transit Reachability Analyser'),
                action)
            self.iface.removeToolBarIcon(action)

    """
    My own methods
    """
    """
    Reachability Analysis Methods
    """
    def get_request_url(self):
        if self.dlg.rb_otp_manually_started_8080.isChecked():
            return "http://localhost:8080/otp/gtfs/v1"
        elif self.dlg.rb_otp_manually_started_changed_port.isChecked():
            port_number = self.dlg.le_port_number.text()
            return f"http://localhost:{port_number}/otp/gtfs/v1"

    def get_walk_speed(self):
        if self.dlg.rb_2kmh.isChecked():
            walk_speed = 2.1 / 3.6
        elif self.dlg.rb_3_5kmh.isChecked():
            walk_speed = 3.0 / 3.6
        elif self.dlg.rb_4_5kmh.isChecked():
            walk_speed = 4.5 / 3.6
        elif self.dlg.rb_5_5kmh.isChecked():
            walk_speed = 5.5 / 3.6
        elif self.dlg.rb_6_5kmh.isChecked():
            walk_speed = 6.5 / 3.6
        elif self.dlg.rb_personalised_speed.isChecked():
            input = self.dlg.le_personalized_tempo.text()
            try:
                walk_speed = float(input) / 3.6
            except ValueError:
                error_message = "The walk_speed has to be a float with '.' as seperator" + "\n"
                self.iface.messageBar().pushMessage(error_message)
                return False
        else:
            self.iface.messageBar().pushMessage("One walk speed option has to be chosen")
            return False
        return walk_speed

    def get_max_walk_time(self):
        if self.dlg.rb_5min.isChecked():
            max_walking_time = 5 * 60  # minutes in seconds
        elif self.dlg.rb_6min.isChecked():
            max_walking_time = 6 * 60
        elif self.dlg.rb_10min.isChecked():
            max_walking_time = 10 * 60
        elif self.dlg.rb_personalised_walktime.isChecked():
            input = self.dlg.le_max_walking_time.text()
            try:
                max_walking_time = int(input) * 60  # minutes in seconds
            except ValueError:
                error_message = "The max_walking_time has to be an integer" + "\n"
                self.iface.messageBar().pushMessage(error_message)
                return False
        else:
            self.iface.messageBar().pushMessage("One walk distance option has to be chosen")
            return False
        return max_walking_time

    def query_all_stops_incl_departure_times(self, analysis_parameters:ReferencePoint):
        unix_timestamp = int(datetime.timestamp(datetime.combine(analysis_parameters.day, analysis_parameters.time_start)))
        time_range = analysis_parameters.search_window
        #the number of departures is depending on the time range and the frequency of the departures
        #To get all departures, even of long time ranges and short frequencies, the value is randomly high.
        plan = f"""
            {{stops {{
                gtfsId
                name
                lat
                lon
                vehicleMode
                stoptimesWithoutPatterns(
                    startTime: {unix_timestamp}
                    timeRange: {time_range}
                    numberOfDepartures:100000) {{
                        scheduledDeparture
                        trip {{
                            route {{
                                gtfsId
                                shortName
                            }}
                        }}
                    }}
            }}
            }}
            """
        if self.check_grizzly_server_is_running():
            url = self.get_request_url()
            queried_stops = requests.post(url, json={"query": plan})
            queried_stops = json.loads(queried_stops.content)
            queried_stops = queried_stops["data"]["stops"]
            return queried_stops
        else:
            print("stops could not be queried because OTP is not reachable")

    def create_request_object(self):
        layer_name = self.dlg.le_layer_name.text()
        if self.dlg.le_filepath_itineraries.text() != "":
            filepath = self.dlg.le_filepath_itineraries.text()
        else:
            self.iface.messageBar().pushMessage("The filepath has to be selected first")
            return

        if not self.get_walk_speed():
            return
        else:
            walk_speed = self.get_walk_speed()

        if not self.get_max_walk_time():
            return
        else:
            max_walking_time = self.get_max_walk_time()
        analysis_parameters = ReferencePoint(
            #TODO make the try except statements not in the setter of RererencePoint, but in this method
            # TODO how to end the code, if except in try except
            lat=self.dlg.le_lat_of_start_end.text(),
            lon=self.dlg.le_lon_of_start_end.text(),
            day=self.dlg.le_date.text(),
            time_start=self.dlg.le_time_start.text(),
            time_end=self.dlg.le_time_end.text(),
            walk_speed=walk_speed,
            max_walking_time=max_walking_time,
            layer_name=layer_name,
            filepath=filepath
        )
        if analysis_parameters.incorrect_input:
            self.iface.messageBar().pushMessage(analysis_parameters.error_message)
            return
        return analysis_parameters

    def create_stop_and_route_objects(self, queried_stops, analysis_parameters):
        stop_objects = []
        all_routes = []
        for stop in queried_stops:
            route_objects = []
            # create a route object for every line which departs at this specific stop
            # the gtfsID of the stop is important too now to which stop the route objet belongs
            for stop_times in stop["stoptimesWithoutPatterns"]:
                route_gtfsId = stop_times["trip"]["route"]["gtfsId"]
                route_shortName = stop_times["trip"]["route"]["shortName"]
                #check if the routnumber already exists
                route_already_created = False
                for route in route_objects:
                    if route.gtfs_id == route_gtfsId:
                        route_already_created = True
                if not(route_already_created):
                    obj = Route(route_gtfsId, route_shortName, stop["gtfsId"])
                    route_objects.append(obj)
            # add the departure times to the related route object
            for stop_times in stop["stoptimesWithoutPatterns"]:
                route_gtfsId = stop_times["trip"]["route"]["gtfsId"]
                seconds_since_midnight = stop_times["scheduledDeparture"]
                m, s = divmod(seconds_since_midnight, 60)
                h, m = divmod(m, 60)
                departure_time = time(hour=h, minute=m, second=s)
                for route in route_objects:
                    if route.gtfs_id == route_gtfsId:
                        route.add_departure_time(departure_time)
                        route.frequency = route.calculate_frequency(analysis_parameters)
                        all_routes.append(route)
            new_stop = Stop(stop["name"], stop["gtfsId"], stop["lat"], stop["lon"], stop["vehicleMode"], route_objects)
            stop_objects.append(new_stop)
        return stop_objects, all_routes

    def create_stations(self, stop_collection):
        station_collection = []
        current_stop_name = stop_collection[0].name
        related_stops = [stop_collection[0]]
        for element in stop_collection[1:]:
            if element.name == current_stop_name:
                related_stops.append(element)
            else:
                station = Station(current_stop_name, related_stops.copy())
                #The runtime will be about 10s longer in total if the distance calculation is enabled
                station.calculate_max_distance_station_to_stop(self.get_request_url())
                station_collection.append(station)
                current_stop_name = element.name
                related_stops.clear()
                related_stops.append(element)
        return station_collection

    def create_dataframe_with_station_attributes(self, station_collection, analysis_parameters:ReferencePoint):
        """
        Codes of the negative numbers:
        -1: there is no Itinerary to/from this station -> not reachable
        -2: This is the point to which/ from which every itinerary goes (referencePoint)
        The renderer used in the symbology methods needs a numeric value to show a point.
        """

        # The first row of the data frame will be the referencePoint
        name_collection = ["Reference Point"]
        trip_time_collection = [-2]
        car_driving_time_collection = [-2]
        travel_time_ratio_collection = [-2]

        number_of_transfers_collection = [-2]
        meters_to_first_stop_collection = [-2]
        walktime_to_first_stop_collection = [-2]
        itinerary_frequency_collection = [-2]
        selected_itineraries_collection = [None]
        possible_itineraries_collection = [None]
        max_distance_station_to_stop_collection = [-2]

        #attributes for the analysis_parameters object
        date_collection = [None]
        time_start_collection = [None]
        time_end_collection = [None]
        walk_speed_collection = [None]
        max_walking_time_collection = [None]
        catchment_area_collection = [None]
        first_possible_stops_collection = [None]


        first_stop_data = ""
        date_collection[0] = analysis_parameters.day.isoformat()
        time_start_collection[0] = analysis_parameters.time_start.isoformat(timespec='minutes')
        time_end_collection[0] = analysis_parameters.time_end.isoformat(timespec='minutes')
        walk_speed_collection[0] = analysis_parameters.walk_speed*3.6
        max_walking_time_collection[0] = analysis_parameters.max_walking_time/60
        catchment_area_collection[0] = analysis_parameters.catchment_area
        for start_stop in analysis_parameters.get_first_possible_stops():
            data = start_stop + ", "
            first_stop_data = first_stop_data + data
        first_possible_stops_collection[0] = first_stop_data


        for station in station_collection:
            selected_itineraries_data = ""
            first_stop_data = ""
            possible_itineraries_data = ""
            name_collection.append(station.name)
            if station.trip_time is not None:
                trip_time_collection.append(round(station.trip_time,1))
            else:
                trip_time_collection.append(-1)
            if station.car_driving_time is not None:
                car_driving_time_collection.append(round(station.car_driving_time,1))
            else:
                car_driving_time_collection.append(-1)
            if station.travel_time_ratio is not None:
                travel_time_ratio_collection.append(round(station.travel_time_ratio,1))
            else:
                travel_time_ratio_collection.append(-1)
            if station.number_of_transfers is not None:
                number_of_transfers_collection.append(round(station.number_of_transfers,1))
            else:
                number_of_transfers_collection.append(-1)
            if station.meters_to_first_stop is not None:
                meters_to_first_stop_collection.append(round(station.meters_to_first_stop,1))
                walktime = (station.meters_to_first_stop/analysis_parameters.walk_speed) / 60 #seconds in minutes
                walktime_to_first_stop_collection.append(round(walktime,1))
            else:
                meters_to_first_stop_collection.append(-1)
                walktime_to_first_stop_collection.append(-1)
            if station.itinerary_frequency is not None:
                itinerary_frequency_collection.append(round(station.itinerary_frequency,1))
            else:
                itinerary_frequency_collection.append(-1)
            for itinerary in station.selected_itineraries:
                data = f"{itinerary.route_numbers}, duration: {itinerary.duration}, frequency: {itinerary.frequency}, meters_to_first_stop: {round(itinerary.meters_first_stop, 1)}, walktime_to_first_stop: {round(((itinerary.meters_first_stop/analysis_parameters.walk_speed) / 60), 1)}, firstStop: {itinerary.first_stop}, lastStop:{itinerary.last_stop};\n"
                selected_itineraries_data = selected_itineraries_data + data
                start_stop = itinerary.first_stop + ", "
                first_stop_data = first_stop_data + start_stop
            selected_itineraries_collection.append(selected_itineraries_data)
            first_possible_stops_collection.append(first_stop_data)
            for itinerary in station.itineraries_with_permissible_catchment_area:
                data = f"{itinerary.route_numbers}, duration: {itinerary.duration}, frequency: {itinerary.frequency}, meters_to_first_stop: {round(itinerary.meters_first_stop, 1)}, walktime_to_first_stop: {round(((itinerary.meters_first_stop/analysis_parameters.walk_speed) / 60), 1)}, firstStop: {itinerary.first_stop}, lastStop:{itinerary.last_stop};\n"
                possible_itineraries_data = possible_itineraries_data + data
            possible_itineraries_collection.append(possible_itineraries_data)
            max_distance_station_to_stop_collection.append(station.max_distance_station_to_stop)


            date_collection.append(analysis_parameters.day.isoformat())
            time_start_collection.append(analysis_parameters.time_start.isoformat(timespec='minutes'))
            time_end_collection.append(analysis_parameters.time_end.isoformat(timespec='minutes'))
            walk_speed_collection.append(round(analysis_parameters.walk_speed*3.6, 1))
            max_walking_time_collection.append(round(analysis_parameters.max_walking_time/60, 1))
            catchment_area_collection.append(round(analysis_parameters.catchment_area, 1))

        df = pd.DataFrame(
            {
                "name": name_collection,
                "travel_time[min]": trip_time_collection,
                "travel_time_ratio": travel_time_ratio_collection,
                "frequency_[min]": itinerary_frequency_collection,
                "walk_time_[min]": walktime_to_first_stop_collection,
                "walk_distance_[m]": meters_to_first_stop_collection,
                "number_of_transfers": number_of_transfers_collection,
                "travel_time_car_[min]": car_driving_time_collection,
                "max_distance_station_to_stop": max_distance_station_to_stop_collection,
                "selected_itinerarie": selected_itineraries_collection,
                "possible_itineraries": possible_itineraries_collection,
                "date": date_collection,
                "time_start": time_start_collection,
                "time_end": time_end_collection,
                "walk_speed_[km/h]": walk_speed_collection,
                "max_walking_time_[min]": max_walking_time_collection,
                "catchment_area_[m]": catchment_area_collection,
                "first_possible_stops": first_possible_stops_collection,
            }
        )
        return df

    def create_dataframe_for_stop_objects(self, stop_collection, analysis_parameters:ReferencePoint):
        name_collection = ["Reference Point"]
        gtfs_id_collection = [None]
        vehicle_mode_collection = [None]
        related_routes_collection = [None]

        # attributes for the analysis_parameters object
        date_collection = [None]
        time_start_collection = [None]
        time_end_collection = [None]
        walk_speed_collection = [None]
        max_walking_time_collection = [None]
        catchment_area_collection = [None]
        possible_start_stations_collection = [None]


        start_station_data = ""
        date_collection[0] = analysis_parameters.day.isoformat()
        time_start_collection[0] = analysis_parameters.time_start.isoformat(timespec='minutes')
        time_end_collection[0] = analysis_parameters.time_end.isoformat(timespec='minutes')
        walk_speed_collection[0] = analysis_parameters.walk_speed * 3.6
        max_walking_time_collection[0] = analysis_parameters.max_walking_time / 60
        catchment_area_collection[0] = analysis_parameters.catchment_area
        for start_station in analysis_parameters.get_first_possible_stops():
            data = start_station + ", "
            start_station_data = start_station_data + data
        possible_start_stations_collection[0] = start_station_data

        for stop in stop_collection:
            departure_data = ""
            name_collection.append(stop.name)
            gtfs_id_collection.append(stop.gtfs_id)
            vehicle_mode_collection.append(stop.vehicle_mode)
            for route in stop.related_routes:
                departure_times = []
                for departure in route.get_departure_times():
                    departure_times.append(departure.isoformat(timespec='minutes'))
                data = f"{route.short_name}: averageFrequency: {route.frequency}, departures: {departure_times} \n"
                departure_data = departure_data + data
            related_routes_collection.append(departure_data)

            date_collection.append(analysis_parameters.day.isoformat())
            time_start_collection.append(analysis_parameters.time_start.isoformat(timespec='minutes'))
            time_end_collection.append(analysis_parameters.time_end.isoformat(timespec='minutes'))
            walk_speed_collection.append(analysis_parameters.walk_speed*3.6)
            max_walking_time_collection.append(analysis_parameters.max_walking_time/60)
            catchment_area_collection.append(analysis_parameters.catchment_area)

        df = pd.DataFrame(
            {
                "name": name_collection,
                "gtfsId": gtfs_id_collection,
                "vehicle_mode": vehicle_mode_collection,
                "related_routes": related_routes_collection,
                "date": date_collection,
                "time_start": time_start_collection,
                "time_end": time_end_collection,
                "walk_speed_[km/h]": walk_speed_collection,
                "max_walking_time_[min]": max_walking_time_collection,
                "catchment_area_[m]": catchment_area_collection
            }
        )
        return df

    def export_stops_as_geopackage(self, stop_collection, analysis_parameters:ReferencePoint):
        lat_collection = []
        lon_collection = []
        if analysis_parameters is not None:
            #this adds the referencePoint to the Pointlayer
            lat_collection.append(analysis_parameters.lat)
            lon_collection.append(analysis_parameters.lon)
        else:
            lat_collection.append(None)
            lon_collection.append(None)
        for stop in stop_collection:
            lat_collection.append(stop.lat)
            lon_collection.append(stop.lon)
        station_attributes = self.create_dataframe_for_stop_objects(stop_collection, analysis_parameters=analysis_parameters)
        gdf = gpd.GeoDataFrame(station_attributes,
                               geometry=gpd.points_from_xy(lon_collection, lat_collection), crs="EPSG:4326")
        gdf.to_file(analysis_parameters.filepath, driver='GPKG', layer=analysis_parameters.layer_name)
        layer = QgsVectorLayer(analysis_parameters.filepath, analysis_parameters.layer_name, "ogr")
        QgsProject.instance().addMapLayer(layer)

    def export_stations_as_geopackage(self, station_collection, analysis_parameters:ReferencePoint):
        mean_lat_collection = []
        mean_lon_collection = []
        if analysis_parameters is not None:
            #this adds the referencePoint to the Pointlayer
            mean_lat_collection.append(analysis_parameters.lat)
            mean_lon_collection.append(analysis_parameters.lon)
        else:
            mean_lat_collection.append(None)
            mean_lon_collection.append(None)
        for station in station_collection:
            mean_lat_collection.append(station.mean_lat)
            mean_lon_collection.append(station.mean_lon)
        station_attributes = self.create_dataframe_with_station_attributes(station_collection, analysis_parameters=analysis_parameters)
        gdf = gpd.GeoDataFrame(station_attributes,
                               geometry=gpd.points_from_xy(mean_lon_collection, mean_lat_collection), crs="EPSG:4326")
        # permanent layer
        gdf.to_file(analysis_parameters.filepath, driver='GPKG', layer=analysis_parameters.layer_name)
        layer = QgsVectorLayer(analysis_parameters.filepath, analysis_parameters.layer_name, "ogr")

        QgsProject.instance().addMapLayer(layer)

    """
    Symbology Methods
    """
    def set_symbol_point_or_polygon(self, layer):
        if layer.geometryType() == QgsWkbTypes.PolygonGeometry:
            symbol = QgsFillSymbol.createSimple({'color': '#9b9b9b', 'outline_style': 'no'})
            return symbol
        elif QgsSymbol.defaultSymbol(layer.geometryType()) == QgsMarkerSymbol:
            symbol = QgsSymbol.defaultSymbol(layer.geometryType())
            return symbol
        else:

            return QgsSymbol.defaultSymbol(layer.geometryType())

    def add_rendererRange_for_particular_points(self, layer, range_list:list):
        # not reachable stations
        label = "no itinerary found"
        lower_limit = -1
        upper_limit = -1
        symbol = self.set_symbol_point_or_polygon(layer)
        grey = "#9b9b9b"
        symbol.setColor(QtGui.QColor(grey))
        range = QgsRendererRange(lower_limit, upper_limit, symbol, label)
        range_list.append(range)

        # start/end point
        label = "start"
        lower_limit = -2
        upper_limit = -2
        if QgsSymbol.defaultSymbol(layer.geometryType()) == QgsMarkerSymbol:
            symbol = QgsMarkerSymbol.createSimple({'name': 'square', "size": 4})
        else:
            symbol = self.set_symbol_point_or_polygon(layer)
        pink = "#fe019a"
        symbol.setColor(QtGui.QColor(pink))
        range = QgsRendererRange(lower_limit, upper_limit, symbol, label)
        range_list.insert(0,range)

        range_list.reverse()
        return range_list

    def get_travel_time_rendererRange(self, layer):
        limits = [0, 5.001, 10.001, 15.001, 20.001, 30.001, 40.001, 50.001, 60.001, 75.001, 90.001, 1000]
        label_limits = [0, 5, 10, 15, 20, 30, 40, 50, 60, 75, 90, 1000]
        colour_gradient = self.get_colors("Turbo", 11)

        range_list = []
        for index, color in enumerate(colour_gradient):
            if index == 10:
                label = ">90 min"
            else:
                label = f"{label_limits[index]}< to ≤{label_limits[index+1]} min"
            lower_limit = limits[index]
            upper_limit = limits[index + 1]
            symbol = self.set_symbol_point_or_polygon(layer)#QgsSymbol.defaultSymbol(layer.geometryType())
            symbol.setColor(QtGui.QColor(color))
            sector = QgsRendererRange(lower_limit, upper_limit, symbol, label)
            range_list.append(sector)

        return range_list
    def get_travel_time_ratio_rendererRange(self, layer):
        limits = [0.0, 1.0, 1.5, 2.1, 2.8, 3.8, 100.0]
        colour_gradient = self.get_colors("Spectral", 6) #RdYlGn
        colour_gradient.reverse()

        range_list = []
        for index, color in enumerate(colour_gradient):
            if index == 5:
                label = "≥3.8"
            else:
                label = f"{limits[index]}≤ to <{limits[index+1]}"
            lower_limit = limits[index]
            upper_limit = limits[index+1]
            symbol = self.set_symbol_point_or_polygon(layer)
            symbol.setColor(QtGui.QColor(color))
            element = QgsRendererRange(lower_limit, upper_limit, symbol, label)
            range_list.append(element)
        return range_list


    def get_frequency_rendererRange(self, layer):
        limits = [0, 5.001, 10.001, 20.001, 40.001, 60.001, 120.001, 1441]  # 1440min = 1trip per day
        label_limits = [0,5,10,20,40,60,120,1441]
        colour_gradient = self.get_colors("Viridis", 7)#PuRd
        colour_gradient.reverse()
        #limits = [0, 5, 8, 10, 15, 20, 30, 40, 60, 120, 1440]

        range_list = []
        for index, color in enumerate(colour_gradient):
            if index == 6:
                label = ">120 min"
            else:
                label = f"{label_limits[index]}< to ≤{label_limits[index+1]} min frequency" # f"{limits[index+1]} min frequency"
            lower_limit = limits[index]
            upper_limit = limits[index + 1]
            symbol = self.set_symbol_point_or_polygon(layer)
            symbol.setColor(QtGui.QColor(color))
            range = QgsRendererRange(lower_limit, upper_limit, symbol, label)
            range_list.append(range)
        return range_list

    def get_walk_time_rendererRange(self, layer):
        limits = [0, 5.001, 10.001, 15.001, 20.001, 120]
        label_limits = [0,5,10,15,20,120]
        colour_gradient = self.get_colors("Purples", 5)
        colour_gradient.reverse()

        range_list = []
        for index, color in enumerate(colour_gradient):
            if index == 4:
                label = ">21 min walktime"
            else:
                label = f"{label_limits[index]}< to ≤{label_limits[index+1]} min walktime"
            lower_limit = limits[index]
            upper_limit = limits[index+1]
            symbol = self.set_symbol_point_or_polygon(layer)
            symbol.setColor(QtGui.QColor(color))
            range = QgsRendererRange(lower_limit, upper_limit, symbol, label)
            range_list.append(range)
        return range_list


    def get_walk_distance_rendererRange(self, layer):
        limits = [0, 100.001, 200.001, 300.001, 500.001, 750.001, 1000.001, 5000.001]  # 1440min = 1trip per day
        label_limits = [0, 100, 200, 300, 500, 750, 1000, 5000]
        colour_gradient = self.get_colors("Purples", 7)
        colour_gradient.reverse()

        range_list = []
        for index, color in enumerate(colour_gradient):
            if index == 6:
                label = ">1000 m walkdistance"
            else:
                label = f"{label_limits[index]}< to ≤{label_limits[index + 1]} m walkdistance"  # f"{limits[index+1]} min frequency"
            lower_limit = limits[index]  # inclusive
            upper_limit = limits[index + 1]  # exclusive
            symbol = self.set_symbol_point_or_polygon(layer)
            symbol.setColor(QtGui.QColor(color))
            range = QgsRendererRange(lower_limit, upper_limit, symbol, label)
            range_list.append(range)
        return range_list

    def get_transfer_rendererRange(self, layer):
        limits = [0.0, 1.0, 2.0, 3.0, 100.0]
        colour_gradient = self.get_colors("Oranges", 4)
        colour_gradient.reverse()

        range_list = []
        for index, color in enumerate(colour_gradient):
            if index == 3:
                label = "≥3"
            else:
                label = f"{index} transfers"
            lower_limit = limits[index] #inclusive
            upper_limit = limits[index + 1] #exclusive
            symbol = self.set_symbol_point_or_polygon(layer)
            symbol.setColor(QtGui.QColor(color))
            range = QgsRendererRange(lower_limit, upper_limit, symbol, label)
            range_list.append(range)
        return range_list

    """
    ChatGPT Code:
    Request: In der QGIS Python API gibt es die Klasse QgsColorRamp. Ich möchte die ColorRamp Turbo von Qgis erstellen und auf die Hexcodes dieser ColorRamp zugreifen
    """

    def get_hex_from_color(self, color: QColor) -> str:
        """Convert a QColor to a hex string."""
        return color.name()

    def get_colors(self, ramp_name, num_colors):
        # Load the QGIS style and get the Turbo Color Ramp
        style = QgsStyle.defaultStyle()
        color_ramp = style.colorRamp(ramp_name)

        if not color_ramp:
            self.iface.messageBar().pushMessage(f"Error: Color ramp '{ramp_name}' not found.")
            return
        else:
            # Create a list to store the hex codes
            hex_colors = []

            for i in range(num_colors):
                # Determine the position on the color ramp (from 0 to 1)
                position = i / (num_colors - 1)

                # Get the color at this position
                color = color_ramp.color(position)

                # Convert the color to a hex code and add it to the list
                hex_colors.append(self.get_hex_from_color(color))
            return hex_colors

    """Request: Ich greife mit folgendem code auf die QGis Layer zu. Leider werden layer, die in Gruppen liegen nicht angezigt. Wie sieht der Code aus, damit auch layer in Gruppen gefunden werden:
    Request2: im folgenden code werden alle Layer in eine Combobox geladen. Ich möchte jetzt nur point layer in eine Combobox laden. Was muss ich ändern? Kommentare im Code bitte auf Englisch:"""

    def get_layers(self, root, layer_type="all"):
        layers = []
        nodes = root.children()

        for node in nodes:
            if isinstance(node, QgsLayerTreeLayer):
                layer = node.layer()
                # Check if the layer is a vector layer before filtering for geometry type
                if isinstance(layer, QgsVectorLayer):
                    # Filter layers based on the specified layer_type
                    if layer_type == "all" or \
                            (layer_type == "points" and layer.geometryType() == QgsWkbTypes.PointGeometry) or \
                            (layer_type == "polygons" and layer.geometryType() == QgsWkbTypes.PolygonGeometry):
                        layers.append(layer)
            elif isinstance(node, QgsLayerTreeGroup):
                # If the node is a group, recursively get layers from the group
                layers.extend(self.get_layers(node, layer_type))

        return layers

    def load_layers_in_combobox(self):
        # Fetch the currently loaded layers including those in groups
        root = QgsProject.instance().layerTreeRoot()
        all_layers = self.get_layers(root, "all")
        # Clear the contents of the comboBox from previous runs
        self.dlg.cb_layer_symbology.clear()
        # Populate the comboBox with names of the specified type of layers
        self.dlg.cb_layer_symbology.addItems([layer.name() for layer in all_layers])

    """
    Request: temporary layer"""
    def determine_qvariant_type(self, dtype):
        if pd.api.types.is_integer_dtype(dtype):
            return QVariant.Int
        elif pd.api.types.is_float_dtype(dtype):
            return QVariant.Double
        elif pd.api.types.is_object_dtype(dtype):
            return QVariant.String
        elif pd.api.types.is_bool_dtype(dtype):
            return QVariant.Bool
        elif pd.api.types.is_datetime64_any_dtype(dtype):
            return QVariant.DateTime
        elif pd.api.types.is_timedelta64_dtype(dtype):
            return QVariant.Time
        else:
            return QVariant.String
    """   
    end of ChatGPT code
    """

    def setText_distance_field(self):
        if not self.get_walk_speed():
            return
        else:
            walk_speed = self.get_walk_speed()
        if not self.get_max_walk_time():
            return
        else:
            walk_time = self.get_max_walk_time()

        walk_distance = walk_speed * walk_time
        self.dlg.le_personalized_tempo.setText(f"{round(walk_speed * 3.6, 1)}")
        self.dlg.le_max_walking_time.setText(f"{round(walk_time / 60)}")
        self.dlg.le_max_walk_distance.setText(f"{round(walk_distance)}")

    def check_grizzly_server_is_running(self):
        if self.dlg.rb_otp_manually_started_8080.isChecked():
            url = "http://localhost:8080/"
        elif self.dlg.rb_otp_manually_started_changed_port.isChecked():
            if self.dlg.le_port_number.text() != "":
                port_number = self.dlg.le_port_number.text()
                url = f"http://localhost:{port_number}/"
            else:
                self.iface.messageBar().pushMessage("You have to enter a port number for OTP")
        else:
            self.iface.messageBar().pushMessage("How to start OTP. There has to be at least one option choosen")
        try:
            # Get Url
            get = requests.get(url)
            # if the request succeeds
            if get.status_code == 200:
                self.dlg.l_otp_connection_test.setText("connection to OTP server")
                return True
            else:
                return False
            # Exception
        except requests.exceptions.RequestException as e:
            self.dlg.l_otp_connection_test.setText("no connection to OTP server")
            self.iface.messageBar().pushMessage("Grizzly server/ OpenTripPlanner is not running or runs on an different port")
            return False

    def select_output_file(self, current_line_edit): #
        # Get the directory of the currently opened QGIS project
        project_path = QgsProject.instance().fileName()
        if project_path:
            default_dir = os.path.dirname(project_path)
        else:
            default_dir = os.path.expanduser("~")  # Fallback to home directory if no project is open

        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Filepath ", "", '*.gpkg')
        if current_line_edit == "itineraries":
            self.dlg.le_filepath_itineraries.setText(filename)
            layer_name = os.path.splitext(os.path.basename(filename))[0]
            self.dlg.le_layer_name.setText(layer_name)

    def stops_and_departure_times_from_otp_to_gpkg(self):
        if self.check_grizzly_server_is_running():
            start_time = datetime.now()
            analysis_parameters = self.create_request_object()
            all_stops_as_dict = self.query_all_stops_incl_departure_times(analysis_parameters=analysis_parameters)
            all_stops, all_routes = self.create_stop_and_route_objects(all_stops_as_dict, analysis_parameters)
            self.export_stops_as_geopackage(all_stops, analysis_parameters=analysis_parameters)
            print('Duration: {}'.format(datetime.now() - start_time))
        else:
            return

    def stations_from_otp_to_gpkg(self):
        if self.check_grizzly_server_is_running():
            start_time = datetime.now()
            analysis_parameters = self.create_request_object()
            stops_as_dict = self.query_all_stops_incl_departure_times(analysis_parameters=analysis_parameters)
            all_stops, all_routes = self.create_stop_and_route_objects(stops_as_dict, analysis_parameters)
            all_stations = self.create_stations(all_stops)
            self.export_stations_as_geopackage(all_stations, analysis_parameters=analysis_parameters)
            end_time = datetime.now()
            print('Duration: {}'.format(end_time - start_time))
        else:
            return

    def reachability_analysis_to_geopackage(self, start_or_end_station):
        #runtime: several minutes
        start_time = datetime.now()
        if self.check_grizzly_server_is_running():
            #create Station objects
            analysis_parameters = self.create_request_object()
            stops_as_dict = self.query_all_stops_incl_departure_times(analysis_parameters=analysis_parameters)
            all_stops, all_routes = self.create_stop_and_route_objects(stops_as_dict, analysis_parameters)
            all_stations = self.create_stations(all_stops)
            # check if the referencePoint is the start or the end of the itinerary
            if start_or_end_station == "start":
                # During the developement you can reduce the stations to be calculated
                # In this way you can check if the code work with shorter runtime
                # If you choose only one station you have to make a list again, so that some methods can iterate over
                    # the element
                selected_stations = all_stations #[400:403] # delete number
                if not isinstance(selected_stations, list):
                    selected_stations = [selected_stations]
                # query and filter the Itineraries
                for item_index, station in enumerate(selected_stations):
                    queried_itineraries = station.query_transit_itineraries(analysis_parameters, "start",
                                                      url=self.get_request_url())
                    station.create_transit_itinerary_objects(queried_itineraries)
                    station.filter_itineraries_with_permissible_catchment_area("start",
                                                                               analysis_parameters.catchment_area)
                    for itinerary in station.itineraries_with_permissible_catchment_area:
                        analysis_parameters.add_first_possible_stop(itinerary.first_stop)
                        # itinerary.frequency = itinerary.calculate_frequency(route_collection) #alternative frequency calculation
                    station.filter_fastest_itinerary()
                    station.set_indicators_of_itinerary(analysis_parameters, "start", url=self.get_request_url())
                # because of the declaration of start_station, there are empty strings in possible_start_station
                analysis_parameters.remove_empty_entries_in_first_possible_stops()
                self.export_stations_as_geopackage(selected_stations, analysis_parameters=analysis_parameters)

            elif start_or_end_station == "end":
                #end = {"lat": lat, "lon":  lon}
                self.not_implemented_yet()

        else:
            return
        end_time = datetime.now()
        print('Duration: {}'.format(end_time - start_time))

    def set_default_symbology(self):
        root = QgsProject.instance().layerTreeRoot()
        layer_collection = self.get_layers(root, "all")
        layer_index = self.dlg.cb_layer_symbology.currentIndex()
        layer = layer_collection[layer_index]
        symbology_theme = self.dlg.cb_symbology_theme.currentIndex()
        if symbology_theme == 0:
            target_field = "travel_time[min]"
            range_list = self.get_travel_time_rendererRange(layer)
        elif symbology_theme == 1:
            target_field = "travel_time_ratio"
            range_list = self.get_travel_time_ratio_rendererRange(layer)
        elif symbology_theme == 2:
            target_field = "frequency_[min]"
            range_list = self.get_frequency_rendererRange(layer)
        elif symbology_theme == 3:
            target_field = "walk_time_[min]"
            range_list = self.get_walk_time_rendererRange(layer)
        elif symbology_theme == 4:
            target_field = "walk_distance_[m]"
            range_list = self.get_walk_distance_rendererRange(layer)
        elif symbology_theme == 5:
            target_field = "number_of_transfers"
            range_list = self.get_transfer_rendererRange(layer)

        range_list = self.add_rendererRange_for_particular_points(layer, range_list)

        layer_renderer = QgsGraduatedSymbolRenderer(target_field, range_list)
        classification_method = QgsApplication.classificationMethodRegistry().method("EqualInterval")
        layer_renderer.setClassificationMethod(classification_method)
        layer_renderer.setClassAttribute(target_field)

        layer.setRenderer(layer_renderer)
        layer.renderer().setUsingSymbolLevels(True)
        layer.triggerRepaint()

    def not_implemented_yet(self):
        self.iface.messageBar().pushMessage("This function is optional and not implemented yet")

    """
    Run is the method, which connects GUI elements to methods and is running, while the plugin is open
    """
    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = TransitReachabilityAnalyserDialog()
            #preparation buttons
            self.dlg.pb_calculate_walk_distance.clicked.connect(self.setText_distance_field)
            self.dlg.pb_start_check_OTP.clicked.connect(self.check_grizzly_server_is_running)
            self.dlg.pb_open_explorer_itineraries.clicked.connect(lambda: self.select_output_file("itineraries"))
            #calculation buttons
            self.dlg.pb_get_stops_from_otp.clicked.connect(self.stops_and_departure_times_from_otp_to_gpkg)
            self.dlg.pb_get_stations_from_otp.clicked.connect(self.stations_from_otp_to_gpkg)
            self.dlg.pb_start_to_all_stations.clicked.connect(lambda: self.reachability_analysis_to_geopackage("start"))
            #symbology buttons
            self.dlg.pb_reload_layer_cb.clicked.connect(self.load_layers_in_combobox)
            self.dlg.pb_set_symbology.clicked.connect(self.set_default_symbology)

        self.load_layers_in_combobox()
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
