# -*- coding: utf-8 -*-

"""
/***************************************************************************
 TrailElevationStats
                                 A QGIS plugin
 Calculate information such as cumulative elevation gain from a DEM and a line layer.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-07-16
        copyright            : (C) 2019 by Gustave Coste
        email                : gustavecoste@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Gustave Coste'
__date__ = '2019-07-16'
__copyright__ = '(C) 2019 by Gustave Coste'

# TODO: Plot elevation profile
# TODO: Add plugin icon

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (QgsFeature,
                       QgsRaster,
                       QgsField,
                       QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterMatrix,
                       QgsProcessingParameterDistance,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterRasterLayer,
                       QgsWkbTypes)


class TrailElevationStatsAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT_POINT = 'OUTPUT_POINT'
    OUTPUT_LINE = 'OUTPUT_LINE'
    ELEVATION = 'ELEVATION'
    TRAIL = 'TRAIL'
    PRECISION = 'PRECISION'
    DENIV_POS_COEFF = 'DENIV_POS_COEFF'
    DENIV_NEG_COEFF = 'DENIV_NEG_COEFF'
    DENIV_COEFF_MATRIX = 'DENIV_COEFF_MATRIX'
    DENIV_COEFF_SOURCE = 'DENIV_COEFF_SOURCE'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.ELEVATION,
                self.tr('Elevation')
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.TRAIL,
                self.tr('Trails'),
                [QgsProcessing.TypeVectorLine]
            )
        )

        self.addParameter(
            QgsProcessingParameterDistance(
                self.PRECISION,
                self.tr('Precision (should be at least twice the raster pixel size)'),
                minValue=0,
                defaultValue=100,
                parentParameterName=self.TRAIL
            )
        )

        self.addParameter(
            QgsProcessingParameterEnum(
                self.DENIV_COEFF_SOURCE,
                self.tr('Source of the elevation gain coefficients'),
                options=['Positive and negative elevation gain coefficients',
                         'Elevation gain coefficient table'],
                defaultValue='Positive and negative elevation gain coefficients'
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.DENIV_POS_COEFF,
                self.tr('Positive elevation gain coefficient (m-1)'),
                defaultValue=100,
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.DENIV_NEG_COEFF,
                self.tr('Negative elevation gain coefficient (m-1)'),
                defaultValue=333,
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterMatrix(
                self.DENIV_COEFF_MATRIX,
                self.tr('Elevation gain coefficient per slope class'),
                headers=['Min slope (%)', 'Max slope (%)', 'Elevation gain coefficient (m-1)'],
                defaultValue=[-100, -15, 200,
                              -15, -5, 333,
                              -5, 0, -500,
                              0, 200, 100],
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_POINT,
                self.tr('Trail points elevation statistics')
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_LINE,
                self.tr('Trail elevation statistics')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):

        trail_layer = self.parameterAsSource(parameters, self.TRAIL, context)
        elevation_layer = self.parameterAsRasterLayer(parameters, self.ELEVATION, context)
        precision = self.parameterAsDouble(parameters, self.PRECISION, context)
        deniv_pos_coeff = self.parameterAsDouble(parameters, self.DENIV_POS_COEFF, context)
        deniv_neg_coeff = self.parameterAsDouble(parameters, self.DENIV_NEG_COEFF, context)
        deniv_coeff_matrix = self.parameterAsMatrix(parameters, self.DENIV_COEFF_MATRIX, context)
        deniv_coeff_source = self.parameterAsEnum(parameters, self.DENIV_COEFF_SOURCE, context)

        # Rebuild the coeff matrix
        deniv_coeff_matrix = [[deniv_coeff_matrix[i], deniv_coeff_matrix[i + 1], deniv_coeff_matrix[i + 2]]
                              for i in range(0, len(deniv_coeff_matrix), 3)]

        # Input data verifications
        if trail_layer.sourceCrs() != elevation_layer.crs():
            feedback.reportError("The elevation and trail layers must have the same CRS.", fatalError=True)
            raise Exception

        if precision < (2 * elevation_layer.rasterUnitsPerPixelX()) or \
                precision < (2 * elevation_layer.rasterUnitsPerPixelY()):
            feedback.reportError("The precision should be at least twice the raster pixel size.")

        if deniv_coeff_source == 1:
            for i, slope_class_1 in enumerate(deniv_coeff_matrix):
                for j, slope_class_2 in enumerate(deniv_coeff_matrix):
                    if i == j:
                        continue
                    elif (slope_class_2[0] < slope_class_1[0] < slope_class_2[1]) or \
                            (slope_class_2[0] < slope_class_1[1] < slope_class_2[1]):
                        feedback.reportError("Elevation gain coefficient classes are overlapping.", fatalError=True)
                        raise Exception

        point_output_fields = trail_layer.fields()
        distance_field = QgsField(name='distance', type=QVariant.Double)
        elevation_field = QgsField(name='elevation', type=QVariant.Double)
        slope_field = QgsField(name='slope', type=QVariant.Double)
        deniv_pos_field = QgsField(name='deniv_pos', type=QVariant.Double)
        deniv_neg_field = QgsField(name='deniv_neg', type=QVariant.Double)
        kme_field = QgsField(name='kme', type=QVariant.Double)
        point_output_fields.append(distance_field)
        point_output_fields.append(elevation_field)
        point_output_fields.append(slope_field)
        point_output_fields.append(deniv_pos_field)
        point_output_fields.append(deniv_neg_field)
        point_output_fields.append(kme_field)
        (point_sink, point_dest_id) = self.parameterAsSink(parameters, self.OUTPUT_POINT, context, point_output_fields,
                                                           QgsWkbTypes.Point, trail_layer.sourceCrs())

        max_elevation_field = QgsField(name='max_elevation', type=QVariant.Double)
        min_elevation_field = QgsField(name='min_elevation', type=QVariant.Double)
        line_output_fields = trail_layer.fields()
        line_output_fields.append(distance_field)
        line_output_fields.append(max_elevation_field)
        line_output_fields.append(min_elevation_field)
        line_output_fields.append(deniv_pos_field)
        line_output_fields.append(deniv_neg_field)
        line_output_fields.append(kme_field)
        (line_sink, line_dest_id) = self.parameterAsSink(parameters, self.OUTPUT_LINE, context, line_output_fields,
                                                         QgsWkbTypes.LineString, trail_layer.sourceCrs())

        # Looping over trail layer features
        line_features = trail_layer.getFeatures()
        for line_feature in line_features:

            distance = deniv_pos = deniv_neg = previous_point_elevation = kme = max_elevation = 0
            min_elevation = 9999999999
            # Creating points along the lines separated by the given precision
            while distance <= line_feature.geometry().length():
                # Stop the algorithm if cancel button has been clicked
                if feedback.isCanceled():
                    break

                new_point_feature = QgsFeature()
                new_point_feature.setFields = point_output_fields
                new_point_feature.setGeometry(line_feature.geometry().interpolate(distance))

                # Computing elevation info
                # Getting elevation at this point
                ident = elevation_layer.dataProvider().identify(new_point_feature.geometry().asPoint(),
                                                                QgsRaster.IdentifyFormatValue)
                elevation = ident.results()[1] if ident.isValid() else None

                # Getting elevation gain
                if distance == 0:
                    elevation_gain = 0
                else:
                    elevation_gain = elevation - previous_point_elevation
                if elevation_gain > 0:
                    deniv_pos += elevation_gain
                else:
                    deniv_neg += -elevation_gain

                # Calculating slope
                slope = 100 * elevation_gain / precision

                # Calculating kme
                if deniv_coeff_source == 0:  # Fixed positive and negative coefficients
                    kme = (distance / 1000) + (deniv_pos / deniv_pos_coeff) + (deniv_neg / deniv_neg_coeff)
                elif deniv_coeff_source == 1:  # Slope defined coefficients
                    coeff = None

                    for slope_class in deniv_coeff_matrix:
                        if float(slope_class[0]) <= slope < float(slope_class[1]):
                            coeff = float(slope_class[2])

                    if coeff is None:
                        feedback.reportError(f"No elevation gain coefficient for slope {slope}.")
                        coeff = 0

                    kme += (precision / 1000)
                    if coeff != 0:
                        kme += abs(elevation_gain) / coeff

                new_point_feature.setAttributes(
                    line_feature.attributes() + [distance, elevation, slope, deniv_pos, deniv_neg, kme])

                point_sink.addFeature(new_point_feature)

                distance += precision
                previous_point_elevation = elevation
                max_elevation = max(max_elevation, elevation)
                min_elevation = min(min_elevation, elevation)

            # Creating a copy of the line feature with additional metadata
            new_line_feature = QgsFeature()
            new_line_feature.setFields = line_output_fields
            new_line_feature.setGeometry(line_feature.geometry())
            new_line_feature.setAttributes(
                line_feature.attributes() + [line_feature.geometry().length(), max_elevation, min_elevation, deniv_pos,
                                             deniv_neg, kme])
            line_sink.addFeature(new_line_feature)

        # Return the results of the algorithm.
        return {self.OUTPUT_POINT: point_dest_id, self.OUTPUT_LINE: line_dest_id}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Calculate trail elevation statistics'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Trail elevation statistics'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return TrailElevationStatsAlgorithm()
