# -*- coding: utf-8 -*-

"""
/***************************************************************************
 TopoDrainPlugin
                                 A QGIS plugin
 A QGIS plugin for planning surface drainage water management. It automates the extraction of main valleys and ridges, and supports various other water retention planning methods, such as Keyline Design (keypoints, keylines), pond siting and various other sponge-scape measures. The algorithms are mainly based on WhiteboxTools. The plugin will be developed further on a regular basis.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-08-06
        copyright            : (C) 2025 by Timo Wicki
        email                : wickitimo@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Timo Wicki'
__date__ = '2025-08-06'
__copyright__ = '(C) 2025 by Timo Wicki'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os
import sys
import inspect
import importlib

from qgis.core import QgsProcessingAlgorithm, QgsApplication, QgsProcessingUtils, QgsCoordinateReferenceSystem, QgsProject, QgsRasterLayer, QgsVectorLayer
from qgis.PyQt.QtWidgets import QMessageBox
from processing.core.ProcessingConfig import ProcessingConfig
from .topo_drain_provider import TopoDrainProvider
from topo_drain.core.topo_drain_core import TopoDrainCore
from .utils import get_crs_from_project

cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]

if cmd_folder not in sys.path:
    sys.path.insert(0, cmd_folder)

class TopoDrainPlugin(object):

    def __init__(self):
        self.provider = None
        self.core = None  # Will hold the TopoDrainCore instance

    def get_whiteboxtools_executable_path(self, silent=False):
        """
        Get WhiteboxTools executable path from QGIS settings.
        
        Args:
            silent (bool): If True, don't show warning messages
            
        Returns:
            str or None: Path to WhiteboxTools executable
        """
        # The key used by the WhiteboxTools plugin for the executable path
        WBT_EXECUTABLE_KEY = "WBT_EXECUTABLE"
        exe_path = ProcessingConfig.getSetting(WBT_EXECUTABLE_KEY)
        print(f"WhiteboxTools executable path from QGIS Settings: {exe_path}")
        
        if not exe_path and not silent:
            msg = (
                "WhiteboxTools executable path could not be determined.\n\n"
                "This is normal during QGIS startup if the WhiteboxTools plugin hasn't loaded yet.\n\n"
                "The TopoDrain plugin will load successfully. WhiteboxTools will be configured automatically when first used."
            )
            QMessageBox.information(None, "TopoDrain Plugin - Initialization", msg)

        return exe_path
    
    def check_python_dependencies(self):
        # Read python dependencies from requirements.txt
        dep_file = os.path.join(cmd_folder, "requirements.txt")
        REQUIRED_PACKAGES = []
        if os.path.exists(dep_file):
            with open(dep_file, "r") as f:
                for line in f:
                    pkg = line.strip()
                    if pkg and not pkg.startswith("#"):
                        # Only take the package name (strip version specifiers)
                        REQUIRED_PACKAGES.append(pkg.split()[0].split("=")[0])
            missing = []
            for pkg in REQUIRED_PACKAGES:
                if importlib.util.find_spec(pkg) is None:
                    missing.append(pkg)
            if missing:
                msg = (
                    "The following Python packages are required but not installed in your QGIS Python environment:\n\n"
                    + ", ".join(missing) +
                    "\n\nPlease install them using your QGIS Python environment before using this plugin."
                )
                QMessageBox.critical(None, "TopoDrain Plugin - Missing Dependencies", msg)
                return False
        return True
    
    def initProcessing(self):
        """Init Processing provider for QGIS >= 3.8."""
        temp_dir = QgsProcessingUtils.tempFolder()
        working_dir = QgsProcessingUtils.tempFolder()
        
        # Try to get WhiteboxTools path silently during initialization
        # This might fail if WhiteboxTools plugin hasn't loaded yet, which is normal
        whitebox_executable_path = self.get_whiteboxtools_executable_path(silent=True)
        
        # Check if WhiteboxTools path was found
        if whitebox_executable_path is None:
            # Use None for whitebox_dir if executable path is not available
            # This will be updated later when WhiteboxTools becomes available
            whitebox_dir = None
            print("WhiteboxTools not available during initialization - will be configured when first needed")
        else:
            whitebox_dir = os.path.dirname(whitebox_executable_path)
            print(f"WhiteboxTools directory found during initialization: {whitebox_dir}")
        
        project_crs = get_crs_from_project()
        # Create the TopoDrainCore instance - it will handle None whitebox_directory gracefully
        self.core = TopoDrainCore(whitebox_directory=whitebox_dir, crs=project_crs, temp_directory=temp_dir, working_directory=working_dir)
        
        # Only check dependencies after core is created
        self.check_python_dependencies()
        
        # Pass the core instance to the provider
        self.provider = TopoDrainProvider(core=self.core, plugin=self)
        QgsApplication.processingRegistry().addProvider(self.provider)

    def initGui(self):
        self.initProcessing()
        
    def ensure_whiteboxtools_configured(self):
        """
        Ensure WhiteboxTools is configured. This method can be called by algorithms
        before they execute to make sure WhiteboxTools is available.
        
        Returns:
            bool: True if WhiteboxTools is configured, False otherwise
        """
        if self.core is None:
            return False
            
        # Check if WhiteboxTools is already configured
        if self.core.wbt is not None and self.core.whitebox_directory is not None:
            return True
        
        # Try to get WhiteboxTools executable path and update core
        exe_path = self.get_whiteboxtools_executable_path(silent=True)
        if exe_path:
            whitebox_dir = os.path.dirname(exe_path)
            print(f"Updating TopoDrainCore with WhiteboxTools directory: {whitebox_dir}")
            # Update the core's whitebox directory
            self.core.whitebox_directory = whitebox_dir
            # Re-initialize WhiteboxTools
            try:
                self.core.wbt = self.core._init_whitebox_tools(whitebox_dir)
                return True
            except Exception as e:
                print(f"Failed to initialize WhiteboxTools: {e}")
        
        # DO NOT show GUI dialogs during algorithm execution - this causes crashes
        # Simply return False to indicate WhiteboxTools is not configured
        return False

    def unload(self):
        QgsApplication.processingRegistry().removeProvider(self.provider)
