# -*- coding: utf-8 -*-
# -----------------------------------------------------------------------------
# Name: topo_drain_core.py
#
# Purpose: Script with python functions of topo drain qgis plugin
#
# -----------------------------------------------------------------------------
import os
import sys
import importlib.util
from typing import Union
import warnings
import uuid
import numpy as np
import geopandas as gpd
from shapely.geometry import LineString, MultiLineString, Point, Polygon
from shapely.ops import linemerge, nearest_points, substring
from scipy.ndimage import gaussian_filter1d
from scipy.signal import savgol_filter
import re
from osgeo import gdal, gdal_array, ogr

# ---  Class TopoDrainCore ---
class TopoDrainCore:
    def __init__(self, whitebox_directory=None, nodata=None, crs=None, temp_directory=None, working_directory=None):
        print("[TopoDrainCore] Initializing TopoDrainCore...")
        self._thisdir = os.path.dirname(__file__)
        print(f"[TopoDrainCore] Module directory: {self._thisdir}")
        
        # Handle None whitebox_directory gracefully
        if whitebox_directory is None:
            print("[TopoDrainCore] No WhiteboxTools directory provided")
            self.whitebox_directory = None  # Keep as None to trigger lazy loading
        else:
            self.whitebox_directory = whitebox_directory
        print(f"[TopoDrainCore] WhiteboxTools directory: {self.whitebox_directory if self.whitebox_directory else 'Not set'}")
        
        self.nodata = nodata if nodata is not None else -32768
        print(f"[TopoDrainCore] NoData value set to: {self.nodata}")
        self.crs = crs if crs is not None else "EPSG:4326"
        print(f"[TopoDrainCore] crs value set to: {self.crs}")
        self.temp_directory = temp_directory if temp_directory is not None else None
        print(f"[TopoDrainCore] Temp directory set to: {self.temp_directory if self.temp_directory else 'Not set'}")
        self.working_directory = working_directory if working_directory is not None else None
        print(f"[TopoDrainCore] Working directory set to: {self.working_directory if self.working_directory else 'Not set'}")
        
        # Define supported GDAL driver mappings for raster formats (has to be compatible with available raster formats for WhiteboxTools)
        self.gdal_driver_mapping = {
            '.tif': 'GTiff',
            '.tiff': 'GTiff',
            '.hdr': 'EHdr',
            '.asc': 'AAIGrid',
            '.bil': 'EHdr',
            '.gpkg': 'GPKG',
            '.sdat': 'SAGA',
            '.sgrd': 'SAGA',
            '.rdc': 'RDxC',
            '.rst': 'RST'

        }
        print(f"[TopoDrainCore] Configured GDAL driver support for {len(self.gdal_driver_mapping)} raster formats")
        
        # Define supported OGR driver mappings for vector formats
        self.ogr_driver_mapping = {
            '.shp': 'ESRI Shapefile',
            '.gpkg': 'GPKG',
            '.geojson': 'GeoJSON',
            '.json': 'GeoJSON',
            '.gml': 'GML'
        }
        print(f"[TopoDrainCore] Configured OGR driver support for {len(self.ogr_driver_mapping)} vector formats")
        
        # Configure GDAL settings for the entire class
        self._configure_gdal()
        
        # Try to initialize WhiteboxTools, but don't fail if it's not available
        self.wbt = self._init_whitebox_tools(self.whitebox_directory)
        print(f"[TopoDrainCore] WhiteboxTools initialized: {self.wbt is not None}")
        print("[TopoDrainCore] Initialization complete.")

    def _init_whitebox_tools(self, whitebox_directory):
        """
        Initialize WhiteboxTools with graceful fallback for plugin loading order issues.
        Returns None if WhiteboxTools cannot be initialized (will be configured later).
        """
        if whitebox_directory is None:
            print("[TopoDrainCore] WhiteboxTools directory not provided - will be configured when available")
            return None
            
        try:
            # Add WhiteboxTools directory to sys path
            if whitebox_directory not in sys.path:
                sys.path.insert(0, whitebox_directory)
                
            # Try to import from the provided location
            wbt_path = os.path.join(whitebox_directory, "whitebox_tools.py")
            if not os.path.exists(wbt_path):
                print(f"[TopoDrainCore] WhiteboxTools not found at {wbt_path}")
                return None
            
            # Step 1: Create the specification for WhiteboxTools module
            spec = importlib.util.spec_from_file_location("whitebox_tools", wbt_path)
            if spec is None or spec.loader is None:
                print(f"[TopoDrainCore] Could not create spec for WhiteboxTools from {wbt_path}")
                return None
            
            # Step 2: Create empty module from spec
            whitebox_tools_mod = importlib.util.module_from_spec(spec)
            # Step 3: Register module in sys.modules
            sys.modules["whitebox_tools"] = whitebox_tools_mod
            # Step 4: Actually execute the module's code
            spec.loader.exec_module(whitebox_tools_mod)
            # Step 5: Access classes/functions from the loaded module
            WhiteboxTools = whitebox_tools_mod.WhiteboxTools
            # Step 6: Reference WhiteboxTools as wbt
            wbt = WhiteboxTools()
            if self.working_directory:
                wbt.set_working_dir(self.working_directory)
            print(f"[TopoDrainCore] Using WhiteboxTools from directory: {whitebox_directory}")
            return wbt
            
        except Exception as e:
            print(f"[TopoDrainCore] WhiteboxTools initialization failed: {e}")
            print("[TopoDrainCore] WhiteboxTools will be configured when available")
            return None

    ## GDAL/OGR setting functions
    def _configure_gdal(self):
        """
        Configure GDAL/OGR settings for the entire class instance.
        Sets up exception handling and error management for all GDAL/OGR operations.
        """
        try:
            # Enable GDAL/OGR exceptions for better error handling
            # Note: gdal.UseExceptions() enables exceptions for both GDAL and OGR
            gdal.UseExceptions()
            
            # Set quiet error handler to suppress console messages
            # We'll capture errors using GetLastErrorMsg() instead
            gdal.PushErrorHandler('CPLQuietErrorHandler')
            
            print("[TopoDrainCore] GDAL/OGR configured: exceptions enabled, quiet error handler set")
            
        except Exception as e:
            print(f"[TopoDrainCore] Warning: Failed to configure GDAL/OGR settings: {e}")
            print("[TopoDrainCore] GDAL/OGR operations may have less detailed error reporting")

    def __del__(self):
        """
        Cleanup method to restore GDAL error handler when object is destroyed.
        """
        try:
            gdal.PopErrorHandler()  # Restore default error handler
        except:
            pass  # Ignore errors during cleanup

    @staticmethod
    def _get_gdal_error_message():
        """
        Helper method to get the last GDAL/OGR error message with proper formatting.
        Note: GDAL and OGR share the same error system (CPL), so this works for both.
        
        Returns:
            str: Formatted GDAL/OGR error message or empty string if no error
        """
        try:
            error_msg = gdal.GetLastErrorMsg()
            return f" GDAL/OGR Error: {error_msg}" if error_msg else ""
        except:
            return ""

    @staticmethod
    def _check_ogr_error(operation_name: str, error_code):
        """
        Helper method to check OGR operation return codes and raise appropriate errors.
        
        Args:
            operation_name (str): Name of the operation for error messages
            error_code: The return code from OGR operation
            
        Raises:
            RuntimeError: If the operation failed
        """
        if error_code != ogr.OGRERR_NONE:
            error_msg = TopoDrainCore._get_gdal_error_message()
            raise RuntimeError(f"{operation_name} failed.{error_msg} (Error code: {error_code})")

    def _get_gdal_driver_from_path(self, file_path: str) -> str:
        """
        Get appropriate GDAL driver name based on file extension.
        
        Args:
            file_path (str): Path to the output file
            
        Returns:
            str: GDAL driver name corresponding to the file extension
        """
        ext = os.path.splitext(file_path)[1].lower()
        return self.gdal_driver_mapping.get(ext, 'GTiff')  # Default to GTiff if unknown extension

    def _get_ogr_driver_from_path(self, file_path: str) -> str:
        """
        Get appropriate OGR driver name based on file extension.
        
        Args:
            file_path (str): Path to the output file
            
        Returns:
            str: OGR driver name corresponding to the file extension
        """
        ext = os.path.splitext(file_path)[1].lower()
        return self.ogr_driver_mapping.get(ext, 'ESRI Shapefile')  # Default to Shapefile if unknown extension


    ## Setters for class configuration
    def set_temp_directory(self, temp_dir):
        self.temp_directory = temp_dir

    def set_working_directory(self, working_dir):
        self.working_directory = working_dir
        if self.wbt is not None and self.working_directory:
            self.wbt.set_working_dir(self.working_directory)

    def set_nodata_value(self, nodata):
        self.nodata = nodata

    def set_crs(self, crs):
        self.crs = crs

    ## WhiteboxTools helper functions
    def _execute_wbt(self, tool_name, feedback=None, report_progress=True, **kwargs):
        """
        Execute a WhiteboxTools command using the Python API.
        
        Args:
            tool_name (str): Name of the WhiteboxTools command
            feedback (QgsProcessingFeedback, optional): Feedback object for progress reporting
            report_progress (bool): Whether to report progress via setProgress calls. Default True.
            **kwargs: Tool parameters as keyword arguments
            
        Returns:
            int: Return code (0 = success)
            
        Raises:
            RuntimeError: If WhiteboxTools is not properly configured
        """
        # Check if WhiteboxTools is available
        if self.wbt is None:
            error_msg = "WhiteboxTools is not properly configured. Please ensure WhiteboxTools plugin is loaded and configured."
            if feedback:
                feedback.reportError(error_msg)
            raise RuntimeError(error_msg)
        
        # Create callback function for progress reporting and logging
        def callback_func(message):
            # Check for cancellation first - this allows immediate cancellation during WhiteboxTools execution
            if feedback and feedback.isCanceled():
                raise RuntimeError("[WBT] Process cancelled by user during WhiteboxTools execution.")
            
            if feedback:
                # Parse progress percentage if available and report_progress is True
                if '%' in message and report_progress:
                    try:
                        # Extract progress from messages like "Progress: 45%"
                        parts = message.split('%')
                        if len(parts) > 1:
                            progress_part = parts[0].strip().split()[-1]
                            progress = int(progress_part)
                            feedback.setProgress(progress)
                    except (ValueError, IndexError):
                        pass  # If parsing fails, just ignore progress
                
                # Send all output to feedback (always enabled regardless of report_progress)
                if message.strip():
                    feedback.pushConsoleInfo(message.strip())
            else:
                # Print to console when no feedback available
                if message.strip():
                    print(f"[WBT] {message.strip()}")
        
        # Get the tool method from WhiteboxTools object
        tool_method = getattr(self.wbt, tool_name, None)
        if tool_method is not None:
            # Use the convenience method if available (cleaner and more reliable)
            try:
                return tool_method(callback=callback_func, **kwargs)
            except Exception as e:
                if feedback:
                    feedback.reportError(f"WhiteboxTools error: {e}")
                raise RuntimeError(f"WhiteboxTools error: {e}")
        else:
            # Fallback to run_tool method for tools without convenience methods
            # Build arguments list for the tool
            args = []
            for param, value in kwargs.items():
                if value is not None:
                    args.append(f"--{param}='{value}'")
            
            try:
                return self.wbt.run_tool(tool_name, args, callback=callback_func)
            except Exception as e:
                if feedback:
                    feedback.reportError(f"WhiteboxTools error: {e}")
                raise RuntimeError(f"WhiteboxTools error: {e}")

    ## Vector geometry functions
    @staticmethod
    def _merge_lines_by_distance(line_geometries):
        """
        Merge multiple LineString geometries into a single LineString by
        connecting them based on closest endpoint distances.
        
        Can handle:
        - List of LineString/MultiLineString objects
        - Single LineString 
        - Single MultiLineString

        Args:
            line_geometries (list|LineString|MultiLineString): 
                Input geometries to merge.

        Returns:
            LineString: Single merged LineString, or None if merging fails.
        """
        print(f"[_merge_lines_by_distance] Start merge lines...")
        # Handle different input types
        if isinstance(line_geometries, LineString):
            return line_geometries
        elif isinstance(line_geometries, MultiLineString):
            line_list = list(line_geometries.geoms)
        elif isinstance(line_geometries, list):
            # Flatten any MultiLineString objects in the list
            line_list = []
            for geom in line_geometries:
                if isinstance(geom, LineString):
                    line_list.append(geom)
                elif isinstance(geom, MultiLineString):
                    line_list.extend(list(geom.geoms))
                else:
                    warnings.warn(f"[_merge_lines_by_distance] Warning: Skipping unsupported geometry type: {type(geom)}")
        else:
            warnings.warn(f"[_merge_lines_by_distance] Error: Unsupported input type: {type(line_geometries)}")
            return None

        
        if not line_list:
            return None
        
        if len(line_list) == 1:
            return line_list[0]
        
        # Convert to list to avoid modifying original
        remaining_lines = line_list.copy()

        remaining_lines.sort(key=lambda line: line.length, reverse=True)
        longest_line = remaining_lines.pop(0)
        merged_coords = list(longest_line.coords)
        iteration = 0
        while remaining_lines:
            iteration += 1
            
            # Get endpoints of current merged line
            start_point = Point(merged_coords[0])
            end_point = Point(merged_coords[-1])
            
            best_line_idx = None
            best_connection = None
            best_distance = float('inf')
            
            # Find the closest connection among all remaining lines
            for idx, line in enumerate(remaining_lines):
                line_start = Point(line.coords[0])
                line_end = Point(line.coords[-1])
                
                # Check all possible connections
                connections = [
                    ('start_to_start', start_point.distance(line_start), 'prepend_reversed'),
                    ('start_to_end', start_point.distance(line_end), 'prepend_normal'),
                    ('end_to_start', end_point.distance(line_start), 'append_normal'),
                    ('end_to_end', end_point.distance(line_end), 'append_reversed')
                ]
                
                for conn_type, distance, action in connections:
                    if distance < best_distance:
                        best_distance = distance
                        best_line_idx = idx
                        best_connection = action
            
            # Connect the best line
            if best_line_idx is not None:
                best_line = remaining_lines.pop(best_line_idx)
                line_coords = list(best_line.coords)
                
                if best_connection == 'prepend_normal':
                    # Add line to start of merged (line_end connects to merged_start)
                    merged_coords = line_coords[:-1] + merged_coords
                elif best_connection == 'prepend_reversed':
                    # Add reversed line to start of merged (line_start connects to merged_start)
                    line_coords.reverse()
                    merged_coords = line_coords[:-1] + merged_coords
                elif best_connection == 'append_normal':
                    # Add line to end of merged (merged_end connects to line_start)
                    merged_coords = merged_coords[:-1] + line_coords
                elif best_connection == 'append_reversed':
                    # Add reversed line to end of merged (merged_end connects to line_end)
                    line_coords.reverse()
                    merged_coords = merged_coords[:-1] + line_coords
                
            else:
                # No valid connection found, break
                break
        
        result = LineString(merged_coords)        

        return result

    @staticmethod
    def _smooth_linestring(geom, sigma: float = 1.0):
        """
        Smooth a LineString or MultiLineString geometry using a Gaussian filter,
        preserving the first and last point of the original geometry.

        Args:
            geom (LineString|MultiLineString): Input geometry to smooth.
            sigma (float): Standard deviation for Gaussian kernel.

        Returns:
            LineString or MultiLineString: Smoothed geometry.
        """
        # Handle MultiLineString by smoothing each part
        if isinstance(geom, MultiLineString):
            smoothed_parts = [TopoDrainCore._smooth_linestring(part, sigma) for part in geom.geoms]
            return MultiLineString(smoothed_parts)

        # Must be a LineString from here on
        if not isinstance(geom, LineString):
            raise TypeError(f"Unsupported geometry type: {type(geom)}")

        coords = np.array(geom.coords)
        if len(coords) < 3 or sigma <= 0:
            # nothing to smooth
            return geom

        # apply gaussian filter separately to x and y
        x_smooth = gaussian_filter1d(coords[:, 0], sigma=sigma)
        y_smooth = gaussian_filter1d(coords[:, 1], sigma=sigma)

        # Ensure first and last points are exactly as in the original
        x_smooth[0], y_smooth[0] = coords[0, 0], coords[0, 1]
        x_smooth[-1], y_smooth[-1] = coords[-1, 0], coords[-1, 1]

        # rebuild as a LineString
        smoothed = LineString(np.column_stack([x_smooth, y_smooth]))
        return smoothed

    @staticmethod
    def _flatten_to_linestrings(gdf: gpd.GeoDataFrame) -> gpd.GeoDataFrame:
        """
        Flatten a GeoDataFrame to individual LineString geometries by:
        1. Converting polygons to boundaries
        2. Extracting individual LineStrings from MultiLineStrings
        3. Ignoring other geometry types
        
        Args:
            gdf (gpd.GeoDataFrame): Input GeoDataFrame to process
            
        Returns:
            gpd.GeoDataFrame: GeoDataFrame with only individual LineString geometries
        """
        # Convert polygons to boundaries if needed
        if gdf.geom_type.isin(["Polygon", "MultiPolygon"]).any():
            g = gdf.copy()
            g["geometry"] = g.boundary
        else:
            g = gdf.copy()
            
        # Flatten to individual LineStrings
        line_geoms = []
        for geom in g.geometry:
            print(f"[FlattenToLinestrings] Processing geometry: {geom.geom_type}")
            if geom.geom_type == "LineString":
                line_geoms.append(geom)
            elif geom.geom_type == "MultiLineString":
                line_geoms.extend(list(geom.geoms))  # Flatten MultiLineString to individual LineStrings
            # Ignore other geometry types
            
        return gpd.GeoDataFrame(geometry=line_geoms, crs=gdf.crs)


    @staticmethod
    def _features_to_single_linestring(features: list[gpd.GeoDataFrame]) -> gpd.GeoDataFrame:
        """
        Convert features to individual LineString geometries by:
        1. Converting polygons to boundaries
        2. Flattening MultiLineStrings to individual LineStrings
        3. Ignoring other geometry types
        
        Args:
            features (list[gpd.GeoDataFrame]): List of GeoDataFrames to process
            
        Returns:
            gpd.GeoDataFrame: Single GeoDataFrame with only LineString geometries from all input features
        """
        all_line_geoms = []
        print(f"[FeaturesToSingleLinestring] Processing {len(features)} GeoDataFrames")
        
        for gdf in features:
            if gdf.empty:
                continue
                
            # Get CRS from first non-empty GeoDataFrame
            crs = gdf.crs
                
            # Use the helper function to flatten to LineStrings
            flattened_gdf = TopoDrainCore._flatten_to_linestrings(gdf)
            
            if not flattened_gdf.empty:
                all_line_geoms.extend(flattened_gdf.geometry.tolist())
                
        if not all_line_geoms:
            # Return empty GeoDataFrame with same structure
            return gpd.GeoDataFrame(geometry=[], crs=crs)
            
        result_gdf = gpd.GeoDataFrame(geometry=all_line_geoms, crs=crs)
        print(f"[FeaturesToSingleLinestring] Processed {len(result_gdf)} LineString geometries")
        return result_gdf

    @staticmethod
    def _snap_line_to_point(line: LineString, snap_point: Point, position: str = None) -> LineString:
        """
        Snap the closest endpoint of a line to a given Point.

        Args:
            line (LineString): Input line geometry.
            snap_point (Point): Point to snap to.
            position (str, optional): "start", "end", or None. 
                - "start" snaps the start of the line to the point, if endpoint is closer reverse line direction first.
                - "end" snaps the end of the line to the point, if startpoint is closer reverse line direction first.
                - None snaps the closer endpoint to the point.

        Returns:
            LineString: Line with endpoint snapped to the given point.
        """
        if not isinstance(line, LineString):
            warnings.warn("Warning: Cannot snap endpoint for MultiLineString geometry")
            return line

        coords = list(line.coords)
        if len(coords) < 2:
            return line

        dist_start = snap_point.distance(Point(coords[0]))
        dist_end = snap_point.distance(Point(coords[-1]))

        # Always snap to the closest endpoint
        if dist_start <= dist_end:
            coords[0] = (snap_point.x, snap_point.y)
            if position == "end":
                # Reverse line direction if snapping to end
                coords.reverse()
        else:
            coords[-1] = (snap_point.x, snap_point.y)
            if position == "start":
                # Reverse line direction if snapping to start
                coords.reverse()

        return LineString(coords)
    
    def _vector_to_mask_raster(
        self,
        features: list[gpd.GeoDataFrame],
        reference_raster_path: str,
        output_path: str = None,
        unique_values: bool = False,
        flatten_lines: bool = True,
        buffer_lines: bool = False
        ) -> Union[str, tuple[str, dict]]:
        """
        Convert one or more GeoDataFrames to a binary raster mask (1 = feature, 0 = background)
        or a multi-value mask with unique values for each geometry and save as raster file.

        Args:
            features (list[GeoDataFrame]): List of GeoDataFrames (polygon or line geometries).
            reference_raster_path (str): Path to a reference raster for shape and transform.
            output_path (str, optional): Path to save the mask raster. If None, generates a temporary path.
            unique_values (bool): If True, assigns unique values (1, 2, 3, ...) to cells for each individual geometry.
                                If False, all features get value 1 (default behavior).
            flatten_lines (bool): If True, flattens MultiLineStrings to individual LineStrings before rasterization.
            buffer_lines (bool): If True, buffers lines by a small distance to ensure no diagonal gaps in rasterization.

        Returns:
            str | tuple[str, dict]: 
                - If unique_values=False: Path to the saved mask raster file
                - If unique_values=True: Tuple of (path, geometry_mapping) where geometry_mapping 
                  is a dict {raster_value: geometry} for each unique geometry
        """
        # Read reference raster information using GDAL
        ref_ds = gdal.Open(reference_raster_path, gdal.GA_ReadOnly)
        if ref_ds is None:
            raise RuntimeError(f"Cannot open reference raster: {reference_raster_path}.{self._get_gdal_error_message()}")
            
        try:
            # Get raster dimensions and geotransform
            width = ref_ds.RasterXSize
            height = ref_ds.RasterYSize
            geotransform = ref_ds.GetGeoTransform()
            projection = ref_ds.GetProjection()
            
            # Get complete spatial reference system
            srs = ref_ds.GetSpatialRef()
            
            res = abs(geotransform[1])  # pixel width
            
            # Validate geotransform
            if geotransform is None or len(geotransform) != 6:
                raise RuntimeError(f"Invalid geotransform in reference raster: {reference_raster_path}.{self._get_gdal_error_message()}")
                
        finally:
            ref_ds = None  # Close dataset

        buffer_distance = res + 0.01  # Small buffer to ensure rasterization of lines has no diagonal gaps ### with gdal version still needed? Debug later...

        # Collect all geometries from the provided GeoDataFrames
        all_geometries = [] 
        for gdf in features:
            if gdf.empty:
                continue
            else:
                # Use the helper function to flatten to LineStrings
                if flatten_lines and gdf.geometry.geom_type.isin(["MultiLineString", "LineString"]).any():
                    flattened_gdf = TopoDrainCore._flatten_to_linestrings(gdf) # flatten all line geometries to individual LineStrings
                    if not flattened_gdf.empty:
                        all_geometries.extend(flattened_gdf.geometry)
                else:
                    all_geometries.extend(gdf.geometry)

        # Generate output path if not provided
        if output_path is None:
            output_path = os.path.join(self.temp_directory, f"mask_{uuid.uuid4().hex[:8]}.tif")

        # Determine GDAL driver based on output file extension
        driver_name = self._get_gdal_driver_from_path(output_path)
        
        # Create empty output raster using GDAL with best practices
        driver = gdal.GetDriverByName(driver_name)
        if driver is None:
            raise RuntimeError(f"{driver_name} driver not available.{self._get_gdal_error_message()}")
            
        creation_options = [
            'COMPRESS=LZW',
            'TILED=YES',
            'BIGTIFF=IF_SAFER'
        ]
        
        out_ds = driver.Create(output_path, width, height, 1, gdal.GDT_UInt32, 
                              options=creation_options)
        if out_ds is None:
            raise RuntimeError(f"Failed to create output raster: {output_path}.{self._get_gdal_error_message()}")
            
        # Initialize geometry mapping for return value
        geometry_mapping = {}
            
        try:
            out_ds.SetGeoTransform(geotransform)
            
            # Set complete spatial reference system
            if srs is not None:
                out_ds.SetSpatialRef(srs)
            elif projection:
                # Fallback to projection string if SRS not available
                out_ds.SetProjection(projection)
            
            # Initialize raster with zeros
            out_band = out_ds.GetRasterBand(1)
            out_band.SetNoDataValue(0) # maybe use self.nodata? or fine for mask raster?
            out_band.Fill(0)

            # Create memory vector layer for rasterization
            mem_driver = ogr.GetDriverByName('Memory')
            if mem_driver is None:
                raise RuntimeError(f"Memory driver not available.{self._get_gdal_error_message()}")
                
            mem_ds = mem_driver.CreateDataSource('')
            if mem_ds is None:
                raise RuntimeError(f"Failed to create memory datasource.{self._get_gdal_error_message()}")
            
            try:
                mem_layer = mem_ds.CreateLayer('temp', None, ogr.wkbUnknown)
                if mem_layer is None:
                    raise RuntimeError(f"Failed to create memory layer.{self._get_gdal_error_message()}")
                
                # Add attribute field for raster values
                field_defn = ogr.FieldDefn('burn_value', ogr.OFTInteger)
                self._check_ogr_error("Create burn_value field", mem_layer.CreateField(field_defn))
                
                # Add geometries to layer
                for i, geom in enumerate(all_geometries):
                    if unique_values:
                        mask_value = i + 1  # Assign unique values to each individual geometry. Start from 1
                        geometry_mapping[mask_value] = geom  # Store the mapping
                    else:
                        mask_value = 1  # Default value for all geometries (binary mask)
                    
                    # Apply buffering to lines if requested
                    if geom.geom_type in ("LineString", "MultiLineString") and buffer_lines:
                        geom = geom.buffer(buffer_distance)
                    
                    # Create OGR feature
                    feature = ogr.Feature(mem_layer.GetLayerDefn())
                    feature.SetField('burn_value', int(mask_value))
                    
                    # Convert Shapely geometry to OGR geometry
                    ogr_geom = ogr.CreateGeometryFromWkt(geom.wkt)
                    if ogr_geom is None:
                        warnings.warn(f"Warning: Failed to convert geometry {i} to OGR format, skipping.{self._get_gdal_error_message()}")
                        continue
                    
                    # Validate geometry before setting
                    if not ogr_geom.IsValid():
                        warnings.warn(f"Warning: Invalid geometry {i}, attempting to fix or skipping.{self._get_gdal_error_message()}")
                        # Try to make it valid
                        try:
                            ogr_geom = ogr_geom.Buffer(0)  # Common trick to fix invalid geometries
                            if ogr_geom is None or not ogr_geom.IsValid():
                                ogr_geom = None
                                continue
                        except:
                            ogr_geom = None
                            continue
                        
                    feature.SetGeometry(ogr_geom)
                    
                    # Use the error checking helper for CreateFeature
                    result = mem_layer.CreateFeature(feature)
                    if result != ogr.OGRERR_NONE:
                        warnings.warn(f"Warning: Failed to create feature {i}, skipping.{self._get_gdal_error_message()} (Error code: {result})")
                    
                    feature = None  # Clean up feature
                    ogr_geom = None  # Clean up geometry
                
                # Rasterize the vector layer
                result = gdal.RasterizeLayer(out_ds, [1], mem_layer, 
                                           options=['ATTRIBUTE=burn_value', 'ALL_TOUCHED=YES'])
                if result != gdal.CE_None:
                    raise RuntimeError(f"Rasterization failed.{self._get_gdal_error_message()}")
                    
            finally:
                mem_layer = None
                mem_ds = None
                
        finally:
            out_ds = None  # Clean up output dataset

        # Return path only for binary mask, or tuple with geometry mapping for unique values
        if unique_values:
            return output_path, geometry_mapping
        else:
            return output_path

    ## Raster functions
    @staticmethod
    def _pixel_indices_to_coords(rows, cols, geotransform):
        """
        Convert pixel row,col indices to world coordinates using GDAL geotransform.
        Returns coordinates at the center of each pixel.
        
        GDAL Geotransform parameters:
        GT(0) x-coordinate of the upper-left corner of the upper-left pixel
        GT(1) w-e pixel resolution / pixel width
        GT(2) row rotation (typically zero)
        GT(3) y-coordinate of the upper-left corner of the upper-left pixel
        GT(4) column rotation (typically zero)
        GT(5) n-s pixel resolution / pixel height (negative value for a north-up image)
        
        Args:
            rows (array-like): Row indices of pixels
            cols (array-like): Column indices of pixels
            geotransform (tuple): GDAL geotransform parameters (6 values)
            
        Returns:
            list: List of (x, y) coordinate tuples in world coordinates
        """
        coords = []
        for row, col in zip(rows, cols):
            # Convert pixel indices to world coordinates (center of pixel)
            x = geotransform[0] + (col + 0.5) * geotransform[1] + (row + 0.5) * geotransform[2]
            y = geotransform[3] + (col + 0.5) * geotransform[4] + (row + 0.5) * geotransform[5]
            coords.append((x, y))
        return coords

    @staticmethod
    def _coords_to_pixel_indices(coords, geotransform):
        """
        Convert world coordinates to pixel indices using GDAL geotransform.
        
        GDAL Geotransform parameters:
        GT(0) x-coordinate of the upper-left corner of the upper-left pixel
        GT(1) w-e pixel resolution / pixel width
        GT(2) row rotation (typically zero)
        GT(3) y-coordinate of the upper-left corner of the upper-left pixel
        GT(4) column rotation (typically zero)
        GT(5) n-s pixel resolution / pixel height (negative value for a north-up image)
        
        Args:
            coords (list): List of (x, y) coordinate tuples in world coordinates
            geotransform (tuple): GDAL geotransform parameters (6 values)
            
        Returns:
            list: List of (px, py) pixel index tuples
        """
        pixel_indices = []
        for x, y in coords:
            px = int((x - geotransform[0]) / geotransform[1])
            py = int((y - geotransform[3]) / geotransform[5])
            pixel_indices.append((px, py))
        return pixel_indices

    # _clip_raster not used yet and therefore not debugged
    def _clip_raster(self, raster_path: str, mask: gpd.GeoDataFrame, out_path: str) -> str:
        """
        Clip a raster file using a polygon mask using GDAL.

        Args:
            raster_path (str): Path to input raster (supports all GDAL formats in gdal_driver_mapping).
            mask (GeoDataFrame): Polygon(s) to use as mask.
            out_path (str): Desired path for output masked raster.

        Returns:
            str: Path to the masked raster.
        """
        if mask.empty:
            raise ValueError("The provided GeoDataFrame is empty. Cannot mask raster.")

        # Create memory vector layer from mask GeoDataFrame
        mem_driver = ogr.GetDriverByName('Memory')
        if mem_driver is None:
            raise RuntimeError(f"Memory driver not available.{self._get_gdal_error_message()}")
            
        mem_ds = mem_driver.CreateDataSource('')
        if mem_ds is None:
            raise RuntimeError(f"Failed to create memory datasource.{self._get_gdal_error_message()}")
        
        try:
            mem_layer = mem_ds.CreateLayer('mask', None, ogr.wkbPolygon)
            if mem_layer is None:
                raise RuntimeError(f"Failed to create memory layer.{self._get_gdal_error_message()}")
            
            # Add mask geometries to layer
            for _, row in mask.iterrows():
                geom = row.geometry
                
                # Create OGR feature
                feature = ogr.Feature(mem_layer.GetLayerDefn())
                
                # Convert Shapely geometry to OGR geometry
                ogr_geom = ogr.CreateGeometryFromWkt(geom.wkt)
                if ogr_geom is None:
                    continue
                
                feature.SetGeometry(ogr_geom)
                
                # Add feature to layer
                result = mem_layer.CreateFeature(feature)
                if result != ogr.OGRERR_NONE:
                    continue
                
                feature = None  # Clean up feature
                ogr_geom = None  # Clean up geometry

            # Use gdal.Warp to crop and mask the raster
            # Ensure output directory exists
            os.makedirs(os.path.dirname(out_path), exist_ok=True)
            
            # Determine GDAL driver based on output file extension
            driver_name = self._get_gdal_driver_from_path(out_path)
            
            # Configure warp options
            warp_options = gdal.WarpOptions(
                format=driver_name,
                cutlineDSName=mem_ds,  # Use memory dataset as cutline
                cutlineLayer='mask',
                cropToCutline=True,
                creationOptions=['COMPRESS=LZW', 'TILED=YES', 'BIGTIFF=IF_SAFER']
            )
            
            # Perform the warp operation
            result_ds = gdal.Warp(out_path, raster_path, options=warp_options)
            if result_ds is None:
                raise RuntimeError(f"Failed to mask raster: {raster_path}.{self._get_gdal_error_message()}")
            
            result_ds = None  # Close result dataset
            
        finally:
            mem_layer = None
            mem_ds = None

        return out_path

    def _invert_dtm(self, dtm_path: str, output_path: str, feedback=None) -> str:
        """
        Create an inverted DTM (multiply by -1) to extract ridges.

        Args:
            dtm_path (str): Path to input DTM raster (supports all GDAL formats in gdal_driver_mapping).
            output_path (str): Path to output inverted DTM raster.
            feedback (QgsProcessingFeedback, optional): Optional feedback object for progress reporting.

        Returns:
            str: Path to inverted DTM raster.
        """
        if self.wbt is None:
            raise RuntimeError("WhiteboxTools not initialized. Check WhiteboxTools configuration: QGIS settings -> Options -> Processing -> Provider -> WhiteboxTools -> WhiteboxTools executable.")

        try:
            ret = self._execute_wbt(
                'multiply',
                feedback=feedback,
                report_progress=False,  # Don't override main progress bar
                input1=dtm_path,
                input2=-1.0,
                output=output_path
            )
            
            if ret != 0 or not os.path.exists(output_path):
                raise RuntimeError(f"DTM inversion failed: WhiteboxTools returned {ret}, output not found at {output_path}")
        except Exception as e:
            # Check if cancellation was the cause
            if feedback and feedback.isCanceled():
                feedback.reportError("Process cancelled by user during DTM inversion.")
                raise RuntimeError('Process cancelled by user.')
            raise RuntimeError(f"DTM inversion failed: {e}")

        return output_path

    def _log_raster(
        self,
        input_raster: str,
        output_path: str,
        overwrite: bool = True,
        val_band: int = 1
            ) -> str:
        """
        Computes the natural logarithm of a specified band in a raster,
        and either overwrites it or appends the result as a new band.

        Args:
            input_raster (str): Path to input raster (supports all GDAL formats in gdal_driver_mapping).
            output_path (str): Path to output raster (format determined by file extension).
            overwrite (bool): If True, replaces the selected band with log values.
                    If False, appends the log values as a new band.
            val_band (int): 1-based index of the band to compute the logarithm from.

        Returns:
            str: Path to the output raster.
        """
        # Read input raster using GDAL
        input_ds = gdal.Open(input_raster, gdal.GA_ReadOnly)
        if input_ds is None:
            raise RuntimeError(f"Cannot open input raster: {input_raster}.{self._get_gdal_error_message()}")
            
        try:
            # Get raster information
            width = input_ds.RasterXSize
            height = input_ds.RasterYSize
            band_count = input_ds.RasterCount
            geotransform = input_ds.GetGeoTransform()
            projection = input_ds.GetProjection()
            srs = input_ds.GetSpatialRef()
            
            # Validate band index
            if not (1 <= val_band <= band_count):
                raise ValueError(f"val_band={val_band} is out of range. Input raster has {band_count} band(s).")
            
            # Get the source band and its properties
            src_band = input_ds.GetRasterBand(val_band)
            nodata = src_band.GetNoDataValue()
            if nodata is None:
                nodata = self.nodata
            
            # Read the band data
            data = src_band.ReadAsArray().astype(np.float32)
            
            # Compute log(x) only for valid values > 0
            log_data = np.where(data > 0, np.log(data), nodata)
            
        finally:
            input_ds = None  # Close input dataset

        # Determine GDAL driver based on output file extension
        driver_name = self._get_gdal_driver_from_path(output_path)
        
        # Create output raster using GDAL with best practices
        driver = gdal.GetDriverByName(driver_name)
        if driver is None:
            raise RuntimeError(f"{driver_name} driver not available.{self._get_gdal_error_message()}")
            
        creation_options = [
            'COMPRESS=LZW',
            'TILED=YES',
            'BIGTIFF=IF_SAFER'
        ]
        
        if overwrite:
            # Create single-band output raster
            out_ds = driver.Create(output_path, width, height, 1, gdal.GDT_Float32, 
                                  options=creation_options)
        else:
            # Create multi-band output raster (original bands + log band)
            new_band_count = band_count + 1
            out_ds = driver.Create(output_path, width, height, new_band_count, gdal.GDT_Float32, 
                                  options=creation_options)
        
        if out_ds is None:
            raise RuntimeError(f"Failed to create output raster: {output_path}.{self._get_gdal_error_message()}")
            
        try:
            out_ds.SetGeoTransform(geotransform)
            
            # Set complete spatial reference system
            if srs is not None:
                out_ds.SetSpatialRef(srs)
            elif projection:
                # Fallback to projection string if SRS not available
                out_ds.SetProjection(projection)
            
            if overwrite:
                # Write only the log-transformed band
                out_band = out_ds.GetRasterBand(1)
                out_band.SetNoDataValue(nodata)
                out_band.WriteArray(log_data)
            else:
                # Re-open input raster to copy all original bands
                input_ds = gdal.Open(input_raster, gdal.GA_ReadOnly)
                if input_ds is None:
                    raise RuntimeError(f"Cannot re-open input raster: {input_raster}.{self._get_gdal_error_message()}")
                    
                try:
                    # Copy all original bands
                    for i in range(band_count):
                        src_band = input_ds.GetRasterBand(i + 1)
                        out_band = out_ds.GetRasterBand(i + 1)
                        
                        # Copy band data and properties
                        band_data = src_band.ReadAsArray().astype(np.float32)
                        out_band.WriteArray(band_data)
                        out_band.SetNoDataValue(src_band.GetNoDataValue())
                    
                    # Write the new log band
                    log_band = out_ds.GetRasterBand(new_band_count)
                    log_band.SetNoDataValue(nodata)
                    log_band.WriteArray(log_data)
                    
                finally:
                    input_ds = None  # Close input dataset again
                    
        finally:
            out_ds = None  # Close output dataset

        return output_path

    # Alternatively define constant value of absolute elevation at masked cells or input raster contains absolute elevation values
    # Not tested yet
    def _modify_dtm_with_mask(
        self,
        dtm_path: str,
        mask: np.ndarray,
        elevation_add: float,
        output_path: str
        ) -> str:
        """
        Modify DTM by adding elevation to masked cells using GDAL.

        Args:
            dtm_path (str): Path to input DTM raster (supports all GDAL formats in gdal_driver_mapping).
            mask (np.ndarray): Binary mask where elevation should be modified.
            elevation_add (float): Value to add to masked cells.
            output_path (str): Path to save modified DTM raster (format determined by file extension).

        Returns:
            str: Path to the modified DTM raster.
        """
        # Read input raster using GDAL
        input_ds = gdal.Open(dtm_path, gdal.GA_ReadOnly)
        if input_ds is None:
            raise RuntimeError(f"Cannot open input DTM raster: {dtm_path}.{self._get_gdal_error_message()}")
            
        try:
            # Get raster information
            width = input_ds.RasterXSize
            height = input_ds.RasterYSize
            geotransform = input_ds.GetGeoTransform()
            projection = input_ds.GetProjection()
            srs = input_ds.GetSpatialRef()
            
            # Read the first band data
            src_band = input_ds.GetRasterBand(1)
            nodata = src_band.GetNoDataValue()
            if nodata is None:
                nodata = self.nodata
                
            data = src_band.ReadAsArray().astype(np.float32)
            
        finally:
            input_ds = None  # Close input dataset

        # Modify data using the mask
        modified = data.copy()
        modified[mask == 1] += elevation_add

        # Determine GDAL driver based on output file extension
        driver_name = self._get_gdal_driver_from_path(output_path)
        
        # Create output raster using GDAL with best practices
        driver = gdal.GetDriverByName(driver_name)
        if driver is None:
            raise RuntimeError(f"{driver_name} driver not available.{self._get_gdal_error_message()}")
            
        creation_options = [
            'COMPRESS=LZW',
            'TILED=YES',
            'BIGTIFF=IF_SAFER'
        ]
        
        out_ds = driver.Create(output_path, width, height, 1, gdal.GDT_Float32, 
                              options=creation_options)
        if out_ds is None:
            raise RuntimeError(f"Failed to create output raster: {output_path}.{self._get_gdal_error_message()}")
            
        try:
            out_ds.SetGeoTransform(geotransform)
            
            # Set complete spatial reference system
            if srs is not None:
                out_ds.SetSpatialRef(srs)
            elif projection:
                # Fallback to projection string if SRS not available
                out_ds.SetProjection(projection)
            
            # Write the modified data
            out_band = out_ds.GetRasterBand(1)
            out_band.SetNoDataValue(nodata)
            out_band.WriteArray(modified)
            
        finally:
            out_ds = None  # Close output dataset

        return output_path

    def _raster_to_linestring_wbt(self, raster_path: str, snap_to_start_point: Point = None, snap_to_endpoint: Point = None, output_vector_path: str = None, feedback=None) -> LineString:
        """
        Uses WhiteboxTools to vectorize a raster and return a merged LineString or MultiLineString.
        Optionally snaps the endpoint to the center of a destination cell.

        Args:
            raster_path (str): Path to input raster where 1-valued pixels form your keyline (supports all GDAL formats in gdal_driver_mapping).
            snap_to_start_point (Point, optional): Point to snap the start of the line to.
            snap_to_endpoint (Point, optional): Point to snap the endpoint of the line to.
            feedback (QgsProcessingFeedback, optional): Optional feedback object for progress reporting.

        Returns:
            LineString or MultiLineString, or None if empty.
        """
        if self.wbt is None:
            raise RuntimeError("WhiteboxTools not initialized. Check WhiteboxTools configuration: QGIS settings -> Options -> Processing -> Provider -> WhiteboxTools -> WhiteboxTools executable.")

        if not output_vector_path:
            base, _ = os.path.splitext(raster_path)
            output_vector_path = base + ".shp"
            
        try:
            ret = self._execute_wbt(
                'raster_to_vector_lines',
                feedback=feedback,
                report_progress=False,  # Don't override main progress bar
                i=raster_path,
                output=output_vector_path
            )
            
            if ret != 0 or not os.path.exists(output_vector_path):
                raise RuntimeError(f"Raster to vector lines failed: WhiteboxTools returned {ret}, output not found at {output_vector_path}")
        except Exception as e:
            # Check if cancellation was the cause
            if feedback and feedback.isCanceled():
                feedback.reportError("Process cancelled by user during raster to vector conversion.")
                raise RuntimeError('Process cancelled by user.')
            raise RuntimeError(f"Raster to vector lines failed: {e}")

        gdf = gpd.read_file(output_vector_path)
        
        if gdf.empty:
            warnings.warn(f"Warning: No vector features found in {output_vector_path}.")
            return None

        all_geometries = list(gdf.geometry) 
        # First linemerge on all geometries
        merged_geom = linemerge(all_geometries)

        # Extract line geometries and filter valid ones
        line_geometries = []
        if isinstance(merged_geom, LineString):
            line_geometries.append(merged_geom)
        elif isinstance(merged_geom, MultiLineString):
            line_geometries.extend(list(merged_geom.geoms))
        else:
            # Fallback: process individual geometries if linemerge didn't work
            for geom in gdf.geometry:
                if isinstance(geom, LineString):
                    line_geometries.append(geom)
                elif isinstance(geom, MultiLineString):
                    # Add individual parts of MultiLineString
                    line_geometries.extend(list(geom.geoms))
        
        if not line_geometries:
            warnings.warn("Warning: No valid LineString geometries found after vectorization.")
            return None
        
        # Merge to one single LineString
        if len(line_geometries) == 1:
            # If only one line, no need to merge
            single_part_line = line_geometries[0]
        else:
            # Merge lines using distance-based approach
            single_part_line = TopoDrainCore._merge_lines_by_distance(line_geometries)
            
        # If single_part_line is empty, return None
        if not single_part_line:
            warnings.warn("Warning: No valid line segments found after vectorization.")
            return None
        
        # 5) Snap start to destination cell center if requested, and ensure correct line direction
        if snap_to_start_point:
            single_part_line = TopoDrainCore._snap_line_to_point(single_part_line, snap_to_start_point, "start")

        # 6) Snap endpoint to destination cell center if requested
        if snap_to_endpoint:
            single_part_line = TopoDrainCore._snap_line_to_point(single_part_line, snap_to_endpoint, "end")
        
        return single_part_line

    ## TopoDrainCore functions
    @staticmethod
    def _find_inflection_candidates(curvature: np.ndarray, window: int) -> list:
        """
        Detect inflection points where the curvature changes from convex to concave,
        using a moving average window. If none found, return point of strongest transition.

        Args:
            curvature (np.ndarray): Smoothed 2nd derivative of elevation profile.
            window (int): Number of points before/after to average.

        Returns:
            list of tuples: Sorted (index, strength) candidates (at least one guaranteed).
        """
        candidates = []

        for i in range(window, len(curvature) - window):
            before_avg = np.mean(curvature[i - window:i])
            after_avg = np.mean(curvature[i + 1:i + 1 + window])
            # Look for convex → concave transitions (positive to negative curvature)
            if before_avg > 0 and after_avg < 0:
                strength = abs(before_avg) + abs(after_avg)
                candidates.append((i, strength))

        # Fallback: if no clear convex → concave transitions found
        if not candidates:
            warnings.warn("Warning: No clear convex → concave inflection points found. Using strongest transition as fallback.")
            best_strength = -np.inf
            best_index = None
            for i in range(window, len(curvature) - window):
                before_avg = np.mean(curvature[i - window:i])
                after_avg = np.mean(curvature[i + 1:i + 1 + window])
                # Look for any significant transition (prioritize convex → concave)
                strength = before_avg - after_avg  # Higher when going from positive to negative
                if strength > best_strength:
                    best_strength = strength
                    best_index = i
            candidates = [(best_index, best_strength)]

        # Sort by strength (strongest transitions first)
        sorted_candidates = sorted(candidates, key=lambda x: x[1], reverse=True)
        return sorted_candidates

    ## Core functions
    def extract_valleys(
        self,
        dtm_path: str,
        filled_output_path: str = None,
        fdir_output_path: str = None,
        facc_output_path: str = None,
        facc_log_output_path: str = None,
        streams_output_path: str  = None,
        accumulation_threshold: int = 1000,
        dist_facc: float = 50,
        postfix: str = None,
        feedback=None
    ) -> gpd.GeoDataFrame:
        """
        Extract valley lines using WhiteboxTools. 

        Args:
            dtm_path (str):
                Path to input DTM raster (supports all GDAL formats in gdal_driver_mapping).
            filled_output_path (str, optional):
                Path to save the depression-filled DTM raster (format determined by file extension).
            fdir_output_path (str, optional):
                Path to save the flow-direction raster (format determined by file extension).
            facc_output_path (str, optional):
                Path to save the flow-accumulation raster (format determined by file extension).
            facc_log_output_path (str, optional):
                Path to save the log-scaled accumulation raster (format determined by file extension).
            streams_output_path (str, optional):
                Path to save the extracted stream raster (format determined by file extension).
            accumulation_threshold (int):
                Threshold for stream extraction (flow accumulation units).
            dist_facc (float):
                Maximum breach distance (in raster units) for depression filling.
            postfix (str, optional):
                Optional string to include in default output filenames.
            feedback (QgsProcessingFeedback, optional):
                Optional feedback object for progress reporting/logging (for QGIS Plugin).

        Returns:
            GeoDataFrame:
                Extracted stream (valley) network with attributes.
        """
        if self.wbt is None:
            raise RuntimeError("WhiteboxTools not initialized. Check WhiteboxTools configuration: QGIS settings -> Options -> Processing -> Provider -> WhiteboxTools -> WhiteboxTools executable.")

        if feedback:
            feedback.pushInfo("[ExtractValleys] Starting valley extraction process...")
            feedback.pushInfo("[ExtractValleys] *Detailed WhiteboxTools output can be viewed in the Python Console")
            feedback.setProgress(0)
        else:
            print("[ExtractValleys] Starting valley extraction process...")

        # Build defaults for everything
        if not postfix:
            d = lambda name: os.path.join(self.temp_directory, name)
            defaults = {
                "filled":         d("filled.tif"),
                "fdir":           d("fdir.tif"),
                "streams":        d("streams.tif"),
                "streams_vec":    d("streams.shp"),
                "streams_linked": d("streams_linked.shp"),
                "facc":           d("facc.tif"),
                "facc_log":       d("facc_log.tif"),
                "network":        d("stream_network.shp"),
            }
        else:
            d = lambda base: os.path.join(self.temp_directory, f"{base}_{postfix}")
            defaults = {
                "filled":         d("filled") + ".tif",
                "fdir":           d("fdir") + ".tif",
                "streams":        d("streams") + ".tif",
                "streams_vec":    d("streams") + ".shp",
                "streams_linked": d("streams_linked") + ".shp",
                "facc":           d("facc") + ".tif",
                "facc_log":       d("facc_log") + ".tif",
                "network":        d("stream_network") + ".shp",
            }

        print(f"[ExtractValleys] Define paths for outputs")
        # Only these four can be overridden
        filled_output_path   = filled_output_path   or defaults["filled"]
        fdir_output_path     = fdir_output_path     or defaults["fdir"]
        facc_output_path     = facc_output_path     or defaults["facc"]
        facc_log_output_path = facc_log_output_path or defaults["facc_log"]
        streams_output_path  = streams_output_path  or defaults["streams"]

        # intermediate paths always use defaults:
        streams_vec_output_path    = defaults["streams_vec"]
        streams_linked_output_path = defaults["streams_linked"]
        stream_network_output_path = defaults["network"]

        try:
            if feedback:
                feedback.pushInfo(f"[ExtractValleys] Step 1/7: Filling depressions → {filled_output_path}")
                feedback.setProgress(10)
                # Check if user has canceled process
                if feedback.isCanceled():
                    feedback.reportError("Process cancelled by user after step 1 initialization.")
                    raise RuntimeError('Process cancelled by user.')
            else:
                print(f"[ExtractValleys] Step 1/7: Filling depressions → {filled_output_path}")
            try:
                ret = self._execute_wbt(
                    'breach_depressions_least_cost',
                    feedback=feedback,  # Pass feedback to enable cancellation during execution
                    report_progress=False,  # Don't override main progress bar
                    dem=dtm_path,
                    output=filled_output_path,
                    dist=int(dist_facc),
                    fill=True,
                    min_dist=True
                )
                if ret != 0 or not os.path.exists(filled_output_path):
                    raise RuntimeError(f"[ExtractValleys] Depression filling failed: WhiteboxTools returned {ret}, output not found at {filled_output_path}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during depression filling.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"[ExtractValleys] Depression filling failed: {e}")

            if feedback:
                feedback.pushInfo(f"[ExtractValleys] Step 2/7: Computing flow direction → {fdir_output_path}")
                feedback.setProgress(25)
            else:
                print(f"[ExtractValleys] Step 2/7: Computing flow direction → {fdir_output_path}")
            try:
                ret = self._execute_wbt(
                    'd8_pointer',
                    feedback=feedback,  # Pass feedback to enable cancellation during execution
                    report_progress=False,  # Don't override main progress bar
                    dem=filled_output_path,
                    output=fdir_output_path
                )
                if ret != 0 or not os.path.exists(fdir_output_path):
                    raise RuntimeError(f"[ExtractValleys] Flow direction failed: WhiteboxTools returned {ret}, output not found at {fdir_output_path}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during flow direction computation.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"[ExtractValleys] Flow direction failed: {e}")

            if feedback:
                feedback.pushInfo(f"[ExtractValleys] Step 3/7: Computing flow accumulation → {facc_output_path}")
                feedback.setProgress(40)
            else:
                print(f"[ExtractValleys] Step 3/7: Computing flow accumulation → {facc_output_path}")
            try:
                ret = self._execute_wbt(
                    'd8_flow_accumulation',
                    feedback=feedback,  # Pass feedback to enable cancellation during execution
                    report_progress=False,  # Don't override main progress bar
                    i=filled_output_path,
                    output=facc_output_path,
                    out_type="specific contributing area"
                )
                if ret != 0 or not os.path.exists(facc_output_path):
                    raise RuntimeError(f"[ExtractValleys] Flow accumulation failed: WhiteboxTools returned {ret}, output not found at {facc_output_path}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during flow accumulation computation.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"[ExtractValleys] Flow accumulation failed: {e}")

            if feedback:
                feedback.pushInfo(f"[ExtractValleys] Step 4/7: Creating log-scaled accumulation → {facc_log_output_path}")
                feedback.setProgress(55)
            else:
                print(f"[ExtractValleys] Step 4/7: Creating log-scaled accumulation → {facc_log_output_path}")
            try:
                self._log_raster(input_raster=facc_output_path, output_path=facc_log_output_path)
                if not os.path.exists(facc_log_output_path):
                    raise RuntimeError(f"[ExtractValleys] Log-scaled accumulation output not found at {facc_log_output_path}")
            except Exception as e:
                msg = f"[ExtractValleys] Warning: Log-scaled accumulation failed: {e}"
                if feedback:
                    feedback.pushWarning(msg)
                else:
                    warnings.warn(msg)

            if feedback:
                feedback.pushInfo(f"[ExtractValleys] Step 5/7: Extracting streams (threshold={accumulation_threshold})")
                feedback.setProgress(70)
            else:
                print(f"[ExtractValleys] Step 5/7: Extracting streams (threshold={accumulation_threshold})")
            try:
                ret = self._execute_wbt(
                    'extract_streams',
                    feedback=feedback,  # Pass feedback to enable cancellation during execution
                    report_progress=False,  # Don't override main progress bar
                    flow_accum=facc_output_path,
                    output=streams_output_path,
                    threshold=accumulation_threshold
                )
                if ret != 0 or not os.path.exists(streams_output_path):
                    raise RuntimeError(f"[ExtractValleys] Stream extraction failed: WhiteboxTools returned {ret}, output not found at {streams_output_path}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during stream extraction.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"[ExtractValleys] Stream extraction failed: {e}")

            if feedback:
                feedback.pushInfo("[ExtractValleys] Step 6/7: Vectorizing streams")
                feedback.setProgress(80)
            else:
                print("[ExtractValleys] Step 6/7: Vectorizing streams")
            try:
                ret = self._execute_wbt(
                    'raster_streams_to_vector',
                    feedback=feedback,  # Pass feedback to enable cancellation during execution
                    report_progress=False,  # Don't override main progress bar
                    streams=streams_output_path,
                    d8_pntr=fdir_output_path,
                    output=streams_vec_output_path
                )
                if ret != 0 or not os.path.exists(streams_vec_output_path):
                    raise RuntimeError(f"[ExtractValleys] Vectorizing streams failed: WhiteboxTools returned {ret}, output not found at {streams_vec_output_path}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during stream vectorization.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"[ExtractValleys] Vectorizing streams failed: {e}")

            streams_vec_id = streams_linked_output_path.replace(".shp", "_id.tif")
            try:
                if feedback:
                    feedback.pushInfo("[ExtractValleys] Step 7/7: Processing network topology - Identifying stream links")
                    feedback.setProgress(85)
                else:
                    print("[ExtractValleys] Step 7/7: Processing network topology - Identifying stream links")
                ret = self._execute_wbt(
                    'stream_link_identifier',
                    feedback=feedback,  # Pass feedback to enable cancellation during execution
                    report_progress=False,  # Don't override main progress bar
                    d8_pntr=fdir_output_path,
                    streams=streams_output_path,
                    output=streams_vec_id
                )
                if ret != 0 or not os.path.exists(streams_vec_id):
                    raise RuntimeError(f"[ExtractValleys] Stream link identifier failed: WhiteboxTools returned {ret}, output not found at {streams_vec_id}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during stream link identification.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"[ExtractValleys] Stream link identifier failed: {e}")

            try:
                if feedback:
                    feedback.pushInfo("[ExtractValleys] Converting linked streams to vectors")
                    feedback.setProgress(90)
                else:
                    print("[ExtractValleys] Converting linked streams to vectors")
                ret = self._execute_wbt(
                    'raster_streams_to_vector',
                    feedback=feedback,  # Pass feedback to enable cancellation during execution
                    report_progress=False,  # Don't override main progress bar
                    streams=streams_vec_id,
                    d8_pntr=fdir_output_path,
                    output=streams_linked_output_path
                )
                if ret != 0 or not os.path.exists(streams_linked_output_path):
                    raise RuntimeError(f"[ExtractValleys] Converting linked streams failed: WhiteboxTools returned {ret}, output not found at {streams_linked_output_path}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during linked stream conversion.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"[ExtractValleys] Converting linked streams failed: {e}")

            try:
                if feedback:
                    feedback.pushInfo("[ExtractValleys] Performing final network analysis")
                    feedback.setProgress(95)
                else:
                    print("[ExtractValleys] Performing final network analysis")
                ret = self._execute_wbt(
                    'VectorStreamNetworkAnalysis',
                    feedback=feedback,  # Pass feedback to enable cancellation during execution
                    report_progress=False,  # Don't override main progress bar
                    streams=streams_linked_output_path,
                    dem=filled_output_path,
                    output=stream_network_output_path
                    )
                if ret != 0 or not os.path.exists(stream_network_output_path):
                    raise RuntimeError(f"[ExtractValleys] Network analysis failed: WhiteboxTools returned {ret}, output not found at {stream_network_output_path}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during network analysis.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"[ExtractValleys] Network analysis failed: {e}")

            if feedback:
                feedback.pushInfo(f"[ExtractValleys] Loading network from {stream_network_output_path}")
            else:
                print(f"[ExtractValleys] Loading network from {stream_network_output_path}")

            # Check if the file exists and is non-empty before reading
            if not os.path.exists(stream_network_output_path):
                raise RuntimeError(f"[ExtractValleys] Network output file not found: {stream_network_output_path}")
            gdf = gpd.read_file(stream_network_output_path)
            if gdf.empty:
                raise RuntimeError(f"[ExtractValleys] Network output file is empty: {stream_network_output_path}")

            # Always create a reliable LINK_ID field as primary identifier
            # This addresses Windows case-sensitivity and temporary layer issues
            if 'FID' in gdf.columns or 'fid' in gdf.columns:
                fid_col = 'FID' if 'FID' in gdf.columns else 'fid'
                gdf['LINK_ID'] = gdf[fid_col]
            else:
                gdf['LINK_ID'] = range(1, len(gdf) + 1)
            
            if feedback:
                feedback.pushInfo("[ExtractValleys] Created reliable 'LINK_ID' field for cross-platform compatibility")
                feedback.pushInfo(f"[ExtractValleys] Completed: {len(gdf)} valley features extracted successfully!")
                feedback.setProgress(100)
            else:
                print("[ExtractValleys] Created reliable 'LINK_ID' field for cross-platform compatibility")
                print(f"[ExtractValleys] Completed: {len(gdf)} valley features extracted successfully!")
            
            return gdf

        except Exception as e:
            raise RuntimeError(f"[ExtractValleys] Failed to extract valleys: {e}")
        

    def extract_ridges(
        self,
        dtm_path: str,
        inverted_filled_output_path: str = None,
        inverted_fdir_output_path: str = None,
        inverted_facc_output_path: str = None,
        inverted_facc_log_output_path: str = None,
        inverted_streams_output_path: str = None,
        accumulation_threshold: int = 1000,
        dist_facc: float = 50,
        postfix: str = "inverted",
        feedback=None
    ) -> gpd.GeoDataFrame:
        """
        Extract ridge lines (watershed divides) from a DTM by inverting the terrain
        and running the valley‐extraction workflow.

        Args:
            dtm_path (str):
                Path to input DTM raster (supports all GDAL formats in gdal_driver_mapping).
            inverted_filled_output_path (str, optional):
                Where to save the inverted‐DTM’s filled DEM (GeoTIFF, “.tif”).
            inverted_fdir_output_path (str, optional):
                Where to save the inverted‐DTM’s flow‐direction raster (GeoTIFF).
            inverted_facc_output_path (str, optional):
                Where to save the inverted‐DTM’s flow‐accumulation raster (GeoTIFF).
            inverted_facc_log_output_path (str, optional):
                Where to save the inverted‐DTM’s log‐scaled accumulation raster (GeoTIFF).
            inverted_streams_output_path (str, optional):
                Where to save the inverted‐DTM’s extracted streams (GeoTIFF).   
            accumulation_threshold (int):
                Threshold for ridge extraction (analogous to stream threshold).
            dist_facc (float):
                Maximum breach distance (in raster units) for depression filling.
            postfix (str):
                Postfix for naming intermediate files (default “inverted”).

        Returns:
            GeoDataFrame:
                Extracted ridge (divide) network as vector geometries.
        """
        if self.wbt is None:
            raise RuntimeError("WhiteboxTools not initialized. Check WhiteboxTools configuration: QGIS settings -> Options -> Processing -> Provider -> WhiteboxTools -> WhiteboxTools executable.")

        if feedback:
            feedback.pushInfo("[ExtractRidges] Starting ridge extraction process...")
            feedback.setProgress(0)
        else:
            print("[ExtractRidges] Starting ridge extraction process...")

        # 1) Invert the DTM
        if feedback:
            feedback.pushInfo("[ExtractRidges] Inverting DTM for ridge extraction...")
        else:
            print("[ExtractRidges] Inverting DTM for ridge extraction...")
        
        inverted_dtm = os.path.join(self.temp_directory, f"inverted_dtm_{postfix}.tif")
        inverted_dtm = self._invert_dtm(dtm_path, inverted_dtm, feedback=feedback)  # Remove feedback to prevent multiple progress bars
        
        if feedback:
            feedback.pushInfo(f"[ExtractRidges] DTM inversion complete: {inverted_dtm}")
            feedback.setProgress(5)
        else:
            print(f"[ExtractRidges] DTM inversion complete: {inverted_dtm}")

        # 2) Compute defaults for the four inverted outputs
        #    We leverage extract_valleys’ own default logic by passing these params through.
        if feedback:
            feedback.pushInfo("[ExtractRidges] Extracting ridges from inverted DTM (using extract_valleys function)...")
        else:
            print("[ExtractRidges] Extracting ridges from inverted DTM (using extract_valleys function)...")
        # If the user did not supply, leave as None—extract_valleys will pick its defaults (which include postfix).
        inv_filled = inverted_filled_output_path
        inv_fdir   = inverted_fdir_output_path
        inv_facc   = inverted_facc_output_path
        inv_facc_log = inverted_facc_log_output_path
        inv_streams = inverted_streams_output_path

        # 3) Call extract_valleys on the inverted DTM (this will handle its own progress reporting)
        ridges_gdf = self.extract_valleys(
            dtm_path=inverted_dtm,
            filled_output_path=inv_filled,
            fdir_output_path=inv_fdir,
            facc_output_path=inv_facc,
            facc_log_output_path=inv_facc_log,
            streams_output_path=inv_streams,
            accumulation_threshold=accumulation_threshold,
            dist_facc=dist_facc,
            postfix=postfix,
            feedback=feedback 
        )

        if feedback:
            feedback.pushInfo(f"[ExtractRidges] Ridge extraction completed successfully: {len(ridges_gdf)} ridge features extracted!")
        else:
            print(f"[ExtractRidges] Ridge extraction completed successfully: {len(ridges_gdf)} ridge features extracted!")

        return ridges_gdf


    def extract_main_valleys(
        self,
        valley_lines: gpd.GeoDataFrame,
        facc_path: str,
        perimeter: gpd.GeoDataFrame = None,
        nr_main: int = 2,
        clip_to_perimeter: bool = True,
        feedback=None
    ) -> gpd.GeoDataFrame:
        """
        Identify and merge main valley lines based on the highest flow accumulation,
        using only points uniquely associated with one TRIB_ID (to avoid confluent points).

        Args:
            valley_lines (GeoDataFrame): Valley line network with 'LINK_ID', 'TRIB_ID', and 'DS_LINK_ID' attributes.
            facc_path (str): Path to the flow accumulation raster.
            perimeter (GeoDataFrame, optional): Polygon defining the area boundary. If None, uses valley_lines extent.
            nr_main (int): Number of main valleys to select.
            clip_to_perimeter (bool): If True, clips output to boundary polygon of perimeter.
            feedback (QgsProcessingFeedback, optional): Optional feedback object for progress reporting/logging.

        Returns:
            GeoDataFrame: Main valley lines with TRIB_ID, LINK_ID, RANK, and POLYGON_ID attributes.
        """
        if feedback:
            feedback.pushInfo("[ExtractMainValleys] Starting main valley extraction...")
            feedback.setProgress(0)
            if feedback.isCanceled():
                feedback.reportError('Process cancelled by user at initialization.')
                raise RuntimeError('Process cancelled by user.')
        else:
            print("[ExtractMainValleys] Starting main valley extraction...")

        if valley_lines.empty:
            raise RuntimeError("[ExtractMainValleys] Input valley_lines is empty")

        # Ensure LINK_ID field is present (required)
        if 'LINK_ID' not in valley_lines.columns:
            raise RuntimeError("[ExtractMainValleys] Input valley_lines must have a 'LINK_ID' attribute. Please use valley lines generated by the Create Valleys algorithm.")
        
        # Ensure TRIB_ID field is present in valley_lines
        if 'TRIB_ID' not in valley_lines.columns:
            # Create TRIB_ID field using LINK_ID values as fallback
            valley_lines = valley_lines.copy()  # Avoid modifying original
            valley_lines['TRIB_ID'] = valley_lines['LINK_ID']
            if feedback:
                feedback.pushWarning("[ExtractMainValleys] Warning: 'TRIB_ID' column not found in valley_lines, using 'LINK_ID' values as fallback")
            else:
                warnings.warn("[ExtractMainValleys] Warning: 'TRIB_ID' column not found in valley_lines, using 'LINK_ID' values as fallback")

        # Ensure DS_LINK_ID field is present in valley_lines (optional field, can be null)
        if 'DS_LINK_ID' not in valley_lines.columns:
            # Create DS_LINK_ID field with null values (not critical for main valley extraction)
            valley_lines = valley_lines.copy()  # Avoid modifying original
            valley_lines['DS_LINK_ID'] = None
            if feedback:
                feedback.pushWarning("[ExtractMainValleys] Warning: 'DS_LINK_ID' column not found in valley_lines, created with null values as fallback")
            else:
                warnings.warn("[ExtractMainValleys] Warning: 'DS_LINK_ID' column not found in valley_lines, created with null values as fallback")

        # Create perimeter from valley_lines extent if not provided
        if perimeter is None:
            if feedback:
                feedback.pushInfo("[ExtractMainValleys] No perimeter provided, using valley lines extent...")
            else:
                print("[ExtractMainValleys] No perimeter provided, using valley lines extent...")
            
            # Get the bounding box of valley_lines and create a polygon
            bounds = valley_lines.total_bounds  # [minx, miny, maxx, maxy]
            bbox_polygon = Polygon([
                (bounds[0], bounds[1]),  # bottom-left
                (bounds[2], bounds[1]),  # bottom-right
                (bounds[2], bounds[3]),  # top-right
                (bounds[0], bounds[3]),  # top-left
                (bounds[0], bounds[1])   # close polygon
            ])
            perimeter = gpd.GeoDataFrame([{'geometry': bbox_polygon}], crs=valley_lines.crs)

        if feedback:
            feedback.pushInfo("[ExtractMainValleys] Reading flow accumulation raster...")
            feedback.setProgress(10)
            if feedback.isCanceled():
                feedback.reportError('Process cancelled by user after reading flow accumulation raster.')
                raise RuntimeError('Process cancelled by user.')
        else:
            print("[ExtractMainValleys] Reading flow accumulation raster...")
        
        # Read flow accumulation raster using GDAL
        facc_ds = gdal.Open(facc_path, gdal.GA_ReadOnly)
        if facc_ds is None:
            raise RuntimeError(f"Cannot open flow accumulation raster: {facc_path}.{self._get_gdal_error_message()}")
            
        try:
            facc_band = facc_ds.GetRasterBand(1)
            facc = facc_band.ReadAsArray()
            if facc is None:
                raise RuntimeError(f"Failed to read flow accumulation data from: {facc_path}.{self._get_gdal_error_message()}")
            
            # Get geotransform for coordinate conversion
            geotransform = facc_ds.GetGeoTransform()
            if geotransform is None:
                raise RuntimeError(f"Failed to get geotransform from: {facc_path}.{self._get_gdal_error_message()}")
            
            res = abs(geotransform[1])  # pixel width
            
        finally:
            facc_ds = None  # Close dataset

        # Process each polygon in the perimeter separately
        all_merged_records = []
        global_fid_counter = 1
        
        for poly_idx, poly_row in perimeter.iterrows():
            single_polygon = gpd.GeoDataFrame([poly_row], crs=perimeter.crs)
            
            # Calculate progress based on polygon processing (20-80% range)
            polygon_progress = 20 + int((poly_idx / len(perimeter)) * 60)
            
            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Processing polygon {poly_idx + 1}/{len(perimeter)}...")
                feedback.setProgress(polygon_progress)
                if feedback.isCanceled():
                    feedback.reportError(f'Process cancelled by user while processing polygon {poly_idx + 1}.')
                    raise RuntimeError('Process cancelled by user.')
            else:
                print(f"[ExtractMainValleys] Processing polygon {poly_idx + 1}/{len(perimeter)}...")

            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Clipping valley lines to polygon {poly_idx + 1}...")
            else:
                print(f"[ExtractMainValleys] Clipping valley lines to polygon {poly_idx + 1}...")
            valley_clipped = gpd.overlay(valley_lines, single_polygon, how="intersection")
            
            if valley_clipped.empty:
                if feedback:
                    feedback.pushInfo(f"[ExtractMainValleys] No valley lines found in polygon {poly_idx + 1}, skipping...")
                else:
                    print(f"[ExtractMainValleys] No valley lines found in polygon {poly_idx + 1}, skipping...")
                continue

            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Rasterizing valley lines for polygon {poly_idx + 1}...")
            else:
                print(f"[ExtractMainValleys] Rasterizing valley lines for polygon {poly_idx + 1}...")
            valley_raster_path = os.path.join(self.temp_directory, f"valley_mask_poly_{poly_idx}.tif")
           
           # All valley lines are rasterized together into a single binary mask (1 = valley cell, 0 = background)
            valley_mask_path = self._vector_to_mask_raster(
                features=[valley_clipped],
                reference_raster_path=facc_path,
                output_path=valley_raster_path,
                unique_values=False,
                flatten_lines=False,
                buffer_lines=False
            )
            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Valley mask created at {valley_mask_path}")
            else:
                print(f"[ExtractMainValleys] Valley mask created at {valley_mask_path}")

            # Read the valley mask data from the saved raster file using GDAL
            valley_lines_ds = gdal.Open(valley_mask_path, gdal.GA_ReadOnly)
            if valley_lines_ds is None:
                raise RuntimeError(f"Cannot open valley mask raster: {valley_mask_path}.{self._get_gdal_error_message()}")
                
            try:
                valley_lines_band = valley_lines_ds.GetRasterBand(1)
                valley_mask = valley_lines_band.ReadAsArray()
                if valley_mask is None:
                    raise RuntimeError(f"Failed to read valley mask data from: {valley_mask_path}.{self._get_gdal_error_message()}")
                    
            finally:
                valley_lines_ds = None  # Close dataset

            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Extracting facc > 0 points for polygon {poly_idx + 1}...")
            else:
                print(f"[ExtractMainValleys] Extracting facc > 0 points for polygon {poly_idx + 1}...")
            mask = (valley_mask == 1) & (facc > 0)
            rows, cols = np.where(mask)
            if len(rows) == 0:
                if feedback:
                    feedback.pushInfo(f"[ExtractMainValleys] No valley cells with flow accumulation > 0 found in polygon {poly_idx + 1}, skipping...")
                else:
                    print(f"[ExtractMainValleys] No valley cells with flow accumulation > 0 found in polygon {poly_idx + 1}, skipping...")
                continue

            # Points are created at the center coordinates of the raster cells containing valley lines with facc > 0
            # Convert row,col indices to world coordinates using GDAL geotransform
            coords = self._pixel_indices_to_coords(rows, cols, geotransform)
            points = gpd.GeoDataFrame(geometry=gpd.points_from_xy(*zip(*coords)), crs=self.crs)
            points["facc"] = facc[rows, cols]

            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Performing spatial join for polygon {poly_idx + 1}...")
            else:
                print(f"[ExtractMainValleys] Performing spatial join for polygon {poly_idx + 1}...")
            
            # Ensure the required columns exist in valley_clipped (they should after validation above)
            join_columns = ["geometry"]
            for col in ["LINK_ID", "TRIB_ID", "DS_LINK_ID"]:
                if col in valley_clipped.columns:
                    join_columns.append(col)
            
            # Spatial Join with Original Vector Lines using buffered points to ensure all valley lines within raster cells are captured
            # Buffer points by half the cell resolution to catch all lines passing through the raster cell
            buffer_distance = res / 2.0  # Half cell size ensures we capture lines at cell edges
            
            points_buffered = points.copy()
            points_buffered.geometry = points.geometry.buffer(buffer_distance)
            
            points_joined = gpd.sjoin(
                points_buffered,
                valley_clipped[join_columns],
                how="inner"
            ).drop(columns="index_right")
            
            # Restore original point geometries for further processing
            points_joined.geometry = points.geometry[points_joined.index]

            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Filtering ambiguous facc points for polygon {poly_idx + 1}...")
            else:
                print(f"[ExtractMainValleys] Filtering ambiguous facc points for polygon {poly_idx + 1}...")
            
            # Removes any point that belongs to multiple TRIB_IDs. Prevents "Flow Accumulation Theft" at confluences.
            points_joined["geom_wkt"] = points_joined.geometry.apply(lambda geom: geom.wkt)
            geom_counts = points_joined.groupby("geom_wkt")["TRIB_ID"].nunique()
            valid_geoms = geom_counts[geom_counts == 1].index
            points_unique = points_joined[points_joined["geom_wkt"].isin(valid_geoms)].copy()

            if points_unique.empty:
                if feedback:
                    feedback.pushWarning(f"[ExtractMainValleys] Warning: No unique valley points found in polygon {poly_idx + 1}, skipping...")
                else:
                    warnings.warn(f"[ExtractMainValleys] Warning: No unique valley points found in polygon {poly_idx + 1}, skipping...")
                continue

            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Selecting top {nr_main} TRIB_IDs for polygon {poly_idx + 1}...")
            else:
                print(f"[ExtractMainValleys] Selecting top {nr_main} TRIB_IDs for polygon {poly_idx + 1}...")
            points_sorted = points_unique.sort_values("facc", ascending=False)
            points_top = points_sorted.drop_duplicates(subset="TRIB_ID").head(nr_main)

            if points_top.empty:
                if feedback:
                    feedback.pushWarning(f"[ExtractMainValleys] Warning: No main valley lines could be selected for polygon {poly_idx + 1}, skipping...")
                else:
                    warnings.warn(f"[ExtractMainValleys] Warning: No main valley lines could be selected for polygon {poly_idx + 1}, skipping...")
                continue

            selected_trib_ids = points_top["TRIB_ID"].unique()
            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Selected TRIB_IDs for polygon {poly_idx + 1}: {list(selected_trib_ids)}")
            else:
                print(f"[ExtractMainValleys] Selected TRIB_IDs for polygon {poly_idx + 1}: {list(selected_trib_ids)}")

            # Create ranking based on flow accumulation values (highest facc gets rank 1)
            # points_top is already sorted by facc descending since it comes from points_sorted
            trib_id_ranking = {}
            for rank, (_, row) in enumerate(points_top.iterrows(), 1):
                trib_id_ranking[row["TRIB_ID"]] = rank
            
            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] TRIB_ID rankings for polygon {poly_idx + 1}: {trib_id_ranking}")
            else:
                print(f"[ExtractMainValleys] TRIB_ID rankings for polygon {poly_idx + 1}: {trib_id_ranking}")

            if feedback:
                feedback.pushInfo(f"[ExtractMainValleys] Merging valley line segments for polygon {poly_idx + 1}...")
            else:
                print(f"[ExtractMainValleys] Merging valley line segments for polygon {poly_idx + 1}...") # because maybe split by perimeter
            for trib_id in selected_trib_ids:
                lines = valley_lines[valley_lines["TRIB_ID"] == trib_id]

                cleaned = []
                for geom in lines.geometry:
                    if geom.is_empty:
                        continue
                    if isinstance(geom, LineString):
                        cleaned.append(geom)
                    elif isinstance(geom, MultiLineString):
                        cleaned.extend([g for g in geom.geoms if isinstance(g, LineString)])

                if cleaned:
                    try:
                        merged_line = linemerge(cleaned)
                        # Get the first matching line to copy attributes from
                        first_line = lines.iloc[0]
                        # Get the rank for this TRIB_ID
                        rank = trib_id_ranking.get(trib_id, 999)  # Default to 999 if not found
                        all_merged_records.append({
                            "geometry": merged_line,
                            "TRIB_ID": trib_id,
                            "LINK_ID": global_fid_counter,
                            "RANK": rank,  # Add ranking based on flow accumulation
                            "POLYGON_ID": poly_idx + 1,
                            # Copy other attributes if they exist
                            "DS_LINK_ID": first_line.get("DS_LINK_ID", None) if hasattr(first_line, 'get') else None
                        })
                        global_fid_counter += 1
                        if feedback:
                            feedback.pushInfo(f"[ExtractMainValleys] Merged TRIB_ID={trib_id} (RANK={rank}) for polygon {poly_idx + 1}, segments={len(cleaned)}")
                        else:
                            print(f"[ExtractMainValleys] Merged TRIB_ID={trib_id} (RANK={rank}) for polygon {poly_idx + 1}, segments={len(cleaned)}")
                    except Exception as e:
                        raise RuntimeError(f"[ExtractMainValleys] Failed to merge lines for TRIB_ID={trib_id} in polygon {poly_idx + 1}: {e}")

        if not all_merged_records:
            raise RuntimeError("[ExtractMainValleys] No main valley lines could be extracted from any polygon.")

        gdf = gpd.GeoDataFrame(all_merged_records, crs=self.crs)

        if clip_to_perimeter:
            if feedback:
                feedback.pushInfo("[ExtractMainValleys] Clipping final valley lines to perimeter...")
                feedback.setProgress(90)
                if feedback.isCanceled():
                    feedback.reportError('Process cancelled by user during final clipping.')
                    raise RuntimeError('Process cancelled by user.')
            else:
                print("[ExtractMainValleys] Clipping final valley lines to perimeter...")
            gdf = gpd.overlay(gdf, perimeter, how="intersection")

        if feedback:
            feedback.pushInfo(f"[ExtractMainValleys] Main valley extraction complete. {len(gdf)} valleys extracted from {len(perimeter)} polygons.")
            feedback.setProgress(100)
            if feedback.isCanceled():
                feedback.reportError('Process cancelled by user at completion.')
                raise RuntimeError('Process cancelled by user.')
        else:
            print(f"[ExtractMainValleys] Main valley extraction complete. {len(gdf)} valleys extracted from {len(perimeter)} polygons.")
        return gdf


    def extract_main_ridges(
        self,
        ridge_lines: gpd.GeoDataFrame,
        facc_path: str,
        perimeter: gpd.GeoDataFrame = None,
        nr_main: int = 2,
        clip_to_perimeter: bool = True,
        feedback=None
    ) -> gpd.GeoDataFrame:
        """
        Identify and trace the main ridge lines (watershed divides) using the same logic as main valley detection.
        Merging based on the highest flow accumulation,
        using only points uniquely associated with one TRIB_ID (to avoid confluent points).

        Args:
            ridge_lines (GeoDataFrame): Ridge line network with 'LINK_ID', 'TRIB_ID', and 'DS_LINK_ID' attributes.
            facc_path (str): Path to the flow accumulation raster (based on inverted DTM).
            perimeter (GeoDataFrame, optional): Polygon defining the area boundary. If None, uses ridge_lines extent.
            nr_main (int): Number of main ridges to select.
            clip_to_perimeter (bool): If True, clips output to boundary polygon of perimeter.
            feedback (QgsProcessingFeedback, optional): Optional feedback object for progress reporting/logging.

        Returns:
            GeoDataFrame: Traced main ridge lines.
        """
        if feedback:
            feedback.pushInfo("[ExtractMainRidges] Starting main ridge extraction using main valleys logic (extract_main_valleys)...")
        else:
            print("[ExtractMainRidges] Starting main ridge extraction using main valleys logic (extract_main_valleys)...")

        gdf = self.extract_main_valleys(
            valley_lines=ridge_lines,
            facc_path=facc_path,
            perimeter=perimeter,
            nr_main=nr_main,
            clip_to_perimeter=clip_to_perimeter,
            feedback=feedback
        )

        return gdf

    def get_keypoints(
        self,
        valley_lines: gpd.GeoDataFrame,
        dtm_path: str,
        smoothing_window: int = 9,
        polyorder: int = 2,
        min_distance: float = 10.0,
        max_keypoints: int = 5,
        find_window_cells: int = 10,
        feedback=None
        ) -> gpd.GeoDataFrame:
        """
        Detect keypoints along valley lines based on curvature of elevation profiles
        (second derivative). Keypoints are locations where the profile changes from 
        convex to concave curvature, indicating morphological transitions like 
        channel heads or slope breaks.

        The elevation profile is extracted along each valley line using the DTM at
        pixel resolution (all values along the line) and smoothed using a Savitzky-Golay 
        filter. The second derivative is then computed, and points with the strongest 
        convex → concave transitions are selected as keypoints.

        Args:
            valley_lines (GeoDataFrame): Valley centerlines with geometries and unique LINK_ID.
            dtm_path (str): Path to the input DTM raster.
            smoothing_window (int): Window size for Savitzky-Golay filter (must be odd).
            polyorder (int): Polynomial order for Savitzky-Golay smoothing.
            min_distance (float): Minimum distance between selected keypoints (in meters).
            max_keypoints (int): Maximum number of keypoints to retain per valley line.
            find_window_cells (int): Number of cells to consider for curvature detection.
            feedback (QgsProcessingFeedback, optional): Optional feedback object for progress reporting/logging.

        Returns:
            GeoDataFrame: Detected keypoints as point geometries with metadata.
        """
        results = []

        # Validate smoothing window (must be odd)
        if smoothing_window % 2 == 0:
            smoothing_window += 1
            if feedback:
                feedback.pushInfo(f"[GetKeypoints] Smoothing window adjusted to {smoothing_window} (must be odd)")
            else:
                print(f"[GetKeypoints] Smoothing window adjusted to {smoothing_window} (must be odd)")
        if feedback:
            feedback.pushInfo(f"[GetKeypoints] Starting keypoint detection on {len(valley_lines)} valley lines...")
            feedback.setProgress(0)
            if feedback.isCanceled():
                feedback.reportError('Process cancelled by user at initialization.')
                raise RuntimeError('Process cancelled by user.')
        else:
            print(f"[GetKeypoints] Starting keypoint detection on {len(valley_lines)} valley lines...")

        # Read DTM raster using GDAL
        dtm_ds = gdal.Open(dtm_path, gdal.GA_ReadOnly)
        if dtm_ds is None:
            raise RuntimeError(f"Cannot open DTM raster: {dtm_path}.{self._get_gdal_error_message()}")
            
        try:
            # Get raster information
            geotransform = dtm_ds.GetGeoTransform()
            if geotransform is None:
                raise RuntimeError(f"Cannot get geotransform from DTM raster: {dtm_path}.{self._get_gdal_error_message()}")
            
            res = abs(geotransform[1])  # pixel width
            dtm_band = dtm_ds.GetRasterBand(1)
            
            # Auto-calculate find_window_distance based on pixel size
            processed_lines = 0
            skipped_lines = 0
            total_lines = len(valley_lines)
            total_keypoints = 0
            
            for idx, row in valley_lines.iterrows():
                line = row.geometry
                line_id = row.LINK_ID
                length = line.length
                # Sample at pixel resolution - use pixel size as sampling distance
                sampling_distance = res
                num_samples = max(int(length / sampling_distance), 2)  # At least 2 samples

                # Progress reporting for every line (or every 5 lines for large datasets)
                current_line = idx + 1
                progress_pct = int((current_line / total_lines) * 100)
                
                if feedback:
                    # Update progress bar for every line
                    feedback.setProgress(progress_pct)
                    if feedback.isCanceled():
                        feedback.reportError(f'Process cancelled by user while processing line {current_line}/{total_lines}.')
                        raise RuntimeError('Process cancelled by user.')
                    # Detailed info for smaller datasets or periodic updates for large datasets
                    if total_lines <= 20 or current_line % 5 == 0 or current_line == total_lines:
                        feedback.pushInfo(f"[GetKeypoints] Processing line {current_line}/{total_lines} ({progress_pct}%) - Line ID: {line_id}, Length: {length:.1f}m, Samples: {num_samples}")
                else:
                    if total_lines <= 20 or current_line % 5 == 0 or current_line == total_lines:
                        print(f"[GetKeypoints] Processing line {current_line}/{total_lines} ({progress_pct}%) - Line ID: {line_id}, Length: {length:.1f}m, Samples: {num_samples}")
                processed_lines += 1

                distances = np.linspace(0, length, num=num_samples)
                sample_points = [line.interpolate(d) for d in distances]
                coords = [(pt.x, pt.y) for pt in sample_points]
                
                # Convert world coordinates to pixel coordinates using utility function
                pixel_indices = TopoDrainCore._coords_to_pixel_indices(coords, geotransform)
                
                # Sample elevations using GDAL
                elevations = []
                for px, py in pixel_indices:
                    # Read elevation value at pixel location
                    try:
                        elevation_array = dtm_band.ReadAsArray(px, py, 1, 1)
                        if elevation_array is not None and elevation_array.size > 0:
                            elevations.append(float(elevation_array[0, 0]))
                        else:
                            # Use a default value if pixel is outside raster bounds
                            elevations.append(0.0)
                    except:
                        # Handle any GDAL errors
                        elevations.append(0.0)

                # Smooth elevation profile first
                elev_smooth = savgol_filter(elevations, smoothing_window, polyorder)
                
                # Calculate curvature (second derivative) directly from smoothed data using numpy gradient
                # This avoids double-smoothing while still working with clean data
                curvature = np.gradient(np.gradient(elev_smooth))
                
                # Alternative approaches:
                # Option 1 - Direct savgol on raw data (single smoothing): 
                # curvature = savgol_filter(elevations, smoothing_window, polyorder, deriv=2)
                # Option 2 - Double smoothing (may over-smooth):
                # curvature = savgol_filter(elev_smooth, smoothing_window, polyorder, deriv=2)

                # Find convex→concave transitions (keypoints)
                find_window = max(3, find_window_cells)  # At least 3 pixels, maybe later as input parameter? e.g. Nr. of cells?
                candidates = TopoDrainCore._find_inflection_candidates(curvature, window=find_window)

                # Sort and select strongest candidates
                sorted_candidates = sorted(candidates, key=lambda x: x[1], reverse=True)

                # Check for minimum distance between keypoints
                accepted = []
                for i, strength in sorted_candidates:
                    pt = sample_points[i]
                    if all(pt.distance(p[0]) >= min_distance for p in accepted):
                        accepted.append((pt, strength, i))
                    if len(accepted) >= max_keypoints:
                        break

                for rank, (pt, _, idx_pt) in enumerate(accepted, start=1):
                    results.append({
                        "geometry": Point(pt),
                        "VALLEY_ID": row["LINK_ID"],
                        "ELEV_INDEX": idx_pt,
                        "RANK": rank,
                        "CURVATURE": curvature[idx_pt]
                    })

                # Update keypoint count and provide feedback
                line_keypoints = len(accepted)
                total_keypoints += line_keypoints
                
                if feedback:
                    if line_keypoints > 0:
                        feedback.pushInfo(f"[GetKeypoints] Line {line_id}: found {line_keypoints} keypoints (total: {total_keypoints})")
                    else:
                        feedback.pushInfo(f"[GetKeypoints] Line {line_id}: no keypoints found")
                else:
                    if line_keypoints > 0:
                        print(f"[GetKeypoints] Line {line_id}: found {line_keypoints} keypoints (total: {total_keypoints})")
                    else:
                        print(f"[GetKeypoints] Line {line_id}: no keypoints found")
                        
        finally:
            dtm_ds = None  # Close GDAL dataset

        gdf = gpd.GeoDataFrame(results, geometry="geometry", crs=self.crs)

        if feedback:
            feedback.pushInfo(f"[GetKeypoints] Keypoint detection complete:")
            feedback.pushInfo(f"[GetKeypoints] - Total valley lines: {total_lines}")
            feedback.pushInfo(f"[GetKeypoints] - Processed lines: {processed_lines}")
            feedback.pushInfo(f"[GetKeypoints] - Skipped lines: {skipped_lines}")
            feedback.pushInfo(f"[GetKeypoints] - Total keypoints found: {len(gdf)}")
            feedback.setProgress(100)
            if feedback.isCanceled():
                feedback.reportError('Process cancelled by user at completion.')
                raise RuntimeError('Process cancelled by user.')
        else:
            print(f"[GetKeypoints] Keypoint detection complete: {len(gdf)} keypoints found from {processed_lines}/{total_lines} valley lines (skipped: {skipped_lines})")

        return gdf

    @staticmethod
    def _get_orthogonal_directions_start_points(
        barrier_raster_path: str,
        point: Point,
        line_geom: LineString,
        max_offset: int = 10
    ) -> tuple[Point, Point]:
        """
        Determine two start points to the left and right of a given point, orthogonal to an input line using GDAL.

        The function searches along the orthogonal direction from a given point until it finds
        a non-barrier cell in the provided raster.

        Args:
            barrier_raster_path (str): Path to binary raster with 1 = barrier, 0 = free (supports all GDAL formats in gdal_driver_mapping).
            point (Point): The reference point (typically a keypoint on a valley line).
            line_geom (LineString): Reference line geometry used to determine orientation, e.g. valley line.
            max_offset (int): Maximum number of cells to move outward when searching.

        Returns:
            tuple: (left_point, right_point), or (None, None) if no valid points found.
        """
        print(f"[GetOrthogonalDirectionsStartPoints] Checking point {point}, max_offset={max_offset}")
        
        # Read barrier raster using GDAL
        barrier_ds = gdal.Open(barrier_raster_path, gdal.GA_ReadOnly)
        if barrier_ds is None:
            print(f"[GetOrthogonalDirectionsStartPoints] Cannot open barrier raster: {barrier_raster_path}")
            return None, None
            
        try:
            # Get raster information
            rows = barrier_ds.RasterYSize
            cols = barrier_ds.RasterXSize
            geotransform = barrier_ds.GetGeoTransform()
            if geotransform is None:
                print(f"[GetOrthogonalDirectionsStartPoints] Cannot get geotransform from: {barrier_raster_path}")
                return None, None

            res = abs(geotransform[1])  # pixel width (assumes square pixels)

            # Read barrier mask data
            barrier_band = barrier_ds.GetRasterBand(1)
            barrier_mask = barrier_band.ReadAsArray()
            if barrier_mask is None:
                print(f"[GetOrthogonalDirectionsStartPoints] Cannot read barrier data from: {barrier_raster_path}")
                return None, None
                
        finally:
            barrier_ds = None  # Close dataset

        # Find nearest segment and compute tangent vector
        nearest_pt = nearest_points(point, line_geom)[1]
        coords = list(line_geom.coords)

        min_dist = float("inf")
        tangent = None
        for i in range(1, len(coords)):
            seg = LineString([coords[i - 1], coords[i]])
            dist = seg.distance(nearest_pt)
            if dist < min_dist:
                min_dist = dist
                dx = coords[i][0] - coords[i - 1][0]
                dy = coords[i][1] - coords[i - 1][1]
                norm = np.linalg.norm([dx, dy])
                if norm > 0:
                    tangent = np.array([dx, dy]) / norm

        if tangent is None:
            return None, None

        # Compute orthogonal direction vectors
        ortho_left = np.array([-tangent[1], tangent[0]])
        ortho_right = np.array([tangent[1], -tangent[0]])

        def find_valid_point(direction_vec):
            for i in range(1, max_offset + 1):
                offset = res * i
                test_x = point.x + direction_vec[0] * offset
                test_y = point.y + direction_vec[1] * offset

                # Convert world coordinates to pixel indices using GDAL geotransform
                px = int((test_x - geotransform[0]) / geotransform[1])
                py = int((test_y - geotransform[3]) / geotransform[5])

                # Bounds check
                if not (0 <= py < rows and 0 <= px < cols):
                    continue

                # barrier >= 1 means forbidden
                if barrier_mask[py, px] >= 1:
                    print(f"[GetOrthogonalDirectionsStartPoints] Still on barrier at offset {i}.")
                else:
                    new_point = Point(test_x, test_y)
                    print(f"[GetOrthogonalDirectionsStartPoints] Found valid point at offset {i}: {new_point.wkt}")
                    return new_point
                
            print("[GetOrthogonalDirectionsStartPoints] No valid point found within max_offset.")
            return None

        left_pt = find_valid_point(ortho_left)
        right_pt = find_valid_point(ortho_right)

        return left_pt, right_pt
        
    @staticmethod
    def _get_linedirection_start_point(
        barrier_raster_path: str,
        line_geom: LineString,
        max_offset: int = 10,
        reverse: bool = False
    ) -> Point:
        """
        Determine a start point in the proceeding direction of a given input line.

        The function searches from the endpoint of the line along its direction until it finds
        a non-barrier cell. If reverse=True, follows the line geometry backwards from the endpoint.
        For forward mode, prioritizes perpendicular direction to the local barrier orientation.

        Args:
            barrier_raster_path (str): Path to binary raster (GeoTIFF) with 1 = barrier, 0 = free.
            line_geom (LineString): Reference line geometry used to determine orientation, e.g. keyline.
            max_offset (int): Maximum number of cells to move outward when searching.
            reverse (bool): If True, follow line geometry backwards from endpoint, if False search forward.

        Returns:
            Point or None: The new start point beyond the barrier, or None if not found.
        """
        coords = list(line_geom.coords)
        if len(coords) >= 2:
            end_point = coords[-1]    # Always start from the endpoint
            ref_point = coords[-2]    # Reference point for direction
        else:
            raise ValueError("LineString must have at least two coordinates to determine direction.")
        
        print(f"[GetLinedirectionStartPoint]: Checking endpoint {end_point}, reverse={reverse}")
        print(f"[GetLinedirectionStartPoint] barrier_raster_path: {barrier_raster_path}")

        # Read barrier raster using GDAL
        barrier_ds = gdal.Open(barrier_raster_path, gdal.GA_ReadOnly)
        if barrier_ds is None:
            print(f"[GetLinedirectionStartPoint] Cannot open barrier raster: {barrier_raster_path}")
            return None
            
        try:
            # Get raster information
            rows = barrier_ds.RasterYSize
            cols = barrier_ds.RasterXSize
            geotransform = barrier_ds.GetGeoTransform()
            if geotransform is None:
                print(f"[GetLinedirectionStartPoint] Cannot get geotransform from: {barrier_raster_path}")
                return None

            res = abs(geotransform[1])  # pixel width (assumes square pixels)

            # Read barrier mask data
            barrier_band = barrier_ds.GetRasterBand(1)
            barrier_mask = barrier_band.ReadAsArray()
            if barrier_mask is None:
                print(f"[GetLinedirectionStartPoint] Cannot read barrier data from: {barrier_raster_path}")
                return None
                
        finally:
            barrier_ds = None  # Close dataset

        # Convert world coordinates to pixel indices using utility function
        pixel_coords = TopoDrainCore._coords_to_pixel_indices([end_point], geotransform)
        col_ep, row_ep = pixel_coords[0]
        print(f"[GetLinedirectionStartPoint] Endpoint raster index: row={row_ep}, col={col_ep}")
        
        # Bounds check for endpoint
        if not (0 <= row_ep < rows and 0 <= col_ep < cols):
            print("[GetLinedirectionStartPoint] Endpoint is outside raster bounds.")
            return None
            
        if not barrier_mask[row_ep, col_ep] >= 1:
            print("[GetLinedirectionStartPoint] No barrier at endpoint, returning None.")
            return None  # no barrier at endpoint, so no need to search

        if reverse:
                # For reverse, follow the line geometry backwards
                print("[GetLinedirectionStartPoint] Reverse mode: following line geometry backwards")
                
                # Get total line length and work backwards from endpoint
                total_length = line_geom.length
                
                def find_valid_point_along_line():
                    for i in range(1, max_offset + 1):
                        # Calculate distance to move back along the line
                        back_distance = res * i
                        
                        # Calculate position along line (from start = 0 to end = total_length)
                        # We want to go backwards from the end, so subtract from total_length
                        target_distance = max(0, total_length - back_distance)
                        
                        # Get point at this distance along the line
                        try:
                            test_point = line_geom.interpolate(target_distance)
                            test_x, test_y = test_point.x, test_point.y
                            
                            # Convert world coordinates to pixel indices using utility function
                            pixel_coords = TopoDrainCore._coords_to_pixel_indices([(test_x, test_y)], geotransform)
                            col_idx, row_idx = pixel_coords[0]
                            print(f"[GetLinedirectionStartPoint] Checking offset {i} along line: ({test_x}, {test_y}) -> row={row_idx}, col={col_idx}")

                            if not (0 <= row_idx < rows and 0 <= col_idx < cols):
                                print(f"[GetLinedirectionStartPoint] Offset {i} out of raster bounds.")
                                continue

                            if barrier_mask[row_idx, col_idx] > 0:
                                print(f"[GetLinedirectionStartPoint] Still on barrier at offset {i} along line.")
                            else:
                                new_point = Point(test_x, test_y)
                                print(f"[GetLinedirectionStartPoint] Found valid point at offset {i} along line: {new_point.wkt}")
                                return new_point

                        except Exception as e:
                            print(f"[GetLinedirectionStartPoint] Error interpolating at distance {target_distance}: {e}")
                            continue
                            
                        # If we've reached the start of the line, stop
                        if target_distance <= 0:
                            print("[GetLinedirectionStartPoint] Reached start of line without finding valid point.")
                            break
                    
                    print("[GetLinedirectionStartPoint] No valid point found following line backwards.")
                    return None
                
                new_pt = find_valid_point_along_line()
                
        else:
                # For forward, use mean direction of the last two line segments
                print("[GetLinedirectionStartPoint] Forward mode: using line direction")
                
                tangent = None
                
                # Use simple mean direction of last two line segments
                if tangent is None:
                    print("[GetLinedirectionStartPoint] Using mean direction of last two line segments")
                    
                    # Calculate tangent vectors for the last two segments (if available)
                    num_segments_to_use = min(2, len(coords) - 1)
                    tangent_vectors = []
                    
                    for i in range(num_segments_to_use):
                        seg_end_idx = len(coords) - 1 - i  # Start from the end
                        seg_start_idx = seg_end_idx - 1
                        
                        if seg_start_idx >= 0:
                            dx = coords[seg_end_idx][0] - coords[seg_start_idx][0]
                            dy = coords[seg_end_idx][1] - coords[seg_start_idx][1]
                            
                            segment_length = np.sqrt(dx*dx + dy*dy)
                            
                            if segment_length > 0:
                                # Normalize to unit vector
                                unit_vector = np.array([dx, dy]) / segment_length
                                tangent_vectors.append(unit_vector)
                    
                    if not tangent_vectors:
                        print("[GetLinedirectionStartPoint] No valid segments found for tangent calculation.")
                        return None
                    
                    # Calculate simple mean of tangent vectors (unweighted average)
                    if len(tangent_vectors) > 0:
                        mean_tangent = np.mean(tangent_vectors, axis=0)
                        norm = np.linalg.norm(mean_tangent)
                        
                        if norm > 0:
                            tangent = mean_tangent / norm
                            print(f"[GetLinedirectionStartPoint] Mean tangent vector from {len(tangent_vectors)} segments: {tangent}")
                        else:
                            print("[GetLinedirectionStartPoint] Zero-length mean tangent vector, cannot proceed.")
                            return None
                    else:
                        print("[GetLinedirectionStartPoint] No tangent vectors calculated.")
                        return None

                def find_valid_point_forward():
                    for i in range(1, max_offset + 1):
                        offset = res * i
                        test_x = end_point[0] + tangent[0] * offset
                        test_y = end_point[1] + tangent[1] * offset

                        # Convert world coordinates to pixel indices using utility function
                        pixel_coords = TopoDrainCore._coords_to_pixel_indices([(test_x, test_y)], geotransform)
                        col_idx, row_idx = pixel_coords[0]
                        print(f"[GetLinedirectionStartPoint] Checking offset {i}: ({test_x}, {test_y}) -> row={row_idx}, col={col_idx}")

                        if not (0 <= row_idx < rows and 0 <= col_idx < cols):
                            print(f"[GetLinedirectionStartPoint] Offset {i} out of raster bounds.")
                            continue

                        # barrier >= 1 means forbidden
                        if barrier_mask[row_idx, col_idx] >= 1:
                            print(f"[GetLinedirectionStartPoint] Still on barrier at offset {i}.")
                        else:
                            new_point = Point(test_x, test_y)
                            print(f"[GetLinedirectionStartPoint] Found valid point at offset {i}: {new_point.wkt}")
                            return new_point

                    print("[GetLinedirectionStartPoint] No valid point found beyond barrier.")
                    return None

                new_pt = find_valid_point_forward()

        if new_pt:
            return new_pt
        else:
            return None

    @staticmethod
    def _reverse_line_direction(input_geometry):
        """
        Reverse the coordinate direction of LineString geometries.
        
        Can handle single LineString or GeoDataFrame with multiple LineString geometries.
        
        Args:
            input_geometry: Either a single LineString or a GeoDataFrame with LineString geometries
            
        Returns:
            Same type as input but with reversed coordinate direction:
            - LineString: Returns reversed LineString
            - GeoDataFrame: Returns GeoDataFrame with all LineString geometries reversed
        """
        if isinstance(input_geometry, LineString):
            # Handle single LineString
            if hasattr(input_geometry, 'coords') and len(input_geometry.coords) >= 2:
                reversed_coords = list(input_geometry.coords)[::-1]
                return LineString(reversed_coords)
            else:
                return input_geometry
                
        elif hasattr(input_geometry, 'geometry') and hasattr(input_geometry, 'iterrows'):
            # Handle GeoDataFrame
            result_gdf = input_geometry.copy()
            reversed_geometries = []
            
            for _, row in input_geometry.iterrows():
                line_geom = row.geometry
                if isinstance(line_geom, LineString) and hasattr(line_geom, 'coords') and len(line_geom.coords) >= 2:
                    reversed_coords = list(line_geom.coords)[::-1]
                    reversed_geometries.append(LineString(reversed_coords))
                else:
                    reversed_geometries.append(line_geom)
                    
            result_gdf.geometry = reversed_geometries
            return result_gdf
            
        else:
            # Return input unchanged if not a recognized type
            return input_geometry

    @staticmethod
    def _create_slope_cost_raster(
        dtm_path: str,
        start_point: Point,
        output_cost_raster_path: str,
        slope: float = 0.01,
        barrier_raster_path: str = None,
        penalty_exp: float = 2.0
    ) -> str:
        """
        Create a raster with cost values based on deviation from desired slope using GDAL.
        You can now set penalty_exp>1 to punish larger deviations more heavily.

        Args:
            dtm_path (str): Path to input DTM raster (supports all GDAL formats in gdal_driver_mapping).
            start_point (Point): Starting point of the constant slope line.
            output_cost_raster_path (str): Path to output cost raster (format determined by file extension).
            slope (float): Desired slope (1% downhill = 0.01).
            barrier_raster_path (str): Path to a binary raster of barriers (1=barrier).
            penalty_exp (float): Exponent on the absolute deviation (>=1) of slope. 2.0 => quadratic penalty --> as higher the exponent as stronger penalty for larger deviations.

        Returns:
            str: Path to the written cost raster.
        """
        # Read DTM raster using GDAL
        dtm_ds = gdal.Open(dtm_path, gdal.GA_ReadOnly)
        if dtm_ds is None:
            raise RuntimeError(f"Cannot open DTM raster: {dtm_path}")
            
        try:
            # Get raster information
            rows = dtm_ds.RasterYSize
            cols = dtm_ds.RasterXSize
            geotransform = dtm_ds.GetGeoTransform()
            if geotransform is None:
                raise RuntimeError(f"Cannot get geotransform from DTM raster: {dtm_path}")
            
            projection = dtm_ds.GetProjection()
            srs = dtm_ds.GetSpatialRef()
            
            # Read DTM data
            dtm_band = dtm_ds.GetRasterBand(1)
            dtm = dtm_band.ReadAsArray().astype(np.float32)
            if dtm is None:
                raise RuntimeError(f"Cannot read DTM data from: {dtm_path}")
            
            # Handle NoData values
            nodata = dtm_band.GetNoDataValue()
            if nodata is not None:
                dtm[dtm == nodata] = np.nan

            # Convert start point coordinates to pixel indices using utility function
            pixel_coords = TopoDrainCore._coords_to_pixel_indices([start_point.coords[0]], geotransform)
            key_col, key_row = pixel_coords[0]
            
            # Bounds check for start point
            if not (0 <= key_row < rows and 0 <= key_col < cols):
                raise ValueError(f"Start point {start_point} is outside raster bounds")

            # Create index arrays for distance calculation
            rr, cc = np.indices((rows, cols))
            
            # Calculate elevation difference and horizontal distance
            dz = dtm - dtm[key_row, key_col]
            res = abs(geotransform[1])  # pixel width
            dist = np.hypot(rr - key_row, cc - key_col) * res
            expected_dz = -dist * slope

            # Calculate linear deviation
            deviation = np.abs(dz - expected_dz)

            # Apply exponentiation for stronger penalty
            cost = deviation ** penalty_exp

            # Enforce NoData areas with high cost
            cost[np.isnan(dtm)] = 1e6

            # Read barrier mask from raster if provided
            if barrier_raster_path is not None:
                barrier_ds = gdal.Open(barrier_raster_path, gdal.GA_ReadOnly)
                if barrier_ds is None:
                    raise RuntimeError(f"Cannot open barrier raster: {barrier_raster_path}")
                    
                try:
                    barrier_band = barrier_ds.GetRasterBand(1)
                    barrier_mask = barrier_band.ReadAsArray()
                    if barrier_mask is None:
                        raise RuntimeError(f"Cannot read barrier data from: {barrier_raster_path}")
                    
                    if barrier_mask.shape != cost.shape:
                        raise ValueError("Barrier raster shape does not match DTM shape.")
                    
                    barrier_mask = barrier_mask.astype(bool)
                    if np.any(barrier_mask):
                        print("[TopoDrainCore] Applying barrier mask to cost raster.")
                        # Set cost to a very high value where barriers are present
                        cost[barrier_mask] = 1e6
                        
                finally:
                    barrier_ds = None  # Close barrier dataset

            # Zero cost at the true start point
            cost[key_row, key_col] = 0
            
        finally:
            dtm_ds = None  # Close DTM dataset

        # Use GTiff driver since we know this is an internal function that creates .tif files
        driver_name = 'GTiff'
        
        # Create output raster using GDAL with best practices
        driver = gdal.GetDriverByName(driver_name)
        if driver is None:
            raise RuntimeError(f"Cannot create GDAL driver for: {driver_name}")
            
        creation_options = [
            'COMPRESS=LZW',
            'TILED=YES',
            'BIGTIFF=IF_SAFER'
        ]
        
        out_ds = driver.Create(output_cost_raster_path, cols, rows, 1, gdal.GDT_Float32, 
                              options=creation_options)
        if out_ds is None:
            raise RuntimeError(f"Cannot create output raster: {output_cost_raster_path}")
            
        try:
            # Set spatial reference and geotransform
            out_ds.SetGeoTransform(geotransform)
            
            # Set complete spatial reference system
            if srs is not None:
                out_ds.SetSpatialRef(srs)
            elif projection:
                # Fallback to projection string if SRS not available
                out_ds.SetProjection(projection)
            
            # Write cost data
            out_band = out_ds.GetRasterBand(1)
            out_band.WriteArray(cost.astype(np.float32))
            out_band.SetNoDataValue(1e6)
            
        finally:
            out_ds = None  # Close output dataset

        return output_cost_raster_path

    @staticmethod
    def _create_source_raster(
        reference_raster_path: str,
        source_point: Point,
        output_source_raster_path: str,
        ) -> str:
        """
        Create a binary raster marking the source cell (value = 1) based on a given Point using GDAL.
        All other cells are set to 0.

        Args:
            reference_raster_path (str): Path to the reference raster (supports all GDAL formats in gdal_driver_mapping).
            source_point (Point): Shapely Point marking the source location.
            output_source_raster_path (str): Path to output binary raster.

        Returns:
            str: Path to the saved binary raster file.
        """
        # Read reference raster using GDAL
        ref_ds = gdal.Open(reference_raster_path, gdal.GA_ReadOnly)
        if ref_ds is None:
            raise RuntimeError(f"Cannot open reference raster: {reference_raster_path}")
            
        try:
            # Get raster information
            rows = ref_ds.RasterYSize
            cols = ref_ds.RasterXSize
            geotransform = ref_ds.GetGeoTransform()
            if geotransform is None:
                raise RuntimeError(f"Cannot get geotransform from reference raster: {reference_raster_path}")
            
            projection = ref_ds.GetProjection()
            srs = ref_ds.GetSpatialRef()
            
            # Convert source point coordinates to pixel indices using utility function
            pixel_coords = TopoDrainCore._coords_to_pixel_indices([source_point.coords[0]], geotransform)
            col, row = pixel_coords[0]
            
            # Bounds check for source point
            if not (0 <= row < rows and 0 <= col < cols):
                raise ValueError(f"Source point {source_point} is outside the bounds of the reference raster.")

            # Create binary data array
            data = np.zeros((rows, cols), dtype=np.uint8)
            data[row, col] = 1
            
        finally:
            ref_ds = None  # Close reference dataset

        # Use GTiff driver since we know this is an internal function that creates .tif files
        driver_name = 'GTiff'
        
        # Create output raster using GDAL with best practices
        driver = gdal.GetDriverByName(driver_name)
        if driver is None:
            raise RuntimeError(f"Cannot create GDAL driver for: {driver_name}")
            
        creation_options = [
            'COMPRESS=LZW',
            'TILED=YES',
            'BIGTIFF=IF_SAFER'
        ]
        
        out_ds = driver.Create(output_source_raster_path, cols, rows, 1, gdal.GDT_Byte, 
                              options=creation_options)
        if out_ds is None:
            raise RuntimeError(f"Cannot create output raster: {output_source_raster_path}")
            
        try:
            # Set spatial reference and geotransform
            out_ds.SetGeoTransform(geotransform)
            
            # Set complete spatial reference system
            if srs is not None:
                out_ds.SetSpatialRef(srs)
            elif projection:
                # Fallback to projection string if SRS not available
                out_ds.SetProjection(projection)
            
            # Write source data
            out_band = out_ds.GetRasterBand(1)
            out_band.WriteArray(data)
            out_band.SetNoDataValue(0)
            
        finally:
            out_ds = None  # Close output dataset

        return output_source_raster_path

    @staticmethod
    def _select_best_destination_cell(
        accum_raster_path: str,
        destination_raster_path: str,
        best_destination_raster_path: str
    ) -> tuple[str, Point]:
        """
        Select the best destination cell from a binary destination raster based on
        minimum accumulated cost, and write it as a single-cell binary raster using GDAL.

        Args:
            accum_raster_path (str): Path to the cost accumulation raster (supports all GDAL formats in gdal_driver_mapping).
            destination_raster_path (str): Path to the binary destination raster (1 = destination, 0 = background).
            best_destination_raster_path (str): Path to output raster with only the best cell marked.

        Returns:
            tuple[str, Point]: Tuple of (path to the output best destination raster, Point with spatial coordinates of the best destination cell).
        """
        # Read accumulation raster using GDAL
        acc_ds = gdal.Open(accum_raster_path, gdal.GA_ReadOnly)
        if acc_ds is None:
            raise RuntimeError(f"Cannot open accumulation raster: {accum_raster_path}")
            
        try:
            # Get accumulation data
            acc_band = acc_ds.GetRasterBand(1)
            acc_data = acc_band.ReadAsArray()
            if acc_data is None:
                raise RuntimeError(f"Cannot read accumulation data from: {accum_raster_path}")
                
            # Get spatial information for coordinate conversion
            geotransform = acc_ds.GetGeoTransform()
            if geotransform is None:
                raise RuntimeError(f"Cannot get geotransform from accumulation raster: {accum_raster_path}")
                
        finally:
            acc_ds = None  # Close accumulation dataset
            
        # Read destination raster using GDAL
        dest_ds = gdal.Open(destination_raster_path, gdal.GA_ReadOnly)
        if dest_ds is None:
            raise RuntimeError(f"Cannot open destination raster: {destination_raster_path}")
            
        try:
            # Get raster information
            rows = dest_ds.RasterYSize
            cols = dest_ds.RasterXSize
            dest_geotransform = dest_ds.GetGeoTransform()
            if dest_geotransform is None:
                raise RuntimeError(f"Cannot get geotransform from destination raster: {destination_raster_path}")
            
            projection = dest_ds.GetProjection()
            srs = dest_ds.GetSpatialRef()
            
            # Get destination data
            dest_band = dest_ds.GetRasterBand(1)
            dest_data = dest_band.ReadAsArray()
            if dest_data is None:
                raise RuntimeError(f"Cannot read destination data from: {destination_raster_path}")
                
        finally:
            dest_ds = None  # Close destination dataset

        # Consider only destination cells (where value == 1)
        mask = (dest_data == 1)
        acc_masked = np.where(mask, acc_data, np.nan)

        # Identify cell with minimum accumulated cost
        if np.all(np.isnan(acc_masked)):
            raise RuntimeError("No valid destination cell found.")

        # Find the minimum value and its index
        min_val = np.nanmin(acc_masked)
        min_indices = np.where(acc_masked == min_val)
        # Take the first occurrence if multiple
        row, col = min_indices[0][0], min_indices[1][0]

        # Calculate spatial coordinates using GDAL geotransform
        coords = TopoDrainCore._pixel_indices_to_coords([row], [col], geotransform)
        x, y = coords[0]
        best_destination_point = Point(x, y)

        # Create output raster marking only the best cell
        best_dest = np.zeros_like(dest_data, dtype=np.uint8)
        best_dest[row, col] = 1

        # Use GTiff driver since we know this is an internal function that creates .tif files
        driver_name = 'GTiff'
        
        # Create output raster using GDAL with best practices
        driver = gdal.GetDriverByName(driver_name)
        if driver is None:
            raise RuntimeError(f"Cannot create GDAL driver for: {driver_name}")
            
        creation_options = [
            'COMPRESS=LZW',
            'TILED=YES',
            'BIGTIFF=IF_SAFER'
        ]
        
        out_ds = driver.Create(best_destination_raster_path, cols, rows, 1, gdal.GDT_Byte, 
                              options=creation_options)
        if out_ds is None:
            raise RuntimeError(f"Cannot create output raster: {best_destination_raster_path}")
            
        try:
            # Set spatial reference and geotransform
            out_ds.SetGeoTransform(dest_geotransform)
            
            # Set complete spatial reference system
            if srs is not None:
                out_ds.SetSpatialRef(srs)
            elif projection:
                # Fallback to projection string if SRS not available
                out_ds.SetProjection(projection)
            
            # Write best destination data
            out_band = out_ds.GetRasterBand(1)
            out_band.WriteArray(best_dest)
            out_band.SetNoDataValue(0)
            
        finally:
            out_ds = None  # Close output dataset

        return best_destination_raster_path, best_destination_point

    def _analyze_slope_deviation_and_cut(
        self,
        line: LineString,
        start_point: Point,
        expected_slope: float,
        slope_deviation_threshold: float
        ) -> Point:
        """
        Analyse the slope deviation of a line compared to the expected slope.
        This method compares the assumed slope (based on euclidean distance) with the 
        real slope (based on actual line distance), without using actual height values.
        
        The least cost algorithm assumes: dz/euclidean_distance = expected_slope
        But the real line gives us: dz/real_distance = actual_slope
        
        Since dz is the same, we can derive: actual_slope = expected_slope * (euclidean_distance / real_distance)
        Cut the line when actual_slope deviates too much from expected_slope.

        Args:
            line (LineString): The line to analyse.
            start_point (Point): Original start point for reference.
            expected_slope (float): Desired slope (e.g., 0.01 for 1% downhill) (assumed euclidean slope).
            slope_deviation_threshold (float): Maximum allowed relative deviation (e.g., 0.1 for 10%).
            
        Returns:
            Point: The point where cutting should occur, or None if no cutting needed.
        """
        # Sample points along the line at regular intervals
        line_length = line.length
        num_samples = max(int(line_length / 5.0), 10)  # Sample every 5 meters or at least 10 points
        
        distances_along_line = np.linspace(0, line_length, num_samples)
        sample_points = [line.interpolate(d) for d in distances_along_line]
        end_point = sample_points[-1]

        print(f"[AnalyzeSlopeDeviation] Analyzing {num_samples} points along {line_length:.1f}m line")
        print(f"[AnalyzeSlopeDeviation] Start point: {start_point}, End point: {end_point}")
        print(f"[AnalyzeSlopeDeviation] Expected slope (euclidean): {expected_slope:.4f}")
        print(f"[AnalyzeSlopeDeviation] Slope deviation threshold: {slope_deviation_threshold:.2f}")

        # Calculate slope deviations based on distance ratios
        for i, (line_distance, point) in enumerate(zip(distances_along_line, sample_points)):
            if line_distance == 0:
                continue  # Skip start point
                
            # Calculate euclidean distance from start point
            euclidean_distance = start_point.distance(point)
            
            if euclidean_distance == 0:
                continue  # Skip if no distance
                
            # Calculate actual slope based on distance ratio
            # actual_slope = expected_slope * (euclidean_distance / real_distance)
            actual_slope = expected_slope * (euclidean_distance / line_distance)
            
            # Calculate relative deviation from expected slope
            if expected_slope != 0:
                slope_deviation_ratio = actual_slope / expected_slope
                relative_deviation = abs(slope_deviation_ratio - 1.0)
            else:
                relative_deviation = 0
            
            print(f"[AnalyzeSlopeDeviation] Point {i}: line_dist={line_distance:.1f}m, "
                  f"euclidean_dist={euclidean_distance:.1f}m, "
                  f"expected_slope={expected_slope:.4f}, actual_slope={actual_slope:.4f}, "
                  f"deviation={relative_deviation:.3f}")

            if relative_deviation > slope_deviation_threshold:
                print(f"[AnalyzeSlopeDeviation] Slope deviation {relative_deviation:.3f} exceeds threshold "
                      f"{slope_deviation_threshold:.3f} at line distance {line_distance:.1f}m")
                print(f"[AnalyzeSlopeDeviation] Expected slope: {expected_slope:.4f}, Actual slope: {actual_slope:.4f}")
                return point

        # No cutting needed - line maintains acceptable slope deviation
        print(f"[AnalyzeSlopeDeviation] Line maintains acceptable slope deviation")
        return None
    

    def _cut_line_at_point(self, line: LineString, cut_point: Point) -> LineString:
        """
        Cut a line at the specified point, returning the portion from start to cut point.
        
        Args:
            line (LineString): The line to cut.
            cut_point (Point): The point where to cut the line.
            
        Returns:
            LineString: The line segment from start to cut point.
        """
        try:
            # Find the distance along the line to the cut point
            cut_distance = line.project(cut_point)
            print(f"[CutLineAtPoint] Cutting line ({line.length:.2f}m) at distance {cut_distance:.2f}m from start to point {cut_point}")

            # Create a new line from start to cut point
            cut_line = substring(line, 0, cut_distance)

            # Ensure the cut line ends exactly at the cut point
            if isinstance(cut_line, LineString) and len(cut_line.coords) >= 2:
                coords = list(cut_line.coords)
                coords[-1] = (cut_point.x, cut_point.y)  # Replace last coordinate with exact cut point
                return LineString(coords)
            else:
                return line  # Fallback to original line if cutting failed
                
        except Exception as e:
            print(f"[CutLineAtPoint] Error cutting line: {e}")
            return line  # Fallback to original line

    def _get_constant_slope_line(
        self,
        dtm_path: str,
        start_point: Point,
        destination_raster_path: str,
        slope: float = 0.01,
        barrier_raster_path: str = None,
        slope_deviation_threshold: float = 0.2,
        max_iterations_slope: int = 20,
        feedback=None
    ) -> LineString:
        """
        Trace lines with constant slope starting from a given point using an iterative approach.
        
        This function traces a line, checks where the line distance deviates too much from 
        the Euclidean distance from start point, cuts the line at that point, and continues 
        from there in subsequent iterations.

        Args:
            dtm_path (str): Path to input DTM raster (supports all GDAL formats in gdal_driver_mapping).
            start_point (Point): Starting point of the constant slope line.
            destination_raster_path (str): Path to the binary raster indicating destination cells (1 = destination).
            slope (float): Desired slope for the line (e.g., 0.01 for 1% downhill or -0.01 for uphill).
            barrier_raster_path (str): Optional path to a binary raster of cells that should not be crossed (1 = barrier).
            slope_deviation_threshold (float): Maximum allowed relative deviation from expected slope (0.0-1.0, e.g., 0.2 for 20% deviation before line cutting). Default 0.2.
            max_iterations_slope (int): Maximum number of iterations for line refinement.
            feedback (QgsProcessingFeedback, optional): Optional feedback object for progress reporting.

        Returns:
            LineString: Refined constant slope path as a Shapely LineString, or None if no path found.
        """
        if self.wbt is None:
            raise RuntimeError("WhiteboxTools not initialized. Check WhiteboxTools configuration: QGIS settings -> Options -> Processing -> Provider -> WhiteboxTools -> WhiteboxTools executable.")

        print(f"[GetConstantSlopeLine] Starting constant slope line tracing")
        if feedback:
            feedback.pushInfo(f"[GetConstantSlopeLine] Starting constant slope line tracing")
            feedback.pushInfo(f"*for more information see in Python Console")

        print(f"[GetConstantSlopeLine] destination raster path: {destination_raster_path}")
        print(f"[GetConstantSlopeLine] barrier raster path: {barrier_raster_path}")
        print(f"[GetConstantSlopeLine] slope: {slope}, max_iterations_slope: {max_iterations_slope}, slope_deviation_threshold: {slope_deviation_threshold}")

        current_start_point = start_point
        accumulated_line_coords = []
        
        for iteration in range(max_iterations_slope):
            # Check for cancellation at the start of each iteration
            if feedback and feedback.isCanceled():
                feedback.reportError("Operation cancelled by user")
                raise RuntimeError("Operation cancelled by user")
            
            print(f"[GetConstantSlopeLine] Iteration {iteration + 1}/{max_iterations_slope}")
            if feedback:
                feedback.pushInfo(f"[GetConstantSlopeLine] Iteration {iteration + 1}/{max_iterations_slope}")
            
            # --- Temporary file paths ---
            cost_raster_path = os.path.join(self.temp_directory, f"cost_iter_{iteration}.tif")
            source_raster_path = os.path.join(self.temp_directory, f"source_iter_{iteration}.tif")
            accum_raster_path = os.path.join(self.temp_directory, f"accum_iter_{iteration}.tif")
            backlink_raster_path = os.path.join(self.temp_directory, f"backlink_iter_{iteration}.tif")
            best_destination_raster_path = os.path.join(self.temp_directory, f"destination_best_iter_{iteration}.tif")
            pathway_raster_path = os.path.join(self.temp_directory, f"pathway_iter_{iteration}.tif")
            pathway_vector_path = os.path.join(self.temp_directory, f"pathway_iter_{iteration}.shp")

            print(f"[GetConstantSlopeLine] Create cost slope raster for iteration {iteration + 1}")
            # --- Create cost raster ---
            cost_raster_path = TopoDrainCore._create_slope_cost_raster(
                dtm_path=dtm_path,
                start_point=current_start_point,
                output_cost_raster_path=cost_raster_path,
                slope=slope,
                barrier_raster_path=barrier_raster_path
            )

            print(f"[GetConstantSlopeLine] Create source raster for iteration {iteration + 1}")
            # --- Create source raster ---
            source_raster_path = TopoDrainCore._create_source_raster(
                reference_raster_path=dtm_path,
                source_point=current_start_point,
                output_source_raster_path=source_raster_path
            )

            print(f"[GetConstantSlopeLine] Starting cost-distance analysis for iteration {iteration + 1}")
            # --- Run cost-distance analysis ---
            try:
                ret = self._execute_wbt(
                    'cost_distance',
                    feedback=feedback,
                    report_progress=False,  # Don't override main progress bar
                    source=source_raster_path,
                    cost=cost_raster_path,
                    out_accum=accum_raster_path,
                    out_backlink=backlink_raster_path
                )
                
                if ret != 0 or not os.path.exists(accum_raster_path) or not os.path.exists(backlink_raster_path):
                    raise RuntimeError(f"Cost distance analysis failed in iteration {iteration + 1}: WhiteboxTools returned {ret}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during cost-distance analysis.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"Cost distance analysis failed in iteration {iteration + 1}: {e}")

            # Check for cancellation after cost-distance analysis
            if feedback and feedback.isCanceled():
                feedback.reportError("Operation cancelled by user")
                raise RuntimeError("Operation cancelled by user")

            print(f"[GetConstantSlopeLine] Selecting best destination cell for iteration {iteration + 1}")
            # --- Select best destination cell ---
            best_destination_raster_path, best_destination_point = TopoDrainCore._select_best_destination_cell(
                accum_raster_path=accum_raster_path,
                destination_raster_path=destination_raster_path,
                best_destination_raster_path=best_destination_raster_path
            )

            print(f"[GetConstantSlopeLine] Tracing least-cost pathway for iteration {iteration + 1}")
            # --- Trace least-cost pathway ---
            try:
                ret = self._execute_wbt(
                    'cost_pathway',
                    feedback=feedback,
                    report_progress=False,  # Don't override main progress bar
                    destination=best_destination_raster_path,
                    backlink=backlink_raster_path,
                    output=pathway_raster_path
                )
                
                if ret != 0 or not os.path.exists(pathway_raster_path):
                    raise RuntimeError(f"Cost pathway analysis failed in iteration {iteration + 1}")
            except Exception as e:
                # Check if cancellation was the cause
                if feedback and feedback.isCanceled():
                    feedback.reportError("Process cancelled by user during pathway tracing.")
                    raise RuntimeError('Process cancelled by user.')
                raise RuntimeError(f"Cost pathway analysis failed in iteration {iteration + 1}: {e}")

            # Check for cancellation after pathway analysis
            if feedback and feedback.isCanceled():
                feedback.reportError("Operation cancelled by user")
                raise RuntimeError("Operation cancelled by user")

            # --- Set correct NoData value for pathway raster --- ## Noch prüfen, ob das notwendig ist
            # Read NoData value from backlink raster using GDAL
            backlink_ds = gdal.Open(backlink_raster_path, gdal.GA_ReadOnly)
            if backlink_ds is not None:
                try:
                    backlink_band = backlink_ds.GetRasterBand(1)
                    nodata_value = backlink_band.GetNoDataValue()
                finally:
                    backlink_ds = None  # Close dataset
                
                # Set NoData value for pathway raster using GDAL
                pathway_ds = gdal.Open(pathway_raster_path, gdal.GA_Update)
                if pathway_ds is not None:
                    try:
                        pathway_band = pathway_ds.GetRasterBand(1)
                        if nodata_value is not None:
                            pathway_band.SetNoDataValue(nodata_value)
                    finally:
                        pathway_ds = None  # Close dataset

            print(f"[GetConstantSlopeLine] Converting pathway raster to LineString for iteration {iteration + 1}")
            # --- Convert to LineString ---
            line_segment = self._raster_to_linestring_wbt(
                pathway_raster_path, 
                snap_to_start_point=current_start_point, 
                snap_to_endpoint=best_destination_point, 
                output_vector_path=pathway_vector_path,
                feedback=feedback
            )

            if line_segment is None:
                print(f"[GetConstantSlopeLine] No valid line found in iteration {iteration + 1}")
                break

            # --- Analyze distance deviation and find cut point ---
            print(f"[GetConstantSlopeLine] Analyzing slope deviation for iteration {iteration + 1}")
            cut_point = self._analyze_slope_deviation_and_cut(
                line=line_segment, 
                start_point=current_start_point, 
                expected_slope=slope,
                slope_deviation_threshold=slope_deviation_threshold
                )

            # Check if the line segment reaches the destination
            if cut_point:
                # Read the destination raster and get the value at the cut point using GDAL
                dest_ds = gdal.Open(destination_raster_path, gdal.GA_ReadOnly)
                if dest_ds is not None:
                    try:
                        # Get raster information
                        rows = dest_ds.RasterYSize
                        cols = dest_ds.RasterXSize
                        geotransform = dest_ds.GetGeoTransform()
                        
                        if geotransform is not None:
                            # Convert cut point coordinates to pixel indices using utility function
                            pixel_coords = TopoDrainCore._coords_to_pixel_indices([cut_point.coords[0]], geotransform)
                            col, row = pixel_coords[0]
                            
                            if 0 <= row < rows and 0 <= col < cols:
                                # Read destination data
                                dest_band = dest_ds.GetRasterBand(1)
                                dest_data = dest_band.ReadAsArray()
                                if dest_data is not None:
                                    dest_value = dest_data[row, col]
                                else:
                                    dest_value = None
                            else:
                                dest_value = None
                        else:
                            dest_value = None
                    finally:
                        dest_ds = None  # Close dataset
                else:
                    dest_value = None
                    
                if dest_value == 1:
                    print(f"[GetConstantSlopeLine] Cut point {cut_point} is a destination cell")
                    reached_destination = True
                else:
                    reached_destination = False

            # Check if last iteration reached
            last_iteration = (iteration == max_iterations_slope - 1)
            
            if last_iteration and cut_point is not None:
                warnings.warn(f"[GetConstantSlopeLine] Warning: Last iteration reached without finding fully valid line segment")
                if feedback:
                    feedback.pushWarning(f"[GetConstantSlopeLine] Warning: Last iteration reached without finding fully valid line segment")

            if last_iteration or cut_point is None or reached_destination:
                # If we are at the last iteration or reached destination, we can wan to add fully line segement instead of doing another iteration
                if accumulated_line_coords:
                    # Skip first coordinate to avoid duplication
                    accumulated_line_coords.extend(line_segment.coords[1:])
                else:
                    accumulated_line_coords.extend(line_segment.coords)
                break

            else:
                # Cut the line at the identified point
                cut_line = self._cut_line_at_point(line_segment, cut_point)
                if cut_line:
                    if accumulated_line_coords:
                        # Skip first coordinate to avoid duplication
                        accumulated_line_coords.extend(cut_line.coords[1:])
                    else:
                        accumulated_line_coords.extend(cut_line.coords)

                    # Store the cut point for the next iteration
                    current_start_point = cut_point
                    iteration += 1
                    print(f"[GetConstantSlopeLine] Continuing from cut point: {cut_point}")
                else:
                    # If cutting failed, use the whole segment
                    warnings.warn(f"[GetConstantSlopeLine] Warning: Cutting failed, using whole line segment")
                    if feedback:
                        feedback.pushWarning(f"[GetConstantSlopeLine] Warning: Cutting failed, using whole line segment")
                    if accumulated_line_coords:
                        # Skip first coordinate to avoid duplication
                        accumulated_line_coords.extend(line_segment.coords[1:])
                    else:
                        accumulated_line_coords.extend(line_segment.coords)
                    break

        # --- Combine all segments into final line ---
        if not accumulated_line_coords or len(accumulated_line_coords) < 2:
            warnings.warn("[GetConstantSlopeLine] Warning: No valid line segments could be extracted.")
            if feedback:
                feedback.pushWarning("[GetConstantSlopeLine] Warning: No valid line segments could be extracted.")                
            return None

        final_line = LineString(accumulated_line_coords)

        print("[GetConstantSlopeLine] Smoothing the resulting line")
        # --- Optional smoothing ---
        final_line = TopoDrainCore._smooth_linestring(final_line, sigma=1.0)

        print(f"[GetConstantSlopeLine] Finished processing") 
        if feedback:
            feedback.pushInfo(f"[GetConstantSlopeLine] Finished processing")

        return final_line

    def _get_iterative_constant_slope_line(
        self,
        dtm_path: str,
        start_point: Point,
        destination_raster_path: str,
        slope: float,
        barrier_raster_path: str,
        initial_barrier_value: int = None,
        max_iterations_barrier: int = 10,
        slope_deviation_threshold: float = 0.2,
        max_iterations_slope: int = 20,
        feedback=None
    ) -> LineString:
        """
        Trace lines with constant slope starting from a given point using a cost-distance approach based on slope deviation.
        Unlike _get_constant_slope_line, this function allows barriers to act as temporary destinations.
        The barrier raster needs in this case different values (1, 2, ...) for different barrier features.

        This function creates a cost raster that penalizes deviation from the desired slope,
        runs a least-cost-path analysis using WhiteboxTools, and returns the resulting line.

        Args:
            dtm_path (str): Path to the digital terrain model (GeoTIFF).
            start_point (Point): Starting point of the constant slope line (not on a barrier, this is handled in get_constant_slope_lines).
            destination_raster_path (str): Path to the binary raster indicating destination cells (1 = destination).
            slope (float): Desired slope for the line (e.g., 0.01 for 1% downhill or -0.01 for uphill).
            barrier_raster_path (str): Path to a raster of cells that should not be crossed (different barriers have unique values 1, 2, ...).
            initial_barrier_value (int, optional): Initial barrier value to start from. Default None.
            max_iterations_barrier (int): Maximum number of iterations for iterative tracing (nr. of times barriers used as temporary destinations). Default 10.
            slope_deviation_threshold (float): Maximum allowed relative deviation from expected slope (0.0-1.0, e.g., 0.2 for 20% deviation before line cutting).
            feedback (QgsProcessingFeedback, optional): Optional feedback object for progress reporting.

        Returns:
            LineString: Least-cost slope path as a Shapely LineString, or None if no path found.
        """
        if self.wbt is None:
            raise RuntimeError("WhiteboxTools not initialized. Check WhiteboxTools configuration: QGIS settings -> Options -> Processing -> Provider -> WhiteboxTools -> WhiteboxTools executable.")
        
        current_iteration = 0
        current_start_point = start_point
        current_barrier_value = initial_barrier_value
        accumulated_line_coords = []
    
        # Read destination raster data using GDAL
        dest_ds = gdal.Open(destination_raster_path, gdal.GA_ReadOnly)
        if dest_ds is None:
            raise RuntimeError(f"Cannot open destination raster: {destination_raster_path}.{self._get_gdal_error_message()}")
            
        try:
            # Get destination raster information
            dest_rows = dest_ds.RasterYSize
            dest_cols = dest_ds.RasterXSize
            dest_geotransform = dest_ds.GetGeoTransform()
            dest_projection = dest_ds.GetProjection()
            dest_srs = dest_ds.GetSpatialRef()
            
            # Read destination data
            dest_band = dest_ds.GetRasterBand(1)
            orig_dest_data = dest_band.ReadAsArray().copy()  # create a copy to avoid modifying the original raster
            if orig_dest_data is None:
                raise RuntimeError(f"Cannot read destination raster data: {destination_raster_path}.{self._get_gdal_error_message()}")
                
            dest_nodata = dest_band.GetNoDataValue()
            dest_dtype = gdal_array.GDALTypeCodeToNumericTypeCode(dest_band.DataType)
            
        finally:
            dest_ds = None  # Close destination dataset
        
        # Read barrier raster data using GDAL
        if barrier_raster_path is None:
            raise RuntimeError("Barrier raster path is None. _get_iterative_constant_slope_line requires a valid barrier raster path.")
            
        barrier_ds = gdal.Open(barrier_raster_path, gdal.GA_ReadOnly)
        if barrier_ds is None:
            raise RuntimeError(f"Cannot open barrier raster: {barrier_raster_path}.{self._get_gdal_error_message()}")
            
        try:
            # Get barrier raster information
            barrier_rows = barrier_ds.RasterYSize
            barrier_cols = barrier_ds.RasterXSize
            barrier_geotransform = barrier_ds.GetGeoTransform()
            barrier_projection = barrier_ds.GetProjection()
            barrier_srs = barrier_ds.GetSpatialRef()
            
            # Read barrier data
            barrier_band = barrier_ds.GetRasterBand(1)
            orig_barrier_data = barrier_band.ReadAsArray().copy()  # create a copy to avoid modifying the original raster
            if orig_barrier_data is None:
                raise RuntimeError(f"Cannot read barrier raster data: {barrier_raster_path}.{self._get_gdal_error_message()}")
                
            barrier_nodata = barrier_band.GetNoDataValue()
            barrier_dtype = gdal_array.GDALTypeCodeToNumericTypeCode(barrier_band.DataType)
            
        finally:
            barrier_ds = None  # Close barrier dataset
        
        while current_iteration < max_iterations_barrier:
            print(f"[IterativeConstantSlopeLine] Iteration {current_iteration + 1}/{max_iterations_barrier}")
            if feedback:
                feedback.pushInfo(f"[IterativeConstantSlopeLine] Iteration {current_iteration + 1}/{max_iterations_barrier}")
                feedback.pushInfo("*for more information see in Python Console")
                
            # Debug: Print start point and extracted value
            print(f"[IterativeConstantSlopeLine] Start point: {current_start_point.wkt}")
            print(f"[IterativeConstantSlopeLine] Start barrier value: {current_barrier_value}")

            print(f"[IterativeConstantSlopeLine] Creating working rasters for _get_constant_slope_line...")
            # --- Create barrier raster for _get_constant_slope_line ---
            # Use dtype information from GDAL data
            
            if current_barrier_value:
                working_barrier_data = np.where(orig_barrier_data == current_barrier_value, 1, 0).astype(orig_barrier_data.dtype) # current barrier act as barrier and not as destination
                working_destination_data = np.where((orig_dest_data == 1) | ((orig_barrier_data >= 1) & (orig_barrier_data != current_barrier_value)), 1, 0).astype(orig_dest_data.dtype)  # all other barriers act as temporary destinations except the current one
            else:
                working_barrier_data = None # all barriers acting as temporary destinations and not as barriers
                working_destination_data = np.where((orig_dest_data == 1) | (orig_barrier_data >= 1), 1, 0).astype(orig_dest_data.dtype) 
                
            # Save barrier mask using GDAL
            if working_barrier_data is not None:
                working_barrier_raster_path = os.path.join(self.temp_directory, f"barrier_iter_{current_iteration}.tif")
                
                # Create barrier raster using GDAL
                driver = gdal.GetDriverByName('GTiff')
                if driver is None:
                    raise RuntimeError(f"GTiff driver not available.{self._get_gdal_error_message()}")
                    
                creation_options = ['COMPRESS=LZW', 'TILED=YES', 'BIGTIFF=IF_SAFER']
                barrier_out_ds = driver.Create(working_barrier_raster_path, barrier_cols, barrier_rows, 1, gdal.GDT_Byte, 
                                              options=creation_options)
                if barrier_out_ds is None:
                    raise RuntimeError(f"Failed to create barrier raster: {working_barrier_raster_path}.{self._get_gdal_error_message()}")
                    
                try:
                    barrier_out_ds.SetGeoTransform(barrier_geotransform)
                    if barrier_srs is not None:
                        barrier_out_ds.SetSpatialRef(barrier_srs)
                    elif barrier_projection:
                        barrier_out_ds.SetProjection(barrier_projection)
                        
                    barrier_out_band = barrier_out_ds.GetRasterBand(1)
                    barrier_out_band.WriteArray(working_barrier_data.astype(np.uint8))
                    barrier_out_band.SetNoDataValue(0)
                    
                finally:
                    barrier_out_ds = None  # Close dataset
                    
                print(f"[IterativeConstantSlopeLine] Working barrier raster created at {working_barrier_raster_path}")
            else:
                working_barrier_raster_path = None
                print("[IterativeConstantSlopeLine] No working barrier raster created (all barriers act as temporary destinations).")
                        
            working_destination_raster_path = os.path.join(self.temp_directory, f"destination_iter_{current_iteration}.tif")
            
            # Save destination mask using GDAL
            driver = gdal.GetDriverByName('GTiff')
            if driver is None:
                raise RuntimeError(f"GTiff driver not available.{self._get_gdal_error_message()}")
                
            creation_options = ['COMPRESS=LZW', 'TILED=YES', 'BIGTIFF=IF_SAFER']
            dest_out_ds = driver.Create(working_destination_raster_path, dest_cols, dest_rows, 1, gdal.GDT_Byte, 
                                       options=creation_options)
            if dest_out_ds is None:
                raise RuntimeError(f"Failed to create destination raster: {working_destination_raster_path}.{self._get_gdal_error_message()}")
                
            try:
                dest_out_ds.SetGeoTransform(dest_geotransform)
                if dest_srs is not None:
                    dest_out_ds.SetSpatialRef(dest_srs)
                elif dest_projection:
                    dest_out_ds.SetProjection(dest_projection)
                    
                dest_out_band = dest_out_ds.GetRasterBand(1)
                dest_out_band.WriteArray(working_destination_data.astype(np.uint8))
                dest_out_band.SetNoDataValue(0)
                
            finally:
                dest_out_ds = None  # Close dataset
                
            print(f"[IterativeConstantSlopeLine] Working destination raster created at {working_destination_raster_path}")

            # Call _get_constant_slope_line with current parameters
            print(f"[IterativeConstantSlopeLine] Tracing from point {current_start_point}")
                
            line_segment = self._get_constant_slope_line(
                dtm_path=dtm_path,
                start_point=current_start_point,
                destination_raster_path=working_destination_raster_path,
                slope=slope,
                barrier_raster_path=working_barrier_raster_path,
                slope_deviation_threshold=slope_deviation_threshold,
                max_iterations_slope=max_iterations_slope,
                feedback=feedback
            )

            # Check if a line segment was found
            if line_segment is None:
                warnings.warn(f"[IterativeConstantSlopeLine] Warning: No line found in iteration {current_iteration + 1}")
                if feedback:
                    feedback.pushWarning(f"[IterativeConstantSlopeLine] Warning: No line found in iteration {current_iteration + 1}")
                break

            line_coords = list(line_segment.coords)
            # Check if endpoint is on original (final) destination
            endpoint = Point(line_coords[-1])
            print(f"[IterativeConstantSlopeLine] Endpoint iteration {current_iteration + 1}: {endpoint.wkt}")
            final_destination_found = False
            # Convert endpoint coordinates to pixel indices using GDAL geotransform
            pixel_coords = TopoDrainCore._coords_to_pixel_indices([endpoint.coords[0]], dest_geotransform)
            end_col, end_row = pixel_coords[0]
            
            if 0 <= end_row < orig_dest_data.shape[0] and 0 <= end_col < orig_dest_data.shape[1]:
                if orig_dest_data[end_row, end_col] == 1:
                    print(f"[IterativeConstantSlopeLine] Reached final destination in iteration {current_iteration + 1}")
                    final_destination_found = True
                else:
                    print(f"[IterativeConstantSlopeLine] Endpoint not on destination in iteration {current_iteration + 1}, checking barriers.")
            
            last_iteration = (current_iteration == max_iterations_barrier - 1)
            if not final_destination_found and last_iteration:
                warnings.warn("[IterativeConstantSlopeLine] Warning: Maximum iterations reached without finding a fully valid line")
                if feedback:
                    feedback.pushWarning("[IterativeConstantSlopeLine] Warning: Maximum iterations reached without finding a fully valid line")                    

            if final_destination_found or last_iteration: 
                # If we reached the last iteration, add this final segment to avoid cutting in the last iteration:
                # If we reached the final destination, add this final segment and stop
                if accumulated_line_coords:
                    # Skip first coordinate to avoid duplication
                    accumulated_line_coords.extend(line_segment.coords[1:])
                else:
                    accumulated_line_coords.extend(line_segment.coords)
                break

            else: 
                # Get barrier value at endpoint for next iteration using GDAL geotransform
                pixel_coords = TopoDrainCore._coords_to_pixel_indices([endpoint.coords[0]], barrier_geotransform)
                barrier_end_col, barrier_end_row = pixel_coords[0]
                
                if 0 <= barrier_end_row < barrier_rows and 0 <= barrier_end_col < barrier_cols:
                    end_barrier_value = int(orig_barrier_data[barrier_end_row, barrier_end_col])
                else:
                    end_barrier_value = None
                print(f"[IterativeConstantSlopeLine] Iteration reached barrier: {end_barrier_value}")

                if end_barrier_value:
                    # Get start point for next iteration next to the barrier (back where the line came from)
                    next_start_point = TopoDrainCore._get_linedirection_start_point(
                        barrier_raster_path=barrier_raster_path,
                        line_geom=line_segment,
                        max_offset=10,  # adjust as needed
                        reverse=True  # always go backward were the line came from
                    )
                else:
                    next_start_point = endpoint  # if no barrier, continue from endpoint (should actually never happen in this case)

                print(f"[IterativeConstantSlopeLine] Start point for next iteration: {next_start_point.wkt}")

                print(f"[IterativeConstantSlopeLine] Adjusting line segment to new start point")
                # Adjust line_segment to only go up to the new next_start_point (not to the endpoint)
                if next_start_point != endpoint:
                    line_segment = self._cut_line_at_point(line_segment, next_start_point)

                # Add line segment to accumulated coordinates for continuing iterations
                if accumulated_line_coords:
                    # Skip first coordinate to avoid duplication
                    accumulated_line_coords.extend(line_segment.coords[1:])
                else:
                    accumulated_line_coords.extend(line_segment.coords)

            # Prepare for next iteration
            current_barrier_value = end_barrier_value if end_barrier_value else None  # Update barrier value for next iteration
            current_start_point = next_start_point  # Update start point for next iteration
            current_iteration += 1


        if len(accumulated_line_coords) >= 2:
            line = LineString(accumulated_line_coords)
            print(f"[IterativeConstantSlopeLine] Completed after {current_iteration + 1} iterations")
            if feedback:
                feedback.pushInfo(f"[IterativeConstantSlopeLine] Completed after {current_iteration + 1} iterations")
            return line
        else:
            warnings.warn("[IterativeConstantSlopeLine] Warning: No valid line could be created")
            if feedback:
                feedback.pushWarning("[IterativeConstantSlopeLine] Warning: No valid line could be created")
            return None

    def get_constant_slope_lines(
        self,
        dtm_path: str,
        start_points: gpd.GeoDataFrame,
        destination_features: list[gpd.GeoDataFrame],
        slope: float = 0.01,
        barrier_features: list[gpd.GeoDataFrame] = None,
        allow_barriers_as_temp_destination: bool = False,
        max_iterations_barrier: int = 30,
        slope_deviation_threshold: float = 0.2,
        max_iterations_slope: int = 20,
        feedback=None,
        report_progress: bool = True
    ) -> gpd.GeoDataFrame:
        """
        Trace lines with constant slope starting from given points using a cost-distance approach
        based on slope deviation, snapping true original start-points only when they overlapped barrier lines.
        All barrier_features (lines, polygons, points) are rasterized into barrier_mask,
        but only the line geometries are used for splitting and offsetting start points.
        
        Args:
            dtm_path (str): Path to the digital terrain model (GeoTIFF).
            start_points (gpd.GeoDataFrame): Starting points for slope line tracing (e.g. Keypoints).
            destination_features (list[gpd.GeoDataFrame]): List of destination features (e.g. main ridge lines, area of interest).
            slope (float): Desired slope for the lines (e.g., 0.01 for 1% downhill).
            barrier_features (list[gpd.GeoDataFrame], optional): List of barrier features to avoid (e.g. main valley lines).
            allow_barriers_as_temp_destination (bool): If True, barriers are included as temporary destinations for iterative tracing.
            max_iterations_barrier (int): Maximum number of iterations for iterative tracing when allowing barriers as temporary destinations. Default 30.
            max_iterations_slope (int): Maximum number of iterations for line refinement (1-50, higher values allow more complex paths). Default 20.
            slope_deviation_threshold (float): Maximum allowed relative deviation from expected slope (0.0-1.0, e.g., 0.2 for 20% deviation before line cutting). Default 0.2.
            feedback (QgsProcessingFeedback, optional): Optional feedback object for progress reporting/logging.
            report_progress (bool): Whether to report progress via setProgress calls. Default True. Set to False when called from create_keylines_core to avoid progress conflicts.
            
        Returns:
            gpd.GeoDataFrame: Traced constant slope lines.
        """
        if feedback:
            if feedback.isCanceled():
                feedback.reportError("Constant slope line tracing was cancelled by user")
                raise RuntimeError("Operation cancelled by user")

        if feedback:
            feedback.pushInfo("[ConstantSlopeLines] Starting tracing")
            if report_progress:
                feedback.setProgress(0)
        else:
            print("[ConstantSlopeLines] Starting tracing")

        # Check for potential configuration issues
        if allow_barriers_as_temp_destination and not barrier_features:
            warning_msg = "[ConstantSlopeLines] Warning: allow_barriers_as_temp_destination=True but no barrier_features provided. This setting will have no effect."
            if feedback:
                feedback.pushWarning(warning_msg)
            else:
                warnings.warn(warning_msg)

        # store original start points for adding later to the traced line geometry if starting from adjusted start point
        original_pts = start_points.copy()

        # --- Destination mask ---
        if feedback:
            feedback.pushInfo("[ConstantSlopeLines] Building destination mask…")
        else:
            print("[ConstantSlopeLines] Building destination mask…")
        
        # Process destination features - convert polygons to boundaries and flatten to LineStrings
        destination_processed = TopoDrainCore._features_to_single_linestring(destination_features)

        # Create binary destination mask
        destination_raster_path = self._vector_to_mask_raster([destination_processed], dtm_path, unique_values=False, flatten_lines=False, buffer_lines=True) # destination_processed are already flattened. Binary mask raster, no need of unique values

        # Read destination mask for later processing
        dest_ds = gdal.Open(destination_raster_path, gdal.GA_ReadOnly)
        if dest_ds is None:
            raise RuntimeError(f"Cannot open destination raster: {destination_raster_path}.{self._get_gdal_error_message()}")
            
        try:
            dest_band = dest_ds.GetRasterBand(1)
            destination_mask = dest_band.ReadAsArray()
            if destination_mask is None:
                raise RuntimeError(f"Cannot read destination mask data from: {destination_raster_path}.{self._get_gdal_error_message()}")
        finally:
            dest_ds = None
            
        if feedback:
            feedback.pushInfo("[ConstantSlopeLines] Destination mask ready")
        else:
            print("[ConstantSlopeLines] Destination mask ready")

        # --- Barrier mask ---
        if barrier_features:
            if feedback:
                feedback.pushInfo("[ConstantSlopeLines] Preparing barrier mask...")
            else:
                print("[ConstantSlopeLines] Preparing barrier mask...")

            # Process barrier features - convert polygons to boundaries like destinations
            barrier_processed = TopoDrainCore._features_to_single_linestring(barrier_features)

            # Create binary barrier mask (always binary: 0=free, 1=barrier)
            barrier_raster_path = self._vector_to_mask_raster([barrier_processed], dtm_path, unique_values=False, flatten_lines=False, buffer_lines=True)
            
            # Create barrier value raster with unique values to identify which barrier feature each cell belongs to (needed for overlap analysis and allow barrier as destination option)
            barrier_value_raster_path, barrier_id_to_geom = self._vector_to_mask_raster([barrier_processed], dtm_path, unique_values=True, flatten_lines=False, buffer_lines=True)

            # Read barrier masks for processing
            barrier_ds = gdal.Open(barrier_raster_path, gdal.GA_ReadOnly)
            if barrier_ds is None:
                raise RuntimeError(f"Cannot open barrier raster: {barrier_raster_path}.{self._get_gdal_error_message()}")
                
            try:
                barrier_band = barrier_ds.GetRasterBand(1)
                barrier_mask = barrier_band.ReadAsArray()
                if barrier_mask is None:
                    raise RuntimeError(f"Cannot read barrier mask data from: {barrier_raster_path}.{self._get_gdal_error_message()}")
            finally:
                barrier_ds = None
                
            barrier_value_ds = gdal.Open(barrier_value_raster_path, gdal.GA_ReadOnly)
            if barrier_value_ds is None:
                raise RuntimeError(f"Cannot open barrier value raster: {barrier_value_raster_path}.{self._get_gdal_error_message()}")
                
            try:
                barrier_value_band = barrier_value_ds.GetRasterBand(1)
                barrier_value_mask = barrier_value_band.ReadAsArray()
                if barrier_value_mask is None:
                    raise RuntimeError(f"Cannot read barrier value mask data from: {barrier_value_raster_path}.{self._get_gdal_error_message()}")
            finally:
                barrier_value_ds = None

            # Handle overlapping barrier and original destination cells --> adjust destination mask
            dest_barrier_overlap = (barrier_mask == 1) & (destination_mask == 1)
            if np.any(dest_barrier_overlap):
                num_overlaps = np.sum(dest_barrier_overlap)
                if feedback:
                    feedback.pushInfo(f"[ConstantSlopeLines] Found {num_overlaps} overlapping barrier/original destination cells")
                else:
                    print(f"[ConstantSlopeLines] Found {num_overlaps} overlapping barrier/original destination cells")
                # Set destination_mask to 0 at overlapping cells, because not possible to be barrier and destination at the same time
                destination_mask[dest_barrier_overlap] = 0
                # update destination raster
                dest_write_ds = gdal.Open(destination_raster_path, gdal.GA_Update)
                if dest_write_ds is None:
                    raise RuntimeError(f"Cannot open destination raster for writing: {destination_raster_path}.{self._get_gdal_error_message()}")
                    
                try:
                    dest_write_band = dest_write_ds.GetRasterBand(1)
                    dest_write_band.WriteArray(destination_mask)
                    dest_write_ds.FlushCache()
                finally:
                    dest_write_ds = None
            else:
                if feedback:
                    feedback.pushInfo("[ConstantSlopeLines] No overlapping barrier/original destination cells found")
                else:
                    print("[ConstantSlopeLines] No overlapping barrier/original destination cells found")
            
            # Check if start points are on barrier raster cells using precise raster-based overlap detection
            if feedback:
                feedback.pushInfo("[ConstantSlopeLines] Check if start points lies on barrier features...")
            else:
                print("[ConstantSlopeLines] Check if start points lies on barrier features...")
            
            # Get barrier values at each start point location using raster lookup
            point_barrier_info = []
            dtm_ds = gdal.Open(dtm_path, gdal.GA_ReadOnly)
            if dtm_ds is None:
                raise RuntimeError(f"Cannot open DTM raster: {dtm_path}.{self._get_gdal_error_message()}")
                
            try:
                dtm_geotransform = dtm_ds.GetGeoTransform()
                
                for orig_idx, row in original_pts.iterrows():
                    pt = row.geometry
                    # Get raster indices for the point using TopoDrainCore utility function
                    pixel_coords = TopoDrainCore._coords_to_pixel_indices([pt.coords[0]], dtm_geotransform)
                    pt_c, pt_r = pixel_coords[0]
                    
                    # Check if point is within raster bounds
                    if (0 <= pt_r < barrier_mask.shape[0] and 0 <= pt_c < barrier_mask.shape[1]):
                        barrier_binary_value = int(barrier_mask[pt_r, pt_c])
                        barrier_feature_id = int(barrier_value_mask[pt_r, pt_c])
                        
                        point_barrier_info.append({
                            'orig_idx': orig_idx,
                            'is_on_barrier': barrier_binary_value == 1,
                            'barrier_feature_id': barrier_feature_id if barrier_binary_value == 1 else None,
                            'row': row
                        })
                    else:
                        # Point outside raster bounds - treat as non-overlapping
                        point_barrier_info.append({
                            'orig_idx': orig_idx,
                            'is_on_barrier': False,
                            'barrier_feature_id': None,
                            'row': row
                        })
            finally:
                dtm_ds = None
            
            # Count overlapping vs non-overlapping for reporting
            overlapping_count = sum(1 for info in point_barrier_info if info['is_on_barrier'])
            non_overlapping_count = len(point_barrier_info) - overlapping_count
            
            if feedback:
                feedback.pushInfo(f"[ConstantSlopeLines]  → {overlapping_count} overlapping on barrier cells, {non_overlapping_count} non-overlapping")
            else:
                print(f"[ConstantSlopeLines]  → {overlapping_count} overlapping on barrier cells, {non_overlapping_count} non-overlapping")
        else:
            if feedback:
                feedback.pushInfo("[ConstantSlopeLines] No barrier features provided")
            else:
                print("[ConstantSlopeLines] No barrier features provided")
            barrier_mask = None
            barrier_raster_path = None
            barrier_value_raster_path = None
            barrier_id_to_geom = {}
            # Create point_barrier_info for all points as non-overlapping
            point_barrier_info = []
            for orig_idx, row in original_pts.iterrows():
                point_barrier_info.append({
                    'orig_idx': orig_idx,
                    'is_on_barrier': False,
                    'barrier_feature_id': None,
                    'row': row
                })
            
            # Count overlapping vs non-overlapping for reporting (all non-overlapping when no barriers)
            overlapping_count = 0
            non_overlapping_count = len(point_barrier_info)

        # --- Trace slope lines directly from point_barrier_info ---
        results = []
        total_points = overlapping_count*2 + non_overlapping_count  # each overlapping point creates two lines (left and right offset)
        current_point = 0

        if feedback:
            feedback.pushInfo("[ConstantSlopeLines] Starting slope line tracing...")
            if report_progress:
                feedback.setProgress(40)
        else:
            print("[ConstantSlopeLines] Starting slope line tracing...")

        for info in point_barrier_info:
            # Check for cancellation at the start of each point processing
            if feedback and feedback.isCanceled():
                feedback.reportError("Constant slope line tracing was cancelled by user")
                raise RuntimeError("Operation cancelled by user")
                
            orig_idx = info['orig_idx']
            orig_pt = info['row'].geometry
            is_on_barrier = info['is_on_barrier']
            barrier_feature_id = info['barrier_feature_id']
            orig_attrs = original_pts.loc[orig_idx].drop(labels="geometry").to_dict()

            if is_on_barrier:
                # Point is on barrier - need to create adjusted start points
                # Point is on barrier - create offset points (reduced logging)

                # Get the specific barrier geometry that this point overlaps with
                barrier_geom = barrier_id_to_geom.get(barrier_feature_id)
                if barrier_geom is None:
                    if feedback:
                        feedback.pushWarning(f"[ConstantSlopeLines] Warning: No barrier geometry found for feature ID {barrier_feature_id} - skipping point {orig_idx}")
                    else:
                        warnings.warn(f"[ConstantSlopeLines] Warning: No barrier geometry found for feature ID {barrier_feature_id} - skipping point {orig_idx}")
                    continue

                # Get orthogonal offset points
                left_pt, right_pt = TopoDrainCore._get_orthogonal_directions_start_points(
                    barrier_raster_path=barrier_raster_path,
                    point=orig_pt,
                    line_geom=barrier_geom
                )

                # Trace from both offset points (if they exist)
                adj_idx = 0
                for offset_pt, offset_name in [(left_pt, "left"), (right_pt, "right")]:
                    current_point += 1
                    if offset_pt is None:
                        # Check if adjusted point is still on barrier (warn if so) ##### maybe not necessary
                        if offset_pt:
                            if feedback:
                                feedback.pushWarning(f"[ConstantSlopeLines] Warning: No {offset_name} offset point could be created for {orig_idx}")
                            else:
                                warnings.warn(f"[ConstantSlopeLines] Warning: No {offset_name} offset point could be created for {orig_idx}")
                        continue

                    # Progress reporting 
                    adj_idx += 1
                    current_point += 1

                    if feedback:
                        feedback.pushInfo(f"[ConstantSlopeLines] Processing point {current_point}/{total_points} - Point {orig_idx}...")
                    elif not feedback:
                        print(f"[ConstantSlopeLines] Processing point {current_point}/{total_points} - Point {orig_idx}...")

                    # Trace line from offset point
                    if allow_barriers_as_temp_destination and barrier_features and barrier_value_raster_path is not None:
                        raw_line = self._get_iterative_constant_slope_line(
                            dtm_path=dtm_path,
                            start_point=offset_pt,
                            destination_raster_path=destination_raster_path,
                            slope=slope,
                            barrier_raster_path=barrier_value_raster_path,  # Use barrier value raster for iterative tracing
                            initial_barrier_value=barrier_feature_id,
                            max_iterations_barrier=max_iterations_barrier,
                            slope_deviation_threshold=slope_deviation_threshold,
                            feedback=feedback
                        )
                    else:
                        raw_line = self._get_constant_slope_line(
                            dtm_path=dtm_path,
                            start_point=offset_pt,
                            destination_raster_path=destination_raster_path,
                            slope=slope,
                            barrier_raster_path=barrier_raster_path,  # Use binary barrier raster for simple tracing
                            max_iterations_slope=max_iterations_slope,
                            slope_deviation_threshold=slope_deviation_threshold,
                            feedback=feedback
                        )

                    # Calculate progress percentage for 40-90% range
                    progress_pct = int(40 + (current_point / total_points) * 50)

                    if feedback:
                        feedback.pushInfo(f"[ConstantSlopeLines] Processed {progress_pct}%")
                    else:
                        print(f"[ConstantSlopeLines] Processed {progress_pct}%")
                    if feedback and report_progress:
                        feedback.setProgress(progress_pct)

                    if not raw_line:
                        # No line generated (reduced logging)
                        continue

                    # Snap line to original start point
                    raw_line = TopoDrainCore._snap_line_to_point(raw_line, orig_pt, "start")

                    # Append to results
                    results.append({
                        "geometry": raw_line,
                        "orig_index": orig_idx,
                        "adj_index": adj_idx,
                        **orig_attrs
                    })

            else:
                # Point is not on barrier - trace directly from original point
                current_point += 1

                # Progress reporting
                if feedback:
                    feedback.pushInfo(f"[ConstantSlopeLines] Processing point {current_point}/{total_points} - Point {orig_idx}...")
                elif not feedback:
                    print(f"[ConstantSlopeLines] Processing point {current_point}/{total_points} - Point {orig_idx}...")

                # Trace line from original point
                if allow_barriers_as_temp_destination and barrier_value_raster_path is not None:
                    raw_line = self._get_iterative_constant_slope_line(
                        dtm_path=dtm_path,
                        start_point=orig_pt,
                        destination_raster_path=destination_raster_path,
                        slope=slope,
                        barrier_raster_path=barrier_value_raster_path,  # Use barrier value raster for iterative tracing
                        initial_barrier_value=None,  # No initial barrier value for non-overlapping points
                        max_iterations_barrier=max_iterations_barrier,  # max iterations for iterative tracing (nr of times barriers used as temporary destinations)
                        slope_deviation_threshold=slope_deviation_threshold,
                        feedback=feedback
                    )
                else:
                    raw_line = self._get_constant_slope_line(
                        dtm_path=dtm_path,
                        start_point=orig_pt,
                        destination_raster_path=destination_raster_path,
                        slope=slope,
                        barrier_raster_path=barrier_raster_path,  # Use binary barrier raster for simple tracing
                        max_iterations_slope=max_iterations_slope,
                        slope_deviation_threshold=slope_deviation_threshold,
                        feedback=feedback
                    )

                # Calculate progress percentage for 40-90% range
                progress_pct = int(40 + (current_point / total_points) * 50)

                if feedback:
                    feedback.pushInfo(f"[ConstantSlopeLines] Processed {progress_pct}%")
                else:
                    print(f"[ConstantSlopeLines] Processed {progress_pct}%")
                if feedback and report_progress:
                    feedback.setProgress(progress_pct)

                if not raw_line:
                    if feedback:
                        feedback.pushInfo(f"[ConstantSlopeLines]   → No line for point {orig_idx}")
                    else:
                        print(f"[ConstantSlopeLines]   → No line for point {orig_idx}")
                    continue

                # Append to results (no snapping needed since we started from original point)
                results.append({
                    "geometry": raw_line,
                    "orig_index": orig_idx,
                    **orig_attrs
                })

                if feedback:
                    feedback.pushInfo(f"[ConstantSlopeLines]   → Line created for {orig_idx}")
                else:
                    print(f"[ConstantSlopeLines]   → Line created for {orig_idx}")

        if not results:
            if feedback:
                feedback.reportError("[ConstantSlopeLines] No slope lines could be created.")
            raise RuntimeError("No slope lines could be created.")

        if feedback:
            if feedback.isCanceled():
                feedback.reportError("Constant slope line tracing was cancelled by user")
                raise RuntimeError("Operation cancelled by user")
            
        if feedback and report_progress:
            feedback.setProgress(100)
        if feedback:
            feedback.pushInfo(f"[ConstantSlopeLines] Done: generated {len(results)} lines")
        else:
            print(f"[ConstantSlopeLines] Done: generated {len(results)} lines")

        # build GeoDataFrame including all original attributes
        out_gdf = gpd.GeoDataFrame(results, crs=self.crs)
        
        return out_gdf
    
    def create_keylines_core(self, dtm_path, start_points, valley_lines, ridge_lines, slope, perimeter, 
                        change_after=None, slope_after=None, slope_deviation_threshold=0.2, max_iterations_slope=20, feedback=None):
        """
        Core keylines creation logic using an iterative process:
        1. Trace from start points to ridges (using valleys as barriers)
        2. Check if endpoints are on ridges, create new start points beyond ridges
        3. Trace from new start points to valleys (using ridges as barriers)
        4. Continue iteratively while endpoints reach target features

        All output keylines will be oriented from valley to ridge (valley → ridge direction).

        Parameters:
        -----------
        dtm_path : str
            Path to the digital terrain model (GeoTIFF)
        start_points : GeoDataFrame
            Input keypoints to start keyline creation from
        valley_lines : GeoDataFrame
            Valley line features to use as barriers/destinations
        ridge_lines : GeoDataFrame
            Ridge line features to use as barriers/destinations
        slope : float
            Target slope for the constant slope lines (e.g., 0.01 for 1%)
        perimeter : GeoDataFrame
            Area of interest (perimeter) that always acts as destination feature (e.g. watershed, parcel polygon)
        change_after : float, optional
            Fraction of line length where slope changes (0.0-1.0, e.g., 0.5 = from halfway). If None, no slope adjustment is applied.
        slope_after : float, optional
            New slope to apply after the change point (e.g., 0.005 for 0.5% downhill). Required if change_after is provided.
        slope_deviation_threshold : float, optional
            Maximum allowed relative deviation from expected slope (0.0-1.0, e.g., 0.2 for 20% deviation before line cutting). Default 0.2.
        max_iterations_slope : int, optional
            Maximum number of iterations for line refinement (1-50). Default 20.
        feedback : QgsProcessingFeedback
            Feedback object for progress reporting

        Returns:
        --------
        GeoDataFrame
            Combined keylines from all stages, all oriented from valley to ridge.
        """
        if feedback:
            feedback.pushInfo("[CreateKeylinesCore] Starting iterative keyline core creation process...")
            feedback.setProgress(5)
            if feedback.isCanceled():
                feedback.reportError("[CreateKeylinesCore] Keyline creation was cancelled by user")
                raise RuntimeError("Operation cancelled by user")
        else:
            print("[CreateKeylinesCore]Starting iterative keyline core creation process...")
            print("[CreateKeylinesCore] Progress: 5%")
        
        # Create temporary raster .tif files for valley_lines and ridge_lines
        valley_lines_raster_path = os.path.join(self.temp_directory, "valley_lines_mask.tif")
        ridge_lines_raster_path = os.path.join(self.temp_directory, "ridge_lines_mask.tif")

        # Rasterize valley_lines - now returns path directly
        valley_lines_raster_path = self._vector_to_mask_raster([valley_lines], dtm_path, output_path=valley_lines_raster_path, unique_values=False, flatten_lines=True, buffer_lines=True)
        # Rasterize ridge_lines - now returns path directly
        ridge_lines_raster_path = self._vector_to_mask_raster([ridge_lines], dtm_path, output_path=ridge_lines_raster_path, unique_values=False, flatten_lines=True, buffer_lines=True)

        # Read valley mask for endpoint checking
        valley_lines_ds = gdal.Open(valley_lines_raster_path, gdal.GA_ReadOnly)
        if valley_lines_ds is None:
            raise RuntimeError(f"Cannot open valley lines raster: {valley_lines_raster_path}.{self._get_gdal_error_message()}")
        try:
            valley_lines_band = valley_lines_ds.GetRasterBand(1)
            valley_mask = valley_lines_band.ReadAsArray()
            if valley_mask is None:
                raise RuntimeError(f"Cannot read valley mask data from: {valley_lines_raster_path}.{self._get_gdal_error_message()}")
        finally:
            valley_lines_ds = None

        # Read ridge mask for endpoint checking
        ridge_lines_ds = gdal.Open(ridge_lines_raster_path, gdal.GA_ReadOnly)
        if ridge_lines_ds is None:
            raise RuntimeError(f"Cannot open ridge lines raster: {ridge_lines_raster_path}.{self._get_gdal_error_message()}")
        try:
            ridge_lines_band = ridge_lines_ds.GetRasterBand(1)
            ridge_mask = ridge_lines_band.ReadAsArray()
            if ridge_mask is None:
                raise RuntimeError(f"Cannot read ridge mask data from: {ridge_lines_raster_path}.{self._get_gdal_error_message()}")
        finally:
            ridge_lines_ds = None

        # Rasterize perimeter for precise endpoint checking
        perimeter_raster_path = None
        perimeter_mask = None
        if perimeter is not None:
            # If perimeter is a polygon, use its boundary for rasterization
            if perimeter.geom_type.isin(["Polygon", "MultiPolygon"]).any():
                perimeter_line = perimeter.copy()
                perimeter_line["geometry"] = perimeter_line.boundary # maybe .unary_union?
            else:
                perimeter_line = perimeter
            # Create temporary .tif file for perimeter raster
            perimeter_raster_path = os.path.join(self.temp_directory, "perimeter_mask.tif")
            perimeter_raster_path = self._vector_to_mask_raster(
                [perimeter_line],
                dtm_path,
                output_path=perimeter_raster_path,
                unique_values=False,
                flatten_lines=True,
                buffer_lines=True
            )
            # Read perimeter mask for endpoint checking
            perimeter_ds = gdal.Open(perimeter_raster_path, gdal.GA_ReadOnly)
            if perimeter_ds is None:
                raise RuntimeError(f"Cannot open perimeter raster: {perimeter_raster_path}.{self._get_gdal_error_message()}")
            try:
                perimeter_band = perimeter_ds.GetRasterBand(1)
                perimeter_mask = perimeter_band.ReadAsArray()
                if perimeter_mask is None:
                    raise RuntimeError(f"Cannot read perimeter mask data from: {perimeter_raster_path}.{self._get_gdal_error_message()}")
            finally:
                perimeter_ds = None
        
        if feedback:
            feedback.pushInfo("[CreateKeylinesCore] Starting iterative keyline creation process...")
            feedback.setProgress(5)
        else:
            print("[CreateKeylinesCore] Starting iterative keyline creation process...")
            print("[CreateKeylinesCore] Progress: 5%")

        # Initialize variables
        all_keylines = []
        current_start_points = start_points.copy()
        stage = 1
        
        # Set a maximum number of iterations to prevent infinite loops
        expected_stages = (len(valley_lines) + len(ridge_lines)) + 1  # Rough estimate assuming valley to ridge to valley alternation
        max_iterations_keyline = expected_stages + 10  # Set a reasonable limit based on input features (+10 for safety)

        # Iterate until no new start points are found or max iterations reached
        while not current_start_points.empty and stage <= max_iterations_keyline:
            # Progress: 5% at start, 95% spread over expected_stages
            progress = 5 + int((stage - 1) * (95 / expected_stages))
            if feedback:
                feedback.pushInfo(f"[CreateKeylinesCore] **** Stage {stage}/~{expected_stages}: Processing {len(current_start_points)} start points...***")
                feedback.setProgress(min(progress, 99))
                if feedback.isCanceled():
                    feedback.reportError("[CreateKeylinesCore] Keyline creation was cancelled by user")
                    raise RuntimeError("Operation cancelled by user")
            else:
                print(f"[CreateKeylinesCore] **** Stage {stage}/~{expected_stages}: Processing {len(current_start_points)} start points...****")
                print(f"[CreateKeylinesCore] Progress: {progress}%")
            
            # Determine destination and barrier features based on stage
            if stage % 2 == 1:  # Odd stages: trace to ridges, valleys as barriers
                destination_features = [ridge_lines]
                barrier_features = [valley_lines]
                target_type = "ridges"
                use_slope = slope  # Use slope as is for downhill tracing
            else:  # Even stages: trace to valleys, ridges as barriers
                destination_features = [valley_lines] 
                barrier_features = [ridge_lines]
                target_type = "valleys"
                # For even stages with slope adjustment: trace with -slope_after, then adjust to -slope
                if change_after is not None and slope_after is not None:
                    use_slope = -slope_after  # Trace with final slope (negative for ridge→valley)
                else:
                    use_slope = -slope  # Invert slope for uphill tracing (no adjustment)
            
            # Always add perimeter as destination feature if provided
            if perimeter is not None:
                destination_features.append(perimeter)
                
            if feedback:
                feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Tracing to {target_type}...")
            else:
                print(f"[CreateKeylinesCore] Stage {stage}: Tracing to {target_type}...")
        
            if feedback:
                feedback.pushInfo(f"[CreateKeylinesCore] Current start points: {current_start_points}")
            else:
                print(f"[CreateKeylinesCore] Current start points: {current_start_points}")

            # Trace constant slope lines
            stage_lines = self.get_constant_slope_lines(
                dtm_path=dtm_path,
                start_points=current_start_points,
                destination_features=destination_features,
                slope=use_slope,
                barrier_features=barrier_features,
                allow_barriers_as_temp_destination=False,
                slope_deviation_threshold=slope_deviation_threshold,
                max_iterations_slope=max_iterations_slope,
                feedback=feedback,
                report_progress=False  # Disable progress reporting to avoid conflicts with main progress
            )

            if feedback:
                feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Tracing complete, processing results...")
            else:
                print(f"[CreateKeylinesCore] Stage {stage}: Tracing complete, processing results...")

            if stage_lines.empty:
                if feedback:
                    feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: No lines generated, stopping...")
                else:
                    print(f"[CreateKeylinesCore] Stage {stage}: No lines generated, stopping...")
                break
                
            if feedback:
                feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage} complete: {len(stage_lines)} lines to {target_type}")
                feedback.pushInfo(f"[CreateKeylinesCore] Stage lines: {stage_lines}")
            else:
                print(f"[CreateKeylinesCore] Stage {stage} complete: {len(stage_lines)} lines to {target_type}")
                print(f"[CreateKeylinesCore] Stage lines: {stage_lines}") 

            # Apply slope adjustment if parameters are provided
            if change_after is not None and slope_after is not None:
                if feedback:
                    feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Applying slope adjustment after {change_after*100:.1f}% with new slope {slope_after}")
                else:
                    print(f"[CreateKeylinesCore] Stage {stage}: Applying slope adjustment after {change_after*100:.1f}% with new slope {slope_after}")
                
                # For even stages, we need to handle slope direction properly
                if stage % 2 == 0:  # Even stage: lines traced ridge→valley with -slope_after
                    # For even stages with slope adjustment:
                    # 1. Lines were traced ridge→valley with -slope_after
                    # 2. Apply slope adjustment to change from -slope_after to -slope at (1-change_after)
                    # 3. Then reverse final lines to valley→ridge orientation
                    
                    if feedback:
                        feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Even stage - applying slope adjustment from -slope_after to -slope...")
                    else:
                        print(f"[CreateKeylinesCore] Stage {stage}: Even stage - applying slope adjustment from -slope_after to -slope...")
                    
                    # Apply slope adjustment: change from -slope_after to -slope
                    # The adjust_constant_slope_after function expects positive slopes, so we pass the absolute values
                    # and handle the direction through the destination/barrier configuration
                    adjusted_lines = self.adjust_constant_slope_after(
                        dtm_path=dtm_path,
                        input_lines=stage_lines,  # Lines are ridge→valley with -slope_after
                        change_after=(1-change_after),  # Change at (1-change_after) distance
                        slope_after=-slope,  # Change to -slope (ridge→valley direction)
                        destination_features=destination_features,
                        barrier_features=barrier_features,
                        allow_barriers_as_temp_destination=False,
                        slope_deviation_threshold=slope_deviation_threshold,
                        max_iterations_slope=max_iterations_slope,
                        feedback=feedback,
                        report_progress=False  # Disable progress reporting to avoid conflicts with main progress
                    )
                    
                    # update stage lines
                    stage_lines = adjusted_lines

                else:  # Odd stage: lines are already valley→ridge oriented
                    # Apply the slope adjustment directly
                    adjusted_lines = self.adjust_constant_slope_after(
                        dtm_path=dtm_path,
                        input_lines=stage_lines,
                        change_after=change_after,
                        slope_after=slope_after,  # Applies correctly in valley→ridge direction
                        destination_features=destination_features,
                        barrier_features=barrier_features,
                        allow_barriers_as_temp_destination=False,
                        slope_deviation_threshold=slope_deviation_threshold,
                        max_iterations_slope=max_iterations_slope,
                        feedback=feedback,
                        report_progress=False  # Disable progress reporting to avoid conflicts with main progress
                    )

                    # update stage lines
                    stage_lines = adjusted_lines
                    
                if feedback:
                    feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Slope adjustment complete, {len(stage_lines)} adjusted lines")
                else:
                    print(f"[CreateKeylinesCore] Stage {stage}: Slope adjustment complete, {len(stage_lines)} adjusted lines")

            # Check endpoints and create new start points if they're on target features
            new_start_points = []
            # Define which raster mask to use for barrier checking
            new_barrier_raster_path = ridge_lines_raster_path if target_type == "ridges" else valley_lines_raster_path
            new_barrier_mask = ridge_mask if target_type == "ridges" else valley_mask
            new_barrier_feature = ridge_lines if target_type == "ridges" else valley_lines
            
            if feedback:    
                feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Checking endpoints on {target_type} using raster-based detection...")
            else:
                print(f"[CreateKeylinesCore] Stage {stage}: Checking endpoints on {target_type} using raster-based detection...")

            # Iterate through each line in the stage_lines GeoDataFrame (for endpoint processing)
            # Note: stage_lines is oriented correctly for endpoint detection
            dtm_ds = gdal.Open(dtm_path, gdal.GA_ReadOnly)
            if dtm_ds is None:
                raise RuntimeError(f"Cannot open DTM raster: {dtm_path}.{self._get_gdal_error_message()}")
                
            try:
                dtm_geotransform = dtm_ds.GetGeoTransform()
                dtm_rows = dtm_ds.RasterYSize
                dtm_cols = dtm_ds.RasterXSize
                
                for line_idx, line_row in stage_lines.iterrows():
                    line_geom = line_row.geometry
                    if hasattr(line_geom, 'coords') and len(line_geom.coords) >= 2:
                        end_point = Point(line_geom.coords[-1])
                        if feedback:
                            feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Checking endpoint {end_point.wkt} for new start point...")
                        else:
                            print(f"[CreateKeylinesCore] Stage {stage}: Checking endpoint {end_point.wkt} for new start point...")
                        
                        # Get raster coordinates for endpoint using TopoDrainCore utility function
                        pixel_coords = TopoDrainCore._coords_to_pixel_indices([end_point.coords[0]], dtm_geotransform)
                        end_c, end_r = pixel_coords[0]
                        
                        # Check if endpoint is within raster bounds
                        if not (0 <= end_r < dtm_rows and 0 <= end_c < dtm_cols):
                            if feedback:
                                feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Endpoint outside raster bounds, skipping...")
                            else:
                                print(f"[CreateKeylinesCore] Stage {stage}: Endpoint outside raster bounds, skipping...")    
                            continue
                        
                        # Check if endpoint reached the perimeter using raster lookup
                        reached_perimeter = False
                        if perimeter is not None and perimeter_mask is not None:
                            perimeter_value = int(perimeter_mask[end_r, end_c])
                            reached_perimeter = (perimeter_value == 1)
                            
                            if feedback:
                                feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Perimeter raster value at endpoint: {perimeter_value}")
                            else:
                                print(f"[CreateKeylinesCore] Stage {stage}: Perimeter raster value at endpoint: {perimeter_value}")  
                            
                            if reached_perimeter:
                                if feedback:
                                    feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Endpoint is on perimeter raster cell, adding line and skipping further tracing.")
                                else:
                                    print(f"[CreateKeylinesCore] Stage {stage}: Endpoint is on perimeter raster cell, adding line and skipping further tracing.")

                                # adjust line orientation for even stages to ensure valley→ridge direction
                                if stage % 2 == 0:  
                                    final_line_geom = TopoDrainCore._reverse_line_direction(line_geom)
                                    if feedback:
                                        feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Reversed line direction (ridge→valley to valley→ridge)")
                                    else:
                                        print(f"[CreateKeylinesCore] Stage {stage}: Reversed line direction (ridge→valley to valley→ridge)") 
                                else:
                                    final_line_geom = line_geom

                                # Add final line to all_keylines (now correctly oriented valley → ridge)
                                all_keylines.append(final_line_geom)
                                continue  # Skip further processing for this line
                                
                        # Check if endpoint is on barrier using raster lookup
                        barrier_value = int(new_barrier_mask[end_r, end_c])
                        is_on_barrier = (barrier_value == 1)
                        
                        if feedback:
                            feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Barrier raster value at endpoint: {barrier_value}")
                        else:
                            print(f"[CreateKeylinesCore] Stage {stage}: Barrier raster value at endpoint: {barrier_value}")
                        
                        if is_on_barrier:
                            if feedback:
                                feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Endpoint is on barrier raster cell, trying to get a new start point...")
                            else:
                                print(f"[CreateKeylinesCore] Stage {stage}: Endpoint is on barrier raster cell, trying to get a new start point...") 
                            new_start_point = TopoDrainCore._get_linedirection_start_point(
                                new_barrier_raster_path, line_geom, max_offset=10
                            )
                            if new_start_point:
                                if feedback:
                                    feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: New start point found at {new_start_point.wkt}")
                                else:
                                    print(f"[CreateKeylinesCore] Stage {stage}: New start point found at {new_start_point.wkt}") 
                                new_start_points.append(new_start_point)
                                # Extend line_geom to include the new start point
                                line_geom = TopoDrainCore._snap_line_to_point(
                                    line_geom, new_start_point, "end"
                                )
                            else:
                                if feedback:
                                    feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: No valid start point found.")
                                else:
                                    print(f"[CreateKeylinesCore] Stage {stage}: No valid start point found.")

                        # No slope adjustment - use the (possibly extended) line from endpoint processing
                        if stage % 2 == 0:
                            # Even stage without slope adjustment: reverse ridge→valley to valley→ridge
                            final_line_geom = TopoDrainCore._reverse_line_direction(line_geom)
                            if feedback:
                                feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Reversed line direction (ridge→valley to valley→ridge)")
                            else:
                                print(f"[CreateKeylinesCore] Stage {stage}: Reversed line direction (ridge→valley to valley→ridge)")
                        else:
                            final_line_geom = line_geom  # Odd stage already valley→ridge
                    
                        # Add line to all_keylines (now correctly oriented valley → ridge)
                        all_keylines.append(final_line_geom)
            finally:
                dtm_ds = None

            if not new_start_points:
                if feedback:
                    feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: No endpoints on {target_type}, stopping iteration...")
                else:
                    print(f"[CreateKeylinesCore] Stage {stage}: No endpoints on {target_type}, stopping iteration...")
                break
            
            # Create GeoDataFrame from new start points
            current_start_points = gpd.GeoDataFrame(
                geometry=new_start_points,
                crs=start_points.crs
            )
        
            if feedback:
                feedback.pushInfo(f"[CreateKeylinesCore] Stage {stage}: Generated {len(new_start_points)} new start points beyond {target_type}")
            else:
                print(f"[CreateKeylinesCore] Stage {stage}: Generated {len(new_start_points)} new start points beyond {target_type}")
            
            # Increment stage
            stage += 1

        if stage > max_iterations_keyline:
            if feedback:
                feedback.reportWarning(f"[CreateKeylinesCore] Maximum iterations ({max_iterations_keyline}) reached, stopping iteration...")
            else:
                warnings.warn(f"[CreateKeylinesCore] Warning: Maximum iterations ({max_iterations_keyline}) reached, stopping iteration...")

        # Create combined GeoDataFrame
        if all_keylines:
            combined_gdf = gpd.GeoDataFrame(
                geometry=all_keylines,
                crs=start_points.crs
            )
        else:
            combined_gdf = gpd.GeoDataFrame(crs=start_points.crs)
        
        if feedback:
            feedback.setProgress(100)
            if feedback.isCanceled():
                feedback.reportError("Keyline creation was cancelled by user")
                raise RuntimeError("Operation cancelled by user")
            feedback.pushInfo(f"[CreateKeylinesCore] Keyline creation complete: {len(combined_gdf)} total keylines from {stage-1} stages")
        else:
            print(f"[CreateKeylinesCore] Keyline creation complete: {len(combined_gdf)} total keylines from {stage-1} stages")
            print("[CreateKeylinesCore] Progress: 100%")
            
        return combined_gdf

    def create_keylines(self, dtm_path, start_points, valley_lines, ridge_lines, slope, perimeter, 
                        change_after=None, slope_after=None, slope_deviation_threshold=0.2, max_iterations_slope=20, feedback=None):
        """
        Create keylines using an iterative process, handling start points on either valley or ridge lines:
        1. Classify start points based on their location (valley lines, ridge lines, or neutral)
        2. Process valley/neutral start points normally (valley → ridge → valley...)
        3. Process ridge start points with swapped valley/ridge roles (ridge → valley → ridge...)
        4. Combine results from both processing paths

        All output keylines will be oriented from valley to ridge (valley → ridge direction).

        Parameters:
        -----------
        dtm_path : str
            Path to the digital terrain model (GeoTIFF)
        start_points : GeoDataFrame
            Input keypoints to start keyline creation from (can be on valley lines, ridge lines, or neutral)
        valley_lines : GeoDataFrame
            Valley line features to use as barriers/destinations
        ridge_lines : GeoDataFrame
            Ridge line features to use as barriers/destinations
        slope : float
            Target slope for the constant slope lines (e.g., 0.01 for 1%)
        perimeter : GeoDataFrame
            Area of interest (perimeter) that always acts as destination feature (e.g. watershed, parcel polygon)
        change_after : float, optional
            Fraction of line length where slope changes (0.0-1.0, e.g., 0.5 = from halfway). If None, no slope adjustment is applied.
        slope_after : float, optional
            New slope to apply after the change point (e.g., 0.005 for 0.5% downhill). Required if change_after is provided.
        slope_deviation_threshold : float, optional
            Maximum allowed relative deviation from expected slope (0.0-1.0, e.g., 0.2 for 20% deviation before line cutting). Default 0.2.
        max_iterations_slope : int, optional
            Maximum number of iterations for line refinement (1-50). Default 20.
        feedback : QgsProcessingFeedback
            Feedback object for progress reporting

        Returns:
        --------
        GeoDataFrame
            Combined keylines from all stages, all oriented from valley to ridge.
        """
        if feedback:
            feedback.pushInfo("Starting keyline creation with start point classification...")
        else:
            print("Starting keyline creation with start point classification...")
        
        # Create raster masks for valley and ridge lines to classify start points
        valley_lines_raster_path = os.path.join(self.temp_directory, "valley_lines_classification.tif")
        ridge_lines_raster_path = os.path.join(self.temp_directory, "ridge_lines_classification.tif")
        
        # Rasterize valley and ridge lines for start point classification
        valley_lines_raster_path = self._vector_to_mask_raster([valley_lines], dtm_path, output_path=valley_lines_raster_path, unique_values=False, flatten_lines=True, buffer_lines=True)
        ridge_lines_raster_path = self._vector_to_mask_raster([ridge_lines], dtm_path, output_path=ridge_lines_raster_path, unique_values=False, flatten_lines=True, buffer_lines=True)
        
        # Read masks for start point classification
        valley_lines_ds = gdal.Open(valley_lines_raster_path, gdal.GA_ReadOnly)
        if valley_lines_ds is None:
            raise RuntimeError(f"Cannot open valley lines raster: {valley_lines_raster_path}.{self._get_gdal_error_message()}")
            
        try:
            valley_lines_band = valley_lines_ds.GetRasterBand(1)
            valley_mask = valley_lines_band.ReadAsArray()
            if valley_mask is None:
                raise RuntimeError(f"Cannot read valley mask data from: {valley_lines_raster_path}.{self._get_gdal_error_message()}")
        finally:
            valley_lines_ds = None
            
        ridge_lines_ds = gdal.Open(ridge_lines_raster_path, gdal.GA_ReadOnly)
        if ridge_lines_ds is None:
            raise RuntimeError(f"Cannot open ridge lines raster: {ridge_lines_raster_path}.{self._get_gdal_error_message()}")
            
        try:
            ridge_lines_band = ridge_lines_ds.GetRasterBand(1)
            ridge_mask = ridge_lines_band.ReadAsArray()
            if ridge_mask is None:
                raise RuntimeError(f"Cannot read ridge mask data from: {ridge_lines_raster_path}.{self._get_gdal_error_message()}")
        finally:
            ridge_lines_ds = None
            
        # Classify start points based on their location
        valley_start_points = []
        ridge_start_points = []
        neutral_start_points = []
        
        if feedback:
            feedback.pushInfo(f"Classifying {len(start_points)} start points...")
        else:
            print(f"Classifying {len(start_points)} start points...")
        
        dtm_ds = gdal.Open(dtm_path, gdal.GA_ReadOnly)
        if dtm_ds is None:
            raise RuntimeError(f"Cannot open DTM raster: {dtm_path}.{self._get_gdal_error_message()}")
            
        try:
            dtm_geotransform = dtm_ds.GetGeoTransform()
            dtm_rows = dtm_ds.RasterYSize
            dtm_cols = dtm_ds.RasterXSize
            
            for idx, row in start_points.iterrows():
                point = row.geometry
                
                # Get raster coordinates for the point using TopoDrainCore utility function
                pixel_coords = TopoDrainCore._coords_to_pixel_indices([point.coords[0]], dtm_geotransform)
                point_c, point_r = pixel_coords[0]
                
                # Check if point is within raster bounds
                if not (0 <= point_r < dtm_rows and 0 <= point_c < dtm_cols):
                    if feedback:
                        feedback.pushWarning(f"Warning: Start point {idx} is outside raster bounds, treating as neutral")
                    else:
                        warnings.warn(f"Warning: Start point {idx} is outside raster bounds, treating as neutral")
                    neutral_start_points.append(row)
                    continue
                
                # Check if point is on valley or ridge mask
                valley_value = int(valley_mask[point_r, point_c])
                ridge_value = int(ridge_mask[point_r, point_c])
                
                if valley_value == 1 and ridge_value == 1:
                    # Point is on both valley and ridge - treat as neutral with warning
                    if feedback:
                        feedback.pushWarning(f"Warning: Start point {idx} is on both valley and ridge lines, treating as neutral")
                    else:
                        warnings.warn(f"Warning: Start point {idx} is on both valley and ridge lines, treating as neutral")
                    neutral_start_points.append(row)
                elif valley_value == 1:
                    # Point is on valley line
                    valley_start_points.append(row)
                elif ridge_value == 1:
                    # Point is on ridge line
                    ridge_start_points.append(row)
                else:
                    # Point is on neither - neutral point
                    neutral_start_points.append(row)
        finally:
            dtm_ds = None
        
        # Create GeoDataFrames for each category
        valley_start_gdf = gpd.GeoDataFrame(valley_start_points, crs=start_points.crs) if valley_start_points else gpd.GeoDataFrame(crs=start_points.crs)
        ridge_start_gdf = gpd.GeoDataFrame(ridge_start_points, crs=start_points.crs) if ridge_start_points else gpd.GeoDataFrame(crs=start_points.crs)
        neutral_start_gdf = gpd.GeoDataFrame(neutral_start_points, crs=start_points.crs) if neutral_start_points else gpd.GeoDataFrame(crs=start_points.crs)
        
        # Report classification results
        if feedback:
            feedback.pushInfo(f"Start point classification: {len(valley_start_gdf)} on valleys, {len(ridge_start_gdf)} on ridges, {len(neutral_start_gdf)} neutral")
            if len(neutral_start_gdf) > 0:
                feedback.pushWarning(f"Warning: {len(neutral_start_gdf)} neutral start points should ideally be positioned on either ridge or valley lines")
        else:
            print(f"Start point classification: {len(valley_start_gdf)} on valleys, {len(ridge_start_gdf)} on ridges, {len(neutral_start_gdf)} neutral")
            if len(neutral_start_gdf) > 0:
                warnings.warn(f"Warning: {len(neutral_start_gdf)} neutral start points should ideally be positioned on either ridge or valley lines")

        # Initialize list to collect all keylines
        all_keylines = []
        
        # Process valley start points and neutral points (treat neutral as valley starts)
        if not valley_start_gdf.empty or not neutral_start_gdf.empty:
            # Combine valley and neutral start points
            valley_and_neutral = []
            if not valley_start_gdf.empty:
                valley_and_neutral.extend([row for _, row in valley_start_gdf.iterrows()])
            if not neutral_start_gdf.empty:
                valley_and_neutral.extend([row for _, row in neutral_start_gdf.iterrows()])
            
            valley_neutral_gdf = gpd.GeoDataFrame(valley_and_neutral, crs=start_points.crs)
            
            if feedback:
                feedback.pushInfo(f"Processing {len(valley_neutral_gdf)} valley/neutral start points...")
            else:
                print(f"Processing {len(valley_neutral_gdf)} valley/neutral start points...")
            
            valley_keylines = self.create_keylines_core(
                dtm_path=dtm_path,
                start_points=valley_neutral_gdf,
                valley_lines=valley_lines,
                ridge_lines=ridge_lines,
                slope=slope,
                perimeter=perimeter,
                change_after=change_after,
                slope_after=slope_after,
                slope_deviation_threshold=slope_deviation_threshold,
                max_iterations_slope=max_iterations_slope,
                feedback=feedback
            )
            
            if not valley_keylines.empty:
                all_keylines.extend([geom for geom in valley_keylines.geometry])
        
        # Process ridge start points (swap valley and ridge roles)
        if not ridge_start_gdf.empty:
            if feedback:
                feedback.pushInfo(f"[CreateKeylines] Processing {len(ridge_start_gdf)} ridge start points (with swapped valley/ridge roles)...")
            else:
                print(f"[CreateKeylines] Processing {len(ridge_start_gdf)} ridge start points (with swapped valley/ridge roles)...")

            ridge_keylines = self.create_keylines_core(
                dtm_path=dtm_path,
                start_points=ridge_start_gdf,
                valley_lines=ridge_lines,  # Swap: use ridge_lines as valley_lines
                ridge_lines=valley_lines,  # Swap: use valley_lines as ridge_lines
                slope=-slope_after if slope_after is not None else -slope, # Start with slope after, since we are going ridge→valley first
                perimeter=perimeter,
                change_after=(1-change_after) if change_after is not None else None, # Invert change_after for swapped roles
                slope_after=-slope, # Invert slope_after to slope for swapped roles
                slope_deviation_threshold=slope_deviation_threshold,
                max_iterations_slope=max_iterations_slope,
                feedback=feedback
            )
            
            if not ridge_keylines.empty:
                # Reverse the line direction to ensure valley→ridge orientation
                # (ridge_keylines are currently ridge→valley due to swapped roles)
                reversed_ridge_keylines = TopoDrainCore._reverse_line_direction(ridge_keylines)
                all_keylines.extend([geom for geom in reversed_ridge_keylines.geometry])
        
        # Combine all keylines
        if all_keylines:
            combined_gdf = gpd.GeoDataFrame(geometry=all_keylines, crs=start_points.crs)
        else:
            combined_gdf = gpd.GeoDataFrame(crs=start_points.crs)
        
        if feedback:
            feedback.pushInfo(f"[CreateKeylines] Keyline creation complete: {len(combined_gdf)} total keylines from {len(valley_start_gdf)} valley, {len(ridge_start_gdf)} ridge, and {len(neutral_start_gdf)} neutral start points")
        else:
            print(f"[CreateKeylines] Keyline creation complete: {len(combined_gdf)} total keylines from {len(valley_start_gdf)} valley, {len(ridge_start_gdf)} ridge, and {len(neutral_start_gdf)} neutral start points")
        
        return combined_gdf

    def adjust_constant_slope_after(
        self,
        dtm_path: str,
        input_lines: gpd.GeoDataFrame,
        change_after: float,
        slope_after: float,
        destination_features: list[gpd.GeoDataFrame],
        barrier_features: list[gpd.GeoDataFrame] = None,
        allow_barriers_as_temp_destination: bool = False,
        max_iterations_barrier: int = 30,
        slope_deviation_threshold: float = 0.2,
        max_iterations_slope: int = 20,
        feedback=None,
        report_progress: bool = True
    ) -> gpd.GeoDataFrame:
        """
        Modify constant slope lines by changing to a secondary slope after a specified distance.
        
        This function splits each input line at a specified fraction of its length and continues 
        with a new slope from that point using the get_constant_slope_lines function.
        
        Args:
            dtm_path (str): Path to the digital terrain model (GeoTIFF).
            input_lines (gpd.GeoDataFrame): Input constant slope lines to modify.
            change_after (float): Fraction of line length where slope changes (0.0-1.0, e.g., 0.5 = from halfway).
            slope_after (float): New slope to apply after the change point (e.g., 0.01 for 1% downhill).
            destination_features (list[gpd.GeoDataFrame]): Destination features for the new slope sections, e.g. ridge lines in case of keylines.
            barrier_features (list[gpd.GeoDataFrame], optional): Barrier features to avoid, e.g. valley lines in case of keylines.
            allow_barriers_as_temp_destination (bool): If True, barriers are included as temporary destinations for iterative tracing.
            max_iterations_barrier (int): Maximum number of iterations when using barriers as temporary destinations.
            slope_deviation_threshold (float): Maximum allowed relative deviation from expected slope (0.0-1.0, e.g., 0.2 for 20% deviation before line cutting).
            max_iterations_slope (int): Maximum number of iterations for line refinement.
            feedback (QgsProcessingFeedback, optional): Optional feedback object for progress reporting.
            report_progress (bool): Whether to report progress via setProgress calls. Default True. Set to False when called from create_keylines_core to avoid progress conflicts.
            
        Returns:
            gpd.GeoDataFrame: Modified lines with secondary slopes applied.
        """
        if feedback:
            if feedback.isCanceled():
                feedback.reportError("Slope adjustment was cancelled by user")
                raise RuntimeError("Operation cancelled by user")
            

        if feedback:
            feedback.pushInfo(f"[AdjustConstantSlopeAfter] Starting adjustment of {len(input_lines)} lines...")
            if report_progress:
                feedback.setProgress(0)
        else:
            print(f"[AdjustConstantSlopeAfter] Starting adjustment of {len(input_lines)} lines...")

        # Validate change_after parameter
        if not (0.0 <= change_after <= 1.0):
            raise ValueError("change_after must be between 0.0 and 1.0")
        
        # Check for potential configuration issues
        if allow_barriers_as_temp_destination and not barrier_features:
            warning_msg = "[AdjustConstantSlopeAfter] Warning: allow_barriers_as_temp_destination=True but no barrier_features provided. This setting will have no effect."
            if feedback:
                feedback.pushWarning(warning_msg)
            else:
                warnings.warn(warning_msg)
        
        # Phase 1: Process all lines to create first parts and collect start points for second parts
        if feedback:
            if feedback.isCanceled():
                feedback.reportError("Slope adjustment was cancelled by user")
                raise RuntimeError("Operation cancelled by user")

        if feedback:
            feedback.pushInfo(f"[AdjustConstantSlopeAfter] Phase 1: Processing {len(input_lines)} lines to create first parts...")
            if report_progress:
                feedback.setProgress(10)
        else:
            print(f"[AdjustConstantSlopeAfter] Phase 1: Processing {len(input_lines)} lines to create first parts...")

        first_parts_data = []  # Store first part data with mapping info
        all_start_points = []  # Collect all start points for second parts
        line_mapping = {}      # Map start point index to original line index
        total_lines = len(input_lines)
        
        for idx, row in input_lines.iterrows():
            line_geom = row.geometry
            
            # Progress reporting for Phase 1 (10-30% range)
            if feedback and report_progress:
                phase1_progress = int(10 + ((idx + 1) / total_lines) * 20)
                feedback.setProgress(phase1_progress)
            
            # Handle MultiLineString by stitching into a single LineString
            if isinstance(line_geom, MultiLineString):
                # Converting MultiLineString to LineString (reduced logging)
                line_geom = TopoDrainCore._merge_lines_by_distance(line_geom)
            elif not isinstance(line_geom, LineString):
                if feedback:
                    feedback.pushInfo(f"[AdjustConstantSlopeAfter] Skipping unsupported geometry type: {type(line_geom)} at index {idx}")
                else:
                    print(f"[AdjustConstantSlopeAfter] Skipping unsupported geometry type: {type(line_geom)} at index {idx}")   
                # Keep original line for unsupported geometry types
                first_parts_data.append({
                    'original_row': row,
                    'first_part_line': None,
                    'needs_second_part': False,
                    'start_point_index': None
                })
                continue
            
            # Calculate split point at specified fraction of line length
            line_length = line_geom.length
            split_distance = line_length * change_after
            
            # Check if split distance is valid
            if split_distance <= 0 or split_distance >= line_length:
                if feedback:
                    feedback.pushInfo(f"[AdjustConstantSlopeAfter] Invalid split distance for line {idx}, keeping original line")
                else:
                    print(f"[AdjustConstantSlopeAfter] Invalid split distance for line {idx}, keeping original line")   
                # Keep original line if split point is invalid
                first_parts_data.append({
                    'original_row': row,
                    'first_part_line': None,
                    'needs_second_part': False,
                    'start_point_index': None
                })
                continue
            
            # Create first part of line (from start to split point)
            try:
                # Get coordinates up to split point
                coords = list(line_geom.coords)
                first_part_coords = []
                remaining_distance = split_distance
                
                for i in range(len(coords) - 1):
                    start_pt = Point(coords[i])
                    end_pt = Point(coords[i + 1])
                    segment_length = start_pt.distance(end_pt)
                    
                    first_part_coords.append(coords[i])
                    
                    if remaining_distance <= segment_length:
                        # Interpolate the exact split point on this segment
                        if remaining_distance > 0:
                            fraction = remaining_distance / segment_length
                            split_x = coords[i][0] + fraction * (coords[i + 1][0] - coords[i][0])
                            split_y = coords[i][1] + fraction * (coords[i + 1][1] - coords[i][1])
                            first_part_coords.append((split_x, split_y))
                        break
                    else:
                        remaining_distance -= segment_length
                
                # Create first part of the line
                if len(first_part_coords) >= 2:
                    first_part_line = LineString(first_part_coords)
                    
                    # Create start point for second part (the split point)
                    start_point_second = Point(first_part_coords[-1])
                    start_point_index = len(all_start_points)
                    all_start_points.append(start_point_second)
                    line_mapping[start_point_index] = len(first_parts_data)
                    
                    first_parts_data.append({
                        'original_row': row,
                        'first_part_line': first_part_line,
                        'needs_second_part': True,
                        'start_point_index': start_point_index
                    })
                else:
                    # Fallback: use original line if we can't create valid first part
                    if feedback:
                        feedback.pushInfo(f"[AdjustConstantSlopeAfter] Could not create valid first part for line {idx}, keeping original line")
                    else:
                        print(f"[AdjustConstantSlopeAfter] Could not create valid first part for line {idx}, keeping original line")
                    first_parts_data.append({
                        'original_row': row,
                        'first_part_line': None,
                        'needs_second_part': False,
                        'start_point_index': None
                    })
                    
            except Exception as e:
                if feedback:
                    feedback.pushInfo(f"[AdjustConstantSlopeAfter] Error processing line {idx}: {str(e)}, keeping original")
                else:
                    print(f"[AdjustConstantSlopeAfter] Error processing line {idx}: {str(e)}, keeping original")
                first_parts_data.append({
                    'original_row': row,
                    'first_part_line': None,
                    'needs_second_part': False,
                    'start_point_index': None
                })
        
        # Phase 2: Trace all second parts in a single call if we have start points
        second_part_lines = gpd.GeoDataFrame(geometry=[])  # Empty GeoDataFrame with geometry column
        if all_start_points:
            if feedback:
                if feedback.isCanceled():
                    feedback.reportError("Slope adjustment was cancelled by user")
                    raise RuntimeError("Operation cancelled by user")
                
            if feedback:
                feedback.pushInfo(f"[AdjustConstantSlopeAfter] Phase 2: Tracing {len(all_start_points)} second parts in parallel...")
                if report_progress:
                    feedback.setProgress(40)
            else:
                print(f"[AdjustConstantSlopeAfter] Phase 2: Tracing {len(all_start_points)} second parts in parallel...")

            # Set CRS after creation when we have geometry column
            second_part_lines = second_part_lines.set_crs(input_lines.crs)
            
            # Create GeoDataFrame with all start points and add mapping information
            start_points_gdf = gpd.GeoDataFrame(geometry=all_start_points, crs=input_lines.crs)
            start_points_gdf['original_line_idx'] = [line_mapping[i] for i in range(len(all_start_points))]
            
            try:
                # Trace all second parts with new slope in a single call
                second_part_lines = self.get_constant_slope_lines(
                    dtm_path=dtm_path,
                    start_points=start_points_gdf,
                    destination_features=destination_features,
                    slope=slope_after,
                    barrier_features=barrier_features,
                    allow_barriers_as_temp_destination=allow_barriers_as_temp_destination,
                    max_iterations_barrier=max_iterations_barrier,
                    slope_deviation_threshold=slope_deviation_threshold,
                    max_iterations_slope=max_iterations_slope,
                    feedback=feedback, 
                    report_progress=True  
                )

                if feedback:
                    feedback.pushInfo(f"[AdjustConstantSlopeAfter] Successfully traced {len(second_part_lines)} second parts")
                else:
                    print(f"[AdjustConstantSlopeAfter] Successfully traced {len(second_part_lines)} second parts")
            except Exception as e:
                if feedback:
                    feedback.reportError(f"[AdjustConstantSlopeAfter] Error tracing second parts: {str(e)}")
                    raise RuntimeError(f"[AdjustConstantSlopeAfter] Error tracing second parts: {str(e)}")
                else:
                    raise RuntimeError(f"[AdjustConstantSlopeAfter] Error tracing second parts: {str(e)}")
                # Continue with empty second_part_lines
        else:
            # No second parts to trace (reduced logging)
            pass
        
        # Phase 3: Combine first and second parts
        if feedback:
            if feedback.isCanceled():
                feedback.reportError("Slope adjustment was cancelled by user")
                raise RuntimeError("Operation cancelled by user")

        if feedback:
            feedback.pushInfo(f"[AdjustConstantSlopeAfter] Phase 3: Combining first and second parts...")
            if report_progress:
                feedback.setProgress(80)
        else:
            print(f"[AdjustConstantSlopeAfter] Phase 3: Combining first and second parts...")
                 
        adjusted_lines = []
        total_parts = len(first_parts_data)
        
        for data_idx, part_data in enumerate(first_parts_data):
            # Progress reporting for Phase 3 (80-90% range)
            if feedback and report_progress and total_parts > 0:
                phase3_progress = int(80 + ((data_idx + 1) / total_parts) * 10)
                feedback.setProgress(phase3_progress)
                
            original_row = part_data['original_row']
            first_part_line = part_data['first_part_line']
            needs_second_part = part_data['needs_second_part']
            start_point_index = part_data['start_point_index']
            
            if not needs_second_part or first_part_line is None:
                # Keep original line
                adjusted_lines.append(original_row)
                continue
            
            # Find corresponding second part line(s)
            if not second_part_lines.empty and 'orig_index' in second_part_lines.columns:
                # Filter second parts that belong to this original line
                matching_second_parts = second_part_lines[second_part_lines['orig_index'] == start_point_index]
                
                if not matching_second_parts.empty:
                    # Get the first (and should be only) matching second part
                    second_part_line = matching_second_parts.iloc[0].geometry
                    
                    # Combine first and second parts
                    if isinstance(second_part_line, LineString):
                        # Merge coordinates, avoiding duplication of the split point
                        first_coords = list(first_part_line.coords)
                        second_coords = list(second_part_line.coords)
                        
                        # Remove duplicate split point if it exists
                        if len(second_coords) > 0 and first_coords[-1] == second_coords[0]:
                            combined_coords = first_coords + second_coords[1:]
                        else:
                            combined_coords = first_coords + second_coords
                        
                        combined_line = LineString(combined_coords)
                        
                        # Create new row with combined geometry and original attributes
                        new_row = original_row.copy()
                        new_row['geometry'] = combined_line
                        adjusted_lines.append(new_row)
                        
                        # Successfully combined line parts (reduced logging)
                    else:
                        # Second part is not LineString, keeping first part only (reduced logging)
                        new_row = original_row.copy()
                        new_row['geometry'] = first_part_line
                        adjusted_lines.append(new_row)
                else:
                    # No matching second part found, keeping first part only (reduced logging)
                    new_row = original_row.copy()
                    new_row['geometry'] = first_part_line
                    adjusted_lines.append(new_row)
            else:
                # No second parts available, keeping first part only (reduced logging)
                new_row = original_row.copy()
                new_row['geometry'] = first_part_line
                adjusted_lines.append(new_row)
        
        # Create result GeoDataFrame
        if adjusted_lines:
            result_gdf = gpd.GeoDataFrame(adjusted_lines, crs=input_lines.crs).reset_index(drop=True)
        else:
            result_gdf = gpd.GeoDataFrame(crs=input_lines.crs)
        
        if feedback:
            if feedback.isCanceled():
                feedback.reportError("Slope adjustment was cancelled by user")
                raise RuntimeError("Operation cancelled by user")
        

        if feedback:
            feedback.pushInfo(f"[AdjustConstantSlopeAfter] Adjustment complete: {len(result_gdf)} adjusted lines")
            if report_progress:
                feedback.setProgress(100)
        else:
            print(f"[AdjustConstantSlopeAfter] Adjustment complete: {len(result_gdf)} adjusted lines")

        return result_gdf


if __name__ == "__main__":
    print("No main part")
