#! python3  # noqa: E265

"""
Main plugin module.
"""
import importlib
import json
import os
import shutil
from functools import partial
from math import pi
from pathlib import Path

import numpy as np
from PyQt5.QtCore import QRect, Qt
from PyQt5.QtWidgets import QDialogButtonBox, QMessageBox, QProgressBar

# PyQGIS
from qgis.core import (
    Qgis,
    QgsApplication,
    QgsFeature,
    QgsGeometry,
    QgsProject,
    QgsVectorLayerUtils,
)
from qgis.gui import QgisInterface
from qgis.PyQt.QtCore import QCoreApplication, QSettings, QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QAction, QDialog, QMenu, QToolBar
from qgis.utils import showPluginHelp

from topaze.__about__ import DIR_PLUGIN_ROOT, __title__, __uri_homepage__
from topaze.gui.additional_canvas_dialog import AdditionalCanvasDialog
from topaze.gui.dlg_settings import PlgOptionsFactory
from topaze.gui.edit_obs_dialog import EditObsDialog
from topaze.gui.free_station_dialog import FreeStationDialog
from topaze.gui.import_field_data_dialog import ImportFieldDataDialog
from topaze.gui.import_points_dialog import ImportPointsDialog
from topaze.gui.intersection_dialog import IntersectionDialog
from topaze.gui.new_survey_dialog import NewSurveyDialog
from topaze.gui.resection_dialog import ResectionDialog
from topaze.gui.topaze_about import TopazeAbout
from topaze.gui.traverse_dialog import TraverseDialog
from topaze.gui.trilateration_dialog import TrilaterationDialog
from topaze.helmert_group import HelmertGroup
from topaze.shared import shared_ptopo_array
from topaze.toolbelt import PlgLogger, PlgOptionsManager, i18n
from topaze.topaze_global import TopazeConst
from topaze.topaze_layers import TopazeLayers
from topaze.totalopenstation import formats
from topaze.totalopenstation.formats import BUILTIN_INPUT_FORMATS
from topaze.totalopenstation.output.tops_azimut import OutputFormat

# project
from . import resources
from .file_utils import FileUtils
from .observation import Observation
from .ptopo import Ptopo, PtopoConst
from .station import Station
from .topaze_calculator import TopazeCalculator
from .topaze_config import TopazeConfig
from .topaze_global import TopazeGlobal
from .topaze_utils import TopazeUtils

# ############################################################################
# ########## Classes ###############
# ##################################

# __topaze_global__ = TopazeGlobal()


class TopazePlugin:
    OBSERVATIONS_LAYER_NAME = "observations"

    def __init__(self, iface: QgisInterface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class which \
        provides the hook by which you can manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        # print(
        #    "++++++++++++++++++++++++++++++++ __init__++++++++++++++++++++++++++++++++ "
        # )
        resources.__name__
        # global __topaze_global__
        self.log = PlgLogger().log
        self.plg_settings = PlgOptionsManager()
        self.settings = self.plg_settings.get_plg_settings()
        self.work_on_survey = False
        self._obs_storage_mode = "file"  # "file" or "db"
        try:
            layers = QgsProject.instance().mapLayersByName(TopazeConst.PTOPO_LAYER_NAME)
            if layers:
                self._ptopo_layer = layers[0]
                self.work_on_survey = True
        except Exception as e:
            pass

        # initialize locale
        i18n.init_i18n()

        self.iface = iface
        self.actions = []
        self.topaze_menu = None

        if self.settings.config_file:
            config_path = self.settings.config_file
        else:
            config_path = Path(__file__).parent / "resources/settings" / "topaze.yml"
        TopazeConfig.load_config(config_path)

        self.dlg_import_field_data = ImportFieldDataDialog()
        self.dlg_import_points = ImportPointsDialog()
        self.dlg_new_survey = NewSurveyDialog()
        self.dlg_edit_obs = EditObsDialog()
        self.dlg_edit_obs.pushButton_print.setVisible(False)
        self.dlg_edit_obs.pushButton_print.setEnabled(False)
        self.dlg_free_station = FreeStationDialog()
        self.dlg_free_station.set_iface(self.iface)
        # self.dlg_helmert_canvas = AdditionalCanvasDialog(self.iface)
        # self.dlg_helmert_canvas.setWindowFlags(Qt.Window)  # Non-modal
        self.dlg_resection = ResectionDialog()
        self.dlg_resection.set_iface(self.iface)
        self.dlg_intersection = IntersectionDialog()
        self.dlg_intersection.set_iface(self.iface)
        self.dlg_trilateration = TrilaterationDialog()
        self.dlg_trilateration.set_iface(self.iface)
        self.dlg_traverse = TraverseDialog()
        self.dlg_traverse.set_iface(self.iface)
        self.action_compute_point = None
        self.log = PlgLogger().log
        self._observations_layer = None
        self._ptopo_array = []
        QgsProject.instance().cleared.connect(self.on_topaze_project_cleared)
        self.iface.projectRead.connect(self.on_topaze_project_read)

    def initGui(self):
        """Set up plugin UI elements."""
        print("initGui")

        # settings page within the QGIS preferences menu
        self.options_factory = PlgOptionsFactory()
        self.iface.registerOptionsWidgetFactory(self.options_factory)

        if self.settings.is_first_use:
            self.settings.is_first_use = False
            PlgOptionsManager.save_from_object(self.settings)
            self.show_help_about()

        # Declare instance attributes
        self._current_obs = None
        self._previous_obs = None

        self.topaze_toolbar = self.iface.addToolBar(__title__)
        self.topaze_toolbar.setObjectName(__title__)
        self.action_new_survey_dialog = QAction(
            QIcon(":/images/themes/default/mActionNewFolder.svg"),
            i18n.tr("New survey"),
            self.iface.mainWindow(),
        )
        self.action_new_survey_dialog.triggered.connect(self.new_survey_dialog)
        self.action_new_survey_dialog.setEnabled(True)
        self.topaze_toolbar.addAction(self.action_new_survey_dialog)
        self.actions.append(self.action_new_survey_dialog)

        if not self.topaze_menu:
            self.topaze_menu = self.iface.pluginMenu().addMenu(
                QIcon(":/plugins/topaze/resources/images/topaze.png"), __title__
            )
        self.action_new_survey_dialog = QAction(
            QIcon(":/images/themes/default/mActionNewFolder.svg"),
            i18n.tr("New survey"),
            self.iface.mainWindow(),
        )
        self.action_new_survey_dialog.triggered.connect(self.new_survey_dialog)
        self.action_new_survey_dialog.setEnabled(True)
        self.topaze_menu.addAction(self.action_new_survey_dialog)
        self.actions.append(self.action_new_survey_dialog)

        self.action_import_points_dialog = QAction(
            QIcon(":/oauth2method/svg/import.svg"),
            i18n.tr("Import points"),
            self.iface.mainWindow(),
        )
        self.action_import_points_dialog.triggered.connect(self.import_points_dialog)
        self.topaze_menu.addAction(self.action_import_points_dialog)
        self.topaze_toolbar.addAction(self.action_import_points_dialog)
        self.actions.append(self.action_import_points_dialog)

        self.action_update_observations_dialog = QAction(
            QIcon(":/images/themes/default/mActionCopyProfileSettings.svg"),
            i18n.tr("Update observations"),
            self.iface.mainWindow(),
        )
        self.action_update_observations_dialog.triggered.connect(
            self.update_observations_dialog
        )
        self.topaze_menu.addAction(self.action_update_observations_dialog)
        self.topaze_toolbar.addAction(self.action_update_observations_dialog)
        self.actions.append(self.action_update_observations_dialog)

        self.action_edit_observations = QAction(
            QIcon(":/images/themes/default/console/iconShowEditorConsole.svg"),
            i18n.tr("Edit observations"),
            self.iface.mainWindow(),
        )
        self.action_edit_observations.triggered.connect(self.edit_observations)
        self.topaze_menu.addAction(self.action_edit_observations)
        self.topaze_toolbar.addAction(self.action_edit_observations)
        self.actions.append(self.action_edit_observations)

        self.sub_menu = QMenu(i18n.tr("Compute reference points"), self.topaze_menu)
        self.sub_menu.setIcon(
            QIcon(":/qt-project.org/styles/commonstyle/images/computer-32.png")
        )
        self.topaze_menu.addMenu(self.sub_menu)
        self.action_compute_free_station = QAction(
            i18n.tr("Free station"),
            self.iface.mainWindow(),
        )
        self.action_compute_free_station.triggered.connect(
            self.command_compute_free_station
        )
        self.sub_menu.addAction(self.action_compute_free_station)

        self.action_compute_intersection = QAction(
            i18n.tr("Intersection"),
            self.iface.mainWindow(),
        )
        self.action_compute_intersection.triggered.connect(
            self.command_compute_intersection
        )
        self.sub_menu.addAction(self.action_compute_intersection)
        self.actions.append(self.action_compute_intersection)

        self.action_compute_resection = QAction(
            i18n.tr("Resection"),
            self.iface.mainWindow(),
        )
        self.action_compute_resection.triggered.connect(self.command_compute_resection)
        self.sub_menu.addAction(self.action_compute_resection)
        self.actions.append(self.action_compute_resection)

        self.action_compute_trilateration = QAction(
            i18n.tr("Trilateration"),
            self.iface.mainWindow(),
        )
        self.action_compute_trilateration.triggered.connect(
            self.command_compute_trilateration
        )
        self.sub_menu.addAction(self.action_compute_trilateration)
        self.actions.append(self.action_compute_trilateration)

        self.action_compute_traverse = QAction(
            i18n.tr("Traverse"),
            self.iface.mainWindow(),
        )
        self.action_compute_traverse.triggered.connect(self.command_compute_traverse)
        self.sub_menu.addAction(self.action_compute_traverse)
        self.actions.append(self.action_compute_traverse)

        self.action_compute_points = QAction(
            QIcon(":plugins/topaze/resources/images/topaze.png"),
            i18n.tr("Compute points"),
            self.iface.mainWindow(),
        )
        self.action_compute_points.triggered.connect(self.command_compute_points)
        self.topaze_menu.addAction(self.action_compute_points)
        self.topaze_toolbar.addAction(self.action_compute_points)
        self.actions.append(self.action_compute_points)

        self.action_process_field_codification = QAction(
            QIcon(":/images/themes/default/console/iconFormatCode.svg"),
            i18n.tr("Process field codification"),
            self.iface.mainWindow(),
        )
        self.action_process_field_codification.triggered.connect(
            self.process_field_codification
        )
        self.topaze_menu.addAction(self.action_process_field_codification)
        self.topaze_toolbar.addAction(self.action_process_field_codification)
        self.actions.append(self.action_process_field_codification)

        self.action_purge_observations = QAction(
            QIcon(":/images/themes/default/mActionRemoveSelectedFeature.svg"),
            i18n.tr("Purge observations"),
            self.iface.mainWindow(),
        )
        self.action_purge_observations.triggered.connect(self.purge_observations)
        self.topaze_menu.addAction(self.action_purge_observations)
        self.topaze_toolbar.addAction(self.action_purge_observations)
        self.actions.append(self.action_purge_observations)

        # -- Menu
        self.dlg_import_points.button_box.button(QDialogButtonBox.Open).clicked.connect(
            self.button_box_import_points_clicked
        )
        self.dlg_edit_obs.pushButton_add.clicked.connect(
            self.on_push_button_add_obs_clicked
        )
        self.dlg_edit_obs.pushButton_change.clicked.connect(
            self.on_push_button_change_obs_clicked
        )
        self.dlg_edit_obs.pushButton_delete.clicked.connect(
            self.on_push_button_delete_obs_clicked
        )
        self.dlg_edit_obs.pushButton_export_lsci.clicked.connect(
            self.on_push_button_export_lsci_clicked
        )
        self.dlg_edit_obs.pushButton_save.clicked.connect(
            self.on_push_button_save_obs_data_clicked
        )
        self.dlg_edit_obs.pushButton_print.clicked.connect(
            self.on_push_button_print_selected_obs_clicked
        )
        self.dlg_edit_obs.pushButton_cancel.clicked.connect(
            self.on_push_button_cancel_edition_clicked
        )
        self.dlg_edit_obs.closeEvent = self.on_edit_observations_close

        self.action_helmert = QAction(
            QIcon(":/images/themes/default/transformation.svg"),
            i18n.tr("Helmert transformation"),
            self.iface.mainWindow(),
        )
        self.action_helmert.triggered.connect(self.helmert_transformation)
        self.topaze_menu.addAction(self.action_helmert)
        self.topaze_toolbar.addAction(self.action_helmert)
        self.actions.append(self.action_helmert)

        self.action_help = QAction(
            QIcon(":/images/themes/default/mActionHelpContents.svg"),
            i18n.tr("{} - Documentation").format(__title__),
            self.iface.mainWindow(),
        )
        self.action_help.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )
        self.topaze_menu.addAction(self.action_help)
        # self.topaze_toolbar.addAction(self.action_help)
        self.actions.append(self.action_help)

        self.action_about = QAction(
            QIcon(":/images/themes/default/mActionHelpAbout.svg"),
            i18n.tr("{} - About").format(__title__),
            self.iface.mainWindow(),
        )
        self.action_about.triggered.connect(self.show_help_about)
        self.topaze_menu.addAction(self.action_about)
        # self.topaze_toolbar.addAction(self.action_about)
        self.actions.append(self.action_about)

        self.action_settings = QAction(
            QgsApplication.getThemeIcon("console/iconSettingsConsole.svg"),
            i18n.tr("Settings"),
            self.iface.mainWindow(),
        )
        self.action_settings.triggered.connect(
            lambda: self.iface.showOptionsDialog(
                currentPage="mOptionsPage{}".format(__title__)
            )
        )
        self.topaze_menu.addAction(self.action_settings)
        # self.topaze_toolbar.addAction(self.action_settings)
        self.actions.append(self.action_settings)

    def unload(self):
        # (print("unload")
        """Cleans up when plugin is disabled/uninstalled."""
        for action in self.actions:
            self.iface.removePluginMenu(__title__, action)
            self.iface.removeToolBarIcon(action)
        self.sub_menu.clear()
        del self.sub_menu
        # remove the toolbar
        try:
            del self.topaze_toolbar
        except Exception as e:
            ...

    def run(self):
        """Main process.

        :raises Exception: if there is no item in the feed
        """
        # (print("run")
        try:
            self.log(
                message=i18n.tr("Everything ran OK."),
                log_level=3,
                push=False,
            )
        except Exception as err:
            self.log(
                message=i18n.tr("Houston, we've got a problem: {}").format(err),
                log_level=2,
                push=True,
            )

    def show_help_about(self):
        dialog = TopazeAbout()
        dialog.exec_()

    def on_topaze_project_read(self):
        # (print("on_topaze_project_read")
        self.dlg_edit_obs._obs_edited = False
        self._ptopo_array = []
        self._ptopo_layer = TopazeLayers.ptopo_layer()
        if self._ptopo_layer:
            self.work_on_survey = True
        else:
            self.iface.messageBar().pushMessage(
                i18n.tr("PTOPO table missing"), "", Qgis.Info
            )
            self.work_on_survey = False
        self._observations_layer = TopazeLayers.field_layer()
        """
        if not self._observations_layer:
            self.iface.messageBar().pushMessage(
                i18n.tr("Observations table missing"),
                "",
                Qgis.Info,
            )
            self.work_on_survey = False
        """
        self.dlg_helmert_canvas = AdditionalCanvasDialog(self.iface)
        self.dlg_helmert_canvas.setWindowFlags(Qt.Window)  # Non-modal

    def on_topaze_project_cleared(self):
        # (print("on_topaze_project_cleared")
        ...

    def helmert_transformation(self):
        """Helmert transformation"""
        # print("helmert_transformation")
        # self.dlg_helmert.showDialog()
        # results = compute_helmert_transformation()
        # print("results:", results)
        # Créer ou vider les couches temporaires "coordinates"
        self.helmert_group = HelmertGroup()
        self.dlg_helmert_canvas.set_helmert_group(self.helmert_group)
        self.dlg_helmert_canvas.show()

    def create_points(self, refresh: bool = True):
        """Create points"""
        # (print("create_points")
        try:
            nb_created = nb_updated = 0
            features = []
            config = TopazeConfig.config_dict()
            tol_xy = config["duplicated"]["xy_treatment"]["dup_tolerance_xy"]
            tol_z = config["duplicated"]["z_treatment"]["dup_tolerance_z"]
            self._ptopo_layer.startEditing()
            for ptopo in self._ptopo_array:
                feature = None
                if TopazeUtils.ptopo_exists_in_layer(
                    ptopo.matricule, self._ptopo_layer
                ):
                    feature = TopazeUtils.get_feature_by_matricule(
                        ptopo.matricule, ptopo_layer=self._ptopo_layer
                    )
                    old_ptopo = Ptopo.from_feature(feature)
                    x, y, z = TopazeUtils.process_duplicated(
                        old_ptopo, ptopo, tol_xy, tol_z
                    )
                    ptopo.x, ptopo.y, ptopo.z = x, y, z
                if feature:
                    Ptopo.to_feature(ptopo, feature)
                    ok = self._ptopo_layer.updateFeature(feature)
                    nb_updated = nb_updated + 1
                else:
                    if ptopo.z is not None:
                        wkt = f"Point({ptopo.x} {ptopo.y} {ptopo.z})"
                    else:
                        wkt = f"Point({ptopo.x} {ptopo.y} {PtopoConst.UNKNOWN_Z})"
                    geom = QgsGeometry.fromWkt(wkt)
                    feature_dict = {}
                    i = self._ptopo_layer.fields().indexFromName("matricule")
                    if i >= 0:
                        feature_dict.update({i: ptopo.matricule})
                    i = self._ptopo_layer.fields().indexFromName("type_pt")
                    if i >= 0:
                        feature_dict.update({i: ptopo.type})
                    i = self._ptopo_layer.fields().indexFromName("prec_xy")
                    if i >= 0:
                        feature_dict.update({i: ptopo.prec_xy})
                    i = self._ptopo_layer.fields().indexFromName("prec_z")
                    if i >= 0:
                        feature_dict.update({i: ptopo.prec_z})
                    i = self._ptopo_layer.fields().indexFromName("codes")
                    if i >= 0:
                        if ptopo.codes:
                            feature_dict.update({i: ptopo.codes})

                    feature = QgsVectorLayerUtils.createFeature(
                        self._ptopo_layer, geom, feature_dict
                    )
                    features.append(feature)
                    nb_created = nb_created + 1

            self._ptopo_layer.commitChanges()
            if len(features) > 0:
                self._ptopo_layer.dataProvider().addFeatures(features)

            if refresh:
                # If caching is enabled, a simple canvas refresh might not be sufficient
                # to trigger a redraw and you must clear the cached image for the layer
                if self.iface.mapCanvas().isCachingEnabled():
                    self._ptopo_layer.triggerRepaint()
                else:
                    self.iface.mapCanvas().refresh()

        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to add points"),
                str(e),
                Qgis.Warning,
            )

        else:
            return nb_created, nb_updated

    def compute_details_from_station(
        self, station, station_index, obs_array, prec_xy, prec_z
    ):
        if station.x is None or station.y is None:
            pt = TopazeUtils.get_ptopo_by_matricule(
                station.matricule, self._ptopo_array, self._ptopo_layer
            )
            try:
                station.x = pt.x
                station.y = pt.y
                station.z = pt.z
            except Exception as e:
                self.iface.messageBar().pushMessage(
                    i18n.tr("Undetermined station"),
                    "Import/input coordinates, please",
                    Qgis.MessageLevel.Critical,
                )
                return

        references, end_index = TopazeUtils.find_references_in_obs_array(
            obs_array, station_index + 1
        )
        TopazeUtils.get_coordinates_from_references(
            references, self._ptopo_array, self._ptopo_layer
        )
        if station.v0 is None:
            v0 = TopazeCalculator.compute_bearing_pq(station, references)
            station.v0 = v0
        if station.v0 is None:
            station.v0 = 0.0
        next_idx = station_index
        to_compute = True
        config = TopazeConfig.config_dict()
        tol_xy = config["duplicated"]["xy_treatment"]["dup_tolerance_xy"]
        tol_z = config["duplicated"]["z_treatment"]["dup_tolerance_z"]
        for obs in obs_array[station_index + 1 : end_index]:
            next_idx = next_idx + 1
            if obs.type == "st":
                to_compute = False
                break
            elif obs.type in ["pnt", "xyz"]:
                x, y, z = TopazeCalculator.obs_to_xyz(station, obs)
                if x is not None and y is not None:
                    ptopo = Ptopo(
                        obs.matricule, "P", x, y, z, obs.codes, prec_xy, prec_z
                    )
                    try:
                        if obs_array[next_idx].type == "code":
                            ptopo.codes = obs_array[next_idx].codes
                    except Exception:
                        pass
                    if TopazeUtils.ptopo_exists_in_array(
                        ptopo.matricule, self._ptopo_array
                    ):
                        old_ptopo = TopazeUtils.get_ptopo_by_matricule(
                            ptopo.matricule, ptopo_array=self._ptopo_array
                        )
                        x, y, z = TopazeUtils.process_duplicated(
                            old_ptopo, ptopo, tol_xy, tol_z
                        )
                        status = TopazeUtils.update_xyz_in_array(
                            self._ptopo_array, ptopo.matricule, x, y, z
                        )
                        if status:
                            print(f"Ptopo {ptopo.matricule} updated in array")
                        else:
                            ptopo.x, ptopo.y, ptopo.z = x, y, z
                            self._ptopo_array.append(ptopo)
                    else:
                        self._ptopo_array.append(ptopo)
                        print(f"Ptopo {ptopo.matricule} added in array")

    def compute_points(self, type: str = "P", prec_xy: str = "A", prec_z: str = "A"):
        """Compute surveyed points"""
        # (print("compute_points")

        try:
            station = None

            nb_stations = 0
            station_index = -1
            while True:
                station, station_index = TopazeUtils.find_station_in_obs_array(
                    self.obs_array, station_index + 1
                )
                if station is None:
                    if nb_stations < 1:
                        self.iface.messageBar().pushMessage(
                            i18n.tr("No station found "),
                            i18n.tr("Check/modify observations"),
                            Qgis.MessageLevel.Critical,
                        )
                        return
                    break
                nb_stations += 1
                self.compute_details_from_station(
                    station, station_index, self.obs_array, prec_xy, prec_z
                )

            (n_cre, n_upd) = self.create_points()
            self.iface.messageBar().pushMessage(
                i18n.tr("{} surveyed point(s) computed").format(len(self._ptopo_array)),
                i18n.tr(" ({} created, {} updated)").format(n_cre, n_upd),
                Qgis.Info,
            )
        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Error computing points"),
                str(e),
                Qgis.MessageLevel.Critical,
            )
        ...

    def command_compute_points(self):
        """Compute points"""
        self.obs_array = TopazeUtils.load_obs_array(self._obs_storage_mode)
        self.compute_points()

    def command_compute_free_station(self):
        """Compute free_station"""
        self.obs_array = TopazeUtils.load_obs_array(self._obs_storage_mode)
        self.dlg_free_station.showDialog(
            self.obs_array, self._ptopo_array, self._ptopo_layer
        )
        # results = compute_free_station()
        # print("results:", results)

    def command_compute_intersection(self):
        """Compute intersection"""
        self.obs_array = TopazeUtils.load_obs_array(self._obs_storage_mode)
        self.dlg_intersection.showDialog(
            self.obs_array, self._ptopo_array, self._ptopo_layer
        )

    def command_compute_resection(self):
        """Compute resection"""
        self.obs_array = TopazeUtils.load_obs_array(self._obs_storage_mode)
        self.dlg_resection.showDialog(
            self.obs_array, self._ptopo_array, self._ptopo_layer
        )

    def command_compute_trilateration(self):
        """Compute trilateration"""
        self.obs_array = TopazeUtils.load_obs_array(self._obs_storage_mode)
        self.dlg_trilateration.showDialog(
            self.obs_array, self._ptopo_array, self._ptopo_layer
        )

    def command_compute_traverse(self):
        """Compute traverse"""
        self.obs_array = TopazeUtils.load_obs_array(self._obs_storage_mode)
        self.dlg_traverse.showDialog(
            self.obs_array, self._ptopo_array, self._ptopo_layer
        )

    def new_survey_dialog(self):
        """Open dialog New survey"""
        # (print("new_survey_dialog")
        self.dlg_new_survey.set_initial_dir(
            str(os.path.dirname(QgsProject.instance().homePath()))
        )
        result = self.dlg_new_survey.exec()
        if result == QDialog.Accepted:
            to_path = self.dlg_new_survey.get_survey_dir()
            self.new_survey(to_path)

    def new_survey(self, to_path: str):
        """Create new survey by copying template directory"""
        # (print("new_survey")
        try:
            if os.path.exists(to_path):
                m1 = i18n.tr("Creating new survey in :", context="TopazePlugin")
                m2 = i18n.tr(
                    "Do you want override existing directory ?", context="TopazePlugin"
                )
                msg = m1 + f"\n\n{to_path}\n\n" + m2
                if self.show_alert(
                    i18n.tr("Create new survey", context="TopazePlugin"), msg
                ):
                    shutil.rmtree(to_path)
                else:
                    return
            from_path = DIR_PLUGIN_ROOT / "resources/templates/surveying"
            shutil.copytree(from_path, to_path)
            old_qgs = os.path.join(to_path, "surveying.qgs")
            new_qgs = os.path.join(
                to_path, os.path.basename(os.path.normpath(to_path)) + ".qgs"
            )
            os.rename(old_qgs, new_qgs)

        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to create new survey "),
                i18n.tr(" in {}".format(to_path)),
                Qgis.MessageLevel.Critical,
            )
        else:
            self.iface.messageBar().pushMessage(
                i18n.tr("New survey created "),
                i18n.tr(" in {}".format(to_path)),
                Qgis.Success,
            )
            QgsProject.instance().read(new_qgs)

    def update_observations_dialog(self):
        """Open dialog Update observations"""
        # (print("update_observations_dialog")
        self.dlg_import_field_data.exec()
        if len(self.dlg_import_field_data.lineEdit.text()) > 0:
            self.update_observations(self.dlg_import_field_data.lineEdit.text())

    def update_observations(self, fullfilepath: str):
        """Update 'observations' table from text file"""
        # (print("update_observations")
        # open .obs file read fied data
        try:
            field_data_type = TopazeUtils.get_field_data_type(fullfilepath)
            if field_data_type not in ["TRIMBLE", "LEICA"]:
                self.iface.messageBar().pushMessage(
                    i18n.tr("Unable to detect format from {}").format(fullfilepath),
                    "",
                    Qgis.MessageLevel.Critical,
                )
                return

            try:
                inputclass = BUILTIN_INPUT_FORMATS[field_data_type]
            except KeyError as e:
                self.iface.messageBar().pushMessage(
                    i18n.tr("{} is not a valid input format").format(field_data_type),
                    "",
                    Qgis.MessageLevel.Critical,
                )
                return
            else:
                if isinstance(inputclass, tuple):
                    try:
                        # builtin format parser
                        mod, cls, name = inputclass
                        format_parser = getattr(
                            importlib.import_module(
                                f"topaze.totalopenstation.formats.{mod}"
                            ),
                            cls,
                        )
                    except ImportError as e:
                        self.iface.messageBar().pushMessage(
                            i18n.tr("Error importing {} from {}").format(cls, mod),
                            str(e),
                            Qgis.MessageLevel.Critical,
                        )
                        return

            with open(
                fullfilepath, "r", encoding=TopazeUtils.find_file_encoding(fullfilepath)
            ) as field_file:
                fp = format_parser(field_file.read(), point_starts_on="5", use_all=True)
                field_file.close()
                rl = fp.raw_line

                of = OutputFormat(rl)
                new_survey = of.process()

        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to open {}").format(fullfilepath),
                str(e),
                Qgis.Warning,
            )

        else:
            try:
                if self._observations_layer or self._obs_storage_mode == "file":
                    db_field_data = ""
                    db_field_data = self.get_observations(mode=self._obs_storage_mode)
                    new_survey += "\n"
                    if len(db_field_data):
                        new_field_data = db_field_data + new_survey
                    else:
                        new_field_data = new_survey
                    self.set_observations(new_field_data, mode=self._obs_storage_mode)

            except Exception as e:
                self.iface.messageBar().pushMessage(
                    i18n.tr(
                        "Unable to update observations with field data",
                        context="TopazePlugin",
                    ),
                    str(e),
                    Qgis.Warning,
                )
            else:
                self.iface.messageBar().pushMessage(
                    i18n.tr("Observations updated"),
                    i18n.tr(" with field data from {}").format(fullfilepath),
                    Qgis.Success,
                )

    def import_points_dialog(self):
        """Open dialog Import points"""
        self.dlg_import_points.show()

    def import_points(self, fullfilepath: str):
        """Import points  from text file"""
        try:
            if fullfilepath.lower().endswith(".jsi") or fullfilepath.lower().endswith(
                ".txt"
            ):
                from topaze.totalopenstation.formats.topstation_jsi import FormatParser
            with open(
                fullfilepath, "r", encoding=TopazeUtils.find_file_encoding(fullfilepath)
            ) as pointsfile:
                fp = FormatParser(pointsfile.read(), ptopo_only=True)
                pointsfile.close()

        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to open {}").format(fullfilepath),
                str(e),
                Qgis.Info,
            )

        else:
            # get points data
            try:
                rl = fp.raw_line
                from topaze.totalopenstation.output.tops_ptopo import OutputFormat

                of = OutputFormat(rl)
                self._ptopo_array.clear
                self._ptopo_array = of.process()
                nb = len(self._ptopo_array)
                n_cre, n_upd = self.create_points()
            except Exception as e:
                self.iface.messageBar().pushMessage(
                    i18n.tr("Unable to import points from {}").format(fullfilepath),
                    str(e),
                    Qgis.Warning,
                )
            else:
                self.iface.messageBar().pushMessage(
                    i18n.tr("{} point(s) imported from {}").format(nb, fullfilepath),
                    i18n.tr(" ({} created, {} updated)").format(n_cre, n_upd),
                    Qgis.Success,
                )

    def get_observations(self, mode: str = "db"):
        """Get observations from .obs file or from 'observations' table in DB

        :param mode: 'db' to get observations from 'observations' table, \
        'file' to get observations from .obs file, defaults to "db"
        :type mode: str, optional
        :param fullfilepath: full path to .obs file, defaults to ""
        :type fullfilepath: str, optional
        :return: observations data
        :rtype: str
        """
        field_data = ""
        if mode == "db":
            field_data = self.get_observations_from_db()
        elif mode == "file":
            fullfilepath = (
                QgsProject.instance().homePath()
                + "/db/"
                + QgsProject.instance().baseName()
                + ".obs"
            )
            field_data = self.get_observations_from_obsfile(fullfilepath)
        if not field_data:
            field_data = ""
        return field_data

    def get_observations_from_obsfile(self, fullfilepath: str):
        """Get observations from .obs file

        :param fullfilepath: _description_
        :type fullfilepath: str
        :return: _description_
        :rtype: _type_
        """
        try:
            with open(
                fullfilepath, "r", encoding=TopazeUtils.find_file_encoding(fullfilepath)
            ) as field_file:
                field_data = field_file.read()
                field_file.close()

        except Exception as e:
            s = str(e)
            return None

        else:
            return field_data

    def get_observations_from_db(self):
        """_summary_

        :param returns_rows: _description_, defaults to True
        :type returns_rows: bool, optional
        :return: _description_
        :rtype: _type_
        """
        try:
            field_data = ""
            if self._observations_layer:
                feature_dict = {}
                idx_obs_column = self._observations_layer.fields().indexFromName("obs")
                obs_iter = self._observations_layer.getFeatures()
                if obs_iter.isValid():
                    feature = QgsFeature()
                    if obs_iter.nextFeature(feature):
                        if idx_obs_column >= 0:
                            field_data = feature.attribute("obs")
                            if field_data == "\n":
                                field_data = ""

        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to get field data"),
                str(e),
                Qgis.Info,
            )
            return None

        else:
            return field_data

    def set_observations(self, new_data, mode: str = "db"):
        """Set observations in .obs file or in 'observations' table in DB

        :param new_data: observations data to set
        :type new_data: str
        :param mode: 'db' to set observations in 'observations' table, \
        'file' to set observations in .obs file, defaults to "db"
        :type mode: str, optional
        :param fullfilepath: full path to .obs file, defaults to ""
        :type fullfilepath: str, optional
        """
        if mode == "db":
            old_data = self.get_observations(mode=self._obs_storage_mode)
            self.set_observations_in_db(new_data, old_data)
        elif mode == "file":
            fullfilepath = (
                QgsProject.instance().homePath()
                + "/db/"
                + QgsProject.instance().baseName()
                + ".obs"
            )
            self.set_observations_in_obsfile(new_data, fullfilepath)

    def set_observations_in_obsfile(self, new_data, fullfilepath):
        try:
            with open(fullfilepath, "w", encoding="utf-8") as field_file:
                field_file.write(new_data)
                field_file.close()
        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to set field observations in {}").format(fullfilepath),
                str(e),
                Qgis.MessageLevel.Critical,
            )

    def set_observations_in_db(self, new_data, old_data):
        try:
            idx_obs_column = self._observations_layer.fields().indexFromName("obs")
            obs_iter = self._observations_layer.getFeatures()
            if obs_iter.isValid():
                feature = QgsFeature()
                if obs_iter.nextFeature(feature):
                    if idx_obs_column >= 0:
                        self._observations_layer.startEditing()
                        self._observations_layer.changeAttributeValue(
                            feature.id(), idx_obs_column, new_data, old_data
                        )
                        self._observations_layer.commitChanges()
        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to set field observations in table"),
                "",
                Qgis.MessageLevel.Critical,
            )

    def edit_observations(self):
        """Load observations edition dialog"""
        # (print("edit_observations")

        self.dlg_edit_obs.reset_widgets()
        self.dlg_edit_obs._field_rows = []
        self.dlg_edit_obs._filter = "all"
        field_data = self.get_observations(mode=self._obs_storage_mode)
        if field_data is None:
            field_data = ""
        self.dlg_edit_obs.load_observations(field_data.splitlines())

        self.dlg_edit_obs._obs_edited = False
        self.dlg_edit_obs.pushButton_save.setEnabled(False)
        self.dlg_edit_obs.show()

        # self.dlg_edit_obs.comboBox_type_obs.setCurrentIndex(0)

        self.dlg_edit_obs.listWidget_observations.currentItemChanged.connect(
            self.on_list_obs_current_item_changed
        )

        self.dlg_edit_obs.comboBox_type_obs.currentTextChanged.connect(
            self.on_combo_type_obs_current_text_changed
        )

    def process_field_codification(self):
        print("TODO : process_field_codification")
        ...

    def purge_observations(self):
        """Purge observations from .obs file or from 'observations' table in DB"""
        # (print("purge_observations")
        if self._obs_storage_mode == "db":
            self.purge_observations_in_db()
        elif self._obs_storage_mode == "file":
            self.purge_observations_in_obsfile()

    def purge_observations_in_obsfile(self):
        """Remove all observations from .obs file"""
        # (print("purge_observations in obsfile")
        try:
            fullfilepath = (
                QgsProject.instance().homePath()
                + "/db/"
                + QgsProject.instance().baseName()
                + ".obs"
            )
            if os.path.exists(fullfilepath):
                os.remove(fullfilepath)

        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to remove {}").format(fullfilepath),
                str(e),
                Qgis.Warning,
            )

        else:
            self.dlg_edit_obs._obs_edited = False
            self.dlg_edit_obs.pushButton_save.setEnabled(False)
            self.iface.messageBar().pushMessage(
                i18n.tr("Observations erased"), "", Qgis.Info
            )

    def purge_observations_in_db(self):
        """Remove all observations"""
        # (print("purge_observations in db")
        try:
            db_field_data = ""
            if self._observations_layer:
                idx_obs_column = self._observations_layer.fields().indexFromName("obs")
                obs_iter = self._observations_layer.getFeatures()
                if obs_iter.isValid():
                    feature = QgsFeature()
                    if obs_iter.nextFeature(feature):
                        if idx_obs_column >= 0:
                            db_field_data = feature.attribute("obs")
                            new_field_data = ""
                            self._observations_layer.startEditing()
                            self._observations_layer.changeAttributeValue(
                                feature.id(),
                                idx_obs_column,
                                new_field_data,
                                db_field_data,
                            )
                            self._observations_layer.commitChanges()

        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to erase observations"),
                str(e),
                Qgis.Warning,
            )

        else:
            self.dlg_edit_obs._obs_edited = False
            self.dlg_edit_obs.pushButton_save.setEnabled(False)
            self.iface.messageBar().pushMessage(
                i18n.tr("Observations erased"), "", Qgis.Info
            )

    def button_box_ok_new_survey_clicked(self):
        self.new_survey(self.dlg_new_survey.lineEdit.text())

    def button_box_save_field_data_clicked(self):
        if len(self.dlg_import_field_data.lineEdit.text()) > 0:
            self.update_observations(self.dlg_import_field_data.lineEdit.text())

    def button_box_import_points_clicked(self):
        if len(
            self.dlg_import_points.lineEdit.text()
        ) > 0 and self.dlg_import_points.lineEdit.text()[-4:].lower() in [
            ".csv",
            ".jsi",
            ".txt",
        ]:
            self.import_points(self.dlg_import_points.lineEdit.text())

    def on_push_button_add_obs_clicked(self):
        """add new observation"""
        try:
            self.dlg_edit_obs.pushButton_add.clicked.disconnect()
        except TypeError:
            pass
        dico = self.dlg_edit_obs.observation_fields_to_dict(
            self.dlg_edit_obs.comboBox_type_obs.currentText(),
            self.dlg_edit_obs.comboBox_type_point.currentText(),
        )
        rows = TopazeUtils.dict_to_obs_str(dico)
        if len(rows) == 1:
            try:
                idx = self.dlg_edit_obs.listWidget_observations.currentIndex().row()
                if idx == self.dlg_edit_obs.listWidget_observations.count() - 1:
                    self.dlg_edit_obs.listWidget_observations.addItem(rows[0])
                else:
                    self.dlg_edit_obs.listWidget_observations.insertItem(
                        idx + 1, rows[0]
                    )
            except Exception as e:
                self.iface.messageBar().pushMessage(
                    i18n.tr("Unable to add observation"),
                    str(e),
                    Qgis.Warning,
                )
            else:
                self.dlg_edit_obs.listWidget_observations.setCurrentRow(idx + 1)
        elif len(rows) > 1:
            ...
        else:
            ...
        ...
        self.dlg_edit_obs._obs_edited = True
        self.dlg_edit_obs.pushButton_save.setEnabled(True)
        self.dlg_edit_obs.pushButton_add.clicked.connect(
            self.on_push_button_add_obs_clicked
        )

    def on_push_button_change_obs_clicked(self):
        """change observation"""
        try:
            self.dlg_edit_obs.pushButton_change.clicked.disconnect()
        except TypeError:
            pass
        dico = self.dlg_edit_obs.observation_fields_to_dict(
            self.dlg_edit_obs.comboBox_type_obs.currentText(),
            self.dlg_edit_obs.comboBox_type_point.currentText(),
        )
        rows = TopazeUtils.dict_to_obs_str(dico)
        if len(rows) == 1:
            if self._current_obs is not None:
                self._current_obs.setText(rows[0])
        elif len(rows) > 1:
            ...
        else:
            ...
        self.dlg_edit_obs._obs_edited = True
        self.dlg_edit_obs.pushButton_save.setEnabled(True)
        self.dlg_edit_obs.pushButton_change.clicked.connect(
            self.on_push_button_change_obs_clicked
        )

    def on_push_button_delete_obs_clicked(self):
        """delete observation"""
        try:
            self.dlg_edit_obs.pushButton_delete.clicked.disconnect()
        except TypeError:
            pass
        current_row = self.dlg_edit_obs.listWidget_observations.currentRow()
        item = self.dlg_edit_obs.listWidget_observations.takeItem(current_row)
        self.dlg_edit_obs._obs_edited = True
        self.dlg_edit_obs.pushButton_save.setEnabled(True)
        self.dlg_edit_obs.pushButton_delete.clicked.connect(
            self.on_push_button_delete_obs_clicked
        )

    def on_push_button_export_lsci_clicked(self):
        """export codification to csv for LSCI"""
        # (print("export codification for lsci")
        # TODO: valider analyse et écrire csv
        try:
            points = []
            for row in self.dlg_edit_obs._field_rows:
                type = TopazeUtils.type_obs(row)
                if type == "pnt" and len(points) < 1:
                    code = []
                    nb = 0
                    csv_line = row[4:]
                elif type == "code" and len(points):
                    codes = []  # TODO: parse(row)
                    for code in codes:
                        nb = 1
                elif type == "pnt" and len(points) > 0:
                    pass
        except Exception as e:
            print(str(e))

    def on_push_button_save_obs_data_clicked(self):
        """save observations data"""
        if not self.dlg_edit_obs.pushButton_save.isEnabled():
            return
        try:
            new_data = ""
            rows = [
                self.dlg_edit_obs.listWidget_observations.item(x).text()
                for x in range(0, self.dlg_edit_obs.listWidget_observations.count())
            ]
            if len(rows) > 0:
                new_data = "\n".join(rows)
                new_data += "\n"
            self.set_observations(new_data, mode=self._obs_storage_mode)
        except Exception as e:
            self.iface.messageBar().pushMessage(
                i18n.tr("Unable to save field observations"),
                str(e),
                Qgis.Warning,
            )
        else:
            self.iface.messageBar().pushMessage(
                i18n.tr("Field observations data saved"),
                "",
                Qgis.Success,
            )
            self.dlg_edit_obs._obs_edited = False
            self.dlg_edit_obs._field_rows = None
            field_data = self.get_observations(mode=self._obs_storage_mode)
            self.dlg_edit_obs.load_observations(
                field_data.splitlines(), self.dlg_edit_obs._filter
            )
        self.dlg_edit_obs.pushButton_save.setEnabled(False)

    def on_push_button_print_selected_obs_clicked(self):
        """TODO : print selected observations"""
        ...

    def on_edit_observations_close(self, result):
        self.on_push_button_cancel_edition_clicked()

    def on_push_button_cancel_edition_clicked(self):
        """cancel changes and close dialog"""
        if self.dlg_edit_obs._obs_edited:
            if self.show_alert(
                i18n.tr("Edit observations"),
                i18n.tr(
                    "Do you want to save or cancel and drop your changes ?",
                    context="TopazePlugin",
                ),
            ):
                self.on_push_button_save_obs_data_clicked()
        self.dlg_edit_obs._obs_edited = False
        self.dlg_edit_obs.hide()

    def on_list_obs_current_item_changed(self, current, previous):
        """process current row changed"""
        self._current_obs = current
        self._previous_obs = previous
        if current is not None:
            type_obs, type_pt = TopazeUtils.types_obs(current.text())
            if type_pt is not None:
                print(f"type_obs {type_obs}, type_pt {type_pt}")
            else:
                print(f"type_obs {type_obs}")
            self.dlg_edit_obs.update_combobox_types(type_obs, type_pt)
            self.dlg_edit_obs.load_observation_fields(
                type_obs, TopazeUtils.obs_str_to_dict(current.text())
            )

    def on_combo_type_obs_current_text_changed(self, type_obs):
        """process curent type obs changed"""
        self.dlg_edit_obs.update_state_of_input_fields(type_obs)

    def show_alert(self, title, msg):
        msgBox = QMessageBox()
        msgBox.setIcon(QMessageBox.Information)
        msgBox.setText(msg)
        msgBox.setWindowTitle(title)
        msgBox.setStandardButtons(QMessageBox.Save | QMessageBox.Cancel)
        # msgBox.buttonClicked.connect(msgButtonClick)
        returnValue = msgBox.exec()
        if returnValue == QMessageBox.Save:
            return True
        else:
            return False
