"""
/***************************************************************************
 IntersectionDialog
                                 A QGIS plugin
 Topaze
                             -------------------
        begin                : 2022-09-29
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Jean-Marie ARSAC
        email                : jmarsac@arsac.wf
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import json
import os

import numpy as np
from PyQt5 import QtCore, QtWidgets, uic
from qgis.core import Qgis, QgsProject
from qgis.PyQt.QtCore import QDir, QUrl

from topaze.calc.calc_helpers import v0_is_computable_map_from_tempfile
from topaze.calc.intersection import Intersection
from topaze.dlg_util import DlgUtil
from topaze.file_utils import FileUtils
from topaze.ptopo import Ptopo, PtopoConst
from topaze.toolbelt import i18n
from topaze.topaze_calculator import TopazeCalculator
from topaze.topaze_utils import TopazeUtils

FORM_CLASS, _ = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "triangulation_dialog.ui")
)


class IntersectionDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(IntersectionDialog, self).__init__(parent)
        # Set up the user interface from Designer.
        # After setupUI you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        self.setWindowTitle(i18n.tr("Compute isolated point by intersection"))
        self.comboBox_compute_station.currentTextChanged.connect(self.load_stations)
        self.listWidget_available.currentTextChanged.connect(
            self.display_available_coordinates
        )
        self.listWidget_useful.currentTextChanged.connect(
            self.display_useful_coordinates
        )
        self.pushButton_use.clicked.connect(self.use_it)
        self.pushButton_unuse.clicked.connect(self.unuse_it)
        self.pushButton_use_all.clicked.connect(self.use_all)
        self.pushButton_unuse_all.clicked.connect(self.unuse_all)
        self.pushButton_compute.clicked.connect(self.on_pushbutton_compute_clicked)
        self.pushButton_quit.clicked.connect(self.on_pushbutton_quit_clicked)
        self.pushButton_save.clicked.connect(self.on_pushbutton_save_clicked)
        self._obs_array = None
        self.iface = None
        self._ptopo_array = None
        self._ptopo_layer = None

    def set_iface(self, iface):
        self.iface = iface

    def showDialog(self, obs_array, ptopo_array=None, ptopo_layer=None):
        self._obs_array = obs_array
        self._ptopo_array = ptopo_array
        self._ptopo_layer = ptopo_layer
        self._station_matricules = None
        matricules = TopazeUtils.find_all_reference_matricules_in_obs_array(obs_array)
        reference_matricules = list(set(matricules))
        i = 0
        while i < len(reference_matricules):
            if TopazeUtils.ptopo_exists_in_array(reference_matricules[i], obs_array):
                reference_matricules.pop(i)
            else:
                i += 1
        DlgUtil.load_combobox_list_array(
            self.comboBox_compute_station, reference_matricules, False
        )
        self.show()

    def on_pushbutton_compute_clicked(self):
        to_compute = [self.comboBox_compute_station.currentText()]
        references = []
        if self.listWidget_useful.count() < 2:
            return
        for index in range(self.listWidget_useful.count()):
            item = self.listWidget_useful.item(index)
            references.append(item.text())
        # print(to_compute)
        # print(references)
        self.save_data_to_compute_intersection(
            "intersection.json", self._obs_array, to_compute, references
        )
        self.clear_coordinates_fields()
        pt_raw, pt_cmp, stats = Intersection.compute_intersection()
        if pt_raw[0] is None and pt_raw[1] is None:
            return

        if pt_raw[0] is not None and pt_raw[1] is not None:
            self.lineEdit_x_raw.setText(f"{pt_raw[0]:>16.4f}")
            self.lineEdit_y_raw.setText(f"{pt_raw[1]:>16.4f}")
            if pt_raw[2] is not None:
                self.lineEdit_z_raw.setText(f"{pt_raw[2]:>16.4f}")
            else:
                self.lineEdit_z_raw.setText("")
        if pt_cmp[0] is not None and pt_cmp[1] is not None:
            self.lineEdit_x_cmp.setText(f"{pt_cmp[0]:>16.4f}")
            self.lineEdit_y_cmp.setText(f"{pt_cmp[1]:>16.4f}")
            if pt_cmp[2] is not None:
                self.lineEdit_z_cmp.setText(f"{pt_cmp[2]:>16.4f}")
            else:
                self.lineEdit_z_cmp.setText("")

        if (
            stats
            and stats.get("std_x_m", None) is not None
            and stats.get("std_y_m", None) is not None
        ):
            self.lineEdit_emq_x.setText(f"{stats['std_x_m']:>16.4f}")
            self.lineEdit_emq_y.setText(f"{stats['std_y_m']:>16.4f}")
            if (
                stats
                and stats["zstats"]
                and stats["zstats"].get("EMQ_Z_m", None) is not None
            ):
                self.lineEdit_emq_z.setText(f"{stats['zstats']['EMQ_Z_m']:>16.4f}")

        target_path = QgsProject.instance().homePath() + "/rapport/intersection.pdf"
        msg1 = i18n.tr("Intersection calculation of ") + to_compute[0]
        msg2 = i18n.tr(' report in folder <a href="{}">{}</a>').format(
            QUrl.fromLocalFile(target_path).toString(),
            QDir.toNativeSeparators(target_path),
        )
        self.iface.messageBar().pushMessage(msg1, msg2, Qgis.Success, 10)

    def on_pushbutton_quit_clicked(self):
        FileUtils.remove_temp_file("intersection.json")
        self.hide()

    def on_pushbutton_save_clicked(self):
        station_id = self.comboBox_compute_station.currentText()
        pt = Ptopo(
            matricule=station_id,
            type="R",
            x=float(self.lineEdit_x_cmp.text()),
            y=float(self.lineEdit_y_cmp.text()),
            z=(
                float(self.lineEdit_z_cmp.text())
                if self.lineEdit_z_cmp.text()
                else PtopoConst.UNKNOWN_Z
            ),
        )
        if TopazeUtils.ptopo_exists_in_array(pt.matricule, self._obs_array):
            TopazeUtils.update_xyz_in_array(
                self._obs_array, pt.matricule, pt.x, pt.y, pt.z
            )
        else:
            self._obs_array.append(pt)
        fid, created = TopazeUtils.upsert_ptopo_in_layer(pt, True, self.iface)
        if fid:
            msg1 = i18n.tr("Intersection")
            if created:
                msg2 = i18n.tr("PTOPO #{} ({}) created.").format(str(fid), station_id)
            else:
                msg2 = i18n.tr("PTOPO #{} ({}) updated.").format(str(fid), station_id)
            self.iface.messageBar().pushMessage(msg1, msg2, Qgis.Success, 10)

    def load_stations(self):
        reference_matricule = self.comboBox_compute_station.currentText()
        if reference_matricule and self._obs_array:
            self._station_matricules = (
                TopazeUtils.find_all_station_matricules_aiming_reference_in_obs_array(
                    self._obs_array, reference_matricule
                )
            )
            # --------------------------
            self.save_data_to_compute_intersection(
                "calculable_v0.json",
                self._obs_array,
                [reference_matricule],
                self._station_matricules,
            )
            stations_with_v0 = v0_is_computable_map_from_tempfile(
                "calculable_v0.json",
                require_ah=True,
                require_known_xy=True,
                min_refs=1,
                verbose=False,
            )
            self._station_matricules = [
                key
                for key, value in stations_with_v0.items()
                if value and key != reference_matricule
            ]
            # --------------------------
            self.listWidget_available.clear()
            self.listWidget_useful.clear()
            DlgUtil.load_combobox_list_array(
                self.listWidget_available, self._station_matricules, False
            )

    def use_it(self):
        selectedItems = self.listWidget_available.selectedItems()
        if not selectedItems:
            return

        for item in selectedItems:
            self.listWidget_available.takeItem(self.listWidget_available.row(item))
            self.listWidget_useful.addItem(item)

        self.clear_coordinates_fields()

    def unuse_it(self):
        selectedItems = self.listWidget_useful.selectedItems()
        if not selectedItems:
            return

        for item in selectedItems:
            self.listWidget_useful.takeItem(self.listWidget_useful.row(item))
            self.listWidget_available.addItem(item)

        self.clear_coordinates_fields()

    def use_all(self):
        items = self.listWidget_available.findItems("", QtCore.Qt.MatchContains)
        if not items:
            return

        for item in items:
            self.listWidget_available.takeItem(self.listWidget_available.row(item))
            self.listWidget_useful.addItem(item)

        self.clear_coordinates_fields()

    def unuse_all(self):
        items = self.listWidget_useful.findItems("", QtCore.Qt.MatchContains)
        if not items:
            return

        for item in items:
            self.listWidget_useful.takeItem(self.listWidget_useful.row(item))
            self.listWidget_available.addItem(item)
        self.clear_coordinates_fields()

    def display_coordinates(self, matricule):
        try:
            # print(matricule)
            self.clear_coordinates_fields()
            pt = TopazeUtils.get_ptopo_by_matricule_in_obs_array(
                matricule, self._obs_array
            )
            if pt is None:
                pt = TopazeUtils.get_ptopo_by_matricule(
                    matricule, self._ptopo_array, self._ptopo_layer
                )
            if pt:
                self.lineEdit_x_raw.setText(str(pt.x))
                self.lineEdit_y_raw.setText(str(pt.y))
                self.lineEdit_z_raw.setText(str(pt.z))
        except Exception as e:
            print(str(e))

    def display_available_coordinates(self):
        try:
            matricule = self.listWidget_available.currentItem().text()
            self.display_coordinates(matricule)
        except Exception as e:
            print(str(e))

    def display_useful_coordinates(self):
        try:
            matricule = self.listWidget_useful.currentItem().text()
            self.display_coordinates(matricule)
        except Exception as e:
            print(str(e))

    def save_data_to_compute_intersection(
        self, tmp_filename, obs_array, to_compute_array, reference_array
    ):
        data_dict = {"calcul": to_compute_array, "stations": [], "obs": []}

        stations = TopazeUtils.find_all_stations_in_obs_array(obs_array)
        for station in stations:
            if station.matricule in list(set(to_compute_array + reference_array)):
                dico = {"matricule": station.matricule}
                pt = TopazeUtils.get_ptopo_by_matricule_in_obs_array(
                    station.matricule, obs_array
                )
                if pt is None:
                    pt = TopazeUtils.get_ptopo_by_matricule(
                        station.matricule, self._ptopo_array, self._ptopo_layer
                    )
                if pt:
                    if not pt.z or pt.z <= PtopoConst.UNKNOWN_Z:
                        dico.update({"x": pt.x, "y": pt.y, "z": PtopoConst.NO_Z})
                    else:
                        dico.update({"x": pt.x, "y": pt.y, "z": pt.z})
                else:
                    dico.update(
                        {
                            "x": PtopoConst.NO_X,
                            "y": PtopoConst.NO_Y,
                            "z": PtopoConst.NO_Z,
                        }
                    )
                data_dict["stations"].append(dico)

            reference_matricules = (
                TopazeUtils.find_all_reference_matricules_in_obs_array(
                    obs_array, station.matricule
                )
            )
            for matricule in reference_matricules:
                dico = {"matricule": matricule}
                pt = TopazeUtils.get_ptopo_by_matricule_in_obs_array(
                    matricule, obs_array
                )
                if pt is None:
                    pt = TopazeUtils.get_ptopo_by_matricule(
                        matricule, self._ptopo_array, self._ptopo_layer
                    )
                if pt:
                    dico.update({"x": pt.x, "y": pt.y, "z": pt.z})
                else:
                    dico.update(
                        {
                            "x": PtopoConst.NO_X,
                            "y": PtopoConst.NO_Y,
                            "z": PtopoConst.NO_Z,
                        }
                    )
                if dico not in data_dict["stations"]:
                    data_dict["stations"].append(dico)
            references = TopazeUtils.find_all_references_in_obs_array(
                obs_array, station.matricule
            )
            for ref in references:
                if ref.matricule in reference_matricules:
                    dico = {"origine": station.matricule, "hi": station.hi}
                    dico.update(
                        {
                            "origine": station.matricule,
                            "hi": station.hi,
                            "cible": ref.matricule,
                            "ah": ref.ah,
                            "av": ref.av,
                            "di": ref.di,
                            "hp": ref.hp,
                        }
                    )
                    data_dict["obs"].append(dico)
        data_s = json.dumps(data_dict, indent=4)
        fullfilepath = FileUtils.create_temp_file(tmp_filename, data_s)
        # print(fullfilepath)

    def clear_coordinates_fields(self):
        self.lineEdit_x_raw.setText("")
        self.lineEdit_y_raw.setText("")
        self.lineEdit_z_raw.setText("")
        self.lineEdit_x_cmp.setText("")
        self.lineEdit_y_cmp.setText("")
        self.lineEdit_z_cmp.setText("")
        self.lineEdit_emq_x.setText("")
        self.lineEdit_emq_y.setText("")
        self.lineEdit_emq_z.setText("")
