# -*- coding: utf-8 -*-
"""
/***************************************************************************
 TimeEditor
                                 A QGIS plugin
 Used to facilitate the editing of features with lifespan information
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-06-05
        git sha              : $Format:%H$
        copyright            : (C) GPL 3.0 2020 by Niklas Alt
        email                : niklas.alt [at] uni-marburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.core import QgsFeature, QgsGeometry, Qgis, QgsVectorLayer, QgsProject, QgsField
from qgis.gui import QgsIFeatureSelectionManager
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QThread, QVariant, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QInputDialog, QToolButton, QMenu

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialogs
from .time_editor_split_dialog import TimeEditorSplitDialog
from .time_editor_base_dialog import TimeEditorBaseDialog
from .time_editor_expand_dialog import TimeEditorExpandDialog
from .time_editor_inspect_dialog import TimeEditorInspectDialog
from .time_editor_proof_dialog import TimeEditorProofDialog
from .time_editor_date_helper import TimeEditorDateHelper
from .time_editor_dockwidget_timesplit import TimeSplitDockWidget

import os.path

import copy
import datetime
import re


class TimeSplitConfig:

    def __init__(self, edited_ids=[], date_str="", documentation_str="", original_layer=None, tmp_layer=None):
        self.edited_ids = edited_ids
        self.date_str = date_str
        self.documentation_str = documentation_str
        self.original_layer = original_layer
        self.tmp_layer = tmp_layer


class TimeEditor:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'time_editor_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&TimeEditor')

        self.date_helper = TimeEditorDateHelper()
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Variable to keep track of the edit values to avoid reentering the same
        # data all over again
        self.last_time_edit = None
        self.last_docstring_edit = None

        self.time_split_dock_widget = None
        self.current_time_split_config = TimeSplitConfig()

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('TimeEditor', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            # Code from reloader_plugin
            # if main:
            #     self.toolButton = QToolButton()
            #     self.toolButton.setMenu(QMenu())
            #     self.toolButton.setPopupMode(QToolButton.MenuButtonPopup)
            #     self.toolBtnAction = self.iface.addToolBarWidget(self.toolButton)
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/time_editor/icon.png'
        self.main_action = self.add_action(
            icon_path,
            text=self.tr(u'Time Editor Settings'),
            callback=self.run,
            parent=self.iface.mainWindow())

        self.proof_action = self.add_action(
            None,
            text=self.tr("Add Proof"),
            callback=self.key_add_proof,
            add_to_toolbar=False,
            parent=self.main_action
        )

        # TODO: Find international keyboard shortcut
        # self.proof_action.setShortcut("Ctrl+Alt+Shift+P")
        self.proof_action.setShortcut("Ctrl+Ä")

        self.time_split_action = self.add_action(
            None,
            text=self.tr("Time Split Features"),
            callback=self.key_time_split_features,
            add_to_toolbar=False,
            parent=self.main_action
        )
        self.time_split_action.setShortcut("Ctrl+T")
        # self.iface.registerMainWindowAction(time_split_action, "Ctrl+T")

        # TODO: set appropriate icon
        self.time_expand_action = self.add_action(
            None,
            text=self.tr("Time Expand Features"),
            callback=self.key_time_expand_features,
            add_to_toolbar=False,
            parent=self.main_action
        )
        self.time_expand_action.setShortcut("Ctrl+E")

        self.inspect_layer_action = self.add_action(
            None,
            text=self.tr("Inspect Layer"),
            callback=self.key_inspect_layer,
            add_to_toolbar=False,
            parent=self.main_action
        )
        self.inspect_layer_action.setShortcut("Ctrl+I")

        self.filter_date_action = self.add_action(
            None,
            text=self.tr("Filter Date"),
            callback=self.key_filter_layer,
            add_to_toolbar=False,
            parent=self.main_action
        )

        # will be set False in run()
        self.first_start = True

    def _check_variables_set(self, layer):
        return layer.customProperty("te_time_start") != None and \
            layer.customProperty("te_time_end") != None and \
            layer.customProperty("te_documentation") != None

    def key_filter_layer(self):
        dlg = QInputDialog()
        text, ok = QInputDialog.getText(dlg, self.tr(
            "Filter Features to Year"), self.tr("Year: "))
        if not ok:
            return False
        date_str = text
        if not self.date_helper.validate_date_string(date_str):
            return False

        layer = self.iface.activeLayer()
        if layer:
            if not self._check_variables_set(layer):
                self._display_set_attributes_first_message()
                self.run()
            filter_str = self.date_helper.build_filter_string(layer, date_str)
            layer.setSubsetString(filter_str)

    # Expand or shorten the lifespan

    def key_time_expand_features(self):
        layer = self.iface.activeLayer()

        if layer:
            if not self._check_variables_set(layer):
                self._display_set_attributes_first_message()
                self.run()

            life_start_name = layer.customProperty("te_time_start")
            life_start_idx = layer.dataProvider().fieldNameIndex(life_start_name)
            life_end_name = layer.customProperty("te_time_end")
            life_end_idx = layer.dataProvider().fieldNameIndex(life_end_name)
            common_id_name = layer.customProperty("te_common_id")
            common_id_idx = layer.dataProvider().fieldNameIndex(common_id_name)
            documentation_name = layer.customProperty("te_documentation")
            documentation_idx = layer.dataProvider().fieldNameIndex(documentation_name)

            count_selected = layer.selectedFeatureCount()
            if count_selected == 0:
                QMessageBox.information(self.iface.mainWindow(), self.tr("Information"),
                                        self.tr("You need to select at least one feature to start editing the lifespan. Select and press Ctrl+E."))

            else:
                # text, ok = QInputDialog.getText(None, 'Enter Date', 'Enter date (YYYY-MM-DD):')
                expand_values = self.runExpandDialog()
                if expand_values:
                    date_str = expand_values["date"]
                    doc_str = expand_values["docstring"]
                    mode = expand_values["mode"].replace(" ", "_").lower()

                    id_features = layer.selectedFeatureIds()

                    # Run some sanity checks on all selected Features
                    errors = []
                    warnings = []
                    # collect all values from the selected Features
                    # to see if the current state of the time-attribute
                    # (depending on mode) is the same for all features
                    time_values = {}
                    for i in id_features:
                        curr_feature = layer.getFeature(i)
                        life_start = curr_feature.attribute(life_start_name)
                        life_end = curr_feature.attribute(life_end_name)
                        time_value_name = None
                        # TODO: use translation safe mode (occurs multiple times)
                        if mode == "life_start" or mode == "zeit_beginn":
                            time_value_name = life_start_name
                            # TODO: handle case where end date is null
                            if date_str >= life_end:
                                errors.append([i, self.tr("The supplied value ({date_str}) exceeds the life_end value ({life_end}) of the feature (ID: {_id})").format(
                                    _id=i, date_str=date_str, life_end=life_end
                                )])
                            # TODO: Check for implications on adjacent feature
                        elif mode == "life_end" or mode == "zeit_ende":
                            time_value_name = life_end_name
                            if date_str <= life_start:
                                errors.append([i, self.tr("The supplied value ({date_str}) preceeds the life_start value ({life_start}) of the feature (ID: {_id})").format(
                                    _id=i, date_str=date_str, life_start=life_start
                                )])
                        time_value = curr_feature.attribute(time_value_name)
                        if time_value not in time_values:
                            time_values[time_value] = 1
                        else:
                            time_values[time_value] += 1
                            # TODO: Check for implications on adjacent features
                    if len(list(time_values.keys())) > 1:
                        msgBox = QMessageBox()
                        date_diff_str = "\n".join(
                            [key + " (" + str(time_values[key]) + " times)" for key in time_values])
                        msgBox.setText((self.tr("Warning: the selected features have different values for {value_type}. ") +
                                        self.tr("Click OK to proceeed, or cancel to review the attributes of the selected features.\n\n") +
                                        date_diff_str).format(value_type=time_value_name))
                        msgBox.setWindowTitle(self.tr("Warning"))
                        msgBox.setStandardButtons(
                            QMessageBox.Ok | QMessageBox.Cancel)
                        msgBox.setIcon(QMessageBox.Warning)
                        returnValue = msgBox.exec()
                        if not returnValue == QMessageBox.Ok:
                            return False
                        # TODO: Display Dialog to check if we should proceed anyway
                    if len(errors):
                        QMessageBox.critical(self.iface.mainWindow(), self.tr("Error"), self.tr("The operation could not be executed, the following {num} critical errors occurred:\n").format(num=len(errors)) +
                                             "\n".join(error[1] for error in errors))
                        # self.iface.messageBar().pushMessage(
                        #     "Changes rejected! Errors...", level=Qgis.Critical)
                        return None
                    for i in id_features:
                        curr_feature = layer.getFeature(i)
                        life_start = curr_feature.attribute(life_start_name)
                        life_end = curr_feature.attribute(life_end_name)
                        # find all features for a common ID
                        all_common_id_features = layer.getFeatures("{common_id_name} = '{common_id_value}'".format(
                            common_id_name=common_id_name,
                            common_id_value=curr_feature.attribute(
                                common_id_name)
                        ))
                        adj_ids = []
                        # some features don't have adjacent values
                        has_adjacent = False  # remove?
                        adjacent_feature = None
                        # TODO: Check if date_str vals affects more than one feature
                        if mode == "life_start" or mode == "zeit_beginn":
                            compare_to_name = life_end_name
                        else:
                            compare_to_name = life_start_name

                        # Iterate over all features linked to the same
                        # id, and check if the dates are adjacent depending
                        # on the selected mode
                        for common_sub_feature in all_common_id_features:
                            # skip the feature itself
                            if common_sub_feature.id() != curr_feature.id():
                                current_feature_life_start = common_sub_feature.attribute(
                                    common_sub_feature.fieldNameIndex(life_start_name))
                                current_feature_life_end = common_sub_feature.attribute(
                                    common_sub_feature.fieldNameIndex(life_end_name))

                                if (mode == "life_start" or mode == "zeit_beginn"):
                                    if current_feature_life_end:
                                        if self.date_helper.dates_touch(current_feature_life_end, life_start):
                                            adjacent_feature = common_sub_feature
                                else:
                                    if current_feature_life_start:
                                        if self.date_helper.dates_touch(life_end, current_feature_life_start):
                                            adjacent_feature = common_sub_feature
                        # apply the dates to the adjacent feature
                        if adjacent_feature:

                            if mode == "life_start" or mode == "zeit_beginn":
                                layer.changeAttributeValue(adjacent_feature.id(),
                                                           life_end_idx,
                                                           self.date_helper.get_adjacent_date(date_str, False))
                                # adjacent_feature.setAttribute(5, date_str)
                            elif mode == "life_end" or mode == "zeit_ende":
                                layer.changeAttributeValue(
                                    adjacent_feature.id(),
                                    life_start_idx,
                                    self.date_helper.get_adjacent_date(date_str, True))
                                # adjacent_feature.setAttribute(4, date_str)
                            # TODO: use function
                            comment = adjacent_feature.attribute(
                                common_sub_feature.fieldNameIndex(documentation_name))
                            if comment.isNull():
                                comment = ""
                            else:
                                comment = str(comment)
                            comment += "@timeexpand " + date_str + "#" + doc_str + "\n"
                            layer.changeAttributeValue(
                                adjacent_feature.id(),
                                documentation_idx,
                                comment)

                        # apply the date change to the selected feature itself
                        if mode == "life_start" or mode == "zeit_beginn":
                            layer.changeAttributeValue(
                                curr_feature.id(),
                                life_start_idx,
                                date_str)

                        elif mode == "life_end" or mode == "zeit_ende":
                            layer.changeAttributeValue(
                                curr_feature.id(),
                                life_end_idx,
                                date_str)

                        # TODO: use function for comment editing
                        comment = curr_feature.attribute(
                            common_sub_feature.fieldNameIndex(documentation_name))
                        if comment.isNull():
                            comment = ""
                        else:
                            comment = str(comment)
                        comment += "@timeexpand " + date_str + "#" + doc_str + "\n"
                        layer.changeAttributeValue(
                            curr_feature.id(),
                            documentation_idx,
                            comment)

                    layer.commitChanges()

    def _display_set_attributes_first_message(self):
        QMessageBox.information(self.iface.mainWindow(), self.tr("Information"),
                                self.tr("Before you start, you need to set some information on the field names for time, documentation and common id"))

    def key_add_proof(self):
        layer = self.iface.activeLayer()

        if layer:
            if not self._check_variables_set(layer):
                self._display_set_attributes_first_message()
                self.run()
            documentation_name = layer.customProperty("te_documentation")
            documentation_idx = layer.dataProvider().fieldNameIndex(documentation_name)
            count_selected = layer.selectedFeatureCount()
            if count_selected == 0:
                QMessageBox.information(self.iface.mainWindow(), self.tr("Information"),
                                        self.tr("You need to select at least one feature to edit. Select at least one feature and press Strg+Alt+Shift+P").format(num=count_selected))
            else:
                result = self.runProofDialog()

                if result:
                    self.last_time_edit = result["date"]
                    self.last_docstring_edit = result["docstring"]
                    date_str = result["date"]
                    proof_level = result["proof_level"]
                    doc_str = result["docstring"]
                    layer.startEditing()
                    id_features = layer.selectedFeatureIds()

                    for i in id_features:
                        curr_feature = layer.getFeature(i)
                        comment = str(
                            curr_feature.attribute(documentation_name))
                        comment += f"@proof-{str(proof_level)} {date_str}#{doc_str}\n"
                        layer.changeAttributeValue(
                            curr_feature.id(),
                            documentation_idx,
                            comment)
                    layer.commitChanges()

    # TODO: should work with deep copy inheritance
    def _copy_feature(self, feature, copy_geom=True):
        """
        Creates and returns a new QGSFeature
        """
        new_feature = QgsFeature()
        new_attribs = [None]
        new_attribs.extend(feature.attributes()[1:])
        new_feature.setAttributes(new_attribs)
        # new_feature.setId(id_val)
        if copy_geom:
            new_feature.setGeometry(feature.geometry())
        return new_feature

    def _check_lifespan_validity(self, source_date, feature, life_start_name, life_end_name):
        life_start = feature.attribute(life_start_name)
        life_end = feature.attribute(life_end_name)
        # TODO: Hint auf eigentliches Objekt dessen lifespan passen würde
        if not self.date_helper.date_within_lifespan(source_date, life_start, life_end, True):
            self.iface.messageBar().pushMessage(self.tr("The date you entered ({date_text}) does not match the lifespan of the feature (id={_id}, {start} - {end}). Aborting...").format(
                date_text=source_date,
                _id=feature.id(),
                start=life_start,
                end=life_end
            ), level=Qgis.Critical)
            return False
        else:
            return True

    # Queries for time split value, copies features and applies date va lue to old and
    # new features
    def key_time_split_features(self):
        layer = self.iface.activeLayer()
        if layer:
            if not self._check_variables_set(layer):
                self._display_set_attributes_first_message()
                self.run()
            layer_data_provider = layer.dataProvider()
            life_start_name = layer.customProperty("te_time_start")
            life_start_idx = layer_data_provider.fieldNameIndex(
                life_start_name)
            life_end_name = layer.customProperty("te_time_end")
            life_end_idx = layer_data_provider.fieldNameIndex(life_end_name)
            common_id_name = layer.customProperty("te_common_id")
            common_id_idx = layer_data_provider.fieldNameIndex(common_id_name)
            documentation_name = layer.customProperty("te_documentation")
            documentation_idx = layer_data_provider.fieldNameIndex(
                documentation_name)

            count_selected = layer.selectedFeatureCount()
            # if count_selected = 0 or count_selected > 2:
            # TODO: Reallow single feature mode
            if count_selected != 2:
                QMessageBox.information(self.iface.mainWindow(), self.tr("Information"),
                                        self.tr("You have to select exactly two features to edit (currently {num}). Select two features and press Strg+T").format(num=count_selected))
                return None
            else:
                # We either edit one ore two features
                # Some operations are needed for both edit modes these
                # are prepended to the actual mode
                split_values = self.runSplitDialog()
                if not split_values:
                    return None
                date_text = split_values["date"]
                documentation_text = split_values["docstring"]
                # TODO: Check in dialog itself (listen to change)
                valid_date, error = self.date_helper.validate_date_string(
                    date_text)
                if not valid_date:
                    self.iface.messageBar().pushMessage(self.tr("Error: Entered Date ({date_text}): {validation_msg}").format(
                        date_text=date_text,
                        validation_msg=error
                    ), level=Qgis.Critical)
                    return None
                id_features = layer.selectedFeatureIds()
                # TODO: check layer validity for selected features here and abort if not valid
                # Check if all selected features life-range matches the selected date value
                for i in id_features:
                    curr_feature = layer.getFeature(i)
                    # TODO: should move up (check before temporary layer)
                    valid_lifespan = self._check_lifespan_validity(
                        date_text, curr_feature, life_start_name, life_end_name)
                    # TODO: display message to user
                    if not valid_lifespan:
                        self.iface.messageBar().pushMessage(self.tr("Error: Timespan violation for feature-id {_id} and date {date_text}").format(
                            date_text=date_text,
                            _id=i
                        ), level=Qgis.Critical)
                        return None

                # layer.startEditing()
                tmp_layer = QgsVectorLayer("MultiPolygon?crs={proj_id}".format(
                    proj_id=layer.crs().authid()
                ), "Temporary Edit Layer", "memory", )
                tmp_layer.setOpacity(0.4)
                tmp_layer_data_provider = tmp_layer.dataProvider()
                # Unnecessary, keep track of new geometries with .id() and .setId()
                tmp_field_defs = [QgsField("fid", QVariant.Int)]
                tmp_layer_data_provider.addAttributes(tmp_field_defs)
                tmp_layer.updateFields()

                layer.removeSelection()

                QgsProject.instance().addMapLayer(tmp_layer)
                # TODO:
                # Set as active layer, possibly with
                # root = QgsProject.instance().layerTreeRoot()

                tmp_layer.startEditing()

                # ONE FEATURE MODE
                # Simply copy the feature and set the dates according to split
                # TODO: Reallow single feature mode
                # if count_selected == 1:
                #     QMessageBox.information(self.iface.mainWindow(), self.tr(
                #         "Information"), self.tr("ONE FEATURE MODE"))
                #     feature = layer.getFeature(id_features[0])
                #     valid_lifespan = self._check_lifespan_validity(
                #         date_text, feature, life_start_name, life_end_name)
                #     if not valid_lifespan:
                #         self.iface.messageBar().pushMessage(self.tr("Error: Cannnot split a single feature at ({date_text}) due to lifespan violation").format(
                #             date_text=date_text
                #         ), level=Qgis.Critical)
                #         return None
                #     if feature.attribute(life_end_name) == date_text:
                #         self.iface.messageBar().pushMessage(self.tr("Error: Cannnot split a single feature with same date ({date_text})").format(
                #             date_text=date_text
                #         ), level=Qgis.Critical)
                #         return None
                #     new_feature = self._copy_feature(feature)
                #     layer_data_provider.addFeature(new_feature)
                #     layer.commitChanges()
                #     # set the date on the old feature
                #     layer.changeAttributeValue(feature.id(),
                #                                life_start_idx,
                #                                date_text)

                #     # TODO: duplicate Code
                #     # TODO: use helper functions for date and comment setting
                #     comment = feature.attribute(documentation_name)
                #     comment = str(comment) + "@timesplit " + \
                #         date_text + "#" + source + "\n"
                #     layer.changeAttributeValue(feature.id(),
                #                                documentation_idx,
                #                                comment)
                #     # Set the life-end and documentation on the newly created feature
                #     layer.changeAttributeValue(new_feature.id(),
                #                                life_end_idx,
                #                                self.date_helper.get_adjacent_date(date_text, False))
                #     layer.changeAttributeValue(new_feature.id(),
                #                                documentation_idx,
                #                                comment + "\n@inherit " + str(feature.id()) + "\n")
                #     layer.commitChanges()
                #     copy_count = 1
                # TWO FEATURE MODE
                # Copies both selected features and put them in edit mode

                # TODO: change wenn single feature mode is reallowed
                if count_selected == 2:
                    copy_count = 0
                    # collects the ids of the features that are scheduled to be edited for
                    # the filter
                    for i in id_features:
                        curr_feature = layer.getFeature(i)
                        # Create a simple copy with 1 attribute (will hold the original id)
                        tmp_feature = QgsFeature()
                        tmp_feature.setGeometry(curr_feature.geometry())
                        tmp_feature.initAttributes(1)
                        tmp_feature.setAttribute(0, i)
                        tmp_layer_data_provider.addFeature(tmp_feature)
                        copy_count += 1
                    tmp_layer.commitChanges()
                    tmp_layer.startEditing()
                    # create the dock widget
                    if self.time_split_dock_widget is None:
                        # TODO: initialize widget with data class TimeSplitConfig
                        self.time_split_dock_widget = TimeSplitDockWidget(
                            doc_string=documentation_text, date_string=date_text)
                        self.time_split_dock_widget.abortTimesplit.connect(
                            self.on_time_split_abort)
                        self.time_split_dock_widget.requestCopy.connect(
                            self.on_time_split_copy_request)
                        self.iface.addDockWidget(
                            Qt.TopDockWidgetArea, self.time_split_dock_widget)
                    if self.time_split_dock_widget.isHidden():
                        self.time_split_dock_widget.show()
                    # save the current settings
                    # read in on copy request
                    self.current_time_split_config.date_str = date_text
                    self.current_time_split_config.documentation_str = documentation_text
                    self.current_time_split_config.original_layer = layer
                    self.current_time_split_config.tmp_layer = tmp_layer
                    self.current_time_split_config.edited_ids = id_features

                    # WIll wait until the "requestCopy" signal has fired

                    # # TODO: use comment function
                    # comment = curr_feature.attribute(documentation_name)
                    # comment = str(comment) + "@timesplit " + \
                    #     date_text + "#" + source + "\n"
                    # curr_feature[documentation_name] = comment
                    # # layer.changeAttributeValue(i,
                    # #                            documentation_idx,
                    # #                            comment)
                    # # Set the life-end and documentation on the newly created feature
                    # new_feature[life_end_name] = self.date_helper.get_adjacent_date(
                    #     date_text, False)
                    # # layer.changeAttributeValue(new_feature.id(),
                    # #                            life_end_idx,
                    # #                            self.date_helper.get_adjacent_date(date_text, False))
                    # new_feature[documentation_name] = comment + \
                    #     "\n@inherit " + str(curr_feature.id()) + "\n"

    def on_time_split_copy_request(self):

        split_date = self.current_time_split_config.date_str
        edited_features = self.current_time_split_config.edited_ids
        layer = self.current_time_split_config.original_layer
        tmp_layer = self.current_time_split_config.tmp_layer

        # TODO: Duplicated code, save as class attributes
        layer_data_provider = layer.dataProvider()
        life_start_name = layer.customProperty("te_time_start")
        life_start_idx = layer_data_provider.fieldNameIndex(life_start_name)
        life_end_name = layer.customProperty("te_time_end")
        life_end_idx = layer_data_provider.fieldNameIndex(life_end_name)
        common_id_name = layer.customProperty("te_common_id")
        common_id_idx = layer_data_provider.fieldNameIndex(common_id_name)
        documentation_name = layer.customProperty("te_documentation")
        documentation_idx = layer_data_provider.fieldNameIndex(documentation_name)
        # Iterate over all edited features
        # if the split date differs from the current start_date
        #  -> copy the feature and set
        #       -> the start date of the original feature to split_date
        #       -> set the end date of the copied feature to split_date
        #       -> copy the edited geometry on the copied feature
        # elif the split date is the same as the current start_date
        #       -> no copy (as the time span fits already)
        #       -> no date set
        #       -> copy the edited geometry on the original feature

        # Maps the fid from the original layer to
        # the actual fid to retrieve the geometry
        # TODO: could be part of the SplitConfig class
        geom_dict = {}
        for feat in tmp_layer.getFeatures():
            geom_dict[feat.attributes()[0]] = feat.id()

        layer.startEditing()
        copied_features = []
        for i in edited_features:
            original_feature = layer.getFeature(i)
            adjacent_date = self.date_helper.get_adjacent_date(
                split_date, False)

            # In case there already is a feature with the adjacent date (from 
            # a previous timesplit) -> only change the geometry, no attributes
            if original_feature.attribute(life_end_name) == adjacent_date:
                new_geom = tmp_layer.getFeature(geom_dict[i]).geometry()
                layer.changeGeometry(i, new_geom)
                # Calling .setGeometry does not work for existing features
                # original_feature.setGeometry(new_geom)
            else:
                copied_feature = self._copy_feature(original_feature, False)
                layer.changeAttributeValue(i, life_start_idx, split_date)
                copied_feature.setAttribute(life_end_idx, adjacent_date)
                copied_feature.setGeometry(
                    tmp_layer.getFeature(geom_dict[i]).geometry())
                layer_data_provider.addFeatures([copied_feature])
            feature_id_info = ":".join([str(_id) for _id in edited_features])
            documentation_str = ""
            documentation_content = original_feature.attributes()[documentation_idx]
            # TODO: read up on str vs. QVariant and write get and write function
            if (documentation_content is not None):
                if isinstance(documentation_content, QVariant):
                    documentation_str = str(documentation_content.value()).strip()
            if documentation_str.strip() != "":
                documentation_str = documentation_str + "\n"
            documentation_str += f'timesplit@{feature_id_info}'
            layer.changeAttributeValue(i, documentation_idx, QVariant(documentation_str))
        layer.commitChanges()

        if self.current_time_split_config.tmp_layer:
            QgsProject.instance().removeMapLayer(self.current_time_split_config.tmp_layer)
        self.current_time_split_config = TimeSplitConfig()
        if self.time_split_dock_widget:
            self.time_split_dock_widget.hide()
        # self.iface.messageBar().pushMessage(
        #     self.tr("Copied and reset date value ({date}) on {copy_cnt} out of {num} selected Features.").format(
        #                 date=date_text, copy_cnt=copy_count, num=count_selected
        #             ), level=Qgis.Success)

    def on_time_split_abort(self):
        if (self.current_time_split_config.tmp_layer):
            QgsProject.instance().removeMapLayer(self.current_time_split_config.tmp_layer)
        self.current_time_split_config = TimeSplitConfig()
        if self.time_split_dock_widget:
            self.time_split_dock_widget.hide()

    # TODO: Move to dialog use progress bar
    def key_inspect_layer(self):
        self.abort_layer_inspection = False
        self.runInspectDialog()

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&TimeEditor'),
                action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        self.dlg = TimeEditorBaseDialog(self.iface)
        self.dlg.show()
        result = self.dlg.exec_()

    def runProofDialog(self):
        self.proof_dlg = TimeEditorProofDialog(
            preset_time=self.last_time_edit,
            preset_docstring=self.last_docstring_edit
        )
        self.proof_dlg.show()
        result = self.proof_dlg.exec_()
        if result:
            proof_level = self.proof_dlg.comboBoxMatchingType.currentIndex() + 1
            return {
                "date": self.proof_dlg.lineEdit.text(),
                "proof_level": proof_level,
                "docstring": self.proof_dlg.textEdit.toPlainText()
            }
        else:
            return False

    # TODO: Set current values in dialog
    # TODO: check if expanding violates prior or following
    def runExpandDialog(self):
        self.expand_dlg = TimeEditorExpandDialog()
        self.expand_dlg.show()
        result = self.expand_dlg.exec_()
        if result:
            return {
                "date": self.expand_dlg.lineEdit.text(),
                "docstring": self.expand_dlg.textEdit.toPlainText(),
                "mode": self.expand_dlg.comboBox.currentText()
                # "radio1": self.split_dlg.radioButton.value()
            }
        else:
            return False

    def runSplitDialog(self):
        """Run method for split Features"""
        self.split_dlg = TimeEditorSplitDialog()
        # show the dialog
        self.split_dlg.show()
        # Run the dialog event loop
        result = self.split_dlg.exec_()
        # See if OK was pressed
        if result:
            return {
                "date": self.split_dlg.lineEdit.text().strip(),
                "docstring": self.split_dlg.textEdit.toPlainText().strip()
            }
        else:
            return False

    def runInspectDialog(self):
        layer = self.iface.activeLayer()
        
        if not self._check_variables_set(layer):
            self._display_set_attributes_first_message()
            self.run()
        
        if layer.isEditable():
            QMessageBox.critical(self.iface.mainWindow(), self.tr("Error"), self.tr("Integrity checks are not allowed in edit mode. Please commit your changes and toggle off the edit mode"))
        else:   
            self.inspect_dlg = TimeEditorInspectDialog(self.iface)
            self.inspect_dlg.show()
            result = self.inspect_dlg.exec()
