# -*- coding: utf-8 -*-
"""
/***************************************************************************
 TileCover
                                 A QGIS plugin
 Creates polygon footprints for tiles covering the current viewport
                              -------------------
        begin                : 2014-01-13
        copyright            : (C) 2014 by Christian Wygoda
        email                : code@wygoda.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import math
import os.path

# Import the PyQt and QGIS libraries
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *

# Initialize Qt resources from file resources.py
import resources_rc
# Import the code for the dialog
from tilecoverdialog import TileCoverDialog


class TileCover:

    layer = None
    fid = None

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        locale = QSettings().value("locale/userLocale")[0:2]
        localePath = os.path.join(self.plugin_dir, 'i18n', 'tilecover_{}.qm'.format(locale))

        if os.path.exists(localePath):
            self.translator = QTranslator()
            self.translator.load(localePath)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        self.dlg = TileCoverDialog()

    def initGui(self):
        self.action = QAction(
            QIcon(":/plugins/tilecover/icon.png"),
            u"Tile Cover", self.iface.mainWindow())
        self.action.triggered.connect(self.run)

        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu(u"&Tile Cover", self.action)

    def unload(self):
        self.iface.removePluginMenu(u"&Tile Cover", self.action)
        self.iface.removeToolBarIcon(self.action)

    def run(self):
        self.dlg.show()
        result = self.dlg.exec_()
        if result == 1:
            self.create_cover()

    def create_cover(self, level=None, extent=None):
        if not level:

            self.layer = QgsVectorLayer(
                "Polygon?crs=epsg:4326&field=id:integer&field=x:integer&field=y:integer&field=z:integer",
                'Tile footprint',
                "memory")
            self.fid = 0

            self.layer.startEditing()

            # Get viewport bounds in EPSG:4326
            map = self.iface.mapCanvas()
            mapCrs = map.mapRenderer().destinationCrs()
            epsg4326 = QgsCoordinateReferenceSystem(4326)
            transform = QgsCoordinateTransform(mapCrs, epsg4326)
            extent = transform.transformBoundingBox(map.extent())

            # Add viewport extent as feature if requested
            if self.dlg.addBounds.isChecked():
                feature = QgsFeature()
                feature.setGeometry(QgsGeometry.fromRect(map.extent()))
                feature.setAttributes([None, None, None, None])
                self.layer.addFeature(feature, False)

            z_from = min(self.dlg.zFrom.value(), self.dlg.zTo.value())
            z_to = max(self.dlg.zFrom.value(), self.dlg.zTo.value())
            for level in range(z_from, z_to + 1):
                self.create_cover(level, extent)

            self.layer.commitChanges()
            self.layer.dataProvider().createSpatialIndex()
            self.layer.updateExtents()
            QgsMapLayerRegistry.instance().addMapLayer(self.layer)
            return

        QgsMessageLog.logMessage(
            'Creating cover for level %d' % level,
            None,
            QgsMessageLog.INFO)

        minTile = self.deg2tile(extent.yMaximum(), extent.xMinimum(), level)
        maxTile = self.deg2tile(extent.yMinimum(), extent.xMaximum(), level)
        QgsMessageLog.logMessage(
            'Extent: %f, %f, %f, %f' % (extent.xMinimum(), extent.yMinimum(), extent.xMaximum(), extent.yMaximum()),
            None,
            QgsMessageLog.INFO)
        QgsMessageLog.logMessage(
            'Tiles for level %d: (%d, %d) to (%d, %d)' % (level, minTile[0], minTile[1], maxTile[0], maxTile[1]),
            None,
            QgsMessageLog.INFO)

        for x in range(minTile[0], maxTile[0] + 1):
            for y in range(minTile[1], maxTile[1] + 1):
                feature = QgsFeature()
                feature.setGeometry(self.tileGeometry(x, y, level))
                feature.setAttributes([self.fid, x, y, level])
                self.layer.addFeature(feature, False)
                self.fid += 1

    def deg2tile(self, lat_deg, lon_deg, zoom):
        lat_rad = math.radians(lat_deg)
        n = 2.0 ** zoom
        xtile = int((lon_deg + 180.0) / 360.0 * n)
        ytile = int((1.0 - math.log(math.tan(lat_rad) + (1 / math.cos(lat_rad))) / math.pi) / 2.0 * n)
        return (xtile, ytile)

    def tileGeometry(self, x, y, z):
        n = 2.0 ** z
        xmin = x / n * 360.0 - 180.0
        xmax = (x + 1) / n * 360.0 - 180
        ymin = math.degrees(math.atan(math.sinh(math.pi * (1 - 2 * y / n))))
        ymax = math.degrees(math.atan(math.sinh(math.pi * (1 - 2 * (y + 1) / n))))
        return QgsGeometry.fromRect(QgsRectangle(xmin, ymin, xmax, ymax))
