# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ThemeSwitcherDialog
                                 A QGIS plugin
 This plugin adds a popup to easily switch between layer themes
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2024-07-23
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Roel Huybrechts
        email                : roel@huybrechts.re
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import math
from qgis.PyQt import QtWidgets, QtGui, QtCore

from .translate import Translatable

class ThemeButton(QtWidgets.QToolButton):
    def __init__(self, themeName, theme, themeConfig, dialog):
        """Initialise a ThemeButton.

        This is a button in the dialog to switch to a particular theme.

        Parameters
        ----------
        themeName : str
            Name of the theme.
        theme : str
            Name of the QGIS theme.
        themeConfig : ThemeConfig
            Reference to the ThemeConfig instance
        dialog : ThemeSwitcherDialog
            Reference to the ThemeSwitcherDialog instance
        """
        super().__init__()
        self.themeName = themeName
        self.theme = theme
        self.themeConfig = themeConfig
        self.dialog = dialog

        self.iconPath = ':/plugins/theme_switcher/map.png'
        self.setIcon(QtGui.QIcon(self.iconPath))
        self.setIconSize(QtCore.QSize(64, 64))

        self.setText(self.themeName)
        self.setToolButtonStyle(
            QtCore.Qt.ToolButtonStyle.ToolButtonTextUnderIcon)

        self.clicked.connect(self.switchToTheme)

    def switchToTheme(self):
        """Called when the button is clicked.

        Switch to the theme and close the dialog.
        """
        self.themeConfig.mapThemeCollection.applyTheme(
            self.theme, self.themeConfig.layerTreeRoot, self.themeConfig.layerTreeModel)

        self.dialog.close()


class ThemeSwitcherWidget(QtWidgets.QWidget):
    def __init__(self, parent, themeConfig):
        """Widget containing the ThemeButtons to switch themes.

        If there are groups available, the buttons are grouped per theme group.

        Parameters
        ----------
        parent : ThemeSwitcherDialog
            Reference to parent, in this case ThemeSwitcherDialog instance.
        themeConfig : ThemeConfig
            Reference to the ThemeConfig instance
        """
        super().__init__(parent)

        self.setLayout(QtWidgets.QGridLayout())
        self.layout().setHorizontalSpacing(10)
        self.layout().setVerticalSpacing(10)

        self.themeConfig = themeConfig
        self.dialog = parent
        self.populate()

        self.themeConfig.configChanged.connect(self.populate)

    def populate(self):
        """Populate the widget based on the theme configuration."""
        # remove all groups
        for i in reversed(range(self.layout().count())):
            try:
                self.layout().itemAt(i).widget().setParent(None)
            except AttributeError:
                continue

        def createThemeButton(themeName, theme):
            return ThemeButton(themeName, theme, self.themeConfig, self.dialog)

        def createGroupWidget(groupName):
            groupWidget = QtWidgets.QGroupBox(self)
            groupWidget.setTitle(groupName)
            groupWidget.setFlat(True)
            groupWidget.setLayout(QtWidgets.QHBoxLayout())

            for t in self.themeConfig.themeGroups[groupName]:
                themeName, theme = t
                btn = createThemeButton(themeName, theme)
                groupWidget.layout().addWidget(btn)

            groupWidget.layout().addStretch()
            return groupWidget

        if len(self.themeConfig.themeGroups) == 1 and self.themeConfig.GROUP_OTHER_NAME in self.themeConfig.themeGroups:
            # no user defined group, skip the groups altogether
            size = math.ceil(math.sqrt(len(self.themeConfig.themes)))

            for i in range(size):
                for ix, theme in enumerate(self.themeConfig.themes[(i*size): ((i*size)+size)]):
                    btn = createThemeButton(theme, theme)
                    self.layout().addWidget(btn, i, ix, QtCore.Qt.AlignmentFlag.AlignCenter)
        else:
            # all user defined groups first
            groupNames = sorted(
                [g for g in self.themeConfig.themeGroups.keys() if g != self.themeConfig.GROUP_OTHER_NAME])
            if self.themeConfig.GROUP_OTHER_NAME in self.themeConfig.themeGroups.keys():
                groupNames.append(self.themeConfig.GROUP_OTHER_NAME)

            size = math.ceil(math.sqrt(len(groupNames)))

            for i in range(size):
                for ix, group in enumerate(groupNames[(i*size): ((i*size)+size)]):
                    self.layout().addWidget(createGroupWidget(group), i,
                                            ix, QtCore.Qt.AlignmentFlag.AlignLeft)

        self.adjustSize()


class ThemeSwitcherDialog(QtWidgets.QDialog, Translatable):
    def __init__(self, main):
        """Initialise the dialog.

        Parameters
        ----------
        main : ThemeSwitcher
            Reference to main ThemeSwitcher instance
        """
        QtWidgets.QDialog.__init__(self)
        self.main = main
        self.themeConfig = self.main.themeConfig

        self.iconPath = ':/plugins/theme_switcher/map.png'
        self.setWindowIcon(QtGui.QIcon(self.iconPath))

        self.setWindowTitle(self.tr(u'Theme switcher'))
        self.setLayout(QtWidgets.QVBoxLayout())

        label = QtWidgets.QLabel(self.tr('Choose map theme'))
        labelFont = label.font()
        labelFont.setBold(True)
        labelFont.setPointSize(12)
        label.setFont(labelFont)
        self.layout().addWidget(label)

        self.layout().addWidget(ThemeSwitcherWidget(self, self.themeConfig))

        self.layout().addStretch()

        buttonBox = QtWidgets.QDialogButtonBox(self)
        buttonBox.setOrientation(QtCore.Qt.Orientation.Horizontal)
        buttonBox.setStandardButtons(
            QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel)
        buttonBox.button(
            QtWidgets.QDialogButtonBox.Ok).clicked.connect(self.close)
        buttonBox.button(
            QtWidgets.QDialogButtonBox.Cancel).clicked.connect(self.close)
        self.layout().addWidget(buttonBox)

        self.adjustSize()
