# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ThemeManagerDialog
                                 A QGIS plugin
 This plugin stands to be a theme manager. It allows you to select the layers you want to set up in the different themes in your projet as a checklist, showing up all themes at once.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-05-22
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Antoine BOYER
        email                : theme.manager.qgis.plugin@gmail.com
 ***************************************************************************/
"""

import os
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets, QtCore
from qgis.core import QgsProject, QgsMapThemeCollection
from qgis.PyQt.QtGui import QColor

# Charge la classe et la base de l'interface utilisateur UI (fichier d'interface graphique) à partir du fichier QtDesigner
# Loads the class and the user interface from UI file created with QtDesigner
FORM_CLASS, _ = uic.loadUiType(os.path.join( # FORM_CLASS is a class which represents the UI
    os.path.dirname(__file__), 'theme_manager_dialog_base.ui'))

class ThemeManagerDialog(QtWidgets.QDialog, FORM_CLASS):
            # Initialisation de l'interface utilisateur
            # Initialising user's interface 
    def __init__(self, iface, parent=None):
        # Initialisation de la QDialog et l'UI héritée de FORM_CLASS (parent)
        # Initialising QDialog and UI herited from FORM_CLASS
        super().__init__(parent)
        # Stockage de l'état de l'interface de QGIS pour gérer les thèmes plus tard
        # Saves the state of QGIS's interface to be able to ad, remove, etc... later
        self.iface = iface
        # Initialise l'interface du plugin depuis le .ui
        # Initialising the plugin's interface from the .ui
        self.setupUi(self)
        # Un attribut stocke le dernier thème modifié
        # This attribute saves the last modified theme
        self.last_modified_theme = None  # Aucun thème modifié au lancement du plugin / No theme already modified by the user at the plugin's start

        """
        Connexion des boutons et tableaux aux méthodes / Préparation de la boîte de dialogue
        Connecting buttons and tables to the methods / Initialising the user's interface
        """
        # Si les boutons existent dans le .ui, ils se connectent à leur méthode lorsque l'utilisateur clique dessus
        # If the buttons present in the .ui, they're connected to their respective methods when the user clicks on them
        if hasattr(self, "btnCreateNewTheme"):
            self.btnCreateNewTheme.clicked.connect(self.create_new_theme)
        if hasattr(self, "btnApplyTheme"):
            self.btnApplyTheme.clicked.connect(self.apply_selected_theme)
        if hasattr(self, "table_FilterAndSuppressThemes"):
            self.table_FilterAndSuppressThemes.itemChanged.connect(self.on_filter_changed)

        # Remplit le premier tableau de gestion des thèmes et de filtre d'affichage dans le tableau principal
        # Fills the first table for managing themes and their appearance in the main table
        self.fill_theme_filter_table()
        # Remplit le tableau principale de gestion des couches par thème
        # Fills the main table for managing layers by theme
        self.fill_theme_table()
        # Check s'il y a des thèmes dans le projet sinon une alerte apparaîtra
        # Checks if there are themes in the project otherwise, an alert would be displayed
        self.check_for_themes()

    def check_for_themes(self):
        # Récupère la liste des thèmes du projet grâce à une fonction QGIS
        # Collects the themes list in the project from QGIS fonction
        theme_collection = QgsProject.instance().mapThemeCollection()
        # Les stocke sous forme de liste
        # Stores them as a list
        themes = list(theme_collection.mapThemes())
        # S'il n'y a pas de thèmes dans le projet, une alerte est renvoyée
        # If there is no theme in the projet, an alert is displayed
        if not themes:
            QtWidgets.QMessageBox.information(
                self,
                self.tr("Absence de thème dans le projet"),
                self.tr("Votre projet ne contient pas de thème. Créez-en un depuis le menu QGIS ou dans le premier volet du plugin.")
            )


    def fill_theme_table(self):
        # Récupère la collection des thèmes du projet
        # Collects the theme collection in the projet
        theme_collection = QgsProject.instance().mapThemeCollection()
        # Les stocke sous forme de liste nommée all_themes
        # Stores them as a list named all_themes
        all_themes = list(theme_collection.mapThemes())
        """
        Sélection des thèmes à afficher en colonne
        Selects displayed themes in column
        """
        # Récupère la liste des thèmes à afficher selon le filtre appliqué par l'utilisateur
        # If the table that filters themes exists, it collects the list of themes that are checked
        if hasattr(self, "table_FilterAndSuppressThemes"):
            selected_themes = []
            # La liste commence à la ligne 1 car car la ligne d'indice 0 contient les boutons d'actions
            # The list begins at line 1 because line 0 contains the buttons
            for row in range(1, self.table_FilterAndSuppressThemes.rowCount()):
                item = self.table_FilterAndSuppressThemes.item(row, 0)
                # Si la case est cochée, le thème est ajouté à la liste
                # If the checkbox is checked, the theme is added to the list
                if item and item.checkState() == QtCore.Qt.Checked:
                    selected_themes.append(self.table_FilterAndSuppressThemes.item(row, 1).text())
            themes = selected_themes if selected_themes else all_themes
        # Si le tableau de filtre ne pouvait pas être chargée, tous les thèmes sont affichés
        # If the table wasn't loaded, all themes are displayed
        else:
            themes = all_themes

        # Liste tous les groupes et couches de l'arbre des couches QGIS
        # Creation of a list which contains groups and layers of the QGIS layer tree
        items = []
        def add_items_rec(node, parent_id=None):
            # Parcourt l'arbre des couches, ajoute chaque groupe/couche à la liste items, parent_id permet de connaître la hiérarchie dans l'arbre des couches
            # Recursive function that runs through the layer tree and adds each layer and groups
            for child in node.children():
                if child.nodeType() == child.NodeGroup:
                    # Si la condition est True, c'est un groupe : son id, son nom et l'id du parent (sous-groupe) sont récupérés
                    # If the condition is True, the item is a group : its id, name and the id of the parent (sub-group) are collected
                    group_id = child.name()
                    items.append((True, group_id, child.name(), parent_id))
                    add_items_rec(child, group_id)
                    # Récursif s'il y a des sous-groupes
                    # Recursive if it's a sub-group
                elif child.nodeType() == child.NodeLayer:
                    # False = c'est une couche : récupération de son id, son nom et son parent (groupe/sous-groupe)
                    # False = it's a layer : Its id, name and parent are collected (group/sub-group)
                    items.append((False, child.layerId(), child.name(), parent_id))
        root = QgsProject.instance().layerTreeRoot()
        add_items_rec(root)

        # Prépare le tableau (en-têtes, lignes, colonnes)
        # Initialise the table (header, row, column)
        self.tableThemes.clear() # Remise à 0 du tableau / Erases former configuration of the table
        # Réserve une ligne pour les boutons d'action sur les colonnes
        # Allocates one row for the column action buttons
        self.tableThemes.setRowCount(len(items) + 1)
        # La première colonne accueille les boutons d'action sur les lignes, la seconde les noms et ensuite n colonne pour n thème
        # The first column prints the action buttons on the lines, the second one the themes names and then n column for n theme
        self.tableThemes.setColumnCount(len(themes) + 2)
        # Nommage des colonnes (la première colonne n'a pas d'en-tête car elle contient les boutons d'action)
        # Naming columns (the first one is empty because it's the buttons columns)
        self.tableThemes.setHorizontalHeaderLabels(["", self.tr("Groupe/Couche")] + themes)
        for col in [0, 1]:
            item = QtWidgets.QTableWidgetItem()
            item.setFlags(QtCore.Qt.ItemIsEnabled)
            item.setBackground(QtCore.Qt.gray)
            self.tableThemes.setItem(0, col, item)

        """
        Ajout des boutons d'action pour les colonnes dans la ligne d'indice 0
        Adding action buttons for columns managing in row 0
        """
        # Pour chaque colonne les boutons 'Tout sélectionner' et 'Inverser la Sélection' sont ajoutés
        # Each column (apart from 0(buttons) and 1(layers)) has buttons 'Select all' and 'Invert selection'
        from functools import partial
        for col, theme in enumerate(themes, start=2):
            header_widget = QtWidgets.QWidget()
            header_layout = QtWidgets.QVBoxLayout(header_widget)
            header_layout.setContentsMargins(0, 0, 0, 0)
            header_layout.setSpacing(2)

            btn_select_all = QtWidgets.QPushButton(self.tr("Tout sélectionner"))
            btn_select_all.setToolTip(self.tr("Tout cocher pour le thème « {theme} »").format(theme=theme))
            btn_select_all.clicked.connect(partial(self.select_all_column, col))
            header_layout.addWidget(btn_select_all)

            btn_invert = QtWidgets.QPushButton(self.tr("Inverser la sélection"))
            btn_invert.setToolTip(self.tr("Inverser la sélection pour le thème « {theme} »").format(theme=theme))
            btn_invert.clicked.connect(partial(self.invert_column, col))
            header_layout.addWidget(btn_invert)

            self.tableThemes.setCellWidget(0, col, header_widget)

        """
        Si la combobox existe, ajout de la liste des thèmes a son contenu
        If the combobox exists, the themes list content is added inside
        """
        if hasattr(self, "comboThemes"):
            self.comboThemes.clear()
            self.comboThemes.addItems(themes)

        """
        Ajout du contenu dans les lignes
        Adding the content in rows
        """
        for row, (is_group, item_id, name, parent_id) in enumerate(items):
            # Si c'est un groupe, la case est grisée, et il n'est pas possible d'intéragir dessus (futur développement)
            # If it's a group, the checkbox isn't displayed (to develop)
            if is_group:
                group_item = QtWidgets.QTableWidgetItem()
                group_item.setFlags(QtCore.Qt.NoItemFlags)
                group_item.setBackground(QColor("#f0f0f0"))
                self.tableThemes.setItem(row + 1, 0, group_item)
            else:
            # Si ce n'est pas un groupe, c'est une couche, elles sont ajoutée et des cases à cocher sont mises en place
            # If it's not a group then it's a layer, it is added to the table and checkboxes are set
                btn_widget = QtWidgets.QWidget()
                btn_layout = QtWidgets.QVBoxLayout(btn_widget)
                btn_layout.setContentsMargins(0, 0, 0, 0)
                btn_layout.setSpacing(2)

                btn_select_all_row = QtWidgets.QPushButton(self.tr("Tout sélectionner"))
                btn_select_all_row.setToolTip(self.tr("Tout cocher pour « {name} »").format(name=name))
                btn_select_all_row.clicked.connect(partial(self.select_all_row, row + 1))
                btn_layout.addWidget(btn_select_all_row)

                btn_invert_row = QtWidgets.QPushButton(self.tr("Inverser la sélection"))
                btn_invert_row.setToolTip(self.tr("Inverser la sélection pour « {name} »").format(name=name))
                btn_invert_row.clicked.connect(partial(self.invert_row, row + 1))
                btn_layout.addWidget(btn_invert_row)
                self.tableThemes.setCellWidget(row + 1, 0, btn_widget)

            # Si c'est un groupe le nom est en gras
            # If it's a group in bold
            label = QtWidgets.QLabel(name)
            if is_group:
                font = label.font()
                font.setBold(True)
                label.setFont(font)
            self.tableThemes.setCellWidget(row + 1, 1, label)

            """
            Ajoute une case à cocher pour chaque thème (couche/thème ou groupe/thème)
            Adding a checkbox for each theme (layer/theme or theme/layer)
            """
            for col, theme in enumerate(themes, start=2):
                # Si c'est un groupe, la cellule est grisée et désactivée
                # If it's a group he box is displayed in grey and disabled
                if is_group:
                    group_item = QtWidgets.QTableWidgetItem()
                    group_item.setFlags(QtCore.Qt.NoItemFlags)
                    group_item.setBackground(QColor("#f0f0f0"))
                    self.tableThemes.setItem(row + 1, col, group_item)
                # Si c'est une couche, ajout d'une case à cocher pour chaque thème
                # If it's a layer, adds a checkbox for each theme
                else:
                    layers_in_theme = theme_collection.mapThemeVisibleLayerIds(theme)
                    checked = item_id in layers_in_theme # La case est déjà cochée si la couche est dans le thème / The checkbox is already checked if the layer is in the theme 
                    chk = QtWidgets.QCheckBox()
                    chk.setChecked(checked)
                    chk.stateChanged.connect(partial(self.on_box_changed, item_id, theme))
                    widget = QtWidgets.QWidget()
                    layout = QtWidgets.QHBoxLayout(widget)
                    layout.addWidget(chk)
                    layout.setAlignment(chk, QtCore.Qt.AlignCenter) # Case à cocher au centre de la cellule / Checkbox centered in the cell 
                    layout.setContentsMargins(0, 0, 0, 0)
                    widget.setLayout(layout)
                    self.tableThemes.setCellWidget(row + 1, col, widget)
        
        """
        Redimensionne les lignes et les colonnes en fonction du contenu
        Resizes the rows and columns to fit to content
        """
        self.tableThemes.resizeColumnsToContents()
        self.tableThemes.resizeRowsToContents()

    """
    Méthode gérant le cochage décochage des cases dans le tableau. Elle met à jour le thème modifié et réapplique le thème courant pour constater les modifications si elles ont été effectuées sur celui-ci
    This method manages the checked or unchecked state of the cells int the table. It updates the modified theme and reload the theme in QGIS interface to display the modifications if they has been done on this one
    """
    def on_box_changed(self, item_id, theme, state):
        theme_collection = QgsProject.instance().mapThemeCollection()
        # Récupère l'état des thèmes depuis la collection des thèmes
        # Collects themes states from QGIS theme collection
        theme_state = theme_collection.mapThemeState(theme)
        root = QgsProject.instance().layerTreeRoot()
        # Récupère la couche depuis son identifiant dans QGIS (c'est ce que je ne peux pas faire avec les groupes qui apparemment n'ont pas d'ID)
        # Collects the layer from his QGIS project's ID (that is what I can't do with groups because it seems that they don't have IDs)
        layer = QgsProject.instance().mapLayer(item_id)
        if layer:
            if state == QtCore.Qt.Checked: # Ajoute la couche comme étant visible dans le thème / If it's checked, the layer is added to the theme
                theme_state.removeLayerRecord(layer)
                lr = QgsMapThemeCollection.MapThemeLayerRecord(layer)
                lr.isVisible = True
                theme_state.addLayerRecord(lr)
            else: # Sinon, retire la couche du thème / Otherwise, the layer is deletd from the theme
                theme_state.removeLayerRecord(layer)
            # Met à jour la collection des themes en fonction des nouveaux paramètres
            # Updates the theme collection with user's choice
            theme_collection.update(theme, theme_state)
            # Conserve le dernier thème modifié
            # Keeps the last modified theme in mind
            self.last_modified_theme = theme

        if hasattr(self, "comboThemes"): # Le thème dans la combobox des thèmes est rechargé / The theme chosen in the comboxbox is reloaded
            current_theme = self.comboThemes.currentText()
            if current_theme:
                theme_collection.applyTheme(current_theme, root, self.iface.layerTreeView().layerTreeModel())

    """
    Méthode réagissant au bouton "Appliquer le thème sélectionné". Elle applique le thème sélectionné.
    This methods applies the theme selected in the combobox when the user clicks on "Apply selected theme".
    """
    def apply_selected_theme(self):
        if hasattr(self, "comboThemes"):
            selected_theme = self.comboThemes.currentText()
        else:
            selected_theme = None
        if selected_theme:
            theme_collection = QgsProject.instance().mapThemeCollection()
            root = QgsProject.instance().layerTreeRoot()
            layerTreeModel = self.iface.layerTreeView().layerTreeModel()
            theme_collection.applyTheme(selected_theme, root, layerTreeModel)

    """
    Remplit le tableau de gestion des thèmes. Une ligne = Un thème, associé à ses boutons de sélection et de modification
    Fills the managing theme table. One row = One theme, associated to its selection and modification buttons
    """
    def fill_theme_filter_table(self):
        # Remplit la table de gestion des thèmes (affichage dans la table de gestion des couches, suppression, renommage), une ligne = un thème
        theme_collection = QgsProject.instance().mapThemeCollection()
        themes = list(theme_collection.mapThemes())
        self.table_FilterAndSuppressThemes.blockSignals(True)
        self.table_FilterAndSuppressThemes.setRowCount(len(themes) + 1)
        # Définit et nomme les quatre colonnes
        # Initialise and name the four columns
        self.table_FilterAndSuppressThemes.setColumnCount(4)
        self.table_FilterAndSuppressThemes.setHorizontalHeaderLabels([self.tr("Afficher"), self.tr("Thème"), self.tr("Supprimer"), self.tr("Modifier le nom")])

        header_widget = QtWidgets.QWidget()
        header_layout = QtWidgets.QVBoxLayout(header_widget)
        header_layout.setContentsMargins(0, 0, 0, 0)
        header_layout.setSpacing(2)

        btn_select_all = QtWidgets.QPushButton(self.tr("Tout sélectionner"))
        btn_select_all.setToolTip(self.tr("Tout cocher/décocher pour tous les thèmes"))
        btn_select_all.clicked.connect(self.select_all_themes_column)
        header_layout.addWidget(btn_select_all)

        btn_invert = QtWidgets.QPushButton(self.tr("Inverser la sélection"))
        btn_invert.setToolTip(self.tr("Inverser la sélection pour tous les thèmes"))
        btn_invert.clicked.connect(self.invert_themes_column)
        header_layout.addWidget(btn_invert)

        self.table_FilterAndSuppressThemes.setCellWidget(0, 0, header_widget)
        # Rend les autres cellules de la première ligne non éditables et grises
        # The other cells from the first line aren't editable and displayed in grey
        for col in range(1, 4):
            item = QtWidgets.QTableWidgetItem("")
            item.setFlags(QtCore.Qt.ItemIsEnabled)
            self.table_FilterAndSuppressThemes.setItem(0, col, item)

        for row, theme in enumerate(themes):
            # Colonne 0 : Base à cocher pour afficher/masquer le thème dans la table de gestion des couches par thème
            # Column 0 : Display or hide a theme in the other table
            chk_item = QtWidgets.QTableWidgetItem()
            chk_item.setFlags(QtCore.Qt.ItemIsUserCheckable | QtCore.Qt.ItemIsEnabled)
            chk_item.setCheckState(QtCore.Qt.Checked)
            self.table_FilterAndSuppressThemes.setItem(row +1, 0, chk_item)
            # Colonne 1 : Nom du thème
            # Column 1 : Theme's name
            name_item = QtWidgets.QTableWidgetItem(theme)
            name_item.setFlags(QtCore.Qt.ItemIsEnabled)
            self.table_FilterAndSuppressThemes.setItem(row+1, 1, name_item)
            # Colonne 2 : Bouton de suppression du thème
            # Column 2 : Delete theme button
            btn = QtWidgets.QPushButton(self.tr("Supprimer"))
            btn.clicked.connect(lambda checked, t=theme: self.delete_theme(t))
            self.table_FilterAndSuppressThemes.setCellWidget(row+1, 2, btn)
            # Colonne 3 : Bouton pour renommer le thème
            # Column 3 : Rename theme button
            btn_rename = QtWidgets.QPushButton(self.tr("Modifier le nom du thème"))
            btn_rename.setToolTip(self.tr("Renommer ce thème"))
            btn_rename.clicked.connect(lambda checked, t=theme, r=row+1: self.rename_theme_from_table(t, r))
            self.table_FilterAndSuppressThemes.setCellWidget(row+ 1, 3, btn_rename)

        # Redimensionne les tailles des colonnes (pas des lignes sinon les boutons sont trop gros)
        # Resizes the columns (not rows, because the buttons were displayed too large)
        self.table_FilterAndSuppressThemes.resizeColumnsToContents()
        self.table_FilterAndSuppressThemes.blockSignals(False)

    """
    Permet à l'utilisateur de renommer un thème grâce à une boîte de dialogue. Vérifie les doublons ou les valeurs nulles et applique le changement. Affiche un message de confirmation et met à jour les tableaux.
    Allows the user to rename a theme thanks to a dialog box. Checks duplicated values or NULL ones and then rename the theme. Prints a confirmation message and reload the tables.
    """
    def rename_theme_from_table(self, old_name, row):
        theme_collection = QgsProject.instance().mapThemeCollection()
        # Demande un nouveau nom à l'utilisateur via une boîte de dialogue
        # Asks the user to enter a new theme name (the former one is displayed in the question)
        new_name, ok = QtWidgets.QInputDialog.getText(self, self.tr("Renommer le thème"), self.tr("Entrer un nouveau nom pour le thème « {old_name} » :").format(old_name=old_name), QtWidgets.QLineEdit.Normal, old_name)
        # Si l'utilisateur quitte la fenêtre ou entre une valeur nulle la fonction est fermée (Aucune action réalisée)
        # If the user leaves the dialog box or sets a NULL value, the fonction is stopped (No action)
        if not ok or not new_name:
            return 
        new_name = new_name.strip()
        # Vérification du nom (vide, existant, doublon) :
        # New name verifications (null, existing or duplicate value) :
        if not new_name: # NULL
            QtWidgets.QMessageBox.warning(self, self.tr("Nom manquant"), self.tr("Veuillez saisir un nom de thème non vide."))
            return
        if new_name == old_name: # Nom identique à l'ancien / Same name as the older one
            QtWidgets.QMessageBox.information(self, self.tr("Renommer le thème"), self.tr("Le nom n'a pas changé."))
            return
        if new_name in theme_collection.mapThemes(): # Identique à un autre thème / Same as another theme
            QtWidgets.QMessageBox.warning(self, self.tr("Thème existant"), self.tr("Ce nom de thème existe déjà."))
            return
        # Si tout est correct, applique le nouveau nom
        # If the new name has passed every tests, it is applyed
        theme_collection.renameMapTheme(old_name, new_name)
        self.fill_theme_filter_table()
        self.fill_theme_table()
        QtWidgets.QMessageBox.information(self, self.tr("Renommé"), self.tr("Le thème « {old_name} » a été renommé en « {new_name} ».").format(old_name=old_name, new_name=new_name))

    """
    Permet à l'utilisateur de créer un thème grâce à une boîte de dialogue. Vérifie les doublons ou les valeurs nulles et applique le changement. Affiches un message de confirmation et met à jour les tableaux.
    Allows the user to create a theme thanks to a dialog box. Checks duplicated values or NULL ones and then rename the theme. Prints a confirmation message and reload the tables. 
    """
    def create_new_theme(self):
        # Demande un nouveau nom à l'utilisateur via une boîte de dialogue
        # Asks the user to enter a new theme name (the former one is displayed in the question)
        name, ok = QtWidgets.QInputDialog.getText(self, self.tr("Création d'un thème"), self.tr("Veuillez saisir le nom du nouveau thème").strip())
        # Si l'utilisateur quitte la fenêtre ou entre une valeur nulle la fonction est fermée (Aucune action réalisée)
        # If the user leaves the dialog box or sets a NULL value, the fonction is stopped (No action)
        if not ok or not name.strip():
            return
        name = name.strip()
        theme_collection = QgsProject.instance().mapThemeCollection()
        if name in theme_collection.mapThemes(): # Identique à un autre thème / Same as another theme
            QtWidgets.QMessageBox.warning(self, self.tr("Thème existant"), self.tr("Ce nom de thème existe déjà."))
            return
        # Le thème est créé à partir de l'état actuel de l'arbre des couches
        # The new theme is created from the current state of the layer tree
        root = QgsProject.instance().layerTreeRoot()
        layerTreeModel = self.iface.layerTreeView().layerTreeModel()
        theme_state = QgsMapThemeCollection.createThemeFromCurrentState(root, layerTreeModel)
        theme_collection.insert(name, theme_state)
        # Recharge les tableaux pour afficher le nouveau thème
        # Reloads the tables to display the new theme
        self.fill_theme_filter_table()
        self.fill_theme_table()

    """
    Supprimes un theme du projet
    Removes a theme from the project
    """
    def delete_theme(self, theme_name):
        theme_collection = QgsProject.instance().mapThemeCollection()
        # Si le thème n'est pas présent dans le projet la fonction est fermée
        # If the theme isn't in the project the functon is cancelled
        if theme_name not in theme_collection.mapThemes():
            return
        # Boîte de dialogue
        # Dialog box
        reply = QtWidgets.QMessageBox.question(
            self, self.tr("Suppression de thème"),
            self.tr("Êtes-vous sûr de vouloir supprimer le thème : « {theme_name} » ?").format(theme_name=theme_name),
            QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.No)
        # Si l'utilisateur confirme son choix, le theme est supprimé
        # If the user confirms, the theme is removed
        if reply == QtWidgets.QMessageBox.Yes:
            theme_collection.removeMapTheme(theme_name)
            # Recharge les tableaux pour afficher le nouveau thème
            # Reloads the tables to display the new theme
            self.fill_theme_filter_table()
            self.fill_theme_table()

    """
    Gestion de l'affichage dans la table de gestion des thèmes : Filtrage des thèmes à afficher, modification d'un thème
    Manages the display in theme management table : filtering the themes to display, modifying a theme
    """
    def on_filter_changed(self, item):
        # Si la modification affecte la colonne 0 (filtre d'affichage des thèmes)
        # If the edit concerns column 0 (theme display filter)
        if item.column() == 0:
            # Le tableau est rechargé pour appliquer le filtre
            # The table is reloaded to apply the user's choices
            self.fill_theme_table()
        # Si la modificaiton affecte la colonne 1 (nom du thème)
        # If the edit concerns column 1 (theme name)
        elif item.column() == 1:
            row = item.row()
            old_name = self._get_theme_name_by_row(row)
            new_name = item.text().strip()
            # Si le nom ne change pas, on ignore la modification
            # If the name does not change, the modification is ignored
            if not new_name or new_name == old_name:
                return
            theme_collection = QgsProject.instance().mapThemeCollection()
            # Si le nom existe déjà, affiche un avertissement
            # If the name already exists, displayed a warn
            if new_name in theme_collection.mapThemes():
                QtWidgets.QMessageBox.warning(self, self.tr("Thème existant"), self.tr("Ce nom de thème existe déjà."))
                item.setText(old_name)
                return
            theme_collection.renameMapTheme(old_name, new_name)
            # Recharge les tableaux pour afficher le nouveau thème
            # Reloads the tables to display the new theme
            self.fill_theme_filter_table()
            self.fill_theme_table()

    """
    Récupère le nom du thème associé au numéro de ligne du tableau
    Gets the name of the theme and its row position in the table
    """
    def _get_theme_name_by_row(self, row):
        item = self.table_FilterAndSuppressThemes.item(row, 1)
        return item.text() if item else ""

    """
    Coche toutes les cases à cocher de la ligne dans le tableau de gestion des couches par thèmes (Bouton 'Tout sélectionner')
    Checks all checkboxes of the row in the table for managing layers by themes ('Select All' button)
    """
    def select_all_row(self, row):
        # Trouves l'emplacement de la case à cocher et si elle n'est pas cochée, elle le devient
        # Find the location of the checkboxes and if it is not checked, it becomes checked
        for col in range(2, self.tableThemes.columnCount()):
            cell_widget = self.tableThemes.cellWidget(row, col)
            if cell_widget:
                chk = cell_widget.findChild(QtWidgets.QCheckBox)
                if chk and not chk.isChecked():
                    chk.setChecked(True)

    """
    Inverse l'état des cases à cocher pour une ligne dans le tableau tableThemes (Bouton 'Inverser la sélection')
    Reverses the state of the checkboxes for a row in the tableThemes table ('Invert selection' button)
    """
    def invert_row(self, row):
        for col in range(2, self.tableThemes.columnCount()):
            cell_widget = self.tableThemes.cellWidget(row, col)
            if cell_widget:
                chk = cell_widget.findChild(QtWidgets.QCheckBox)
                if chk:
                    chk.setChecked(not chk.isChecked())

    """
    # Trouves l'emplacement de la case à cocher et si elle n'est pas cochée, elle le devient pour les colonnes de tableThemes
    # Find the location of the checkboxes and if it is not checked, it becomes checked for the colums in tableThemes
    """
    def select_all_column(self, col):
        for row in range(1, self.tableThemes.rowCount()):
            cell_widget = self.tableThemes.cellWidget(row, col)
            if cell_widget:
                chk = cell_widget.findChild(QtWidgets.QCheckBox)
                if chk and not chk.isChecked():
                    chk.setChecked(True)

    """
    Inverse l'état des cases à cocher pour une colonne dans le tableau tableThemes (Bouton 'Inverser la sélection')
    Toggles the state of the checkboxes for a column in the tableThemes table ('Invert selection' button)
    """
    def invert_column(self, col):
        for row in range(1, self.tableThemes.rowCount()):
            cell_widget = self.tableThemes.cellWidget(row, col)
            if cell_widget:
                chk = cell_widget.findChild(QtWidgets.QCheckBox)
                if chk:
                    chk.setChecked(not chk.isChecked())

    """
    Coche toutes les cases à cocher de la colonne "Afficher" (col 0) dans la table de gestion des thèmes (table_FilterAndSuppressThemes) : Bouton 'Tout sélectionner'
    Checks all the checkboxes in the column 0 in the theme management table (table_FilterAndSuppressThemes) :  'Select All' button
    """
    def select_all_themes_column(self):
        for row in range(1, self.table_FilterAndSuppressThemes.rowCount()):
            item = self.table_FilterAndSuppressThemes.item(row, 0)
            if item and item.flags() & QtCore.Qt.ItemIsUserCheckable:
                item.setCheckState(QtCore.Qt.Checked)

    """
    Inverse l'état des cases à cocher pour une colonne dans le tableau table_FilterAndSuppressThemes (Bouton 'Inverser la sélection')
    Toggles the state of the checkboxes for a column in the table_FilterAndSuppressThemes table ('Invert selection' button)
    """
    def invert_themes_column(self):
        for row in range(1, self.table_FilterAndSuppressThemes.rowCount()):
            item = self.table_FilterAndSuppressThemes.item(row, 0)
            if item and item.flags() & QtCore.Qt.ItemIsUserCheckable:
                item.setCheckState(QtCore.Qt.Unchecked if item.checkState() == QtCore.Qt.Checked else QtCore.Qt.Checked)

"""
LA FIN
THE END
"""