# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Thematics
                                 A QGIS plugin
 open projects and layer groups from list
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-01-19
        git sha              : $Format:%H$
        copyright            : (C) 2020 by Zoltan Siki
        email                : siki1958@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os
import platform
import configparser
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox

from qgis.core import QgsApplication, QgsProject, QgsVectorLayer, QgsRasterLayer, QgsLayerTreeLayer
# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .thematics_dockwidget import ThematicsDockWidget

DEFAULT_CONFIG = "L:/xxii/thematics.cfg"

class Thematics:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        try:
            locale = QSettings().value('locale/userLocale')[0:2]
        except Exception:
            locale = "hu"
        locale_path = os.path.join(self.plugin_dir, 'i18n',
            '{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr('&thematics')

        self.pluginIsActive = False
        self.dockwidget = None
        # load and process configuration
        self.side = "left"
        if os.path.exists(DEFAULT_CONFIG):
            self.projects, self.layers, self.side = self.config(DEFAULT_CONFIG)
        else:
            self.projects, self.layers, self.side = self.config(os.path.join(self.plugin_dir, "default.cfg"))
        # open the panel
        #self.run()

    def config(self, path):
        """ load and parse config file """
        parser = configparser.ConfigParser()
        if not os.path.exists(path):
            QMessageBox.warning(None, self.tr("Missing file"),
                self.tr("Config file not found: {}").format(path))
            return ({}, {}, "left")
        try:
            parser.read(path)
        except Exception:
            QMessageBox.warning(None, self.tr("Error in file"),
                self.tr("Config file is not valid: {}").format(path))
            return ({}, {}, "left")
        projects = {}
        layers = {}
        base_dir = ""
        side = ""
        for section in parser.sections():
            if section == "include":
                return self.config(parser[section]['path'])
            if section == "base":
                base_dir = parser[section].get('dir', self.plugin_dir)
                side = parser[section].get('side', "left")
            elif section == "projects":
                for key in parser[section]:
                    # join is wrong on windows \\ inserted while provider returns path with /
                    projects[key] = base_dir+'/'+parser[section][key]
            elif section.startswith('layer_group'):
                layers[parser[section]['name']] = []
                for key in parser[section]:
                    if key.startswith('layer'):
                        if parser[section][key].startswith("url=") or \
                           parser[section][key].startswith("type=xyz"):
                            # wms layer
                            layers[parser[section]['name']].append(parser[section][key])
                        else:
                            # local raster/vector layer
                            # join is wrong on windows \\ inserted while provider returns path with /
                            layers[parser[section]['name']].append(base_dir+'/'+parser[section][key])
        return projects, layers, side

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('Thematics', message)


    def add_action(self, icon_path, text, callback, enabled_flag=True,
        add_to_menu=True, add_to_toolbar=True, status_tip=None,
        whats_this=None, parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/thematics/icon.png'
        self.add_action(icon_path, text=self.tr('Thematics'),
            callback=self.run, parent=self.iface.mainWindow())

    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        #self.dockwidget = None
        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr('&thematics'),
                action)
            self.iface.removeToolBarIcon(action)
            try:
                self.dockwidget.close()
            except Exception:
                pass
            self.dockwidget = None

    #--------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""
        if not self.pluginIsActive:
            self.pluginIsActive = True

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget is None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = ThematicsDockWidget(self)
                # fill list with project names
                for project in self.projects:
                    self.dockwidget.list_projects.addItem(project)
                # fill list with layer_group names
                for layer in self.layers:
                    self.dockwidget.list_layers.addItem(layer)

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            if self.side == "right":
                self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            else:
                self.iface.addDockWidget(Qt.LeftDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

    def new_config(self, f):
        """ load a new config file """
        if len(f) > 0:
            self.dockwidget.list_projects.clear()
            self.dockwidget.list_layers.clear()
            self.projects = {}
            self.layers = {}
            self.projects, self.layers, self.side = self.config(f)
            # fill list with project names
            for project in self.projects:
                self.dockwidget.list_projects.addItem(project)
            # fill list with layer_group names
            for layer in self.layers:
                self.dockwidget.list_layers.addItem(layer)

    def open_project(self, name, newwin=False):
        """ open the selected project """
        # get canvas extent
        project = QgsProject.instance()
        pname = project.fileName()
        if len(pname) > 0:
            cext = self.iface.mapCanvas().extent()
            sext = "--extent {},{},{},{}".format(cext.xMinimum(),
                cext.yMinimum(), cext.xMaximum(), cext.yMaximum())
        else:
            #sext = ""
            sext = []
        if newwin:
            qgis_exe = QgsApplication.instance().applicationFilePath()
            if platform.system() == "Windows":
                code = os.system('{} --project {} {}'.format(qgis_exe,
                    self.projects[name], sext))
            else: # add execute in background
                code = os.system('{} --project {} {} &'.format(qgis_exe,
                    self.projects[name], sext))
            if code != 0:
                QMessageBox.warning(None, self.tr("Project"),
                    self.tr("Cannot start qgis: {}").format(code))
        else:
            if not project.read(self.projects[name]):
                QMessageBox.warning(None, self.tr("Project"),
                    self.tr("Cannot open project: {}").format(
                        self.projects[name]))
            else:
                # set canvas extent
                if len(pname) > 0:
                    self.iface.mapCanvas().setExtent(cext, True)
                    self.iface.mapCanvas().refresh()

    def wms_name(self, l):
        """ get wms layer name """
        if l.startswith("url="):
            for tag in l.split("&"):
                if tag.startswith("layers="):
                    return tag.split("=")[1]
        else:
            for tag in l.split("&"):
                if tag.startswith("url="):
                    return tag.split("=")[1]
        return None

    def reorder(self, r):
        """ move vector layer to resonable position in layer order """
        gtyp = r.geometryType() # point/line/polygon
        rg = self.iface.layerTreeCanvasBridge().rootGroup()
        order = rg.customLayerOrder()
        for i in range(len(order)):
            if isinstance(order[i], QgsRasterLayer) or order[i].geometryType() > gtyp:
                order.insert(i, order.pop())    # move from end
                rg.setCustomLayerOrder(order)
                return

    def open_layer_group(self, name):
        """ add layer group to canvas """
        project = QgsProject.instance() # get project instance
        # enable custom layer order
        rg = self.iface.layerTreeCanvasBridge().rootGroup()
        if not rg.hasCustomLayerOrder():
            rg.setHasCustomLayerOrder(True)
        # check if thematics group in tree
        root = project.layerTreeRoot()
        themGroup = root.findGroup(self.tr("Thematics"))
        if themGroup is None:
            # create the group at the end of tree
            themGroup = root.insertGroup(-1, self.tr("Thematics"))
        if name in self.layers:
            for l in self.layers[name]:
                valid = False
                typ = ""
                if l.startswith("url=") or l.startswith("type=xyz"):
                    # wms or XYZ tile layer
                    print(l)
                    nam = self.wms_name(l)
                    print(nam)
                    if nam:
                        lobjs = project.mapLayersByName(nam)
                        if not lobjs:   # not in project yet
                            # open wms
                            r = QgsRasterLayer(l, nam, "wms")
                            valid = r.isValid()
                            typ = "wms"
                else:
                    # local vector/raster layer
                    ext = os.path.splitext(os.path.split(l)[1])[1]
                    nam = os.path.splitext(os.path.split(l)[1])[0]
                    lobjs = project.mapLayersByName(nam)
                    if lobjs:   # layer in project yet
                        continue
                    if ext in ('.tif', '.jpg', '.png', '.vrt'):
                        # open raster
                        r = QgsRasterLayer(l, nam)
                        valid = r.isValid()
                        typ = "raster"
                    else:
                        # open vector
                        r = QgsVectorLayer(l, nam, 'ogr')
                        valid = r.isValid()
                        typ = "vector"
                        # check qml & load
                        qname = os.path.splitext(l)[0] + '.qml'
                        if os.path.exists(qname):
                            r.loadNamedStyle(qname)
                if valid:
                    project.addMapLayer(r, False)
                    themGroup.insertChildNode(-1, QgsLayerTreeLayer(r))
                    if typ == "vector":
                        self.reorder(r)   # change layer position in layer order
                else:
                    QMessageBox.warning(None, self.tr("Layer"),
                        self.tr("Cannot open layer: {}").format(nam))

    def remove_layer_group(self, name):
        """ remove layer group from canvas """
        project = QgsProject.instance()
        if name in self.layers:
            for l in self.layers[name]:
                if l.startswith("url="):
                    nam = self.wms_name(l)
                else:
                    nam = os.path.splitext(os.path.split(l)[1])[0]
                lobjs = project.mapLayersByName(nam)
                if lobjs:
                    project.removeMapLayers([lobjs[0].id()])
        self.iface.mapCanvas().refresh()
