"""
Settings Dock Widget for Terrascope Plugin

Provides authentication UI (login/logout), general settings (default collection,
cloud cover, STAC URL), and advanced settings (token refresh, debug mode).
"""

import os

from qgis.PyQt.QtCore import Qt, QSettings, QThread, pyqtSignal
from qgis.PyQt.QtWidgets import (
    QDockWidget,
    QWidget,
    QVBoxLayout,
    QHBoxLayout,
    QFormLayout,
    QTabWidget,
    QLabel,
    QLineEdit,
    QPushButton,
    QCheckBox,
    QSpinBox,
    QGroupBox,
    QMessageBox,
)


class LoginWorker(QThread):
    """Worker thread for non-blocking authentication setup."""

    finished = pyqtSignal(bool, str)

    def __init__(self, auth, username, password):
        """Initialize the login worker.

        Args:
            auth: TerrascopeAuth instance.
            username: Terrascope username.
            password: Terrascope password.
        """
        super().__init__()
        self.auth = auth
        self.username = username
        self.password = password

    def run(self):
        """Set up basic authentication."""
        try:
            self.auth.login(self.username, self.password)
            self.finished.emit(True, "Authentication configured successfully")
        except Exception as e:
            self.finished.emit(False, str(e))


class SettingsDockWidget(QDockWidget):
    """Settings dock widget with authentication and configuration tabs."""

    def __init__(self, iface, get_auth, parent=None):
        """Initialize the settings dock.

        Args:
            iface: QGIS interface instance.
            get_auth: Callable that returns the shared TerrascopeAuth instance.
            parent: Parent widget.
        """
        super().__init__("Terrascope Settings", parent)
        self.iface = iface
        self._get_auth = get_auth
        self._workers = []

        self.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)

        self._settings = QSettings()
        self._setup_ui()
        self._load_settings()
        self._update_auth_status()

    def _start_worker(self, worker):
        """Start a QThread worker and track it for cleanup.

        Args:
            worker: QThread worker to start.
        """

        def _cleanup():
            if worker in self._workers:
                self._workers.remove(worker)

        worker.finished.connect(_cleanup)
        self._workers.append(worker)
        worker.start()

    def _stop_all_workers(self):
        """Wait for all active workers to finish."""
        for worker in list(self._workers):
            worker.wait(5000)
        self._workers.clear()

    def closeEvent(self, event):
        """Handle dock close event, ensuring workers are stopped.

        Args:
            event: The close event.
        """
        self._stop_all_workers()
        super().closeEvent(event)

    def _setup_ui(self):
        """Set up the dock widget UI."""
        container = QWidget()
        layout = QVBoxLayout(container)

        tabs = QTabWidget()

        # Authentication tab
        auth_tab = QWidget()
        auth_layout = QVBoxLayout(auth_tab)

        auth_group = QGroupBox("Authentication")
        auth_form = QFormLayout(auth_group)

        self.username_edit = QLineEdit()
        self.username_edit.setPlaceholderText("Terrascope username")
        auth_form.addRow("Username:", self.username_edit)

        self.password_edit = QLineEdit()
        self.password_edit.setEchoMode(QLineEdit.Password)
        self.password_edit.setPlaceholderText("Terrascope password")
        auth_form.addRow("Password:", self.password_edit)

        self.save_credentials_cb = QCheckBox("Save credentials")
        auth_form.addRow(self.save_credentials_cb)

        self.auto_login_cb = QCheckBox("Auto-login on startup")
        auth_form.addRow(self.auto_login_cb)

        auth_layout.addWidget(auth_group)

        # Login/Logout buttons
        btn_layout = QHBoxLayout()
        self.login_btn = QPushButton("Login")
        self.login_btn.clicked.connect(self._on_login)
        btn_layout.addWidget(self.login_btn)

        self.logout_btn = QPushButton("Logout")
        self.logout_btn.clicked.connect(self._on_logout)
        self.logout_btn.setEnabled(False)
        btn_layout.addWidget(self.logout_btn)
        auth_layout.addLayout(btn_layout)

        # Status
        self.auth_status_label = QLabel("Not authenticated")
        self.auth_status_label.setStyleSheet("color: gray; font-weight: bold;")
        self.auth_status_label.setAlignment(Qt.AlignCenter)
        auth_layout.addWidget(self.auth_status_label)

        auth_layout.addStretch()
        tabs.addTab(auth_tab, "Authentication")

        # General tab
        general_tab = QWidget()
        general_layout = QVBoxLayout(general_tab)
        general_form = QFormLayout()

        self.default_collection_edit = QLineEdit()
        self.default_collection_edit.setPlaceholderText("terrascope-s2-ndvi-v2")
        general_form.addRow("Default collection:", self.default_collection_edit)

        cloud_row = QHBoxLayout()
        self.default_cloud_cover_cb = QCheckBox()
        self.default_cloud_cover_cb.setChecked(False)
        self.default_cloud_cover_cb.setToolTip(
            "Uncheck to disable cloud cover filtering\n"
            "(required for collections without cloud cover metadata)"
        )
        cloud_row.addWidget(self.default_cloud_cover_cb)
        self.default_cloud_cover_spin = QSpinBox()
        self.default_cloud_cover_spin.setRange(0, 100)
        self.default_cloud_cover_spin.setValue(30)
        self.default_cloud_cover_spin.setSuffix("%")
        self.default_cloud_cover_spin.setEnabled(False)
        cloud_row.addWidget(self.default_cloud_cover_spin)
        self.default_cloud_cover_cb.toggled.connect(
            self.default_cloud_cover_spin.setEnabled
        )
        general_form.addRow("Default max cloud cover:", cloud_row)

        self.default_max_results_spin = QSpinBox()
        self.default_max_results_spin.setRange(1, 500)
        self.default_max_results_spin.setValue(50)
        general_form.addRow("Default max results:", self.default_max_results_spin)

        self.stac_url_edit = QLineEdit()
        self.stac_url_edit.setText("https://stac.terrascope.be")
        general_form.addRow("STAC URL:", self.stac_url_edit)

        general_layout.addLayout(general_form)

        save_general_btn = QPushButton("Save Settings")
        save_general_btn.clicked.connect(self._save_settings)
        general_layout.addWidget(save_general_btn)

        general_layout.addStretch()
        tabs.addTab(general_tab, "General")

        # Advanced tab
        advanced_tab = QWidget()
        advanced_layout = QVBoxLayout(advanced_tab)
        advanced_form = QFormLayout()

        self.auth_method_label = QLabel("Basic Authentication (GDAL_HTTP_AUTH)")
        self.auth_method_label.setStyleSheet("font-weight: bold;")
        advanced_form.addRow("Auth method:", self.auth_method_label)

        self.env_username_label = QLabel("Not configured")
        advanced_form.addRow("GDAL_HTTP_USERPWD:", self.env_username_label)

        self.debug_cb = QCheckBox("Enable debug logging")
        advanced_form.addRow(self.debug_cb)

        advanced_layout.addLayout(advanced_form)

        # Dependencies section
        dep_group = QGroupBox("Dependencies")
        dep_layout = QVBoxLayout(dep_group)

        self.dep_status_label = QLabel("Not checked")
        self.dep_status_label.setWordWrap(True)
        dep_layout.addWidget(self.dep_status_label)

        self.check_deps_btn = QPushButton("Check / Install Dependencies...")
        self.check_deps_btn.clicked.connect(self._on_check_dependencies)
        dep_layout.addWidget(self.check_deps_btn)

        from html import escape

        from ..venv_manager import VENV_DIR

        self.venv_path_label = QLabel(f"<small>{escape(VENV_DIR)}</small>")
        self.venv_path_label.setStyleSheet("color: gray;")
        dep_layout.addWidget(self.venv_path_label)

        advanced_layout.addWidget(dep_group)

        advanced_layout.addStretch()
        tabs.addTab(advanced_tab, "Advanced")

        self._update_dep_status()

        layout.addWidget(tabs)
        self.setWidget(container)

    def _on_login(self):
        """Handle login button click.

        Uses the UI fields first, falling back to TERRASCOPE_USERNAME
        and TERRASCOPE_PASSWORD environment variables if fields are empty.
        """
        username = self.username_edit.text().strip()
        password = self.password_edit.text().strip()

        if not username:
            username = os.environ.get("TERRASCOPE_USERNAME", "")
        if not password:
            password = os.environ.get("TERRASCOPE_PASSWORD", "")

        if not username or not password:
            QMessageBox.warning(
                self,
                "Terrascope",
                "Please enter both username and password, or set\n"
                "TERRASCOPE_USERNAME and TERRASCOPE_PASSWORD\n"
                "environment variables.",
            )
            return

        self.login_btn.setEnabled(False)
        self.auth_status_label.setText("Logging in...")
        self.auth_status_label.setStyleSheet("color: blue; font-weight: bold;")

        auth = self._get_auth()
        worker = LoginWorker(auth, username, password)
        worker.finished.connect(self._on_login_finished)
        self._start_worker(worker)

    def _on_login_finished(self, success, message):
        """Handle login result.

        Args:
            success: Whether login succeeded.
            message: Result message.
        """
        self.login_btn.setEnabled(True)

        if success:
            self.auth_status_label.setText("Authenticated")
            self.auth_status_label.setStyleSheet("color: green; font-weight: bold;")
            self.login_btn.setEnabled(False)
            self.logout_btn.setEnabled(True)

            if self.save_credentials_cb.isChecked():
                self._settings.setValue(
                    "Terrascope/username", self.username_edit.text().strip()
                )
                self._settings.setValue(
                    "Terrascope/password", self.password_edit.text().strip()
                )
            else:
                self._settings.remove("Terrascope/username")
                self._settings.remove("Terrascope/password")

            self._settings.setValue(
                "Terrascope/auto_login", self.auto_login_cb.isChecked()
            )
        else:
            self.auth_status_label.setText(f"Login failed: {message}")
            self.auth_status_label.setStyleSheet("color: red; font-weight: bold;")

    def _on_logout(self):
        """Handle logout button click."""
        auth = self._get_auth()
        auth.logout()
        self.auth_status_label.setText("Not authenticated")
        self.auth_status_label.setStyleSheet("color: gray; font-weight: bold;")
        self.login_btn.setEnabled(True)
        self.logout_btn.setEnabled(False)

    def _update_auth_status(self):
        """Update the authentication status display."""
        auth = self._get_auth()
        if auth.is_authenticated():
            self.auth_status_label.setText("Authenticated")
            self.auth_status_label.setStyleSheet("color: green; font-weight: bold;")
            self.login_btn.setEnabled(False)
            self.logout_btn.setEnabled(True)

    def update_env_status(self):
        """Update the environment variable status label."""
        username = os.environ.get("TERRASCOPE_USERNAME", "")
        if username:
            self.env_username_label.setText(f"TERRASCOPE_USERNAME={username}")
            self.env_username_label.setStyleSheet("color: green;")
        else:
            self.env_username_label.setText("Not set (use login form)")
            self.env_username_label.setStyleSheet("color: gray;")

    def _save_settings(self):
        """Save settings to QSettings."""
        self._settings.setValue(
            "Terrascope/default_collection",
            self.default_collection_edit.text().strip(),
        )
        self._settings.setValue(
            "Terrascope/default_cloud_cover",
            (
                self.default_cloud_cover_spin.value()
                if self.default_cloud_cover_cb.isChecked()
                else -1
            ),
        )
        self._settings.setValue(
            "Terrascope/default_max_results",
            self.default_max_results_spin.value(),
        )
        self._settings.setValue(
            "Terrascope/stac_url",
            self.stac_url_edit.text().strip(),
        )
        self._settings.setValue(
            "Terrascope/debug_mode",
            self.debug_cb.isChecked(),
        )
        self._settings.setValue(
            "Terrascope/auto_login",
            self.auto_login_cb.isChecked(),
        )

        QMessageBox.information(self, "Terrascope", "Settings saved.")

    def _load_settings(self):
        """Load settings from QSettings."""
        self.default_collection_edit.setText(
            self._settings.value("Terrascope/default_collection", "")
        )
        default_cc = int(self._settings.value("Terrascope/default_cloud_cover", -1))
        self.default_cloud_cover_cb.setChecked(default_cc >= 0)
        self.default_cloud_cover_spin.setValue(max(default_cc, 30))
        self.default_cloud_cover_spin.setEnabled(default_cc >= 0)
        self.default_max_results_spin.setValue(
            int(self._settings.value("Terrascope/default_max_results", 50))
        )
        stac_url = self._settings.value(
            "Terrascope/stac_url", "https://stac.terrascope.be"
        )
        self.stac_url_edit.setText(stac_url)

        self.debug_cb.setChecked(
            self._settings.value("Terrascope/debug_mode", False, type=bool)
        )
        self.auto_login_cb.setChecked(
            self._settings.value("Terrascope/auto_login", False, type=bool)
        )
        self.save_credentials_cb.setChecked(
            bool(self._settings.value("Terrascope/username", ""))
        )

        # Load saved credentials, then fall back to environment variables
        saved_username = self._settings.value("Terrascope/username", "")
        saved_password = self._settings.value("Terrascope/password", "")

        if not saved_username:
            saved_username = os.environ.get("TERRASCOPE_USERNAME", "")
        if not saved_password:
            saved_password = os.environ.get("TERRASCOPE_PASSWORD", "")

        if saved_username:
            self.username_edit.setText(saved_username)
        if saved_password:
            self.password_edit.setText(saved_password)

    def _on_check_dependencies(self):
        """Open the dependency installation dialog."""
        from .dependency_dialog import DependencyDialog

        dialog = DependencyDialog(self)
        dialog.exec_()
        self._update_dep_status()

    def _update_dep_status(self):
        """Update the dependency status label."""
        from ..venv_manager import get_venv_status

        is_ready, _message, missing_req, missing_opt = get_venv_status()
        all_missing = missing_req + missing_opt

        if not all_missing:
            self.dep_status_label.setText("All dependencies installed")
            self.dep_status_label.setStyleSheet("color: green; font-weight: bold;")
        elif missing_req:
            self.dep_status_label.setText(f"Missing: {', '.join(all_missing)}")
            self.dep_status_label.setStyleSheet("color: red; font-weight: bold;")
        else:
            self.dep_status_label.setText(f"Optional missing: {', '.join(missing_opt)}")
            self.dep_status_label.setStyleSheet("color: orange; font-weight: bold;")
