from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant)
from qgis.core import (
  QgsProcessingParameterExtent,
  QgsProcessingParameterCrs,
  QgsWkbTypes,
  QgsProcessingParameterFeatureSink,
  QgsProcessingContext,
  QgsProcessingFeedback,
  QgsFields,
  QgsField,
  QgsFeature,
  QgsGeometry,
  QgsRectangle

  )
from .algabstract import algabstract

import requests
import concurrent

class listtengungrid(algabstract):
  
  # UIs
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("listtengungrid","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("listtengungrid","CRS for determining point cloud coordinates")
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("listtengungrid","Point-cloud grid")
      }
    }
  }
  
  
  CRS = ""
  
  # initialization of the algorithm
  def initAlgorithm(self, config):    
    self.initUsingCanvas()
    self.initParameters()
  
    
  
  
  def setWebFetchArgs(self, parameters, context, feedback):
    
    self.WEBFETCH_ARGS["REQUEST_ARGS"] = []
    
    for db in self.DBS:
      base_url = db["url"]
      pcode_format = db["pcode_format"]
      dx = db["dx"]
      dy = db["dy"]
      
    
      if "{pcode}" not in base_url:
        raise Exception(self.tr("URL must contain {pcode}"))
      
      x = self.FETCH_AREA.xMinimum()
      while x < self.FETCH_AREA.xMaximum():
        y = self.FETCH_AREA.yMinimum()
        while y < self.FETCH_AREA.yMaximum():
          try:
            meshcode_str = self.cmptMeshCodeFromXY(x, y, pcode_format=pcode_format)
            download_url = base_url.replace("{pcode}", meshcode_str)
            self.WEBFETCH_ARGS["REQUEST_ARGS"].append(
              {
                "method": "GET",
                "url": download_url
              }
            )
            self.WEBFETCH_ARGS["URL_DICT"].update(
              {
                download_url: {
                  "meshcode": meshcode_str, 
                  "pcode_format": pcode_format,
                  "description": db["description"]
                }
              }
            )
          except Exception as e:
            feedback.reportError(self.tr("Failed to get a mesh code. CRS settings may be wrong."))
            raise e
          y += dy
        x += dx
    
    if self.WEBFETCH_ARGS["REQUEST_ARGS"] is not None:
      self.WEBFETCH_ARGS["SET"] = True
    
  
  # execution of the algorithm
  def processAlgorithm(self, parameters, context, feedback):    
    self.FETCH_AREA = self.parameterAsExtent(
      parameters, "FETCH_EXTENT", context, self.parameterAsCrs(parameters, "TARGET_CRS", context)
      )
    
    
    # set the meta information for obtaining SRTM data
    self.setWebFetchArgs(parameters, context, feedback)
    
    # download files using the session info
    self.fetchFileHeaderConcurrently(parameters, context, feedback, parallel=True)
    
    
    output_flds = QgsFields()
    output_flds.append(QgsField("meshcode", QVariant.String))
    output_flds.append(QgsField("description", QVariant.String))
    output_flds.append(QgsField("url", QVariant.String))
    output_flds.append(QgsField("file_size", QVariant.Double))
    (sink, dest_id) = self.parameterAsSink(
      parameters, "OUTPUT", context, output_flds, QgsWkbTypes.Polygon, self.parameterAsCrs(parameters, "TARGET_CRS", context)
    )
    
    for file in self.WEBFETCH_ARGS["DOWNLOADED_FILE"]:
      ft = QgsFeature(output_flds)
      for k, v in file.items():
        if k != "geometry":
          ft[k] = v
      ft.setGeometry(file["geometry"])
      sink.addFeature(ft)
      
    return {"OUTPUT": dest_id}
  
  
  # download a file from the URL
  def fetchFileHeader(self, session, request_args:dict, feedback: QgsProcessingFeedback) -> dict:
    try:
      response = session.request(stream = True, timeout = 12.0, **request_args)
    except Exception as e:
      feedback.reportError(self.tr("Download was failed."))
      raise e
    # if download is successful
    if response.status_code == 200:
      file_size = int(response.headers.get('content-length', 0))
      return {"url": request_args["url"], "file_size": file_size}
    
  
  # download file(s) from the URL (asynchronously)
  def fetchFileHeaderConcurrently(self, parameters: dict, context: QgsProcessingContext, feedback: QgsProcessingFeedback, parallel = True) -> None:
    self.WEBFETCH_ARGS["DOWNLOADED_FILE"] = []
    
    if not self.WEBFETCH_ARGS["SET"]:
      return
    # if login is needed, session is also needed
    if self.WEBFETCH_ARGS["LOGIN"] is None:
      session = requests.Session()
    else:
      session = self.WEBFETCH_ARGS["LOGIN"]["SESSION"]
      if not isinstance(session, requests.sessions.Session):
        feedback.reportError(self.tr("Login session is not given."))
        raise Exception(self.tr("Login session is not given"))
        
    # set download arguments
    download_args = self.WEBFETCH_ARGS["REQUEST_ARGS"]
    mx_wk = None if parallel else 1
    
    # fetch in parallel
    with concurrent.futures.ThreadPoolExecutor(max_workers=mx_wk) as executor:
      for _ in executor.map(
        lambda d_arg : self.fetchFileHeader(
          session, 
          request_args = d_arg, 
          feedback = feedback
          ), 
        download_args
        ):
        if _ is not None:
          url_dict = self.WEBFETCH_ARGS["URL_DICT"][_["url"]]
          self.WEBFETCH_ARGS["DOWNLOADED_FILE"].append(
            {
              "meshcode": url_dict["meshcode"],
              "url": _["url"],
              "file_size": _["file_size"] / 1e6,
              "description": url_dict["description"],
              "geometry": self.cmptGeomFromMeshCode(url_dict["meshcode"], url_dict["pcode_format"])
            }
          )
          feedback.pushInfo(_["url"] + " " + f"{_['file_size']/1e6:.1f}MB")
        if feedback.isCanceled():
          feedback.reportError(self.tr("Fetching was canceled."))
          raise Exception(self.tr("Fetching was canceled."))
    
  
  # compute the geometry from the mesh code of 国土基本図図郭
  def cmptGeomFromMeshCode(self, meshcode, pcode_format = {"level": "500","prefix": ""}) -> QgsGeometry:
    meshcode = meshcode.replace(pcode_format["prefix"], "", 1).upper()
    
    dx1 = 40.0e3
    dy1 = 30.0e3
    nx1 = ord(meshcode[1]) - 69 
    ny1 = 74 - ord(meshcode[0])
    x0 = dx1 * nx1
    y0 = dy1 * ny1
    
    if pcode_format["level"] == "50000":
      return QgsGeometry.fromRect(QgsRectangle(x0, y0, x0 + dx1, y0 + dy1))
    
    dx2 = 4.0e3
    dy2 = 3.0e3
    nx2 = int(meshcode[3])
    ny2 = 9 - int(meshcode[2])
    x0 = x0 + dx2 * nx2
    y0 = y0 + dy2 * ny2
    
    if pcode_format["level"] == "5000":
      return QgsGeometry.fromRect(QgsRectangle(x0, y0, x0 + dx2, y0 + dy2))
    
    
    if pcode_format["level"] == "2500":
      dx3 = 2.0e3
      dy3 = 1.5e3
      nx3 = int(meshcode[4]) % 2 - 1
      ny3 = 1 - int(meshcode[4]) // 2
      x0 = x0 + dx3 * nx3
      y0 = y0 + dy3 * ny3  
      return QgsGeometry.fromRect(QgsRectangle(x0, y0, x0 + dx3, y0 + dy3))
    
    elif pcode_format["level"] == "1000":
      dx4 = 800.0
      dy4 = 600.0
      nx4 = ord(meshcode[5]) - 65
      ny4 = 4 - int(meshcode[4])
      x0 = x0 + dx4 * nx4
      y0 = y0 + dy4 * ny4
      return QgsGeometry.fromRect(QgsRectangle(x0, y0, x0 + dx4, y0 + dy4))
    
    else:
      dx5 = 400.0
      dy5 = 300.0
      nx5 = int(meshcode[5])
      ny5 = 9 - int(meshcode[4])
      x0 = x0 + dx5 * nx5
      y0 = y0 + dy5 * ny5
      return QgsGeometry.fromRect(QgsRectangle(x0, y0, x0 + dx5, y0 + dy5))
  
  # Post processing; append layers
  def postProcessAlgorithm(self, context, feedback):
    return {}

  def displayName(self):
    return self.tr("Show point cloud grid")

  def createInstance(self):
    return listtengungrid()
