from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP)
from qgis.core import (
  QgsCoordinateReferenceSystem,
  QgsProcessingParameterExtent,
  QgsProject,
  QgsProcessingParameterNumber,
  QgsProcessingParameterEnum,
  QgsProcessingContext,
  QgsProcessingFeedback,
  QgsProcessingUtils
  )
from qgis import processing

from .algabstract import algabstract

import os
import re
import shutil
import glob
import os
import uuid
import requests
import concurrent

class fetchtengun(algabstract):
  
  # UIs
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchtengun","Extent for fetching data")
      }
    },
    "MAX_NUM_FILES": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "type": QgsProcessingParameterNumber.Integer,
        "description": QT_TRANSLATE_NOOP("fetchtengun","Maximum number of files to be downloaded"),
        "defaultValue": 1
      }
    },       
    "PC_DATABASE": {
      "ui_func": QgsProcessingParameterEnum,
      "ui_args": {
        "options": [],
        "description": QT_TRANSLATE_NOOP("fetchtengun","Point cloud database"),
        "defaultValue": 0
      }
    }
  }
    
  CRS = ""
    
  # initialization of the algorithm
  def initAlgorithm(self, config):    
    self.initUsingCanvas()
    self.PARAMETERS["PC_DATABASE"]["ui_args"]["options"] = [item["description"] for item in self.DBS]
    self.initParameters()
  
  
  # download a file from the URL
  def downloadFile(self, session, request_args:dict, download_msg:str, feedback_progress: bool, feedback: QgsProcessingFeedback) -> str:
    if len(download_msg) > 0:
      feedback.pushInfo(download_msg)
    try:
      response = session.request(stream = True, timeout = 12.0, **request_args)
    except Exception as e:
      feedback.reportError(self.tr("Download was failed."))
      raise e
    # if download is successful
    if response.status_code == 200:
      file_size = int(response.headers.get('content-length', 0))
      
      file_path = os.path.join(
        os.path.normpath(os.path.dirname(QgsProcessingUtils.generateTempFilename(""))), 
        str(uuid.uuid4())
      )

      # write downloaded file and show progress
      with open(file_path, 'wb') as file:
        for data in response.iter_content(chunk_size=1024):
          file.write(data)
          if feedback_progress:
            feedback.setProgress(int((file.tell() / file_size) * 100))
      
      if feedback_progress:
        feedback.setProgress(0)
      
      return file_path
      
    else:
      # if download is failed
      raise Exception(self.tr("Download was failed."))
  
  # download file(s) from the URL (asynchronously)
  def downloadFilesConcurrently(self, parameters: dict, context: QgsProcessingContext, feedback: QgsProcessingFeedback, parallel = True) -> None:
    self.WEBFETCH_ARGS["DOWNLOADED_FILE"] = []
    
    if not self.WEBFETCH_ARGS["SET"]:
      return
    # if login is needed, session is also needed
    if self.WEBFETCH_ARGS["LOGIN"] is None:
      session = requests.Session()
    else:
      session = self.WEBFETCH_ARGS["LOGIN"]["SESSION"]
      if not isinstance(session, requests.sessions.Session):
        feedback.reportError(self.tr("Login session is not given."))
        raise Exception(self.tr("Login session is not given"))
        
    # set download arguments
    n_url = len(self.WEBFETCH_ARGS["REQUEST_ARGS"])
    download_args = [
      {
        "download_msg": f"{i}/{n_url} Downloading {ra['url']}", 
        "request_args": ra
      } for i, ra in zip(range(1, n_url + 1), self.WEBFETCH_ARGS["REQUEST_ARGS"])
    ]
    mx_wk = None if parallel else 1
    
    # fetch in parallel
    with concurrent.futures.ThreadPoolExecutor(max_workers=mx_wk) as executor:
      for _ in executor.map(
        lambda d_arg : self.downloadFile(
          session, 
          request_args = d_arg["request_args"], 
          download_msg=d_arg["download_msg"], 
          feedback_progress = not parallel, feedback = feedback
          ), 
        download_args
        ):
        self.WEBFETCH_ARGS["DOWNLOADED_FILE"].append(_)
        if feedback.isCanceled():
          feedback.reportError(self.tr("Fetching was canceled."))
          raise Exception(self.tr("Fetching was canceled."))
  
  
  def setWebFetchArgs(self, parameters, context, feedback):
    
    self.WEBFETCH_ARGS["REQUEST_ARGS"] = []
    
    # set url and file names
    webfetch_option = self.parameterAsEnum(parameters, "PC_DATABASE", context)
    base_url = self.DBS[webfetch_option]["url"]
    pcode_format = self.DBS[webfetch_option]["pcode_format"]
    dx = self.DBS[webfetch_option]["dx"]
    dy = self.DBS[webfetch_option]["dy"]
         
    if "{pcode}" not in base_url:
      raise Exception(self.tr("The URL must contain {pcode}"))
    
    self.WEBFETCH_ARGS["REQUEST_ARGS"] = []
    x = self.FETCH_AREA.xMinimum()
    while x < self.FETCH_AREA.xMaximum():
      y = self.FETCH_AREA.yMinimum()
      while y < self.FETCH_AREA.yMaximum():
        self.WEBFETCH_ARGS["REQUEST_ARGS"].append(
          {
            "method": "GET",
            "url": base_url.replace("{pcode}", self.cmptMeshCodeFromXY(x, y, pcode_format=pcode_format))
          }
        )
        y += dy
      x += dx
    
    if len(self.WEBFETCH_ARGS["REQUEST_ARGS"]) > self.parameterAsInt(parameters, "MAX_NUM_FILES", context):
      raise Exception(self.tr("Too many files to be downloaded: ") + str(len(self.WEBFETCH_ARGS["REQUEST_ARGS"])))
    
    if self.WEBFETCH_ARGS["REQUEST_ARGS"] is not None:
      self.WEBFETCH_ARGS["SET"] = True
    
  
  # create the raster from downloaded hgt files
  def fetchFeaturesFromWeb(self, parameters: dict, context: QgsProcessingContext, feedback: QgsProcessingFeedback, parallel = False) -> None:
    self.downloadFilesConcurrently(parameters, context, feedback, parallel=parallel)
    
    # unpack zip files
    tengun_files = []
    try:
      for zip_dir in self.WEBFETCH_ARGS["DOWNLOADED_FILE"]:
        os.rename(zip_dir, zip_dir + ".zip")
        shutil.unpack_archive(zip_dir + ".zip", zip_dir)
        # archived_files = glob.glob(os.path.join(zip_dir, "*"))
        for unpacked_file in glob.glob(os.path.join(zip_dir, "*")):
          if re.search(".las", unpacked_file):
            tengun_files.append(unpacked_file)
    except Exception as e:
      feedback.reportError(self.tr("Unpacking zip files failed."))
      raise Exception(self.tr("Unpacking zip files failed."))
    
    # if there are no files
    if len(tengun_files) == 0:
      feedback.reportError(self.tr("No files were downloaded."))
      raise Exception(self.tr("No files were downloaded."))
    
    # if there are files
    else:      
      for pc in tengun_files:
        pc_projected = processing.run(
          "pdal:assignprojection", 
          {
            "INPUT": pc,
            "CRS": self.CRS,
            "OUTPUT": "TEMPORARY_OUTPUT"
          },
          feedback = feedback,
          context = context
        )["OUTPUT"]
        context.addLayerToLoadOnCompletion(pc_projected, context.LayerDetails(os.path.splitext(os.path.basename(pc))[0], QgsProject.instance(), ""))
  
  # execution of the algorithm
  def processAlgorithm(self, parameters, context, feedback):        
    # set url and file names
    self.CRS = QgsCoordinateReferenceSystem(self.DBS[self.parameterAsEnum(parameters, "PC_DATABASE", context)]["crs"])
    self.FETCH_AREA = self.parameterAsExtent(parameters, "FETCH_EXTENT", context, self.CRS)
    
    # set the meta information for obtaining SRTM data
    self.setWebFetchArgs(parameters, context, feedback)
    
    # download files using the session info
    self.fetchFeaturesFromWeb(parameters, context, feedback)
    
    return {"OUTPUT": None}
  
  # Post processing; append layers
  def postProcessAlgorithm(self, context, feedback):
    return {}

  def displayName(self):
    return self.tr("Fetch Point Cloud")

  def createInstance(self):
    return fetchtengun()
