from qgis.PyQt.QtCore import (
  QCoreApplication, QT_TRANSLATE_NOOP, QSettings
  )
from qgis.core import (
  QgsProcessingAlgorithm,
  QgsProcessingParameterDefinition, 
  QgsProcessingContext,
  QgsCoordinateReferenceSystem,
  QgsProcessingFeedback,
  QgsReferencedRectangle,
  QgsCoordinateTransform,
  QgsProject
  )

import os
from qgis.utils import iface
import math
import json

class algabstract(QgsProcessingAlgorithm):
        
  # fetch area
  FETCH_AREA = None
    
  # parameters for fetching data from other URL (e.g. SRTM)
  WEBFETCH_ARGS = {
    "SET": False,
    "REQUEST_ARGS": [],
    "DOWNLOADED_FILE": [],
    "URL_DICT": {},
    "LOGIN": None
  }
  
  
  def __init__(self):  
    super().__init__()
    with open(os.path.join(os.path.dirname(__file__),"pc-databases.json")) as f:
      self.DBS = json.load(f)
    locale = QSettings().value("locale/userLocale", "en", type=str)[0:2]
    if locale == "ja":
      self.DBS = [
        {**db_info, **{"description": db_info["description_ja"]}} for db_info in self.DBS
      ]
      
    
  # add parameters using PARAMETERS attribute
  def initParameters(self) -> None:    
    for key, value in self.PARAMETERS.items():
      try:
        args = value.get("ui_args")
        args["name"] = key
        args["description"] = self.tr(args["description"])
                
        ui = value.get("ui_func")(**args)
        
        if value.get("advanced") != None and value.get("advanced") == True:
          ui.setFlags(QgsProcessingParameterDefinition.FlagAdvanced)
          
        self.addParameter(ui)  
      except Exception as e:
        raise e
  
  # get the fetch extent and crs using the canvas
  def getExtentAndCrsUsingCanvas(self, get_xy_coords= True) -> (QgsReferencedRectangle, QgsCoordinateReferenceSystem):
    try:
      map_rectangle = iface.mapCanvas().extent()
      proj_crs = iface.mapCanvas().mapSettings().destinationCrs()
      
      use_rectangle = QgsReferencedRectangle(map_rectangle, proj_crs)
      
      use_crs = proj_crs
      if get_xy_coords:
        # if the CRS is geographic, input the long/lat to the getUtmCrs
        if proj_crs.isGeographic():
          use_crs = self.getUtmCrs(map_rectangle.center().x(), map_rectangle.center().y())
        # if the CRS is web-mercator, transform long/lat coordinates and input to the getUtmCrs
        elif proj_crs.authid() == "EPSG:3857":
          transform = QgsCoordinateTransform(proj_crs, QgsCoordinateReferenceSystem("EPSG:4326"), QgsProject.instance())
          map_rectangle = transform.transformBoundingBox(map_rectangle)
          use_crs = self.getUtmCrs(map_rectangle.center().x(), map_rectangle.center().y())
            
      return use_rectangle, use_crs
    except:
      return None, None
  
  # get CRS of the UTM using longitude and latitude
  def getUtmCrs(self, lng: float, lat: float) -> QgsCoordinateReferenceSystem:
    epsg_code = 32600 + 100 * (lat < 0) + int((lng + 180) / 6) + 1
    crs = QgsCoordinateReferenceSystem(f'EPSG:{epsg_code}')
    if crs.isValid():
        return crs
    else:
        return None
      
  # initialize extent and CRS using the canvas
  def initUsingCanvas(self) -> None:
    try:
      (rect, target_crs) = self.getExtentAndCrsUsingCanvas()

      self.PARAMETERS["FETCH_EXTENT"]["ui_args"]["defaultValue"] = ",".join(
        [str(rect.xMinimum()), str(rect.xMaximum()), str(rect.yMinimum()), str(rect.yMaximum())]
        ) + f" [{rect.crs().authid()}]"
      self.PARAMETERS["TARGET_CRS"]["ui_args"]["defaultValue"] = target_crs.authid()
    except:
      pass
      
    
  # to set the parameters for using data over the Internet
  def setWebFetchArgs(self, parameters: dict, context: QgsProcessingContext, feedback: QgsProcessingFeedback) -> None:
    pass 
  
  # compute the mesh code of 国土基本図図郭 from x and y coordinates
  def cmptMeshCodeFromXY(self, x, y, pcode_format = {"level": "500","prefix": "","case":"lower"}) -> str:
    dx1 = 40.0e3
    dy1 = 30.0e3
    nx1 = math.floor(x / dx1) 
    ny1 = math.floor(y / dy1)
    code1 = pcode_format["prefix"] + chr(-ny1 + 74) + chr(nx1 + 69)
    if pcode_format["case"] == "lower":
      code1 = code1.lower()
    
    if pcode_format["level"] == "50000":
      return code1
    dx2 = 4.0e3
    dy2 = 3.0e3
    nx2 = math.floor((x - nx1 * dx1) / dx2)
    ny2 = math.floor((y - ny1 * dy1) / dy2)
    code2 = f"{code1}{9-ny2}{nx2}"
    
    if pcode_format["level"] == "5000":
      return code2
    
    dx3 = 2.0e3
    dy3 = 1.5e3
    nx3 = math.floor((x - nx1 * dx1 - nx2 * dx2) / dx3)
    ny3 = math.floor((y - ny1 * dy1 - ny2 * dy2) / dy3)
    code3 = f"{code2}{2 * (1 - ny3) + nx3 + 1}"
    
    if pcode_format["level"] == "2500":
      return code3
    
    dx4 = 800.0
    dy4 = 600.0
    nx4 = math.floor((x - nx1 * dx1 - nx2 * dx2) / dx4)
    ny4 = math.floor((y - ny1 * dy1 - ny2 * dy2) / dy4)
    code4 = f"{code2}{4-ny4}" + chr(nx4 + 65)
    
    if pcode_format["level"] == "1000":
      return code4
    
    dx5 = 400.0
    dy5 = 300.0
    nx5 = math.floor((x - nx1 * dx1 - nx2 * dx2) / dx5)
    ny5 = math.floor((y - ny1 * dy1 - ny2 * dy2) / dy5)
    code5 = f"{code2}{9 - ny5}{nx5}"
    
    return code5
  
  
  
  def name(self):
    return self.__class__.__name__

  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
