#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
/**
 *   Copyright (C) 2012-2013 IFSTTAR (http://www.ifsttar.fr)
 *   Copyright (C) 2012-2013 Oslandia <infos@oslandia.com>
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *   
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

/***************************************************************************
 IfsttarRouting
                                 A QGIS plugin
 Get routing informations with various algorithms
                              -------------------
        begin                : 2012-04-12
        copyright            : (C) 2012-2013 by Oslandia
        email                : hugo.mercier@oslandia.com
 ***************************************************************************/
"""
import re
from .wps_client import *
from . import config
import binascii
import os
import sys
import math
import random
import colorsys
import time
import signal
import pickle
from datetime import datetime

# Import the PyQt and QGIS libraries
from qgis.PyQt.QtCore import *
from qgis.PyQt.QtGui import *
from qgis.PyQt.QtWidgets import *
#from PyQt4.QtCore import *
#from PyQt4.QtGui import *
from qgis.core import *
from qgis.gui import *
#import PyQt4.QtCore
#import PyQt4.QtGui

from xml.etree import ElementTree as etree
XML_SCHEMA_VALIDATION = True
try:
        from lxml import etree as ET
except ImportError:
        XML_SCHEMA_VALIDATION = False

# Initialize Qt resources from file resources.py
from . import resources_rc
# Import the code for the dialog
from .routing_dock import RoutingDock
# Import the splash screen
from .ui_splash_screen import Ui_SplashScreen

from .history_file import ZipHistoryFile
from .result_selection import ResultSelection
from .altitude_profile import AltitudeProfile
from .wkb import WKB
from . import tempus_request as Tempus
from .polygon_selection import PolygonSelection
import psycopg2
from .consolelauncher import ConsoleLauncher

from .compat import QgsMapLayerRegistry, QgsGeometry, \
    QgsRuleBasedRendererV2, QgsLineSymbolV2

import six


qgis_version_major = sys.version_info[0]
HISTORY_FILE = os.path.expanduser('~/.tempus_qgis.db')
PREFS_FILE = os.path.expanduser('~/.tempus_qgis_{}.prefs'.format(qgis_version_major))

ROADMAP_LAYER_NAME = "Tempus_Roadmap_"
ISOCHRONE_LAYER_NAME = "Tempus_Isochrone_"
TRACE_LAYER_NAME = "Tempus_Trace_"

# There has been an API change regarding vector layer on 1.9 branch
NEW_API = 'commitChanges' in dir(QgsVectorLayer)

#
# clears a FormLayout
def clearLayout( lay ):
    # clean the widget list
    for r in range(lay.count()):
        l = lay.takeAt(0)
        if l is not None:
            w = l.widget()
            if w is not None:
                w.setParent(None)
                w.deleteLater()

#
# clears a BoxLayout
def clearBoxLayout( lay ):
    rc = lay.count()
    if rc == 0:
        return
    for row in range(0, rc):
        # remove in reverse
        item = lay.itemAt(rc - row - 1)
        lay.removeItem(item)
        w = item.widget()
        lay.removeWidget(w)
        w.close()

# minutes (float) to hh:mm:ss
def min2hm( m ):
        hh = int(m/60)
        mm = int(m-hh*60)
        ss = int((m-hh*60-mm) * 60)
        return "%02d:%02d:%02d" % (hh,mm,ss)

class SplashScreen(QDialog):
    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_SplashScreen()
        self.ui.setupUi(self)

# process error message returned by WPS
def displayError( error ):
    if error.startswith('<ows:ExceptionReport'):
        # WPS exception
        t = ET.fromstring( error )
        QMessageBox.warning( None, "Tempus error", t[0][0].text )
    else:
        # HTML
        QMessageBox.warning( None, "Error", error )

class TempusPlugin:

    def __init__(self, iface):
        # Save reference to the QGIS interface
        self.iface = iface
        # Create the dialog and keep reference
        self.canvas = self.iface.mapCanvas()
        self.dlg = RoutingDock( self.canvas )
        if os.path.exists( PREFS_FILE ):
            f = open( PREFS_FILE, 'rb' )
            prefs = pickle.load( f )
            self.dlg.loadState( prefs['query'] )

        self.wps = None
        self.historyFile = ZipHistoryFile( HISTORY_FILE )
        self.dlg.ui.historyFileLbl.setText( HISTORY_FILE )

        # list of transport types
        # array of Tempus.TransportType
        self.transport_modes = []
        self.transport_modes_dict = {}
        # list of public networks
        # array of Tempus.TransportNetwork
        self.networks = []

        self.metadata = {}

        # plugin descriptions
        # dict plugin_name =>  Tempus.Plugin
        self.plugins = {}
        # where to store plugin options
        # dict plugin_name => Tempus.OptionValue
        self.plugin_options = {}

        self.currentRoadmap = None

        # prevent reentrance when roadmap selection is in progress
        self.selectionInProgress = False

        # profile widget
        self.profile = None

        self.setStateText("DISCONNECTED")

        self.server = None

    def initGui(self):
        # Create action that will start plugin configuration
        self.action = QAction(QIcon(":plugins/tempus/icon.png"), \
            u"Compute route", self.iface.mainWindow())
        # connect the action to the run method
        self.action.triggered.connect(self.run)

        self.dlg.ui.connectBtn.clicked.connect(self.onConnect)
        self.dlg.ui.tempusBinPathBrowse.clicked.connect( self.onBinPathBrowse )
        self.dlg.ui.addPluginToLoadBtn.clicked.connect( self.onAddPluginToLoad )
        self.dlg.ui.removePluginToLoadBtn.clicked.connect( self.onRemovePluginToLoad )
        self.dlg.ui.startServerBtn.clicked.connect( self.onStartServer )
        self.dlg.ui.stopServerBtn.clicked.connect( self.onStopServer )

        self.dlg.ui.computeBtn.clicked.connect(self.onCompute)
        self.dlg.ui.resetBtn.clicked.connect(self.onReset)
        self.dlg.ui.verticalTabWidget.currentChanged.connect(self.onTabChanged)

        self.dlg.ui.pluginCombo.currentIndexChanged.connect(self.update_plugin_options)

        # double click on a history's item
        self.dlg.ui.historyList.itemDoubleClicked.connect(self.onHistoryItemSelect)
        
        # click on the 'reset' button in history tab
        self.dlg.ui.reloadHistoryBtn.clicked.connect(self.loadHistory)
        self.dlg.ui.deleteHistoryBtn.clicked.connect(self.onDeleteHistoryItem)
        self.dlg.ui.importHistoryBtn.clicked.connect(self.onImportHistory)

        # click on a result radio button
        ResultSelection.buttonGroup.buttonClicked.connect(self.onResultSelected)

        self.dlg.ui.roadmapTable.itemSelectionChanged.connect(self.onRoadmapSelectionChanged)

        # show elevations button
        self.dlg.ui.showElevationsBtn.clicked.connect(self.onShowElevations)

        self.originPoint = QgsPoint()
        self.destinationPoint = QgsPoint()

        # Add toolbar button and menu item
        self.iface.addToolBarIcon(self.action)
        self.clipAction = QAction( u"Polygon subset (from mouse)", self.iface.mainWindow())
        self.clipAction.triggered.connect( self.onClip )
        self.clipActionFromSel = QAction( u"Polygon subset from selection", self.iface.mainWindow())
        self.clipActionFromSel.triggered.connect( self.onClipFromSel )
        self.loadLayersAction = QAction( u"Load layers", self.iface.mainWindow())
        self.loadLayersAction.triggered.connect( self.onLoadLayers )
        self.exportTraceAction = QAction( u"Export trace", self.iface.mainWindow())
        self.exportTraceAction.triggered.connect( self.onExportTrace )

        self.iface.addPluginToMenu(u"&Tempus", self.action)
        self.iface.addPluginToMenu(u"&Tempus", self.clipAction)
        self.iface.addPluginToMenu(u"&Tempus", self.clipActionFromSel)
        self.iface.addPluginToMenu(u"&Tempus", self.loadLayersAction)
        self.iface.addPluginToMenu(u"&Tempus", self.exportTraceAction)
        self.iface.addDockWidget( Qt.LeftDockWidgetArea, self.dlg )

        # init the history
        self.loadHistory()

        # init server configuration from settings
        s = QSettings()
        self.dlg.ui.tempusBinPathEdit.setText( s.value("IfsttarTempus/startTempus", "/usr/local/bin/startTempus.sh" ) )
        self.dlg.ui.dbOptionsEdit.setText( s.value("IfsttarTempus/dbOptions", "dbname=tempus_test_db user=postgres" ) )
        self.dlg.ui.schemaNameEdit.setText( s.value("IfsttarTempus/schemaName", "tempus" ) )
        plugins = s.value("IfsttarTempus/plugins", ["sample_road_plugin", "sample_multi_plugin", "dynamic_multi_plugin"] )
        for p in plugins:
            item = QListWidgetItem( p )
            item.setFlags( item.flags() | Qt.ItemIsEditable )
            self.dlg.ui.pluginsToLoadList.insertItem(0, item )

        self.clear()

    # reset widgets contents and visibility
    # as if it was just started
    def clear( self ):
        self.dlg.ui.verticalTabWidget.setTabEnabled( 1, False )
        self.dlg.ui.verticalTabWidget.setTabEnabled( 2, False )
        self.dlg.ui.verticalTabWidget.setTabEnabled( 3, False )
        self.dlg.ui.verticalTabWidget.setTabEnabled( 4, False )        
        self.dlg.ui.showElevationsBtn.hide()
        clearLayout( self.dlg.ui.resultLayout )
        self.dlg.ui.roadmapTable.clear()
        self.dlg.ui.roadmapTable.setRowCount(0)
        self.dlg.ui.roadmapTable.setHorizontalHeaderLabels( ["", "Direction", "Costs"] )

    def setStateText( self, text ):
        self.dlg.setWindowTitle( "Routing - " + text + "" )

    def get_pgsql_connection( self ):
        dlg = QInputDialog( None )

        q = QSettings()
        q.beginGroup("/PostgreSQL/connections")
        items = q.childGroups()
        current = items.index(q.value('selected'))
        connection, ok = QInputDialog.getItem(None, "Db connection", "Database to connect to", items, current, False )
        if not ok:
            return
        q.beginGroup(connection)
        db_params = "dbname='%s'" % q.value("database")
        host = q.value("host")
        if host:
            db_params += " host='%s'" % host
        port = q.value("port")
        if port:
            db_params += " port=%s" % port
        user = q.value("username")
        if user:
            db_params += " user='%s'" % user
        passwd = q.value("password")
        if passwd:
            db_params += " password='%s'" % passwd
        return db_params

    def onLoadLayers( self ):
        cwd = os.path.dirname(os.path.abspath(__file__))
        project_tmpl = cwd + '/tempus.qgs.tmpl'
        project = cwd + '/tempus.qgs'

        db_params = self.get_pgsql_connection()
        if db_params is None:
            return

        # FIXME better use psycopg2 here ?
        conn = psycopg2.connect(db_params)
        cur = conn.cursor()
        cur.execute("SELECT schema_name FROM tempus.subset")
        
        items = ('tempus', cur.fetchone()[0])
        schema, ok = QInputDialog.getItem(None, "Schema", "Schema to load", items, 0, False )
        if not ok:
            return

        fo = open(project, 'w+')
        with open(project_tmpl) as f:
            s = f.read()
            s = s.replace('%DB_PARAMS%', str(db_params))
            s = s.replace('%SCHEMA%', str(schema))
            fo.write( s )
            fo.close()
        QgsMapLayerRegistry.instance().removeAllMapLayers()
        ok = QgsProject.instance().read( QFileInfo( project ) )
        if not ok:
            QMessageBox.warning( None, "Project error", QgsProject.instance().error())

    def onExportTrace( self ):
        # look for the trace layer
        maps = QgsMapLayerRegistry.instance().mapLayers()
        trace_layer = None
        for k,v in maps.items():
            if v.name().startswith(TRACE_LAYER_NAME):
                trace_layer = v
                break
        if v is None:
            QMessageBox.warning( None, "No trace layer", "Cannot find a trace layer to export !")
            return

        # ask for a filename
        fn = QFileDialog.getSaveFileName( self.dlg, "GraphViz file where to export to", "", "GraphViz file (*.dot)" )
        if not fn:
            return

        nodes = {}
        edges = {}
        d = trace_layer.dataProvider()
        attr_names = [ f.name() for f in d.fields() if f.name() not in ('origin', 'destination') ]
        for f in d.getFeatures():
            attr = f.fields()
            o = f['origin']
            d = f['destination']
            g = f.geometry().asPolyline()
            nodes[ o ] = g[0]
            nodes[ d ] = g[-1]
            edge_attr = {}
            edges[ '%s->%s' % (o,d) ] = dict([(k, f[k]) for k in attr_names])

        with open(fn, 'w+') as f:
            f.write("digraph G {\n")
            for n,p in six.iteritems(nodes):
                f.write('%s[pos="%f,%f"];\n' % (n, p[0], p[1]))
            for n,attrs in six.iteritems(edges):
                attr_str = ' '.join(['%s=%s' % (k,v) for k,v in six.iteritems(attrs)])
                f.write(n)
                f.write('[label="' + attr_str + '"]\n')
            f.write('}\n')

    def onClip( self ):
        # select a database
        db_params = self.get_pgsql_connection()
        if db_params is None:
            return

        QMessageBox.information( None, "Polygon selection",
                                     "Please select the polygon used for subsetting. Cancel with right click. Confirm the last point with a double click.")
        self.select = PolygonSelection( self.canvas )
        self.select.polygonCaptured.connect( lambda valid : self.onPolygonCaptured(valid, db_params) )
        self.canvas.setMapTool( self.select )

    def onClipFromSel( self ):
        layer = self.iface.activeLayer()
        if not isinstance(layer, QgsVectorLayer):
            QMessageBox.critical( None, "Polygon selection",
                                  "No vector layer selected" )
            return
        if len(layer.selectedFeatures()) == 0:
            QMessageBox.critical( None, "Polygon selection",
                                  "No polygon selected")
            return
        if len(layer.selectedFeatures()) > 1:
            QMessageBox.critical( None, "Polygon selection",
                                  "More than one polygon selected" )
            return
        feature = layer.selectedFeatures()[0]
        if feature.geometry() is None or feature.geometry().type() != QGis.Polygon:
            QMessageBox.critical( None, "Polygon selection",
                                  "No polygon selected" )
            return

        geom = QgsGeometry(feature.geometry())
        trans = QgsCoordinateTransform(layer.crs(), QgsCoordinateReferenceSystem("EPSG:4326"))
        geom.transform( trans )
        polygon_wkt = geom.exportToWkt()

        # select a database
        db_params = self.get_pgsql_connection()
        if db_params is None:
            return

        self.doPolygonSubset( polygon_wkt, db_params )

    def onPolygonCaptured( self, valid, db_params ):
        self.canvas.unsetMapTool( self.select )
        if not valid:
            return
        wkt = self.select.polygon
        self.doPolygonSubset( wkt, db_params )

    def doPolygonSubset( self, wkt, db_params ):
        schema, ok = QInputDialog.getText(None, "Name of the subset to create (schema)", "Schema name")

        sql = "SELECT tempus.create_subset('%s','SRID=4326;%s');" % (schema, wkt)

        dlg = QDialog(self.dlg)
        v = QVBoxLayout()
        log = QLabel( dlg )
        v.addWidget(log)
        btn = QDialogButtonBox( dlg )
        btn.addButton(QDialogButtonBox.Ok)
        v.addWidget(btn)
        dlg.setLayout(v)
        log.setText('Executing script ...\n')
        dlg.show()

        QCoreApplication.processEvents()

        conn = psycopg2.connect(db_params)
        n_notices = len(conn.notices)
        conn.cursor().execute(sql)
        conn.commit()

        btn.accepted.connect( dlg.close )
        log.setText(''.join(conn.notices[n_notices:]))
        dlg.setModal( True )

    def onBinPathBrowse( self ):
        d = QFileDialog.getOpenFileName( self.dlg, "Tempus wps executable path" )
        if d is not None:
            self.dlg.ui.tempusBinPathEdit.setText( d )

    def onAddPluginToLoad( self ):
        item = QListWidgetItem("new_plugin")
        item.setFlags( item.flags() | Qt.ItemIsEditable)
        self.dlg.ui.pluginsToLoadList.insertItem(0,item)

    def onRemovePluginToLoad( self ):
        r = self.dlg.ui.pluginsToLoadList.currentRow()
        if r != -1:
            self.dlg.ui.pluginsToLoadList.takeItem( r )

    def onStartServer( self ):
        self.onStopServer()
        executable = self.dlg.ui.tempusBinPathEdit.text()
        dbOptions = self.dlg.ui.dbOptionsEdit.text()
        schemaName = self.dlg.ui.schemaNameEdit.text()
        plugins = []
        for i in range(self.dlg.ui.pluginsToLoadList.count()):
            plugins.append( self.dlg.ui.pluginsToLoadList.item(i).text() )
        cmdLine = [executable]
        cmdLine += ['-s', schemaName]
        for p in plugins:
            cmdLine += ['-l', p]
        if dbOptions != '':
            cmdLine += ['-d', dbOptions ]

        self.server = ConsoleLauncher( cmdLine )
        self.server.launch()

        # save parameters
        s = QSettings()
        s.setValue("IfsttarTempus/startTempus", executable )
        s.setValue("IfsttarTempus/dbOptions", dbOptions )
        s.setValue("IfsttarTempus/schemaName", schemaName )
        s.setValue("IfsttarTempus/plugins", plugins )

    def onStopServer( self ):
        if self.server is not None:
            self.server.stop()

    # when the 'connect' button gets clicked
    def onConnect( self ):
        try:
            self.wps = Tempus.TempusRequest( self.dlg.ui.wpsUrlText.text() )
        except RuntimeError as e:
            displayError( str(e))
            return

        self.getPluginList()
        if len(self.plugins) == 0:
            QMessageBox.warning( self.dlg, "Warning", "No plugin loaded server side, no computation could be done" )
        else:
            # initialize plugin option values
            self.initPluginOptions()
            self.displayPlugins( self.plugins )

            self.dlg.ui.pluginCombo.setEnabled( True )
            self.dlg.ui.verticalTabWidget.setTabEnabled( 1, True )
            self.dlg.ui.verticalTabWidget.setTabEnabled( 2, True )
            self.dlg.ui.verticalTabWidget.setTabEnabled( 3, True )
            self.dlg.ui.verticalTabWidget.setTabEnabled( 4, True )
            self.dlg.ui.computeBtn.setEnabled( True )
            self.setStateText("connected")

        # restore the history to HISTORY_FILE, if needed (after import)
        if self.historyFile.filename != HISTORY_FILE:
            self.historyFile = ZipHistoryFile( HISTORY_FILE )
            self.dlg.ui.historyFileLbl.setText( HISTORY_FILE )
            self.loadHistory()

    def getPluginList( self ):
        # get plugin list
        try:
            plugins = self.wps.plugin_list()
        except RuntimeError as e:
            displayError( str(e) )
            return
        self.plugins.clear()
        for plugin in plugins:
            self.plugins[plugin.name] = plugin

    def initPluginOptions( self ):
        self.plugin_options.clear()
        for name, plugin in six.iteritems(self.plugins):
            optval = {}
            for k,v in six.iteritems(plugin.options):
                optval[k] = v.default_value.value
            self.plugin_options[name] = optval

    def getConstants( self, plugin_name ):
        if self.wps is None:
            return
        try:
            constants = self.wps.constant_list(plugin_name)
            if len(constants) == 3:
                transport_modes, transport_networks, metadata = constants
            elif len(constants) == 2:
                transport_modes, transport_networks = constants
                metadata = {}
        except RuntimeError as e:
            displayError( str(e) )
            return
        self.transport_modes = transport_modes
        self.transport_modes_dict = {}
        for t in self.transport_modes:
                self.transport_modes_dict[t.id] = t

        self.networks = transport_networks

        self.metadata = metadata

    def update_plugin_options( self, plugin_idx ):
        if plugin_idx == -1:
            return

        plugin_name = str(self.dlg.ui.pluginCombo.currentText())
        self.displayPluginOptions( plugin_name )

        self.getConstants(plugin_name)
        self.displayTransportAndNetworks()

    def onTabChanged( self, tab ):
        # Plugin tab
        #if tab == 1:
        #    self.update_plugin_options( 0 )

        # 'Query' tab
        if tab == 2:
                # prepare for a new query
#            self.dlg.reset()
            self.dlg.inQuery()

    def loadHistory( self ):
        #
        # History tab
        #
        self.dlg.ui.historyList.clear()
        r = self.historyFile.getRecordsSummary()
        for record in r:
            id = record[0]
            date = record[1]
            dt = datetime.strptime( date, "%Y-%m-%dT%H:%M:%S.%f" )
            item = QListWidgetItem()
            item.setText(dt.strftime( "%Y-%m-%d at %H:%M:%S" ))
            item.setData( Qt.UserRole, int(id) )
            self.dlg.ui.historyList.addItem( item )

    def onDeleteHistoryItem( self ):
        msgBox = QMessageBox()
        msgBox.setIcon( QMessageBox.Warning )
        msgBox.setText( "Warning" )
        msgBox.setInformativeText( "Are you sure you want to remove these items from the history ?" )
        msgBox.setStandardButtons(QMessageBox.Yes | QMessageBox.No)
        ret = msgBox.exec_()
        if ret == QMessageBox.No:
            return

        sel = self.dlg.ui.historyList.selectedIndexes()
        for item in sel:
            id = item.data( Qt.UserRole )
            self.historyFile.removeRecord( id )
        # reload
        self.loadHistory()

    def onOptionChanged( self, plugin_name, option_name, option_type, val ):
            if option_type == Tempus.OptionType.Bool:
                    val = True if val else False
            elif option_type == Tempus.OptionType.Int:
                    # int
                    if val:
                            val = int(val)
                    else:
                            val = 0
            elif option_type == Tempus.OptionType.Float:
                    # float
                    if val:
                            val = float(val)
                    else:
                            val = 0.0
            self.plugin_options[plugin_name][option_name] = val

    def displayIsochrone(self, isochrone, lid):
        lname = "%s%d" % (ISOCHRONE_LAYER_NAME, lid)
        # create a new vector layer
        vl = QgsVectorLayer("Point?crs=epsg:4326", lname, "memory")

        pr = vl.dataProvider()

        vl.startEditing()
        vl.addAttribute( QgsField( "transport_mode", QVariant.Int ) )
        vl.addAttribute( QgsField( "mode_changes", QVariant.Int ) )
        vl.addAttribute( QgsField( "pt_changes", QVariant.Int ) )
        vl.addAttribute( QgsField( "cost", QVariant.Double ) )
        vl.commitChanges()

        for point in isochrone.points:
            x, y, mode, mode_changes, pt_changes, cost = point
            f = QgsFeature(vl.fields())
            f.setAttributes([mode, mode_changes, pt_changes, cost])
            f.setGeometry(QgsGeometry.fromPoint(QgsPoint(x, y)))
            pr.addFeatures([f])
            
        vl.updateExtents()

        QgsMapLayerRegistry.instance().addMapLayers( [vl] )
    #
    # Take a XML tree from the WPS 'result' operation
    # add an 'Itinerary' layer on the map
    #
    def displayRoadmapLayer(self, steps, lid):
        #
        # create layer

        lname = "%s%d" % (ROADMAP_LAYER_NAME, lid)
        # create a new vector layer
        vl = QgsVectorLayer("LineString?crs=epsg:4326", lname, "memory")

        pr = vl.dataProvider()

        if NEW_API:
            vl.startEditing()
            vl.addAttribute( QgsField( "transport_mode", QVariant.Int ) )
        else:
            pr.addAttributes( [ QgsField( "transport_mode", QVariant.Int ) ] )


        root_rule = QgsRuleBasedRendererV2.Rule( QgsLineSymbolV2.createSimple({'width': '1.5', 'color' : '255,255,0'} ))

        for mode_id, mode in six.iteritems(self.transport_modes_dict):
                p = float(mode_id-1) / len(self.transport_modes_dict)
                rgb = colorsys.hsv_to_rgb( p, 0.7, 0.9 )
                color = '%d,%d,%d' % (rgb[0]*255, rgb[1]*255, rgb[2]*255)
                rule = QgsRuleBasedRendererV2.Rule( QgsLineSymbolV2.createSimple({'width': '1.5', 'color' : color}),
                                                    0,
                                                    0,
                                                    "transport_mode=%d" % mode_id,
                                                    mode.name)
                root_rule.appendChild( rule )

        renderer = QgsRuleBasedRendererV2( root_rule );
        if hasattr(vl, "setRendererV2"):
            vl.setRendererV2( renderer )
        else:
            vl.setRenderer(renderer)

        # browse steps
        for step in steps:
            fet = QgsFeature()
            if step.wkb != '':
                # find wkb geometry
                wkb = WKB(step.wkb)
                wkb = wkb.force2d()
                geo = QgsGeometry()
                geo.fromWkb( binascii.unhexlify(wkb) )
                fet.setGeometry( geo )
            else:
                geo = QgsGeometry.fromWkt('POINT(0 0)')
                fet.setGeometry(geo)

            if isinstance(step, Tempus.TransferStep):
                mode = step.final_mode
            else:
                mode = step.mode
            if NEW_API:
                fet.setAttributes( [ mode ] )
            else:
                fet.setAttributeMap( { 0: mode } )
            pr.addFeatures( [fet] )

        if NEW_API:
            vl.commitChanges()
        else:
            # We MUST call this manually (see http://hub.qgis.org/issues/4687)
            vl.updateFieldMap()
        # update layer's extent when new features have been added
        # because change of extent in provider is not propagated to the layer
        vl.updateExtents()

        QgsMapLayerRegistry.instance().addMapLayers( [vl] )
        self.selectRoadmapLayer( lid )

        # connect selection change signal
        vl.selectionChanged.connect(lambda layer=vl: self.onLayerSelectionChanged(layer))

    def displayTrace(self, trace, lid, vl):

        # first pass to get the list of variants
        variants_type = {}
        for ve in trace:
            for k,v in six.iteritems(ve.variants):
                if variants_type.get(k) is None:
                    variants_type[k] = v.__class__

        if vl is None:
            lname = "%s%d" % ( TRACE_LAYER_NAME, lid )
            # create a new vector layer
            vl = QgsVectorLayer("LineString?crs=epsg:4326", lname, "memory")
            QgsMapLayerRegistry.instance().addMapLayers( [vl] )

        pr = vl.dataProvider()

        vl.startEditing()
        vl.addAttribute( QgsField( "type", QVariant.Int ) )
        vl.addAttribute( QgsField( "origin", QVariant.Int ) )
        vl.addAttribute( QgsField( "destination", QVariant.Int ) )
        for n, t in six.iteritems(variants_type):
            if t == int:
                vt = QVariant.Int
            elif t == float:
                vt = QVariant.Double
            elif t == str:
                vt = QVariant.String
            elif t == bool:
                vt = QVariant.Bool
            vl.addAttribute( QgsField( n, vt ) )

        features = []
        ntrace = len(trace)
        i = 0
        progressDlg = QProgressDialog("Loading traces ...", "Cancel", 0, ntrace)
        progressDlg.setWindowModality(Qt.WindowModal)
        for ve in trace:
            fet = QgsFeature()
            if ve.wkb != '':
                # find wkb geometry
                wkb = WKB(ve.wkb)
                wkb = wkb.force2d()
                geo = QgsGeometry()
                geo.fromWkb( binascii.unhexlify(wkb) )
                fet.setGeometry( geo )
            else:
                geo = QgsGeometry.fromWkt('POINT(0 0)')
                fet.setGeometry(geo)

            if isinstance(ve.origin, Tempus.RoadVertex) and isinstance(ve.destination, Tempus.RoadVertex):
                type = Tempus.ConnectionType.Road2Road
            elif isinstance(ve.origin, Tempus.RoadVertex) and isinstance(ve.destination, Tempus.PtVertex):
                type = Tempus.ConnectionType.Road2Transport
            elif isinstance(ve.origin, Tempus.PtVertex) and isinstance(ve.destination, Tempus.RoadVertex):
                type = Tempus.ConnectionType.Transport2Road
            elif isinstance(ve.origin, Tempus.PtVertex) and isinstance(ve.destination, Tempus.PtVertex):
                type = Tempus.ConnectionType.Transport2Transport
            elif isinstance(ve.origin, Tempus.RoadVertex) and isinstance(ve.destination, Tempus.PoiVertex):
                type = Tempus.ConnectionType.Road2Poi
            elif isinstance(ve.origin, Tempus.PoiVertex) and isinstance(ve.destination, Tempus.RoadVertex):
                type = Tempus.ConnectionType.Poi2Road
            attrs = [ type, ve.origin.id, ve.destination.id ]
            attrs += [ve.variants.get(k) for k in ve.variants.keys()]
            fet.setAttributes( attrs )

            features.append( fet )
            progressDlg.setValue(i)
            if progressDlg.wasCanceled():
                break
            i = i + 1

        pr.addFeatures( features )

        vl.commitChanges()
        vl.updateExtents()

    #
    # Select the roadmap layer
    #
    def selectRoadmapLayer( self, id ):
        lname = "%s%d" % (ROADMAP_LAYER_NAME, id)
        if hasattr(self.iface, "legendInterface"):
            # qgis 2
            legend = self.iface.legendInterface()
            maps = QgsMapLayerRegistry.instance().mapLayers()
            for k,v in maps.items():
                if v.name()[0:len(ROADMAP_LAYER_NAME)] == ROADMAP_LAYER_NAME:
                    if v.name() == lname:
                        legend.setLayerVisible( v, True )
                    else:
                        legend.setLayerVisible( v, False )
        else:
            # qgis 3
            ltr = QgsProject.instance().layerTreeRoot()
            for child in ltr.children():
                if child.nodeType() == QgsLayerTreeNode.NodeLayer:
                    vname = child.layer().name()
                    if vname[0:len(ROADMAP_LAYER_NAME)] == ROADMAP_LAYER_NAME:
                        child.setItemVisibilityChecked(vname == lname)
                    
    #
    # Take a XML tree from the WPS 'result' operation
    # and fill the "roadmap" tab
    #
    def displayRoadmapTab( self, result ):
        last_movement = 0
        roadmap = result.steps
        row = 0
        self.dlg.ui.roadmapTable.clear()
        self.dlg.ui.roadmapTable.setRowCount(0)
        self.dlg.ui.roadmapTable.setHorizontalHeaderLabels( ["", "", "Direction"] )

        # array of altitudes (distance, elevation)

        # get or create the graphics scene
        w = self.dlg.ui.resultSelectionLayout
        lastWidget = w.itemAt( w.count() - 1 ).widget()
        if isinstance(lastWidget, AltitudeProfile):
            self.profile = lastWidget
        else:
            self.profile = AltitudeProfile( self.dlg )
            self.profile.hide()
            self.dlg.ui.resultSelectionLayout.addWidget( self.profile )

        self.profile.clear()

        current_time = result.starting_date_time.hour*60 + result.starting_date_time.minute + result.starting_date_time.second / 60.0;

        first = True
        leaving_time = None

        for step in roadmap:
            if first:
                if isinstance(step, Tempus.TransferStep ) and step.poi == '':
                    from_private_parking = True
                    text = ''
                else:
                    from_private_parking = False
                    text = "Initial mode: %s<br/>\n" % self.transport_modes_dict[step.mode].name
                first = False
            else:
                text = ''
            icon_text = ''
            cost_text = ''

            if step.wkb and step.wkb != '':
                wkb = WKB(step.wkb)
                pts = wkb.dumpPoints()
                if len(pts) > 0: # if 3D points ...
                        prev = pts[0]
                        for p in pts[1:]:
                                dist = math.sqrt((p[0]-prev[0])**2 + (p[1]-prev[1])**2)
                                self.profile.addElevation( dist, prev[2], p[2], row )
                                prev = p

            if isinstance(step, Tempus.RoadStep):
                road_name = step.road
                movement = step.end_movement
                text += "<p>"
                action_txt = 'Continue on '
                if last_movement == Tempus.EndMovement.TurnLeft:
                    icon_text += "<img src=\"%s/turn_left.png\" width=\"24\" height=\"24\"/>" % config.DATA_DIR
                    action_txt = "Turn left on "
                elif last_movement == Tempus.EndMovement.TurnRight:
                    icon_text += "<img src=\"%s/turn_right.png\" width=\"24\" height=\"24\"/>" % config.DATA_DIR
                    action_txt = "Turn right on "
                elif last_movement >= Tempus.EndMovement.RoundAboutEnter and last_movement < Tempus.EndMovement.YouAreArrived:
                    icon_text += "<img src=\"%s/roundabout.png\" width=\"24\" height=\"24\"/>" % config.DATA_DIR
                text += action_txt + road_name + "<br/>\n"
                text += "</p>"
                last_movement = movement

            elif isinstance(step, Tempus.PublicTransportStep ):
                # set network text as icon
                icon_text = step.network
                if step.wait_time > 0.0:
                        text += "Wait %s <br/>" % min2hm(step.wait_time)
                text += "At %s<br/>\n" % min2hm(step.departure_time+step.wait_time)
                pt_name = self.transport_modes_dict[step.mode].name
                text += "%s Take the %s %s from '%s' to '%s'" % (step.trip_id, pt_name, step.route, step.departure, step.arrival)
                leaving_time = step.arrival_time

            elif isinstance(step, Tempus.RoadTransportStep ):
                icon_text = step.network
                if step.type == Tempus.ConnectionType.Road2Transport:
                    text += "Go to the '%s' station from %s" % (step.stop, step.road)
                elif step.type == Tempus.ConnectionType.Transport2Road:
                        if leaving_time is not None:
                                text += "At %s<br/>\n" % min2hm(leaving_time)
                        text += "Leave the '%s' station to %s" % (step.stop, step.road)
                else:
                    text += "Connection between '%s' and '%s'" % (step.stop, step.road)

            elif isinstance(step, Tempus.TransferStep ):
                if from_private_parking:
                    # my private parking
                    text += "Take your %s<br/>\n" % self.transport_modes_dict[step.final_mode].name
                    text += "Go on %s" % step.road
                    from_private_parking = False
                else:
                    text += "On %s,<br/>At %s:<br/>\n" % (step.road, step.poi)
                    if step.mode != step.final_mode:
                            imode = self.transport_modes_dict[step.mode]
                            fmode = self.transport_modes_dict[step.final_mode]
                            add_t = []
                            if imode.need_parking:
                                    add_t += ["park your %s" % imode.name]
                            add_t += ["take a %s" % fmode.name]
                            text += ' and '.join(add_t)
                    else:
                            text = "Continue on %s" % step.road

            for k,v in six.iteritems(step.costs):
                if k == Tempus.Cost.Duration:                    
                    cost_text += "At %s<br/>\n" % min2hm(current_time)
                    current_time += v
                    cost_text += "Duration " + min2hm(v)
                else:
                    cost_text += "%s: %.1f %s<br/>\n" % (Tempus.CostName[k], v, Tempus.CostUnit[k])
                
            self.dlg.ui.roadmapTable.insertRow( row )
            descLbl = QLabel()
            descLbl.setText( text )
            descLbl.setMargin( 5 )
            self.dlg.ui.roadmapTable.setCellWidget( row, 2, descLbl )

            if icon_text != '':
                lbl = QLabel()
                lbl.setText( icon_text )
                lbl.setMargin(5)
                self.dlg.ui.roadmapTable.setCellWidget( row, 1, lbl )

            costText = QLabel()
            costText.setText( cost_text )
            costText.setMargin( 5 )
            self.dlg.ui.roadmapTable.setCellWidget( row, 0, costText )
            self.dlg.ui.roadmapTable.resizeRowToContents( row )
            row += 1

        # Adjust column widths
        w = self.dlg.ui.roadmapTable.sizeHintForColumn(0)
        self.dlg.ui.roadmapTable.horizontalHeader().resizeSection( 0, w )
        w = self.dlg.ui.roadmapTable.sizeHintForColumn(1)
        self.dlg.ui.roadmapTable.horizontalHeader().resizeSection( 1, w )
        w = self.dlg.ui.roadmapTable.sizeHintForColumn(2)
        self.dlg.ui.roadmapTable.horizontalHeader().resizeSection( 2, w )
        self.dlg.ui.roadmapTable.horizontalHeader().setStretchLastSection( True )

        if not self.profile.empty():
            self.dlg.ui.showElevationsBtn.setEnabled(True)
            self.profile.displayElevations()
        else:
            self.dlg.ui.showElevationsBtn.setEnabled(False)
    #
    # Take a XML tree from the WPS 'metrics' operation
    # and fill the 'Metrics' tab
    #
    def displayMetrics( self, metrics ):
        row = 0
        clearLayout( self.dlg.ui.resultLayout )

        for name, metric in six.iteritems(metrics):
            lay = self.dlg.ui.resultLayout
            lbl = QLabel( self.dlg )
            lbl.setText( name )
            lay.setWidget( row, QFormLayout.LabelRole, lbl )
            widget = QLineEdit( self.dlg )
            widget.setText( str(metric) )
            widget.setEnabled( False )
            lay.setWidget( row, QFormLayout.FieldRole, widget )
            row += 1

    #
    # Take XML trees of 'plugins' and an index of selection
    # and fill the 'plugin' tab
    #
    def displayPlugins( self, plugins ):
        self.dlg.ui.pluginCombo.blockSignals( True )
        self.dlg.ui.pluginCombo.clear()
        self.options_desc = {}
        for name, plugin in six.iteritems(plugins):
            self.dlg.ui.pluginCombo.insertItem(0, name )
        self.dlg.ui.pluginCombo.blockSignals( False )
        self.update_plugin_options(0)

    #
    # Take XML tree of 'options' and a dict 'option_values'
    # and fill the options part of the 'plugin' tab
    #
    def displayPluginOptions( self, plugin_name ):
        vlay = self.dlg.ui.optionsLayout
        clearLayout( vlay )

        # sort options by category
        options_by_cat = {}
        for name, option in six.iteritems(self.plugins[plugin_name].options):
            s = name.split("/")
            if len(s)>1:
                cat_name = s[0]
                option_name = s[1]
            else:
                cat_name = ""
                option_name = name
            if options_by_cat.get( cat_name ) is None:
                options_by_cat[cat_name] = []
            options_by_cat[cat_name].append( (option_name, option) )

        for cat_name, options in six.iteritems(options_by_cat):

            if cat_name != "":
                q = QGroupBox( cat_name )
            else:
                q = QWidget()

            lay = QFormLayout( q )
            vlay.addWidget( q )

            row = 0
            for name, option in options:
                if cat_name != "":
                    complete_name = cat_name + "/" + name
                else:
                    complete_name = name

                lbl = QLabel()
                lbl.setText( option.description )
                lay.setWidget( row, QFormLayout.LabelRole, lbl )

                t = option.type()

                val = self.plugin_options[plugin_name][complete_name]
                if t == Tempus.OptionType.Bool:
                    widget = QCheckBox()
                    if val == True:
                        widget.setCheckState( Qt.Checked )
                    else:
                        widget.setCheckState( Qt.Unchecked )
                    widget.toggled.connect(lambda checked, name=complete_name, t=t, pname=plugin_name: self.onOptionChanged( pname, name, t, checked ))
                else:
                    widget = QLineEdit( self.dlg )
                    if t == Tempus.OptionType.Int:
                        valid = QIntValidator( widget )
                        widget.setValidator( valid )
                    if t == Tempus.OptionType.Float:
                        valid = QDoubleValidator( widget )
                        widget.setValidator( valid )
                    widget.setText( str(val) )
                    widget.textChanged.connect(lambda text, name=complete_name, t=t, pname=plugin_name: self.onOptionChanged( pname, name, t, text ))
                lay.setWidget( row, QFormLayout.FieldRole, widget )

                row += 1

        self.dlg.set_supported_criteria( self.plugins[plugin_name].supported_criteria )
        self.dlg.set_intermediate_steps_support( self.plugins[plugin_name].intermediate_steps )
        self.dlg.set_depart_after_support( self.plugins[plugin_name].depart_after )
        self.dlg.set_arrive_before_support( self.plugins[plugin_name].arrive_before )

    #
    # Take an XML trees from 'constant_list'
    # load them and display them
    def displayTransportAndNetworks( self ):

        listModel = QStandardItemModel()
        for ttype in self.transport_modes:
            item = QStandardItem( ttype.name )
            item.setFlags(Qt.ItemIsUserCheckable | Qt.ItemIsEnabled)
            if ttype.id != 1:
                    item.setData( Qt.Unchecked, Qt.CheckStateRole)
            else:
                    item.setData( Qt.Checked, Qt.CheckStateRole )
            item.setData( ttype.id, Qt.UserRole )
            listModel.appendRow(item)

        def on_transport_changed( item ):
            t = self.dlg.selected_transports()
            c = 1 in t or 2 in t # private cycle or car
            self.dlg.parkingEnabledBox.setEnabled( c )
            self.dlg.parkingChooser.setEnabled( c and self.dlg.parkingEnabledBox.checkState() == Qt.Checked )

        listModel.itemChanged.connect( on_transport_changed )
        self.dlg.ui.transportList.setModel( listModel )
        #on_transport_changed( item )

    #
    # Take an XML tree from the WPS 'results'
    # and load/display them
    #
    def displayResults( self, results ):
        # result radio button id
        self.result_ids=[]
        # clear the vbox layout
        clearBoxLayout( self.dlg.ui.resultSelectionLayout )
        k = 1
        for result in results:
            rselect = ResultSelection()
            if isinstance(result, Tempus.Roadmap):
                start = "%02d:%02d:%02d" % (result.starting_date_time.hour,result.starting_date_time.minute,result.starting_date_time.second)
                name = "%s%d (start: %s)" % (ROADMAP_LAYER_NAME,k, start)
                for k,v in six.iteritems(result.costs):
                    rselect.addCost(Tempus.CostName[k], "%.1f%s" % (v, Tempus.CostUnit[k]))
                    if k == Tempus.Cost.Duration:
                        s = result.starting_date_time.hour * 60 + result.starting_date_time.minute + result.starting_date_time.second / 60.0;
                        s += v
                        name = "%s%d (%s -> %s)" % (ROADMAP_LAYER_NAME,k, start, min2hm(s) )
                rselect.setText( name )
            elif isinstance(result, Tempus.Isochrone):
                rselect.setText("Isochrone")
            self.dlg.ui.resultSelectionLayout.addWidget( rselect )
            self.result_ids.append( rselect.id() )
            k += 1

        # delete pre existing roadmap layers
        maps = QgsMapLayerRegistry.instance().mapLayers()
        trace_layer = None
        for k,v in maps.items():
            if v.name().startswith(ROADMAP_LAYER_NAME):
                QgsMapLayerRegistry.instance().removeMapLayers( [k] )
            elif v.name().startswith(TRACE_LAYER_NAME):
                trace_layer = v
                d = v.dataProvider()
                ids = [f.id() for f in d.getFeatures()]
                d.deleteFeatures(ids)
                attr_ids = list(range(len(d.fields())))
                d.deleteAttributes(attr_ids)

        # then display each layer
        k = 1
        for result in results:
            if isinstance(result, Tempus.Roadmap):
                self.displayRoadmapLayer( result.steps, k )
                if result.trace is not None:
                    self.displayTrace( result.trace, k, trace_layer )
            elif isinstance(result, Tempus.Isochrone):
                self.displayIsochrone(result, k)
            k += 1


    def onResultSelected( self, id ):
        for i in range(0, len(self.result_ids)):
            if id == self.result_ids[i]:
                self.displayRoadmapTab( self.results[i] )
                self.selectRoadmapLayer( i+1 )
                self.currentRoadmap = i
                self.dlg.ui.showElevationsBtn.show()
                break


    def onReset( self ):
        if os.path.exists( PREFS_FILE ):
            f = open( PREFS_FILE, 'rb' )
            prefs = pickle.load( f )
            self.dlg.loadState( prefs['query'] )
            self.plugin_options = prefs['plugin_options']
            self.update_plugin_options(0)

    #
    # When the 'compute' button gets clicked
    #
    def onCompute(self):
        if self.wps is None:
            return

        # retrieve request data from dialogs
        coords = self.dlg.get_coordinates()
        [ox, oy] = coords[0]
        criteria = self.dlg.get_selected_criteria()
        constraints = self.dlg.get_constraints()
        has_constraint = sum([ x for x,_ in constraints ]) > 0
        for i in range(len(constraints)-1):
            ci, ti = constraints[i]
            cj, tj = constraints[i+1]
            dti = datetime.strptime(ti, "%Y-%m-%dT%H:%M:%S")
            dtj = datetime.strptime(tj, "%Y-%m-%dT%H:%M:%S")
            if ci == 2 and cj == 1 and dtj < dti:
                QMessageBox.warning( self.dlg, "Warning", "Impossible constraint : " + tj + " < " + ti )
                return

        parking = self.dlg.get_parking()
        pvads = self.dlg.get_pvads()
        #networks = [ self.networks[x].id for x in self.dlg.selected_networks() ]
        transports = [ self.transport_modes[x].id for x in self.dlg.selected_transports() ]
        has_pt = False
        for x in self.dlg.selected_transports():
            if self.transport_modes[x].is_public_transport:
                has_pt = True
                break

        if transports == []:
            QMessageBox.warning( self.dlg, "Warning", "No transport mode is selected, defaulting to pedestrian walk")
            transports = [ 1 ]

        if has_pt and not has_constraint:
            QMessageBox.critical( self.dlg, "Inconsistency", "Some public transports are selected, but not time constraint is specified" )
            return

        # build the request
        currentPlugin = str(self.dlg.ui.pluginCombo.currentText())

        steps = []
        for i in range(0, len(constraints)):
            steps.append( Tempus.RequestStep( private_vehicule_at_destination = pvads[i],
                                              destination = Tempus.Point( coords[i+1][0], coords[i+1][1] ),
                                constraint = Tempus.Constraint( type = constraints[i][0], date_time = constraints[i][1] )
                                )
                          )

        try:
            select_xml = self.wps.request( plugin_name = currentPlugin,
                                           plugin_options = self.plugin_options[currentPlugin],
                                           origin = Tempus.Point(ox, oy),
                                           allowed_transport_modes = transports,
                                           parking_location = None if parking == [] else Tempus.Point(parking[0], parking[1]),
                                           criteria = criteria,
                                           steps = steps
                                           )
        except RuntimeError as e:
            displayError( str(e) )
            return

        self.results = self.wps.results
        self.displayResults( self.results )
        self.displayMetrics( self.wps.metrics )

        # enable 'metrics' and 'roadmap' tabs
        self.dlg.ui.verticalTabWidget.setTabEnabled( 3, True )
        self.dlg.ui.verticalTabWidget.setTabEnabled( 4, True )
        # clear the roadmap table
        self.dlg.ui.roadmapTable.clear()
        self.dlg.ui.roadmapTable.setRowCount(0)

        # save query / plugin options
        prefs = {}
        prefs['query'] = self.dlg.saveState()
        prefs['plugin_options'] = dict(self.plugin_options)
        f = open( PREFS_FILE, 'wb+' )
        pickle.dump( prefs, f )

        # save the request and the server state
        xml_record = '<record>' + select_xml
        if "metadata" in self.wps.save:
            server_state = to_xml([ 'server_state',
                                    etree.tostring(self.wps.save['plugins']),
                                    [ 'plugin_info', [ 'plugin', {'name': currentPlugin} ],
                                      etree.tostring(self.wps.save[currentPlugin+'/transport_modes']),
                                      etree.tostring(self.wps.save[currentPlugin+'/transport_networks']),
                                      etree.tostring(self.wps.save[currentPlugin+'/metadata'])
                                      ]
                                    ])
        else:
            server_state = to_xml([ 'server_state',
                                    etree.tostring(self.wps.save['plugins']),
                                    [ 'plugin_info', [ 'plugin', {'name': currentPlugin} ],
                                      etree.tostring(self.wps.save[currentPlugin+'/transport_modes']),
                                      etree.tostring(self.wps.save[currentPlugin+'/transport_networks'])
                                  ]
                                ])
        xml_record += server_state + '</record>'
        self.historyFile.addRecord( xml_record )

    def onHistoryItemSelect( self, item ):
        msgBox = QMessageBox()
        msgBox.setIcon( QMessageBox.Warning )
        msgBox.setText( "Warning" )
        msgBox.setInformativeText( "Do you want to load this item from the history ? Current options and query parameters will be overwritten" )
        msgBox.setStandardButtons(QMessageBox.Yes | QMessageBox.No)
        ret = msgBox.exec_()
        if ret == QMessageBox.No:
            return

        id = item.data( Qt.UserRole )
        # load from db
        (id, date, xmlStr) = self.historyFile.getRecord( id )

        if XML_SCHEMA_VALIDATION:
                # validate entry
                schema_root = ET.parse( os.path.join(config.TEMPUS_DATA_DIR, "wps_schemas", "record.xsd") )
                schema = ET.XMLSchema(schema_root)
                parser = ET.XMLParser(schema = schema)
                try:
                        tree = ET.fromstring( xmlStr, parser)
                except ET.XMLSyntaxError as e:
                        QMessageBox.warning( self.dlg, "XML parsing error", e.msg )
                        return

        else:
                tree = etree.XML(xmlStr)
        loaded = {}
        for child in tree:
            loaded[ child.tag ] = child

        # reset
        self.clear()

        # update UI
        self.plugins.clear()
        plugins = Tempus.parse_plugins(loaded['server_state'][0])
        for plugin in plugins:
            self.plugins[plugin.name] = plugin
        self.initPluginOptions()
        self.displayPlugins( self.plugins )

        # get current plugin option values
        currentPlugin = loaded['select'][0].attrib['name']
        myoptions = Tempus.parse_plugin_options( loaded['select'][2] )
        for k, v in six.iteritems(myoptions):
            self.plugin_options[currentPlugin][k] = v.value
        # select currendt plugin
        idx = self.dlg.ui.pluginCombo.findText( currentPlugin )
        self.dlg.ui.pluginCombo.setCurrentIndex( idx )

        plugin_infos = loaded['server_state'][1:]
        for plugin_info in plugin_infos:
            pn = plugin_info[0].attrib['name']
            print(pn)
            if pn != currentPlugin:
                continue

            # select the current plugin info
            self.transport_modes = Tempus.parse_transport_modes( plugin_info[1] )
            self.transport_modes_dict = {}
            for t in self.transport_modes:
                self.transport_modes_dict[t.id] = t
            self.networks = Tempus.parse_transport_networks( plugin_info[2] )
            self.displayTransportAndNetworks()
            if len(plugin_info)>3: # has metadata
                self.metadata = Tempus.parse_metadata( plugin_info[3] )
            else:
                self.metadata = {}

        self.dlg.loadFromXML( loaded['select'][1] )
        self.displayMetrics( Tempus.parse_metrics(loaded['select'][4]) )
        self.results = Tempus.parse_results(loaded['select'][3])
        self.displayResults( self.results )

        # enable tabs
        self.dlg.ui.verticalTabWidget.setTabEnabled( 1, True )
        self.dlg.ui.verticalTabWidget.setTabEnabled( 2, True )
        self.dlg.ui.verticalTabWidget.setTabEnabled( 3, True )
        self.dlg.ui.verticalTabWidget.setTabEnabled( 4, True )

        # update pinpoints layers
        self.dlg.updateLayers()
        
        # simulate a disconnection
        self.wps = None
        self.dlg.ui.computeBtn.setEnabled(False)
        self.setStateText("DISCONNECTED")

    def onImportHistory( self ):
        fname = QFileDialog.getOpenFileName( None, 'Import history file', '')
        if not fname:
            return

        # switch historyFile
        self.historyFile = ZipHistoryFile( fname )
        self.dlg.ui.historyFileLbl.setText( fname )
        self.loadHistory()

    # when the user selects a row of the roadmap
    def onRoadmapSelectionChanged( self ):
        if self.selectionInProgress:
            return

        selected = self.dlg.ui.roadmapTable.selectedIndexes()
        rows = []
        for s in selected:
            r = s.row()
            if r not in rows: rows.append( r )

        roadmap = self.results[ self.currentRoadmap ]

        # find layer
        layer = None
        lname = "%s%d" % (ROADMAP_LAYER_NAME, self.currentRoadmap)
        maps = QgsMapLayerRegistry.instance().mapLayers()
        for k,v in maps.items():
            if v.name()[0:len(ROADMAP_LAYER_NAME)] == ROADMAP_LAYER_NAME:
                layer = v
                break
        # the layer may have been deleted
        if not layer:
            return

        self.selectionInProgress = True
        layer.selectAll()
        layer.invertSelection() # no deselectAll ??
        for row in rows:
            # row numbers start at 0 and ID starts at 1
            layer.select( row + 1)

        # highlight selection on the elevation profile
        self.profile.highlightSelection( rows )
        self.selectionInProgress = False

    # when selection on the itinerary layer changes
    def onLayerSelectionChanged( self, layer ):
        if self.selectionInProgress:
            return

        n = len(ROADMAP_LAYER_NAME)
        if len(layer.name()) <= n:
            return
        layerid = int(layer.name()[n:])-1
        self.currentRoadmap = layerid

        # in case the layer still exists, but not the underlying result
        if len(self.results) <= layerid:
            return

        # block signals to prevent infinite loop
        self.selectionInProgress = True
        self.displayRoadmapTab( self.results[layerid-1] )

        selected = [ feature.id()-1 for feature in layer.selectedFeatures() ]
        k = 0
        if len(selected) > 0:
                for fid in selected:
                        self.dlg.ui.roadmapTable.selectRow( fid )
                        if k == 0:
                                selectionModel = self.dlg.ui.roadmapTable.selectionModel()
                                selectionItem = selectionModel.selection()
                                k += 1
                        else:
                                selectionItem.merge( selectionModel.selection(), QItemSelectionModel.Select )
                selectionModel.select( selectionItem, QItemSelectionModel.Select )

        self.profile.highlightSelection( selected )
        self.selectionInProgress = False

    def onShowElevations( self ):
        if not self.profile:
            return
        if self.profile.isVisible():
            self.profile.hide()
            self.dlg.ui.showElevationsBtn.setText("Show elevations")
        else:
            self.profile.show()
            self.dlg.ui.showElevationsBtn.setText("Hide elevations")

    def unload(self):
        # Remove the plugin menu item and icon
        self.iface.removePluginMenu(u"&Tempus",self.action)
        self.iface.removePluginMenu(u"&Tempus",self.clipAction)
        self.iface.removePluginMenu(u"&Tempus",self.clipActionFromSel)
        self.iface.removePluginMenu(u"&Tempus",self.loadLayersAction)
        self.iface.removePluginMenu(u"&Tempus",self.exportTraceAction)
        self.iface.removeToolBarIcon(self.action)
        if self.server is not None:
            self.server.stop()

    # run method that performs all the real work
    def run(self):
        # show the dialog
        self.dlg.show()

