from PyQt5.QtWidgets import QInputDialog, QInputDialog, QTreeView, QStyledItemDelegate, QColorDialog, QMenu, QLineEdit, QDialog, QVBoxLayout, QLabel, QPushButton, QHBoxLayout, QFileDialog, QComboBox, QFrame, QCheckBox, QDoubleSpinBox, QRadioButton, QButtonGroup, QProgressBar, QDialogButtonBox, QGraphicsView, QListWidget, QScrollBar, QDesktopWidget, QGraphicsEllipseItem, QGraphicsScene, QToolTip, QGraphicsPathItem, QGraphicsRectItem, QGraphicsLineItem, QListWidgetItem, QWidget, QListView, QAbstractItemView, QScrollArea, QSizePolicy, QMessageBox
from qgis.core import QgsProject, QgsMapLayer, QgsWkbTypes, QgsSingleSymbolRenderer, QgsCategorizedSymbolRenderer, QgsSymbol, Qgis, QgsVectorLayerSimpleLabeling, QgsSimpleLineSymbolLayer, QgsRenderContext, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsMessageLog, QgsLayerTreeLayer, QgsSymbolLayer, QgsGeometry, QgsSpatialIndex, QgsVectorFileWriter, QgsDefaultValue, QgsEditorWidgetSetup, QgsEditFormConfig, QgsVectorLayer, QgsFields, QgsField, QgsFeature, QgsPointXY, QgsPoint, QgsPalLayerSettings, QgsProperty, QgsMarkerSymbol
from PyQt5.QtGui import QStandardItemModel, QStandardItem, QIcon, QPixmap, QPainter, QColor, QPen, QFont, QBrush, QGuiApplication, QTransform, QCursor, QPainterPath, QPolygonF, QMouseEvent, QWheelEvent
from PyQt5.QtCore import Qt, QPoint, QRect, QEvent, QCoreApplication, QSettings, QItemSelectionModel, QPointF, QSize, QUrl, QVariant, QObject
from PyQt5.QtNetwork import QNetworkAccessManager, QNetworkRequest
from qgis.gui import QgsProjectionSelectionDialog
from PIL import Image, UnidentifiedImageError
import xml.etree.ElementTree as ET
from itertools import zip_longest
from ezdxf.colors import rgb2int, aci2rgb
from qgis.utils import iface
from ezdxf import colors
from io import BytesIO
import processing
import openpyxl
import requests
import random
import ezdxf
import time
import math
import csv
import os
import re
import gc

# Importe as Classes Externas
from .gerenciar_etiquetas_dialog import GerenciarEtiquetasDialog
from .salvar_multiplos import DialogoSalvarFormatos
from .criar_pontos import CamadaPontosManager

class UiManagerT:
    """
    Gerencia a interface do usuário, interagindo com um QTreeView para listar e gerenciar camadas de pontos no QGIS.
    """
    def __init__(self, iface, dialog):
        """
        Inicializa a instância da classe UiManagerO, responsável por gerenciar a interface do usuário
        que interage com um QTreeView para listar e gerenciar camadas de pontos no QGIS.

        :param iface: Interface do QGIS para interagir com o ambiente.
        :param dialog: Diálogo ou janela que esta classe gerenciará.

        Funções e Ações Desenvolvidas:
        - Configuração inicial das variáveis de instância.
        - Associação do modelo de dados com o QTreeView.
        - Inicialização da configuração do QTreeView.
        - Seleção automática da última camada no QTreeView.
        - Conexão dos sinais do QGIS e da interface do usuário com os métodos correspondentes.
        """
        # Salva as referências para a interface do QGIS e o diálogo fornecidos
        self.iface = iface
        self.dlg = dialog

        # Cria e configura o modelo de dados para o QTreeView
        self.treeViewModel = QStandardItemModel()
        self.dlg.treeViewListaPonto.setModel(self.treeViewModel)

        # Inicializa o QTreeView com as configurações necessárias
        self.init_treeView()

        # Conecta os sinais do QGIS e da interface do usuário para sincronizar ações e eventos
        self.connect_signals()

        # Adiciona o filtro de eventos ao treeView
        self.tree_view_event_filter = TreeViewEventFilter(self)
        self.dlg.treeViewListaPonto.viewport().installEventFilter(self.tree_view_event_filter)

        # instancia uma vez só o manager
        self.pontos_manager = CamadaPontosManager(iface) 

        # Controla os ítens da etiqueta
        self.fieldColors = {}  # Armazenará as cores atribuídas aos campos
        self.fieldVisibility = {}  # Armazena a visibilidade dos campos

        # Mantém na memória a última posição  
        self.ultimo_caminho_salvo = ""  # Inicializa a variável para armazenar o último caminho

    def init_treeView(self):
        """
        Configura o QTreeView para listar e gerenciar camadas de pontos. 
        Este método inicializa a visualização da árvore com os itens e configurações necessárias,
        conecta os eventos de interface do usuário e estiliza os componentes visuais.
        
        Funções e Ações Desenvolvidas:
        - Atualização inicial da lista de camadas no QTreeView.
        - Conexão do evento de duplo clique em itens para tratamento.
        - Conexão do evento de alteração em itens para tratamento.
        - Configuração de delegado para customização da apresentação de itens.
        - Configuração do menu de contexto para interações adicionais.
        - Aplicação de estilos CSS para melhor visualização dos itens.
        - Conexão do botão de exportação para ação de exportar dados.
        """
        # Atualiza a visualização da lista de camadas no QTreeView
        self.atualizar_treeView_lista_ponto()

        # Conecta o evento de duplo clique em um item para manipulação de cores da camada
        self.dlg.treeViewListaPonto.doubleClicked.connect(self.on_item_double_clicked)

        # Conecta o evento de mudança em um item para atualizar a visibilidade da camada
        self.treeViewModel.itemChanged.connect(self.on_item_changed)

        # Define e aplica um delegado personalizado para customização da exibição de itens no QTreeView
        self.treeViewDelegate = CircleDelegate(self.dlg.treeViewListaPonto)
        self.dlg.treeViewListaPonto.setItemDelegate(self.treeViewDelegate)

        # Configura a política de menu de contexto para permitir menus personalizados em cliques com o botão direito
        self.dlg.treeViewListaPonto.setContextMenuPolicy(Qt.CustomContextMenu)
        self.dlg.treeViewListaPonto.customContextMenuRequested.connect(self.open_context_menu)

        # Aplica estilos CSS para aprimorar a interação visual com os itens do QTreeView
        self.dlg.treeViewListaPonto.setStyleSheet("""
            QTreeView::item:hover:!selected {
                background-color: #def2fc;
            }
            QTreeView::item:selected {
            }""")

        # Conecta o botão para criar uma camada de pontos ao método que adiciona a camada e atualiza o treeView
        self.dlg.ButtonCriarPonto.clicked.connect(self.adicionar_camada_e_atualizar)

        # Conecta o botão ao novo método do botão cria uma camada como o nome
        self.dlg.ButtonCriarPontoNome.clicked.connect(self.abrir_caixa_nome_camada)

        # Conecta o botão ao método de exportação do KML
        self.dlg.pushButtonExportaKmlT.clicked.connect(self.exportar_para_kml)
        
        # Conecta o botão ao método de exportação do DXF
        self.dlg.pushButtonExportaDXFT.clicked.connect(self.exportar_para_dxf)

        # Conecta o botão para reprojetar a camada
        self.dlg.pushButtonReprojetarT.clicked.connect(self.abrir_dialogo_crs)

        # seleção via mouse (funciona mesmo com 1 Camada de Pontos)
        self.dlg.treeViewListaPonto.clicked.connect(lambda _: self.on_treeview_selection_changed(None, None))

    def connect_signals(self):
        """
        Conecta os sinais do QGIS e do QTreeView para sincronizar a interface com o estado atual do projeto.
        Este método garante que mudanças no ambiente do QGIS se reflitam na interface do usuário e que ações na
        interface desencadeiem reações apropriadas no QGIS.

        Funções e Ações Desenvolvidas:
        - Conexão com sinais de adição e remoção de camadas para atualizar a visualização da árvore.
        - Sincronização do modelo do QTreeView com mudanças de seleção e propriedades das camadas no QGIS.
        - Tratamento da mudança de nome das camadas para manter consistência entre a interface e o estado interno.
        """
        # Conecta sinais do QGIS para lidar com a adição e remoção de camadas no projeto
        QgsProject.instance().layersAdded.connect(self.layers_added)

        # Conecta o evento de mudança em um item do QTreeView para atualizar a visibilidade da camada no QGIS
        self.treeViewModel.itemChanged.connect(self.on_item_changed)

        # Sincroniza o estado das camadas no QGIS com o checkbox do QTreeView sempre que as camadas do mapa mudam
        self.iface.mapCanvas().layersChanged.connect(self.sync_from_qgis_to_treeview)

        # Conecta mudanças na seleção do QTreeView para atualizar a camada ativa no QGIS
        self.dlg.treeViewListaPonto.selectionModel().selectionChanged.connect(self.on_treeview_selection_changed)

        # Sincroniza a seleção no QGIS com a seleção no QTreeView quando a camada ativa no QGIS muda
        self.iface.currentLayerChanged.connect(self.on_current_layer_changed)

        # Inicia a conexão de sinais para tratar a mudança de nome das camadas no projeto
        self.connect_name_changed_signals()

        # Conectando o botão pushButtonFecharT à função que fecha o diálogo
        self.dlg.pushButtonFecharT.clicked.connect(self.close_dialog)

        # Remove outras camadas sem alterar ações de camadas de outros tipos
        QgsProject.instance().layerWillBeRemoved.connect(self._on_layer_will_be_removed)

        #Diálogo que permite ao usuário selecionar campos da camada atual que serão
        self.dlg.pushButtonCampoT.clicked.connect(self.abrir_dialogo_selecao_campos_ponto)

        # Conecta o botão para Zoom
        self.dlg.pushButtonVisualizarT.clicked.connect(self.visualizar_ponto_selecionado)

        # Conectar o botão de renomear à função de renomear a camada
        self.dlg.pushButtonRenomeT.clicked.connect(self.renomear_camada_selecionada)

        # Conectar o botão de deletar à função de remover a camada
        self.dlg.pushButtonDelT.clicked.connect(self.remover_camada_selecionada)

        # Conectar o botão 'salvar como' à função 'salvar_camada_multiplo' para salvar em múltiplos formatos
        self.dlg.pushButtonSalvaMultiplosT.clicked.connect(self.salvar_camada_multiplo)

        # Conectar o botão de salvar permanente à função de salvar a camada
        self.dlg.pushButton_PermanenteT.clicked.connect(self.salvar_camada_permanente)

        # Conectar o botão 'pushButtonTabelaT' à função de abrir a tabela de atributos
        self.dlg.pushButtonTabelaT.clicked.connect(self.abrir_tabela_atributos)

        # Conecta o botão para Clocar a Camada de Polígono
        self.dlg.pushButtonClonarPonto.clicked.connect(self.clone_layer_ponto)

        # Conectando o botão à função pushButtonAbrirT
        self.dlg.pushButtonAbrirT.clicked.connect(self.abrir_adicionar_arquivo_ponto)

    def atualizar_estado_botoes(self):
        """
        Atualiza o estado (habilitado/desabilitado) dos botões da interface gráfica conforme a presença de camadas no treeView de pontos.

        Detalhamento do Processo:
        1. Verifica se o modelo associado ao treeViewListaPonto está vazio, ou seja, sem nenhuma camada listada.
        2. Se o modelo estiver vazio, todos os botões que dependem de camada devem ser desabilitados para evitar operações inválidas.
        3. Caso contrário, os botões são habilitados, permitindo ao usuário interagir normalmente com as funções associadas a uma camada de ponto.
        4. Os botões afetados incluem:
            - Deletar camada (`pushButtonDelT`)
            - Renomear camada (`pushButtonRenomeT`)
            - Tornar camada permanente (`pushButton_PermanenteT`)
            - Salvar em múltiplos formatos (`pushButtonSalvaMultiplosT`)
            - Abrir tabela de atributos (`pushButtonTabelaT`)
            - Gerenciar campos/etiquetas (`pushButtonCampoT`)
            - Exportar para DXF (`pushButtonExportaDXFT`)
            - Exportar para KML (`pushButtonExportaKmlT`)
            - Visualizar pontos associados (`pushButtonVisualizarT`)
            - Clonar pontos (`pushButtonClonarT`)
            - Reprojetar camada (`pushButtonReprojetarT`)
        5. Garante uma experiência segura e intuitiva, evitando erros por ações em contexto inválido.
        """
        # Verifica se o modelo do treeView está vazio
        modelo_vazio = self.dlg.treeViewListaPonto.model().rowCount() == 0

        # Atualiza o estado dos botões baseado na presença ou ausência de itens no modelo
        self.dlg.pushButtonDelT.setEnabled(not modelo_vazio)
        self.dlg.pushButtonRenomeT.setEnabled(not modelo_vazio)
        self.dlg.pushButton_PermanenteT.setEnabled(not modelo_vazio)
        self.dlg.pushButtonSalvaMultiplosT.setEnabled(not modelo_vazio)
        self.dlg.pushButtonTabelaT.setEnabled(not modelo_vazio)
        self.dlg.pushButtonCampoT.setEnabled(not modelo_vazio)
        self.dlg.pushButtonExportaDXFT.setEnabled(not modelo_vazio)
        self.dlg.pushButtonExportaKmlT.setEnabled(not modelo_vazio)
        self.dlg.pushButtonVisualizarT.setEnabled(not modelo_vazio)
        self.dlg.pushButtonClonarPonto.setEnabled(not modelo_vazio)
        self.dlg.pushButtonReprojetarT.setEnabled(not modelo_vazio)

    def abrir_tabela_atributos(self):
        """
        Abre a tabela de atributos para a camada de pontos selecionada no treeView, permitindo a visualização e edição dos atributos da camada no QGIS.

        Funcionalidades:
        - Verifica se existe alguma camada de polígono selecionada no treeView.
        - Se uma camada estiver selecionada, obtém o ID dessa camada a partir do índice selecionado.
        - Usa o ID para obter a camada correspondente do projeto QGIS.
        - Se a camada for válida, invoca o método para mostrar a tabela de atributos da camada na interface do usuário do QGIS.
        """
        # Acesse o treeView específico de polígonos!
        selected_indexes = self.dlg.treeViewListaPonto.selectedIndexes()
        if selected_indexes:
            selected_index = selected_indexes[0]  # Assume o primeiro item selecionado (se houver)
            # Tenta pegar o ID usando UserRole, se não achar, pega o padrão
            layer_id = selected_index.model().itemFromIndex(selected_index).data(Qt.UserRole)
            if layer_id is None:
                layer_id = selected_index.model().itemFromIndex(selected_index).data()
            layer = QgsProject.instance().mapLayer(layer_id)
            if layer:
                self.iface.showAttributeTable(layer)

    def salvar_camada_multiplo(self):
        """
        Permite ao usuário salvar uma camada selecionada em múltiplos formatos de arquivo simultaneamente, utilizando um diálogo de seleção.

        Funcionalidades:
        - Verifica se alguma camada está selecionada na árvore de visualização de camadas.
        - Extrai o identificador (ID) da camada selecionada para localizar a camada no projeto QGIS.
        - Apresenta um diálogo para que o usuário escolha quais formatos de arquivo deseja salvar a camada (DXF, KML, GeoJSON, etc.).
        - Permite ao usuário selecionar um diretório onde os arquivos serão salvos.
        - Salva a camada nos formatos escolhidos no diretório especificado, aplicando as configurações necessárias para cada formato.
        - Retorna um status de sucesso ou falha com base na ação do usuário (selecionar formatos, diretório, ou cancelar operação).
        """
        # Obtém os índices das linhas selecionadas no treeView de polígonos
        selected_indexes = self.dlg.treeViewListaPonto.selectedIndexes()
        if selected_indexes:
            selected_index = selected_indexes[0]  # Obtém o índice da primeira linha selecionada
            
            # Obtém o ID da camada selecionada
            layer_id = selected_index.model().itemFromIndex(selected_index).data(Qt.UserRole)
            
            # Busca a camada no projeto QGIS pelo ID
            layer_to_save = QgsProject.instance().mapLayer(layer_id)
            if layer_to_save:
                # Dicionário de formatos disponíveis e suas extensões
                formatos = {
                    "DXF": ".dxf",
                    "KML": ".kml",
                    "GeoJSON": ".geojson",
                    "CSV": ".csv",
                    "Shapefile": ".shp",
                    "TXT": ".txt",
                    "Excel": ".xlsx",
                    "Geopackage": ".gpkg"}

                # Mostra o diálogo para escolha dos formatos
                dialogo = DialogoSalvarFormatos(formatos, self.dlg)
                resultado = dialogo.exec_()

                # Se o usuário confirmou seleção dos formatos
                if resultado:
                    # Solicita o diretório de destino
                    diretorio = QFileDialog.getExistingDirectory(self.dlg, "Escolha um diretório para salvar os arquivos")
                    if diretorio:
                        for extensao in dialogo.formatos_selecionados:
                            # Monta o caminho completo para cada arquivo
                            nome_arquivo = os.path.join(diretorio, layer_to_save.name() + extensao)
                            # Exporta no formato correspondente
                            self.salvar_no_formato_especifico(layer_to_save, extensao, nome_arquivo)
                        return True  # Sucesso
                    else:
                        return False  # Usuário cancelou escolha do diretório
                else:
                    return False  # Usuário cancelou o diálogo de formatos
            else:
                return False  # Camada não localizada
        return False  # Nenhuma seleção

    def salvar_camada(self, layer, fileName, driverName):
        """
        Salva uma camada especificada para um arquivo externo usando o formato especificado pelo driverName.

        Parâmetros:
        - layer (QgsVectorLayer): A camada do QGIS que será salva.
        - fileName (str): O caminho completo do arquivo onde a camada será salva.
        - driverName (str): O nome do driver que define o formato de arquivo (por exemplo, "DXF", "KML").

        Funcionalidades:
        - Configura as opções de salvamento adequadas para o formato especificado, como codificação de arquivo e tratamento de simbologia.
        - Aplica configurações específicas para formatos como KML e DXF, adaptando o processo de salvamento às particularidades desses formatos.
        - Executa a operação de salvamento e verifica o resultado. Se bem-sucedido, exibe uma mensagem de sucesso. Caso contrário, exibe uma mensagem de erro detalhando o problema.
        - Para KML, realiza modificações adicionais no arquivo salvo para garantir a adequação às necessidades específicas (como tessellation).
        """
        options = QgsVectorFileWriter.SaveVectorOptions() # Configurações de salvamento da camada
        options.driverName = driverName # Define o driver de acordo com o formato desejado
        options.fileEncoding = "UTF-8" # Define a codificação do arquivo como UTF-8

        # Configurações específicas para KML
        if driverName == "KML":
            options.symbologyExport = QgsVectorFileWriter.FeatureSymbology

        # Configurações específicas para DXF
        if driverName == "DXF":
            options.skipAttributeCreation = True # Pula a criação de atributos para DXF

        # Tenta salvar a camada com as opções configuradas
        error, errorMessage = QgsVectorFileWriter.writeAsVectorFormat(layer, fileName, options)

        # Verifica se a camada foi salva com sucesso
        if error == QgsVectorFileWriter.NoError:
            self.mostrar_mensagem(f"Camada salva como {driverName} com sucesso.", "Sucesso")
            # Modificações adicionais para KML, se necessário
            if driverName == "KML":
                self.modificar_kml_para_tessellation(fileName)
            return True
        else:
            # Exibe uma mensagem de erro com a descrição do problema
            self.mostrar_mensagem(f"Erro ao salvar a camada como {driverName}: {errorMessage}", "Erro")
            return False

    def salvar_camada_como(self):
        """
        Permite ao usuário salvar uma camada selecionada em um formato de arquivo específico através de uma caixa de diálogo interativa.

        Funcionalidades:
        - Obtém o índice da camada selecionada no treeView. Se uma camada estiver selecionada, recupera o ID dessa camada.
        - Usa o ID para obter a camada correspondente do projeto QGIS.
        - Apresenta uma caixa de diálogo que permite ao usuário escolher entre vários formatos de arquivo para salvar a camada, como DXF, KML, Shapefile, GeoJSON, CSV, TXT, Excel e Geopackage.
        - Se o usuário confirmar a escolha de um formato, tenta salvar a camada no formato escolhido usando a função `salvar_no_formato_especifico`, que gerencia o processo de salvamento baseado no formato.
        - Retorna True se o arquivo for salvo com sucesso, False se o usuário cancelar a caixa de diálogo ou se ocorrer um erro no salvamento.
        """
        selected_indexes = self.dlg.treeViewListaPonto.selectedIndexes()
        if selected_indexes:
            selected_index = selected_indexes[0]
            layer_id = selected_index.model().itemFromIndex(selected_index).data(Qt.UserRole)
            layer_to_save = QgsProject.instance().mapLayer(layer_id)
            if layer_to_save:
                # Lista de formatos disponíveis
                formatos = {"DXF": ".dxf", "KML": ".kml", "Shapefile": ".shp", "GeoJSON": ".geojson",
                    "CSV": ".csv", "TXT": ".txt", "Excel": ".xlsx", "Geopackage": ".gpkg"}
                formato, ok = QInputDialog.getItem(self.dlg, "Salvar Como", "Escolha o formato de arquivo:", formatos.keys(), 0, False)
                if ok and formato:
                    return self.salvar_no_formato_especifico(layer_to_save, formatos[formato])
                else:
                    return False  # Usuário cancelou o diálogo
            else:
                return False
        return False

    def salvar_como_txt(self, layer, fileName):
        """
        Salva a camada especificada em um arquivo de texto (.txt) delimitado por tabulações, incluindo todos os campos e feições da camada.

        Parâmetros:
        - layer (QgsVectorLayer): A camada do QGIS que será salva no arquivo.
        - fileName (str): O caminho completo do arquivo onde a camada será salva.

        Funcionalidades:
        - Abre o arquivo especificado em modo de escrita, configurando para usar a codificação UTF-8 e para não adicionar novas linhas automaticamente.
        - Utiliza o módulo csv para criar um escritor que delimita os dados com tabulações.
        - Obtém os campos da camada e escreve os nomes dos campos como cabeçalho do arquivo.
        - Itera sobre cada feição da camada, escrevendo seus atributos no arquivo.
        - Exibe uma mensagem de sucesso na interface do usuário se o arquivo for salvo com sucesso.
        - Em caso de erro durante o salvamento, captura a exceção, exibe uma mensagem de erro na interface do usuário e retorna False.

        Atribuição no código:
        Proporciona uma maneira eficiente e direta de exportar dados geoespaciais para um formato .txt
        """
        try:
            # Abre o arquivo para escrita, configurando para não adicionar novas linhas e usando UTF-8
            with open(fileName, 'w', newline='', encoding='utf-8') as txtfile:
                writer = csv.writer(txtfile, delimiter='\t')  # Cria um objeto writer para csv delimitado por tabulações
                fields = layer.fields()  # Obtém os campos da camada
                writer.writerow([field.name() for field in fields])  # Escreve o cabeçalho com os nomes dos campos
                
                # Itera sobre cada feição da camada
                for feature in layer.getFeatures():
                    writer.writerow(feature.attributes())  # Escreve os atributos de cada feição

            # Exibe uma mensagem de sucesso se não houver exceções
            self.mostrar_mensagem("Camada salva como TXT com sucesso.", "Sucesso")
            return True
        except Exception as e:
            # Captura qualquer exceção que ocorra durante o processo de escrita e exibe uma mensagem de erro
            self.mostrar_mensagem(f"Erro ao salvar a camada como TXT: {e}", "Erro")
            return False

    def salvar_como_xlsx(self, layer, fileName):
        """
        Salva os atributos de uma camada QGIS em um arquivo Excel (XLSX).
        
        Funções e Ações Desenvolvidas:
        - Cria um novo livro do Excel e seleciona a planilha ativa.
        - Escreve o nome de cada campo da camada como cabeçalho da planilha.
        - Itera sobre cada feição da camada e escreve seus atributos nas linhas subsequentes da planilha.
        - Salva o livro do Excel no caminho especificado.
        - Exibe uma mensagem de sucesso ou erro após a tentativa de salvar os dados.

        Args:
        - layer: A camada QGIS da qual os dados são extraídos.
        - fileName: O nome do arquivo para o qual os dados serão salvos (deve terminar em .xlsx).

        Returns:
        - True se os dados forem salvos com sucesso, False caso contrário.
        """
        try:
            # Cria um novo livro do Excel
            workbook = openpyxl.Workbook()
            sheet = workbook.active  # Seleciona a planilha ativa

            # Obtém os campos da camada e escreve como cabeçalho da planilha
            fields = layer.fields()
            sheet.append([field.name() for field in fields])  # Adiciona os nomes dos campos como cabeçalho

            # Itera sobre cada feição da camada
            for feature in layer.getFeatures():
                sheet.append(feature.attributes())  # Adiciona os atributos de cada feição na planilha

            # Salva o livro do Excel no caminho especificado
            workbook.save(fileName)

            # Exibe uma mensagem de sucesso
            self.mostrar_mensagem("Camada salva como XLSX com sucesso.", "Sucesso")
            return True
        except Exception as e:
            # Captura qualquer exceção que ocorra durante o processo e exibe uma mensagem de erro
            self.mostrar_mensagem(f"Erro ao salvar a camada como XLSX: {e}", "Erro")
            return False

    def salvar_no_formato_especifico(self, layer, extensao, nome_arquivo=None):
        """
        Salva uma camada QGIS em um formato de arquivo específico escolhido pelo usuário, com suporte para vários formatos comuns.
        
        Funções e Ações Desenvolvidas:
        - Determina o tipo de arquivo e o nome do driver baseado na extensão fornecida.
        - Oferece ao usuário a possibilidade de escolher um local para salvar o arquivo se o nome do arquivo não for fornecido.
        - Checa se o arquivo já existe para evitar sobreposições e renomeia o arquivo se necessário.
        - Invoca funções específicas de salvamento baseadas na extensão do arquivo.
        
        Args:
        - layer: A camada QGIS que será salva.
        - extensao: A extensão do arquivo que indica o formato desejado.
        - nome_arquivo: O caminho completo do arquivo para salvar. Se None, será solicitado ao usuário.

        Returns:
        - True se o arquivo for salvo com sucesso, False se o salvamento falhar ou for cancelado.
        """
        # Mapeia as extensões para os tipos de arquivo correspondentes e seus drivers
        tipos_de_arquivo = {
            ".dxf": ("DXF Files (*.dxf)", "DXF"),
            ".kml": ("KML Files (*.kml)", "KML"),
            ".shp": ("Shapefile Files (*.shp)", "ESRI Shapefile"),
            ".txt": ("Text Files (*.txt)", "TXT"),
            ".csv": ("CSV Files (*.csv)", "CSV"),
            ".geojson": ("GeoJSON Files (*.geojson)", "GeoJSON"),
            ".gpkg": ("Geopackage Files (*.gpkg)", "GPKG"),
            ".xlsx": ("Excel Files (*.xlsx)", "XLSX")}
            
        tipo_arquivo, driver_name = tipos_de_arquivo.get(extensao, ("", ""))

        if tipo_arquivo:
            if nome_arquivo is None:  # Se o nome do arquivo não foi fornecido, abra o diálogo para escolher o local para salvar
                nome_arquivo = self.escolher_local_para_salvar(
                    os.path.join(self.ultimo_caminho_salvo, layer.name() + extensao), tipo_arquivo)

            # Checagem e renomeação do arquivo, se necessário
            if nome_arquivo:
                base_nome_arquivo = os.path.splitext(nome_arquivo)[0]
                contador = 1
                while os.path.exists(nome_arquivo):  # Se o arquivo já existir
                    nome_arquivo = f"{base_nome_arquivo}_{contador}{extensao}"
                    contador += 1 
                    
                if nome_arquivo and not nome_arquivo.endswith(extensao):
                    nome_arquivo += extensao

            if nome_arquivo:  # Se o nome do arquivo foi fornecido ou escolhido, proceda com o salvamento
                if extensao == ".txt":
                    return self.salvar_como_txt(layer, nome_arquivo)
                elif extensao == ".xlsx":
                    return self.salvar_como_xlsx(layer, nome_arquivo)
                else:
                    return self.salvar_camada(layer, nome_arquivo, driver_name)
            else:
                return False  # O usuário cancelou o diálogo ou ocorreu um erro
        else:
            return False # Extensão não suportada

        # Atualiza o último caminho de salvamento usado
        self.ultimo_caminho_salvo = os.path.dirname(nome_arquivo)

    def remover_camada_selecionada(self):
        """
        Remove a camada selecionada no treeView após verificar o estado de edição e as mudanças pendentes, solicitando ao usuário que salve as alterações se necessário.

        Funcionalidades:
        - Obtém os índices selecionados no treeView. Se um índice estiver selecionado, identifica a camada associada ao item selecionado.
        - Verifica se a camada está atualmente em um estado editável com mudanças não salvas. Se estiver, apresenta uma caixa de diálogo perguntando se o usuário deseja salvar as mudanças antes de remover a camada.
        - Se o usuário optar por salvar, tenta salvar a camada usando o método `salvar_camada_como`. Se o usuário confirmar o salvamento e este for bem-sucedido, a camada é removida do projeto QGIS.
        - Se o usuário optar por não salvar, descarta as mudanças pendentes e desliga a edição antes de remover a camada do projeto.
        - Se o usuário cancelar a ação, a remoção é abortada.
        - Se a camada não estiver editável ou não tiver mudanças pendentes, remove a camada diretamente do projeto.
        - Após a remoção da camada, atualiza o treeView para refletir a alteração na lista de camadas.
        """
        # Obtém o(s) índice(s) selecionado(s) no treeView de polígonos
        selected_indexes = self.dlg.treeViewListaPonto.selectedIndexes()
        if selected_indexes:
            selected_index = selected_indexes[0]
            # Obtém o ID da camada pelo papel UserRole (seguro para QTreeView customizado)
            layer_id = selected_index.model().itemFromIndex(selected_index).data(Qt.UserRole)
            layer_to_remove = QgsProject.instance().mapLayer(layer_id)
            if layer_to_remove:
                # Se a camada estiver editável e modificada, pergunta ao usuário
                if layer_to_remove.isEditable() and layer_to_remove.isModified():
                    resposta = QMessageBox.question(
                        self.dlg, "Confirmar Remoção",
                        "Existem alterações não salvas, deseja salvar antes de Remover?",
                        QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel,
                        QMessageBox.Yes)

                    if resposta == QMessageBox.Yes:
                        if self.salvar_camada_como():  # Se o usuário salvou a camada com sucesso
                            if layer_to_remove.isEditable():
                                layer_to_remove.commitChanges()  # Fecha a edição e confirma alterações
                            QgsProject.instance().removeMapLayer(layer_id)
                            # Remove referência e força coleta de lixo
                            del layer_to_remove
                            gc.collect()
                            self.atualizar_treeView_lista_ponto()
                            self.iface.mapCanvas().refresh()
                        else:
                            return  # O usuário cancelou o salvamento
                    elif resposta == QMessageBox.No:
                        layer_to_remove.rollBack()  # Descarta as mudanças
                        self.iface.actionToggleEditing().trigger()  # Desliga a edição
                        QgsProject.instance().removeMapLayer(layer_id)
                        # Remove referência e força coleta de lixo
                        del layer_to_remove
                        gc.collect()
                        self.atualizar_treeView_lista_ponto()
                        self.iface.mapCanvas().refresh()
                    elif resposta == QMessageBox.Cancel:
                        return  # Ação cancelada pelo usuário
                else:
                    # Se a camada não estiver editável ou não tiver mudanças pendentes, remova diretamente
                    QgsProject.instance().removeMapLayer(layer_id)
                    # Remove referência e força coleta de lixo
                    del layer_to_remove
                    gc.collect()
                    self.atualizar_treeView_lista_ponto()
                    self.iface.mapCanvas().refresh()

    def renomear_camada_selecionada(self):
        """
        Permite ao usuário renomear uma camada de Ponto selecionado no treeViewListaPonto,
        garantindo unicidade do nome no projeto.

        Fluxo:
        - Obtém o índice selecionado.
        - Recupera a camada associada ao item (usando Qt.UserRole).
        - Exibe um diálogo para o usuário digitar o novo nome.
        - Garante que o nome será único (ajusta se necessário).
        - Atualiza o nome da camada no QGIS e o texto do item no treeView.
        """
        # Obtém os índices atualmente selecionados no treeView de pontos
        selected_indexes = self.dlg.treeViewListaPonto.selectedIndexes()
        if selected_indexes:
            # Pega o primeiro índice selecionado
            selected_index = selected_indexes[0]
            # Recupera o ID da camada usando o papel UserRole (garante que é o ID real da camada QGIS)
            layer_id = selected_index.model().itemFromIndex(selected_index).data(Qt.UserRole)
            # Busca a camada correspondente no projeto QGIS
            selected_layer = QgsProject.instance().mapLayer(layer_id)
            if selected_layer:
                # Exibe um diálogo para o usuário digitar o novo nome (pré-preenchido com o nome atual)
                novo_nome, ok = QInputDialog.getText(
                    self.dlg,
                    "Renomear Camada",
                    "Digite o novo nome da camada:",
                    text=selected_layer.name())
                # Se o usuário confirmou e digitou algo
                if ok and novo_nome:
                    # Gera um nome único para evitar duplicidade no projeto
                    novo_nome = self.gerar_nome_unico(novo_nome, selected_layer.id())
                    # Renomeia a camada no QGIS (isso pode disparar sinais que atualizam a interface automaticamente)
                    selected_layer.setName(novo_nome)
                    # Em vez de tentar alterar diretamente o item do modelo (risco de crash), reconstrói o treeView do zero
                    self.atualizar_treeView_lista_ponto()

    def gerar_nome_unico(self, base_nome, current_layer_id):
        """
        Gera um nome único para uma camada de ponto dentro do projeto QGIS,
        assegurando que não haja conflitos de nomeação entre as camadas de pontos.

        Parâmetros:
        - base_nome (str): O nome base proposto para a camada.
        - current_layer_id (str): O ID da camada que está sendo renomeada, usado para excluir seu nome atual da verificação de duplicidade.

        Funcionalidades:
        - Cria um dicionário dos nomes de todas as camadas de pontos existentes no projeto QGIS, excluindo a camada que está sendo renomeada.
        - Verifica se o nome base já existe entre as camadas de pontos. Se não existir, retorna o nome base como válido.
        - Se o nome base já existir, gera variações do nome acrescentando um sufixo numérico (por exemplo, "Nome_1", "Nome_2") até encontrar um nome não utilizado.
        - Retorna o novo nome único gerado.
        """
        # Restringe apenas para camadas vetoriais de pontos
        existing_names = {layer.name(): layer.id()
            for layer in QgsProject.instance().mapLayers().values()
            if (layer.id() != current_layer_id and
                layer.type() == QgsMapLayer.VectorLayer and
                layer.geometryType() == QgsWkbTypes.PolygonGeometry)}
        if base_nome not in existing_names:
            return base_nome
        else:
            i = 1
            novo_nome = f"{base_nome}_{i}"
            while novo_nome in existing_names:
                i += 1
                novo_nome = f"{base_nome}_{i}"
            return novo_nome

    def visualizar_ponto_selecionado(self):
        """
        Aproxima a visualização do mapa para a camada de ponto selecionada no treeView.

        - Obtém a camada de ponto atualmente selecionada.
        - Centraliza e ajusta a extensão do mapa para mostrar toda a camada.
        - Se não houver feições ou seleção, não faz nada.
        """
        index = self.dlg.treeViewListaPonto.currentIndex()  # Obtém o índice selecionado
        if not index.isValid():
            return  # Não há seleção

        # Obtém o ID da camada do UserRole (melhor do que usar .data() sem argumentos)
        layer_id = index.model().itemFromIndex(index).data(Qt.UserRole)
        layer = QgsProject.instance().mapLayer(layer_id)
        if not layer or layer.extent().isEmpty():
            return  # Camada não encontrada ou sem feições

        # Garante que a camada esteja ativa (opcional, mas recomendado)
        self.iface.setActiveLayer(layer)
        # Ajusta o zoom para a extensão da camada ativa
        self.iface.zoomToActiveLayer()

    def configurar_tooltip(self, index):
        """
        Configura um tooltip para exibir informações sobre a camada de ponto selecionada no treeView.

        A função extrai informações sobre a camada de ponto, como o tipo de geometria (ex: Point, MultiPoint) 
        e o sistema de referência de coordenadas (SRC) atual da camada. Essas informações são exibidas em 
        um tooltip que aparece quando o usuário passa o mouse sobre o item correspondente no treeView.

        Parâmetros:
        - index: QModelIndex do item atualmente sob o cursor no treeView.
        """
        item = index.model().itemFromIndex(index)  # Obtém o item do modelo de dados com base no índice fornecido
        layer_id = item.data(Qt.UserRole)  # Obtém o ID da camada associada ao item
        layer = QgsProject.instance().mapLayer(layer_id)  # Recupera a camada correspondente ao ID no projeto QGIS
        if layer:  # Verifica se a camada existe
            tipo_ponto = self.obter_tipo_de_ponto(layer)  # Obtém o tipo de geometria da camada (ex: Point, MultiPoint)
            crs = layer.crs().description() if layer.crs().isValid() else "Sem Georreferenciamento"  # Obtém a descrição do SRC da camada ou "Sem Georreferenciamento" se inválido
            tooltip_text = f"Tipo: {tipo_ponto}\nSRC: {crs}"  # Formata o texto do tooltip com as informações da camada
            QToolTip.showText(QCursor.pos(), tooltip_text)  # Exibe o tooltip na posição atual do cursor

    def obter_tipo_de_ponto(self, layer):
        """
        Retorna uma string que descreve o tipo de geometria da camada fornecida.

        A função obtém o tipo de geometria WKB (Well-Known Binary) da camada e converte esse tipo
        em uma string legível, como 'Point', 'MultiPoint', etc.

        Parâmetros:
        - layer: Objeto QgsVectorLayer representando a camada de onde o tipo de ponto será extraído.

        Retorno:
        - tipo_ponto (str): Uma string que descreve o tipo de geometria da camada.
        """
        geometry_type = layer.wkbType()  # Obtém o tipo de geometria WKB (Well-Known Binary) da camada
        tipo_ponto = QgsWkbTypes.displayString(geometry_type)  # Converte o tipo de geometria em uma string legível
        return tipo_ponto  # Retorna a string que descreve o tipo de geometria

    def adicionar_camada_e_atualizar(self):
        """
        Método chamado ao clicar no botão para criar uma camada de ponto.
        Cria a camada de ponto e atualiza o treeView.
        """
        # Chamada para a função que cria uma nova camada de pontos
        self.pontos_manager.criar_camada_pontos() 

        # Após adicionar a camada, atualize o treeView
        self.atualizar_treeView_lista_ponto()

    def abrir_caixa_nome_camada(self):
        """
        Esta função cria uma caixa de diálogo que permite ao usuário inserir o nome de uma nova camada.
        A caixa de diálogo contém um campo de texto e dois botões: 'OK' e 'Cancelar'.
        O botão 'OK' é ativado somente quando o campo de texto não está vazio.
        Se o usuário clicar em 'OK', a função 'criar_camada_pontos' é chamada e a árvore de visualização é atualizada.
        """
        dialog = QDialog(self.dlg) # Cria uma nova caixa de diálogo
        dialog.setWindowTitle("Nome da Camada") # Define o título da caixa de diálogo
        layout = QVBoxLayout(dialog) # Define o layout da caixa de diálogo
        layout.addWidget(QLabel("Digite o nome da camada:")) # Adiciona um rótulo ao layout

        lineEdit = QLineEdit() # Cria um novo campo de texto
        lineEdit.setPlaceholderText("Camada Temporária") # Define o texto do espaço reservado para o campo de texto
        layout.addWidget(lineEdit) # Adiciona o campo de texto ao layout

        okButton = QPushButton("OK") # Cria botões OK e Cancelar
        cancelButton = QPushButton("Cancelar") # Cria um novo botão 'Cancelar'

        okButton.clicked.connect(dialog.accept) # Conecta o clique do botão 'OK' à função 'accept' da caixa de diálogo
        cancelButton.clicked.connect(dialog.reject) # Conecta o clique do botão 'Cancelar' à função 'reject' da caixa de diálogo

        okButton.setEnabled(False) # Desativa o botão 'OK' por padrão

        # Ativa o botão 'OK' quando o campo de texto não está vazio
        lineEdit.textChanged.connect(lambda: okButton.setEnabled(bool(lineEdit.text().strip())))

        buttonLayout = QHBoxLayout()  # Cria um novo layout horizontal para os botões
        buttonLayout.addWidget(okButton)  # Adiciona o botão 'OK' ao layout do botão
        buttonLayout.addWidget(cancelButton)  # Adiciona o botão 'Cancelar' ao layout do botão
        layout.addLayout(buttonLayout)  # Adiciona o layout do botão ao layout principal

        # Se a caixa de diálogo for aceita e o campo de texto não estiver vazio, cria uma nova camada e atualiza a árvore de visualização
        if dialog.exec_() == QDialog.Accepted and lineEdit.text().strip():
            nome_camada = lineEdit.text().strip()  # Obtém o nome da camada do campo de texto
            self.pontos_manager.criar_camada_pontos(nome_camada)  # Cria uma nova camada de pontos
            self.atualizar_treeView_lista_ponto()  # Atualiza a árvore de visualização

    def close_dialog(self):
        """
        Fecha o diálogo associado a este UiManagerT:
        """
        self.dlg.close()

    def _on_layer_will_be_removed(self, layer_id):
        """
        Remove do TreeView **apenas** se a camada que vai sair
        for vetor de pontos.
        """
        layer = QgsProject.instance().mapLayer(layer_id)
        if not layer:
            return  # segurança

        if layer.type() != QgsMapLayer.VectorLayer or layer.geometryType() != QgsWkbTypes.PointGeometry:
            # não é ponto → ignoramos
            return

        # Remove a linha correspondente no modelo
        for row in range(self.treeViewModel.rowCount()):
            item = self.treeViewModel.item(row)
            if item and item.data(Qt.UserRole) == layer_id:
                self.treeViewModel.removeRow(row)
                break

        # Garante que fique alguma seleção válida
        model = self.dlg.treeViewListaPonto.model()
        if model.rowCount() > 0 and not self.dlg.treeViewListaPonto.currentIndex().isValid():
            target_row = min(row, model.rowCount() - 1)
            idx = model.index(target_row, 0)
            self.dlg.treeViewListaPonto.setCurrentIndex(idx)
            self.dlg.treeViewListaPonto.scrollTo(idx)

        # Chama atualização dos botões após toda modificação
        self.atualizar_estado_botoes()

    def abrir_dialogo_crs(self):
        """
        Abre um diálogo de seleção de CRS e reprojeta a camada de pontos selecionada no treeViewListaPonto.

        A função permite ao usuário escolher um novo sistema de referência de coordenadas (SRC) para a camada 
        selecionada no treeViewListaPonto. Após a seleção, a camada é reprojetada usando o novo SRC, e a nova camada é 
        adicionada ao projeto QGIS com a mesma cor de ícone e rótulo da camada original.

        Parâmetros:
        - self: Referência à instância atual do objeto. (UiManagerT)

        A função não retorna valores, mas adiciona uma nova camada reprojetada ao projeto QGIS.
        """
        index = self.dlg.treeViewListaPonto.currentIndex()  # Obtém o índice atualmente selecionado no treeViewListaPonto
        if not index.isValid():  # Verifica se o índice é válido (se há uma seleção)
            return  # Sai da função se o índice não for válido
        
        layer_id = index.model().itemFromIndex(index).data(Qt.UserRole)  # Obtém o ID da camada associada ao item selecionado
        layer = QgsProject.instance().mapLayer(layer_id)  # Recupera a camada correspondente ao ID no projeto QGIS
        if not layer or layer.geometryType() != QgsWkbTypes.PointGeometry:  # Verifica se a camada existe e é de pontos
            self.mostrar_mensagem("Por favor, selecione uma camada de pontos válida.", "Aviso")  # Exibe uma mensagem de aviso
            return  # Sai da função se a camada não for de pontos
        
        # Abre o diálogo de seleção de CRS para que o usuário possa escolher um novo SRC
        dialog = QgsProjectionSelectionDialog(self.dlg)  # Cria uma instância do diálogo de seleção de CRS
        dialog.setCrs(layer.crs())  # Configura o CRS atual da camada como o CRS padrão no diálogo
        if dialog.exec_():  # Exibe o diálogo e verifica se o usuário confirmou a seleção
            novo_crs = dialog.crs()  # Obtém o novo CRS selecionado pelo usuário

            # Usa o processamento do QGIS para reprojetar a camada
            params = {
                'INPUT': layer,  # Define a camada original como entrada
                'TARGET_CRS': novo_crs,  # Define o novo CRS como o CRS alvo
                'OUTPUT': 'memory:'  # Salva o resultado na memória
            }
            resultado = processing.run('qgis:reprojectlayer', params)  # Executa o processo de reprojeção
            nova_camada = resultado['OUTPUT']  # Obtém a nova camada reprojetada a partir dos resultados

            # Verifique se a nova camada tem feições válidas
            if nova_camada and nova_camada.isValid():  # Verifica se a nova camada foi criada corretamente
                # Gerar nome único para a nova camada reprojetada
                novo_nome = self.gerar_nome_unico_rep(layer.name())  # Gera um nome único para evitar duplicidades
                nova_camada.setName(novo_nome)  # Define o nome da nova camada

                # Aplicar as cores do ícone e do rótulo da camada original
                cor_icone = self.obter_cor_icone(layer)  # Obtém a cor do ícone da camada original
                cor_rotulo = self.obter_cor_rotulo(layer)  # Obtém a cor do rótulo da camada original

                # self.aplicar_simbologia_com_cores_qgis(nova_camada, cor_icone, cor_rotulo)  # Aplica a simbologia

                QgsProject.instance().addMapLayer(nova_camada)  # Adiciona a nova camada reprojetada ao projeto

                # Atualiza a camada na tela
                layer.triggerRepaint()  # Recarrega a camada original (caso necessário)
                self.iface.mapCanvas().refresh()  # Atualiza a tela do mapa para refletir a nova camada

                # Exibe uma mensagem informando que a camada foi reprojetada
                texto_mensagem = f"A camada '{layer.name()}' foi reprojetada para {novo_crs.authid()} ({novo_crs.description()}) com o nome '{novo_nome}'."  # Cria a mensagem de sucesso
                self.mostrar_mensagem(texto_mensagem, "Sucesso")  # Exibe a mensagem na barra de mensagens do QGIS
            else:
                self.mostrar_mensagem(f"Erro na reprojeção da camada '{layer.name()}'.", "Erro")  # Exibe uma mensagem de erro se a reprojeção falhar

    def gerar_nome_unico_rep(self, base_nome):
        """
        Gera um nome único para uma nova camada baseada no nome da camada original.

        A função cria um novo nome para a camada reprojetada, adicionando um sufixo incremental 
        (_rep1, _rep2, etc.) ao nome base, garantindo que o novo nome seja único dentro do projeto QGIS.

        Parâmetros:
        - self: Referência à instância atual do objeto. (UiManager)
        - base_nome (str): O nome original da camada, usado como base para gerar o novo nome.

        Retorno:
        - novo_nome (str): Um nome único gerado para a nova camada reprojetada.
        """
        # Cria um conjunto contendo todos os nomes de camadas existentes no projeto QGIS
        existing_names = {layer.name() for layer in QgsProject.instance().mapLayers().values()}

        # Se o nome base ainda não existir no projeto, retorna com o sufixo _rep1
        if base_nome not in existing_names:
            return f"{base_nome}_rep1"
        # Inicia o contador de sufixos em 1
        i = 1
        # Gera o primeiro nome com o sufixo _rep1
        novo_nome = f"{base_nome}_rep{i}"

        # Incrementa o sufixo até encontrar um nome que não exista no projeto
        while novo_nome in existing_names:
            i += 1
            novo_nome = f"{base_nome}_rep{i}"
        # Retorna o nome único gerado
        return novo_nome

    def on_treeview_selection_changed(self, selected, deselected):
        """
        Define a camada ativa no QGIS com base na seleção do TreeView.

        Funcionalidades:
        - Acessa os índices selecionados no TreeView.
        - Recupera o ID da camada associado ao item selecionado.
        - Define a camada correspondente como ativa no QGIS (iface.setActiveLayer).
        """
        # Obtém os índices selecionados no TreeView
        indexes = self.dlg.treeViewListaPonto.selectionModel().selectedIndexes()
        if indexes:
            # Recupera o ID da camada a partir do primeiro item selecionado
            layer_id = self.treeViewModel.itemFromIndex(indexes[0]).data(Qt.UserRole)
            # Obtém a camada no projeto QGIS através do ID
            layer = QgsProject.instance().mapLayer(layer_id)
            if layer:
                # Define a camada como ativa no QGIS
                self.iface.setActiveLayer(layer)

    def connect_name_changed_signals(self):
        """
        Conecta o sinal de mudança de nome de todas as camadas de ponto existentes no projeto QGIS.
        Este método percorre todas as camadas listadas no QgsLayerTreeRoot e, para cada camada,
        conecta o evento de mudança de nome à função de callback on_layer_name_changed.

        Funções e Ações Desenvolvidas:
        - Busca e iteração por todas as camadas no projeto QGIS.
        - Conexão do sinal de mudança de nome da camada ao método correspondente para tratamento.
        """
        # Acessa a raiz da árvore de camadas do projeto QGIS
        root = QgsProject.instance().layerTreeRoot()
        # Itera por todos os nós de camadas na árvore de camadas
        for layerNode in root.findLayers():
            # Verifica se o nó é uma instância de QgsLayerTreeLayer
            if isinstance(layerNode, QgsLayerTreeLayer):
                # Conecta o sinal de mudança de nome da camada ao método de tratamento on_layer_name_changed
                layerNode.layer().nameChanged.connect(self.on_layer_name_changed)

    def layers_added(self, layers):
        """
        Responde ao evento de adição de camadas no projeto QGIS, atualizando a lista de camadas no QTreeView
        e conectando sinais de mudança de nome para camadas de pontos recém-adicionadas.

        Este método verifica cada camada adicionada para determinar se é uma camada de vetor de pontos.
        Se for, ele atualiza a lista de camadas no QTreeView e conecta o sinal de mudança de nome à função
        de callback apropriada.

        :param layers: Lista de camadas recém-adicionadas ao projeto.

        Funções e Ações Desenvolvidas:
        - Verificação do tipo e da geometria das camadas adicionadas.
        - Atualização da visualização da lista de camadas no QTreeView para incluir novas camadas de pontos.
        - Conexão do sinal de mudança de nome da camada ao método de tratamento correspondente.
        """
        # Itera por todas as camadas adicionadas
        for layer in layers:
            # Verifica se a camada é do tipo vetor e se sua geometria é de ponto
            if layer.type() == QgsMapLayer.VectorLayer and layer.geometryType() == QgsWkbTypes.PointGeometry:
                # Atualiza a lista de camadas no QTreeView
                self.atualizar_treeView_lista_ponto()
                # Conecta o sinal de mudança de nome da nova camada ao método on_layer_name_changed
                layer.nameChanged.connect(self.on_layer_name_changed)
                # Interrompe o loop após adicionar o sinal à primeira camada de ponto encontrada
                break

    def on_layer_name_changed(self):
        """
        Responde ao evento de mudança de nome de qualquer camada no projeto QGIS. 
        Este método é chamado automaticamente quando o nome de uma camada é alterado,
        e sua função é garantir que a lista de camadas no QTreeView seja atualizada para refletir
        essa mudança.

        Funções e Ações Desenvolvidas:
        - Atualização da lista de camadas no QTreeView para assegurar que os nomes das camadas estejam corretos.
        """
        # Atualiza a lista de camadas no QTreeView para refletir a mudança de nome
        self.atualizar_treeView_lista_ponto()

    def abrir_dialogo_selecao_campos_ponto(self):
        """
        Abre o diálogo de gerenciamento de etiquetas para a camada de pontos selecionada em treeViewListaPonto.

        Detalhamento do Processo:
        1. Coleta os índices atualmente selecionados no QTreeView de pontos.
        2. Se não houver seleção, retorna imediatamente (nenhuma ação).
        3. Garante que usamos sempre a coluna 0 do modelo, onde está armazenado o ID da camada.
        4. Lê o ID da camada selecionada através de Qt.UserRole (não confundir com o texto do item).
        5. Recupera a camada no projeto QGIS usando este ID.
        6. Verifica se a camada existe e se sua geometria é do tipo ponto; caso contrário, aborta.
        7. Instancia o diálogo GerenciarEtiquetasDialog, passando:
           - a camada de pontos,
           - os dicionários de cores e visibilidade (stateful),
           - a interface QGIS (iface),
           - e o próprio diálogo principal como parent.
        8. Exibe o diálogo de forma modal com `exec_()`, permitindo ao usuário selecionar campos e cores.
        9. Ao fechar, quaisquer alterações de etiquetas ficam aplicadas na camada.

        Esta função facilita a configuração de etiquetas para pontos, reutilizando a mesma UI
        originalmente criada para pontos/linhas, mas validando geometria de ponto.
        """
        # 1) Obter todos os índices selecionados no treeView de pontos
        selected_indexes = self.dlg.treeViewListaPonto.selectedIndexes()
        # 2) Se não há seleção, interrompe aqui
        if not selected_indexes:
            return

        # 3) Garante uso da coluna zero onde o ID está armazenado
        idx = selected_indexes[0].sibling(selected_indexes[0].row(), 0)

        # 4) Extrai o ID da camada a partir do papel Qt.UserRole
        layer_id = idx.model().itemFromIndex(idx).data(Qt.UserRole)

        # 5) Busca a camada no projeto QGIS pelo ID
        layer = QgsProject.instance().mapLayer(layer_id)
        # 6) Verifica existência e tipo de geometria (ponto)
        if not layer or layer.geometryType() != QgsWkbTypes.PointGeometry:
            return

        # 7) Cria o diálogo de gerenciamento de etiquetas
        dialog = GerenciarEtiquetasDialog(
            layer,
            self.fieldColors,       # dicionário que mapeia campos→cores
            self.fieldVisibility,   # dicionário que mapeia campos→visibilidade
            self.iface,             # interface do QGIS
            self.dlg)                # diálogo principal como parent

        # 8) Exibe o diálogo de forma modal, bloqueando até o usuário fechar
        dialog.exec_()

    def adjust_item_font(self, item, layer):
        """
        Ajusta a fonte do item no QTreeView com base no tipo de fonte de dados da camada associada.
        Este método modifica a fonte para itálico se a camada for temporária (em memória) e para negrito se for permanente.

        A visualização visual das camadas como itálicas ou negritas ajuda o usuário a identificar rapidamente
        o tipo de camada (temporária vs. permanente) diretamente pela interface do usuário.

        :param item: O item no QTreeView que representa uma camada no QGIS.
        :param layer: A camada do QGIS associada ao item no QTreeView.

        :return: Retorna o item com a fonte ajustada.

        Funções e Ações Desenvolvidas:
        - Configuração da fonte para itálico se a fonte de dados da camada for 'memory' (temporária).
        - Configuração da fonte para negrito se a fonte de dados da camada for permanente.
        """
        # Cria um objeto QFont para ajustar a fonte do item
        fonte_item = QFont()

        # Verifica se a camada é temporária (dados em memória) e ajusta a fonte para itálico
        if layer.dataProvider().name() == 'memory':
            fonte_item.setItalic(True)
        # Se não for temporária, ajusta a fonte para negrito, indicando uma camada permanente
        else:
            fonte_item.setBold(True)

        # Aplica a fonte ajustada ao item no QTreeView
        item.setFont(fonte_item)

        # Retorna o item com a fonte ajustada para uso posterior se necessário
        return item

    def atualizar_treeView_lista_ponto(self):
        """
        Atualiza o TreeView que lista as camadas de pontos no QGIS.

        Funcionalidades:
        1) Salva o ID da camada atualmente selecionada no TreeView, se houver.
        2) Limpa e reconstrói o modelo do TreeView:
           - Define o cabeçalho centralizado.
           - Itera sobre todas as camadas do projeto e adiciona ao modelo apenas as camadas vetoriais de pontos.
           - Para cada camada encontrada:
             * Cria um item com o nome da camada.
             * Torna o item checkable para controlar visibilidade.
             * Define flags para impedir edição direta do texto.
             * Armazena o ID único da camada no Qt.UserRole.
             * Ajusta o estado do checkbox de acordo com a visibilidade atual da camada no projeto.
             * Formata a fonte do item (itálico para memória, negrito para outras).
        3) Tenta restaurar a seleção anterior, buscando no modelo o item cujo ID corresponda ao salvo.
        4) Caso não haja seleção anterior ou a camada não exista mais, seleciona o primeiro item do modelo.
        5) Move o scroll para garantir que o item selecionado esteja visível.
        """
        # 1) Salva o ID da camada atualmente selecionada, se houver
        current_layer_id = None  # Armazena o layer.id() da seleção atual
        current_index = self.dlg.treeViewListaPonto.currentIndex()  # Índice selecionado no TreeView
        if current_index.isValid():  # Verifica se há uma seleção válida
            # Extrai o ID da camada do dado armazenado em Qt.UserRole
            current_layer_id = current_index.model().itemFromIndex(current_index).data(Qt.UserRole)

        # 2) Reconstrói o modelo do TreeView
        self.treeViewModel.clear()  # Limpa todos os itens existentes no modelo

        # Configura o cabeçalho da coluna com alinhamento central
        headerItem = QStandardItem("Lista de Camadas de Pontos")  # Título do cabeçalho
        headerItem.setTextAlignment(Qt.AlignCenter)  # Centraliza o texto
        self.treeViewModel.setHorizontalHeaderItem(0, headerItem)  # Aplica o cabeçalho

        # Pega a raiz da árvore de camadas do QGIS para verificar visibilidade
        root = QgsProject.instance().layerTreeRoot()
        # Itera por todas as camadas do projeto
        for layer in QgsProject.instance().mapLayers().values():
            # Seleciona apenas camadas vetoriais do tipo polígono
            if (
                layer.type() == QgsMapLayer.VectorLayer
                and layer.geometryType() == QgsWkbTypes.PointGeometry):
                item = QStandardItem(layer.name())  # Cria o item com o nome da camada
                item.setCheckable(True)  # Permite marcar/desmarcar para visibilidade
                # Remove flag de edição para impedir alteração do nome pelo usuário
                item.setFlags(item.flags() & ~Qt.ItemIsEditable)
                # Armazena o ID da camada no item para referência futura
                item.setData(layer.id(), Qt.UserRole)

                # Ajusta o estado inicial do checkbox baseado na visibilidade da camada
                layerNode = root.findLayer(layer.id())  # Nó de camada na árvore do QGIS
                is_visible = layerNode and layerNode.isVisible()  # True se visível
                item.setCheckState(
                    Qt.Checked if is_visible else Qt.Unchecked)  # Marca ou desmarca

                # Aplica formatação de fonte: itálico para memória, negrito para outras
                self.adjust_item_font(item, layer)
                # Adiciona o item ao modelo, criando uma nova linha
                self.treeViewModel.appendRow(item)

        # 3) Restaura a seleção anterior, se possível
        model = self.dlg.treeViewListaPonto.model()  # Modelo atualizado
        target_index = None  # Índice que será selecionado

        if current_layer_id:
            # Busca pelo item que tenha o mesmo ID salvo anteriormente
            for row in range(model.rowCount()):
                item = model.item(row)
                if item and item.data(Qt.UserRole) == current_layer_id:
                    target_index = model.indexFromItem(item)
                    break  # Interrompe ao encontrar a correspondência

        # 4) Se não houver seleção anterior (ou não encontrada), seleciona o primeiro item
        if target_index is None and model.rowCount() > 0:
            target_index = model.index(0, 0)  # Pega a primeira linha do modelo

        # 5) Aplica a seleção e move o scroll para exibir o item
        if target_index is not None:
            self.dlg.treeViewListaPonto.setCurrentIndex(target_index)  # Seleciona
            self.dlg.treeViewListaPonto.scrollTo(target_index)  # Move o scroll

        # Chama atualização dos botões após toda modificação
        self.atualizar_estado_botoes()

    def on_current_layer_changed(self, layer):
        """
        Atualiza a seleção do treeView quando a camada ativa do QGIS muda.

        Funcionalidades:
        - Verifica se a nova camada ativa é uma camada vetorial de pontos.
        - Percorre o modelo do treeView para encontrar o item correspondente ao ID da nova camada.
        - Se encontrar, seleciona e rola até o item correspondente no treeView.

        Parâmetro:
        - layer (QgsMapLayer): Nova camada ativa no QGIS.
        """
        # Obtém o modelo associado ao treeView de pontos
        model = self.dlg.treeViewListaPonto.model()

        # Verifica se a camada é válida, do tipo vetorial e com geometria de polígono
        if layer and layer.type() == QgsMapLayer.VectorLayer and layer.geometryType() == QgsWkbTypes.PointGeometry:
            layer_id = layer.id()  # Obtém o ID único da camada

            # Percorre as linhas do modelo do treeView
            for row in range(model.rowCount()):
                item = model.item(row, 0)

                # Verifica se o item corresponde ao ID da camada ativa
                if item and item.data(Qt.UserRole) == layer_id:
                    index = model.indexFromItem(item)

                    # Define a seleção atual e rola até a linha correspondente
                    self.dlg.treeViewListaPonto.setCurrentIndex(index)
                    self.dlg.treeViewListaPonto.scrollTo(index)
                    break  # Encerra após encontrar a camada correspondente

    def on_layer_was_renamed(self, layerId, newName):
        """
        Responde ao evento de renomeação de uma camada no QGIS, atualizando o nome da camada no QTreeView.
        Este método garante que as mudanças de nome no projeto sejam refletidas na interface do usuário.

        :param layerId: ID da camada que foi renomeada.
        :param newName: Novo nome atribuído à camada.

        Funções e Ações Desenvolvidas:
        - Pesquisa no modelo do QTreeView pelo item que corresponde à camada renomeada.
        - Atualização do texto do item no QTreeView para refletir o novo nome.
        """
        # Itera sobre todos os itens no modelo do QTreeView para encontrar o item correspondente
        for i in range(self.treeViewModel.rowCount()):
            item = self.treeViewModel.item(i)
            layer = QgsProject.instance().mapLayer(layerId)
            # Verifica se o item corresponde à camada que foi renomeada
            if layer and item.text() == layer.name():
                item.setText(newName)  # Atualiza o nome do item no QTreeView
                break  # Sai do loop após atualizar o nome

    def on_item_changed(self, item):
        """
        Atualiza a visibilidade da camada no QGIS quando o checkbox correspondente
        no treeView é marcado ou desmarcado.

        Funcionalidades:
        - Recupera o ID da camada associada ao item alterado no treeView.
        - Obtém a camada correspondente no projeto.
        - Atualiza a visibilidade no painel de camadas do QGIS conforme o estado do checkbox.

        Parâmetro:
        - item (QStandardItem): Item do treeView que foi modificado (normalmente um checkbox marcado/desmarcado).
        """
        # Obtém o ID da camada armazenado no item do treeView
        layer_id = item.data(Qt.UserRole)

        # Recupera a camada correspondente no projeto QGIS
        layer = QgsProject.instance().mapLayer(layer_id)
        if not layer:
            return  # Se a camada não existir mais, sai da função

        # Acessa a estrutura da árvore de camadas (Layer Tree)
        root = QgsProject.instance().layerTreeRoot()

        # Procura o nó da camada dentro da árvore
        node = root.findLayer(layer_id)
        if node:
            # Atualiza a visibilidade da camada com base no estado do checkbox (checked ou unchecked)
            node.setItemVisibilityChecked(item.checkState() == Qt.Checked)

    def sync_from_qgis_to_treeview(self):
        """
        Sincroniza o estado de visibilidade das camadas no treeView com o estado real
        das camadas no painel de camadas do QGIS.

        Funcionalidades:
        - Percorre todos os itens do modelo do treeView.
        - Obtém o ID da camada associado a cada item.
        - Busca o nó correspondente na árvore de camadas do QGIS.
        - Atualiza o checkbox de visibilidade no treeView de acordo com a visibilidade da camada no QGIS.
        """
        # Acessa a raiz da árvore de camadas do QGIS
        root = QgsProject.instance().layerTreeRoot()

        # Itera sobre todos os itens do modelo do treeView
        for i in range(self.treeViewModel.rowCount()):
            item = self.treeViewModel.item(i)

            # Obtém o ID da camada associado ao item, se existir
            layer_id = item.data(Qt.UserRole) if item else None

            # Obtém o nó da camada correspondente na árvore do QGIS
            node = root.findLayer(layer_id) if layer_id else None

            # Se o nó for válido, atualiza o estado do checkbox do item conforme a visibilidade do nó
            if node:
                item.setCheckState(Qt.Checked if node.isVisible() else Qt.Unchecked)

    def on_item_double_clicked(self, index):
        """
        Manipula o evento de duplo clique em um item do QTreeView. Este método é acionado quando um usuário
        efetua um duplo clique em uma camada listada, permitindo que altere as cores de preenchimento e borda da camada.

        Funções e Ações Desenvolvidas:
        - Recuperação da camada associada ao item clicado.
        - Obtenção das cores atuais de preenchimento e borda da camada.
        - Exibição de um diálogo para que o usuário selecione novas cores.
        - Aplicação das cores selecionadas à camada, se novas cores foram escolhidas.

        :param index: QModelIndex que representa o item no modelo que foi duplo clicado.
        """
        # Obtém o ID da camada do item clicado usando o UserRole
        layer_id = index.model().itemFromIndex(index).data(Qt.UserRole)
        # Busca a camada correspondente no projeto QGIS usando o ID
        layer = QgsProject.instance().mapLayer(layer_id)
        if layer:
            # Obtém as cores atuais de preenchimento e borda da camada
            current_fill_color, current_border_color = self.get_point_colors(layer)
            # Solicita ao usuário que selecione novas cores para preenchimento e borda
            new_fill_color, new_border_color = self.prompt_for_new_colors(current_fill_color, current_border_color)
            # Se novas cores forem selecionadas, aplica estas cores à camada
            if new_fill_color and new_border_color:
                self.apply_new_colors(layer, new_fill_color, new_border_color)

    def tratar_pontos(self, new_layer):
        """
        Configura sinais e comportamentos personalizados para uma nova camada de pontos:
        - ID automático
        - Campos X, Y, X_DMS, Y_DMS atualizados dinamicamente
        - Oculta campos extras na interface
        - Suprime formulário ao adicionar feições
        """
        # ID automático
        idx_id = new_layer.fields().indexOf("ID")
        if idx_id != -1:
            new_layer.setDefaultValueDefinition(
                idx_id, QgsDefaultValue('coalesce(maximum("ID"),0)+1'))
            new_layer.setEditorWidgetSetup(idx_id, QgsEditorWidgetSetup("Hidden", {}))

        # Verifica se existe campo X/Y
        idx_x = new_layer.fields().indexOf("X")
        idx_y = new_layer.fields().indexOf("Y")
        idx_x_dms = new_layer.fields().indexOf("X_DMS") if new_layer.fields().indexOf("X_DMS") != -1 else None
        idx_y_dms = new_layer.fields().indexOf("Y_DMS") if new_layer.fields().indexOf("Y_DMS") != -1 else None

        # Sinal para atualizar X/Y/DMS sempre que um ponto for adicionado ou movido
        new_layer.featureAdded.connect(lambda fid: self.atualizar_posicao_xy(new_layer, fid))
        new_layer.geometryChanged.connect(lambda fid, _g: self.atualizar_posicao_xy(new_layer, fid))

        # Oculta todos os campos exceto ID, X, Y, X_DMS, Y_DMS
        keep = set(i for i in [idx_id, idx_x, idx_y, idx_x_dms, idx_y_dms] if i is not None)
        for i in range(new_layer.fields().count()):
            if i not in keep:
                new_layer.setEditorWidgetSetup(i, QgsEditorWidgetSetup("Hidden", {}))

        # Suprime formulário de atributos ao adicionar ponto
        cfg = new_layer.editFormConfig()
        cfg.setSuppress(QgsEditFormConfig.SuppressOn)
        new_layer.setEditFormConfig(cfg)

        # Atualiza todos os X/Y já existentes
        new_layer.startEditing()
        for f in new_layer.getFeatures():
            self.atualizar_posicao_xy(new_layer, f.id())
        new_layer.commitChanges()

    def atualizar_posicao_xy(self, camada, fid):
        """
        Atualiza os campos X, Y, X_DMS e Y_DMS (se existirem) para a feição especificada, conforme a geometria do ponto.
        """
        feat = camada.getFeature(fid)
        geom = feat.geometry()
        if geom is None or geom.isEmpty():
            return

        # Busca índices dos campos
        idx_x = camada.fields().indexOf("X")
        idx_y = camada.fields().indexOf("Y")
        idx_x_dms = camada.fields().indexOf("X_DMS") if camada.fields().indexOf("X_DMS") != -1 else None
        idx_y_dms = camada.fields().indexOf("Y_DMS") if camada.fields().indexOf("Y_DMS") != -1 else None

        pt = geom.asPoint()
        is_geografico = camada.crs().isGeographic()

        # Define número de casas decimais
        casas = 6 if is_geografico else 3

        # Atualiza valores X/Y
        camada.changeAttributeValue(fid, idx_x, round(pt.x(), casas))
        camada.changeAttributeValue(fid, idx_y, round(pt.y(), casas))

        # Se for geográfico e existir campo DMS, atualiza também
        if is_geografico and idx_x_dms is not None and idx_y_dms is not None:
            camada.changeAttributeValue(fid, idx_x_dms, self.calcular_posicao_xy(pt.x()))
            camada.changeAttributeValue(fid, idx_y_dms, self.calcular_posicao_xy(pt.y()))

    def calcular_posicao_xy(self, valor):
        """
        Converte coordenada decimal para DMS (graus, minutos, segundos).
        Exemplo de saída: -36° 2' 4.20"
        """
        sinal = "-" if valor < 0 else ""
        valor = abs(valor)
        graus = int(valor)
        minutos = int((valor - graus) * 60)
        segundos = (valor - graus - minutos / 60) * 3600
        return f'{sinal}{graus}° {minutos}\' {segundos:.2f}"'

    def salvar_camada_permanente(self):
        """
        Salva uma camada temporária de pontos como permanente (arquivo Shapefile), preservando estilo e etiquetas.
        """
        # Obtém a seleção atual do treeView de pontos
        selected_indexes = self.dlg.treeViewListaPonto.selectedIndexes()
        if selected_indexes:
            selected_index = selected_indexes[0]
            # Pega o layer_id usando UserRole, se não encontrar pega padrão
            layer_id = selected_index.model().itemFromIndex(selected_index).data(Qt.UserRole)
            if layer_id is None:
                layer_id = selected_index.model().itemFromIndex(selected_index).data()
            layer_to_save = QgsProject.instance().mapLayer(layer_id)

            if layer_to_save:
                is_temporary = layer_to_save.dataProvider().name() == 'memory'

                # Clona cor, renderizador e etiquetas
                point_colors = self.get_point_colors(layer_to_save)  # ADAPTADO PARA PONTOS
                renderer = layer_to_save.renderer().clone()
                etiquetas = layer_to_save.labeling().clone() if layer_to_save.labeling() else None

                # Salva edições pendentes antes de exportar
                if layer_to_save.isEditable():
                    layer_to_save.commitChanges()

                # Caixa de diálogo para escolher caminho de exportação
                nome_arquivo = self.escolher_local_para_salvar(
                    os.path.join(self.ultimo_caminho_salvo, layer_to_save.name() + ".shp"),
                    "ESRI Shapefile Files (*.shp)")

                if nome_arquivo:
                    resultado_salvar = self.salvar_camada(layer_to_save, nome_arquivo, "ESRI Shapefile")

                    if resultado_salvar:
                        # Recarrega a camada salva
                        new_layer = QgsVectorLayer(nome_arquivo, layer_to_save.name(), "ogr")

                        if new_layer.isValid():
                            # Aplica o renderizador e as etiquetas originais
                            new_layer.setRenderer(renderer)
                            if etiquetas:
                                new_layer.setLabeling(etiquetas)
                                new_layer.setLabelsEnabled(layer_to_save.labelsEnabled())

                            # Atualiza campos automáticos de ponto (X/Y/DMS/ID)
                            self.tratar_pontos(new_layer)

                            # Adiciona ao projeto no grupo "Camadas Salvas"
                            QgsProject.instance().addMapLayer(new_layer, False)
                            root = QgsProject.instance().layerTreeRoot()
                            my_group = root.findGroup("Camadas Salvas")
                            if not my_group:
                                my_group = root.addGroup("Camadas Salvas")
                            my_group.addLayer(new_layer)

                            # Remove camada temporária se for o caso
                            if is_temporary:
                                QgsProject.instance().removeMapLayer(layer_id)

                            # Aplica cor aos pontos
                            if point_colors:
                                fill_color, border_color = point_colors
                                self.apply_new_colors(new_layer, fill_color, border_color)

                            new_layer.startEditing()
                        else:
                            self.mostrar_mensagem("Falha ao carregar a nova camada. A camada não é válida.", "Erro")
                    else:
                        self.mostrar_mensagem("Falha ao salvar a camada no formato Shapefile.", "Erro")

    def clone_layer_ponto(self):
        """
        Dispara a clonagem de uma camada de pontos selecionada no treeViewListaPonto.

        Funcionalidades:
        - Recupera o índice da camada de ponto selecionada na interface.
        - Obtém o ID da camada salvo no Qt.UserRole do item do treeView.
        - Busca a camada correspondente no projeto QGIS usando o ID.
        - Verifica se a camada existe e é do tipo ponto.
        - Instancia o gerenciador de clonagem para pontos (CloneManagerPt) e exibe as opções de clonagem ao usuário.
        - Atualiza o treeViewListaPonto para refletir eventuais novas camadas criadas.

        Observações:
        - Permite diferentes tipos de clonagem (somente feições, todos os atributos, combinação, exclusão de atributos).
        - Exibe mensagens de erro apropriadas para seleção inválida ou tipo de camada errado.
        """

        # Recupera os índices selecionados no treeViewListaPonto
        indexes = self.dlg.treeViewListaPonto.selectedIndexes()
        if not indexes:
            # Nenhuma camada selecionada: exibe mensagem de erro
            self.mostrar_mensagem("Nenhuma camada de ponto selecionada.", "Erro")
            return

        # Pega o primeiro índice selecionado (assumindo seleção simples)
        index = indexes[0]
        # Obtém o ID da camada salvo em Qt.UserRole
        layer_id = index.data(Qt.UserRole)
        # Busca a camada correspondente no projeto QGIS
        layer = QgsProject.instance().mapLayer(layer_id)

        if not layer:
            # Camada não localizada (ID inválido ou removido)
            self.mostrar_mensagem("Não foi possível localizar a camada no projeto.", "Erro")
            return

        # Garante que a camada selecionada seja de pontos
        if layer.geometryType() != QgsWkbTypes.PointGeometry:
            self.mostrar_mensagem("A camada selecionada não é de pontos.", "Erro")
            return

        # Instancia o gerenciador de clonagem de pontos e exibe o diálogo de opções
        CloneManagerPt(self, layer).show_clone_options()
        # Atualiza a visualização do treeView de pontos na interface
        self.atualizar_treeView_lista_ponto()  # Mantenha igual ao dos outros tipos

    def get_point_colors(self, layer):
        """
        Obtém as cores de preenchimento e borda de um ponto a partir da camada especificada.
        Este método acessa o renderizador da camada para extrair as configurações de cor atuais do símbolo do ponto.
        
        Funções e Ações Desenvolvidas:
        - Acesso ao renderizador da camada para obter os símbolos usados na renderização.
        - Extração da cor de preenchimento e da cor da borda do primeiro símbolo de ponto.
        - Retorno das cores obtidas ou (None, None) se as cores não puderem ser determinadas.
        
        :param layer: Camada do QGIS de onde as cores serão obtidas (deve ser uma camada de vetor de ponto).
        
        :return: Uma tupla contendo a cor de preenchimento e a cor da borda do ponto, respectivamente.
                 Retorna (None, None) se não conseguir extrair as cores.
        """
        # Acessa o renderizador da camada para obter a lista de símbolos usados na renderização
        symbols = layer.renderer().symbols(QgsRenderContext())
        if symbols:
            # Extrai a cor de preenchimento do primeiro símbolo (geralmente usado para o preenchimento de pontos)
            fill_color = symbols[0].color()
            # Define uma cor padrão para a borda caso não seja especificada
            border_color = Qt.black  # Cor padrão se não houver contorno definido
            # Verifica se há camadas de símbolo no símbolo do ponto
            if symbols[0].symbolLayerCount() > 0:
                border_layer = symbols[0].symbolLayer(0)
                # Se o símbolo da borda tiver uma propriedade de cor de borda, extrai essa cor
                if hasattr(border_layer, 'strokeColor'):
                    border_color = border_layer.strokeColor()
            # Retorna as cores de preenchimento e borda
            return fill_color, border_color
        # Retorna None para ambos se não houver símbolos ou se as cores não puderem ser determinadas
        return None, None

    def apply_new_colors(self, layer, fill_color, border_color):
        """
        Aplica novas cores de preenchimento e borda à camada especificada. Este método atualiza o renderizador
        da camada com um novo símbolo criado a partir das cores selecionadas, garantindo que as alterações de cor
        sejam visualmente refletidas no mapa do QGIS.

        Funções e Ações Desenvolvidas:
        - Criação de um novo símbolo com as cores de preenchimento e borda especificadas.
        - Configuração de um novo renderizador para a camada usando o símbolo criado.
        - Atualização da camada no QGIS para refletir as novas cores.

        :param layer: Camada do QGIS que terá suas cores alteradas.
        :param fill_color: Nova cor de preenchimento para a camada.
        :param border_color: Nova cor de borda para a camada.
        """
        # Cria um novo símbolo com as cores selecionadas baseado no tipo de geometria da camada
        new_symbol = QgsSymbol.defaultSymbol(layer.geometryType())
        new_symbol.setColor(fill_color)  # Define a cor de preenchimento do novo símbolo
        new_symbol.symbolLayer(0).setStrokeColor(border_color)  # Define a cor da borda no primeiro símbolo da camada

        # Cria um novo renderizador usando o novo símbolo e aplica à camada
        new_renderer = QgsSingleSymbolRenderer(new_symbol)
        layer.setRenderer(new_renderer)  # Atualiza o renderizador da camada

        # Dispara o processo de repintura para atualizar a visualização da camada no mapa
        layer.triggerRepaint()
        # Emite um sinal para notificar que a camada foi adicionada (usado para atualizar interfaces de usuário dependentes)
        QgsProject.instance().layerWasAdded.emit(layer)
        # Refresca a simbologia da camada na árvore de camadas do QGIS para garantir que as mudanças sejam visíveis
        self.iface.layerTreeView().refreshLayerSymbology(layer.id())

    def prompt_for_new_colors(self, current_fill_color, current_border_color):
        """
        Exibe diálogos de seleção de cor para que o usuário escolha novas cores de preenchimento e borda para uma camada.
        Este método usa o QColorDialog para permitir a seleção visual das cores, proporcionando uma interface amigável
        e intuitiva.

        Funções e Ações Desenvolvidas:
        - Exibição de diálogos para seleção de cores para a borda e o preenchimento da camada.
        - Verificação da validade das cores selecionadas.
        - Retorno das novas cores selecionadas, ou cores transparentes se nenhuma cor válida for escolhida.

        :param current_fill_color: Cor atual de preenchimento da camada, usada como valor inicial no diálogo.
        :param current_border_color: Cor atual da borda da camada, usada como valor inicial no diálogo.

        :return: Uma tupla contendo as novas cores de preenchimento e borda. Retorna cores transparentes se o usuário
                 cancelar a seleção ou selecionar uma cor inválida.
        """
        # Inicializa as variáveis de cor como None para verificar se houve seleção
        new_fill_color = None
        new_border_color = None

        # Solicita ao usuário que selecione uma nova cor da borda
        border_color_dialog = QColorDialog.getColor(current_border_color, self.dlg, "Escolha a Cor da Borda")
        if border_color_dialog.isValid():
            new_border_color = border_color_dialog
        else:
            new_border_color = QColor(0, 0, 0, 0)  # Define cor transparente se nenhuma cor válida for selecionada

        # Solicita ao usuário que selecione uma nova cor de preenchimento
        fill_color_dialog = QColorDialog.getColor(current_fill_color, self.dlg, "Escolha a Cor de Preenchimento")
        if fill_color_dialog.isValid():
            new_fill_color = fill_color_dialog
        else:
            new_fill_color = QColor(0, 0, 0, 0)  # Define cor transparente se nenhuma cor válida for selecionada

        # Retorna as novas cores selecionadas
        return new_fill_color, new_border_color

    def abrir_layer_properties(self, index):
        """
        Abre a janela de propriedades da camada selecionada no QTreeView. Este método é chamado quando o usuário deseja
        ver ou editar as propriedades de uma camada específica, como símbolos, campos e outras configurações.

        Funções e Ações Desenvolvidas:
        - Obtenção do ID da camada a partir do item selecionado no QTreeView.
        - Recuperação da camada correspondente no projeto QGIS.
        - Exibição da janela de propriedades da camada se ela for encontrada.

        :param index: O índice do modelo que representa o item selecionado no QTreeView.
        """
        # Obtém o ID da camada do item selecionado no treeView
        layer_id = index.model().itemFromIndex(index).data(Qt.UserRole)
        # Busca a camada correspondente no projeto QGIS usando o ID
        layer = QgsProject.instance().mapLayer(layer_id)
        # Se a camada for encontrada, exibe a janela de propriedades da camada
        if layer:
            self.iface.showLayerProperties(layer)

    def open_context_menu(self, position):
        """
        Cria e exibe um menu de contexto para ações específicas em um item selecionado no QTreeView.
        Este método é acionado com um clique do botão direito do mouse sobre um item da lista,
        apresentando opções como abrir propriedades da camada e alterar tamanho dos pontos da camada.

        Funções e Ações Desenvolvidas:
        - Verificação da seleção atual no QTreeView para determinar o item sobre o qual o menu será aberto.
        - Criação das opções do menu para manipulação das propriedades da camada e ajuste o tamanho do ponto.
        - Execução do menu no local do clique e execução da ação selecionada.

        :param position: A posição do cursor no momento do clique, usada para posicionar o menu de contexto.
        """
        # Obtém os índices selecionados no QTreeView
        indexes = self.dlg.treeViewListaPonto.selectedIndexes()
        if indexes:
            # Obter o índice da primeira coluna, que deve conter o ID da camada
            index = indexes[0].sibling(indexes[0].row(), 0)
            # Cria o menu de contexto
            menu = QMenu()
            # Adiciona opção para abrir propriedades da camada
            layer_properties_action = menu.addAction("Abrir Propriedades da Camada")
            # Adiciona opção para alterar a espessura da borda
            change_border_thickness_action = menu.addAction("Tamanho do Ponto")
            # Exibe o menu no local do clique e aguarda ação do usuário
            action = menu.exec_(self.dlg.treeViewListaPonto.viewport().mapToGlobal(position))

            # Executa a ação correspondente à escolha do usuário
            if action == layer_properties_action:
                self.abrir_layer_properties(index)
            elif action == change_border_thickness_action:
                self.prompt_for_new_point_size(index)

    def prompt_for_new_point_size(self, index):
        """
        Exibe um diálogo que permite ao usuário ajustar o tamanho do ponto de uma camada de ponto.
        O método recupera o tamanho atual do ponto, apresenta um QDoubleSpinBox para seleção do novo valor,
        e aplica a alteração se o usuário confirmar.

        Funções e Ações Desenvolvidas:
        - Recuperação da camada associada ao item selecionado no QTreeView.
        - Obtenção do tamanho atual do ponto da camada.
        - Criação de um diálogo com um QDoubleSpinBox para o usuário escolher o novo tamanho.
        - Aplicação do novo tamanho ao ponto da camada se o usuário confirmar a mudança.

        :param index: Índice do item no modelo de onde o ID da camada é extraído.
        """
        # Recupera o ID da camada do item selecionado no QTreeView
        layer_id = index.model().itemFromIndex(index).data(Qt.UserRole)
        # Busca a camada correspondente no projeto QGIS usando o ID
        layer = QgsProject.instance().mapLayer(layer_id)
        if layer:
            # Obtém o tamanho atual do ponto da camada
            current_size = self.get_current_point_size(layer)

            # Cria um diálogo personalizado para ajuste do tamanho
            dlg = QDialog(self.dlg)
            dlg.setWindowTitle("Tamanho do Ponto")
            layout = QVBoxLayout(dlg)

            # Configura um QDoubleSpinBox para escolha do novo tamanho
            spinBox = QDoubleSpinBox(dlg)
            spinBox.setRange(0, 100)  # Define o intervalo de valores
            spinBox.setSingleStep(0.2)  # Define o incremento de ajuste
            spinBox.setValue(current_size)  # Define o valor inicial com o tamanho atual
            spinBox.setDecimals(2)  # Define a precisão decimal
            layout.addWidget(spinBox)

            # Cria botões de OK e Cancelar
            buttonLayout = QHBoxLayout()
            okButton = QPushButton("OK", dlg)
            okButton.clicked.connect(dlg.accept)  # Conecta o botão OK à ação de aceitar o diálogo
            buttonLayout.addWidget(okButton)
            layout.addLayout(buttonLayout)

            # Exibe o diálogo e espera pela ação do usuário
            if dlg.exec_():
                new_size = spinBox.value()  # Obtém o novo valor do tamanho
                # Aplica o novo tamanho ao ponto da camada se o usuário confirmar
                self.apply_new_point_size(layer, new_size)

    def get_current_point_size(self, layer):
        """
        Recupera o tamanho atual do ponto de uma camada específica no QGIS. Este método acessa o renderizador da camada,
        e extrai o tamanho do ponto do primeiro símbolo de ponto encontrado.

        Funções e Ações Desenvolvidas:
        - Acesso ao renderizador da camada para obter os símbolos usados na renderização.
        - Extração do tamanho do ponto do primeiro símbolo de ponto se disponível.
        - Retorno do tamanho atual do ponto ou 0 se não for possível determinar.

        :param layer: Camada do QGIS cujo tamanho do ponto precisa ser obtido.
        :return: Tamanho do ponto da camada ou 0 se não for possível determinar o tamanho.
        """
        # Acessa o renderizador da camada para obter a lista de símbolos utilizados
        symbols = layer.renderer().symbols(QgsRenderContext())
        if symbols and symbols[0].symbolLayerCount() > 0:
            # Extrai o primeiro símbolo de ponto
            point_layer = symbols[0].symbolLayer(0)
            # Verifica se o símbolo do ponto possui um atributo para tamanho
            if hasattr(point_layer, 'size'):
                # Retorna o tamanho do ponto se disponível
                return point_layer.size()
        # Retorna 0 como valor padrão caso o tamanho do ponto não seja acessível ou não esteja definido
        return 0

    def apply_new_point_size(self, layer, size):
        """
        Aplica um novo tamanho de ponto para a camada especificada no QGIS. Este método modifica diretamente
        o símbolo de ponto usado para renderizar a camada, garantindo que as alterações sejam visíveis imediatamente
        no mapa.

        Funções e Ações Desenvolvidas:
        - Acesso ao renderizador da camada para modificar o tamanho do ponto de cada símbolo de ponto.
        - Aplicação do novo tamanho do ponto.
        - Disparo de eventos para atualizar a visualização e a interface do usuário no QGIS.

        :param layer: Camada do QGIS que terá o tamanho do ponto ajustado.
        :param size: Novo tamanho do ponto a ser aplicado.
        """
        # Acessa o renderizador da camada para obter os símbolos usados na renderização
        symbols = layer.renderer().symbols(QgsRenderContext())
        if symbols:
            # Itera sobre cada símbolo na lista de símbolos da camada
            for symbol in symbols:
                # Verifica se há camadas de símbolo disponíveis no símbolo atual
                if symbol.symbolLayerCount() > 0:
                    point_layer = symbol.symbolLayer(0)
                    # Verifica se o símbolo do ponto possui um método para definir o tamanho do ponto
                    if hasattr(point_layer, 'setSize'):
                        # Aplica o novo tamanho do ponto
                        point_layer.setSize(size)
            # Dispara o repintura da camada para atualizar a visualização no mapa
            layer.triggerRepaint()
            # Emite um sinal indicando que a camada foi adicionada para atualizar interfaces de usuário dependentes
            QgsProject.instance().layerWasAdded.emit(layer)
            # Refresca a simbologia da camada na árvore de camadas do QGIS
            self.iface.layerTreeView().refreshLayerSymbology(layer.id())

    def escolher_local_para_salvar(self, nome_padrao, tipo_arquivo):
        """
        Permite ao usuário escolher um local e um nome de arquivo para salvar uma camada, usando uma caixa de diálogo.
        O método também gerencia nomes de arquivos para evitar sobreposição e lembra o último diretório utilizado.

        Funções e Ações Desenvolvidas:
        - Recuperação do último diretório utilizado através das configurações do QGIS.
        - Geração de um nome de arquivo único para evitar sobreposição.
        - Exibição de uma caixa de diálogo para escolha do local de salvamento.
        - Atualização do último diretório utilizado nas configurações do QGIS.

        :param nome_padrao: Nome padrão proposto para o arquivo a ser salvo.
        :param tipo_arquivo: Descrição do tipo de arquivo para a caixa de diálogo (ex. "Arquivos DXF (*.dxf)").

        :return: O caminho completo do arquivo escolhido para salvar ou None se nenhum arquivo foi escolhido.
        """
        # Acessa as configurações do QGIS para recuperar o último diretório utilizado
        settings = QSettings()
        lastDir = settings.value("lastDir", "")  # Usa uma string vazia como padrão se não houver último diretório

        # Configura as opções da caixa de diálogo para salvar arquivos
        options = QFileDialog.Options()
        
        # Gera um nome de arquivo com um sufixo numérico caso o arquivo já exista
        base_nome_padrao, extensao = os.path.splitext(nome_padrao)
        numero = 1
        nome_proposto = base_nome_padrao
        
        # Incrementa o número no nome até encontrar um nome que não exista
        while os.path.exists(os.path.join(lastDir, nome_proposto + extensao)):
            nome_proposto = f"{base_nome_padrao}_{numero}"
            numero += 1

        # Propõe o nome completo no último diretório utilizado
        nome_completo_proposto = os.path.join(lastDir, nome_proposto + extensao)

        # Exibe a caixa de diálogo para salvar arquivos com o nome proposto
        fileName, _ = QFileDialog.getSaveFileName(
            self.dlg,
            "Salvar Camada",
            nome_completo_proposto,
            tipo_arquivo,
            options=options)

        # Verifica se um nome de arquivo foi escolhido
        if fileName:
            # Atualiza o último diretório usado nas configurações do QGIS
            settings.setValue("lastDir", os.path.dirname(fileName))

            # Assegura que o arquivo tenha a extensão correta
            if not fileName.endswith(extensao):
                fileName += extensao

        return fileName  # Retorna o caminho completo do arquivo escolhido ou None se cancelado

    def iniciar_progress_bar(self, layer):
        """
        Inicia e exibe uma barra de progresso na interface do usuário para o processo de exportação de uma camada para DXF.

        Parâmetros:
        - layer (QgsVectorLayer): A camada para a qual a barra de progresso será exibida, indicando o progresso da exportação.

        Funcionalidades:
        - Cria uma mensagem personalizada na barra de mensagens para acompanhar o progresso.
        - Configura e estiliza uma barra de progresso.
        - Adiciona a barra de progresso à barra de mensagens e a exibe na interface do usuário.
        - Define o valor máximo da barra de progresso com base no número de feições na camada.
        - Retorna os widgets de barra de progresso e de mensagem para que possam ser atualizados durante a exportação.
        """
        # Cria uma mensagem personalizada na barra de progresso
        progressMessageBar = self.iface.messageBar().createMessage("Exportando camada para DXF: " + layer.name())
        progressBar = QProgressBar()  # Cria uma instância da QProgressBar
        progressBar.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)  # Alinha a barra de progresso à esquerda e verticalmente ao centro
        progressBar.setFormat("%p% - %v de %m Feições processadas")  # Define o formato da barra de progresso
        progressBar.setMinimumWidth(300)  # Define a largura mínima da barra de progresso

        # Estiliza a barra de progresso
        progressBar.setStyleSheet("""
            QProgressBar {
                border: 1px solid grey;
                border-radius: 2px;
                background-color: #cddbde;
                text-align: center;
            }
            QProgressBar::chunk {
                background-color: #55aaff;
                width: 5px;
                margin: 1px;
            }
            QProgressBar {
                min-height: 5px;}""")

        # Adiciona a progressBar ao layout da progressMessageBar e exibe na interface
        progressMessageBar.layout().addWidget(progressBar)
        self.iface.messageBar().pushWidget(progressMessageBar, Qgis.Info)

        # Define o valor máximo da barra de progresso com base no número de feições da camada
        feature_count = layer.featureCount()
        progressBar.setMaximum(feature_count)

        # Retorna o progressBar e o progressMessageBar para que possam ser atualizados durante o processo de exportação
        return progressBar, progressMessageBar

    def mostrar_mensagem(self, texto, tipo, duracao=3, caminho_pasta=None, caminho_arquivos=None):
        """
        Exibe uma mensagem na barra de mensagens do QGIS, proporcionando feedback ao usuário baseado nas ações realizadas.
        As mensagens podem ser de erro ou de sucesso, com uma duração configurável e uma opção de abrir uma pasta.

        :param texto: Texto da mensagem a ser exibida.
        :param tipo: Tipo da mensagem ("Erro" ou "Sucesso") que determina a cor e o ícone da mensagem.
        :param duracao: Duração em segundos durante a qual a mensagem será exibida (padrão é 3 segundos).
        :param caminho_pasta: Caminho da pasta a ser aberta ao clicar no botão (padrão é None).
        :param caminho_arquivos: Caminho do arquivo a ser executado ao clicar no botão (padrão é None).
        """
        # Obtém a barra de mensagens da interface do QGIS
        bar = iface.messageBar()  # Acessa a barra de mensagens da interface do QGIS

        # Exibe a mensagem com o nível apropriado baseado no tipo
        if tipo == "Erro":
            # Mostra uma mensagem de erro na barra de mensagens com um ícone crítico e a duração especificada
            bar.pushMessage("Erro", texto, level=Qgis.Critical, duration=duracao)
        elif tipo == "Sucesso":
            # Cria o item da mensagem
            msg = bar.createMessage("Sucesso", texto)
            
            # Se o caminho da pasta for fornecido, adiciona um botão para abrir a pasta
            if caminho_pasta:
                botao_abrir_pasta = QPushButton("Abrir Pasta")
                botao_abrir_pasta.clicked.connect(lambda: os.startfile(caminho_pasta))
                msg.layout().insertWidget(1, botao_abrir_pasta)  # Adiciona o botão à esquerda do texto
            
            # Se o caminho do arquivo for fornecido, adiciona um botão para executar o arquivo
            if caminho_arquivos:
                botao_executar = QPushButton("Executar")
                botao_executar.clicked.connect(lambda: os.startfile(caminho_arquivos))
                msg.layout().insertWidget(2, botao_executar)  # Adiciona o botão à esquerda do texto
            
            # Adiciona a mensagem à barra com o nível informativo e a duração especificada
            bar.pushWidget(msg, level=Qgis.Info, duration=duracao)

    def obter_cor_rotulo(self, layer):
        """
        Obtém a cor do rótulo da camada e a converte para o formato de cor KML (AABBGGRR).

        Parâmetros:
        - layer (QgsVectorLayer): A camada de origem para obter a cor do rótulo.

        Funcionalidades:
        - Verifica se os rótulos estão habilitados para a camada.
        - Obtém o renderizador de rótulo e as configurações de texto.
        - Converte a cor do rótulo do formato Qt para o formato KML.
        - Retorna a cor do rótulo em formato hexadecimal (AABBGGRR).
        - Retorna preto como padrão se os rótulos não estiverem habilitados ou configurados.
        """
        # Verifica se os rótulos estão habilitados para a camada
        if layer.labelsEnabled():
            # Obtenha o renderizador de rótulo e as configurações de texto
            labeling = layer.labeling()
            if labeling:
                text_format = labeling.settings().format()
                cor_texto = text_format.color()

                # Converta a cor do Qt para o formato de cor KML (AABBGGRR)
                cor_kml = cor_texto.alpha() << 24 | cor_texto.blue() << 16 | cor_texto.green() << 8 | cor_texto.red()
                cor_kml_hex = format(cor_kml, '08x')
                return cor_kml_hex

        # Retorna preto como padrão se os rótulos não estiverem habilitados ou não configurados
        return 'ff000000'

    def obter_cor_icone(self, layer):
        """
        Obtém a cor do ícone da camada e a converte para o formato de cor KML (AABBGGRR).

        Parâmetros:
        - layer (QgsVectorLayer): A camada de origem para obter a cor do ícone.

        Funcionalidades:
        - Verifica se a geometria da camada é do tipo ponto.
        - Obtém o renderizador de símbolo da camada.
        - Verifica se o renderizador é do tipo 'singleSymbol' e se o símbolo é um marcador.
        - Obtém a cor do símbolo do ícone.
        - Converte a cor do símbolo do formato Qt para o formato KML.
        - Retorna a cor do ícone em formato hexadecimal (AABBGGRR).
        - Retorna branco como padrão se a geometria não for do tipo ponto ou se o símbolo não for um marcador.
        """
        # Verifica se a geometria da camada é do tipo ponto
        if layer.geometryType() == QgsWkbTypes.PointGeometry:
            # Obtém o símbolo da camada
            renderer = layer.renderer()
            if renderer and renderer.type() == 'singleSymbol':
                symbol = renderer.symbol()
                if symbol and symbol.type() == QgsSymbol.Marker:
                    # Obtém a cor do símbolo
                    cor_icone = symbol.color()
                    cor_kml = cor_icone.alpha() << 24 | cor_icone.blue() << 16 | cor_icone.green() << 8 | cor_icone.red()
                    cor_kml_hex = format(cor_kml, '08x')
                    return cor_kml_hex

        # Retorna branco como padrão se não for um símbolo de ponto padrão
        return 'ffffffff'

    def url_para_link_html(self, url):
        """
        Converte um URL em um link HTML clicável. A função realiza as seguintes etapas:

        1. Verifica se o texto fornecido é um URL válido.
        2. Se for um URL válido, formata o texto "Link de acesso" como um link HTML com o URL correto.
        3. Se não for um URL, retorna o texto como está.

        Parâmetros:
        - url (str): O texto a ser verificado e possivelmente convertido em um link HTML.

        Funcionalidades:
        - Verificação de validade do URL usando expressão regular.
        - Formatação do texto como link HTML se o URL for válido.
        - Retorno do texto original se não for um URL válido.
        """
        # Verifica se o texto é um URL válido
        if re.match(r'https?://[^\s]+', url):
            # Retorna o texto "Acesso Aqui" formatado como link HTML com o URL correto
            return f"<a href='{url}'>Link de acesso</a>"
        else:
            # Se não for um URL, retorna o texto como está
            return url

    def gerar_cor_suave(self):
        """
        Gera uma cor suave aleatória no formato hexadecimal. A função realiza as seguintes etapas:

        1. Gera valores aleatórios para os componentes RGB dentro do intervalo de cores suaves (180-255).
        2. Formata e retorna a cor como uma string hexadecimal.

        Funcionalidades:
        - Geração de valores aleatórios para os componentes vermelho, verde e azul (RGB).
        - Formatação da cor no formato hexadecimal.

        Retorna:
        - str: A cor gerada no formato hexadecimal (e.g., '#aabbcc').
        """
        # Gera valores aleatórios para os componentes RGB dentro do intervalo de cores suaves
        r = random.randint(180, 255) # Componente vermelho
        g = random.randint(180, 255) # Componente verde
        b = random.randint(180, 255) # Componente azul
        return f'#{r:02x}{g:02x}{b:02x}' # Retorna a cor formatada como uma string hexadecimal

    def exportar_para_kml(self):
        """
        Exporta a camada selecionada para um arquivo KML. A função realiza as seguintes etapas:

        1. Obtém a seleção atual no QTreeView.
        2. Verifica se alguma camada foi selecionada. Se não, exibe uma mensagem de erro.
        3. Obtém o nome da camada selecionada.
        4. Verifica se a camada selecionada está presente no projeto. Se não, exibe uma mensagem de erro.
        5. Verifica se a camada possui feições para exportar. Se não, exibe uma mensagem de erro.
        6. Abre um diálogo para o usuário selecionar o campo de identificação e a URL do ícone.
           - Se o usuário cancelar o diálogo, exibe uma mensagem informativa.
           - Se o usuário aceitar, obtém o campo de identificação, URL do ícone e o estado do checkbox de rótulos.
        7. Obtém o caminho do arquivo para salvar o KML.
           - Se o usuário cancelar a seleção do caminho, exibe uma mensagem informativa.
        8. Inicia a barra de progresso após selecionar o caminho do arquivo.
        9. Mede o tempo de execução da criação do KML e da escrita no arquivo.
           - Cria o conteúdo KML em memória.
           - Escreve o conteúdo KML no arquivo.
        10. Calcula o tempo de execução.
        11. Remove a barra de progresso.
        12. Exibe uma mensagem de sucesso indicando o tempo de execução.

        Funcionalidades:
        - Verificação da seleção e existência da camada.
        - Verificação da presença de feições na camada.
        - Diálogo de seleção de campo de identificação e URL de ícone.
        - Seleção do caminho do arquivo para salvar o KML.
        - Criação e escrita do conteúdo KML.
        - Exibição de mensagens informativas e de erro.
        - Exibição de barra de progresso durante o processo de exportação.
        """
        # Obtém a seleção atual no QTreeView
        indexes = self.dlg.treeViewListaPonto.selectionModel().selectedIndexes()
        if not indexes:
            self.mostrar_mensagem("Selecione uma camada para exportar.", "Erro")
            return

        # Obtém o nome da camada selecionada
        selected_layer_name = self.treeViewModel.itemFromIndex(indexes[0]).text()
        layer = QgsProject.instance().mapLayersByName(selected_layer_name)[0]
        if not layer:
            self.mostrar_mensagem("Camada não encontrada.", "Erro")
            return

        # Verifica se há feições na camada
        if layer.featureCount() == 0:
            self.mostrar_mensagem("Nenhuma feição para ser exportada.", "Erro")
            return

        # Obtenha o campo de identificação desejado e a URL do ícone do usuário
        dialog = IconFieldSelectionDialog(layer)
        if dialog.exec_() == QDialog.Accepted:
            campo_id, icon_url, image_url, overlay_url = dialog.get_selections()  # Captura campo, URL do ícone e URL da imagem
            exportar_rotulos = dialog.check_box.isChecked()  # Obtem o estado do checkbox diretamente do diálogo
        else:
            self.mostrar_mensagem("Seleção de ícone cancelada.", "Info")
            return

        # Obtenha o caminho do arquivo para salvar o KML
        nome_padrao = f"{layer.name()}.kml"
        tipo_arquivo = "KML Files (*.kml)"
        caminho_arquivo = self.escolher_local_para_salvar(nome_padrao, tipo_arquivo)
        if not caminho_arquivo:
            self.mostrar_mensagem("Exportação cancelada.", "Info")
            return

        # Inicia a barra de progresso após selecionar o caminho do arquivo
        progressBar, progressMessageBar = self.iniciar_progress_bar(layer)

        # Medir o tempo de execução da criação do KML e da escrita no arquivo
        start_time = time.time()
        kml_element = self.criar_kml_em_memoria(layer, campo_id, icon_url, exportar_rotulos, progressBar, image_url, overlay_url)  # Passa o estado do checkbox como argumento
        tree = ET.ElementTree(kml_element)
        tree.write(caminho_arquivo, xml_declaration=True, encoding='utf-8', method="xml")
        end_time = time.time()

        # Calcula o tempo de execução
        execution_time = end_time - start_time

        # Remove a barra de progresso
        self.iface.messageBar().clearWidgets()

        # Exibir mensagem de sucesso com o tempo de execução e caminhos dos arquivos
        self.mostrar_mensagem(
            f"Camada exportada para KMZ em {execution_time:.2f} segundos", 
            "Sucesso", 
            caminho_pasta=os.path.dirname(caminho_arquivo), 
            caminho_arquivos=caminho_arquivo)

    def criar_kml_em_memoria(self, layer, campo_id, icon_url, exportar_rotulos, progressBar, image_url, overlay_url):
        """
        Cria o conteúdo KML em memória para a camada fornecida. A função realiza as seguintes etapas:

        1. Cria o elemento raiz do KML e o elemento Document.
        2. Obtém a cor do ícone e do rótulo da camada.
        3. Verifica se a camada está no sistema de referência de coordenadas WGS84 (EPSG:4326) e, se necessário, prepara a transformação de coordenadas.
        4. Itera sobre cada feição na camada e cria um Placemark para cada uma:
           - Adiciona um nome ao Placemark com o valor do campo ID.
           - Cria a tag Point com as coordenadas da feição.
           - Adiciona ExtendedData com os atributos da feição.
           - Define um estilo para o Placemark, incluindo ícone e rótulo.
        5. Atualiza a barra de progresso durante o processamento.
        6. Retorna o elemento raiz do KML.
        7. Cria um índice espacial para a camada

        Parâmetros:
        - layer (QgsVectorLayer): A camada de origem para a exportação.
        - campo_id (str): O campo de identificação a ser usado como nome dos Placemarks.
        - icon_url (str): A URL do ícone a ser usado para os Placemarks.
        - exportar_rotulos (bool): Indica se os rótulos devem ser exportados.
        - Adiciona um ScreenOverlay ao documento KML.
        - progressBar (QProgressBar): A barra de progresso a ser atualizada durante o processamento.

        Funcionalidades:
        - Criação de KML em memória com elementos e estilos configuráveis.
        - Transformação de coordenadas para WGS84, se necessário.
        - Adição de dados estendidos e estilos personalizados para cada Placemark.
        - Atualização da barra de progresso para informar o progresso do usuário.
        
        Retorna:
        - xml.etree.ElementTree.Element: O elemento raiz do KML criado.
        """
        # Cria um índice espacial para a camada
        spatial_index = QgsSpatialIndex()
        for feature in layer.getFeatures():
            spatial_index.insertFeature(feature)

        # Cria o elemento raiz do KML
        kml = ET.Element('kml', xmlns="http://www.opengis.net/kml/2.2")
        document = ET.SubElement(kml, 'Document')
        cor_icone = self.obter_cor_icone(layer)

        # Obtém a cor do rótulo em formato KML e converte para hexadecimal RGB
        cor_rotulo_kml = self.obter_cor_rotulo(layer)
        # Converter de AABBGGRR para RRGGBB para uso em HTML
        cor_rotulo_html = '#' + cor_rotulo_kml[6:] + cor_rotulo_kml[4:6] + cor_rotulo_kml[2:4]

        # Verifica se a camada já está em WGS84 (EPSG:4326)
        if layer.crs().authid() != 'EPSG:4326':
            # Define o sistema de referência de coordenadas para WGS84 (usado pelo KML)
            crsDestino = QgsCoordinateReferenceSystem(4326)  # WGS84
            transform = QgsCoordinateTransform(layer.crs(), crsDestino, QgsProject.instance())
            transformar = True
        else:
            transformar = False

        # Itera sobre cada feição na camada e cria um Placemark
        for i, feature in enumerate(layer.getFeatures(), start=1):
            placemark = ET.SubElement(document, 'Placemark')

            # Adiciona um nome ao Placemark com o valor do campo ID
            name = ET.SubElement(placemark, 'name')
            name.text = str(feature[campo_id])

            # Cria a tag Point com as coordenadas
            point = ET.SubElement(placemark, 'Point')
            coordinates = ET.SubElement(point, 'coordinates')
            geom = feature.geometry()
            # Aplica a transformação de coordenadas se necessário
            if transformar:
                geom.transform(transform)

            coord_text = f"{geom.asPoint().x()},{geom.asPoint().y()},0"  # Adicione altitude se necessário
            coordinates.text = coord_text

            # Adiciona ExtendedData com os atributos da feição
            extended_data = ET.SubElement(placemark, 'ExtendedData')
            for field in layer.fields():
                data = ET.SubElement(extended_data, 'Data', name=field.name())
                value = ET.SubElement(data, 'value')
                value.text = str(feature[field.name()])

            # Define um estilo para o Placemark
            style = ET.SubElement(placemark, 'Style')
            icon_style = ET.SubElement(style, 'IconStyle')
            label_style = ET.SubElement(style, 'LabelStyle')

            self.gerar_balloon_style(feature, campo_id, style, cor_rotulo_html, image_url, overlay_url)

            # Condições para escala do ícone e rótulo
            if icon_url:
                icon = ET.SubElement(icon_style, 'Icon')
                href = ET.SubElement(icon, 'href')
                href.text = icon_url  # Use a URL do ícone escolhido
                color = ET.SubElement(icon_style, 'color')
                color.text = cor_icone  # A cor é usada aqui
                # Escala do ícone quando não há rótulos para exportar
                if not exportar_rotulos:
                    label_scale = ET.SubElement(label_style, 'scale')
                    label_scale.text = '0'  # Esconde o rótulo

            if exportar_rotulos:
                label_scale = ET.SubElement(label_style, 'scale')
                label_scale.text = '1.0'  # Mostra o rótulo
                color = ET.SubElement(label_style, 'color')
                color.text = cor_rotulo_kml  # A cor é usada aqui
                # Escala do ícone quando não há ícone para exportar
                if not icon_url:
                    icon_scale = ET.SubElement(icon_style, 'scale')
                    icon_scale.text = '0'  # Esconde o ícone

            progressBar.setValue(i)  # Atualiza a barra de progresso

        # Adiciona ScreenOverlay apenas se overlay_url for fornecida e não for vazia
        if overlay_url:
            # Redimensiona a imagem obtida a partir do URL
            imagem_redimensionada, nova_largura, nova_altura = self.redimensionar_imagem_proporcional_url(overlay_url, 300, 150)

            if imagem_redimensionada is not None:
                # Adiciona o ScreenOverlay ao KML usando a imagem redimensionada
                screen_overlay = ET.SubElement(document, 'ScreenOverlay') # Cria o elemento ScreenOverlay no documento KML
                name = ET.SubElement(screen_overlay, 'name')  # Define o nome do ScreenOverlay
                name.text = 'logo'

                # Define o ícone do ScreenOverlay, utilizando a URL da imagem fornecida
                icon = ET.SubElement(screen_overlay, 'Icon')
                href = ET.SubElement(icon, 'href')
                href.text = overlay_url

                # Configura a posição e o tamanho do overlay na tela
                overlay_xy = ET.SubElement(screen_overlay, 'overlayXY', x="1", y="1", xunits="fraction", yunits="fraction")
                screen_xy = ET.SubElement(screen_overlay, 'screenXY', x=f"{nova_largura}", y=f"{nova_altura}", xunits="pixels", yunits="pixels")
                rotation_xy = ET.SubElement(screen_overlay, 'rotationXY', x="0", y="0", xunits="fraction", yunits="fraction")
                # Define o tamanho do ScreenOverlay
                size = ET.SubElement(screen_overlay, 'size', x=f"{nova_largura}", y=f"{nova_altura}", xunits="pixels", yunits="pixels")

        # Retorna o elemento raiz do KML
        return kml

    def gerar_balloon_style(self, feature, campo_id, style, cor_rotulo_html, image_url, overlay_url):
        """
        Gera o estilo de balão (BalloonStyle) para uma feição KML. A função realiza as seguintes etapas:

        1. Constrói uma tabela HTML com os atributos da feição.
        2. Adiciona a tabela HTML ao BalloonStyle do KML.

        Parâmetros:
        - feature (QgsFeature): A feição para a qual o BalloonStyle será gerado.
        - campo_id (str): O campo de identificação a ser usado no balão.
        - style (xml.etree.ElementTree.Element): O elemento de estilo do KML onde o BalloonStyle será adicionado.
        - cor_rotulo_html (str): A cor do rótulo em formato hexadecimal RGB para uso em HTML.
        - image_url: A URL da imagem a ser usada como ícone do placemark.
        - overlay_url: A URL da imagem a ser usada como overlay.

        Funcionalidades:
        - Construção de tabela HTML com os atributos da feição.
        - Geração de cores suaves para o fundo das células da tabela.
        - Conversão de URLs em links HTML clicáveis.
        - Adição da tabela HTML ao BalloonStyle do KML.
        - image_url: A URL da imagem a ser usada como ícone do Placemark.
        """
        # Construir a tabela HTML
        tabela_geral_html = '<table border="1" style="border-collapse: collapse; border: 2px solid black; width: 100%;">'
        for field in feature.fields():
            cor_fundo = self.gerar_cor_suave()  # Gera uma cor suave
            field_value = str(feature[field.name()])
            field_value_as_link = self.url_para_link_html(field_value)
            tabela_geral_html += f'<tr><td><table border="0" style="background-color: {cor_fundo}; width: 100%;">'
            tabela_geral_html += f'<tr><td style="text-align: left;"><b>{field.name()}</b></td>'
            tabela_geral_html += f'<td style="text-align: right;">{str(feature[field.name()])}</td></tr></table></td></tr>'
        tabela_geral_html += '</table>'

        # Redimensiona a imagem para 150x75 antes de gerar o HTML
        imagem_html = ""
        if image_url:  # Verifica se a URL da imagem foi fornecida
            imagem_redimensionada, nova_largura, nova_altura = self.redimensionar_imagem_proporcional_url(image_url, 150, 75)
            if imagem_redimensionada:
                # Gera o HTML com a nova largura e altura
                imagem_html = f'<div style="text-align: center; padding: 10px 0;"><img src="{image_url}" alt="Imagem" width="{nova_largura}" height="{nova_altura}"></div>'
            else:
                # Se não foi possível redimensionar, usar dimensões padrão ou uma mensagem de erro
                imagem_html = f'<div style="text-align: center; padding: 10px 0;"><p>Imagem não pôde ser carregada.</p></div>'

        # BalloonStyle com a imagem e a tabela de atributos diretamente
        balloon_style = ET.SubElement(style, 'BalloonStyle')
        text = ET.SubElement(balloon_style, 'text')
        balloon_html = f"""
        {imagem_html}
            <h3 style="margin-bottom:1px;">{campo_id}: {str(feature[campo_id])}</h3>
            <p>Tabela de Informações:</p>
            {tabela_geral_html}
            """
        text.text = balloon_html

    def redimensionar_imagem_proporcional_url(self, url_imagem, largura_max, altura_max):
        """
        Baixa a imagem de um URL e redimensiona proporcionalmente para caber dentro de uma caixa de largura e altura máximas.

        Parâmetros:
        - url_imagem (str): URL da imagem a ser redimensionada.
        - largura_max (int): Largura máxima da caixa.
        - altura_max (int): Altura máxima da caixa.

        Retorna:
        - Uma instância de `PIL.Image.Image` redimensionada proporcionalmente.
        - As novas dimensões da imagem (largura, altura).
        """
        headers = {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.110 Safari/537.3'
        }  # Define o cabeçalho de User-Agent para a solicitação HTTP

        try:
            response = requests.get(url_imagem, headers=headers)  # Faz o download da imagem do URL com o cabeçalho definido
            response.raise_for_status()  # Verifica se o download foi bem-sucedido

            imagem = Image.open(BytesIO(response.content))  # Abre a imagem a partir do conteúdo baixado
            largura_original, altura_original = imagem.size  # Obtém as dimensões originais da imagem

            proporcao_largura = largura_max / largura_original  # Calcula a proporção para a largura
            proporcao_altura = altura_max / altura_original  # Calcula a proporção para a altura
            proporcao_final = min(proporcao_largura, proporcao_altura)  # Usa a menor proporção para manter a imagem dentro da caixa

            nova_largura = int(largura_original * proporcao_final)  # Calcula a nova largura proporcional
            nova_altura = int(altura_original * proporcao_final)  # Calcula a nova altura proporcional

            imagem_redimensionada = imagem.resize((nova_largura, nova_altura), Image.LANCZOS)  # Redimensiona a imagem

            return imagem_redimensionada, nova_largura, nova_altura  # Retorna a imagem redimensionada e as novas dimensões

        except UnidentifiedImageError:
            # Captura o erro se a imagem não puder ser identificada e exibe uma mensagem de erro
            self.mostrar_mensagem("Erro ao abrir a imagem. O arquivo não é uma imagem válida ou está corrompido.", "Erro")
            return None, None, None  # Retorna None se ocorrer um erro

        except Exception as e:
            # Captura qualquer outro erro, exibe uma mensagem de erro e continua
            self.mostrar_mensagem(f"Erro ao processar a imagem: {e}", "Erro")
            return None, None, None  # Retorna None se ocorrer um erro

    def exportar_para_dxf(self):
        """
        Exporta a camada selecionada para um arquivo DXF. A função realiza as seguintes etapas:

        1. Obtém a seleção atual no QTreeView.
        2. Verifica se alguma camada foi selecionada. Se não, exibe uma mensagem de erro.
        3. Obtém o nome da camada selecionada.
        4. Verifica se a camada selecionada está presente no projeto. Se não, exibe uma mensagem de erro.
        5. Obtém os nomes dos campos da camada.
        6. Verifica se os valores dos campos contêm caracteres inválidos.
        7. Se forem encontrados campos inválidos, notifica o usuário e encerra a função.
        8. Cria um novo documento DXF.
        9. Inicializa o diálogo de escolha de blocos e configurações.
        10. Executa o diálogo e obtém as configurações escolhidas pelo usuário.
        11. Chama a função para exportar a camada para DXF com as configurações selecionadas.

        Funcionalidades:
        - Verificação da seleção e existência da camada.
        - Verificação de caracteres inválidos nos valores dos campos.
        - Inicialização e execução de um diálogo de escolha de configurações.
        - Exportação da camada para um arquivo DXF com as configurações selecionadas.
        """
        # Obtém a seleção atual no QTreeView
        indexes = self.dlg.treeViewListaPonto.selectionModel().selectedIndexes()
        if not indexes:
            self.mostrar_mensagem("Selecione uma camada para exportar.", "Erro")
            return

        # Obtém o nome da camada selecionada
        selected_layer_name = self.treeViewModel.itemFromIndex(indexes[0]).text()
        layer = QgsProject.instance().mapLayersByName(selected_layer_name)[0]
        if not layer:
            self.mostrar_mensagem("Camada não encontrada.", "Erro")
            return

        # Obtém os nomes dos campos da camada
        campos = [field.name() for field in layer.fields()]

        # Dicionário para armazenar campos com valores inválidos
        campos_invalidos = {}

        # Definição de quais caracteres são considerados inválidos
        caracteres_invalidos = set("/|\*?%$#@!~'")

        # Função para verificar se o valor é inválido
        def valor_invalido(valor):
            if valor is None or valor == "":  # Checa se o valor é nulo ou vazio
                return True
            if any(char in caracteres_invalidos for char in str(valor)):
                return True
            return False  # Retorna False se o valor for válido
        
        # Verificação de cada feição e cada campo relevante
        for feature in layer.getFeatures():
            for campo in campos:
                valor = feature[campo]
                if valor_invalido(valor):  # Usa a função valor_invalido para verificar
                    if campo not in campos_invalidos:
                        campos_invalidos[campo] = []
                    campos_invalidos[campo].append(feature.id())

        # Se foram encontrados campos inválidos, notifica o usuário e encerra a função
        if campos_invalidos:
            mensagem_erro = "Valores inválidos ou vazios encontrados nos seguintes campos:\n"
            for campo, ids in campos_invalidos.items():
                mensagem_erro += f"{campo} (IDs: {', '.join(map(str, ids))})\n"
            self.mostrar_mensagem(mensagem_erro, "Erro")
            return

        doc = ezdxf.new(dxfversion='R2013')  # Cria um novo documento DXF
        cores = {}  # Inicializa um dicionário de cores
        self.dialogo = BlocoEscolhaDialogo(campos, layer, self.dlg)  # A instância é criada aqui e atribuída a self.dialogo
        dialogo = BlocoEscolhaDialogo(campos, layer, self.dlg)  # Inicializa o diálogo de escolha
        # nomes_blocos = dialogo.criar_blocos(doc, cores)  # Cria blocos no documento DXF

        resultado = dialogo.exec_()  # Executa o diálogo
        if resultado == QDialog.Accepted:
            cores = dialogo.cores  # Atualiza cores com as escolhas do usuário
            nomes_blocos = self.dialogo.criar_blocos(doc, cores)  # Chama o método criar_blocos da instância self.dialogo
            campoEscolhido = dialogo.getCampoEscolhido()  # Obtém o campo escolhido no diálogo
            selecoes = dialogo.getSelecoes()  # Obtém as seleções feitas no diálogo
            camposSelecionados = dialogo.getCamposSelecionados() if hasattr(dialogo, 'camposCheckBoxes') else []  # Obter campos selecionados se existir
            campoZ = dialogo.getCampoZ()  # Obter o campo Z selecionado
            blocoSelecionado = dialogo.getBlocoSelecionado()  # Obter o bloco selecionado

            if campoEscolhido and selecoes:  # Chama a função para exportar a camada para DXF
                self.exportar_camada_para_dxf(layer, campoEscolhido, selecoes, cores, camposSelecionados, campoZ, blocoSelecionado)

    def exportar_camada_para_dxf(self, layer, campoEscolhido, selecoes, cores, camposSelecionados, campoZ, blocoSelecoes):
        """
        Exporta a camada selecionada para um arquivo DXF. A função realiza as seguintes etapas:

        1. Obtém o nome da camada e solicita ao usuário o caminho do arquivo para salvar o DXF.
        2. Cria um novo documento DXF.
        3. Cria camadas no documento DXF com as cores escolhidas.
        4. Inicia a barra de progresso.
        5. Itera sobre cada feição na camada e adiciona pontos, textos e blocos ao documento DXF.
        6. Salva o documento DXF no caminho especificado.
        7. Calcula o tempo de execução.
        8. Remove a barra de progresso e exibe uma mensagem de sucesso.
        9. Cria um índice espacial para a camada antes da exportação.

        Parâmetros:
        - layer (QgsVectorLayer): A camada de origem para a exportação.
        - campoEscolhido (str): O campo de identificação a ser usado como nome dos Placemarks.
        - selecoes (list): As seleções feitas no diálogo de escolha.
        - cores (dict): Dicionário de cores escolhidas.
        - camposSelecionados (list): Lista de campos selecionados para exportar.
        - campoZ (str): Campo Z para coordenadas 3D, se aplicável.
        - blocoSelecoes (dict): Dicionário de blocos selecionados para cada valor de atributo.

        Funcionalidades:
        - Criação de documento DXF com camadas e estilos personalizados.
        - Adição de pontos, textos e blocos ao documento DXF.
        - Atualização da barra de progresso durante o processamento.
        - Cálculo do tempo de execução e exibição de mensagem de sucesso.
        """

        # Cria um índice espacial para a camada
        spatial_index = QgsSpatialIndex()
        for feature in layer.getFeatures():
            spatial_index.insertFeature(feature)

        # Obtém o nome da camada
        nome_camada = layer.name()
        caminho_arquivo = self.escolher_local_para_salvar(f"{nome_camada}.dxf", "DXF Files (*.dxf)")
        if not caminho_arquivo:
            return  # Usuário cancelou a operação

        doc = ezdxf.new(dxfversion='R2013')
        msp = doc.modelspace()

        self.nomes_blocos = self.dialogo.criar_blocos(doc, cores)

        # Criando camadas com as cores escolhidas no documento DXF
        for valor_atributo, cor_rgb in cores.items():
            cor_int = rgb2int(cor_rgb)
            doc.layers.new(name=str(valor_atributo), dxfattribs={'true_color': cor_int})

        altura_texto = 0.5
        deslocamento_y = altura_texto * 1.2  # Ajuste o deslocamento_y para a distância vertical entre as linhas de texto
        deslocamento_x = altura_texto * (-2.7)  # O deslocamento_x é ajustado para centralizar o texto em relação ao ponto

        # Inicia a barra de progresso
        progressBar, progressMessageBar = self.iniciar_progress_bar(layer)

        start_time = time.time()

        for i, feature in enumerate(layer.getFeatures(), start=1):
            valor_atributo = feature[campoEscolhido]
            if valor_atributo in selecoes:
                geom = feature.geometry()
                if geom is not None:
                    point = geom.asPoint()
                    x, y = point.x(), point.y()
                    z = feature[campoZ] if campoZ else 0  # Usar valor Z se campoZ estiver definido

                    # Adiciona o ponto (em 3D se z estiver definido)
                    if campoZ:
                        msp.add_point((x, y, z), dxfattribs={'layer': str(valor_atributo)})
                    else:
                        msp.add_point((x, y), dxfattribs={'layer': str(valor_atributo)})

                    # Adiciona a referência ao bloco selecionado para este valor de atributo (em 2D)
                    blocoSelecionado = blocoSelecoes.get(valor_atributo)
                    if blocoSelecionado and blocoSelecionado in self.nomes_blocos:
                        msp.add_blockref(blocoSelecionado, (x, y), dxfattribs={
                            'layer': str(valor_atributo),
                            'insert': (x, y),
                            'xscale': 1.0,
                            'yscale': 1.0,
                            'zscale': 1.0,
                            'rotation': 0
                        })

                    # O texto é colocado acima do ponto inicialmente (em 2D)
                    y_offset = y + (altura_texto / 2)
                    x_offset = x - (deslocamento_x / 2)  # Ajusta o X para centralizar o texto
                    for campo in camposSelecionados:
                        texto = feature[campo]
                        msp.add_text(str(texto), dxfattribs={
                            'insert': (x_offset, y_offset),
                            'height': altura_texto,
                            'layer': str(valor_atributo)
                        })
                        y_offset -= deslocamento_y  # Move para baixo para o próximo texto

            progressBar.setValue(i)  # Atualiza a barra de progresso
        
        # Salva o documento DXF no caminho especificado
        doc.saveas(caminho_arquivo)
        end_time = time.time()

        execution_time = end_time - start_time

        # Remove a barra de progresso
        self.iface.messageBar().clearWidgets()

        # Exibir mensagem de sucesso com o tempo de execução e caminhos dos arquivos
        self.mostrar_mensagem(
            f"Arquivo DXF salvo com sucesso em {execution_time:.2f} segundos", 
            "Sucesso", 
            caminho_pasta=os.path.dirname(caminho_arquivo), 
            caminho_arquivos=caminho_arquivo)

    def _processar_kml_kmz_pontos(self, file_path: str, layer_name: str) -> QgsVectorLayer:
        """
        Processa arquivos KML ou KMZ, mantendo apenas feições do tipo ponto (Point/MultiPoint),
        copiando-as para uma camada em memória com os campos:
        [ID, X, Y, X_DMS, Y_DMS] + outros (exceto campos_para_remover).
        Para MultiPoint, cada ponto vira uma feição independente.
        """
        origem = QgsVectorLayer(file_path, layer_name, "ogr")
        if not origem.isValid():
            self.mostrar_mensagem("Falha ao carregar KML/KMZ.", "Erro")
            return None

        campos_para_remover = {
            "description", "timestamp", "begin", "end",
            "altitudeMode", "tessellate", "extrude", "drawOrder",
            "visibility", "icon"}

        campos_fixos = [
            QgsField("ID", QVariant.Int),
            QgsField("X", QVariant.Double),
            QgsField("Y", QVariant.Double),
            QgsField("X_DMS", QVariant.String),
            QgsField("Y_DMS", QVariant.String)]

        campos_origem = [f for f in origem.fields()
            if f.name() not in campos_para_remover and f.name() not in [c.name() for c in campos_fixos]]

        uri = f"Point?crs={origem.crs().authid()}"
        mem = QgsVectorLayer(uri, f"{layer_name}", "memory")
        prov = mem.dataProvider()
        prov.addAttributes(campos_fixos + campos_origem)
        mem.updateFields()

        feats = []
        idx = 1
        for feat in origem.getFeatures():
            geom = feat.geometry()
            if QgsWkbTypes.geometryType(geom.wkbType()) == QgsWkbTypes.PointGeometry:
                # Feição simples do tipo ponto
                pt = geom.asPoint()
                f = QgsFeature()
                f.setGeometry(QgsGeometry.fromPointXY(pt))
                x = round(pt.x(), 6 if mem.crs().isGeographic() else 3)
                y = round(pt.y(), 6 if mem.crs().isGeographic() else 3)
                x_dms = self.calcular_posicao_xy(pt.x())
                y_dms = self.calcular_posicao_xy(pt.y())
                attrs = [idx, x, y, x_dms, y_dms]
                attrs += [feat[field.name()] for field in campos_origem]
                f.setAttributes(attrs)
                feats.append(f)
                idx += 1

            elif QgsWkbTypes.geometryType(geom.wkbType()) == QgsWkbTypes.PointGeometry and geom.isMultipart():
                # MultiPoint → cada ponto gera uma feição independente
                for pt in geom.asMultiPoint():
                    f = QgsFeature()
                    f.setGeometry(QgsGeometry.fromPointXY(pt))
                    x = round(pt.x(), 6 if mem.crs().isGeographic() else 3)
                    y = round(pt.y(), 6 if mem.crs().isGeographic() else 3)
                    x_dms = self.calcular_posicao_xy(pt.x())
                    y_dms = self.calcular_posicao_xy(pt.y())
                    attrs = [idx, x, y, x_dms, y_dms]
                    attrs += [feat[field.name()] for field in campos_origem]
                    f.setAttributes(attrs)
                    feats.append(f)
                    idx += 1

        prov.addFeatures(feats)
        mem.updateExtents()
        return mem

    def _copiar_para_memoria_ponto(self, src: QgsVectorLayer, new_name: str) -> QgsVectorLayer:
        """
        Cria uma cópia em memória da camada `src`, mantendo apenas feições de ponto.

        O que esta função faz:
        1. Cria uma camada vetorial em memória, com o mesmo tipo de geometria (Point) e CRS da camada original.
        2. Copia todos os campos (atributos) da camada original para a camada em memória.
        3. Filtra e copia apenas as feições cuja geometria é do tipo ponto (Point ou MultiPoint).
        4. Copia a simbologia (renderer) da camada original para manter o mesmo estilo visual.
        5. Retorna a nova camada criada em memória.
        
        Parâmetros:
            src      : QgsVectorLayer
                Camada de origem, pode conter vários tipos de feição (serão filtrados apenas pontos).
            new_name : str
                Nome desejado para a camada em memória.

        Retorno:
            QgsVectorLayer
                Nova camada de pontos criada em memória.
        """

        # Define URI da camada (tipo de geometria + CRS)
        mem_uri = f"{QgsWkbTypes.displayString(src.wkbType())}?crs={src.crs().authid()}"
        mem = QgsVectorLayer(mem_uri, new_name, "memory")
        mem_dp = mem.dataProvider()

        # Copia todos os campos da camada original
        mem_dp.addAttributes(src.fields())
        mem.updateFields()

        # Filtra e copia apenas feições do tipo ponto
        feats = [f for f in src.getFeatures()
                 if QgsWkbTypes.geometryType(f.geometry().wkbType()) == QgsWkbTypes.PointGeometry]
        mem_dp.addFeatures(feats)
        mem.updateExtents()

        # Copia a simbologia do layer original (mantém visual)
        mem.setRenderer(src.renderer().clone())

        return mem

    @staticmethod
    def carregar_texto_dxf(path_dxf: str, nome_layer: str,
                          crs_authid: str = "EPSG:4326") -> QgsVectorLayer | None:
        """
        Lê um arquivo DXF e, se houver entidades de texto, cria uma camada de pontos em memória
        rotulada com o conteúdo e a formatação original, com suporte à cor ByLayer.

        Funcionalidades principais:
        1. Valida o caminho do arquivo DXF.
        2. Lê todas as entidades do tipo TEXT e MTEXT do modelspace do DXF usando ezdxf.
        3. Cria uma camada de pontos em memória (QGIS) com os campos:
            - text: conteúdo do texto.
            - height: altura do texto (quando disponível).
            - rotation: rotação (em graus, ajustada para sentido do QGIS).
            - x, y: coordenadas de inserção.
            - r, g, b: cor do texto (detecta true-color, ByLayer ou ACI, com fallback para preto em ByBlock).
        4. Monta um dicionário de cores das layers do DXF, para suportar textos com cor ByLayer.
        5. Para cada entidade, extrai atributos, converte e ajusta rotação, determina cor conforme prioridade:
            a. True-color direto (rgb);
            b. ByLayer (herda cor da layer do DXF);
            c. ByBlock ou sem cor definida: preto;
            d. Cor direta (ACI - AutoCAD Color Index).
        6. Adiciona todos os textos como feições à camada de pontos.
        7. Configura a camada para exibir rótulos (labels) com rotação e cor dinâmicas por campo.
        8. Torna o símbolo de ponto invisível para que apenas o texto apareça no mapa.

        Parâmetros:
            path_dxf   : str
                Caminho completo do arquivo DXF a ser lido.
            nome_layer : str
                Nome desejado para a camada criada.
            crs_authid : str (opcional)
                Código de referência espacial (ex: "EPSG:4326") para o sistema de coordenadas.

        Retorno:
            QgsVectorLayer | None
                Camada de pontos QGIS com textos e propriedades originais, ou None se não houver textos.
        """
        # Verifica existência do arquivo DXF
        if not path_dxf or not os.path.isfile(path_dxf):
            raise FileNotFoundError(f"Caminho DXF inválido: {path_dxf!r}")

        doc = ezdxf.readfile(path_dxf)
        msp = doc.modelspace()
        entities = list(msp.query("TEXT MTEXT"))
        if not entities:
            return None  # Sem entidades de texto, retorna None

        # Cria camada de pontos em memória
        vl = QgsVectorLayer(f"Point?crs={crs_authid}", nome_layer, "memory")
        pr = vl.dataProvider()

        # Define os campos que serão utilizados na camada
        pr.addAttributes([
            QgsField("text",     QVariant.String),
            QgsField("height",   QVariant.Double),
            QgsField("rotation", QVariant.Double),
            QgsField("x",        QVariant.Double),
            QgsField("y",        QVariant.Double),
            QgsField("r",        QVariant.Int),
            QgsField("g",        QVariant.Int),
            QgsField("b",        QVariant.Int)])
        vl.updateFields()

        # Monta dicionário de cores das layers (para suporte ao ByLayer)
        dxf_layers = {}
        for lay in doc.layers:
            aci = lay.color
            if hasattr(lay, 'rgb') and lay.rgb is not None:
                rgb = lay.rgb
            else:
                rgb = aci2rgb(aci)
            dxf_layers[lay.dxf.name] = rgb

        feats = []
        for ent in entities:
            # Obtém o texto (TEXT ou MTEXT)
            txt = ent.plain_text() if ent.dxftype() == "MTEXT" else ent.dxf.text
            ins = ent.dxf.insert  # Ponto de inserção
            rot = getattr(ent.dxf, "rotation", 0.0)  # Rotação bruta

            if rot is None:
                rot = 0.0
            # Converte rotação para graus, se necessário
            if rot > 2 * math.pi:
                rot = rot
            else:
                rot = math.degrees(rot)
            rot = -rot  # Ajusta sinal para QGIS

            ht = getattr(ent.dxf, "height", None)  # Altura do texto

            # === Determina a cor ===
            if hasattr(ent, 'rgb') and ent.rgb is not None:
                r, g, b = ent.rgb  # True-color
            else:
                aci = ent.dxf.color
                if aci == 256:  # ByLayer: pega cor do layer do DXF
                    layer_name = ent.dxf.layer
                    r, g, b = dxf_layers.get(layer_name, (0, 0, 0))
                elif aci == 0:  # ByBlock (ou indefinido): preto
                    r, g, b = (0, 0, 0)
                else:
                    r, g, b = aci2rgb(aci)  # ACI normal

            # Cria feature com atributos
            f = QgsFeature(vl.fields())
            f.setAttributes([txt, ht, rot, ins.x, ins.y, r, g, b])
            f.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(ins.x, ins.y)))
            feats.append(f)

        # Adiciona feições à camada
        if feats:
            pr.addFeatures(feats)
            vl.updateExtents()
            vl.triggerRepaint()

        # Configura os rótulos da camada
        pal = QgsPalLayerSettings()  # Cria configurações de rótulos
        pal.fieldName = "text"  # Define o campo usado para os rótulos (conteúdo do texto)
        pal.placement = QgsPalLayerSettings.OverPoint  # Define o posicionamento dos rótulos sobre os pontos

        # Configura a rotação dos rótulos com base no campo "rotation"
        pal.dataDefinedProperties().setProperty(
            QgsPalLayerSettings.Rotation,
            QgsProperty.fromField("rotation"))  # Vincula o campo "rotation" à propriedade de rotação

        # Configura o posicionamento dos rótulos com base no campo "rotation"
        expression_rotation = f'"rotation"'  # Usa o campo "rotation" diretamente como expressão
        pal.dataDefinedProperties().setProperty(
            QgsPalLayerSettings.LabelRotation,
            QgsProperty.fromExpression(expression_rotation))  # Aplica a rotação à posição dos rótulos

        # Configura a cor dos rótulos com base nos campos r, g, b
        pal.dataDefinedProperties().setProperty(
            QgsPalLayerSettings.Color,
            QgsProperty.fromExpression('color_rgba("r","g","b",255)'))  # Define a cor usando RGBA

        # Aplica as configurações de rótulos à camada
        vl.setLabeling(QgsVectorLayerSimpleLabeling(pal))
        vl.setLabelsEnabled(True)  # Habilita os rótulos

        # Define um símbolo de ponto "invisível" para evitar sobreposição visual
        symbol = QgsMarkerSymbol.createSimple({"name": "circle", "size": "0"})  # Cria símbolo circular com tamanho 0
        vl.renderer().setSymbol(symbol)  # Define o símbolo para o renderizador da camada

        return vl  # Retorna a camada vetorial criada

    def _processar_dxf_pontos(self, file_path: str, layer_name: str, manter_original: bool = False) -> QgsVectorLayer | None:
        """
        Lê um DXF e devolve uma camada de pontos (Point ou PointZ) em memória,
        com campos [ID, X, Y, Z] + demais campos relevantes.
        """

        # Carrega via OGR apenas entidades Point
        ogr_layer = QgsVectorLayer(
            f"{file_path}|layername=entities|geometrytype=Point",
            f"{layer_name}_pt", "ogr")

        if not ogr_layer.isValid():
            self.mostrar_mensagem("Falha ao carregar PONTOS do DXF.", "Erro")
            return None

        # Carrega o DXF bruto via ezdxf para obter valores Z
        try:
            dxf_doc = ezdxf.readfile(file_path)
        except Exception as e:
            self.mostrar_mensagem(f"Erro ao abrir o DXF: {e}", "Erro")
            return None

        msp = dxf_doc.modelspace()

        # Lista de coordenadas (x, y, z) na MESMA ordem em que OGR costuma devolver
        coords_z = [(float(ent.dxf.location.x),
                     float(ent.dxf.location.y),
                     float(ent.dxf.location.z))
                    for ent in msp.query("POINT")]

        # Garante um CRS válido
        if not ogr_layer.crs().isValid():
            dlg = QgsProjectionSelectionDialog()
            if not dlg.exec_():
                self.mostrar_mensagem("Operação cancelada pelo usuário.", "Aviso")
                return None
            chosen = dlg.crs()
            if not chosen.isValid():
                self.mostrar_mensagem("Nenhum SRC selecionado – operação cancelada.", "Aviso")
                return None
            ogr_layer.setCrs(chosen)

        # Define campos
        skip_fields   = {"PaperSpace", "SubClasses", "Linetype", "EntityHandle", "Text"}
        campos_fixos  = [
            QgsField("ID", QVariant.Int),
            QgsField("X",  QVariant.Double),
            QgsField("Y",  QVariant.Double),
            QgsField("Z",  QVariant.Double)]
        campos_ogr = [f for f in ogr_layer.fields()
            if f.name() not in skip_fields and f.name() not in {c.name() for c in campos_fixos}]

        # Decide se a camada será Point ou PointZ
        tem_z = any(abs(z) > 1e-9 for _, _, z in coords_z)  # tolerância
        geom_str = "PointZ" if tem_z else "Point"
        uri = f"{geom_str}?crs={ogr_layer.crs().authid()}"

        mem = QgsVectorLayer(uri, f"{layer_name}_pt", "memory")
        prov = mem.dataProvider()
        prov.addAttributes(campos_fixos + campos_ogr)
        mem.updateFields()

        # Copia feições, mantendo atributos OGR
        feats_out = []
        for idx, (ogr_feat, xyz) in enumerate(
            zip_longest(ogr_layer.getFeatures(), coords_z), start=1):
            if ogr_feat is None or xyz is None:                       # segurança
                continue
            x, y, z_val = xyz

            # Cria geometria 2D ou 3D conforme necessário
            if tem_z:
                geom = QgsGeometry.fromPoint(QgsPoint(x, y, z_val))
            else:
                geom = QgsGeometry.fromPointXY(QgsPointXY(x, y))

            new_feat = QgsFeature(mem.fields())
            new_feat.setGeometry(geom)

            # Monta lista de atributos:
            attrs = [idx, x, y, z_val]
            attrs += [ogr_feat[field.name()] for field in campos_ogr]
            new_feat.setAttributes(attrs)

            feats_out.append(new_feat)

        prov.addFeatures(feats_out)
        mem.updateExtents()

        return mem

    def abrir_adicionar_arquivo_ponto(self):
        """
        Importa arquivos de pontos (DXF, KML, KMZ, SHP, GeoJSON, GPKG) como
        camadas em memória, evitando duplicidades.

        · DXF  →  cria camada de pontos e, se houver TEXT/MTEXT, cria também
                   a camada de textos rotulados.
        · KML/KMZ → filtra apenas Point/MultiPoint.
        · Outros  → copia somente feições de ponto da fonte.
        """
        file_filter = (
            "All Supported Formats (*.dxf *.kml *.kmz *.shp *.geojson *.gpkg);;"
            "DXF Files (*.dxf);;KML Files (*.kml);;KMZ Files (*.kmz);;"
            "Shapefiles (*.shp);;GeoJSON Files (*.geojson);;Geopackage Files (*.gpkg)")
        file_path, _ = QFileDialog.getOpenFileName(self.dlg, "Abrir Arquivo de Pontos", "", file_filter)
        if not file_path:
            return

        layer_name = os.path.splitext(os.path.basename(file_path))[0]
        ext        = os.path.splitext(file_path)[1].lower()

        pt_layer   = None
        txt_layer  = None  # só usado para DXF

        # ---------- DXF --------------------------------------------------------
        if ext == ".dxf":
            # pontos
            pt_layer = self._processar_dxf_pontos(file_path, layer_name)
            # textos (pode retornar None se não houver TEXT/MTEXT)
            try:
                txt_layer = self.carregar_texto_dxf(
                    file_path, f"{layer_name}_txt", pt_layer.crs().authid() if pt_layer else "EPSG:4326")
            except Exception as exc:
                self.mostrar_mensagem(f"Erro ao ler TEXTOS do DXF: {exc}", "Erro")

        # ---------- KML / KMZ --------------------------------------------------
        elif ext in (".kml", ".kmz"):
            pt_layer = self._processar_kml_kmz_pontos(file_path, layer_name)

        # ---------- SHP / GeoJSON / GPKG ---------------------------------------
        else:
            origem = QgsVectorLayer(file_path, layer_name, "ogr")
            if not origem.isValid():
                self.mostrar_mensagem("Falha ao carregar o arquivo vetorial.", "Erro")
                return
            pt_layer = self._copiar_para_memoria_ponto(origem, layer_name)
            del origem

        # ---------- Validação --------------------------------------------------
        if pt_layer is None or not pt_layer.isValid():
            self.mostrar_mensagem("Nenhuma feição de ponto encontrada.", "Aviso")
            return

        # ---------- Adiciona camada de pontos (evita duplicata) ----------------
        for lyr in list(QgsProject.instance().mapLayers().values()):
            if lyr.name() == pt_layer.name():
                QgsProject.instance().removeMapLayer(lyr.id())
        QgsProject.instance().addMapLayer(pt_layer)
        node = QgsProject.instance().layerTreeRoot().findLayer(pt_layer.id())
        if node:
            node.setItemVisibilityChecked(True)

        # ---------- Adiciona camada de textos, se existir ----------------------
        if txt_layer and txt_layer.isValid():
            for lyr in list(QgsProject.instance().mapLayers().values()):
                if lyr.name() == txt_layer.name():
                    QgsProject.instance().removeMapLayer(lyr.id())
            QgsProject.instance().addMapLayer(txt_layer)
            node_txt = QgsProject.instance().layerTreeRoot().findLayer(txt_layer.id())
            if node_txt:
                node_txt.setItemVisibilityChecked(True)

        # ---------- Feedback & UI ---------------------------------------------
        self.mostrar_mensagem(f"Camada de pontos '{pt_layer.name()}' adicionada com sucesso.", "Sucesso")
        self.atualizar_treeView_lista_ponto()

class CloneManagerPt:
    def __init__(self, ui_manager, layer_to_clone):
        """
        Inicializa o gerenciador de clonagem de camadas de pontos.

        Parâmetros:
        - ui_manager: Referência ao gerenciador da interface do usuário (geralmente responsável por mensagens, atualizações e utilidades da UI).
        - layer_to_clone: A camada de pontos (QgsVectorLayer) que será clonada.

        Detalhes:
        - Armazena as referências para permitir o acesso a métodos utilitários e manipulação da camada durante o processo de clonagem.
        """

        self.ui = ui_manager                   # Referência à interface de gerenciamento
        self.layer = layer_to_clone            # Camada de pontos a ser clonada (QgsVectorLayer)

    def show_clone_options(self):
        """
        Exibe um diálogo para o usuário escolher o tipo de clonagem para a camada de pontos.

        Funcionalidades:
        - Cria um diálogo modal com quatro opções de clonagem, cada uma apresentada como botão de rádio estilizado.
        - Disponibiliza opções para: clonar só feições, copiar tabela de atributos, combinar tabela e tratar pontos, excluir tabela e tratar pontos.
        - Centraliza os botões de ação (OK/Cancelar) na janela.
        - Após a seleção e confirmação do usuário, chama o método de clonagem correspondente à opção escolhida.
        """

        # Cria o diálogo modal como filho do diálogo principal da interface
        dialog = QDialog(self.ui.dlg)
        dialog.setWindowTitle("Clonar Camada de Pontos")
        layout = QVBoxLayout(dialog)               # Layout vertical para os elementos do diálogo
        button_group = QButtonGroup(dialog)        # Grupo para garantir seleção única dos rádios

        # Define as opções de clonagem via QRadioButton
        self.radio_buttons = {
            1: QRadioButton("Clonar Feição, sem Tabela"),
            2: QRadioButton("Copiar Tabela de Atributos"),
            3: QRadioButton("Combinar Tabela e Tratar Pontos"),
            4: QRadioButton("Excluir Tabela e Tratar Pontos")}

        # Adiciona cada botão de rádio dentro de um QFrame estilizado
        for id, radio in self.radio_buttons.items():
            frame = QFrame(dialog)
            frame_layout = QVBoxLayout(frame)
            frame.setFrameShape(QFrame.StyledPanel)
            frame.setFrameShadow(QFrame.Raised)
            frame.setFixedSize(220, 30)                # Tamanho para melhor alinhamento visual
            frame_layout.addWidget(radio)
            layout.addWidget(frame)
            button_group.addButton(radio, id)          # Adiciona ao grupo para controle de seleção
            if id == 1:
                radio.setChecked(True)                 # Primeira opção selecionada por padrão

        # Adiciona caixa de botões "Ok" e "Cancelar" ao diálogo
        buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttonBox.accepted.connect(dialog.accept)      # OK fecha e aceita
        buttonBox.rejected.connect(dialog.reject)      # Cancelar fecha sem ação
        layout.addWidget(buttonBox)

        # Layout horizontal para centralizar os botões de ação
        hLayout = QHBoxLayout()
        hLayout.addStretch()
        hLayout.addWidget(buttonBox)
        hLayout.addStretch()
        layout.addLayout(hLayout)

        # Executa o diálogo e verifica a ação do usuário
        result = dialog.exec_()
        if result == QDialog.Accepted:
            # Percorre os botões para encontrar o selecionado
            for id, radio in self.radio_buttons.items():
                if radio.isChecked():
                    self.realizar_clonagem(id)         # Executa a clonagem conforme a escolha
                    break

    def criar_camada_clonada(self, suf, fields=None, feats=None, copy_renderer=True):
        """
        Cria uma nova camada de pontos clonada em memória, baseada em uma camada de referência.

        Funcionalidades:
        - Define o nome da nova camada com um sufixo e o nome original.
        - Mantém o mesmo SRC/CRS (sistema de referência de coordenadas) da camada original.
        - Permite especificar campos (QgsFields) e feições (lista de QgsFeature), ou copiar todos da camada original.
        - Copia a simbologia/renderizador da camada original, se solicitado.
        - Adiciona a camada ao grupo "Camadas Clonadas" no projeto QGIS.
        - Retorna a camada criada para possíveis manipulações posteriores.

        Parâmetros:
        - suf: Sufixo a ser adicionado ao nome da nova camada.
        - fields: (Opcional) Estrutura de campos (QgsFields) da camada.
        - feats: (Opcional) Lista de feições (QgsFeature) a serem adicionadas à nova camada.
        - copy_renderer: (bool) Se True, copia a simbologia/renderizador da camada original.

        Retorna:
        - clone: A camada de pontos criada (QgsVectorLayer), ou None se houver erro.
        """

        # Define o nome da nova camada com o sufixo e o nome original
        name = f"{suf}_{self.layer.name()}"
        # Recupera o CRS da camada original
        crs = self.layer.crs().authid()
        # Cria a camada em memória com tipo Point e mesmo CRS
        clone = QgsVectorLayer(f"Point?crs={crs}", name, "memory")

        # Verifica se a camada foi criada corretamente
        if not clone.isValid():
            QMessageBox.critical(None, "Erro", "Não foi possível criar a camada clonada.")
            return None

        # Adiciona os campos (estrutura de atributos) à nova camada
        prov = clone.dataProvider()
        prov.addAttributes((fields or self.layer.fields()).toList())
        clone.updateFields()

        # Adiciona as feições (geometrias e atributos) à nova camada
        prov.addFeatures(feats or list(self.layer.getFeatures()))

        # Copia simbologia/renderizador da camada original, se solicitado
        if copy_renderer:
            clone.setRenderer(self.layer.renderer().clone())

        # Procura (ou cria) o grupo "Camadas Clonadas" na árvore do projeto
        root = QgsProject.instance().layerTreeRoot()
        grp = root.findGroup("Camadas Clonadas") or root.addGroup("Camadas Clonadas")

        # Adiciona a nova camada ao grupo no projeto, mas não a exibe imediatamente
        QgsProject.instance().addMapLayer(clone, False)
        grp.addLayer(clone)

        return clone  # Retorna a camada criada para manipulação posterior

    def realizar_clonagem(self, tipo):
        """
        Executa a clonagem da camada de pontos conforme o tipo selecionado pelo usuário.

        Parâmetros:
        - tipo: (int) Número correspondente à opção de clonagem escolhida:
            1 - Clonar apenas as feições (somente geometria, sem atributos)
            2 - Clonar todos os atributos e geometrias (cópia completa)
            3 - Combinar tabela e tratar campos específicos (adiciona/calcula campos especiais)
            4 - Excluir tabela e tratar apenas campos essenciais (campos mínimos)

        Funcionamento:
        - Usa um dicionário para mapear o tipo à função correspondente.
        - Executa diretamente a função mapeada conforme o tipo selecionado.
        """

        # Seleciona e executa a função correspondente ao tipo de clonagem escolhido
        {1: self.clone_feicao,
         2: self.clone_atributos,
         3: self.clone_combinar,
         4: self.clone_excluir}[tipo]()

    def clone_feicao(self):
        """
        Clona somente as geometrias (feições) da camada de pontos original,
        sem copiar nenhum atributo (tabela vazia).

        Funcionalidades:
        - Cria novas feições mantendo apenas a geometria de cada ponto.
        - Não inclui atributos além dos padrões mínimos obrigatórios (estrutura vazia).
        - Cria uma nova camada em memória com essas feições clonadas.
        - Não copia a tabela de atributos da camada original.

        Fluxo:
        1. Percorre todas as feições da camada original, copiando apenas as geometrias.
        2. Cria uma camada clonada sem campos (tabela de atributos vazia).
        3. Adiciona todas as novas feições (com geometria apenas) à nova camada.
        """
        feats = []
        # Percorre todas as feições da camada original
        for f in self.layer.getFeatures():
            nf = QgsFeature()              # Cria uma nova feição vazia
            nf.setGeometry(f.geometry())   # Copia apenas a geometria
            feats.append(nf)               # Adiciona à lista de feições clonadas

        # Cria camada clonada sem tabela de atributos, apenas geometrias
        self.criar_camada_clonada("Clone_SemTabela", fields=QgsFields(), feats=feats)

    def clone_atributos(self):
        """
        Clona a camada de pontos mantendo todas as geometrias e atributos originais.

        Funcionalidades:
        - Copia todos os campos (estrutura da tabela de atributos) da camada original.
        - Para cada feição, clona tanto a geometria quanto todos os atributos.
        - Cria uma nova camada em memória idêntica à original.
        - Exibe mensagem de sucesso ao usuário ao finalizar.

        Processo:
        1. Percorre todas as feições da camada de pontos original.
        2. Cria novas feições com os mesmos campos e atributos, copiando também a geometria.
        3. Cria uma nova camada clonada com esses campos e feições.
        4. Informa o usuário que a clonagem foi concluída.
        """

        feats = []
        # Percorre todas as feições da camada original
        for f in self.layer.getFeatures():
            nf = QgsFeature(self.layer.fields())   # Cria nova feição com mesma estrutura de campos
            nf.setGeometry(f.geometry())           # Copia a geometria
            nf.setAttributes(f.attributes())       # Copia todos os atributos
            feats.append(nf)                       # Adiciona à lista de feições clonadas

        # Cria a nova camada clonada com campos e feições completos
        self.criar_camada_clonada("Clone_Tabela", fields=self.layer.fields(), feats=feats)

        # Exibe mensagem de sucesso ao usuário
        self.ui.mostrar_mensagem("Clonagem completa concluída.", "Sucesso")

    def clone_combinar(self):
        """
        Clona a camada de pontos combinando campos essenciais ("ID", "X", "Y", "X_DMS", "Y_DMS") 
        e todos os outros campos originais (sem duplicidade), já preenchendo os valores calculados 
        para cada ponto.

        Funcionalidades:
        - Garante que os campos essenciais existam na camada clonada, criando-os se necessário.
        - Copia todos os demais campos da camada original, exceto os essenciais para evitar duplicidade.
        - Calcula e atribui valores para ID sequencial, coordenadas X e Y, e suas versões DMS.
        - Copia os demais atributos conforme presentes na camada original.
        - Chama tratamento adicional para pontos (sinais para atualização dinâmica de X, Y, DMS).
        - Exibe mensagem de sucesso ao usuário.

        Processo:
        1. Define a estrutura de campos da nova camada, priorizando os essenciais.
        2. Percorre todas as feições, calcula valores necessários e copia demais atributos.
        3. Cria a nova camada clonada e ativa o tratamento dinâmico para atualização dos campos especiais.
        4. Notifica o usuário do sucesso.
        """

        base = self.layer.fields()
        keep = {"ID", "X", "Y", "X_DMS", "Y_DMS"}
        fields = QgsFields()
        # Garante que os campos essenciais estejam presentes na camada
        for fld in ["ID", "X", "Y", "X_DMS", "Y_DMS"]:
            if base.indexOf(fld) == -1:
                # Cria campo se não existir
                if fld == "ID":
                    fields.append(QgsField("ID", QVariant.Int))
                else:
                    fields.append(QgsField(fld, QVariant.Double if fld in {"X", "Y"} else QVariant.String))
            else:
                # Usa campo original se já existir
                fields.append(base.field(fld))
        # Adiciona todos os outros campos originais que não sejam essenciais
        for f in base:
            if f.name() not in keep:
                fields.append(f)

        feats = []
        # Para cada feição, calcula e preenche os valores dos campos essenciais e copia demais atributos
        for idx, f in enumerate(self.layer.getFeatures(), start=1):
            nf = QgsFeature(fields)
            nf.setGeometry(f.geometry())
            atr = [None] * len(fields)
            pt = f.geometry().asPoint()
            X = round(pt.x(), 6 if self.layer.crs().isGeographic() else 3)
            Y = round(pt.y(), 6 if self.layer.crs().isGeographic() else 3)
            atr[fields.indexOf("ID")] = idx          # ID sequencial
            atr[fields.indexOf("X")] = X             # Coordenada X
            atr[fields.indexOf("Y")] = Y             # Coordenada Y
            # Preenche os campos DMS se existirem na estrutura
            if fields.indexOf("X_DMS") != -1:
                atr[fields.indexOf("X_DMS")] = self.ui.calcular_posicao_xy(pt.x())
            if fields.indexOf("Y_DMS") != -1:
                atr[fields.indexOf("Y_DMS")] = self.ui.calcular_posicao_xy(pt.y())
            # Copia demais atributos
            for i, fld in enumerate(base):
                if fld.name() not in keep:
                    atr[fields.indexOf(fld.name())] = f[i]
            nf.setAttributes(atr)
            feats.append(nf)

        # Cria a camada clonada com os campos e feições preparados
        clone = self.criar_camada_clonada("Clone_Combinado", fields=fields, feats=feats)
        # Ativa o tratamento de pontos na nova camada (sinais para atualização dinâmica)
        self.ui.tratar_pontos(clone)
        # Exibe mensagem de sucesso ao usuário
        self.ui.mostrar_mensagem("Combinação de tabela + tratamento concluída.", "Sucesso")

    def clone_excluir(self):
        """
        Cria uma camada clonada de pontos contendo apenas os campos essenciais:
        "ID", "X", "Y", "X_DMS", "Y_DMS" – todos os demais atributos são descartados.

        Funcionalidades:
        - Cria uma nova camada de pontos apenas com campos mínimos e valores calculados.
        - Para cada ponto, calcula as coordenadas X e Y (em decimal) e em DMS.
        - Define um ID sequencial para cada ponto.
        - Não copia a simbologia/renderizador da camada original.
        - Ativa o tratamento de atualização dinâmica dos campos calculados.
        - Exibe mensagem de sucesso ao usuário ao finalizar.

        Processo:
        1. Define os campos essenciais na nova camada.
        2. Percorre todas as feições da camada original, calculando valores para os campos essenciais.
        3. Cria uma nova camada em memória com esses campos e feições.
        4. Ativa sinais dinâmicos para atualizar os campos em futuras edições.
        5. Notifica o usuário do sucesso.
        """

        fields = QgsFields()
        fields.append(QgsField("ID", QVariant.Int))             # Campo ID sequencial
        fields.append(QgsField("X", QVariant.Double))           # Coordenada X (decimal)
        fields.append(QgsField("Y", QVariant.Double))           # Coordenada Y (decimal)
        fields.append(QgsField("X_DMS", QVariant.String))       # Coordenada X em DMS
        fields.append(QgsField("Y_DMS", QVariant.String))       # Coordenada Y em DMS

        feats = []
        # Para cada ponto, calcula valores essenciais e cria nova feição
        for idx, f in enumerate(self.layer.getFeatures(), start=1):
            nf = QgsFeature(fields)
            nf.setGeometry(f.geometry())
            pt = f.geometry().asPoint()
            X = round(pt.x(), 6 if self.layer.crs().isGeographic() else 3)
            Y = round(pt.y(), 6 if self.layer.crs().isGeographic() else 3)
            nf.setAttribute("ID", idx)                                  # ID sequencial
            nf.setAttribute("X", X)                                     # Coordenada X decimal
            nf.setAttribute("Y", Y)                                     # Coordenada Y decimal
            nf.setAttribute("X_DMS", self.ui.calcular_posicao_xy(pt.x())) # X em DMS
            nf.setAttribute("Y_DMS", self.ui.calcular_posicao_xy(pt.y())) # Y em DMS
            feats.append(nf)

        # Cria camada clonada apenas com os campos essenciais e feições calculadas
        clone = self.criar_camada_clonada("Clone_Excluir", fields=fields, feats=feats, copy_renderer=False)
        # Ativa sinais de atualização automática para os campos na nova camada
        self.ui.tratar_pontos(clone)
        # Exibe mensagem de sucesso ao usuário
        self.ui.mostrar_mensagem("Exclusão de atributos + tratamento concluída.", "Sucesso")

class TreeViewEventFilter(QObject):
    """
    Filtro de eventos personalizado para detectar movimentos do mouse sobre itens em um treeView.

    Esta classe herda de QObject e implementa um filtro de eventos que detecta quando o mouse se move
    sobre itens específicos em um treeView. Quando o mouse se move sobre um item, a classe chama um 
    método no UiManager para exibir um tooltip com informações sobre o item.

    Parâmetros:
    - ui_manager: Referência à instância do objeto UiManager, que gerencia a interface do usuário.
    """

    def __init__(self, ui_manager):
        """
        Inicializa o filtro de eventos com uma referência ao UiManager.

        Parâmetros:
        - ui_manager: Instância do UiManager que será usada para acessar e manipular a interface do usuário.
        """
        super().__init__()  # Inicializa a classe base QObject
        self.ui_manager = ui_manager  # Armazena a referência ao UiManagerT para uso posterior

    def eventFilter(self, obj, event):
        """
        Filtra os eventos de movimentação do mouse sobre o treeView e exibe tooltips quando aplicável.

        Esta função intercepta eventos que ocorrem no treeView especificado. Se o evento for de movimento
        do mouse (QEvent.MouseMove) e o mouse estiver sobre um item válido no treeView, a função chama
        o método 'configurar_tooltip' do UiManager para exibir um tooltip com informações sobre o item.

        Parâmetros:
        - obj: O objeto que está sendo monitorado (neste caso, o viewport do treeView).
        - event: O evento que está sendo filtrado (como QEvent.MouseMove).

        Retorno:
        - bool: O resultado da chamada à função 'eventFilter' da classe base, indicando se o evento foi processado.
        """
        # Verifica se o objeto é o viewport do treeView e se o evento é de movimento do mouse
        if obj == self.ui_manager.dlg.treeViewListaPonto.viewport() and event.type() == QEvent.MouseMove:
            # Obtém o índice do item no treeView sob o cursor do mouse
            index = self.ui_manager.dlg.treeViewListaPonto.indexAt(event.pos())
            if index.isValid():  # Verifica se o índice é válido (se o mouse está sobre um item)
                self.ui_manager.configurar_tooltip(index)  # Chama o método para configurar e exibir o tooltip
        # Retorna o resultado padrão do filtro de eventos
        return super().eventFilter(obj, event)  # Chama a implementação da classe base para continuar o processamento normal

class BlocoEscolhaDialogo(QDialog):
    """
    Diálogo para escolher campos e valores para exportação de pontos. Este diálogo permite ao usuário selecionar campos,
    definir cores, e escolher blocos para exportação em um formato específico.

    Parâmetros:
    - campos (list): Lista de nomes de campos disponíveis na camada.
    - layer (QgsVectorLayer): A camada de origem.
    - nomes_blocos (list): Lista de nomes de blocos disponíveis.
    - parent (QWidget): Widget pai do diálogo (opcional).

    Funcionalidades:
    - Seleção de campos de texto para exportação.
    - Definição de cores aleatórias para os valores selecionados.
    - Seleção de blocos específicos para cada valor.
    - Verificação de condições para habilitar/desabilitar o botão de exportação.
    """
    def __init__(self, campos, layer, parent=None):
        """
        Inicializa uma instância do diálogo BlocoEscolhaDialogo.
        
        Funções principais:
        - Configura a interface do usuário do diálogo.
        - Inicializa variáveis e estruturas de dados para armazenar informações sobre os campos e cores selecionados.
        - Configura e organiza os layouts e widgets do diálogo.

        Etapas detalhadas:
        1. Inicializa a superclasse QDialog.
        2. Armazena os parâmetros `campos` e `layer`.
        3. Cria um dicionário para armazenar as cores selecionadas.
        4. Inicializa um dicionário para armazenar checkboxes dos campos.
        5. Chama o método `criar_blocos` para inicializar os blocos.
        6. Inicializa um dicionário para armazenar o estado dos campos, todos desmarcados.
        7. Inicializa o QButtonGroup para botões de rádio.
        8. Configura o título, tamanho mínimo e máximo do diálogo.
        9. Configura o layout principal do diálogo.
        10. Configura e organiza widgets e layouts adicionais dentro do diálogo.
        11. Configura a lista de widgets para exibir valores de campos.
        12. Adiciona botões de controle (OK e Cancelar) ao diálogo.
        13. Inicializa o QListWidget com os valores do campo string.
        14. Embaralha uma lista de cores vibrantes.
        15. Verifica as condições para habilitar/desabilitar o botão Executar.
        16. Conecta sinais de checkboxes às funções de atualização de estado.

        Parâmetros:
        - campos (list): Lista de nomes dos campos.
        - layer (QgsVectorLayer): Camada de vetor do QGIS.
        - parent (QWidget): Widget pai opcional.
        """
        super().__init__(parent)
        self.campos = campos # Armazena os campo
        self.layer = layer # Armazena a camada
        self.cores = {}  # Dicionário para armazenar as cores selecionadas
        self.camposCheckBoxes = {} # Dicionário para armazenar checkboxes dos campos
        # Inicialização dos blocos  # Adicionando a lista de nomes de blocos
        self.nomes_blocos = self.criar_blocos(ezdxf.new(dxfversion='R2013'), {})
        self.estadoCampos = {campo: False for campo in campos}  # Inicializa todos os campos como desmarcados
        
        # Inicialize radioGroup no construtor
        self.radioGroup = QButtonGroup(self)        

        self.setWindowTitle('Escolha o Campo e o Valor para os Pontos') # Define o título da janela
        self.setMinimumSize(325, 400)  # Define o tamanho mínimo do diálogo
        self.setMaximumSize(325, 600)  # Define o tamanho máximo do diálogo

        layout = QVBoxLayout(self) # Layout principal vertical

        self.frame = QFrame(self) # Cria um frame
        self.frame.setFrameShape(QFrame.StyledPanel) # Define o formato do frame
        self.frame.setLineWidth(int(0.6)) # Define a largura da linha do frame
        frameLayout = QVBoxLayout(self.frame) # Layout vertical para o frame

        self.comboBoxLayout = QHBoxLayout() # Layout horizontal para o comboBox
        self.label = QLabel('Selecione o campo:') # Label para o comboBox
        self.comboBoxLayout.addWidget(self.label) # Adiciona o label ao layout

        self.comboBox = QComboBox(self.frame) # Cria o comboBox
        # Adicionar apenas campos do tipo string ao comboBox
        for campo in self.campos:
            field = self.layer.fields().field(campo)  # Obtém o campo
            # Verifica se o campo é do tipo string
            if field.type() in (QVariant.String, QVariant.StringList): 
                self.comboBox.addItem(campo)
        self.comboBox.setStyleSheet("""
        QComboBox { combobox-popup: 0; }
        QComboBox QAbstractItemView {
            min-height: 80px; /* 10 itens */
            max-height: 80px; /* 10 itens */
            min-width: 100px; /* Ajuste conforme necessário */
        }
        """) # Define o estilo do comboBox

        # Conecta o sinal de mudança de índice à função atualizarListWidget
        self.comboBox.currentIndexChanged.connect(self.atualizarListWidget)
        self.comboBoxLayout.addWidget(self.comboBox) # Adiciona o comboBox ao layout
        frameLayout.addLayout(self.comboBoxLayout) # Adiciona o layout do comboBox ao layout do frame

        # Layout horizontal para o checkbox e o botão de cores aleatórias
        topLayout = QHBoxLayout() # Layout horizontal para o topo
        frameLayout.addLayout(topLayout) # Adiciona o layout ao frame

        # Adiciona QCheckBox para selecionar todos
        self.selectAllCheckBox = QCheckBox("Selecionar Camadas") # Cria o checkbox para selecionar todos
        self.selectAllCheckBox.setChecked(False) # Define o estado inicial como desmarcado
        # Conecta o sinal de mudança de estado à função selecionarTodos
        self.selectAllCheckBox.stateChanged.connect(self.selecionarTodos)
        topLayout.addWidget(self.selectAllCheckBox) # Adiciona o checkbox ao layout

        # Layout horizontal para os botões "Rótulos" e "3D"
        buttonsLayout = QHBoxLayout() # Layout horizontal para os botões
        frameLayout.addLayout(buttonsLayout) # Adiciona o layout ao frame

        # Adiciona botão para abrir a lista de campos numéricos para 3D
        self.btn3D = QPushButton("Altimetria") # Cria o botão "Altimetria"
        # Conecta o sinal de clique à função mostrarCamposNumericos
        self.btn3D.clicked.connect(self.mostrarCamposNumericos)
        buttonsLayout.addWidget(self.btn3D) # Adiciona o botão ao layout

        # Botão "Rótulos"
        self.camposButton = QPushButton("Rótulos")
        # Conecta o sinal de clique à função mostrarCampos
        self.camposButton.clicked.connect(self.mostrarCampos) 
        buttonsLayout.addWidget(self.camposButton) # Adiciona o botão ao layout

        # Adiciona botão para definir cores aleatórias
        self.randomColorButton = QPushButton("Cores Aleatórias")
        # Conecta o sinal de clique à função definirCoresAleatorias
        self.randomColorButton.clicked.connect(self.definirCoresAleatorias)
        topLayout.addWidget(self.randomColorButton) # Adiciona o botão ao layout

        # Layout para o QLineEdit e o novo botão
        lineEditLayout = QHBoxLayout() # Layout horizontal para o QLineEdit
        self.lineEdit = QLineEdit(self.frame) # Cria o QLineEdit
        self.lineEdit.setReadOnly(True)  # Torna o QLineEdit somente leitura
        self.lineEdit.setPlaceholderText("Feições: Selecione!") # Define o texto de espaço reservado
        self.lineEdit.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed) # Define a política de tamanho
        lineEditLayout.addWidget(self.lineEdit) # Adiciona o QLineEdit ao layout

        # Criação do botão "Bloco Aleatório"
        self.blocoAleatorioButton = QPushButton("Bloco Aleatório")
        # Conecta o sinal de clique à função definirBlocoAleatorio
        self.blocoAleatorioButton.clicked.connect(self.definirBlocoAleatorio)
        lineEditLayout.addWidget(self.blocoAleatorioButton) # Adiciona o botão ao layout

        # Adicionando o layout do QLineEdit e botão ao layout principal
        frameLayout.addLayout(lineEditLayout)

        self.listWidget = QListWidget(self.frame) # Cria o QListWidget
        self.listWidget.setMinimumSize(200, 220) # Define o tamanho mínimo do QListWidget
        # Conecta o sinal de mudança de seleção à função atualizarLineEdit
        self.listWidget.itemSelectionChanged.connect(self.atualizarLineEdit)
        frameLayout.addWidget(self.listWidget)  # Adiciona o QListWidget ao frame

        # Adiciona os botões OK e Cancelar
        self.buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, self)
        self.buttonBox.button(QDialogButtonBox.Ok).setText("Exportar")  # Altera o texto do botão OK para Exportar
        self.buttonBox.accepted.connect(self.accept)  # Se Exportar for clicado, aceita o diálogo
        self.buttonBox.rejected.connect(self.reject)  # Se Cancelar for clicado, rejeita o diálogo

        # Centraliza o buttonBox no layout
        frameLayout.addWidget(self.buttonBox, 0, Qt.AlignCenter)  # Adiciona o buttonBox ao layout com alinhamento centralizado

        layout.addWidget(self.frame) # Adiciona o frame ao layout principal

        # Isto carregará os valores do campo string no QListWidget
        self.atualizarListWidget(0)

        # Embaralha uma lista de cores vibrantes
        self.cores_vibrantes_embaralhadas = self.embaralhar_cores_vibrantes()

        self.verificarCondicoesParaExecutar()  # Chama a função verificarCondicoesParaExecutar

        # Conecta o sinal de mudança de estado à função atualizarEstadoBotaoExportar
        self.selectAllCheckBox.stateChanged.connect(self.atualizarEstadoBotaoExportar)
        # Conecta o sinal de mudança de estado à função verificarCondicoesParaExecutar
        self.selectAllCheckBox.stateChanged.connect(self.verificarCondicoesParaExecutar)

    def definirBlocoAleatorio(self):
        """
        Define um bloco aleatório para cada item no QListWidget. A função realiza as seguintes etapas:

        1. Cria uma cópia da lista de blocos disponíveis.
        2. Inicializa uma lista para armazenar os blocos já utilizados.
        3. Percorre cada item do QListWidget:
           - Obtém o texto do QLabel e o QComboBox associados ao item.
           - Verifica se o texto do label corresponde a algum nome de bloco.
           - Se houver correspondência, seleciona o bloco correspondente e o remove da lista de blocos disponíveis.
           - Se não houver correspondência, escolhe um bloco aleatório dos disponíveis.
           - Atualiza as listas de controle de blocos disponíveis e utilizados.

        Funcionalidades:
        - Seleção automática de blocos para itens em uma lista.
        - Evita a repetição de blocos até que todos tenham sido utilizados.
        """
        blocos_disponiveis = self.nomes_blocos.copy()  # Cria uma cópia da lista de blocos disponíveis
        blocos_utilizados = []  # Lista para armazenar os blocos já utilizados

        # Percorre cada item do QListWidget
        for i in range(self.listWidget.count()):
            item = self.listWidget.item(i)
            widget = self.listWidget.itemWidget(item)
            label = widget.findChild(QLabel).text()  # Obtém o texto do QLabel
            combo = widget.findChild(QComboBox)
            
            # Verifica se o nome no label corresponde a algum nome de bloco
            bloco_correspondente = next((bloco for bloco in self.nomes_blocos if bloco.lower() == label.lower()), None)

            if bloco_correspondente:
                # Se houver correspondência, seleciona o bloco correspondente
                combo.setCurrentText(bloco_correspondente)
                # Remove o bloco correspondente da lista de disponíveis, se necessário
                if bloco_correspondente in blocos_disponiveis:
                    blocos_disponiveis.remove(bloco_correspondente)
                    blocos_utilizados.append(bloco_correspondente)
            else:
                # Se não houver blocos disponíveis, reinicia a lista
                if not blocos_disponiveis:
                    blocos_disponiveis = blocos_utilizados.copy()
                    blocos_utilizados.clear()

                # Escolhe um bloco aleatoriamente dos disponíveis
                bloco_aleatorio = random.choice(blocos_disponiveis)
                combo.setCurrentText(bloco_aleatorio)
                # Atualiza as listas de controle
                blocos_disponiveis.remove(bloco_aleatorio)
                blocos_utilizados.append(bloco_aleatorio)

    def atualizarEstadoBotaoExportar(self):
        """
        Atualiza o estado do botão Exportar com base no estado do checkbox Selecionar Camadas.
        A função realiza as seguintes etapas:

        1. Verifica o estado do checkbox Selecionar Camadas.
        2. Habilita ou desabilita o botão Exportar de acordo com o estado do checkbox.

        Funcionalidades:
        - Habilitação e desabilitação dinâmica do botão Exportar.
        """
        # Atualiza o estado do botão Exportar com base no checkbox Selecionar Camadas
        self.buttonBox.button(QDialogButtonBox.Ok).setEnabled(self.selectAllCheckBox.isChecked())

    def verificarCondicoesParaExecutar(self):
        """
        Verifica se todas as condições necessárias para habilitar o botão Executar são satisfeitas.
        A função realiza as seguintes etapas:

        1. Verifica se a camada possui feições.
        2. Verifica se a camada possui campos.
        3. Verifica se o checkbox "Selecionar Camadas" está marcado.
        4. Habilita o botão Executar somente se todas as condições forem verdadeiras.

        Funcionalidades:
        - Verificação da presença de feições na camada.
        - Verificação da presença de campos na camada.
        - Verificação do estado do checkbox "Selecionar Camadas".
        - Habilitação do botão Executar apenas quando todas as condições são atendidas.
        """
        # Verifica se a camada possui feições
        tem_feicoes = self.layer.featureCount() > 0

        # Verifica se a camada possui campos
        tem_campos = len(self.campos) > 0

        # Verifica se o checkBox "Selecionar Camadas" está marcado
        selecao_ativa = self.selectAllCheckBox.isChecked()

        # Habilita o botão Executar somente se todas as condições forem verdadeiras
        self.buttonBox.button(QDialogButtonBox.Ok).setEnabled(tem_feicoes and tem_campos and selecao_ativa)

    def mostrarCamposNumericos(self):
        """
        Exibe um diálogo para selecionar campos numéricos para altimetria. A função realiza as seguintes etapas:

        1. Cria e configura o diálogo.
        2. Adiciona botões de rádio para campos numéricos da camada.
        3. Configura a área de rolagem para acomodar muitos campos.
        4. Adiciona botões OK e Cancelar ao diálogo.
        5. Atualiza o texto do botão "3D" com base na seleção de campos numéricos.
        6. Exibe o diálogo.

        Funcionalidades:
        - Seleção de campos numéricos para altimetria.
        - Atualização dinâmica do texto do botão "3D".
        """

        # Cria o diálogo para seleção de campos numéricos
        self.campos3DDialog = QDialog(self)
        self.campos3DDialog.setWindowTitle("Campos para Altimetria")
        self.campos3DDialog.resize(150, 150)  # Define o tamanho do diálogo
        
        dialogLayout = QVBoxLayout(self.campos3DDialog)  # Layout vertical para o diálogo

        campos3DWidget = QWidget()  # Cria um widget para conter os campos numéricos
        campos3DLayout = QVBoxLayout(campos3DWidget)  # Layout vertical para os campos numéricos

        # Adiciona botões de rádio para cada campo numérico na camada
        for campo in self.campos:
            if self.layer.fields().field(campo).isNumeric():  # Verifica se o campo é numérico
                radioButton = QRadioButton(campo)  # Cria um botão de rádio para o campo
                self.radioGroup.addButton(radioButton)  # Adiciona o botão de rádio ao grupo de botões
                campos3DLayout.addWidget(radioButton)  # Adiciona o botão de rádio ao layout de campos numéricos
                # Conecta o sinal toggled de cada radioButton ao método de atualização
                radioButton.toggled.connect(self.atualizarTextoBotao3D)  # Conecta o sinal toggled ao método de atualização do texto do botão 3D

        scrollArea = QScrollArea(self.campos3DDialog)  # Cria uma área de rolagem para o diálogo
        scrollArea.setWidgetResizable(True)  # Define a área de rolagem como redimensionável
        scrollArea.setWidget(campos3DWidget)  # Define o widget de campos numéricos como o widget da área de rolagem
        dialogLayout.addWidget(scrollArea)  # Adiciona a área de rolagem ao layout do diálogo

        buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, self.campos3DDialog)  # Cria botões OK e Cancelar para o diálogo
        buttonBox.accepted.connect(self.campos3DDialog.accept)  # Conecta o botão OK ao método de aceitação do diálogo
        buttonBox.rejected.connect(self.campos3DDialog.reject)  # Conecta o botão Cancelar ao método de rejeição do diálogo
        dialogLayout.addWidget(buttonBox)  # Adiciona o buttonBox ao layout do diálogo

        # Atualiza o texto do botão "3D" ao exibir o diálogo
        self.atualizarTextoBotao3D()  # Atualiza o texto do botão 3D com base na seleção atual de campos numéricos

        self.campos3DDialog.exec_()  # Exibe o diálogo

    def atualizarTextoBotao3D(self):
        """
        Atualiza o texto e o estilo do botão "Altimetria" com base na seleção de campos numéricos.
        A função realiza as seguintes etapas:

        1. Verifica se algum botão de rádio no grupo está selecionado.
        2. Se algum botão estiver selecionado, altera o texto do botão para "Altimetria ✓" e muda a cor para verde.
        3. Se nenhum botão estiver selecionado, retorna o texto e o estilo do botão ao padrão.

        Funcionalidades:
        - Atualização dinâmica do texto e estilo do botão "Altimetria" com base na seleção de campos numéricos.
        """

        # Verifica se algum botão está selecionado e atualiza o texto do botão "3D"
        if any(button.isChecked() for button in self.radioGroup.buttons()):
            self.btn3D.setText("Altimetria ✓")  # Altera o texto do botão para indicar seleção
            self.btn3D.setStyleSheet("color: blue;")  # Muda a cor do texto do botão para verde
        else:
            self.btn3D.setText("Altimetria")  # Retorna o texto do botão ao padrão
            self.btn3D.setStyleSheet("")  # Retorna o estilo do botão ao padrão

    def atualizarEstadoCampos(self, campo, checked):
        """
        Atualiza o estado de seleção de um campo específico.

        Parâmetros:
        - campo (str): O nome do campo cujo estado está sendo atualizado.
        - checked (bool): O novo estado do campo (True se selecionado, False se desmarcado).

        Funcionalidades:
        - Atualiza o dicionário self.estadoCampos com o novo estado do campo.
        """

        # Atualiza o estado de seleção do campo específico
        self.estadoCampos[campo] = checked

    def mostrarCampos(self):
        """
        Exibe um diálogo para selecionar campos da tabela de atributos. A função realiza as seguintes etapas:

        1. Cria e configura o diálogo.
        2. Adiciona checkboxes para cada campo da camada, restaurando o estado anterior de seleção.
        3. Configura a área de rolagem para acomodar muitos campos.
        4. Adiciona botões OK e Cancelar ao diálogo.
        5. Exibe o diálogo.

        Funcionalidades:
        - Seleção de campos para exportação.
        - Atualização do estado dos checkboxes com base na seleção anterior.
        - Verificação dinâmica da seleção de checkboxes.
        """

        # Cria o diálogo para seleção de campos
        self.camposDialog = QDialog(self)
        self.camposDialog.setWindowTitle("Campos da Tabela de Atributos")  # Define o título do diálogo
        self.camposDialog.resize(150, 150)  # Define o tamanho do diálogo
        
        dialogLayout = QVBoxLayout(self.camposDialog)  # Layout vertical para o diálogo

        camposWidget = QWidget()  # Cria um widget para conter os checkboxes
        camposLayout = QVBoxLayout(camposWidget)  # Layout vertical para os checkboxes

        self.camposCheckBoxes = {}  # Dicionário para armazenar os checkboxes
        for campo in self.campos:
            checkBox = QCheckBox(campo)  # Cria um checkbox para cada campo
            checkBox.setChecked(self.estadoCampos[campo])  # Restaura o estado anterior de seleção
            checkBox.stateChanged.connect(lambda checked, c=campo: self.atualizarEstadoCampos(c, checked))  # Conecta o estado do checkbox ao método de atualização de estado
            checkBox.stateChanged.connect(self.verificarSelecao)  # Conecta o estado do checkbox ao método de verificação de seleção
            self.camposCheckBoxes[campo] = checkBox  # Armazena o checkbox no dicionário
            camposLayout.addWidget(checkBox)  # Adiciona o checkbox ao layout de campos

        scrollArea = QScrollArea(self.camposDialog)  # Cria uma área de rolagem para o diálogo
        scrollArea.setWidgetResizable(True)  # Define a área de rolagem como redimensionável
        scrollArea.setWidget(camposWidget)  # Define o widget de campos como o widget da área de rolagem
        dialogLayout.addWidget(scrollArea)  # Adiciona a área de rolagem ao layout do diálogo

        buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, self.camposDialog)  # Cria botões OK e Cancelar para o diálogo
        buttonBox.accepted.connect(self.camposDialog.accept)  # Conecta o botão OK ao método de aceitação do diálogo
        buttonBox.rejected.connect(self.camposDialog.reject)  # Conecta o botão Cancelar ao método de rejeição do diálogo
        dialogLayout.addWidget(buttonBox)  # Adiciona o buttonBox ao layout do diálogo

        self.camposDialog.exec_()  # Exibe o diálogo

    def verificarSelecao(self):
        """
        Verifica a seleção dos checkboxes representando os campos de atributos na interface gráfica.
        A função realiza as seguintes etapas:

        1. Calcula a quantidade de checkboxes selecionados.
        2. Atualiza o texto e a cor do botão 'Rótulos' com base na quantidade de seleções.
        3. Desabilita checkboxes não selecionados se três ou mais campos estiverem selecionados.
        4. Habilita todos os checkboxes se menos de três campos estiverem selecionados.

        Funcionalidades:
        - Verificação dinâmica da seleção de checkboxes.
        - Atualização visual do botão 'Rótulos' com base na seleção.
        - Gerenciamento do estado (habilitado/desabilitado) dos checkboxes com base na quantidade de seleções.
        """
        # Calcula a quantidade de checkboxes selecionados
        selecionados = sum(checkBox.isChecked() for checkBox in self.camposCheckBoxes.values())
        # Atualiza o texto e a cor do botão 'Rótulos' com base na quantidade de seleções
        if selecionados > 0:
            self.camposButton.setText("Rótulos ✓") # Adiciona um checkmark se houver seleção
            self.camposButton.setStyleSheet("QPushButton { color: green; }") # Muda a cor para verde
        else:
            self.camposButton.setText("Rótulos")  # Texto padrão sem seleção
            self.camposButton.setStyleSheet("")  # Retorna ao estilo padrão

        # Se três ou mais campos estão selecionados, desabilita os checkboxes não selecionados
        if selecionados >= 3:
            for checkBox in self.camposCheckBoxes.values():
                if not checkBox.isChecked():
                    checkBox.setDisabled(True) # Desabilita checkboxes não selecionados
        else:
            # Se menos de três campos estão selecionados, habilita todos os checkboxes
            for checkBox in self.camposCheckBoxes.values():
                checkBox.setEnabled(True) # Habilita todos os checkboxes
    @staticmethod
    def embaralhar_cores_vibrantes():
        """
        Embaralha uma lista de cores vibrantes. A função realiza as seguintes etapas:

        1. Define uma lista de cores vibrantes usando objetos QColor.
        2. Embaralha aleatoriamente a lista de cores.
        3. Retorna a lista embaralhada de cores.

        Funcionalidades:
        - Criação de uma lista de cores vibrantes predefinidas.
        - Embaralhamento aleatório da lista de cores.
        - Retorno da lista de cores embaralhadas.

        Retorna:
        - list: Lista de cores vibrantes embaralhadas.
        """

        cores_vibrantes = [
            QColor(255, 0, 0),  # Vermelho
            QColor(0, 255, 0),  # Verde
            QColor(0, 0, 255),  # Azul
            QColor(255, 255, 0),  # Amarelo
            QColor(0, 255, 255),  # Ciano
            QColor(255, 0, 255),  # Magenta
            QColor(255, 165, 0),  # Laranja
            QColor(128, 0, 128),  # Roxo
            # Adicione mais cores conforme necessário
        ]
        random.shuffle(cores_vibrantes)  # Embaralha a lista de cores
        return cores_vibrantes  # Retorna a lista embaralhada

    def definirCoresAleatorias(self):
        # Embaralha uma lista de cores vibrantes
        cores_vibrantes = self.embaralhar_cores_vibrantes()
        cores_iter = iter(cores_vibrantes)  # Cria um iterador para a lista de cores

        # Percorre cada item no QListWidget
        for i in range(self.listWidget.count()):
            item = self.listWidget.item(i)
            widget = self.listWidget.itemWidget(item)
            
            # Encontrando o botão e o label no widget
            button = widget.findChildren(QPushButton)[0]
            label = widget.findChildren(QLabel)[0]
            
            try:
                cor_aleatoria = next(cores_iter)  # Obtém a próxima cor do iterador
            except StopIteration:
                # Se chegar ao final da lista de cores, comece novamente
                cores_iter = iter(cores_vibrantes)
                cor_aleatoria = next(cores_iter)

            # Definindo a cor do botão e do texto do label
            button.setStyleSheet(f"QPushButton {{ background-color: {cor_aleatoria.name()}; }}")
            label.setStyleSheet(f"QLabel {{ color: {cor_aleatoria.name()}; font-weight: bold; font-style: italic; background-color: white; border: 1px solid gray; }}")

            valor = button.property('valor_atributo')  # Obtém o valor do atributo associado ao botão
            self.cores[valor] = (cor_aleatoria.red(), cor_aleatoria.green(), cor_aleatoria.blue())  # Armazena a cor selecionada

            # Atualiza o bloco gráfico com a nova cor
            combo = widget.findChildren(QComboBox)[0]  # Encontra o QComboBox no widget do item
            self.atualizarBlocoGrafico(widget, combo, button)

    def selecionarTodos(self, state):
        """
        Seleciona ou desmarca todos os checkboxes no QListWidget com base no estado do checkbox "Selecionar Todos".
        A função realiza as seguintes etapas:

        1. Itera sobre todos os itens no QListWidget.
        2. Para cada item, obtém o widget associado e encontra o QCheckBox.
        3. Define o estado do QCheckBox com base no estado do checkbox "Selecionar Todos".
        4. Atualiza o estado do botão Exportar.

        Funcionalidades:
        - Seleção ou desmarcação de todos os checkboxes no QListWidget.
        - Atualização do estado do botão Exportar com base na seleção.

        Parâmetros:
        - state (Qt.CheckState): O estado do checkbox "Selecionar Todos" (Qt.Checked ou Qt.Unchecked).
        """

        # Itera sobre todos os itens no QListWidget
        for i in range(self.listWidget.count()):
            item = self.listWidget.item(i)
            widget = self.listWidget.itemWidget(item)
            checkbox = widget.findChild(QCheckBox)  # Encontra o QCheckBox no widget do item
            checkbox.setChecked(state == Qt.Checked)  # Define o estado do QCheckBox com base no estado do checkbox "Selecionar Todos"
        
        # Chama a função para atualizar o estado do botão Exportar sempre que selecionar todos é alterado
        self.atualizarEstadoBotaoExportar()

    def selecionarCor(self, button, label):
        """
        Abre um diálogo de seleção de cor e aplica a cor escolhida ao botão e ao label.
        A função realiza as seguintes etapas:

        1. Abre um diálogo de seleção de cor.
        2. Verifica se uma cor válida foi selecionada.
        3. Se uma cor válida foi selecionada:
           - Define a cor de fundo do botão com a cor selecionada.
           - Define a cor do texto e o estilo do label com a cor selecionada.
           - Armazena a cor selecionada em um dicionário de cores.
           - Atualiza a cor do bloco gráfico correspondente.

        Funcionalidades:
        - Seleção de cor pelo usuário.
        - Aplicação da cor selecionada a um botão e um label.
        - Armazenamento da cor selecionada para referência futura.
        - Atualização da cor do bloco gráfico.

        Parâmetros:
        - button (QPushButton): O botão cuja cor de fundo será alterada.
        - label (QLabel): O label cujo texto e estilo serão alterados.
        """

        color = QColorDialog.getColor()  # Abre o diálogo de seleção de cor
        if color.isValid():  # Verifica se uma cor foi selecionada
            button.setStyleSheet(f"QPushButton {{ background-color: {color.name()}; }}")  # Define a cor de fundo do botão
            label.setStyleSheet(f"QLabel {{ color: {color.name()}; font-weight: bold; font-style: italic; background-color: white; border: 1px solid gray; }}")  # Define a cor do texto e o estilo do label
            valor = button.property('valor_atributo')  # Obtém o valor do atributo associado ao botão
            # Armazena a cor como uma tupla RGB
            self.cores[valor] = (color.red(), color.green(), color.blue())  # Armazena a cor selecionada

            # Atualiza a cor do bloco gráfico
            item_widget = button.parentWidget()
            combo = item_widget.findChild(QComboBox)
            self.atualizarBlocoGrafico(item_widget, combo, button)

    def atualizarLineEdit(self):
        """
        Atualiza o texto do QLineEdit com a contagem total de feições do item selecionado no QListWidget.
        A função realiza as seguintes etapas:

        1. Obtém os itens selecionados no QListWidget.
        2. Se houver itens selecionados:
           - Recupera o texto do primeiro item selecionado.
           - Obtém a contagem de feições para o valor selecionado.
           - Atualiza o texto do QLineEdit com a contagem total de feições.
        3. Se não houver itens selecionados, limpa o texto do QLineEdit.

        Funcionalidades:
        - Atualização dinâmica do QLineEdit com informações sobre o item selecionado.
        - Limpeza do QLineEdit quando nenhum item está selecionado.
        """

        selectedItems = self.listWidget.selectedItems()  # Obtém os itens selecionados no QListWidget
        if selectedItems:
            valor = selectedItems[0].text()  # Recupera o texto do primeiro item selecionado
            contagem = self.valorContagem.get(valor, 0)  # Obtém a contagem para esse valor
            self.lineEdit.setText(f"Total de Feições: {contagem}")  # Atualiza o QLineEdit com a contagem total de feições
        else:
            self.lineEdit.clear()  # Limpa o texto do QLineEdit

    def verificarCheckBoxes(self):
        """
        Verifica o estado dos checkboxes e atualiza o estado do checkbox "Selecionar Todos" e do botão Exportar.
        A função realiza as seguintes etapas:

        1. Verifica se todos os checkboxes estão selecionados.
        2. Verifica se algum checkbox está selecionado.
        3. Atualiza o estado do checkbox "Selecionar Todos" com base nas seleções dos checkboxes.
        4. Habilita o botão Exportar apenas se algum item estiver selecionado.

        Funcionalidades:
        - Verificação dinâmica da seleção de todos os checkboxes.
        - Atualização do estado do checkbox "Selecionar Todos".
        - Habilitação do botão Exportar com base na seleção dos checkboxes.
        """

        # Verifica se todos os checkboxes estão selecionados
        todos_selecionados = all(
            checkbox.isChecked() for i in range(self.listWidget.count())
            for checkbox in [self.listWidget.itemWidget(self.listWidget.item(i)).findChild(QCheckBox)]
        )

        # Verifica se algum checkbox está selecionado
        algum_selecionado = any(
            checkbox.isChecked() for i in range(self.listWidget.count())
            for checkbox in [self.listWidget.itemWidget(self.listWidget.item(i)).findChild(QCheckBox)]
        )

        self.selectAllCheckBox.blockSignals(True)  # Bloqueia sinais para evitar loops de sinal
        self.selectAllCheckBox.setChecked(todos_selecionados or algum_selecionado)  # Atualiza o estado do checkbox "Selecionar Todos"
        self.selectAllCheckBox.setTristate(algum_selecionado and not todos_selecionados)  # Define o estado tri-state
        self.selectAllCheckBox.blockSignals(False)  # Desbloqueia sinais

        # Habilita o botão Exportar apenas se algum item estiver selecionado
        self.buttonBox.button(QDialogButtonBox.Ok).setEnabled(algum_selecionado)

    def atualizarListWidget(self, index):
        """
        Atualiza o QListWidget com os valores únicos do campo selecionado no comboBox.
        
        Funções principais:
        - Obtém os valores únicos do campo selecionado na camada.
        - Conta as ocorrências de cada valor.
        - Atualiza o QListWidget com os valores únicos.
        - Configura widgets adicionais (QLabel, QCheckBox, QPushButton, QComboBox, QGraphicsView) para cada item.
        - Conecta sinais para interações dos widgets.
        - Desenha o bloco inicial e configura a atualização dos gráficos.

        Etapas detalhadas:
        1. Obtém o nome do campo selecionado no comboBox.
        2. Verifica se o campo selecionado é válido.
        3. Itera sobre as feições da camada para obter os valores únicos e suas contagens.
        4. Limpa o QListWidget.
        5. Adiciona cada valor único ao QListWidget.
        6. Cria e configura widgets adicionais para cada item.
        7. Desenha o bloco inicial e configura a atualização dos gráficos.
        8. Conecta sinais de checkboxes à função de verificação de checkboxes.

        Parâmetros:
        - index (int): Índice do campo selecionado no comboBox.
        """
        # Obtém o nome do campo selecionado no comboBox
        campoSelecionado = self.comboBox.itemText(index)
        if not campoSelecionado:  # Verifica se campoSelecionado é uma string vazia
            self.buttonBox.button(QDialogButtonBox.Ok).setEnabled(False)  # Desativa o botão Executar
            return  # Encerra a função para evitar mais execuções

        valoresUnicos = set()  # Conjunto para armazenar valores únicos
        self.valorContagem = {}  # Dicionário para contar as ocorrências de cada valor

        # Itera sobre as feições da camada para obter os valores únicos e suas contagens
        for feature in self.layer.getFeatures():
            valor = feature[campoSelecionado]
            valoresUnicos.add(str(valor))
            self.valorContagem[str(valor)] = self.valorContagem.get(str(valor), 0) + 1

        self.listWidget.clear() # Limpa o QListWidget

        # Adiciona cada valor único ao QListWidget
        for valor in sorted(valoresUnicos):
            item = QListWidgetItem(valor)
            item.setForeground(QBrush(QColor(255, 255, 255))) # Define a cor do texto do item
            self.listWidget.addItem(item)

            item_widget = QWidget()  # Cria um widget para conter o layout do item
            item_layout = QHBoxLayout(item_widget)  # Layout horizontal para o item
            item_layout.setSpacing(5)  # Define o espaçamento entre os widgets do layout
            item_layout.setContentsMargins(2, 1, 2, 1)  # Define as margens do layout

            # Primeiro, o QLabel com o texto
            label = QLabel(valor)
            label.setStyleSheet("font-weight: bold; font-style: italic; background-color: white; border: 1px solid gray;")
            label.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
            item_layout.addWidget(label)

            # O checkbox
            checkbox = QCheckBox()
            checkbox.setChecked(self.selectAllCheckBox.isChecked())
            checkbox.stateChanged.connect(self.verificarCheckBoxes)  # Conecta o sinal stateChanged à função
            item_layout.addWidget(checkbox)

            # O botão de cor
            button = QPushButton('Cor')
            button.setFixedSize(50, 20)
            button.clicked.connect(lambda checked, b=button, l=label: self.selecionarCor(b, l))
            item_layout.addWidget(button)

            # O combobox
            combo = QComboBox()
            combo.setMaxVisibleItems(5)  # Define o número máximo de itens visíveis
            combo.setStyleSheet("""
            QComboBox { 
                combobox-popup: 0; 
            }
            QComboBox QAbstractItemView {
                min-height: 140px; /* 10 itens */
                max-height: 140px; /* 10 itens */
                min-width: 100px; /* ajuste conforme necessário */
                max-width: 100px; /* ajuste conforme necessário */
            }
            """) # Define o estilo do comboBox

            for nome_bloco in self.nomes_blocos:
                combo.addItem(nome_bloco)  # Adiciona os nomes dos blocos ao QComboBox
            combo.setFixedSize(80, 20)  # Define o tamanho fixo do QComboBox
            item_layout.addWidget(combo)  # Adiciona o QComboBox ao layout do item

            # Adicionando QGraphicsView
            graphics_view = QGraphicsView()
            # Habilita o antialiasing e suavização de transformações
            graphics_view.setRenderHints(QPainter.Antialiasing | QPainter.SmoothPixmapTransform)
            graphics_view.setFixedSize(20, 20) # Define o tamanho fixo do QGraphicsView
            scene = QGraphicsScene()
            graphics_view.setScene(scene) # Define a cena do QGraphicsView
            item_layout.addWidget(graphics_view) # Adiciona o QGraphicsView ao layout

            button.setProperty('valor_atributo', valor)  # Define uma propriedade personalizada no QPushButton

            item_widget.setLayout(item_layout)  # Define o layout do item_widget
            item.setSizeHint(item_widget.sizeHint())  # Define o tamanho do item com base no tamanho do item_widget
            self.listWidget.setItemWidget(item, item_widget)  # Define o widget personalizado para o item

            # Chama a função atualizarBlocoGrafico para desenhar o bloco inicial
            self.atualizarBlocoGrafico(item_widget, combo, button)

            # Conecta o sinal de mudança de índice à função atualizarBlocoGrafico
            combo.currentIndexChanged.connect(lambda index, w=item_widget, c=combo, b=button: self.atualizarBlocoGrafico(w, c, b))

            # Conecta o sinal stateChanged de cada QCheckBox ao método verificarCheckBoxes
            checkbox.stateChanged.connect(self.verificarCheckBoxes)

        # Chama a função verificarCheckBoxes para definir o estado inicial do botão Exportar
        self.verificarCheckBoxes()

    def atualizarBlocoGrafico(self, item_widget, combo, button):
        """
        Atualiza o gráfico do bloco no QGraphicsView com base na seleção do QComboBox e na cor do QPushButton.
        
        Funções principais:
        - Limpa a cena atual no QGraphicsView.
        - Desenha o bloco correspondente à seleção atual do QComboBox.
        - Aplica a cor selecionada ao bloco.
        - Centraliza o bloco na cena do QGraphicsView.
        
        Etapas detalhadas:
        1. Encontra o QGraphicsView no item_widget.
        2. Limpa a cena atual do QGraphicsView.
        3. Obtém o nome do bloco selecionado no QComboBox.
        4. Obtém a cor do QPushButton associado.
        5. Desenha o bloco na cena com a cor selecionada.
        6. Centraliza a cena e ajusta a visualização.

        Parâmetros:
        - item_widget (QWidget): O widget do item contendo o QGraphicsView.
        - combo (QComboBox): O QComboBox com a seleção do bloco.
        - button (QPushButton): O QPushButton com a cor selecionada.
        """
        graphics_view = item_widget.findChild(QGraphicsView)  # Encontra o QGraphicsView no item_widget
        scene = graphics_view.scene()  # Obtém a cena do QGraphicsView
        scene.clear()  # Limpa a cena atual

        nome_bloco = combo.currentText()  # Obtém o nome do bloco selecionado no QComboBox
        cor = button.palette().button().color()  # Obtém a cor do botão
        if nome_bloco:
            self.desenharBloco(scene, nome_bloco, cor)  # Desenha o bloco na cena com a cor selecionada

        # Centralizar a cena
        rect = scene.itemsBoundingRect()  # Obtém o retângulo delimitador dos itens na cena
        scene.setSceneRect(rect)  # Define o retângulo delimitador como o retângulo da cena
        graphics_view.setSceneRect(rect)  # Define o retângulo delimitador como o retângulo da visualização
        graphics_view.fitInView(rect, Qt.KeepAspectRatio)  # Ajusta a visualização para manter a proporção do retângulo

    def desenharBloco(self, scene, nome_bloco, cor):
        """
        Desenha um bloco específico na cena fornecida com base no nome do bloco e na cor fornecida.
        
        Funções principais:
        - Define a cor e a largura da caneta para desenhar.
        - Ajusta a escala dos blocos.
        - Desenha diferentes formas geométricas (círculos, linhas, pontos) para representar vários blocos.
        
        Etapas detalhadas:
        1. Define a cor e a largura da caneta.
        2. Define um fator de escala para ajustar os blocos ao tamanho desejado.
        3. Desenha o bloco específico com base no nome do bloco fornecido.
        4. Cada bloco é desenhado com as formas geométricas apropriadas (círculos, linhas, pontos).

        Parâmetros:
        - scene (QGraphicsScene): A cena onde o bloco será desenhado.
        - nome_bloco (str): O nome do bloco a ser desenhado.
        - cor (QColor): A cor para desenhar o bloco.
        """

        pen = QPen(cor)  # Define a cor da caneta
        pen.setWidth(int(0.80))  # Define a largura da caneta
        scale_factor = 1  # Escala para ajustar os blocos ao tamanho desejado

        if nome_bloco == 'CirculoX':
            # Desenha um círculo com um X dentro
            scene.addEllipse(-0.5 * scale_factor, -0.5 * scale_factor, 1.0 * scale_factor, 1.0 * scale_factor, pen)
            scene.addLine(-0.35 * scale_factor, -0.35 * scale_factor, 0.35 * scale_factor, 0.35 * scale_factor, pen)
            scene.addLine(-0.35 * scale_factor, 0.35 * scale_factor, 0.35 * scale_factor, -0.35 * scale_factor, pen)

        elif nome_bloco == 'CirculoCruz':
            # Desenha um círculo com uma cruz dentro
            scene.addEllipse(-0.5 * scale_factor, -0.5 * scale_factor, 1.0 * scale_factor, 1.0 * scale_factor, pen)
            scene.addLine(-0.5 * scale_factor, 0, 0.5 * scale_factor, 0, pen)
            scene.addLine(0, -0.5 * scale_factor, 0, 0.5 * scale_factor, pen)

        elif nome_bloco == 'CirculoTraço':
            # Desenha um círculo com um traço vertical no centro
            scene.addEllipse(-0.5 * scale_factor, -0.5 * scale_factor, 1.0 * scale_factor, 1.0 * scale_factor, pen)
            scene.addLine(0, 0, 0, -0.6 * scale_factor, pen)

        elif nome_bloco == 'CirculoPonto':
            # Desenha um círculo com um ponto no centro
            scene.addEllipse(-0.5 * scale_factor, -0.5 * scale_factor, 1.0 * scale_factor, 1.0 * scale_factor, pen)
            scene.addEllipse(-0.05 * scale_factor, -0.05 * scale_factor, 0.1 * scale_factor, 0.1 * scale_factor, pen)

        elif nome_bloco == 'Árvore':
            # Desenha a copa da árvore com elipses e o tronco com uma linha
            scene.addEllipse(-0.4 * scale_factor, -0.1 * scale_factor, 0.8 * scale_factor, 0.6 * scale_factor, pen)
            scene.addEllipse(-0.3 * scale_factor, -0.3 * scale_factor, 0.6 * scale_factor, 0.7 * scale_factor, pen)
            scene.addEllipse(-0.2 * scale_factor, -0.2 * scale_factor, 0.4 * scale_factor, 0.7 * scale_factor, pen)
            scene.addLine(0, 0.5 * scale_factor, 0, 1.0 * scale_factor, pen)

        elif nome_bloco == 'Poste':
            # Desenha o poste principal e suas travessas e fios
            scene.addLine(0, 0, 0, -2.4 * scale_factor, pen)
            # Desenha a travessa superior
            scene.addLine(-0.4 * scale_factor, -2.0 * scale_factor, 0.4 * scale_factor, -2.0 * scale_factor, pen)
            # Desenha a travessa inferior
            scene.addLine(-0.8 * scale_factor, -1.6 * scale_factor, 0.8 * scale_factor, -1.6 * scale_factor, pen)
            # Desenha os fios inclinados
            scene.addLine(0, -1.6 * scale_factor, -0.5656 * scale_factor, -1.0344 * scale_factor, pen)
            scene.addLine(0, -1.6 * scale_factor, 0.5656 * scale_factor, -1.0344 * scale_factor, pen)

        elif nome_bloco == 'Bandeira':
            # Desenha a haste da bandeira e um triângulo como bandeira
            scene.addLine(0, 0, 0, -1.5 * scale_factor, pen)

            # Desenha a bandeira como um triângulo maior à esquerda
            base_triangular = -0.85 * scale_factor
            altura_triangular = 0.75 * scale_factor
            pontos_bandeira = [(0, -1.5 * scale_factor), 
                               (base_triangular, -1.5 * scale_factor + altura_triangular / 2), 
                               (0, -1.5 * scale_factor + altura_triangular), 
                               (0, -1.5 * scale_factor)]
            scene.addPolygon(QPolygonF([QPointF(p[0], p[1]) for p in pontos_bandeira]), pen)

            # Desenha a linha horizontal maior na base da haste
            scene.addLine(-0.5 * scale_factor, 0, 0.5 * scale_factor, 0, pen)

        elif nome_bloco == 'Bandeira 2':
            # Desenha a haste da bandeira e um retângulo como bandeira
            scene.addLine(0, 0, 0, -1.5 * scale_factor, pen)

            # Desenha a bandeira como um retângulo à esquerda
            largura_bandeira = -0.75 * scale_factor
            altura_bandeira = 0.6 * scale_factor
            topo_bandeira = -1.5 * scale_factor
            base_bandeira = topo_bandeira + altura_bandeira
            pontos_bandeira = [
                (0, topo_bandeira),
                (largura_bandeira, topo_bandeira),
                (largura_bandeira, base_bandeira),
                (0, base_bandeira),
                (0, topo_bandeira)  # Fecha o ponto voltando ao ponto inicial
            ]
            scene.addPolygon(QPolygonF([QPointF(p[0], p[1]) for p in pontos_bandeira]), pen)

            # Desenha a linha horizontal na base da haste
            scene.addLine(-0.25 * scale_factor, 0, 0.25 * scale_factor, 0, pen)

        elif nome_bloco == 'Cerca':
            # Desenha postes verticais e painéis horizontais para a cerca
            altura_poste = 0.8 * scale_factor  # Altura dos postes da cerca
            largura_painel = 0.1 * scale_factor  # Largura dos painéis horizontais
            numero_postes = 3  # Número de postes na cerca

            # Desenha os postes verticais e os painéis horizontais
            for i in range(numero_postes):
                x_pos = i * largura_painel * 2
                # Desenha o poste vertical
                scene.addLine(x_pos, 0, x_pos, -altura_poste, pen)

                # Desenha os painéis horizontais conectando os postes
                if i < numero_postes - 1:
                    scene.addLine(x_pos, -altura_poste / 2, x_pos + largura_painel * 2, -altura_poste / 2, pen)
                    scene.addLine(x_pos, -altura_poste / 4, x_pos + largura_painel * 2, -altura_poste / 4, pen)
                    scene.addLine(x_pos, -altura_poste * 3 / 4, x_pos + largura_painel * 2, -altura_poste * 3 / 4, pen)

        elif nome_bloco == 'Rocha':
            # Desenha a rocha como uma forma poligonal
            pontos_rocha = [
                (-0.5 * scale_factor, 0),
                (-0.3 * scale_factor, 0.3 * scale_factor),
                (0, 0.5 * scale_factor),
                (0.3 * scale_factor, 0.3 * scale_factor),
                (0.5 * scale_factor, 0),
                (0.3 * scale_factor, -0.3 * scale_factor),
                (0, -0.5 * scale_factor),
                (-0.3 * scale_factor, -0.3 * scale_factor),
                (-0.5 * scale_factor, 0)
            ]
            scene.addPolygon(QPolygonF([QPointF(p[0], p[1]) for p in pontos_rocha]), pen)

        elif nome_bloco == 'Bueiro':
            # Desenha dois círculos concêntricos e uma cruz no centro para representar um bueiro
            scene.addEllipse(-0.5 * scale_factor, -0.5 * scale_factor, 1.0 * scale_factor, 1.0 * scale_factor, pen)
            scene.addEllipse(-0.4 * scale_factor, -0.4 * scale_factor, 0.8 * scale_factor, 0.8 * scale_factor, pen)

            # Adiciona uma cruz pequena no centro
            cruz_tamanho = 0.1 * scale_factor
            scene.addLine(-cruz_tamanho, 0, cruz_tamanho, 0, pen)
            scene.addLine(0, -cruz_tamanho, 0, cruz_tamanho, pen)

        elif nome_bloco == 'QuadradoX':
            # Desenha um quadrado com um X dentro
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, -0.5 * scale_factor),
                QPointF(-0.5 * scale_factor, 0.5 * scale_factor),
                QPointF(0.5 * scale_factor, 0.5 * scale_factor),
                QPointF(0.5 * scale_factor, -0.5 * scale_factor),
                QPointF(-0.5 * scale_factor, -0.5 * scale_factor)
            ]), pen)

            # Desenha um X dentro do quadrado
            scene.addLine(-0.5 * scale_factor, -0.5 * scale_factor, 0.5 * scale_factor, 0.5 * scale_factor, pen)
            scene.addLine(0.5 * scale_factor, -0.5 * scale_factor, -0.5 * scale_factor, 0.5 * scale_factor, pen)

        elif nome_bloco == 'QuadradoTraço':
            # Desenha um quadrado com um traço vertical no centro
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, -0.5 * scale_factor),
                QPointF(-0.5 * scale_factor, 0.5 * scale_factor),
                QPointF(0.5 * scale_factor, 0.5 * scale_factor),
                QPointF(0.5 * scale_factor, -0.5 * scale_factor),
                QPointF(-0.5 * scale_factor, -0.5 * scale_factor)
            ]), pen)

            # Desenha um traço vertical no centro do quadrado, se estendendo um pouco acima
            scene.addLine(0, 0 * scale_factor, 0, -0.6 * scale_factor, pen)

        elif nome_bloco == 'QuadradoCruz':
            # Desenha um quadrado com uma cruz dentro
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, -0.5 * scale_factor),
                QPointF(-0.5 * scale_factor, 0.5 * scale_factor),
                QPointF(0.5 * scale_factor, 0.5 * scale_factor),
                QPointF(0.5 * scale_factor, -0.5 * scale_factor),
                QPointF(-0.5 * scale_factor, -0.5 * scale_factor)
            ]), pen)

            # Desenha uma cruz dentro do quadrado
            scene.addLine(-0.4 * scale_factor, 0, 0.4 * scale_factor, 0, pen)
            scene.addLine(0, -0.4 * scale_factor, 0, 0.4 * scale_factor, pen)

        elif nome_bloco == 'TrianguloX':
            # Desenha um triângulo com um X dentro
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, 0.289 * scale_factor),
                QPointF(0.5 * scale_factor, 0.289 * scale_factor),
                QPointF(0, -0.577 * scale_factor),
                QPointF(-0.5 * scale_factor, 0.289 * scale_factor)
            ]), pen)

            # Desenha o X, centralizado no ponto (0, 0)
            x_size = 0.20 * scale_factor
            y_size = 0.289 * scale_factor
            scene.addLine(-x_size, -y_size, x_size, y_size, pen)
            scene.addLine(-x_size, y_size, x_size, -y_size, pen)

        elif nome_bloco == 'QuadradoPonto':
            # Desenha um quadrado com um ponto no centro
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, -0.5 * scale_factor),
                QPointF(-0.5 * scale_factor, 0.5 * scale_factor),
                QPointF(0.5 * scale_factor, 0.5 * scale_factor),
                QPointF(0.5 * scale_factor, -0.5 * scale_factor),
                QPointF(-0.5 * scale_factor, -0.5 * scale_factor)
            ]), pen)
            # Desenha um ponto no centro do quadrado
            brush = QBrush(cor)
            scene.addEllipse(-0.05 * scale_factor, -0.05 * scale_factor, 0.1 * scale_factor, 0.1 * scale_factor, pen, brush)

        elif nome_bloco == 'TrianguloPonto':
            # Desenha um triângulo com um ponto no centro
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, 0.289 * scale_factor),
                QPointF(0.5 * scale_factor, 0.289 * scale_factor),
                QPointF(0, -0.577 * scale_factor),
                QPointF(-0.5 * scale_factor, 0.289 * scale_factor)
            ]), pen)
            # Desenha o ponto
            brush = QBrush(cor)
            scene.addEllipse(-0.05 * scale_factor, -0.05 * scale_factor, 0.1 * scale_factor, 0.1 * scale_factor, pen, brush)

        elif nome_bloco == 'TrianguloTraço':
            # Desenha um triângulo com um traço vertical no centro
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, 0.289 * scale_factor),
                QPointF(0.5 * scale_factor, 0.289 * scale_factor),
                QPointF(0, -0.577 * scale_factor),
                QPointF(-0.5 * scale_factor, 0.289 * scale_factor)
            ]), pen)
            # Desenha o traço vertical no centro do triângulo
            scene.addLine(0, 0, 0, -0.577 * scale_factor, pen)

        elif nome_bloco == 'Casa':
            # Desenha a base da casa e o telhado
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, 0.25 * scale_factor),
                QPointF(-0.5 * scale_factor, -0.25 * scale_factor),
                QPointF(0.5 * scale_factor, -0.25 * scale_factor),
                QPointF(0.5 * scale_factor, 0.25 * scale_factor),
                QPointF(-0.5 * scale_factor, 0.25 * scale_factor)
            ]), pen)
            # Desenha o telhado
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, -0.25 * scale_factor),
                QPointF(0, -0.5 * scale_factor),
                QPointF(0.5 * scale_factor, -0.25 * scale_factor)
            ]), pen)

        elif nome_bloco == 'TrianguloCruz':
            # Desenha um triângulo com uma cruz dentro
            scene.addPolygon(QPolygonF([
                QPointF(-0.5 * scale_factor, 0.289 * scale_factor),
                QPointF(0.5 * scale_factor, 0.289 * scale_factor),
                QPointF(0 * scale_factor, -0.577 * scale_factor),
                QPointF(-0.5 * scale_factor, 0.289 * scale_factor)
            ]), pen)
            # Desenha a cruz centralizada
            scene.addLine(0, -0.289 * scale_factor, 0, 0.289 * scale_factor, pen)
            scene.addLine(-0.289 * scale_factor, 0, 0.289 * scale_factor, 0, pen)

        elif nome_bloco == 'Topografia GPS':
            # Desenha o tripé e o GPS no topo
            altura_tripe = 1.5 * scale_factor
            largura_base = 0.8 * scale_factor
            largura_gps = 0.2 * scale_factor
            altura_gps = 0.1 * scale_factor
            offset = altura_tripe / 2

            # Desenha as pernas do tripé apontando para baixo
            scene.addLine(0, -offset, -largura_base / 2, offset, pen)
            scene.addLine(0, -offset, largura_base / 2, offset, pen)
            scene.addLine(0, -offset, 0, -offset + 0.2 * scale_factor, pen)

            # Desenha a linha vertical no centro para simular um tripé
            scene.addLine(0, -offset, 0, offset, pen)

            # Desenha o retângulo para representar o GPS no topo do tripé
            scene.addPolygon(QPolygonF([
                QPointF(-largura_gps, -offset + altura_gps),
                QPointF(-largura_gps, -offset - altura_gps),
                QPointF(largura_gps, -offset - altura_gps),
                QPointF(largura_gps, -offset + altura_gps),
                QPointF(-largura_gps, -offset + altura_gps)
            ]), pen)

            # Adiciona um detalhe no GPS, como um pequeno círculo (botão)
            scene.addEllipse(-0.02 * scale_factor, -offset - 0.02 * scale_factor, 0.04 * scale_factor, 0.04 * scale_factor, pen)

    def criar_blocos(self, doc, cores):
        """
        Cria e adiciona blocos predefinidos ao documento DXF. A função realiza as seguintes etapas:

        1. Verifica se cada bloco predefinido já existe no documento DXF.
        2. Se o bloco não existir, cria o bloco usando a função correspondente.
        3. Adiciona o nome do bloco criado à lista de nomes de blocos.

        Parâmetros:
        - doc (ezdxf.document): O documento DXF onde os blocos serão adicionados.
        - cores (dict): Dicionário de cores escolhidas.

        Funcionalidades:
        - Criação de vários tipos de blocos predefinidos (círculos, quadrados, triângulos, etc.).
        - Verificação da existência dos blocos antes de criá-los.
        - Retorno de uma lista com os nomes dos blocos criados.

        Retorna:
        - list: Lista com os nomes dos blocos criados.
        """
        nomes_blocos = []

        # Verifica se os blocos já existem antes de criar novos
        if 'CirculoX' not in doc.blocks:
            nome_bloco_x = self.criar_bloco_circulo_com_linhas(doc, 'CirculoX', cores, [((-0.35, -0.35), (0.35, 0.35)), ((-0.35, 0.35), (0.35, -0.35))])
            nomes_blocos.append(nome_bloco_x)

        if 'CirculoCruz' not in doc.blocks:
            nome_bloco_cruz = self.criar_bloco_circulo_com_linhas(doc, 'CirculoCruz', cores, [((-0.5, 0), (0.5, 0)), ((0, -0.5), (0, 0.5))])
            nomes_blocos.append(nome_bloco_cruz)

        if 'CirculoTraço' not in doc.blocks:
            nome_bloco_traco_vertical = self.criar_bloco_circulo_com_linhas(doc, 'CirculoTraço', cores, [((0, 0), (0, 0.6))])
            nomes_blocos.append(nome_bloco_traco_vertical)

        # Criação do bloco CirculoComPontoCentral
        if 'CirculoPonto' not in doc.blocks:
            nome_bloco_ponto_central = self.criar_bloco_circulo_com_ponto(doc, 'CirculoPonto', cores)
            nomes_blocos.append(nome_bloco_ponto_central)

        # Criação do bloco ÁRVORE
        if 'Árvore' not in doc.blocks:
            nome_bloco_arvore = self.criar_bloco_arvore(doc, 'Árvore', cores)
            nomes_blocos.append(nome_bloco_arvore)

        # Criação do bloco POSTE
        if 'Poste' not in doc.blocks:
            nome_bloco_poste = self.criar_bloco_poste(doc, 'Poste', cores)
            nomes_blocos.append(nome_bloco_poste)

        # Criação do bloco BANDEIRA
        if 'Bandeira' not in doc.blocks:
            nome_bloco_bandeira = self.criar_bloco_bandeira(doc, 'Bandeira', cores)
            nomes_blocos.append(nome_bloco_bandeira)

        # Criação do bloco BANDEIRA 2
        if 'Bandeira 2' not in doc.blocks:
            nome_bloco_bandeira_Q = self.criar_bloco_bandeira_Q(doc, 'Bandeira 2', cores)
            nomes_blocos.append(nome_bloco_bandeira_Q)

        # Criação do bloco CERCA
        if 'Cerca' not in doc.blocks:
            nome_bloco_cerca = self.criar_bloco_cerca(doc, 'Cerca', cores)
            nomes_blocos.append(nome_bloco_cerca)

        # Verifica e cria o bloco de rocha
        if 'Rocha' not in doc.blocks:
            nome_bloco_rocha = self.criar_bloco_rocha(doc, 'Rocha', cores)
            nomes_blocos.append(nome_bloco_rocha)

        # Verifica e cria o bloco de bueiro
        if 'Bueiro' not in doc.blocks:
            nome_bloco_bueiro = self.criar_bloco_bueiro(doc, 'Bueiro', cores)
            nomes_blocos.append(nome_bloco_bueiro)

        # Criação do o bloco QuadradoComX 
        if 'QuadradoX' not in doc.blocks:
            nome_bloco_QuadradoComX = self.criar_bloco_quadrado_com_x(doc, 'QuadradoX', cores)
            nomes_blocos.append(nome_bloco_QuadradoComX)

        # Criação do bloco QuadradoComCruz
        if 'QuadradoCruz' not in doc.blocks:
            nome_bloco_QuadradoComCruz = self.criar_bloco_quadrado_com_cruz(doc, 'QuadradoCruz', cores)
            nomes_blocos.append(nome_bloco_QuadradoComCruz)

        # Criação do bloco QuadradoComTraco
        if 'QuadradoTraço' not in doc.blocks:
            nome_bloco_QuadradoComTraco = self.criar_bloco_quadrado_com_traco(doc, 'QuadradoTraço', cores)
            nomes_blocos.append(nome_bloco_QuadradoComTraco)

        # Criação do bloco QuadradoComPonto
        if 'QuadradoPonto' not in doc.blocks:
            nome_bloco_QuadradoComPonto = self.criar_bloco_quadrado_com_ponto(doc, 'QuadradoPonto', cores)
            nomes_blocos.append(nome_bloco_QuadradoComPonto)

        # Criação do bloco TrianguloComX
        if 'TrianguloX' not in doc.blocks:
            nome_bloco_triangulo_com_X = self.criar_bloco_triangulo_com_x(doc, 'TrianguloX', cores)
            nomes_blocos.append(nome_bloco_triangulo_com_X)

        # Criação do bloco TrianguloComCruz
        if 'TrianguloCruz' not in doc.blocks:
            nome_bloco_triangulo_cruz = self.criar_bloco_triangulo_com_cruz(doc, 'TrianguloCruz', cores)
            nomes_blocos.append(nome_bloco_triangulo_cruz)

        # Criação do bloco TrianguloComPonto
        if 'TrianguloPonto' not in doc.blocks:
            nome_bloco_ponto = self.criar_bloco_triangulo_com_ponto(doc, 'TrianguloPonto', cores)
            nomes_blocos.append(nome_bloco_ponto)

        # Criação do bloco TrianguloComTraco
        if 'TrianguloTraço' not in doc.blocks:
            nome_bloco_traco = self.criar_bloco_triangulo_com_traco(doc, 'TrianguloTraço', cores)
            nomes_blocos.append(nome_bloco_traco)

        # Criação do bloco Casa
        if 'Casa' not in doc.blocks:
            nome_bloco_casa = self.criar_bloco_casa(doc, 'Casa', cores)
            nomes_blocos.append(nome_bloco_casa)

        # Criação do bloco Topografia
        if 'Topografia GPS' not in doc.blocks:
            nome_bloco_gps = self.criar_bloco_gps_tripe(doc, 'Topografia GPS', cores)
            nomes_blocos.append(nome_bloco_gps)

            return nomes_blocos

    def criar_bloco_gps_tripe(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_tripe = cores.get('cor_tripe', (0, 0, 0))  # Cor padrão preta para o tripé
        cor_gps = cores.get('cor_gps', (0, 0, 0))  # Cor padrão 
        cor_detalhe = cores.get('cor_detalhe', (0, 0, 0))  # Cor padrão 

        # Altura e largura do tripé
        altura_tripe = 1.5
        largura_base = 0.8

        # Desenha as pernas do tripé apontando para baixo
        bloco.add_line(start=(0, altura_tripe), end=(-largura_base / 2, 0), dxfattribs={'color': rgb2int(cor_tripe)})
        bloco.add_line(start=(0, altura_tripe), end=(largura_base / 2, 0), dxfattribs={'color': rgb2int(cor_tripe)})
        bloco.add_line(start=(0, altura_tripe), end=(0, 0.2), dxfattribs={'color': rgb2int(cor_tripe)})

        # Desenha o retângulo para representar o GPS no topo do tripé
        largura_gps = 0.2
        altura_gps = 0.1
        bloco.add_lwpolyline([(0 - largura_gps / 2, altura_tripe + altura_gps / 2), 
                              (0 - largura_gps / 2, altura_tripe - altura_gps / 2),
                              (0 + largura_gps / 2, altura_tripe - altura_gps / 2), 
                              (0 + largura_gps / 2, altura_tripe + altura_gps / 2), 
                              (0 - largura_gps / 2, altura_tripe + altura_gps / 2)], 
                              dxfattribs={'color': rgb2int(cor_gps)})

        # Adiciona um detalhe no GPS, como um pequeno círculo (botão)
        bloco.add_circle(center=(0, altura_tripe), radius=0.02, dxfattribs={'color': rgb2int(cor_detalhe)})

        return nome_bloco

    def criar_bloco_casa(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get(nome_bloco, (128, 0, 0))  # Cor padrão vermelha para a edificação

        # Desenha a base da casa
        bloco.add_lwpolyline([(-0.5, -0.25), (-0.5, 0.25), (0.5, 0.25), (0.5, -0.25), (-0.5, -0.25)], dxfattribs={'color': rgb2int(cor_linha)})

        # Desenha o telhado
        bloco.add_lwpolyline([(-0.5, 0.25), (0, 0.5), (0.5, 0.25)], dxfattribs={'color': rgb2int(cor_linha)})

        return nome_bloco

    def criar_bloco_rocha(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get(nome_bloco, (105, 105, 105))  # Cor padrão cinza

        # Ajustando os pontos para que o ponto esteja centralizado no ponto (0,0)
        pontos_rocha = [(-0.5, 0), (-0.3, 0.3), (0, 0.5), (0.3, 0.3), (0.5, 0), (0.3, -0.3), (0, -0.5), (-0.3, -0.3), (-0.5, 0)]
        bloco.add_lwpolyline(pontos_rocha, dxfattribs={'color': rgb2int(cor_linha)})

        return nome_bloco

    def criar_bloco_bueiro(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get(nome_bloco, (0, 0, 0))  # Cor padrão preta para o bueiro

        # Adiciona dois círculos concêntricos para representar a parte superior do bueiro
        bloco.add_circle(center=(0, 0), radius=0.5, dxfattribs={'color': rgb2int(cor_linha)})
        bloco.add_circle(center=(0, 0), radius=0.4, dxfattribs={'color': rgb2int(cor_linha)})

        # Adiciona uma cruz pequena no centro
        cruz_tamanho = 0.1
        bloco.add_line(start=(-cruz_tamanho, 0), end=(cruz_tamanho, 0), dxfattribs={'color': rgb2int(cor_linha)})
        bloco.add_line(start=(0, -cruz_tamanho), end=(0, cruz_tamanho), dxfattribs={'color': rgb2int(cor_linha)})

        return nome_bloco

    def criar_bloco_triangulo_com_x(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get(nome_bloco, (0, 0, 0))  # Cor padrão preta
        # Desenha o triângulo
        bloco.add_lwpolyline([(-0.5, -0.289), (0.5, -0.289), (0, 0.577), (-0.5, -0.289)], dxfattribs={'color': rgb2int(cor_linha)})
        # Desenha o X, centralizado no ponto (0, 0.192)
        x_center = 0
        y_center = 0
        x_size = 0.35
        y_size = 0.54 # A altura de um triângulo menor que seria formado pelo X
        bloco.add_line(start=(x_center - x_size / 2, y_center - y_size / 2), end=(x_center + x_size / 2, y_center + y_size / 2), dxfattribs={'color': rgb2int(cor_linha)})
        bloco.add_line(start=(x_center - x_size / 2, y_center + y_size / 2), end=(x_center + x_size / 2, y_center - y_size / 2), dxfattribs={'color': rgb2int(cor_linha)})
        return nome_bloco

    def criar_bloco_triangulo_com_cruz(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get(nome_bloco, (0, 0, 0))
        # Desenha o triângulo
        bloco.add_lwpolyline([(-0.5, -0.289), (0.5, -0.289), (0, 0.577), (-0.5, -0.289)], dxfattribs={'color': rgb2int(cor_linha)})
        # Desenha a cruz
        bloco.add_line(start=(0, -0.1156), end=(0, 0.404), dxfattribs={'color': rgb2int(cor_linha)})
        bloco.add_line(start=(-0.3, 0.1442), end=(0.3, 0.1442), dxfattribs={'color': rgb2int(cor_linha)})
        return nome_bloco

    def criar_bloco_triangulo_com_ponto(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get(nome_bloco, (0, 0, 0))
        # Desenha o triângulo
        bloco.add_lwpolyline([(-0.5, -0.289), (0.5, -0.289), (0, 0.577), (-0.5, -0.289)], dxfattribs={'color': rgb2int(cor_linha)})
        # Desenha o ponto
        bloco.add_circle(center=(0, 0), radius=0.05, dxfattribs={'color': rgb2int(cor_linha)})
        return nome_bloco

    def criar_bloco_triangulo_com_traco(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get(nome_bloco, (0, 0, 0))
        # Desenha o triângulo
        bloco.add_lwpolyline([(-0.5, -0.289), (0.5, -0.289), (0, 0.577), (-0.5, -0.289)], dxfattribs={'color': rgb2int(cor_linha)})

        bloco.add_line(start=(0, 0), end=(0, 0.677), dxfattribs={'color': rgb2int(cor_linha)})
        return nome_bloco

    def criar_bloco_quadrado_com_x(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get('cor_linha', (0, 0, 0))  # Cor padrão preta
        # Desenha o quadrado centralizado
        bloco.add_lwpolyline([(-0.5, -0.5), (-0.5, 0.5), (0.5, 0.5), (0.5, -0.5), (-0.5, -0.5)], dxfattribs={'color': rgb2int(cor_linha)})
        # Desenha um X dentro do quadrado
        bloco.add_line(start=(-0.5, -0.5), end=(0.5, 0.5), dxfattribs={'color': rgb2int(cor_linha)})
        bloco.add_line(start=(0.5, -0.5), end=(-0.5, 0.5), dxfattribs={'color': rgb2int(cor_linha)})
        return nome_bloco

    def criar_bloco_quadrado_com_cruz(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get('cor_linha', (0, 0, 0))
        # Desenha o quadrado centralizado
        bloco.add_lwpolyline([(-0.5, -0.5), (-0.5, 0.5), (0.5, 0.5), (0.5, -0.5), (-0.5, -0.5)], dxfattribs={'color': rgb2int(cor_linha)})
        # Desenha uma cruz dentro do quadrado
        bloco.add_line(start=(-0.4, 0), end=(0.4, 0), dxfattribs={'color': rgb2int(cor_linha)})
        bloco.add_line(start=(0, -0.4), end=(0, 0.4), dxfattribs={'color': rgb2int(cor_linha)})
        return nome_bloco

    def criar_bloco_quadrado_com_traco(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_linha = cores.get('cor_linha', (0, 0, 0))
        # Desenha o quadrado centralizado
        bloco.add_lwpolyline([(-0.5, -0.5), (-0.5, 0.5), (0.5, 0.5), (0.5, -0.5), (-0.5, -0.5)], dxfattribs={'color': rgb2int(cor_linha)})
        # Desenha um traço vertical no centro do quadrado, se estendendo um pouco acima
        bloco.add_line(start=(0, 0), end=(0, 0.6), dxfattribs={'color': rgb2int(cor_linha)})
        return nome_bloco

    def criar_bloco_quadrado_com_ponto(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_ponto = cores.get('cor_ponto', (0, 0, 0))
        # Desenha o quadrado centralizado
        bloco.add_lwpolyline([(-0.5, -0.5), (-0.5, 0.5), (0.5, 0.5), (0.5, -0.5), (-0.5, -0.5)], dxfattribs={'color': rgb2int(cor_ponto)})
        # Desenha um ponto no centro do quadrado
        bloco.add_point((0, 0), dxfattribs={'color': rgb2int(cor_ponto)})
        return nome_bloco

    def criar_bloco_cerca(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)

        # Define a cor para a cerca
        cor_cerca = cores.get('cor_cerca', (139, 69, 19))  # Cor padrão marrom para a cerca

        # Parâmetros para o desenho da cerca
        altura_poste = 0.8  # Altura dos postes da cerca
        largura_painel = 0.1  # Largura dos painéis horizontais
        numero_postes = 3  # Número de postes na cerca

        # Desenha os postes verticais e os painéis horizontais
        for i in range(numero_postes):
            x_pos = i * largura_painel * 2
            # Desenha o poste vertical
            bloco.add_line(start=(x_pos, 0), end=(x_pos, altura_poste), dxfattribs={'color': rgb2int(cor_cerca)})

            # Desenha os painéis horizontais conectando os postes
            if i < numero_postes - 1:
                bloco.add_line(start=(x_pos, altura_poste / 2), end=(x_pos + largura_painel * 2, altura_poste / 2), dxfattribs={'color': rgb2int(cor_cerca)})
                bloco.add_line(start=(x_pos, altura_poste / 4), end=(x_pos + largura_painel * 2, altura_poste / 4), dxfattribs={'color': rgb2int(cor_cerca)})
                bloco.add_line(start=(x_pos, altura_poste * 3 / 4), end=(x_pos + largura_painel * 2, altura_poste * 3 / 4), dxfattribs={'color': rgb2int(cor_cerca)})

        return nome_bloco

    def criar_bloco_bandeira(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        
        # Define as cores
        cor_haste = cores.get('cor_haste', (105, 105, 105))  # Cor padrão cinza para a haste
        cor_bandeira = cores.get('cor_bandeira', (255, 0, 0))  # Cor padrão vermelha para a bandeira

        # Adiciona uma linha vertical para representar a haste da bandeira
        altura_haste = 1.5
        bloco.add_line(start=(0, 0), end=(0, altura_haste), dxfattribs={'color': rgb2int(cor_haste)})

        # Adiciona um triângulo maior para representar a bandeira à esquerda
        base_triangular = -0.85  # Agora negativo para ir à esquerda
        altura_triangular = 0.75
        pontos_bandeira = [(0, altura_haste), (base_triangular, altura_haste - altura_triangular / 2), (0, altura_haste - altura_triangular), (0, altura_haste)]
        bloco.add_lwpolyline(points=pontos_bandeira, dxfattribs={'color': rgb2int(cor_bandeira)})

        # Adiciona uma linha horizontal maior na base da haste
        bloco.add_line(start=(-0.5, 0), end=(0.5, 0), dxfattribs={'color': rgb2int(cor_haste)})

        return nome_bloco

    def criar_bloco_bandeira_Q(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        
        # Define as cores
        cor_haste = cores.get('cor_haste', (105, 105, 105))  # Cor padrão cinza para a haste
        cor_bandeira = cores.get('cor_bandeira', (255, 0, 0))  # Cor padrão vermelha para a bandeira

        # Adiciona uma linha vertical para representar a haste da bandeira
        altura_haste = 1.5
        bloco.add_line(start=(0, 0), end=(0, altura_haste), dxfattribs={'color': rgb2int(cor_haste)})

        # Adiciona um retângulo para representar a bandeira no topo da haste e à esquerda
        largura_bandeira = -0.75  # Largura negativa para a esquerda
        altura_bandeira = 0.6
        topo_bandeira = altura_haste
        base_bandeira = topo_bandeira - altura_bandeira
        bloco.add_lwpolyline([
            (0, topo_bandeira),
            (largura_bandeira, topo_bandeira),
            (largura_bandeira, base_bandeira),
            (0, base_bandeira),
            (0, topo_bandeira)  # Fecha o ponto voltando ao ponto inicial
        ], dxfattribs={'color': rgb2int(cor_bandeira)})

        # Adiciona uma linha horizontal na base da haste
        bloco.add_line(start=(-0.25, 0), end=(0.25, 0), dxfattribs={'color': rgb2int(cor_haste)})

        return nome_bloco

    def criar_bloco_poste(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_poste = cores.get('cor_poste', (0, 0, 0))

        # Poste principal (linha vertical)
        bloco.add_line(start=(0, 0), end=(0, 2.4), dxfattribs={'color': rgb2int(cor_poste)})

        # Travessa superior (linha horizontal menor)
        bloco.add_line(start=(-0.4, 2.0), end=(0.4, 2.0), dxfattribs={'color': rgb2int(cor_poste)})

        # Travessa inferior (linha horizontal maior)
        bloco.add_line(start=(-0.8, 1.6), end=(0.8, 1.6), dxfattribs={'color': rgb2int(cor_poste)})

        # Fio inclinado esquerdo (a partir da interseção da linha vertical com a linha horizontal mais baixa)
        bloco.add_line(start=(0, 1.6), end=(-0.5656, 1.0344), dxfattribs={'color': rgb2int(cor_poste)})

        # Fio inclinado direito (a partir da interseção da linha vertical com a linha horizontal mais baixa)
        bloco.add_line(start=(0, 1.6), end=(0.5656, 1.0344), dxfattribs={'color': rgb2int(cor_poste)})

        return nome_bloco

    def criar_bloco_arvore(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        
        # Define as cores
        cor_copa = cores.get('cor_copa', (34, 139, 34))  # Cor padrão verde escuro para a copa
        cor_tronco = cores.get('cor_tronco', (139, 69, 19))  # Cor padrão marrom para o tronco

        # Adiciona as elipses para representar a copa da árvore
        # Primeira elipse
        bloco.add_ellipse(center=(0, 1), major_axis=(0.4, 0), ratio=0.6, start_param=0, end_param=2 * math.pi, dxfattribs={'color': rgb2int(cor_copa)})

        # Segunda elipse, rotacionada
        bloco.add_ellipse(center=(0.2, 0.8), major_axis=(0.3, 0), ratio=0.7, start_param=0, end_param=2 * math.pi, dxfattribs={'color': rgb2int(cor_copa)})

        # Terceira elipse, também rotacionada
        bloco.add_ellipse(center=(-0.2, 0.8), major_axis=(0.3, 0), ratio=0.7, start_param=0, end_param=2 * math.pi, dxfattribs={'color': rgb2int(cor_copa)})

        # Adiciona uma linha para representar o tronco da árvore
        bloco.add_line(start=(0, 0), end=(0, 0.6), dxfattribs={'color': rgb2int(cor_tronco)})

        return nome_bloco

    def criar_bloco_circulo_com_linhas(self, doc, nome_bloco, cores, linhas):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_bloco = cores.get(nome_bloco, (255, 0, 0))  # Cor padrão vermelha
        cor_int = rgb2int(cor_bloco)

        # Adiciona um círculo ao bloco
        bloco.add_circle(center=(0, 0), radius=0.5, dxfattribs={'color': cor_int})

        # Adiciona linhas específicas ao bloco
        for linha in linhas:
            bloco.add_line(start=linha[0], end=linha[1], dxfattribs={'color': cor_int})

        return nome_bloco

    def criar_bloco_circulo_com_ponto(self, doc, nome_bloco, cores):
        bloco = doc.blocks.new(name=nome_bloco)
        cor_bloco = cores.get(nome_bloco, (255, 0, 0))  # Cor padrão vermelha
        cor_int = rgb2int(cor_bloco)

        # Adiciona um círculo ao bloco
        bloco.add_circle(center=(0, 0), radius=0.5, dxfattribs={'color': cor_int})

        # Adiciona um ponto no centro
        bloco.add_point((0, 0), dxfattribs={'color': cor_int})

        return nome_bloco

    def getCampoEscolhido(self):
        """
        Retorna o campo atualmente selecionado no comboBox.
        
        Funcionalidades:
        - Recupera o texto do campo selecionado no comboBox.

        Retorna:
        - str: O texto do campo atualmente selecionado no comboBox.
        """
        
        return self.comboBox.currentText()  # Retorna o texto do campo selecionado no comboBox

    def getSelecoes(self):
        """
        Retorna um conjunto com os valores dos atributos selecionados pelo usuário.
        A função realiza as seguintes etapas:

        1. Cria um conjunto vazio para armazenar as seleções.
        2. Itera sobre todos os itens no QListWidget.
        3. Para cada item, obtém o widget associado e encontra o QCheckBox.
        4. Se o QCheckBox estiver marcado, adiciona o texto do item ao conjunto de seleções.
        5. Retorna o conjunto de seleções.

        Funcionalidades:
        - Recuperação dos valores dos atributos selecionados pelo usuário no QListWidget.

        Retorna:
        - set: Conjunto com os valores dos atributos selecionados.
        """

        selecoes = set()  # Cria um conjunto vazio para armazenar as seleções
        for i in range(self.listWidget.count()):  # Itera sobre todos os itens no QListWidget
            item = self.listWidget.item(i)
            widget = self.listWidget.itemWidget(item)
            checkbox = widget.findChild(QCheckBox)  # Encontra o QCheckBox no widget do item
            if checkbox.isChecked():  # Verifica se o QCheckBox está marcado
                selecoes.add(item.text())  # Adiciona o valor do atributo ao conjunto
        return selecoes  # Retorna o conjunto de seleções

    def getCamposSelecionados(self):
        """
        Retorna uma lista com os nomes dos campos selecionados pelo usuário.
        A função realiza as seguintes etapas:

        1. Cria uma lista vazia para armazenar os campos selecionados.
        2. Verifica se o objeto possui o atributo 'camposCheckBoxes'.
        3. Itera sobre os itens no dicionário 'camposCheckBoxes'.
        4. Para cada campo, verifica se o checkbox está marcado.
        5. Se o checkbox estiver marcado, adiciona o nome do campo à lista de campos selecionados.
        6. Retorna a lista de campos selecionados.

        Funcionalidades:
        - Recuperação dos nomes dos campos selecionados pelo usuário.

        Retorna:
        - list: Lista com os nomes dos campos selecionados.
        """

        camposSelecionados = []  # Cria uma lista vazia para armazenar os campos selecionados
        if hasattr(self, 'camposCheckBoxes'):  # Verifica se o objeto possui o atributo 'camposCheckBoxes'
            for campo, checkbox in self.camposCheckBoxes.items():  # Itera sobre os itens no dicionário 'camposCheckBoxes'
                if checkbox.isChecked():  # Verifica se o checkbox está marcado
                    camposSelecionados.append(campo)  # Adiciona o nome do campo à lista de campos selecionados
        return camposSelecionados  # Retorna a lista de campos selecionados

    def getCampoZ(self):
        """
        Retorna o campo selecionado para a dimensão Z.
        A função realiza as seguintes etapas:

        1. Obtém o botão de rádio selecionado no grupo de botões de rádio.
        2. Verifica se um botão de rádio está selecionado.
        3. Se um botão de rádio estiver selecionado, retorna o texto do botão.
        4. Se nenhum botão de rádio estiver selecionado, retorna None.

        Funcionalidades:
        - Recuperação do nome do campo selecionado para a dimensão Z.

        Retorna:
        - str or None: O nome do campo selecionado para a dimensão Z, ou None se nenhum campo estiver selecionado.
        """

        selecionado = self.radioGroup.checkedButton()  # Obtém o botão de rádio selecionado
        return selecionado.text() if selecionado else None  # Retorna o texto do botão selecionado ou None

    def getBlocoSelecionado(self):
        """
        Retorna um dicionário com os valores dos atributos e os blocos selecionados pelo usuário.
        A função realiza as seguintes etapas:

        1. Cria um dicionário vazio para armazenar as seleções de blocos.
        2. Itera sobre todos os itens no QListWidget.
        3. Para cada item, obtém o valor do atributo e o widget associado.
        4. Encontra o QComboBox no widget do item.
        5. Se o QComboBox existir, adiciona o valor do atributo e o texto do bloco selecionado ao dicionário.
        6. Retorna o dicionário de seleções de blocos.

        Funcionalidades:
        - Recuperação dos blocos selecionados pelo usuário para cada valor de atributo.

        Retorna:
        - dict: Dicionário com os valores dos atributos como chaves e os blocos selecionados como valores.
        """

        blocoSelecoes = {}  # Cria um dicionário vazio para armazenar as seleções de blocos
        for i in range(self.listWidget.count()):  # Itera sobre todos os itens no QListWidget
            item = self.listWidget.item(i)
            valor = item.text()  # Obtém o valor do atributo
            widget = self.listWidget.itemWidget(item)
            combo = widget.findChild(QComboBox)  # Encontra o QComboBox no widget do item
            if combo:
                blocoSelecoes[valor] = combo.currentText()  # Adiciona o valor do atributo e o texto do bloco selecionado ao dicionário
        return blocoSelecoes  # Retorna o dicionário de seleções de blocos

class IconFieldSelectionDialog(QDialog):
    icon_cache = {}  # Dicionário de cache de ícones como atributo da classe
    # Atributos de classe para armazenar os URLs
    ultimoTextoUrl = ""
    ultimoTextoUrl2 = ""
    def __init__(self, layer, parent=None):
        """
        Inicializa o diálogo de seleção de campo e ícone.

        Parâmetros:
        - layer: A camada para a qual os campos e ícones serão selecionados.
        - parent: O widget pai do diálogo.
        """
        super(IconFieldSelectionDialog, self).__init__(parent)
        self.setWindowTitle("Escolher Campo e Ícone")

        self.setMinimumWidth(300)  # Define a largura mínima do diálogo
        self.setMaximumWidth(300)  # Define a largura máxima do diálogo

        self.layer = layer # Armazena a camada GIS
        self.selected_icon_url = None # URL do ícone selecionado
        self.selected_field = None # Campo selecionado
        self.network_manager = QNetworkAccessManager() # Gerenciador de rede para baixar ícones
        
        layout = QVBoxLayout(self) # Layout vertical principal do diálogo

        # Parte para escolha do campo e CheckBox
        field_layout = QHBoxLayout()  # Layout horizontal para o ComboBox e o CheckBox

        field_layout.addWidget(QLabel("Campo de identificação:"))
        self.combo_box = QComboBox() # ComboBox para selecionar o campo
        for field in self.layer.fields():
            self.combo_box.addItem(field.name()) # Adiciona os nomes dos campos ao ComboBox
        field_layout.addWidget(self.combo_box) # Adiciona o ComboBox ao layout

        self.check_box = QCheckBox("Rótulos")  # Criar o CheckBox
        self.check_box.setChecked(True)  # Marque por padrão, desmarque se não quiser exportar rótulos
        field_layout.addWidget(self.check_box)  # Adicionar o CheckBox ao layout horizontal

        layout.addLayout(field_layout)  # Adicionar o layout horizontal ao layout principal

        # Parte para escolha do ícone
        layout.addWidget(QLabel("Selecione um ícone:"))
        self.list_widget = QListWidget()
        self.list_widget.setIconSize(QSize(30, 30))  # Ajusta o tamanho do ícone
        self.list_widget.setViewMode(QListView.IconMode)  # Exibe os itens em modo de ícone
        self.list_widget.setSpacing(5)  # Adiciona espaço entre os ícones
        self.list_widget.setSelectionMode(QAbstractItemView.SingleSelection)  # Modo de seleção única
        self.list_widget.setDragEnabled(False)  # Desativa o arrastar-e-soltar

        # Limita a largura do QListWidget para forçar os ícones a se organizarem em múltiplas linhas
        self.list_widget.setMaximumWidth(280)  # Ajuste este valor conforme necessário

        # Configura a folha de estilos para alterar a aparência dos itens selecionados
        self.list_widget.setStyleSheet("""
        QListWidget::item:selected {
            border: 2px solid #0563c1;
            background-color: #d8eaff;
        }
        QListWidget::item:hover {
            background-color: #b8daff;
        }
        """)
        self.setup_icon_list() # Configura a lista de ícones
        layout.addWidget(self.list_widget) # Adiciona o QListWidget ao layout principal

        # Adicione um atributo para rastrear o item anteriormente selecionado
        self.previous_selected_item = None  
        # Conecte o sinal itemClicked ao slot que irá desselecionar o ícone se ele já estiver selecionado
        self.list_widget.itemClicked.connect(self.toggle_icon_selection)

        # Criação do QFrame para conter os campos de URL
        frame = QFrame()
        frame.setFrameShape(QFrame.Box)  # Define a forma do frame como uma caixa
        frame.setFrameShadow(QFrame.Raised)  # Define o estilo Raised
        frame.setLineWidth(1)  # Define a largura da borda

        # Layout para o QFrame
        frame_layout = QVBoxLayout(frame)

        # Primeiro QLineEdit e QPushButton para o URL da imagem
        self.labelImageUrl = QLabel("URL da Imagem para a Tabela:")
        frame_layout.addWidget(self.labelImageUrl)

        urlLayout1 = QHBoxLayout()
        self.lineEditImageUrl = QLineEdit()
        self.lineEditImageUrl.setPlaceholderText("Colar o URL da IMG para a Tabela: Opcional")
        self.lineEditImageUrl.setClearButtonEnabled(True)  # Habilita o botão de limpeza
        self.btnAbrirImagem = QPushButton("Colar")
        self.btnAbrirImagem.setMaximumWidth(35)
        urlLayout1.addWidget(self.lineEditImageUrl)
        urlLayout1.addWidget(self.btnAbrirImagem)

        frame_layout.addLayout(urlLayout1)   # Adiciona layout de URL da imagem ao layout do frame

        self.btnAbrirImagem.clicked.connect(self.colarTexto) # Conecta botão para colar texto

        # Segundo QLineEdit e QPushButton para o URL da imagem
        self.labelImageUrl2 = QLabel("URL para ScreenOverlay:")
        frame_layout.addWidget(self.labelImageUrl2)

        urlLayout2 = QHBoxLayout()
        self.lineEditImageUrl2 = QLineEdit()
        self.lineEditImageUrl2.setPlaceholderText("Colar o URL para o ScreenOverlay: Opcional")
        self.lineEditImageUrl2.setClearButtonEnabled(True)  # Habilita o botão de limpeza
        self.btnAbrirImagem2 = QPushButton("Colar")
        self.btnAbrirImagem2.setMaximumWidth(35)
        urlLayout2.addWidget(self.lineEditImageUrl2)
        urlLayout2.addWidget(self.btnAbrirImagem2)
        frame_layout.addLayout(urlLayout2)

        self.btnAbrirImagem2.clicked.connect(self.colarTexto2)

        # Setar o texto dos QLineEdit com os últimos valores usados
        self.lineEditImageUrl.setText(self.ultimoTextoUrl)
        self.lineEditImageUrl2.setText(self.ultimoTextoUrl2)

        # Conecta o sinal textChanged a um novo método para lidar com a atualização do texto
        self.lineEditImageUrl.textChanged.connect(self.verificarTexto)
        self.lineEditImageUrl2.textChanged.connect(self.verificarTexto2)

        layout.addWidget(frame)  # Adiciona o QFrame ao layout principal

        # Botões de Exportar e Cancelar
        buttons_layout = QHBoxLayout()
        self.button_ok = QPushButton("Exportar")
        self.button_ok.clicked.connect(self.accept)
        buttons_layout.addWidget(self.button_ok)

        button_cancel = QPushButton("Cancelar")
        button_cancel.clicked.connect(self.reject)
        buttons_layout.addWidget(button_cancel)

        # Conecta a mudança de índice do ComboBox ao método validate_ok_button_state
        self.combo_box.currentIndexChanged.connect(self.validate_ok_button_state)

        # Conecta a mudança de estado do CheckBox ao método validate_ok_button_state
        self.check_box.stateChanged.connect(self.validate_ok_button_state)
        # Conecta a mudança de seleção do QListWidget ao método validate_ok_button_state
        self.list_widget.itemSelectionChanged.connect(self.validate_ok_button_state)

        layout.addLayout(buttons_layout) # Adiciona o layout dos botões ao layout principal

        self.validate_ok_button_state() # Valida o estado inicial do botão OK

    def verificarValidadeURL(self, url):
        """
        Verifica se a string fornecida é uma URL válida usando uma expressão regular.

        Parâmetros:
        - url (str): A URL a ser validada.

        Funcionalidades:
        - Compila uma expressão regular que valida URLs de forma abrangente, cobrindo protocolos, domínios, IPs, portas, caminhos, query strings e fragmentos.
        - Verifica se a URL fornecida corresponde à expressão regular.

        Retorno:
        - Retorna True se a URL for válida, False caso contrário.

        Utilização:
        - Utilizada para garantir que as URLs inseridas para imagens ou links em campos de URL sejam válidas antes de serem utilizadas para evitar erros na aplicação ou na visualização do KML.
        """
        # Expressão regular atualizada para validar URLs de forma mais abrangente.
        padrao_url = re.compile(
            r'^(https?:\/\/)?'  # http:// ou https://
            r'((([a-z\d]([a-z\d-]*[a-z\d])*)\.)+[a-z]{2,}|'  # domínio
            r'((\d{1,3}\.){3}\d{1,3}))'  # ou ip
            r'(\:\d+)?(\/[-a-z\d%_.~+]*)*'  # porta e caminho
            r'(\?[;&a-z\d%_.~+=-]*)?'  # query string
            r'(\#[-a-z\d_]*)?$', re.IGNORECASE)  # fragmento
        return re.match(padrao_url, url) is not None

    def colarTexto(self):
        """
        Obtém o texto do clipboard e insere no campo QLineEdit destinado ao URL da imagem, 
        se o texto for uma URL válida.

        Funcionalidades:
        - Acessa o conteúdo do clipboard do sistema.
        - Verifica se o texto é uma URL válida.
        - Insere a URL válida no campo de texto se for válida.

        Utilização:
        - Usado para facilitar a inserção de URLs por meio do uso de clipboard, evitando a necessidade de digitação manual, 
          aumentando a eficiência do usuário e reduzindo a chance de erro de entrada.
        """
        # Acessa o clipboard do sistema
        clipboard = QGuiApplication.clipboard() # Obtém o objeto clipboard do sistema
        texto = clipboard.text() # Lê o texto atualmente armazenado no clipboard
        
        # Verifica se o texto copiado é uma URL válida
        if self.verificarValidadeURL(texto): # Chama o método para verificar a validade da URL
            self.lineEditImageUrl.setText(texto) # Insere o texto no QLineEdit se for uma URL válida

    def colarTexto2(self):
        """
        Obtém o texto do clipboard e insere no campo QLineEdit destinado ao URL para ScreenOverlay,
        se o texto for uma URL válida.

        Funcionalidades:
        - Acessa o conteúdo do clipboard do sistema.
        - Verifica se o texto é uma URL válida.
        - Insere a URL válida no campo de texto destinado ao ScreenOverlay se for válida.

        Utilização:
        - Usado para facilitar a inserção de URLs de ScreenOverlay por meio do uso de clipboard, 
          evitando a necessidade de digitação manual e melhorando a eficiência do usuário.
        """
        # Acessa o clipboard do sistema
        clipboard = QGuiApplication.clipboard() # Obtém o objeto clipboard do sistema
        texto = clipboard.text() # Lê o texto atualmente armazenado no clipboard

        # Verifica se o texto copiado é uma URL válida
        if self.verificarValidadeURL(texto): # Chama o método para verificar a validade da URL
            self.lineEditImageUrl2.setText(texto) # Insere o texto no QLineEdit destinado ao ScreenOverlay se for uma URL válida

    def verificarValidadeURLImagem(self, url):
        """
        Verifica se a URL fornecida termina com uma extensão de arquivo de imagem aceitável. 
        Este método é usado para garantir que URLs inseridas para imagens sejam de formatos compatíveis para visualização.

        Parâmetros:
        - url (str): A URL da imagem a ser validada.

        Funcionalidades:
        - Define uma lista de extensões de arquivo de imagem que são aceitáveis.
        - Verifica se a URL fornecida termina com uma dessas extensões.

        Retorno:
        - Retorna True se a URL terminar com uma extensão de arquivo de imagem válida, False caso contrário.

        Utilização:
        - Utilizada para validar URLs de imagens, assegurando que os links fornecidos apontem para arquivos de imagem reais que podem ser carregados e exibidos corretamente no KML.
        """
        # Define as extensões de arquivo de imagem que são aceitáveis
        extensoes_validas = ['.jpg', '.jpeg', '.png', '.gif', '.bmp', '.tif', '.tiff', '.webp']
        # Verifica se a URL termina com alguma das extensões válidas
        return any(url.lower().endswith(ext) for ext in extensoes_validas) # Verifica se a URL corresponde a uma extensão válida

    def verificarTexto(self):
        """
        Verifica o conteúdo do campo de entrada para URL da imagem, ajusta a cor do texto baseada na validade da URL
        e atualiza o último texto de URL válido armazenado. Este método é usado para fornecer feedback visual imediato 
        sobre a validade da URL inserida.

        Funcionalidades:
        - Obtém o texto atual do campo QLineEdit destinado à URL da imagem.
        - Verifica se o texto é uma URL válida e se corresponde a uma extensão de arquivo de imagem aceitável.
        - Atualiza o armazenamento do último texto válido ou o limpa se o texto atual não for válido.
        - Altera a cor do texto do QLineEdit para azul se a URL for válida, para vermelho se for inválida, e retorna para a cor padrão se o campo estiver vazio.

        Utilização:
        - Essencial para fornecer uma resposta visual instantânea ao usuário sobre a validade da URL inserida, 
          facilitando a correção de erros e garantindo que apenas URLs válidas sejam utilizadas.
        """
        # Obtém o texto atual do QLineEdit
        texto = self.lineEditImageUrl.text() # Lê o texto da caixa de entrada de URL da imagem
        
        # Verifica a validade da URL e se a URL é uma imagem
        if self.verificarValidadeURL(texto) and self.verificarValidadeURLImagem(texto):
            IconFieldSelectionDialog.ultimoTextoUrl = texto # Atualiza o último texto válido se a URL for válida
            self.lineEditImageUrl.setStyleSheet("QLineEdit { color: blue; }") # Muda a cor do texto para azul
        else:
            IconFieldSelectionDialog.ultimoTextoUrl = "" # Limpa o último texto válido se a URL for inválida
            if texto.strip() != "": # Verifica se o campo não está vazio
                self.lineEditImageUrl.setStyleSheet("QLineEdit { color: red; }") # Muda a cor do texto para vermelho se houver texto inválido
            else:
                self.lineEditImageUrl.setStyleSheet("") # Retorna a cor do texto para o padrão se o campo estiver vazio

    def verificarTexto2(self):
        """
        Verifica o conteúdo do campo de entrada para o URL do ScreenOverlay, ajusta a cor do texto baseada na validade da URL,
        e atualiza o último texto de URL válido armazenado. Este método é usado para fornecer feedback visual imediato 
        sobre a validade da URL inserida.

        Funcionalidades:
        - Obtém o texto atual do campo QLineEdit destinado ao URL do ScreenOverlay.
        - Verifica se o texto é uma URL válida e se corresponde a uma extensão de arquivo de imagem aceitável.
        - Atualiza o armazenamento do último texto válido ou o limpa se o texto atual não for válido.
        - Altera a cor do texto do QLineEdit para azul se a URL for válida, para vermelho se for inválida, e retorna para a cor padrão se o campo estiver vazio.

        Utilização:
        - Essencial para fornecer uma resposta visual instantânea ao usuário sobre a validade da URL inserida para o ScreenOverlay,
          facilitando a correção de erros e garantindo que apenas URLs válidas sejam utilizadas para o ScreenOverlay.
        """
        # Obtém o texto atual do QLineEdit destinado ao URL do ScreenOverlay
        texto = self.lineEditImageUrl2.text()  # Lê o texto da caixa de entrada de URL para o ScreenOverlay

        # Verifica a validade da URL e se a URL é uma imagem
        if self.verificarValidadeURL(texto) and self.verificarValidadeURLImagem(texto):
            IconFieldSelectionDialog.ultimoTextoUrl2 = texto # Atualiza o último texto válido se a URL for válida
            self.lineEditImageUrl2.setStyleSheet("QLineEdit { color: blue; }") # Muda a cor do texto para azul
        else:
            IconFieldSelectionDialog.ultimoTextoUrl2 = "" # Limpa o último texto válido se a URL for inválida
            if texto.strip() != "": # Verifica se o campo não está vazio
                self.lineEditImageUrl2.setStyleSheet("QLineEdit { color: red; }") # Muda a cor do texto para vermelho se houver texto inválido
            else:
                self.lineEditImageUrl2.setStyleSheet("") # Retorna a cor do texto para o padrão se o campo estiver vazio

    def toggle_icon_selection(self, item):
        """
        Alterna a seleção de um ícone na lista de ícones. Se o ícone já estiver selecionado,
        ele será desmarcado. Caso contrário, ele será selecionado.

        Ações executadas pela função:
        1. Verifica se o item clicado é o mesmo que o item anteriormente selecionado.
        2. Se for o mesmo item, desmarca-o, limpa a URL do ícone selecionado e
           reseta a referência ao item anteriormente selecionado.
        3. Se for um item diferente, seleciona o novo item, atualiza a URL do ícone
           selecionado e a referência ao item anteriormente selecionado.
        4. Valida o estado do botão OK após a alteração na seleção.

        Parâmetros:
        - item: O item clicado na lista de ícones.
        """
        # Verifique se o item clicado já está selecionado
        if self.previous_selected_item == item:
            # Se for o mesmo item que o anteriormente selecionado, deselecione-o
            self.list_widget.clearSelection() # Limpa a seleção na lista de ícones
            self.selected_icon_url = None # Reseta a URL do ícone selecionado
            self.previous_selected_item = None  # Limpa o item anteriormente selecionado
        else:
            # Se for um item diferente, selecione-o e atualize o item anteriormente selecionado
            self.list_widget.setCurrentItem(item)  # Define o item atual na lista de ícones
            self.selected_icon_url = item.data(Qt.UserRole) # Obtém a URL do ícone selecionado
            self.previous_selected_item = item  # Atualiza o item anteriormente selecionado

        # Após a alteração na seleção, valide o estado do botão OK
        self.validate_ok_button_state()

    def validate_ok_button_state(self):
        """
        Valida o estado do botão OK no diálogo.

        Ações executadas pela função:
        1. Verifica se algum ícone está selecionado na lista de ícones.
        2. Verifica se o checkbox está marcado.
        3. Verifica se o ComboBox não está vazio.
        4. Habilita o botão OK apenas se:
           - Algum ícone estiver selecionado OU o checkbox estiver marcado,
           E
           - O ComboBox não estiver vazio.

        Linhas de código explicadas:
        """
        # Verifica se algum ícone está selecionado
        has_icon_selected = any(item.isSelected() for item in self.list_widget.selectedItems())
        # Verifica se o checkbox está marcado
        is_checkbox_checked = self.check_box.isChecked()
        # Verifica se o QComboBox não está vazio
        is_combobox_not_empty = self.combo_box.count() > 0

        # O botão "OK" só será habilitado se houver ícones selecionados OU o checkbox estiver marcado,
        # E o QComboBox não estiver vazio.
        self.button_ok.setEnabled((is_checkbox_checked or has_icon_selected) and is_combobox_not_empty)

    def setup_icon_list(self):
        """
        Configura a lista de ícones no diálogo.

        Ações executadas pela função:
        1. Define uma lista de URLs de ícones a serem baixados.
        2. Itera sobre cada URL na lista.
        3. Chama a função `download_icon` para cada URL para baixar e exibir o ícone na lista de ícones (`QListWidget`).

        Linhas de código explicadas:
        """
        # Define uma lista de URLs de ícones a serem baixados
        icon_urls = [
            "http://maps.google.com/mapfiles/kml/shapes/triangle.png",
            "http://maps.google.com/mapfiles/kml/shapes/parks.png",
            "http://maps.google.com/mapfiles/kml/shapes/campground.png",
            "http://maps.google.com/mapfiles/kml/shapes/info-i.png",
            "http://maps.google.com/mapfiles/kml/shapes/square.png",
            "http://maps.google.com/mapfiles/kml/shapes/placemark_square.png",
            "http://maps.google.com/mapfiles/kml/shapes/man.png",
            "http://maps.google.com/mapfiles/kml/shapes/arrow.png",
            "http://maps.google.com/mapfiles/kml/paddle/wht-blank.png",
            "http://maps.google.com/mapfiles/kml/paddle/wht-circle.png",
            "http://maps.google.com/mapfiles/kml/paddle/wht-diamond.png",
            "http://maps.google.com/mapfiles/kml/paddle/wht-square.png",
            "http://maps.google.com/mapfiles/kml/paddle/wht-stars.png",
            "http://maps.google.com/mapfiles/kml/shapes/cabs.png",
            "http://maps.google.com/mapfiles/kml/shapes/airports.png",
            "http://maps.google.com/mapfiles/kml/shapes/caution.png",
            "http://maps.google.com/mapfiles/kml/shapes/placemark_circle.png",
            "http://maps.google.com/mapfiles/kml/shapes/flag.png",
            "http://maps.google.com/mapfiles/kml/pal5/icon13.png",
            "http://mw1.google.com/mw-ocean/ocean/media/swc/en/icons/swc.png",
            "http://maps.google.com/mapfiles/kml/shapes/picnic.png",
            "http://maps.google.com/mapfiles/kml/shapes/church.png",
        ]
        for url in icon_urls:
            # Chama a função download_icon para baixar e exibir o ícone na lista de ícones
            self.download_icon(url)

    def download_icon(self, url):
        """
        Baixa um ícone a partir de uma URL e o adiciona à lista de ícones.

        Ações executadas pela função:
        1. Verifica se a URL do ícone já está no cache.
        2. Se estiver no cache, usa o ícone armazenado.
        3. Se não estiver no cache, faz uma requisição de rede para baixar o ícone.
        4. Conecta o sinal de término do download a uma função que processa o ícone baixado.

        Linhas de código explicadas:
        """
        if url in IconFieldSelectionDialog.icon_cache:
            # Se o ícone já estiver no cache, use-o
            self.add_icon_to_list(url, IconFieldSelectionDialog.icon_cache[url])
        else:
            # Se não estiver no cache, baixe-o
            req = QNetworkRequest(QUrl(url)) # Cria uma requisição de rede com a URL do ícone
            reply = self.network_manager.get(req) # Envia a requisição e obtém a resposta
            # Conecta o sinal finished à função lambda que chama on_download_finished
            reply.finished.connect(lambda: self.on_download_finished(reply))

    def on_download_finished(self, reply):
        """
        Processa a resposta do download de um ícone.

        Ações executadas pela função:
        1. Verifica se houve erro na resposta.
        2. Se houver erro, limpa a resposta e retorna.
        3. Obtém a URL da resposta.
        4. Carrega os dados da resposta em um QPixmap.
        5. Armazena o ícone no cache.
        6. Adiciona o ícone à lista de ícones.
        7. Limpa a resposta.

        Linhas de código explicadas:
        """
        if reply.error():
            reply.deleteLater()  # Limpa a resposta se houver erro
            return

        url = reply.url().toString()  # Obtém a URL da resposta
        pixmap = QPixmap()
        if pixmap.loadFromData(reply.readAll()):  # Carrega os dados da resposta em um QPixmap
            IconFieldSelectionDialog.icon_cache[url] = pixmap  # Armazena o ícone no cache
            self.add_icon_to_list(url, pixmap)  # Adiciona o ícone à lista de ícones
        reply.deleteLater()  # Limpa a resposta

    def add_icon_to_list(self, url, pixmap):
        """
        Adiciona um ícone à lista de ícones no QListWidget.

        Ações executadas pela função:
        1. Cria um objeto QIcon a partir do QPixmap fornecido.
        2. Cria um item de QListWidget sem texto.
        3. Define o ícone do item de lista para o QIcon criado.
        4. Armazena a URL do ícone no item de lista usando Qt.UserRole.
        5. Adiciona o item de lista ao QListWidget.

        Linhas de código explicadas:
        """
        icon = QIcon(pixmap)  # Cria um objeto QIcon a partir do QPixmap fornecido
        item = QListWidgetItem()  # Cria um item de lista sem texto
        item.setIcon(icon)  # Define o ícone do item
        item.setData(Qt.UserRole, url)  # Armazena a URL do ícone no item de lista usando Qt.UserRole
        self.list_widget.addItem(item)  # Adiciona o item de lista ao QListWidget

    def accept(self):
        """
        Aceita o diálogo e salva as seleções feitas pelo usuário.

        Ações executadas pela função:
        1. Obtém o campo selecionado no QComboBox e armazena em self.selected_field.
        2. Obtém o item atualmente selecionado no QListWidget.
        3. Se um item estiver selecionado, obtém a URL do ícone associada e armazena em self.selected_icon_url.
        4. Chama o método accept da classe base QDialog para fechar o diálogo.

        Linhas de código explicadas:
        """
        self.selected_field = self.combo_box.currentText()  # Obtém o campo selecionado no QComboBox e armazena em self.selected_field
        current_item = self.list_widget.currentItem()  # Obtém o item atualmente selecionado no QListWidget
        if current_item:
            self.selected_icon_url = current_item.data(Qt.UserRole)  # Se um item estiver selecionado, obtém a URL do ícone associada e armazena em self.selected_icon_url
        self.selected_image_url = self.lineEditImageUrl.text()  # Captura o texto (URL) inserido no lineEditImageUrl
        self.selected_image_url2 = self.lineEditImageUrl2.text() # Captura o texto (URL) inserido no lineEditImageUr2
        super(IconFieldSelectionDialog, self).accept()  # Chama o método accept da classe base QDialog para fechar o diálogo

    def get_selections(self):
        """
        Retorna as seleções feitas pelo usuário.

        Ações executadas pela função:
        1. Retorna o campo selecionado no QComboBox e a URL do ícone selecionado.

        Retornos:
        - Uma tupla contendo o campo selecionado (self.selected_field) e a URL do ícone selecionado (self.selected_icon_url).
        """
        # Retorna o campo selecionado, a URL do ícone e o URL da imagem
        return self.selected_field, self.selected_icon_url, self.selected_image_url, self.selected_image_url2

class CircleDelegate(QStyledItemDelegate):
    """
    Delegate para desenhar um círculo colorido ao lado de cada item em uma lista de itens.
    
    Ações executadas pela função:
    1. Desenha um círculo colorido ao lado do item, com cor e borda configuráveis.
    2. Ajusta a posição do texto do item para não sobrepor o círculo.
    3. Busca a camada no projeto do QGIS usando o ID e tenta obter as cores do símbolo da camada.
    """
    def paint(self, painter, option, index):
        """
        Sobrescreve o método paint para desenhar um círculo ao lado do item.
        
        Parâmetros:
        - painter (QPainter): O objeto de pintura usado para desenhar o item.
        - option (QStyleOptionViewItem): Opções de estilo para o item.
        - index (QModelIndex): Índice do item no modelo.
        """
        super().paint(painter, option, index)

        # Obtém o ID da camada do item do modelo
        layer_id = index.data(Qt.UserRole)
        if not layer_id:
            layer_id = index.model().itemFromIndex(index).data()

        # Busca a camada no projeto do QGIS usando o ID
        layer = QgsProject.instance().mapLayer(layer_id)

        # Define as cores padrão para o círculo e a borda
        circle_color = Qt.white
        border_color = Qt.black
        border_width = 2  # Definindo a espessura da borda

        # Tenta obter as cores do símbolo da camada, se disponível
        if layer:
            symbols = layer.renderer().symbols(QgsRenderContext())
            if symbols:
                circle_color = symbols[0].color()
                # Verifica se existe um contorno e obtém a cor do contorno
                if symbols[0].symbolLayerCount() > 0:
                    border_layer = symbols[0].symbolLayer(0)
                    if hasattr(border_layer, 'strokeColor'):
                        border_color = border_layer.strokeColor()
 
        # Calcula a posição e o tamanho do círculo
        offset = -15
        radius = 6
        # Assegurando que os valores passados para QRect sejam inteiros
        x = int(option.rect.left() + offset)
        y = int(option.rect.top() + (option.rect.height() - radius * 2) / 2)
        diameter = int(radius * 2)
        circleRect = QRect(x, y, diameter, diameter) # Retângulo que define o círculo

        painter.setBrush(QBrush(circle_color)) # Define a cor de preenchimento e a borda para o círculo
        painter.setPen(QPen(border_color, border_width))  # Configurando a cor e a espessura da borda
        painter.drawEllipse(circleRect) # Desenha o círculo

        # Ajusta a posição do texto no item para não sobrepor o círculo
        option.rect.setLeft(option.rect.left() + offset + radius*2 + 8)
