# Import basic libs
import json

# Import Qgis libs
from qgis.core import NULL, QgsFeature

# Import PyQt libs
from qgis.PyQt.QtCore import QObject, QUrl, pyqtSignal
from qgis.PyQt.QtNetwork import QNetworkReply, QNetworkRequest


class GetTaxrefFromMNHN(QObject):
    finished_dl = pyqtSignal()
    """
    API Request to the MNHN to get the taxon for every occurence by name and possibly rank.
    network_manager : the one
    project : qgis project
    layer : initial layer
    no_occurence_layer : the layer containing occurence with no matching taxon
    one_occurence_layer : the layer containing occurence with one matching taxon
    multiple_occurence_layer : the layer containing occurence with multiple matching taxon
    dlg : main dialog widget
    field_name : name of the field containing the taxon name
    field_rank : name of the field containing the taxon rank
    list_rank : list of the rank from the MNHN
    selected_only : know if only the selected occurences must be processed
    """

    def __init__(
        self,
        network_manager=None,
        project=None,
        layer=None,
        no_occurence_layer=None,
        one_occurence_layer=None,
        multiple_occurence_layer=None,
        dlg=None,
        field_name=None,
        field_rank=None,
        list_rank=None,
        selected_only=None,
    ):
        super().__init__()
        self.network_manager = network_manager
        self.project = project
        self.layer = layer
        self.no_occurence_layer = no_occurence_layer
        self.one_occurence_layer = one_occurence_layer
        self.multiple_occurence_layer = multiple_occurence_layer
        self.thread = dlg.thread
        self.progress_bar = dlg.select_progress_bar_label
        self.field_name = field_name
        self.field_rank = field_rank
        self.list_rank = list_rank
        selected_only = selected_only
        self._pending_downloads = 0

        self.ids = []
        self.rank = []

        # results list
        self.no_occurence = []
        self.multi_occurence = []
        self.multiples_occurences_values = {}

        if not selected_only:
            selected_features = self.layer.selectedFeatureIds()
            self.layer.selectAll()

        for obs in self.layer.getSelectedFeatures():
            self.ids.append(obs.id())
            if self.field_rank != "":
                self.rank.append(obs[self.field_rank])

        if not selected_only:
            self.layer.removeSelection()
            self.layer.selectByIds(selected_features)

        self._pending_downloads = len(self.ids)
        self._iterate_names = 0

        self.thread.set_max(len(self.ids))
        self.thread.add_one(0)
        self.progress_bar.setText(
            self.tr("Downloaded data : " + str(0) + "/" + str(len(self.ids)))
        )

        self.download(self.ids[self._iterate_names])

    @property
    def pending_downloads(self):
        return self._pending_downloads

    @property
    def iterate_names(self):
        return self._iterate_names

    def get_request(self, feature_id, name, rank):
        if name != NULL:
            url = "https://taxref.mnhn.fr/taxref-web/api/taxa/search?cdContext=&cdDoc=&fuzzyMatching=on&NOM_COMPLET={name}&nomVernFr=&ANNEE=&validite=&rang.rang={rank}&HABITAT=&inpn=&FR=&GF=&MAR=&GUA=&SM=&SB=&SPM=&MAY=&REU=&EPA=&TAAF=&SA=&TA=&NC=&WF=&PF=&CLI=".format(  # noqa: E501
                name=name, rank=rank
            )

            request = QNetworkRequest(QUrl(url))
            request.setHeader(QNetworkRequest.ContentTypeHeader, "application/json")
            reply = self.network_manager.get(request)
            reply.finished.connect(lambda: self.handle_finished(reply, feature_id))
        else:
            self._pending_downloads -= 1
            if self.pending_downloads == 0:
                self.project.addMapLayer(self.layer)
                self.finished_dl.emit()
            else:
                self.thread.set_max(len(self.ids))
                self.thread.add_one(1)
                self.progress_bar.setText(
                    self.tr(
                        "Downloaded data : "
                        + str(self.thread.value)
                        + "/"
                        + str(len(self.ids))
                    )
                )
                self.download(self.ids[self._iterate_names])

    def download(self, feature_id):
        feature = self.layer.getFeature(feature_id)
        self._iterate_names += 1
        if self.field_rank != "":
            rank = feature[self.field_rank].lower()
            if rank.lower() not in list(self.list_rank.keys()):
                name = feature[self.field_name]
                rank = ""
                self.get_request(feature_id, name, rank)
            else:
                name = feature[self.field_name]
                rank = self.list_rank[rank]
                self.get_request(feature_id, name, rank)
        else:
            name = feature[self.field_name]
            rank = ""
            self.get_request(feature_id, name, rank)

    def handle_finished(self, reply, feature_id):
        self._pending_downloads -= 1
        if reply.error() != QNetworkReply.NetworkError.NoError:
            print(f"code: {reply.error()} message: {reply.errorString()}")
            if reply.error() == 403:
                print("Service down")
        else:
            data_request = reply.readAll().data().decode()
            if data_request != "":
                res = json.loads(data_request)
                res = [i for i in res if i["cdRef"]]
                if len(res) == 0:
                    self.no_occurence.append(feature_id)
                    self.no_occurence_layer.startEditing()
                    feature = self.layer.getFeature(feature_id)
                    self.no_occurence_layer.dataProvider().addFeature(feature)
                    self.no_occurence_layer.updateExtents()
                    self.no_occurence_layer.commitChanges()
                    self.no_occurence_layer.triggerRepaint()
                else:
                    taxref_id = None
                    for elem in res:
                        if not taxref_id:
                            taxref_id = elem["cdRef"]
                        elif taxref_id != elem["cdRef"]:
                            taxref_id = None
                            taxref_name = None
                            taxref_url = None
                            break
                        else:
                            taxref_name = res[0]["lbNom"] + " " + res[0]["lbAuteur"]
                            taxref_url = (
                                "https://inpn.mnhn.fr/espece/cd_nom/{cd_nom}".format(
                                    cd_nom=taxref_id
                                )
                            )
                    if not taxref_id:
                        feature = self.layer.getFeature(feature_id)
                        self.multi_occurence.append(feature_id)
                        self.multiple_occurence_layer.startEditing()
                        feature = self.layer.getFeature(feature_id)
                        self.multiple_occurence_layer.dataProvider().addFeature(feature)
                        self.multiple_occurence_layer.updateExtents()
                        self.multiple_occurence_layer.commitChanges()
                        self.multiple_occurence_layer.triggerRepaint()

                        occurences_available = {}
                        for occurence in res:
                            occurences_available[occurence["cdRef"]] = (
                                occurence["lbNom"] + " " + occurence["lbAuteur"]
                            )
                        self.multiples_occurences_values[feature[self.field_name]] = (
                            occurences_available
                        )
                    else:
                        feature = QgsFeature(self.one_occurence_layer.fields())
                        feature.setGeometry(
                            self.layer.getFeature(feature_id).geometry()
                        )
                        attributes = self.layer.getFeature(feature_id).attributes()
                        taxref_id = res[0]["cdRef"]
                        taxref_name = res[0]["lbNom"] + " " + res[0]["lbAuteur"]
                        attributes.append(taxref_id)
                        attributes.append(taxref_name)
                        attributes.append(
                            "https://inpn.mnhn.fr/espece/cd_nom/{cd_nom}".format(
                                cd_nom=taxref_id
                            )
                        )
                        feature.setAttributes(attributes)
                        self.one_occurence_layer.startEditing()
                        self.one_occurence_layer.dataProvider().addFeature(feature)
                        self.one_occurence_layer.updateExtents()
                        self.one_occurence_layer.commitChanges()
                        self.one_occurence_layer.triggerRepaint()
            else:
                self.no_occurence.append(feature_id)
                self.no_occurence_layer.startEditing()
                feature = self.layer.getFeature(feature_id)
                self.no_occurence_layer.dataProvider().addFeature(feature)
                self.no_occurence_layer.updateExtents()
                self.no_occurence_layer.commitChanges()
                self.no_occurence_layer.triggerRepaint()
        if self.pending_downloads == 0:
            self.finished_dl.emit()
        else:
            self.thread.set_max(len(self.ids))
            self.thread.add_one(1)
            self.progress_bar.setText(
                self.tr(
                    "Downloaded data : "
                    + str(self.thread.value)
                    + "/"
                    + str(len(self.ids))
                )
            )
            self.download(self.ids[self._iterate_names])
