# Import basic libs
import json

# Import Qgis libs
from qgis.core import NULL, QgsFeature

# Import PyQt libs
from qgis.PyQt.QtCore import QObject, QUrl, pyqtSignal
from qgis.PyQt.QtNetwork import QNetworkReply, QNetworkRequest


class GetTaxrefFromGBIF_ID(QObject):
    finished_dl = pyqtSignal()
    """Get multiples informations from a getcapabilities request.
    List all layers available, get the maximal extent of all the Wfs' data."""

    def __init__(
        self,
        network_manager=None,
        project=None,
        layer=None,
        no_occurence_layer=None,
        one_occurence_layer=None,
        dlg=None,
        field=None,
        selected_only=None,
    ):
        super().__init__()
        self.network_manager = network_manager
        self.project = project
        self.layer = layer
        self.no_occurence_layer = no_occurence_layer
        self.one_occurence_layer = one_occurence_layer
        self.thread = dlg.thread
        self.progress_bar = dlg.select_progress_bar_label
        self.field = field
        selected_only = selected_only
        self._pending_downloads = 0

        # results list
        self.no_occurence = []
        self.multi_occurence = []

        self.ids = {}

        if not selected_only:
            selected_features = self.layer.selectedFeatureIds()
            self.layer.selectAll()

        for obs in self.layer.getSelectedFeatures():
            if obs[self.field] not in self.ids:
                self.ids[obs[self.field]] = [obs.id()]
            else:
                self.ids[obs[self.field]].append(obs.id())

        if not selected_only:
            self.layer.removeSelection()
            self.layer.selectByIds(selected_features)

        self._pending_downloads = len(self.ids)
        self._iterate_ids = 0

        self.thread.set_max(len(self.ids))
        self.thread.add_one(0)
        self.progress_bar.setText(
            self.tr("Downloaded data : " + str(0) + "/" + str(len(self.ids)))
        )

        self.download(list(self.ids.keys())[self._iterate_ids])

    @property
    def pending_downloads(self):
        return self._pending_downloads

    @property
    def iterate_ids(self):
        return self._iterate_ids

    def download(self, gbif_id):
        self._iterate_ids += 1
        if gbif_id != NULL:
            features_id = self.ids[gbif_id]
            url = "https://www.gbif.org/api/species/{gbif_id}/checklistdatasets?limit=100".format(
                gbif_id=gbif_id
            )
            request = QNetworkRequest(QUrl(url))
            request.setHeader(QNetworkRequest.ContentTypeHeader, "application/json")
            reply = self.network_manager.get(request)
            reply.finished.connect(
                lambda: self.handle_finished(reply, features_id, gbif_id)
            )
        else:
            self.no_occurence.append(gbif_id)
            self._pending_downloads -= 1
            if self.pending_downloads == 0:
                self.finished_dl.emit()
            else:
                self.thread.set_max(len(self.ids))
                self.thread.add_one(1)
                self.progress_bar.setText(
                    self.tr(
                        "Downloaded data : "
                        + str(self.thread.value)
                        + "/"
                        + str(len(self.ids))
                    )
                )
                self.download(list(self.ids.keys())[self._iterate_ids])

    def handle_finished(self, reply, features_id, gbif_id):
        self._pending_downloads -= 1
        if reply.error() != QNetworkReply.NetworkError.NoError:
            print(f"code: {reply.error()} message: {reply.errorString()}")
            if reply.error() == 403:
                print("Service down")
            self.no_occurence.append(gbif_id)
            self.no_occurence_layer.startEditing()
            for feature_id in features_id:
                self.no_occurence_layer.dataProvider().addFeature(
                    self.layer.getFeature(feature_id)
                )
            self.no_occurence_layer.updateExtents()
            self.no_occurence_layer.commitChanges()
            self.no_occurence_layer.triggerRepaint()
        else:
            data_request = reply.readAll().data().decode()
            if data_request != "":
                res = json.loads(data_request)
                if "results" in res:
                    if len(res["results"]) == 0:
                        self.no_occurence.append(gbif_id)
                        self.no_occurence_layer.startEditing()
                        for feature_id in features_id:
                            self.no_occurence_layer.dataProvider().addFeature(
                                self.layer.getFeature(feature_id)
                            )
                        self.no_occurence_layer.updateExtents()
                        self.no_occurence_layer.commitChanges()
                        self.no_occurence_layer.triggerRepaint()
                    else:
                        for elem in res["results"]:
                            if "TAXREF" in [
                                elem["title"]
                                for elem in res["results"]
                                if "title" in elem
                            ]:
                                if elem["title"] == "TAXREF":
                                    taxref_id = elem["_relatedTaxon"]["taxonID"]
                                    taxref_name = elem["_relatedTaxon"][
                                        "scientificName"
                                    ]
                                    self.one_occurence_layer.startEditing()
                                    for feature_id in features_id:
                                        feature = QgsFeature(
                                            self.one_occurence_layer.fields()
                                        )
                                        feature.setGeometry(
                                            self.layer.getFeature(feature_id).geometry()
                                        )
                                        attributes = self.layer.getFeature(
                                            feature_id
                                        ).attributes()
                                        attributes.append(taxref_id)
                                        attributes.append(taxref_name)
                                        attributes.append(
                                            "https://inpn.mnhn.fr/espece/cd_nom/{cd_nom}".format(
                                                cd_nom=taxref_id
                                            )
                                        )
                                        feature.setAttributes(attributes)
                                        self.one_occurence_layer.startEditing()
                                        self.one_occurence_layer.dataProvider().addFeature(
                                            feature
                                        )
                                        self.one_occurence_layer.updateExtents()
                                        self.one_occurence_layer.commitChanges()
                                        self.one_occurence_layer.triggerRepaint()
                                else:
                                    pass
                            else:
                                pass
                else:
                    self.no_occurence.append(gbif_id)
                    self.no_occurence_layer.startEditing()
                    for feature_id in features_id:
                        self.no_occurence_layer.dataProvider().addFeature(
                            self.layer.getFeature(feature_id)
                        )
                    self.no_occurence_layer.updateExtents()
                    self.no_occurence_layer.commitChanges()
                    self.no_occurence_layer.triggerRepaint()
            else:
                self.no_occurence.append(gbif_id)
                self.no_occurence_layer.startEditing()
                for feature_id in features_id:
                    self.no_occurence_layer.dataProvider().addFeature(
                        self.layer.getFeature(feature_id)
                    )
                self.no_occurence_layer.updateExtents()
                self.no_occurence_layer.commitChanges()
                self.no_occurence_layer.triggerRepaint()
        if self.pending_downloads == 0:
            self.finished_dl.emit()
        else:
            self.thread.set_max(len(self.ids))
            self.thread.add_one(1)
            self.progress_bar.setText(
                self.tr(
                    "Downloaded data : "
                    + str(self.thread.value)
                    + "/"
                    + str(len(self.ids))
                )
            )
            self.download(list(self.ids.keys())[self._iterate_ids])


class GetTaxrefFromGBIF_Scientic_Name(QObject):
    finished_dl = pyqtSignal()
    """Get multiples informations from a getcapabilities request.
    List all layers available, get the maximal extent of all the Wfs' data."""

    def __init__(
        self,
        network_manager=None,
        project=None,
        layer=None,
        no_occurence_layer=None,
        one_occurence_layer=None,
        multiple_occurence_layer=None,
        dlg=None,
        field=None,
        selected_only=None,
    ):
        super().__init__()
        self.network_manager = network_manager
        self.project = project
        self.layer = layer
        self.no_occurence_layer = no_occurence_layer
        self.one_occurence_layer = one_occurence_layer
        self.multiple_occurence_layer = multiple_occurence_layer
        self.thread = dlg.thread
        self.progress_bar = dlg.select_progress_bar_label
        self.field = field
        selected_only = selected_only
        self._pending_downloads = 0

        # results list
        self.no_occurence = []
        self.multi_occurence = []
        self.multiples_occurences_values = {}

        self.ids = {}

        if not selected_only:
            selected_features = self.layer.selectedFeatureIds()
            self.layer.selectAll()

        for obs in self.layer.getSelectedFeatures():
            if obs[self.field] not in self.ids:
                self.ids[obs[self.field]] = [obs.id()]
            else:
                self.ids[obs[self.field]].append(obs.id())

        if not selected_only:
            self.layer.removeSelection()
            self.layer.selectByIds(selected_features)

        self._pending_downloads = len(self.ids)
        self._iterate_ids = 0

        self.thread.set_max(len(self.ids))
        self.thread.add_one(0)
        self.progress_bar.setText(
            self.tr("Downloaded data : " + str(0) + "/" + str(len(self.ids)))
        )

        self.download(list(self.ids.keys())[self._iterate_ids])

    @property
    def pending_downloads(self):
        return self._pending_downloads

    @property
    def iterate_ids(self):
        return self._iterate_ids

    def download(self, scientific_name):
        self._iterate_ids += 1
        if scientific_name != NULL:
            features_id = self.ids[scientific_name]
            url = "https://api.gbif.org/v1/species/search?datasetKey=0e61f8fe-7d25-4f81-ada7-d970bbb2c6d6&q={scientific_name}".format(
                scientific_name=scientific_name
            )
            request = QNetworkRequest(QUrl(url))
            request.setHeader(QNetworkRequest.ContentTypeHeader, "application/json")
            reply = self.network_manager.get(request)
            reply.finished.connect(
                lambda: self.handle_finished(reply, features_id, scientific_name)
            )
        else:
            self.no_occurence.append(scientific_name)
            self._pending_downloads -= 1
            if self.pending_downloads == 0:
                self.finished_dl.emit()
            else:
                self.thread.set_max(len(self.ids))
                self.thread.add_one(1)
                self.progress_bar.setText(
                    self.tr(
                        "Downloaded data : "
                        + str(self.thread.value)
                        + "/"
                        + str(len(self.ids))
                    )
                )
                self.download(list(self.ids.keys())[self._iterate_ids])

    def handle_finished(self, reply, features_id, scientific_name):
        self._pending_downloads -= 1
        if reply.error() != QNetworkReply.NetworkError.NoError:
            print(f"code: {reply.error()} message: {reply.errorString()}")
            if reply.error() == 403:
                print("Service down")
            self.no_occurence.append(scientific_name)
            self.no_occurence_layer.startEditing()
            for feature_id in features_id:
                self.no_occurence_layer.dataProvider().addFeature(
                    self.layer.getFeature(feature_id)
                )
            self.no_occurence_layer.updateExtents()
            self.no_occurence_layer.commitChanges()
            self.no_occurence_layer.triggerRepaint()
        else:
            data_request = reply.readAll().data().decode()
            if data_request != "":
                res = json.loads(data_request)
                print(res)
                if "results" in res:
                    if len(res["results"]) == 0:
                        self.no_occurence.append(scientific_name)
                        self.no_occurence_layer.startEditing()
                        for feature_id in features_id:
                            self.no_occurence_layer.dataProvider().addFeature(
                                self.layer.getFeature(feature_id)
                            )
                        self.no_occurence_layer.updateExtents()
                        self.no_occurence_layer.commitChanges()
                        self.no_occurence_layer.triggerRepaint()
                    else:
                        accepted_taxon = {}
                        other_taxon = {}
                        for elem in res["results"]:
                            taxref_name = None
                            for ver_names in elem["vernacularNames"]:
                                if ver_names["language"] == "fra":
                                    taxref_name = ver_names["vernacularName"]
                                else:
                                    pass
                            if not taxref_name:
                                taxref_name = elem["scientificName"]
                            if elem["taxonomicStatus"] == "ACCEPTED":
                                if elem["taxonID"] not in accepted_taxon:
                                    accepted_taxon[elem["taxonID"]] = taxref_name
                                else:
                                    pass
                            else:
                                if elem["taxonID"] not in other_taxon:
                                    other_taxon[elem["taxonID"]] = taxref_name
                        if len(accepted_taxon) == 1:
                            taxref_id = list(accepted_taxon.keys())[0]
                            taxref_name = accepted_taxon[taxref_id]
                            self.one_occurence_layer.startEditing()
                            for feature_id in features_id:
                                feature = QgsFeature(self.one_occurence_layer.fields())
                                feature.setGeometry(
                                    self.layer.getFeature(feature_id).geometry()
                                )
                                attributes = self.layer.getFeature(
                                    feature_id
                                ).attributes()
                                attributes.append(taxref_id)
                                attributes.append(taxref_name)
                                attributes.append(
                                    "https://inpn.mnhn.fr/espece/cd_nom/{cd_nom}".format(
                                        cd_nom=taxref_id
                                    )
                                )
                                feature.setAttributes(attributes)
                                self.one_occurence_layer.startEditing()
                                self.one_occurence_layer.dataProvider().addFeature(
                                    feature
                                )
                                self.one_occurence_layer.updateExtents()
                                self.one_occurence_layer.commitChanges()
                                self.one_occurence_layer.triggerRepaint()
                        elif len(accepted_taxon) == 0 and len(other_taxon) == 1:
                            taxref_id = list(other_taxon.keys())[0]
                            taxref_name = other_taxon[taxref_id]
                        elif len(accepted_taxon) == 0 and len(other_taxon) == 0:
                            self.no_occurence.append(scientific_name)
                            self.no_occurence_layer.startEditing()
                            for feature_id in features_id:
                                self.no_occurence_layer.dataProvider().addFeature(
                                    self.layer.getFeature(feature_id)
                                )
                        else:
                            feature = self.layer.getFeature(feature_id)
                            self.multi_occurence.append(feature_id)
                            self.multiple_occurence_layer.startEditing()
                            feature = self.layer.getFeature(feature_id)
                            self.multiple_occurence_layer.dataProvider().addFeature(
                                feature
                            )
                            self.multiple_occurence_layer.updateExtents()
                            self.multiple_occurence_layer.commitChanges()
                            self.multiple_occurence_layer.triggerRepaint()
                            occurences_available = {}
                            for occurence in accepted_taxon:
                                occurences_available[str(occurence)] = accepted_taxon[
                                    occurence
                                ]
                                self.multiples_occurences_values[
                                    str(feature[self.field])
                                ] = occurences_available
                    self.no_occurence_layer.updateExtents()
                    self.no_occurence_layer.commitChanges()
                    self.no_occurence_layer.triggerRepaint()
                else:
                    self.no_occurence.append(scientific_name)
                    self.no_occurence_layer.startEditing()
                    for feature_id in features_id:
                        self.no_occurence_layer.dataProvider().addFeature(
                            self.layer.getFeature(feature_id)
                        )
                    self.no_occurence_layer.updateExtents()
                    self.no_occurence_layer.commitChanges()
                    self.no_occurence_layer.triggerRepaint()
            else:
                self.no_occurence.append(scientific_name)
                self.no_occurence_layer.startEditing()
                for feature_id in features_id:
                    self.no_occurence_layer.dataProvider().addFeature(
                        self.layer.getFeature(feature_id)
                    )
                self.no_occurence_layer.updateExtents()
                self.no_occurence_layer.commitChanges()
                self.no_occurence_layer.triggerRepaint()
        if self.pending_downloads == 0:
            self.finished_dl.emit()
        else:
            self.thread.set_max(len(self.ids))
            self.thread.add_one(1)
            self.progress_bar.setText(
                self.tr(
                    "Downloaded data : "
                    + str(self.thread.value)
                    + "/"
                    + str(len(self.ids))
                )
            )
            self.download(list(self.ids.keys())[self._iterate_ids])
