# Import basic libs
import json

# Import PyQt libs
from qgis.PyQt.QtCore import QObject, QUrl, pyqtSignal
from qgis.PyQt.QtNetwork import QNetworkReply, QNetworkRequest


class ImportRank(QObject):
    finished_dl = pyqtSignal()
    """Get multiples informations from a getcapabilities request.
    List all layers available, get the maximal extent of all the Wfs' data."""

    def __init__(self, url=None, network_manager=None):
        super().__init__()
        self.url = url

        self.rank_list = {}

        self._pending_downloads = 0
        self.network_manager = network_manager

        self.download()

    @property
    def pending_downloads(self):
        return self._pending_downloads

    def download(self):
        url = QUrl(self.url)
        request = QNetworkRequest(url)
        request.setHeader(QNetworkRequest.ContentTypeHeader, "application/json")
        reply = self.network_manager.get(request)
        reply.finished.connect(lambda: self.handle_finished(reply))
        self._pending_downloads += 1

    def handle_finished(self, reply):
        self._pending_downloads -= 1
        if reply.error() != QNetworkReply.NetworkError.NoError:
            print(f"code: {reply.error()} message: {reply.errorString()}")
            if reply.error() == 403:
                print("service is down")
        else:
            data_request = reply.readAll().data().decode()
            if data_request != "":
                res = json.loads(data_request)
                for elem in res["_embedded"]["taxonomicRanks"]:
                    if elem["geoSpeciesUri"]:
                        self.rank_list[
                            elem["geoSpeciesUri"]
                            .split(
                                "http://rdf.geospecies.org/ont/geospecies#TaxonRank_"
                            )[1]
                            .lower()
                        ] = elem["id"]
                    else:
                        self.rank_list[
                            elem["taxrefUri"]
                            .split("http://taxref.mnhn.fr/lod/taxrank/")[1]
                            .lower()
                        ] = elem["id"]
                    self.rank_list[elem["name"]] = elem["id"]
                    self.rank_list[elem["id"]] = elem["id"]
                if self.pending_downloads == 0:
                    self.finished_dl.emit()
