#! python3  # noqa: E265

"""
Main plugin module.
"""

import datetime

# standard
from functools import partial
from pathlib import Path

# PyQGIS
from qgis.core import (
    QgsApplication,
    QgsFeatureRequest,
    QgsField,
    QgsProject,
    QgsSettings,
)
from qgis.gui import QgisInterface
from qgis.PyQt.QtCore import (
    QCoreApplication,
    QLocale,
    QObject,
    QTranslator,
    QUrl,
    QVariant,
    pyqtSignal,
)
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtNetwork import QNetworkAccessManager, QNetworkReply, QNetworkRequest
from qgis.PyQt.QtWidgets import QAction, QMessageBox

# project
from taxref_collector.__about__ import (
    DIR_PLUGIN_ROOT,
    __icon_path__,
    __plugin_name__,
    __title__,
    __uri_homepage__,
    __uri_tracker__,
)
from taxref_collector.gui.dlg_main import TaxrefCollectorDialog
from taxref_collector.gui.dlg_multiple_occurences import MultipleTaxonSelectionDialog
from taxref_collector.gui.dlg_settings import PlgOptionsFactory
from taxref_collector.processing import (
    GetTaxrefFromGBIF_ID,
    GetTaxrefFromGBIF_Scientic_Name,
    GetTaxrefFromMNHN,
    GetTaxrefFromWiki,
    ImportRank,
    TaxrefCollectorProvider,
)
from taxref_collector.toolbelt import PlgLogger

# ############################################################################
# ########## Classes ###############
# ##################################


class TaxrefCollectorPlugin:
    def __init__(self, iface: QgisInterface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class which \
        provides the hook by which you can manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.project = QgsProject.instance()
        self.manager = QNetworkAccessManager()
        self.log = PlgLogger().log
        self.provider = None
        self.pluginIsActive = False
        self.action_launch = None

        # translation

        # initialize the locale
        self.locale: str = QgsSettings().value("locale/userLocale", QLocale().name())[
            0:2
        ]

        locale_path: Path = (
            DIR_PLUGIN_ROOT
            / "resources"
            / "i18n"
            / f"{__plugin_name__.lower()}_{self.locale}.qm"
        )
        self.log(message=f"Translation: {self.locale}, {locale_path}", log_level=4)
        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path.resolve()))
            QCoreApplication.installTranslator(self.translator)

    def initGui(self):
        """Set up plugin UI elements."""

        # settings page within the QGIS preferences menu
        self.options_factory = PlgOptionsFactory()
        self.iface.registerOptionsWidgetFactory(self.options_factory)

        # -- Actions
        self.action_launch = QAction(
            QIcon(str(__icon_path__)),
            self.tr(__title__),
            self.iface.mainWindow(),
        )
        self.iface.addToolBarIcon(self.action_launch)
        self.action_launch.triggered.connect(lambda: self.run())
        self.action_help = QAction(
            QgsApplication.getThemeIcon("mActionHelpContents.svg"),
            self.tr("Help"),
            self.iface.mainWindow(),
        )
        self.action_help.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.action_settings = QAction(
            QgsApplication.getThemeIcon("console/iconSettingsConsole.svg"),
            self.tr("Settings"),
            self.iface.mainWindow(),
        )
        self.action_settings.triggered.connect(
            lambda: self.iface.showOptionsDialog(
                currentPage="mOptionsPage{}".format(__title__)
            )
        )

        # -- Menu
        self.iface.addPluginToMenu(__title__, self.action_launch)
        self.iface.addPluginToMenu(__title__, self.action_settings)
        self.iface.addPluginToMenu(__title__, self.action_help)

        # -- Processing
        self.initProcessing()

        # -- Help menu

        # documentation
        self.iface.pluginHelpMenu().addSeparator()
        self.action_help_plugin_menu_documentation = QAction(
            QIcon(str(__icon_path__)),
            f"{__title__} - Documentation",
            self.iface.mainWindow(),
        )
        self.action_help_plugin_menu_documentation.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.iface.pluginHelpMenu().addAction(
            self.action_help_plugin_menu_documentation
        )

    def initProcessing(self):
        self.provider = TaxrefCollectorProvider()
        QgsApplication.processingRegistry().addProvider(self.provider)

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)

    def unload(self):
        """Cleans up when plugin is disabled/uninstalled."""
        # -- Clean up menu
        self.iface.removePluginMenu(__title__, self.action_launch)
        self.iface.removeToolBarIcon(self.action_launch)
        self.iface.removePluginMenu(__title__, self.action_help)
        self.iface.removePluginMenu(__title__, self.action_settings)

        # -- Clean up preferences panel in QGIS settings
        self.iface.unregisterOptionsWidgetFactory(self.options_factory)

        # -- Unregister processing
        QgsApplication.processingRegistry().removeProvider(self.provider)

        # remove from QGIS help/extensions menu
        if self.action_help_plugin_menu_documentation:
            self.iface.pluginHelpMenu().removeAction(
                self.action_help_plugin_menu_documentation
            )

        # remove actions
        del self.action_launch
        del self.action_settings
        del self.action_help
        self.pluginIsActive = False

    def service_connexion(self):
        if self.internet_checker.reply.error() != QNetworkReply.NoError:
            print(self.internet_checker.reply.url())

    def run(self):
        """Main process.

        Try to connect to internet, if successfull, the dialog appear.
        Else an error message appear.
        """
        self.internet_checker = InternetChecker(None, self.manager)
        self.internet_checker.finished.connect(self.handle_finished)
        self.internet_checker.errors.connect(self.service_connexion)
        self.internet_checker.ping("https://github.com/")

    def handle_finished(self):
        # Check if plugin is already launched
        if not self.pluginIsActive:
            self.pluginIsActive = True
            # Open Dialog
            self.dlg = TaxrefCollectorDialog(
                self.project, self.iface, self.manager, self.locale
            )

            result = self.dlg.exec_()
            if result:
                # If dialog is accepted, "OK" is pressed, the process is launch
                tab_index = self.dlg.tab_widget_api.currentIndex()
                api = list(self.dlg.api_list.keys())[tab_index]
                stack_index = self.dlg.organiser[tab_index]["stack"].currentIndex()
                api_road = self.dlg.api_list[api][stack_index]
                field = self.dlg.organiser[tab_index][stack_index]["field"]
                if "optional" in self.dlg.organiser[tab_index][stack_index]:
                    optional = self.dlg.organiser[tab_index][stack_index][
                        "optional"
                    ].currentField()
                else:
                    optional = None
                self.processing(
                    api=api,
                    api_road=api_road,
                    field=field.currentField(),
                    optional=optional,
                )
            else:
                # Else the dialog close and plugin can be launched again
                self.pluginIsActive = False

    def processing(self, api, api_road, field, optional):
        """Processing chain if the dialog is accepted
        Depending on user's choices, a folder can be created, the service is
        requested and the layers in the specific extent can be added to
        the QGIS project

        """
        self.dlg.activate_window()
        layer = self.dlg.select_layer_combo_box.currentLayer()

        # Creation of the group and folder name
        today = datetime.datetime.now()
        year = today.year
        month = today.strftime("%m")
        day = today.strftime("%d")
        hour = today.strftime("%H")
        minute = today.strftime("%M")
        folder = (
            str("TAXREF_Collector_")
            + "_"
            + str(year)
            + str(month)
            + str(day)
            + "_"
            + str(hour)
            + str(minute)
        )
        self.project.instance().layerTreeRoot().insertGroup(0, folder)
        self.group = self.project.instance().layerTreeRoot().findGroup(folder)

        # Creation of the layer to fetch all the result from the API.
        self.no_occurence_layer = layer.materialize(
            QgsFeatureRequest().setFilterFids([])
        )
        self.no_occurence_layer.setName(str(layer.name()) + "_no_occurrence")

        self.one_occurence_layer = layer.materialize(
            QgsFeatureRequest().setFilterFids([])
        )
        self.one_occurence_layer.setName(str(layer.name()) + "_one_occurrence")
        self.one_occurence_layer.startEditing()
        self.one_occurence_layer.addAttribute(
            QgsField("cd_nom", QVariant.Int, "integer", 10)
        )
        self.one_occurence_layer.addAttribute(
            QgsField("taxref_name", QVariant.String, "string", 254)
        )
        self.one_occurence_layer.addAttribute(
            QgsField("taxref_url", QVariant.String, "string", 254)
        )
        self.one_occurence_layer.commitChanges()
        self.one_occurence_layer.triggerRepaint()

        self.multiple_occurence_layer = layer.materialize(
            QgsFeatureRequest().setFilterFids([])
        )
        self.multiple_occurence_layer.setName(
            str(layer.name()) + "_multiple_occurrences"
        )

        self.multiple_occurence_layer.startEditing()
        self.multiple_occurence_layer.addAttribute(
            QgsField("cd_nom", QVariant.Int, "integer", 10)
        )
        self.multiple_occurence_layer.addAttribute(
            QgsField("taxref_name", QVariant.String, "string", 254)
        )
        self.multiple_occurence_layer.addAttribute(
            QgsField("taxref_url", QVariant.String, "string", 254)
        )
        self.multiple_occurence_layer.commitChanges()
        self.multiple_occurence_layer.triggerRepaint()

        if api == "GBIF":
            print("OK")
            if api_road == "GBIF_ID":
                print("weird")
                self.collect_taxref = GetTaxrefFromGBIF_ID(
                    network_manager=self.manager,
                    project=self.project,
                    layer=layer,
                    no_occurence_layer=self.no_occurence_layer,
                    one_occurence_layer=self.one_occurence_layer,
                    dlg=self.dlg,
                    field=field,
                    selected_only=self.dlg.layer_selection_checkbox.isChecked(),
                )
            elif api_road == "scientific_name":
                print("ok")
                self.collect_taxref = GetTaxrefFromGBIF_Scientic_Name(
                    network_manager=self.manager,
                    project=self.project,
                    layer=layer,
                    no_occurence_layer=self.no_occurence_layer,
                    one_occurence_layer=self.one_occurence_layer,
                    dlg=self.dlg,
                    field=field,
                    selected_only=self.dlg.layer_selection_checkbox.isChecked(),
                )
            else:
                print(api_road)
                # NOT SUPPOSED TO BE HERE
            self.collect_taxref.finished_dl.connect(
                lambda: self.finished_import(field=field)
            )
        elif api == "INPN":
            if api_road == "scientific_name":
                if optional:
                    self.imported_rank = ImportRank(
                        network_manager=self.manager,
                        url="https://taxref.mnhn.fr/api/taxonomicRanks",
                    )
                    self.imported_rank.finished_dl.connect(
                        lambda: self.collect_from_name(field, optional)
                    )
                else:
                    self.collect_from_name(field, optional)
            else:
                print(api_road)
                # NOT SUPPOSED TO BE HERE
        elif api == "Wikidata":
            self.collect_taxref = GetTaxrefFromWiki(
                network_manager=self.manager,
                project=self.project,
                layer=layer,
                no_occurence_layer=self.no_occurence_layer,
                one_occurence_layer=self.one_occurence_layer,
                multiple_occurence_layer=self.multiple_occurence_layer,
                dlg=self.dlg,
                field=field,
                selected_only=self.dlg.layer_selection_checkbox.isChecked(),
                api_road=api_road,
            )
            self.collect_taxref.finished_dl.connect(
                lambda: self.finished_import(field=field)
            )
        else:
            print(api)
            # NOT SUPPOSED TO BE HERE
            self.pluginIsActive = False

    def collect_from_name(self, field, optional):
        if optional:
            list_rank = self.imported_rank.rank_list
        else:
            list_rank = None
        self.collect_taxref = GetTaxrefFromMNHN(
            network_manager=self.manager,
            project=self.project,
            layer=self.dlg.select_layer_combo_box.currentLayer(),
            no_occurence_layer=self.no_occurence_layer,
            one_occurence_layer=self.one_occurence_layer,
            multiple_occurence_layer=self.multiple_occurence_layer,
            dlg=self.dlg,
            field_name=field,
            field_rank=optional,
            list_rank=list_rank,
            selected_only=self.dlg.layer_selection_checkbox.isChecked(),
        )

        self.collect_taxref.finished_dl.connect(
            lambda: self.finished_import(field=field)
        )

    def finished_import(self, field):
        # Once it's finished, the ProgressBar is set back to 0
        if len(self.collect_taxref.no_occurence) > 0:
            self.project.instance().addMapLayer(self.no_occurence_layer, False)
            self.group.addLayer(self.no_occurence_layer)
        if len(self.collect_taxref.no_occurence) + len(
            self.collect_taxref.multi_occurence
        ) < len(self.collect_taxref.ids):
            self.project.instance().addMapLayer(self.one_occurence_layer, False)
            self.group.addLayer(self.one_occurence_layer)
        if len(self.collect_taxref.multi_occurence) > 0:
            self.project.instance().addMapLayer(self.multiple_occurence_layer, False)
            self.group.addLayer(self.multiple_occurence_layer)
        self.dlg.thread.finish()
        self.dlg.select_progress_bar_label.setText("")
        self.dlg.thread.reset_value()
        msg = QMessageBox()
        msg.information(
            None,
            self.tr("Informations"),
            self.tr("Species with occurences : ")
            + str(
                len(self.collect_taxref.ids)
                - len(self.collect_taxref.no_occurence)
                - len(self.collect_taxref.multi_occurence)
            )
            + "\n"
            + self.tr("Species without occurence : ")
            + str(len(self.collect_taxref.no_occurence))
            + "\n"
            + self.tr("Species with multiple occurences : ")
            + str(len(self.collect_taxref.multi_occurence)),
        )
        self.dlg.close()
        if len(self.collect_taxref.multi_occurence) > 0:
            multiples_occurences_dlg = MultipleTaxonSelectionDialog(
                layer=self.multiple_occurence_layer,
                field_name=field,
                multiple_values=self.collect_taxref.multiples_occurences_values,
            )
            multiples_occurences_dlg.show()

        self.pluginIsActive = False


class InternetChecker(QObject):
    """Constructor.

    Class wich is going to ping a website
    to know if the user is connected to internet.
    """

    finished = pyqtSignal()
    errors = pyqtSignal()

    def __init__(self, parent=None, manager=None):
        super().__init__(parent)
        self._manager = manager

    @property
    def manager(self):
        return self._manager

    @property
    def pending_ping(self):
        return self._pending_ping

    def ping(self, url):
        qrequest = QNetworkRequest(QUrl(url))
        self.reply = self.manager.get(qrequest)
        self.reply.finished.connect(self.handle_finished)

    def handle_finished(self):
        if self.reply.error() != QNetworkReply.NoError:
            # If the user does not have an internet connexion,
            # the plugin does not launch.
            msg = QMessageBox()
            if self.reply.error() == 403:
                msg.critical(
                    None,
                    self.tr("Error"),
                    self.tr("Service is down."),
                )
            elif self.reply.error() == 3:
                msg.critical(
                    None,
                    self.tr("Error"),
                    self.tr("You are not connected to the Internet."),
                )
            else:
                msg.critical(
                    None,
                    self.tr("Error"),
                    self.tr(
                        "Code error : {code}\nGo to\n{tracker}\nto report the issue.".format(
                            code=str(self.reply.error()), tracker=__uri_tracker__
                        )
                    ),
                )
            self.errors.emit()
        else:
            self.finished.emit()
