#! python3  # noqa: E265

"""
Plugin dialog.
"""

# PyQGIS
from qgis.core import QgsMapLayerProxyModel
from qgis.gui import QgsMapLayerComboBox
from qgis.PyQt.Qt import QUrl
from qgis.PyQt.QtCore import QSize, QThread, pyqtSignal
from qgis.PyQt.QtGui import QDesktopServices, QIcon, QPixmap
from qgis.PyQt.QtWidgets import (
    QCheckBox,
    QDialog,
    QDialogButtonBox,
    QGridLayout,
    QLabel,
    QProgressBar,
    QPushButton,
    QTabWidget,
    QToolButton,
    QVBoxLayout,
    QWidget,
)

# project
from taxref_collector.__about__ import (
    __service_credit_gbif__,
    __service_credit_inpn__,
    __service_credit_patri__,
    __service_credit_wikidata__,
    __service_logo_gbif__,
    __service_logo_inpn__,
    __service_logo_patri__,
    __service_logo_wikidata__,
    __service_metadata__,
    __title__,
    __uri_homepage__,
)
from taxref_collector.gui.dlg_tab_gbif import GbifTab
from taxref_collector.gui.dlg_tab_inpn import InpnTab
from taxref_collector.gui.dlg_tab_wikidata import WikidataTab

# ############################################################################
# ########## Classes ###############
# ##################################


class TaxrefCollectorDialog(QDialog):
    def __init__(self, project=None, iface=None, manager=None, locale=None):
        """Constructor.
        :param
        project: The current QGIS project instance
        iface: An interface instance that will be passed to this class which \
        provides the hook by which you can manipulate the QGIS application \
        at run time.
        """
        super(TaxrefCollectorDialog, self).__init__()
        self.setObjectName(__title__)

        self.iface = iface
        self.project = project
        self.manager = manager
        self.locale = locale
        self.canvas = self.iface.mapCanvas()

        self.layer = None
        self.rectangle = None
        tab_index = 0
        self.organiser = {}
        self.api_list = {}

        self.setWindowTitle(__title__)

        self.layout = QVBoxLayout()

        # Source and credit
        self.source_doc_layout = QGridLayout()
        credit_label = QLabel(self)
        credit_label.setText(self.tr("Data provided by :"))
        self.layout.addWidget(credit_label)

        pixmap = QPixmap(str(__service_logo_inpn__))
        self.producer_inpn_label = QToolButton(self)
        self.producer_inpn_label.setObjectName(__service_credit_inpn__)
        icon = QIcon()
        icon.addPixmap(pixmap)
        self.producer_inpn_label.setIcon(icon)
        self.producer_inpn_label.setIconSize(QSize(60, 60))
        self.source_doc_layout.addWidget(self.producer_inpn_label, 0, 0, 1, 2)
        pixmap = QPixmap(str(__service_logo_patri__))
        self.producer_col_label = QToolButton(self)
        self.producer_col_label.setObjectName(__service_credit_patri__)
        icon = QIcon()
        icon.addPixmap(pixmap)
        self.producer_col_label.setIcon(icon)
        self.producer_col_label.setIconSize(QSize(60, 60))
        self.source_doc_layout.addWidget(self.producer_col_label, 0, 1, 1, 2)
        pixmap = QPixmap(str(__service_logo_gbif__))
        self.producer_gbif_label = QToolButton(self)
        self.producer_gbif_label.setObjectName(__service_credit_gbif__)
        icon = QIcon()
        icon.addPixmap(pixmap)
        self.producer_gbif_label.setIcon(icon)
        self.producer_gbif_label.setIconSize(QSize(60, 60))
        self.source_doc_layout.addWidget(self.producer_gbif_label, 1, 0, 1, 2)
        pixmap = QPixmap(str(__service_logo_wikidata__))
        self.producer_wikidata_label = QToolButton(self)
        self.producer_wikidata_label.setObjectName(__service_credit_wikidata__)
        icon = QIcon()
        icon.addPixmap(pixmap)
        self.producer_wikidata_label.setIcon(icon)
        self.producer_wikidata_label.setIconSize(QSize(60, 60))
        self.source_doc_layout.addWidget(self.producer_wikidata_label, 1, 1, 1, 2)

        widget = QWidget()
        self.doc_layout = QVBoxLayout()
        self.documentation_button = QPushButton(self)
        if self.locale == "fr":
            doc_url = __uri_homepage__ + f"usage/{self.locale}_how_to_use.html"
        else:
            doc_url = __uri_homepage__ + "usage/{locale}_how_to_use.html".format(
                locale="en"
            )
        self.documentation_button.setObjectName(doc_url)
        self.documentation_button.setText(self.tr("Documentation"))
        self.doc_layout.addWidget(self.documentation_button)

        self.doc_layout.addStretch()

        self.metadata_button = QPushButton(self)
        self.metadata_button.setObjectName(__service_metadata__)
        self.metadata_button.setText(self.tr("Metadata"))
        self.doc_layout.addWidget(self.metadata_button)
        widget.setLayout(self.doc_layout)
        self.source_doc_layout.addWidget(widget, 0, 5, 1, 2)

        self.layout.addLayout(self.source_doc_layout)
        self.layout.insertSpacing(100, 25)

        # Select layer tool
        select_layer_label = QLabel(self)
        select_layer_label.setText(self.tr("Select a layer with data species"))
        self.layout.addWidget(select_layer_label)
        self.select_layer_combo_box = QgsMapLayerComboBox(self)
        self.select_layer_combo_box.setFilters(
            QgsMapLayerProxyModel.PolygonLayer
            | QgsMapLayerProxyModel.LineLayer
            | QgsMapLayerProxyModel.PointLayer
        )
        self.select_layer_combo_box.setAllowEmptyLayer(False)
        self.layout.addWidget(self.select_layer_combo_box)

        # Use only selection checkbox
        self.layer_selection_checkbox = QCheckBox(self)
        self.layer_selection_checkbox.setText(self.tr("Selected feature(s) only"))
        self.layer_selection_checkbox.setChecked(False)
        self.layout.addWidget(self.layer_selection_checkbox)

        self.layout.insertSpacing(100, 25)

        # QTabWidget test
        self.tab_widget_api = QTabWidget(self)

        # INPN TAB Warning to add
        inpn_tab = InpnTab(self)
        self.tab_widget_api.addTab(inpn_tab, "INPN (hacked)")
        self.tab_widget_api.setTabEnabled(tab_index, False)
        self.organiser = inpn_tab.append_organiser(self.organiser, tab_index)
        self.api_list["INPN"] = ["scientific_name"]
        tab_index = tab_index + 1

        # Wikidata TAB
        wikidata_tab = WikidataTab(self)
        self.tab_widget_api.addTab(wikidata_tab, "Wikidata")
        self.tab_widget_api.setTabEnabled(tab_index, True)
        self.organiser = wikidata_tab.append_organiser(self.organiser, tab_index)
        self.api_list["Wikidata"] = ["GBIF_ID", "iNaturalist_ID"]
        tab_index = tab_index + 1

        # GBIF TAB, api road not supported for now
        gbif_tab = GbifTab(self)
        self.tab_widget_api.addTab(gbif_tab, "GBIF")
        self.tab_widget_api.setTabEnabled(tab_index, True)
        self.organiser = gbif_tab.append_organiser(self.organiser, tab_index)
        self.api_list["GBIF"] = ["GBIF_ID", "scientific_name"]
        tab_index = tab_index + 1

        self.layout.addWidget(self.tab_widget_api)
        self.layout.insertSpacing(100, 25)

        # Accept and reject button box
        self.button_box = QDialogButtonBox(self)
        self.button_box.setEnabled(False)
        self.button_box.addButton(self.tr("Ok"), QDialogButtonBox.AcceptRole)
        self.button_box.addButton(self.tr("Cancel"), QDialogButtonBox.RejectRole)
        self.layout.addWidget(self.button_box)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)

        # Progress Bar
        self.select_progress_bar_label = QLabel(self)
        self.select_progress_bar_label.setText("")
        self.layout.addWidget(self.select_progress_bar_label)

        self.progress_bar = QProgressBar(self)
        self.progress_bar.setValue(0)
        self.thread = Thread()
        self.thread._signal.connect(self.signal_accept)
        self.layout.addWidget(self.progress_bar)

        # Add layout
        self.setLayout(self.layout)

        # Ui signals
        self.producer_gbif_label.clicked.connect(self.open_url)
        self.producer_col_label.clicked.connect(self.open_url)
        self.producer_inpn_label.clicked.connect(self.open_url)
        self.producer_wikidata_label.clicked.connect(self.open_url)
        self.metadata_button.clicked.connect(self.open_url)
        self.documentation_button.clicked.connect(self.open_url)
        self.select_layer_combo_box.layerChanged.connect(self.check_valid)

        self.check_valid()

    def check_valid(self):
        if self.select_layer_combo_box.currentLayer():
            if self.select_layer_combo_box.currentLayer().fields().count() > 0:
                self.button_box.setEnabled(True)
            else:
                self.button_box.setEnabled(False)
        else:
            self.button_box.setEnabled(False)

    def open_url(self):
        # Function to open the url of the buttons
        url = QUrl(self.sender().objectName())
        QDesktopServices.openUrl(url)

    def signal_accept(self, msg):
        # Update the progress bar when result is pressed
        self.progress_bar.setValue(int(msg))
        if self.progress_bar.value() == 101:
            self.progress_bar.setValue(0)

    def activate_window(self):
        # Put the dialog on top once the rectangle is drawn
        self.showNormal()
        self.activateWindow()


class Thread(QThread):
    """Thread used fot the ProgressBar"""

    _signal = pyqtSignal(int)

    def __init__(self):
        super(Thread, self).__init__()
        self.max_value = 1
        self.value = 0

    def __del__(self):
        self.wait()

    def set_max(self, max_value):
        self.max_value = max_value

    def add_one(self, to_add):
        self.value = self.value + to_add
        self._signal.emit(int((self.value / self.max_value) * 100))

    def finish(self):
        self._signal.emit(101)

    def reset_value(self):
        self.value = 0
