from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import (
    QComboBox,
    QDialog,
    QDialogButtonBox,
    QGridLayout,
    QHeaderView,
    QTableWidget,
    QTableWidgetItem,
    QVBoxLayout,
)


class MultipleTaxonSelectionDialog(QDialog):
    def __init__(self, parent=None, layer=None, field_name=None, multiple_values=None):
        """Dialog used to manage multiple taxon available for an occurence.
        layer : multiple occurences layer
        field_name : name of the field containing the taxon name
        multiple_values : list of the possible taxon for every occurences to fill the combobox
        """
        super(MultipleTaxonSelectionDialog, self).__init__(parent)

        self.layer = layer
        self.field_name = field_name

        self.setWindowTitle(self.tr("Multiple occurences management"))

        self.resize(400, 300)

        self.layout = QVBoxLayout()
        self.table = QTableWidget(self)

        # Set the size of the table.
        column_name = [self.tr("CD NOM"), self.tr("TAXREF's name")]
        self.table.setColumnCount(len(column_name))
        # Set label of the table.
        self.table.setHorizontalHeaderLabels(column_name)
        self.add_lines(multiple_values)
        self.layout.addWidget(self.table)

        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        # Accept and reject button box
        self.test_layout = QGridLayout()
        self.button_box = QDialogButtonBox(self)
        self.button_box.addButton(self.tr("Ok"), QDialogButtonBox.AcceptRole)
        self.button_box.addButton(self.tr("Cancel"), QDialogButtonBox.RejectRole)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)
        self.accepted.connect(self.update_layer)
        self.rejected.connect(self.close_window)
        self.test_layout.addWidget(self.button_box, 0, 2)
        self.layout.addLayout(self.test_layout)

        # Add layout
        self.setLayout(self.layout)

    def add_lines(self, values):

        self.table.setRowCount(len(values))
        print(list(values.keys()))
        self.table.setVerticalHeaderLabels(list(values.keys()))
        row_count = 0
        for elem in values:
            item = QTableWidgetItem(str(list(values[elem].keys())[0]))
            item.setFlags(Qt.ItemIsEditable)
            self.table.setItem(row_count, 0, item)
            combo_box = QComboBox()
            combo_box.addItems(list(values[elem].values()))
            combo_box.currentIndexChanged.connect(
                lambda state, cmb=combo_box, row_count=row_count, values=values, elem=elem: self.update_cd_nom(
                    cmb, row_count, values, elem
                )
            )
            self.table.setCellWidget(row_count, 1, combo_box)
            row_count = row_count + 1

    def update_cd_nom(self, cmb, row_count, values, elem):
        item = QTableWidgetItem(str(list(values[elem].keys())[cmb.currentIndex()]))
        item.setFlags(Qt.ItemIsEditable)
        self.table.setItem(
            row_count,
            0,
            item,
        )

    def update_layer(self):
        self.layer.startEditing()
        for row in range(self.table.rowCount()):
            old_name = self.table.verticalHeaderItem(row).text()
            new_cd_nom = self.table.item(row, 0).text()
            new_name = self.table.cellWidget(row, 1).currentText()
            # Create an expression with format
            expr = "\"{name_field}\" = '{old_name}' ".format(
                name_field=self.field_name, old_name=old_name
            )
            # Use the expression to select features
            self.layer.selectByExpression(expr)
            for feature in self.layer.getSelectedFeatures():
                self.layer.changeAttributeValue(
                    feature.id(),
                    self.layer.fields().indexFromName("taxref_name"),
                    new_name,
                )
            for feature in self.layer.getSelectedFeatures():
                self.layer.changeAttributeValue(
                    feature.id(),
                    self.layer.fields().indexFromName("cd_nom"),
                    new_cd_nom,
                )
            taxref_url = "https://inpn.mnhn.fr/espece/cd_nom/{cd_nom}".format(
                cd_nom=new_cd_nom
            )
            for feature in self.layer.getSelectedFeatures():
                self.layer.changeAttributeValue(
                    feature.id(),
                    self.layer.fields().indexFromName("taxref_url"),
                    taxref_url,
                )
        self.layer.commitChanges()
        self.layer.triggerRepaint()

    def close_window(self):
        self.close()
