"""
/***************************************************************************
 SwissGeoDownloader
                                 A QGIS plugin
 This plugin lets you comfortably download swiss geo data.
                             -------------------
        begin                : 2021-03-14
        copyright            : (C) 2025 by Patricia Moll
        email                : pimoll.dev@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os

from osgeo import gdal
from qgis.core import (
    QgsProject,
    QgsRasterLayer,
    QgsTask,
    QgsVectorLayer,
    QgsMessageLog,
    Qgis
)

from swissgeodownloader.api.responseObjects import STREAMED_SOURCE_PREFIX
from swissgeodownloader.ui.ui_utilities import MESSAGE_CATEGORY


class QgisLayerCreatorTask(QgsTask):
    """ QGIS can freeze when a lot of layers have to be created in the main
     thread. Instead, layers are created in this separate QTask and moved to
     the main thread. After the task has finished, they are added to the map
     in the main thread."""
    
    def __init__(self, description, fileList, vrtOutputPath):
        super().__init__(description, QgsTask.Flag.CanCancel)
        self.fileList = fileList
        self.layerList = []
        self.alreadyAdded = 0
        self.exception = None
        self.vrtOutputPath = vrtOutputPath
    
    def run(self):
        
        if not self.fileList or len(self.fileList) == 0:
            return True
        
        qgsProject = QgsProject.instance()
        already_added = [lyr.source() for lyr in
                         qgsProject.mapLayers().values()]
        
        progressStep = 100 / len(self.fileList)
        
        if self.vrtOutputPath:
            return self.combineTiles()
        
        for i, file in enumerate(self.fileList):
            if self.isCanceled():
                return False
            
            self.setProgress(i * progressStep)
            
            # Adding the file to QGIS if it's (1) a streamed file or (2) is
            #  present in the file system and (3) is not a .zip
            if (file.path.startswith(STREAMED_SOURCE_PREFIX) or os.path.exists(
                    file.path)) and '.zip' not in file.id:
                if file.path in already_added:
                    self.alreadyAdded += 1
                    continue
                
                # Try to create a raster layer first, if that fails, create a
                #  vector layer
                success = self.createRasterLayer(file.path, file.id)
                if success:
                    continue
                
                self.createVectorLayer(file.path, file.id)
        return True
    
    def combineTiles(self):
        try:
            pathList = [file.path for file in self.fileList]
            gdal.BuildVRT(self.vrtOutputPath, pathList)
        except Exception as e:
            self.exception = str(e)
            return False
        layerName = os.path.splitext(os.path.basename(self.vrtOutputPath))[0]
        return self.createRasterLayer(self.vrtOutputPath, layerName)
    
    def createRasterLayer(self, filepath, filename):
        try:
            rasterLyr = QgsRasterLayer(filepath, filename)
            if rasterLyr.isValid():
                self.layerList.append(rasterLyr)
                return True
            else:
                del rasterLyr
        except Exception as e:
            self.exception = f'{filename}: {e}' if self.exception is None else self.exception + f'\n{filename}: {e}'
        return False
    
    def createVectorLayer(self, filepath, filename):
        try:
            vectorLyr = QgsVectorLayer(filepath, filename, "ogr")
            if vectorLyr.isValid():
                self.layerList.append(vectorLyr)
                return True
            else:
                del vectorLyr
        except Exception as e:
            self.exception = f'{filename}: {e}' if self.exception is None else self.exception + f'\n{filename}: {e}'
        return False
    
    def finished(self, result):
        self.setProgress(100)
        if not result:
            if self.isCanceled():
                self.exception = self.tr('Aborted by user')
            elif self.exception is None:
                self.exception = self.tr('An unknown error occurred')
        for e in str(self.exception).split('\n'):
            QgsMessageLog.logMessage(e, MESSAGE_CATEGORY,
                                     Qgis.MessageLevel.Warning)
