"""
/***************************************************************************
 SwissGeoDownloader
                                 A QGIS plugin
 This plugin lets you comfortably download swiss geo data.
                             -------------------
        begin                : 2021-03-14
        copyright            : (C) 2025 by Patricia Moll
        email                : pimoll.dev@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os

from qgis.core import (QgsProject, QgsRasterLayer, QgsTask, QgsVectorLayer)

from swissgeodownloader import DEBUG
from swissgeodownloader.api.responseObjects import STREAMED_SOURCE_PREFIX


class QgisLayerCreatorTask(QgsTask):
    """ QGIS can freeze when a lot of layers have to be created in the main
     thread. Instead, layers are created in this separate QTask and moved to
     the main thread. After the task has finished, they are added to the map
     in the main thread."""
    
    def __init__(self, description, fileList):
        super().__init__(description, QgsTask.Flag.CanCancel)
        self.fileList = fileList
        self.layerList = []
        self.alreadyAdded = 0
        self.exception = None
    
    def run(self):
        if DEBUG:
            try:
                # Add pydevd to path
                import sys
                sys.path.insert(0,
                                '/snap/pycharm-professional/current/debug-eggs/pydevd-pycharm.egg')
                import pydevd_pycharm
                pydevd_pycharm.settrace('localhost', port=53100, suspend=False,
                                        stdoutToServer=True,
                                        stderrToServer=True)
            except ConnectionRefusedError:
                pass
            except ImportError:
                pass
        
        if not self.fileList or len(self.fileList) == 0:
            return True
        
        qgsProject = QgsProject.instance()
        already_added = [lyr.source() for lyr in
                         qgsProject.mapLayers().values()]
        
        progressStep = 100 / len(self.fileList)
        
        for i, file in enumerate(self.fileList):
            if self.isCanceled():
                return False
            
            self.setProgress(i * progressStep)
            
            # Adding the file to QGIS if it's (1) a streamed file or (2) is
            #  present in the file system and (3) is not a .zip
            if (file.path.startswith(STREAMED_SOURCE_PREFIX) or os.path.exists(
                    file.path)) and '.zip' not in file.id:
                if file.path in already_added:
                    self.alreadyAdded += 1
                    continue
                try:
                    rasterLyr = QgsRasterLayer(file.path, file.id)
                    if rasterLyr.isValid():
                        self.layerList.append(rasterLyr)
                        continue
                    else:
                        del rasterLyr
                except Exception:
                    pass
                try:
                    vectorLyr = QgsVectorLayer(file.path, file.id, "ogr")
                    if vectorLyr.isValid():
                        self.layerList.append(vectorLyr)
                        continue
                    else:
                        del vectorLyr
                except Exception:
                    pass
        
        self.setProgress(100)
        return True
    
    def finished(self, result):
        if not result:
            if self.isCanceled():
                self.exception = self.tr('Aborted by user')
            elif self.exception is None:
                self.exception = self.tr('An unknown error occurred')
