# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SwisstopoGeocoder
                                 A QGIS plugin
 This is a process designed to provide geocoding functionality using the SwissTopo geocoding service.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-11-17
        copyright            : (C) 2023 by Geometa Lab
        email                : geometalab@ost.ch
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Geometa Lab'
__date__ = '2023-11-17'
__copyright__ = '(C) 2023 by Geometa Lab'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import requests
import time
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterField,
                       QgsPointXY,
                       QgsGeometry,
                       QgsWkbTypes,
                       QgsProject,
                       QgsCoordinateReferenceSystem,
                       QgsCoordinateTransform)


class SwisstopoGeocoderAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    LAYER_INPUT = 'LAYER_INPUT'
    COLUMN_INPUT = 'COLUMN_INPUT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.LAYER_INPUT,
                self.tr('Input layer'),
                [QgsProcessing.TypeFile]
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.COLUMN_INPUT,
                self.tr('Select an attribute'),
                parentLayerParameterName=self.LAYER_INPUT
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Output layer')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        swisstopo_crs = QgsCoordinateReferenceSystem("EPSG:4326")

        target_crs_code = QgsProject.instance().crs().authid()  # Use the active QGIS project CRS
        target_crs = QgsCoordinateReferenceSystem(target_crs_code)

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.LAYER_INPUT, context)
        column_source = self.parameterAsString(parameters, self.COLUMN_INPUT, context)

        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                                               context, source.fields(), QgsWkbTypes.Point, target_crs)

        transform = QgsCoordinateTransform(swisstopo_crs, target_crs, QgsProject.instance())

        # Compute the number of steps to display within the progress bar and
        total = 100.0 / source.featureCount() if source.featureCount() else 0

        # get features from source
        features = source.getFeatures()

        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            # Get address from feature
            address = feature[column_source]

            try:
                get_feature_by_address(feature, address, transform)
            except Exception as e:
                feedback.pushInfo(address + " failed\nError: " + str(e))

            # Add a feature in the sink
            sink.addFeature(feature, QgsFeatureSink.FastInsert)

            # Update the progress bar
            feedback.setProgress(int(current * total))
            time.sleep(1)

        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {self.OUTPUT: dest_id}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Swiss GeoAdmin Bulk Geocoder'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return SwisstopoGeocoderAlgorithm()


def get_feature_by_address(fet, address, transform):
    url = "https://api3.geo.admin.ch/rest/services/api/SearchServer?origins=address&type=locations&sr=2056&searchText="
    query = address

    response = requests.get(url + query)
    response_json = response.json()

    latitude = response_json['results'][0]['attrs']['lat']
    longitude = response_json['results'][0]['attrs']['lon']

    if latitude is not None and longitude is not None:
        # Create a point with the API coordinates
        point = QgsPointXY(float(longitude), float(latitude))
        transformed_point = QgsPointXY(transform.transform(point))

        fet.setGeometry(QgsGeometry.fromPointXY(transformed_point))
