# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SubsetRaster
                                 A QGIS plugin
 This plugin allows user to create subset of a raster.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2026-01-26
        git sha              : $Format:%H$
        copyright            : (C) 2026 by Murat Çalışkan
        email                : caliskan.murat.20@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon, QColor
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .subset_raster_dialog import SubsetRasterDialog
import os.path

from .DrawRect import RectangleMapTool

from qgis.core import QgsProject, Qgis, QgsRasterLayer, QgsProcessingUtils

from pathlib import Path
from osgeo import ogr, gdal, osr
from pyproj import CRS, Transformer
from shapely.wkt import loads
from shapely.ops import transform
from shapely.geometry import box
from datetime import date

class SubsetRaster:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'SubsetRaster_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Subset Raster')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('SubsetRaster', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/subset_raster/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Subset Raster'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Subset Raster'),
                action)
            self.iface.removeToolBarIcon(action)
        
    def selectOutputFolder(self):
        self.dlg.le_outputFolder.setText("")
        output_dir = QFileDialog.getExistingDirectory(None, 'Select working directory', "", QFileDialog.ShowDirsOnly)
        self.dlg.le_outputFolder.setText(output_dir)
        
    def getDrawnCoor(self, map_canvas):
        self.srs_wkt = self.canvas.crs().toWkt()
        self.dlg.showMinimized()
        self.rect = RectangleMapTool(map_canvas, self.dlg)
        map_canvas.setMapTool(self.rect)
        
        self.checkCRS()
    
    def getCanvasExtent(self):
        self.srs_wkt = self.canvas.crs().toWkt()
        ext = self.iface.mapCanvas().extent()
        
        self.dlg.sb_extent_minx.setValue(ext.xMinimum())
        self.dlg.sb_extent_miny.setValue(ext.yMinimum())
        self.dlg.sb_extent_maxx.setValue(ext.xMaximum())
        self.dlg.sb_extent_maxy.setValue(ext.yMaximum())
        
        self.checkCRS()
     
    def getLayerExtent(self):
        layerName = self.dlg.cb_layers.currentText()
        layers = QgsProject.instance().mapLayersByName(layerName)
        if len(layers) > 0:     
            ext = layers[0].extent()
            self.srs_wkt = layers[0].crs().toWkt()
            
            self.dlg.sb_extent_minx.setValue(ext.xMinimum())
            self.dlg.sb_extent_miny.setValue(ext.yMinimum())
            self.dlg.sb_extent_maxx.setValue(ext.xMaximum())
            self.dlg.sb_extent_maxy.setValue(ext.yMaximum())
        
        self.checkCRS()

    def getRasters(self):
        sender = self.dlg.sender()
        name = sender.objectName()
        
        if name == "pb_browse":
            existing_items = [Path(self.dlg.lw_input_ql.item(i).text()).resolve() for i in range(self.dlg.lw_input_ql.count())]
            
            filePaths, _filter = QFileDialog.getOpenFileNames(self.dlg, "Select Raster File(s)", "", 'All Files (*.*)')
            for fp in filePaths:
                if not Path(fp).resolve() in existing_items:
                    try:
                        _ = gdal.Open(fp)
                        self.dlg.lw_input_ql.addItem(fp)
                    except:
                        pass
            
        elif name == "pb_getLayers":
            existing_items = [Path(self.dlg.lw_input_ql.item(i).text()).resolve() for i in range(self.dlg.lw_input_ql.count())]
            
            for layer in QgsProject.instance().mapLayers().values():
                if layer.type()== 1:
                    src = layer.source()
                    if not Path(src).resolve() in existing_items:
                        try:
                            _ = gdal.Open(src)
                            self.dlg.lw_input_ql.addItem(src)
                        except:
                            pass
        
        elif name == "pb_remove":
            if self.dlg.lw_input_ql.count() > 0:
                selected_items = self.dlg.lw_input_ql.selectedItems()
                
                if len(selected_items) > 0:
                    for item in selected_items:
                        self.dlg.lw_input_ql.takeItem(self.dlg.lw_input_ql.row(item))
        
        elif name == "pb_clear":
            self.dlg.lw_input_ql.clear()
        
        
        self.dlg.label.setText(f"Raster List ({self.dlg.lw_input_ql.count()})")
        
        self.checkCRS()

    def fillCompressWidget(self):
        self.dlg.cb_compressions.clear()            
        if self.dlg.rb_vrt.isChecked():
            self.dlg.cb_compressions.addItem("NONE")
            
        elif self.dlg.rb_gtiff.isChecked():
            compression_list = ["DEFLATE", "NONE", "LZW", "JPEG", "PACKBITS", "CCITTRLE", "CCITTFAX3", "CCITTFAX4", "LZMA", "ZSTD", "LERC", 
                                "LERC_DEFLATE", "LERC_ZSTD", "WEBP", "JXL"]
            self.dlg.cb_compressions.addItems(compression_list)
            
        elif self.dlg.rb_cog.isChecked():
            compression_list = ["DEFLATE", "NONE", "LZW", "JPEG", "ZSTD", "WEBP", "LERC", "LERC_DEFLATE", "LERC_ZSTD", "LZMA"]
            self.dlg.cb_compressions.addItems(compression_list)
    
    def checkCRS(self):
        ql_count = self.dlg.lw_input_ql.count()
        srs_conflicts = 0
        srs1 = CRS.from_wkt(self.srs_wkt)
        
        epsg = srs1.to_epsg()
        if epsg:
            self.dlg.lbl_epsg.setText(f"EPSG : {epsg}")
        else:
            self.dlg.lbl_epsg.setText("EPSG : Unknown")
        
        for i in range(ql_count):
            it = self.dlg.lw_input_ql.item(i)
            rp = it.text()
            ds = gdal.Open(rp)
            srs2_wkt = ds.GetSpatialRef().ExportToWkt()
            srs2 = CRS.from_wkt(srs2_wkt)
            
            if not srs1.equals(srs2):
                it.setForeground(QColor("red"))
                srs_conflicts += 1
            else:
                it.setForeground(QColor("black"))
            ds = None
        
        if srs_conflicts > 0:
            self.dlg.label_2.setText("Rasters with conflicting SRS are shown in red.")
        else:
            self.dlg.label_2.setText("")
    
    def refreshEpsg(self, *arg, **kwargs):
        srs_old = self.srs_wkt
        self.srs_wkt = self.canvas.crs().toWkt()
        
        minX_ = float(self.dlg.sb_extent_minx.value())
        maxX_ = float(self.dlg.sb_extent_maxx.value())
        minY_ = float(self.dlg.sb_extent_miny.value())
        maxY_ = float(self.dlg.sb_extent_maxy.value()) 
        
        geom_wkt = box(minX_, minY_, maxX_, maxY_).wkt
        minX, maxX, minY, maxY = self.getTransformedGeometry(geom_wkt, srs_old, self.srs_wkt)
        
        self.dlg.sb_extent_minx.setValue(minX)
        self.dlg.sb_extent_miny.setValue(minY)
        self.dlg.sb_extent_maxx.setValue(maxX)
        self.dlg.sb_extent_maxy.setValue(maxY)
        
        self.checkCRS()
    
    def getTransformedGeometry(self, geom_wkt, src_srs_wkt, dest_srs_wkt):
        geom = loads(geom_wkt)
        
        from_crs = CRS.from_wkt(src_srs_wkt)
        to_crs = CRS.from_wkt(dest_srs_wkt)
        
        transformer = Transformer.from_crs(from_crs, to_crs, always_xy=True)
        projected = transform(transformer.transform, geom)
        
        proj_geom = ogr.CreateGeometryFromWkt(projected.wkt)       
        
        return proj_geom.GetEnvelope()
    
    def generateOutputName(self, output_folder_path, name, ext):
        out_name_ = name + "_clipped"

        if not os.path.isfile(os.path.join(output_folder_path, f"{out_name_}.{ext}")):
            out_name = out_name_
        else:
            n = 2
            while True:
                out_name = out_name_ + "_" + str(n)
                if not os.path.isfile(os.path.join(output_folder_path, f"{out_name}.{ext}")):
                    break
                else:
                    n += 1    
        
        return f"{out_name}.{ext}"
    
    def subsetRaster(self, input_raster_path, output_folder_path, name, creationOptions, minX_, minY_, maxX_, maxY_):
        srs1 = CRS.from_wkt(self.srs_wkt)
        
        ds = gdal.Open(input_raster_path)
        srs2_wkt = ds.GetSpatialRef().ExportToWkt()
        srs2 = CRS.from_wkt(srs2_wkt)
        ds = None
        
        if not srs1.equals(srs2):
            geom_wkt = box(minX_, minY_, maxX_, maxY_).wkt
            minX, maxX, minY, maxY = self.getTransformedGeometry(geom_wkt, self.srs_wkt, srs2_wkt)
        else:
            minX, minY, maxX, maxY = minX_, minY_, maxX_, maxY_
        
        if self.dlg.rb_gtiff.isChecked():
            output_name = self.generateOutputName(output_folder_path, name, "tif")            
            output_file_path = os.path.join(output_folder_path, output_name)
            
            out_raster = gdal.Warp(
                output_file_path,
                input_raster_path,
                outputBounds = (minX, minY, maxX, maxY),
                format = "GTiff",
                creationOptions = creationOptions,
                dstAlpha = True if self.dlg.chc_addAlpha.isChecked() else False
            )
            out_raster = None
            
        
        elif self.dlg.rb_cog.isChecked():
            output_name = self.generateOutputName(output_folder_path, name, "tif")            
            output_file_path = os.path.join(output_folder_path, output_name)
            
            out_raster = gdal.Warp(
                output_file_path,
                input_raster_path,
                outputBounds = (minX, minY, maxX, maxY),
                format = "COG",
                creationOptions = creationOptions,
                dstAlpha = True if self.dlg.chc_addAlpha.isChecked() else False
            )
            out_raster = None
        
        
        elif self.dlg.rb_vrt.isChecked():
            output_name = self.generateOutputName(output_folder_path, name, "vrt")            
            output_file_path = os.path.join(output_folder_path, output_name)
            
            out_raster = gdal.BuildVRT(
                output_file_path,
                input_raster_path,
                outputBounds = (minX, minY, maxX, maxY),
                addAlpha = True if self.dlg.chc_addAlpha.isChecked() else False
            )
            out_raster = None
            
        
        if self.dlg.chb_addToMap.isChecked():
            layer = QgsRasterLayer(output_file_path, output_name+"_clipped", "gdal")
            QgsProject.instance().addMapLayer(layer)
            layer = None
        
        return out_raster
        
    def executeSubset(self):
        ql_count = self.dlg.lw_input_ql.count()
        
        if self.dlg.le_outputFolder.text() == "":
            output_folder_path = QgsProcessingUtils.tempFolder()
        else:
            output_folder_path = self.dlg.le_outputFolder.text()
            
        if not os.path.isdir(output_folder_path):
            QMessageBox.critical(None, "ERROR", """Invalid folder path""")
            return
        
        if ql_count == 0:
            QMessageBox.critical(None, "ERROR", """No raster in raster list!""")
            return
                
        self.dlg.setEnabled(False)
        self.dlg.pushButton.setText("RUNNING...")
        self.dlg.processEvents() 
        
        minX_ = float(self.dlg.sb_extent_minx.value())
        maxX_ = float(self.dlg.sb_extent_maxx.value())
        minY_ = float(self.dlg.sb_extent_miny.value())
        maxY_ = float(self.dlg.sb_extent_maxy.value())
        
        compress = None if self.dlg.cb_compressions.currentText() == "NONE" else self.dlg.cb_compressions.currentText()
        creationOptions = [f"COMPRESS={compress}", *[i.strip() for i in self.dlg.le_options.text().split(",") if not i.strip().lower().startswith("compress")]]
        
        for i in range(ql_count):
            input_raster_path = self.dlg.lw_input_ql.item(i).text()
            name = os.path.split(input_raster_path)[-1].split(".")[0]
            
            _ = self.subsetRaster(input_raster_path, output_folder_path, name, creationOptions, minX_, minY_, maxX_, maxY_)
        
        
        self.iface.messageBar().pushMessage("Successful", "Subset(s) created!" , level=Qgis.Success, duration=5)
        
        self.dlg.setEnabled(True)
        self.dlg.pushButton.setText("RUN")
        self.dlg.processEvents()
        

    def run(self):
        if self.first_start == True:
            self.dlg = SubsetRasterDialog()
            
            if any([
                (date.today().day == 18 and date.today().month == 3),
                (date.today().day == 23 and date.today().month == 4),
                (date.today().day == 19 and date.today().month == 5),
                (date.today().day == 30 and date.today().month == 8),
                (date.today().day == 29 and date.today().month == 10),
                (date.today().day == 10 and date.today().month == 11)
                ]):
                
                self.dlg.setWindowIcon(QIcon(':/plugins/subset_raster/mka.png'))
            else:
                self.dlg.setWindowIcon(QIcon(':/plugins/subset_raster/icon.png'))
            
            self.dlg.lbl_info.setText("""<html><head/><body><a href="https://github.com/caliskanmurat/qgis_subset_raster_plugin"><img width="20" height="20" src=":/plugins/subset_raster/info.svg"/></a></body></html>""")
            
            self.canvas = QgsProject.instance()            
            self.srs_wkt = self.canvas.crs().toWkt()
            
            layers = [v.name() for v in QgsProject.instance().mapLayers().values()]
            self.dlg.cb_layers.clear()
            self.dlg.cb_layers.addItems(layers)
            
            self.dlg.lw_input_ql.clear()
            for layer in QgsProject.instance().mapLayers().values():
                if layer.type()== 1:
                    src = layer.source()
                    try:
                        ttt = gdal.Open(src)
                        if ttt:
                            self.dlg.lw_input_ql.addItem(layer.source())
                    except:
                        pass
            
            self.dlg.label.setText(f"Raster List ({self.dlg.lw_input_ql.count()})")
            self.checkCRS()
            
            self.dlg.btn_canvasExtent.clicked.connect(self.getCanvasExtent)
            self.dlg.btn_layerextent.clicked.connect(self.getLayerExtent)
            self.dlg.tbtn_draw.clicked.connect(lambda x:self.getDrawnCoor(self.iface.mapCanvas()))
            self.dlg.rb_vrt.toggled.connect(self.fillCompressWidget)
            self.dlg.rb_gtiff.toggled.connect(self.fillCompressWidget)
            self.dlg.rb_cog.toggled.connect(self.fillCompressWidget)
            
            self.dlg.pushButton.clicked.connect(self.executeSubset)
            self.dlg.tb_browse.clicked.connect(self.selectOutputFolder)
            
            self.dlg.pb_browse.clicked.connect(self.getRasters)
            self.dlg.pb_getLayers.clicked.connect(self.getRasters)
            self.dlg.pb_remove.clicked.connect(self.getRasters)
            self.dlg.pb_clear.clicked.connect(self.getRasters)
            
            self.dlg.lbl_refresh.mousePressEvent = self.refreshEpsg
        
        self.dlg.show()
