"""
/***************************************************************************
 srid part of Coordinate Systems Updater
 A QGIS plugin for updating SRID's that need to be improved

 srid contains classes used to define srid items and provide functionality to make items visible in list.
 -------------------
 begin                : 2011-07-24
 copyright            : (C) 2011 by Diethard Jansen
 email                : Diethard.Jansen at Gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt4.QtCore import *
from PyQt4.QtGui import *
import os
import qgis
"""
Next import statement using try is neccesary to make the connection possible
to sqlite database on Mac (which includes pysqlite) or Windows (which includes
sqlite3) in it's python distribution (actually it probably depends more on the
python distribution itself).
sqlite3 like sqlite itself does not do much on type checking which is by design.
pysqlite2 does type checking and needs additionally a commit for every mutation
on sqlite database.
"""
try:
    from pysqlite2 import dbapi2 as sqlite
except:
    from sqlite3 import dbapi2 as sqlite
    
def path_srs_db_qgis():
    """
    returns full path to sqlite database used by qgis to store
    many coordinate system used for projection worldwide
    """
    l_path1 = os.path.join(qgis.__path__[0], "../../resources/srs.db")
    l_path_linux = '/usr/share/qgis/resources/srs.db'
    l_path_srs_db = None
    for i_path in [l_path1, l_path_linux]:
        l_path = os.path.abspath(i_path)
        if os.path.exists(l_path):
            l_path_srs_db = l_path
            break
    return l_path_srs_db

def path_srs_db4update():
    """
    returns full path to csv file that contains coordinate systems
    that may need an update
    """
    l_dir = os.path.dirname(__file__)
    l_path = os.path.join(l_dir, "srids2update.csv")
    return os.path.abspath(l_path)

class Cs_ref(object):
    
    def __init__(self, p_srid, p_description, p_parameters):
        self.srid = p_srid
        self.description = QString(p_description)
        self.parameters = QString(p_parameters)
        self.available = None # definition available in QGIS?
        self.update = None # does it need to be updated?
        self.restore = None # can it be restored?

    def __lt__(self, p_other):
        """
        compares two Cs_ref objects, used for sorting
        """
        return int(self.srid) < int(p_other.srid)

    def color(self):
        # Returns colour to be used to highlight item in list
        # green means, no update needed
        # red means, no update possible, because definition is missing in QGIS.
        # blue means, update available, after update these should become green.
        # darkgreen means, coord system def is updated and can be restored
        l_colour = Qt.red
        if self.available == True:
            if self.update == True:
                l_colour = Qt.blue
            else:
                if self.restore == True:
                    l_colour = Qt.darkGreen
                else:
                    l_colour = Qt.green
        return l_colour

class Cs_ref_container(object):

    def __init__(self, p_owner=None):
        # the owner is only used for debugging purposes, show the contents
        # of some variables in a message when neccesary.
        self.owner = p_owner
        self.cs_refs = {}
        # create connection to qgis srs database and initialise
        self.authorised = True
        self.srs_db = self.initialise_srs_db_qgis()

    def cs_ref(self, identity):
        return self.cs_refs.get(identity)
        
    def __len__(self):
        return len(self.cs_refs)

    def __iter__(self):
        for cs_ref in self.cs_refs.values():
            yield cs_ref

    def inOrder(self):
        return sorted(self.cs_refs.values())

    def load(self):
        """
        load csv file holding updated coordinate system references
        """
        l_file = open("%s" % path_srs_db4update())
        for i_line in l_file:
            l_record = i_line.strip().split(';')
            if len(l_record)<3 or l_record[0] == 'srid':
                continue
            l_srid = int(l_record[0])
            l_description = unicode(l_record[1])
            l_parameters = unicode(l_record[2])
            l_cs_ref = Cs_ref(l_srid, l_description, l_parameters)
            self.cs_refs[id(l_cs_ref)] = l_cs_ref
        self.set_statusses_cs_refs()
        l_file.close()
        
    def initialise_srs_db_qgis(self):
        """
        connect to srs database of qgis and create tables necessary for
        restore and another virtual one for easy compare (which srids needs
        update and all).
        return open connection to database so it can be stored in object
        attribute.
        """
        l_con = sqlite.connect(path_srs_db_qgis())
        l_cursor = l_con.cursor()
        # find out if there allready exists a table for restore
        l_request = "select tbl_name from sqlite_master"
        l_records = l_cursor.execute(l_request)
        
        # create first a list of table names from record selection
        l_table_names = []
        for i_record in l_records:
            l_table_name = i_record[0]
            l_table_names.append(l_table_name)

        if not u"srids4restore" in l_table_names:
            # there is no restore table yet, create it!
            l_request = 'create table srids4restore(srid INTEGER, description \
            TEXT, parameters TEXT)'
            try:
                l_cursor.execute(l_request)
                l_con.commit()
            except:
                self.authorised = False
                l_dlg = self.owner
                l_msg = l_dlg.tr("Not authorised to update SRS database!\n\n\
Unfortunately you are not authorised to\n\
change the file srs.db which holds all\n\
Coordinate Reference Systems Definitions\n\
delivered with QGIS.\n\n\
The list will remain empty untill authorisation\n\
problem is solved.\n\n\
Use [Help] button in menu to open manual\n\
which explains how to change authorisation!")
                QMessageBox.warning(self.owner,
                                    self.owner.tr("No table access"),
                                    l_msg)
               
        l_cursor.close()
        
        return l_con

    def set_statusses_cs_refs(self):
        """
        set the statusses of coordinate systems references provided for
        update, using three fields in each cs_ref we can cover all possible
        mutations done to the actual coordinate reference system database
        provided with qgis.
        """
        l_cursor = self.srs_db.cursor()
        
        for i_cs_ref in self.cs_refs.itervalues():
            l_srid = i_cs_ref.srid
            # find out if coordinate system is available.
            l_request = "select srid, parameters from tbl_srs \
            where srid = %s" % str(l_srid)
            l_record = l_cursor.execute(l_request).fetchone()
            if l_record is None:
                i_cs_ref.available = False
                i_cs_ref.update = False
            else:
                i_cs_ref.available = True
                if l_record[1] == i_cs_ref.parameters:
                    i_cs_ref.update = False
                else:
                    #for debug purposes only!
                    #l_msg = "%s\n%s" % (i_cs_ref.parameters, unicode(l_record[1]))
                    #QMessageBox.warning(self.owner, "compare", l_msg)
                                        
                    i_cs_ref.update = True
                    
            # now search if there is a record to restore in table srids4restore
            # if there is this means it was update and can be restored if
            # necessary
            l_request = "select srid from srids4restore \
            where srid = %s" % str(l_srid)
            l_record = l_cursor.execute(l_request).fetchone()
            if l_record is None:
                i_cs_ref.restore = False
            else:
                i_cs_ref.restore = True
        l_cursor.close

    def update_all(self):
        """
        updating all coordinate systems that are indicated for update
        """
        l_cs_refs = self.cs_refs.values()
        self._update(l_cs_refs)

    def update_selected(self, p_selected_items):
        """
        updating all selected coordinate systems that are indicated for update
        """
        l_cs_refs = self._values_from_selected(p_selected_items)
        self._update(l_cs_refs)
        
    def _update(self, p_cs_refs):
        """
        before updating all coordinate systems that are indicate
        for update, first store original coordinate system to be able to
        restore when necessary
        """
        l_con = self.srs_db
        l_cursor = l_con.cursor()
        for i_cs_ref in p_cs_refs:
            if i_cs_ref.update == True:
                l_srid = str(i_cs_ref.srid)
                # first store cs_ref to update in table for restore
                l_request = "select srid, description, parameters from tbl_srs \
                where srid=%s" % l_srid
                l_record = l_cursor.execute(l_request).fetchone()
                l_request = 'insert into srids4restore \
                (srid, description, parameters) values (%s, "%s", "%s")' % \
                (str(l_record[0]), l_record[1], l_record[2])
                self.execute_sql_safe(l_request)

                # now update cs_ref with new provided cs_ref with plugin
                l_request= 'update tbl_srs set parameters="%s" where \
                srid=%s' % (unicode(i_cs_ref.parameters), l_srid)
                self.execute_sql_safe(l_request)
        l_cursor.close()
        self.set_statusses_cs_refs()
        
    def restore_all(self):
        """
        restore all coordinate systems that are indicated for restore
        """
        l_cs_refs = self.cs_refs.values()
        self._restore(l_cs_refs)
        
    def restore_selected(self, p_selected_items):
        """
        restore selected coordinate systems that are indicated for restore
        """
        l_cs_refs = self._values_from_selected(p_selected_items)
        self._restore(l_cs_refs)
        
    def _restore(self, p_cs_refs):
        """
        restore coordinate systems that are indicated for restore
        """
        l_con = self.srs_db
        l_cursor = l_con.cursor()
        for i_cs_ref in p_cs_refs:
            if i_cs_ref.restore == True:
                l_srid = str(i_cs_ref.srid)

                # first get cs_ref from table holding restore values
                l_request = "select srid, parameters from srids4restore \
                where srid=%s" % l_srid
                l_record = l_cursor.execute(l_request).fetchone()

                # now restore back in table used by qgis!
                l_request= 'update tbl_srs set parameters="%s" where \
                srid=%s' % (l_record[1], l_srid)
                self.execute_sql_safe(l_request)

                # after updating, delete record from srids4restore
                l_request = 'delete from srids4restore \
                where srid = %s' % l_srid
                self.execute_sql_safe(l_request)
        l_cursor.close()
        self.set_statusses_cs_refs()

    def _values_from_selected(self, p_selected_items):
        """
        return a list of Cs_ref objects using a give list of selected
        widget list items
        """
        l_cs_ref_values = []
        for i_item in p_selected_items:
            l_srid = int(i_item.text().split(':')[0])
            for i_cs_ref in self.cs_refs.itervalues():
                if l_srid == i_cs_ref.srid:
                    l_cs_ref_values.append(i_cs_ref)
        return l_cs_ref_values

    def execute_sql_safe(self, p_sql_request):
        """
        handles the sql requests to insert, update or delete in a more
        secure way
        """
        l_con = self.srs_db
        l_cursor = l_con.cursor()
    
        try:
            l_cursor.execute(p_sql_request)
            l_con.commit()
        except sqlite.DatabaseError:
            l_con.rollback()
    
        l_cursor.close()
