# -*- coding: utf-8 -*-

from DlgDbError import DlgDbError

from PyQt4.QtCore import *
from PyQt4.QtGui import *

import spatialite_utils


class TreeItem:
	
	def __init__(self, parent):
		self.parentItem = parent
		self.childItems = []
		
		if parent:
			parent.appendChild(self)
			
	def appendChild(self, child):
		self.childItems.append(child)
	
	def child(self, row):
		return self.childItems[row]
	
	def childCount(self):
		return len(self.childItems)
	
	def row(self):
		
		if self.parentItem:
			items = self.parentItem.childItems
			for (row,item) in enumerate(items):
				if item is self:
					return row
			
		return 0
	
	def parent(self):
		return self.parentItem
	
	def icon(self):
		return None



class DatabaseItem(TreeItem):
	def __init__(self, parent=None):
		TreeItem.__init__(self, parent)
	
	def data(self, column):
		return None

	def constructTreeFromDb(self, db):
		""" creates a tree of schemas and tables from current DB connection """
		
		self.tableCount = 0
		
		if not db:
			return
		
		# there is only one db
		item = DBItem(db.dbname, self)
			
		try:
			list_tables = db.list_geotables()
		except spatialite_utils.DbError, e:
			DlgDbError.showError(e, None)
			return
		
		self.tableCount = 0
		# add all tables
		for tbl in list_tables:
			tablename, reltype, geom_col, geom_type, geom_dim, geom_srid = tbl
			
			is_view = (reltype == 'view')
			tableItem = TableItem(tablename, is_view, geom_type, geom_col, geom_dim, geom_srid, item)

			self.tableCount += 1

	
class DBItem(TreeItem):
	def __init__(self, name, parent):
		TreeItem.__init__(self, parent)
		self.name = name
		
		# load (shared) icon with first instance of db item
		if not hasattr(DBItem, 'dbIcon'):
			DBItem.dbIcon = QIcon(":/icons/namespace.png")
	
	def data(self, column):
		if column == 0:
			return self.name
		else:
			return None
	
	def icon(self):
		return self.dbIcon


class TableItem(TreeItem):
	
	def __init__(self, name, is_view, geom_type, geom_column, geom_dim, geom_srid, parent):
		TreeItem.__init__(self, parent)
		self.name, self.is_view = name, is_view
		self.geom_type, self.geom_column = geom_type, geom_column

		self.geom_dim = geom_dim
		if self.geom_dim == None or self.geom_dim == "" or \
				(isinstance(self.geom_dim, int) and self.geom_dim <= 0):
			self.geom_dim = 'Unknown'
			
		self.geom_srid = geom_srid if geom_srid != None else -1
		self.row_count_real = -1 # not known (expensive to find out)
		
		# load (shared) icon with first instance of table item
		if not hasattr(TableItem, 'tableIcon'):
			TableItem.tableIcon = QIcon(":/icons/table.png")
			TableItem.viewIcon = QIcon(":/icons/view.png")
			TableItem.layerPointIcon = QIcon(":/icons/layer_point.png")
			TableItem.layerLineIcon = QIcon(":/icons/layer_line.png")
			TableItem.layerPolygonIcon = QIcon(":/icons/layer_polygon.png")
			TableItem.layerUnknownIcon = QIcon(":/icons/layer_unknown.png")
			
	def db(self):
		return self.parent()
		
	def data(self, column):
		if column == 0:
			return self.name
		elif column == 1:
			return self.geom_type
		else:
			return None
		
	def icon(self):
		if self.geom_type is not None:
			if self.geom_type.find('POINT') != -1:
				return self.layerPointIcon
			elif self.geom_type.find('LINESTRING') != -1:
				return self.layerLineIcon
			elif self.geom_type.find('POLYGON') != -1:
				return self.layerPolygonIcon
			else:
				return self.layerUnknownIcon
		if self.is_view:
			return self.viewIcon
		else:
			return self.tableIcon


def new_tree():
	
	rootItem = DatabaseItem()
	sPublic = DBItem('public', rootItem)
	sG = DBItem('gis', rootItem)
	t1 = TableItem('roads', 'ozefo', 123, 4, 'LINESTRING', False, sPublic)
	t2 = TableItem('sidla', 'ozefo', 66, 2, 'POINT', False, sG)
	return rootItem


class DatabaseModel(QAbstractItemModel):
	
	def __init__(self, parent=None):
		QAbstractItemModel.__init__(self, parent)

		self.header = ['Table']
		self.readOnlyMode = False
		self.showOnlyReadable = False

		self.tree = DatabaseItem()

	def setReadOnlyMode(self, value=True):
		self.readOnlyMode = value

	def showOnlyReadableTables(self, value=True):
		self.showOnlyReadable = value
		
	def loadFromDb(self, db):
		self.db = db
		self.tree = DatabaseItem(None)
		self.tree.constructTreeFromDb(self.db)

	def columnCount(self, parent):
		return 1
		
	def data(self, index, role):
		if not index.isValid():
			return QVariant()
		
		if role == Qt.DecorationRole and index.column() == 0:
			icon = index.internalPointer().icon()
			if icon: return QVariant(icon)
			
		if role != Qt.DisplayRole and role != Qt.EditRole:
			return QVariant()
		
		retval = index.internalPointer().data(index.column())
		if retval:
			return QVariant(retval)
		else:
			return QVariant()
	
	def flags(self, index):
		if not index.isValid():
			return 0
		
		flags = Qt.ItemIsEnabled | Qt.ItemIsSelectable 
		if index.column() == 0:
			if not self.readOnlyMode:
				flags |= Qt.ItemIsEditable
		return flags
	
	def headerData(self, section, orientation, role):
		if orientation == Qt.Horizontal and role == Qt.DisplayRole and section < len(self.header):
			return QVariant(self.header[section])
		return QVariant()

	def index(self, row, column, parent):
		if not self.hasIndex(row, column, parent):
			return QModelIndex()
		
		if not parent.isValid():
			parentItem = self.tree
		else:
			parentItem = parent.internalPointer()
		
		childItem = parentItem.child(row)
		if childItem:
			return self.createIndex(row, column, childItem)
		else:
			return QModelIndex()

	def parent(self, index):
		if not index.isValid():
			return QModelIndex()
		
		childItem = index.internalPointer()
		parentItem = childItem.parent()

		if parentItem == self.tree:
			return QModelIndex()
		
		return self.createIndex(parentItem.row(), 0, parentItem)

	def rowCount(self, parent):
		if parent.column() > 0:
			return 0
		
		if not parent.isValid():
			parentItem = self.tree
		else:
			parentItem = parent.internalPointer()
			
		return parentItem.childCount()

	def setData(self, index, value, role):
		if role != Qt.EditRole or index.column() != 0:
			return False
			
		item = index.internalPointer()
		new_name = unicode(value.toString())
		if new_name == item.name:
			return False
		
		if isinstance(item, TableItem):
			if self.readOnlyMode:
				return False

			# rename table or view
			try:
				self.db.rename_table(item.name, new_name)
				self.emit(SIGNAL('dataChanged(const QModelIndex &, const QModelIndex &)'), index, index)
				return True
			except spatialite_utils.DbError, e:
				DlgDbError.showError(e, None)
				return False
			
		elif isinstance(item, DBItem):
			if self.readOnlyMode:
				return False

			# rename schema
			try:
				self.db.rename_schema(item.name, new_name)
				self.emit(SIGNAL('dataChanged(const QModelIndex &, const QModelIndex &)'), index, index)
				return True
			except spatialite_utils.DbError, e:
				DlgDbError.showError(e, None)
				return False
			
		else:
			print "set", unicode(value.toString()), role
			return False

