# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SmoothGeometry
                                 A QGIS plugin
 This plugin smooths selected geometries
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-12-31
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Valerio Pinna
        email                : pinnavalerio@yahoo.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from qgis.PyQt import uic
from qgis.PyQt.QtCore import QCoreApplication, QSettings, QTranslator
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import (
    QAction,
    QDialog,
    QMenu,
    QMessageBox,
    QToolButton,
)

from qgis.core import (
    Qgis,
    QgsFeatureRequest,
    QgsGeometry,
    QgsMapLayerType,
    QgsMessageLog,
)

# Initialize Qt resources from file resources.py
from .resources import *  # noqa: F401, F403


Ui_ConfigureSmoothGeometryDialogBase = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "configuresmoothingbase.ui")
)[0]


# -----------------------------------------------------------------------------
# Settings helpers (namespaced + defaults)
# -----------------------------------------------------------------------------
_SETTINGS_GROUP = "SmoothGeometry"


def _key(name: str) -> str:
    return f"{_SETTINGS_GROUP}/{name}"


def iterations() -> int:
    return QSettings().value(_key("iterations"), 1, type=int)


def setIterations(itera: int) -> None:
    QSettings().setValue(_key("iterations"), int(itera))


def offset() -> float:
    return QSettings().value(_key("offset"), 0.3, type=float)


def setOffset(val: float) -> None:
    QSettings().setValue(_key("offset"), float(val))


def minDistance() -> float:
    return QSettings().value(_key("minDistance"), -1, type=float)


def setMinDistance(val: float) -> None:
    QSettings().setValue(_key("minDistance"), float(val))


def maxAngle() -> int:
    return QSettings().value(_key("maxAngle"), 180, type=int)


def setMaxAngle(val: int) -> None:
    QSettings().setValue(_key("maxAngle"), int(val))


# -----------------------------------------------------------------------------
# UX + logging helpers
# -----------------------------------------------------------------------------
def _notify(iface, text: str, level=Qgis.Warning, duration: int = 5, title: str = "Smooth Geometry") -> None:
    """Non-blocking user feedback."""
    if iface:
        iface.messageBar().pushMessage(title, text, level=level, duration=duration)


def _log(text: str, level=Qgis.Info, tag: str = "SmoothGeometry") -> None:
    """Developer-facing log."""
    QgsMessageLog.logMessage(text, tag, level)


# -----------------------------------------------------------------------------
# Dialog
# -----------------------------------------------------------------------------
class ConfigureSmoothGeometryDialog(QDialog, Ui_ConfigureSmoothGeometryDialogBase):
    def __init__(self, parent):
        # parent is iface in your code; use mainWindow when possible
        super().__init__(parent.mainWindow() if hasattr(parent, "mainWindow") else None)
        self.iface = parent
        self.setupUi(self)

        # Populate UI with persisted values (safe defaults guaranteed)
        self.boxIterations.setValue(iterations())
        self.boxOffset.setValue(offset())
        self.boxMinDistance.setValue(minDistance())
        self.boxMaxAngle.setValue(maxAngle())


# -----------------------------------------------------------------------------
# Plugin main class
# -----------------------------------------------------------------------------
class SmoothGeometry:
    def __init__(self, iface):
        """Constructor.

        :param iface: QGIS interface instance (QgsInterface)
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        # Translator (guard against missing locale)
        user_locale = QSettings().value("locale/userLocale", "en")
        locale = str(user_locale)[0:2] if user_locale else "en"
        locale_path = os.path.join(self.plugin_dir, "i18n", f"SmoothGeometry_{locale}.qm")

        self.translator = None
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # UI references
        self.toolbar = None
        self.toolButton = None
        self.actionRun = None
        self.actionConfigure = None

        self.menu = self.tr("&Smooth Geometry Tool")

    def tr(self, message: str) -> str:
        return QCoreApplication.translate("SmoothGeometry", message)

    # -------------------------------------------------------------------------
    # QGIS GUI hooks
    # -------------------------------------------------------------------------
    def initGui(self):
        # Create a dedicated toolbar (clean unload + no orphan widgets)
        self.toolbar = self.iface.addToolBar("Smooth Geometry")
        self.toolbar.setObjectName("SmoothGeometryToolbar")

        # Toolbutton with dropdown menu
        self.toolButton = QToolButton()
        
        # Qt5/Qt6 compatible popup mode enum
        try:
            popup_mode = QToolButton.MenuButtonPopup  # Qt5
        except AttributeError:
            popup_mode = QToolButton.ToolButtonPopupMode.MenuButtonPopup  # Qt6

        self.toolButton.setPopupMode(popup_mode)
        
        self.toolButton.setMenu(QMenu(self.toolButton))

        # Actions
        self.actionRun = QAction(
            QIcon(os.path.join(self.plugin_dir, "icon.png")),
            self.tr("Smooth the selected geometries"),
            self.iface.mainWindow(),
        )
        self.actionRun.triggered.connect(self.run)

        self.actionConfigure = QAction(
            QIcon(os.path.join(self.plugin_dir, "smooth_geometry_config_icon.png")),
            self.tr("Configure"),
            self.iface.mainWindow(),
        )
        self.actionConfigure.setToolTip(self.tr("Configure the smoothing settings"))
        self.actionConfigure.triggered.connect(self.configure)

        # Put actions in the toolbutton menu
        m = self.toolButton.menu()
        m.addAction(self.actionRun)
        m.addAction(self.actionConfigure)
        self.toolButton.setDefaultAction(self.actionRun)

        # Add widget to toolbar
        self.toolbar.addWidget(self.toolButton)

        # Add to plugin menu as well
        self.iface.addPluginToMenu(self.menu, self.actionRun)
        self.iface.addPluginToMenu(self.menu, self.actionConfigure)

        # Optional: keep this if you need main-window action registration
        self.iface.registerMainWindowAction(self.actionConfigure, "")

    def unload(self):
        # Remove menu actions
        if self.actionRun:
            self.iface.removePluginMenu(self.menu, self.actionRun)
        if self.actionConfigure:
            self.iface.removePluginMenu(self.menu, self.actionConfigure)
            self.iface.unregisterMainWindowAction(self.actionConfigure)

        # Remove toolbar (and its widget)
        if self.toolbar:
            self.iface.mainWindow().removeToolBar(self.toolbar)
            self.toolbar = None

        self.toolButton = None
        self.actionRun = None
        self.actionConfigure = None

        # Remove translator if installed
        if self.translator:
            QCoreApplication.removeTranslator(self.translator)
            self.translator = None

    # -------------------------------------------------------------------------
    # Core logic
    # -------------------------------------------------------------------------
    def smooth(self, layer, ext_canvas_geom: QgsGeometry) -> None:
        itera = iterations()
        off = offset()
        min_dist = minDistance()
        max_ang = maxAngle()

        fids = layer.selectedFeatureIds()
        if not fids:
            _notify(self.iface, "No selected features to smooth.", Qgis.Info)
            return

        #req = QgsFeatureRequest().setFilterFids(set(fids))
        req = QgsFeatureRequest().setFilterFids(list(fids))


        layer.beginEditCommand("Smooth Geometry")

        try:
            # Warn if any selected feature is outside current view
            outside = False
            for f in layer.getFeatures(req):
                geom = f.geometry()
                if not geom or geom.isEmpty():
                    continue
                if not geom.intersects(ext_canvas_geom):
                    outside = True
                    break

            if outside:
                reply = QMessageBox.warning(
                    self.iface.mainWindow(),
                    "Smooth Geometry",
                    "Some selected features are outside the current map view. Continue?",
                    QMessageBox.Yes | QMessageBox.No,
                    QMessageBox.No,
                )
                if reply == QMessageBox.No:
                    layer.destroyEditCommand()
                    _notify(self.iface, "Cancelled.", Qgis.Info, duration=3)
                    return

            # Apply smoothing
            changed = 0
            for f in layer.getFeatures(req):
                g = f.geometry()
                if not g or g.isEmpty():
                    continue

                sm = g.smooth(itera, off, min_dist, max_ang)
                if sm and not sm.isEmpty():
                    if layer.changeGeometry(f.id(), sm):
                        changed += 1

            layer.endEditCommand()
            layer.triggerRepaint()

            _notify(self.iface, f"Smoothing completed ({changed} feature(s) updated).", Qgis.Success, duration=4)

        except Exception as e:
            layer.destroyEditCommand()
            _log(f"Smoothing failed: {e!r}", Qgis.Critical)
            _notify(self.iface, "Smoothing failed. See Log Messages Panel for details.", Qgis.Critical, duration=8)
            raise

    def run(self):
        layer = self.iface.activeLayer()
        if layer is None:
            _notify(self.iface, "No active layer selected.", Qgis.Warning)
            return

        if layer.type() != QgsMapLayerType.VectorLayer:
            _notify(self.iface, "Please select a vector layer.", Qgis.Warning)
            return

        if not layer.selectedFeatureIds():
            _notify(self.iface, "No selected features to smooth.", Qgis.Info)
            return

        # Ensure the layer is editable
        if not layer.isEditable():
            if not layer.startEditing():
                QMessageBox.warning(self.iface.mainWindow(), "Smooth Geometry", "Cannot start editing on this layer.")
                return

        ext_canvas_geom = QgsGeometry.fromRect(self.iface.mapCanvas().extent())
        self.smooth(layer, ext_canvas_geom)

    def configure(self):
        dlg = ConfigureSmoothGeometryDialog(self.iface)

        # Qt6 uses exec(); Qt5 used exec_()
        try:
            result = dlg.exec()
        except AttributeError:
            result = dlg.exec_()

        if result:
            setIterations(dlg.boxIterations.value())
            setOffset(dlg.boxOffset.value())
            setMinDistance(dlg.boxMinDistance.value())
            setMaxAngle(dlg.boxMaxAngle.value())
            _notify(self.iface, "Settings saved.", Qgis.Success, duration=3)
