# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CropSiteSuitabilityAnalysisDialog
                                 A QGIS plugin
 Define crop-specific thresholds and criteria to generate customized suitability maps.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-02-25
        git sha              : $Format:%H$
        copyright            : (C) 2025 by ICIMOD
        email                : nimesh.bhandari@icimod.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import json
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QListWidget, QPushButton,
    QGroupBox, QSpinBox, QLabel, QFileDialog, QMessageBox, QTableWidget,
    QTableWidgetItem, QHeaderView, QDialogButtonBox
)
from qgis.PyQt.QtCore import Qt, QTimer
from qgis.PyQt.QtWidgets import QProgressDialog
from qgis.core import QgsProject, QgsRasterLayer, QgsMapLayer
from qgis.analysis import QgsRasterCalculator, QgsRasterCalculatorEntry

class ThresholdMappingDialog(QDialog):
    def __init__(self, saved_params, current_layers, parent=None):
        super().__init__(parent)
        self.saved_params = saved_params
        self.current_layers = current_layers
        self.mappings = {}
        self.initUI()

    def initUI(self):
        self.setWindowTitle("Current Layers and Saved Layers Mapping")
        layout = QVBoxLayout()
        
        self.table = QTableWidget()
        self.table.setColumnCount(2)
        self.table.setHorizontalHeaderLabels(["Current Layer Name", "Saved Layer Name"])
        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        
        # Create rows for current layers with saved layer comboboxes
        for idx, current_layer in enumerate(self.current_layers):
            self.table.insertRow(idx)
            self.table.setItem(idx, 0, QTableWidgetItem(current_layer.name()))
            
            combo = QtWidgets.QComboBox()
            combo.addItem("-- Select Saved Layer --", None)
            for param in self.saved_params:
                combo.addItem(param["layer_name"], param)
            self.table.setCellWidget(idx, 1, combo)
        
        layout.addWidget(self.table)
        
        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)
        
        self.setLayout(layout)

    def get_mappings(self):
        mappings = {}
        for row in range(self.table.rowCount()):
            current_layer = self.current_layers[row]
            combo = self.table.cellWidget(row, 1)
            saved_param = combo.currentData()
            if saved_param:
                mappings[current_layer] = saved_param
        return mappings

class CropSiteSuitabilityAnalysisDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setAttribute(Qt.WA_DeleteOnClose)
        self.setWindowTitle("Crop Suitability Analysis")
        self.thresholds = {}
        self.selected_layers = []
        self.initUI()
        self.populate_available_layers()

    def initUI(self):
        layout = QVBoxLayout()

        # Layer selection section
        layer_layout = QHBoxLayout()
        # Available layers column
        available_col = QVBoxLayout()
        available_col.addWidget(QLabel("Available Layers"))  # Label added here
        self.available_layers_list = QListWidget()
        available_col.addWidget(self.available_layers_list)
        
        # Selected layers column
        selected_col = QVBoxLayout()
        selected_col.addWidget(QLabel("Selected Layers"))  # Label added here
        self.selected_layers_list = QListWidget()
        selected_col.addWidget(self.selected_layers_list)
        
        # Buttons between columns
        button_layout = QVBoxLayout()
        self.add_button = QPushButton(">")
        self.add_button.clicked.connect(self.add_layers)
        self.remove_button = QPushButton("<")
        self.remove_button.clicked.connect(self.remove_layers)
        button_layout.addStretch()
        button_layout.addWidget(self.add_button)
        button_layout.addWidget(self.remove_button)
        button_layout.addStretch()
        
        # Add all components to main layer layout
        layer_layout.addLayout(available_col)
        layer_layout.addLayout(button_layout)
        layer_layout.addLayout(selected_col)
        
        layout.addLayout(layer_layout)

        # Threshold input section
        self.threshold_table = QTableWidget()
        self.threshold_table.setColumnCount(3)
        self.threshold_table.setHorizontalHeaderLabels(["Layer", "Min Value", "Max Value"])
        self.threshold_table.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        layout.addWidget(self.threshold_table)

        # Save/Load buttons
        btn_layout = QHBoxLayout()
        self.save_btn = QPushButton("Save Thresholds")
        self.save_btn.clicked.connect(self.save_thresholds)
        self.load_btn = QPushButton("Load Thresholds")
        self.load_btn.clicked.connect(self.load_thresholds)

        btn_layout.addWidget(self.load_btn)
        btn_layout.addWidget(self.save_btn)

        layout.addLayout(btn_layout)

        # Process button
        self.process_btn = QPushButton("Generate Suitability Map")
        self.process_btn.clicked.connect(self.process_analysis)
        layout.addWidget(self.process_btn)

        self.setLayout(layout)
        self.selected_layers_list.itemSelectionChanged.connect(self.update_threshold_table)

    def populate_available_layers(self):
        self.available_layers_list.clear()
        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsRasterLayer):
                self.available_layers_list.addItem(layer.name())

    def add_layers(self):
        selected = self.available_layers_list.selectedItems()
        for item in selected:
            if self.selected_layers_list.findItems(item.text(), Qt.MatchExactly):
                continue
            self.selected_layers_list.addItem(item.text())
            self.available_layers_list.takeItem(self.available_layers_list.row(item))
            self.update_threshold_table()

    def remove_layers(self):
        selected = self.selected_layers_list.selectedItems()
        for item in selected:
            self.available_layers_list.addItem(item.text())
            self.selected_layers_list.takeItem(self.selected_layers_list.row(item))
            self.update_threshold_table()

    def update_threshold_table(self):
        self.threshold_table.setRowCount(0)
        self.selected_layers = []
        
        for i in range(self.selected_layers_list.count()):
            layer_name = self.selected_layers_list.item(i).text()
            layer = QgsProject.instance().mapLayersByName(layer_name)[0]
            self.selected_layers.append(layer)
            
            row = self.threshold_table.rowCount()
            self.threshold_table.insertRow(row)
            
            self.threshold_table.setItem(row, 0, QTableWidgetItem(layer_name))
            
            min_spin = QtWidgets.QDoubleSpinBox()
            min_spin.setDecimals(1)  # 2 decimal places
            min_spin.setRange(-9999.9, 9999.9)  # Extended range for decimal values
            min_spin.setSingleStep(0.1)  # Step by 0.01 when using arrows
            
            max_spin = QtWidgets.QDoubleSpinBox()
            max_spin.setDecimals(1)
            max_spin.setRange(-9999.9, 9999.9)
            max_spin.setSingleStep(0.1)
            
            self.threshold_table.setCellWidget(row, 1, min_spin)
            self.threshold_table.setCellWidget(row, 2, max_spin)

    def get_current_thresholds(self):
        thresholds = {}
        for row in range(self.threshold_table.rowCount()):
            layer_name = self.threshold_table.item(row, 0).text()
            min_val = self.threshold_table.cellWidget(row, 1).value()
            max_val = self.threshold_table.cellWidget(row, 2).value()
            thresholds[layer_name] = {"min": min_val, "max": max_val}
        return thresholds

    def save_thresholds(self):
        filepath, _ = QFileDialog.getSaveFileName(
            self, "Save Thresholds", "", "JSON files (*.json)"
        )
        if not filepath:
            return

        thresholds = self.get_current_thresholds()
        data = {"parameters": []}
        
        for layer in self.selected_layers:
            if layer.name() in thresholds:
                param = {
                    "layer_name": layer.name() + "_thresh",
                    "layer_source": layer.source(),
                    "min": thresholds[layer.name()]["min"],
                    "max": thresholds[layer.name()]["max"]
                }
                data["parameters"].append(param)
        
        with open(filepath, 'w') as f:
            json.dump(data, f, indent=4)

    def load_thresholds(self):
        filepath, _ = QFileDialog.getOpenFileName(
            self, "Load Thresholds", "", "JSON files (*.json)"
        )
        if not filepath:
            return

        with open(filepath, 'r') as f:
            data = json.load(f)
        
        current_layers = [QgsProject.instance().mapLayersByName(name)[0] 
                         for name in self.get_selected_layer_names()]
        
        mapping_dialog = ThresholdMappingDialog(
            data["parameters"], current_layers, self
        )
        if mapping_dialog.exec_():
            mappings = mapping_dialog.get_mappings()
            self.apply_loaded_thresholds(mappings, data)

    def apply_loaded_thresholds(self, mappings, data):
        # Clear existing thresholds first
        for row in range(self.threshold_table.rowCount()):
            self.threshold_table.cellWidget(row, 1).setValue(0)
            self.threshold_table.cellWidget(row, 2).setValue(0)

        # Apply new thresholds from mappings
        for current_layer, saved_param in mappings.items():
            for row in range(self.threshold_table.rowCount()):
                if self.threshold_table.item(row, 0).text() == current_layer.name():
                    self.threshold_table.cellWidget(row, 1).setValue(saved_param["min"])
                    self.threshold_table.cellWidget(row, 2).setValue(saved_param["max"])

    def validate_inputs(self):
        # Check layer compatibility
        ref_layer = self.selected_layers[0]
        for layer in self.selected_layers[1:]:
            if (layer.crs() != ref_layer.crs() or
                layer.width() != ref_layer.width() or
                layer.height() != ref_layer.height()):
                return False, "All layers must have same CRS, resolution and extent"
        
        # Check threshold validity
        thresholds = self.get_current_thresholds()
        for layer_name, values in thresholds.items():
            if values["min"] > values["max"]:
                return False, f"Invalid thresholds for {layer_name}: min > max"
        
        return True, ""

    def process_analysis(self):
        if len(self.selected_layers) == 0:
            QMessageBox.warning(self, "Error", "Please select at least one layer")
            return

        valid, msg = self.validate_inputs()
        if not valid:
            QMessageBox.critical(self, "Validation Error", msg)
            return

        try:
            # Disable UI during processing
            self.setEnabled(False)

            # Create and show progress dialog
            self.progress = QtWidgets.QProgressDialog(
                "Processing suitability analysis...", 
                None,  # No cancel button
                0, 0,  # Indeterminate mode
                self
            )
            self.progress.setWindowTitle("Processing")
            self.progress.setWindowModality(Qt.WindowModal)
            self.progress.show()

            # Force UI update
            QtWidgets.QApplication.processEvents()

            # Use a different approach - create a single combined expression
            ref_layer = self.selected_layers[0]
            thresholds = self.get_current_thresholds()
            
            # Setup entries for calculator
            entries = []
            for layer in self.selected_layers:
                entry = QgsRasterCalculatorEntry()
                entry.ref = f'{layer.name()}@1'
                entry.raster = layer
                entry.bandNumber = 1
                entries.append(entry)
            
            # Build a single combined condition
            conditions = []
            for layer in self.selected_layers:
                # Update progress message
                self.progress.setLabelText(f"Processing {layer.name()}...")
                QtWidgets.QApplication.processEvents()  # Update UI

                threshold = thresholds[layer.name()]
                condition = f'(({layer.name()}@1 >= {threshold["min"]}) AND ({layer.name()}@1 <= {threshold["max"]}))'
                conditions.append(condition)

            # Final calculation
            self.progress.setLabelText("Combining layers...")
            QtWidgets.QApplication.processEvents()
            
            # Join all conditions with AND and explicitly cast to 0/1
            final_expr = f'1 * ({" AND ".join(conditions)})'
            
            output_path, _ = QFileDialog.getSaveFileName(
                self, "Save Output Raster", "", "GeoTIFF files (*.tif)"
            )
            
            if output_path:
                calc = QgsRasterCalculator(final_expr, output_path, 'GTiff',
                                        ref_layer.extent(),
                                        ref_layer.width(),
                                        ref_layer.height(),
                                        entries)
                result = calc.processCalculation()
                
                if result == 0:  # 0 means success in QgsRasterCalculator
                    QMessageBox.information(self, "Success", "Suitability map generated successfully")
                    
                    # Add result to QGIS
                    if os.path.exists(output_path):
                        layer_name = os.path.splitext(os.path.basename(output_path))[0]
                        result_layer = QgsRasterLayer(output_path, layer_name)
                        QgsProject.instance().addMapLayer(result_layer)
                else:
                    QMessageBox.critical(self, "Error", f"Calculation failed with error code {result}")

        except Exception as e:
            QMessageBox.critical(self, "Error", str(e))

        finally:
            # Cleanup progress dialog
            self.progress.close()
            self.setEnabled(True)
            QtWidgets.QApplication.processEvents()

    def get_selected_layer_names(self):
        return [self.selected_layers_list.item(i).text() 
               for i in range(self.selected_layers_list.count())]
    
    def cleanup(self):
        """Reset all UI elements to initial state"""
        # Clear layer lists
        self.available_layers_list.clear()
        self.selected_layers_list.clear()
        self.populate_available_layers()
        
        # Clear threshold table
        self.threshold_table.setRowCount(0)
        

    def closeEvent(self, event):
        """Handle window close event"""
        self.cleanup()
        event.accept()
    
# Required QGIS Plugin Boilerplate
def classFactory(iface):
    from .site_suitability_analysis import CropSiteSuitabilityAnalysisDialog
    return CropSiteSuitabilityAnalysisDialog(iface)

