# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SimstockQGIS
                                 A QGIS plugin
 Feeds QGIS data into Simstock
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-05-17
        git sha              : $Format:%H$
        copyright            : (C) 2023 by UCL
        email                : shyam.amrith.14@ucl.ac.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from unittest import result
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant, QUrl
from qgis.PyQt.QtGui import QIcon, QDesktopServices
from qgis.PyQt.QtWidgets import QAction

from qgis.core import QgsProject, QgsVectorDataProvider, QgsVectorLayer, QgsField, QgsFields, QgsVectorFileWriter, QgsCoordinateTransformContext, QgsApplication

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .Simstock_QGIS_dialog import SimstockQGISDialog
import os

#my imports
#import venv
import subprocess
import platform
import sys
#sys.path.insert(0, os.path.dirname(__file__))
import pandas as pd
#import multiprocessing as mp
import qgis.utils
from qgis.core import Qgis
from qgis.core import NULL as qgis_null
import time
import shutil
import numpy as np
import json
from zipfile import ZipFile

class SimstockQGIS:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.iface.actionShowPythonDialog().trigger() #show console upon launch
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'SimstockQGIS_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Simstock QGIS')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        
        # Custom additions
        self.custom_initialisation()


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('SimstockQGIS', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            # Add plugin to plugins dropdown
            self.iface.addPluginToMenu(
                self.menu,
                action)
            
            # Add plugin to Vector menu
            #self.iface.addPluginToVectorMenu(
            #    self.menu,
            #    action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        # Default action
        icon_path = ':/plugins/Simstock_QGIS/icon.svg'
        self.add_action(
            icon_path,
            text=self.tr(u'Simstock'),
            callback=self.run,
            parent=self.iface.mainWindow())
        # Run "pyrcc5 -o resources.py resources.qrc" to update icon
        
        # Help action
        self.help_action = QAction(QgsApplication.getThemeIcon("/mActionContextHelp.png"),
                                   self.tr("Simstock Documentation"),
                                   self.iface.mainWindow())
        # Add the action to the Help menu
        self.iface.pluginHelpMenu().addAction(self.help_action)
        self.help_action.triggered.connect(self.show_help)

        # will be set False in run()
        self.first_start = True
    

    def show_help(self):
        """Display documentation online"""
        QDesktopServices.openUrl(QUrl('https://simstock.readthedocs.io/en/latest/simstockqgis.html'))


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Simstock QGIS'),
                action)
            self.iface.removeToolBarIcon(action)



##################################### CUSTOM CODE ##################################################
    def custom_initialisation(self):
        ### CUSTOM ADDITIONS
        # Update path to access Simstock scripts
        sys.path.insert(0, self.plugin_dir)
        
        # Update path to packaged eppy
        self.eppy_dir = os.path.join(self.plugin_dir, "eppy-scripts")
        sys.path.append(self.eppy_dir)
        
        # Various check trackers
        self.initial_setup_worked = None #check if initial setup worked
        self.simulation_started = False #check if the run button was pressed
        self.cwd_set = False #check if the user set the cwd
        self.added_fields = False #check if Add Fields has already been used
        
        # Startup E+ stuff
        self.EP_DIR = os.path.join(self.plugin_dir, "EnergyPlus")

        # Find the computer's operating system and find energyplus version
        self.system = platform.system().lower()
        if self.system in ['windows', 'linux', 'darwin']:
            self.energyplusexe = os.path.join(self.EP_DIR, 'ep8.9_{}/energyplus'.format(self.system))
            self.readvarseso = os.path.join(self.EP_DIR, 'ep8.9_{}/ReadVarsESO'.format(self.system))
        
        # Locate QGIS Python, differs by OS
        qgis_python_dir = sys.exec_prefix
        if self.system == "windows":
            self.qgis_python_location = os.path.join(qgis_python_dir, "python")
        if self.system == "darwin":
            self.qgis_python_location = os.path.join(qgis_python_dir, "bin", "python3")

        # Set up Eppy
        from eppy.modeleditor import IDF, IDDAlreadySetError
        if self.system in ['windows', 'linux', 'darwin']:
            iddfile = os.path.join(self.EP_DIR, 'ep8.9_{}/Energy+.idd'.format(self.system))
        try:
            IDF.setiddname(iddfile)
        except IDDAlreadySetError:
            pass

        # The prepended tag on database files used to identify them
        self.database_tag = "DB-"

        # The headings that Simstock expects and the QVariant type of each
        # Format is: "heading-QVariantType"
        self.headings = ["polygon-None",
                         "UID-String",
                         "shading-String",
                         "height-Double",
                         "wwr-Double",
                         "nofloors-Int",
                         "construction-String",
                         "glazing_const-String",
                         "infiltration_rate-Double",
                         "ventilation_rate-Double",
                         "overhang_depth-Double"]

        # Load config file
        with open(os.path.join(self.plugin_dir, "config.json"), "r") as read_file:
            self.config = json.load(read_file)



    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = SimstockQGISDialog()

        # show the dialog
        self.dlg.show()
        
        # Check if the buttons were clicked and run function if so
        self.dlg.pbInitialSetup.clicked.connect(self.initial_setup)
        self.dlg.pbRunSim.clicked.connect(self.run_plugin)
        self.dlg.pbOptions.clicked.connect(self.add_fields)
        self.dlg.pbSetcwd.clicked.connect(self.set_cwd)
        self.dlg.label_4.linkActivated.connect(self.open_config)
        
        # Run the dialog event loop
        result = self.dlg.exec_()
        
        # See if OK was pressed
        if result:
            pass #don't do anything if OK was pressed



    def open_config(self):
        """Reveals the config file to the user directly"""
        config_location = os.path.join(self.plugin_dir, "config.json")

        if self.system == "windows":
            subprocess.run(["explorer", "config.json"], cwd=self.plugin_dir)

        if self.system == "darwin": #TODO: needs testing
            subprocess.run(["open", "config.json"], cwd=self.plugin_dir)



    def initial_setup(self):
        # TODO:
        #   - Print out more useful information in the case that failures occured. This will be 
        #      useful if users need to report bugs, and can provide the important info for them.
        #   - Put into functions, and maybe even separate script
        print("Initial setup starting...")
        
        # Set up list to track success of each test
        self.initial_tests = []
        
        # Tracks unimportant issues, unless something fails in which case could be useful to debug
        self.initial_tests_warnings = [] 


        # This is to select a different EnergyPlus source
        # TODO: decide on the best source and delete the other methods
        ep_source = "download" # "packaged" or "download" or "user"

        if not os.path.exists(os.path.dirname(self.energyplusexe)):
            if ep_source == "packaged":
                # Unzip EnergyPlus according to platform
                EP_zipfile = os.path.join(self.EP_DIR, f"ep8.9_{self.system}.zip")

                print("    Extracting EnergyPlus...")
                with ZipFile(EP_zipfile, "r") as fp:
                    fp.extractall(self.EP_DIR)

                # Delete all EP zipfiles
                [os.remove(f) for f in os.scandir(self.EP_DIR) if f.name[-4:]==".zip"]


            if ep_source == "download":
                # Download EnergyPlus option

                if self.system == "windows":
                    ep_link = "https://github.com/NREL/EnergyPlus/releases/download/v8.9.0/EnergyPlus-8.9.0-40101eaafd-Windows-x86_64.zip"

                elif self.system == "darwin":
                    ep_link = "https://github.com/NREL/EnergyPlus/releases/download/v8.9.0/EnergyPlus-8.9.0-40101eaafd-Darwin-x86_64.tar.gz"

                EP_zipfile = os.path.join(self.EP_DIR, ep_link.split("/")[-1])
                ep_plat_dir = os.path.dirname(self.energyplusexe)

                # Get user's permission before downloading EnergyPlus
                self.ask_permission()
                if self.permission == "n":
                    print("User permission not granted to download EnergyPlus.")

                elif self.permission == "y":
                    print("User permission granted to download EnergyPlus.")
                    import requests

                    # Download EnergyPlus if permission given
                    if not os.path.exists(EP_zipfile):
                        r = requests.get(ep_link, stream=True)
                        if r.ok:
                            print("Downloading EnergyPlus...")
                            with open(EP_zipfile, "wb") as f:
                                for chunk in r.iter_content(chunk_size=1024):
                                    f.write(chunk)
                

                if self.system == "windows":

                    # Extract zip archive
                    print("    Extracting EnergyPlus...")
                    with ZipFile(EP_zipfile, "r") as fp:
                        fp.extractall(self.EP_DIR)

                    # Arrange in expected file path structure
                    shutil.move(os.path.join(EP_zipfile[:-4], "EnergyPlus-8-9-0"), ep_plat_dir)
                    shutil.rmtree(EP_zipfile[:-4]) #delete empty dir
                    ep_files = ["Energy+.idd", "energyplus.exe", "energyplusapi.dll", "ReadVarsESO.exe"]

                if self.system == "darwin":

                    # Extract tar.gz archive
                    #import extracttargz
                    #extracttargz.main()        #this throws an error for some reason
                    print("    Extracting EnergyPlus...")
                    # TODO: feed in name of tar.gz file to script
                    subprocess.run([self.qgis_python_location, os.path.join(self.plugin_dir, "extracttargz.py")])

                    # Arrange in expected file path structure
                    shutil.move(os.path.join(EP_zipfile[:-7], "EnergyPlus-8-9-0"), ep_plat_dir)
                    shutil.rmtree(EP_zipfile[:-7]) #delete empty dir
                    ep_files = ["Energy+.idd", "energyplus", "libenergyplusapi.8.9.0.dylib", "libgcc_s.1.dylib",
                            "libgfortran.3.dylib", "libquadmath.0.dylib", "ReadVarsESO"]
                

                # Check that all required files are present and report status
                clean_up = True
                for f in ep_files:
                    fp = os.path.join(ep_plat_dir, f)
                    if os.path.exists(fp):
                        print(f"Located '{f}'")
                    else:
                        # Move file if not in expected directory
                        fp = os.path.join(ep_plat_dir, "PostProcess", f)
                        if os.path.exists(fp):
                            print(f"Located '{f}'")
                            shutil.move(fp, os.path.join(ep_plat_dir, f))
                        else:
                            self.initial_tests.append(f"Could not locate '{f}'")
                            clean_up = False #do not perform clean-up if some files are missing

                # Clean up if all files were located
                if clean_up:
                    os.remove(EP_zipfile)
                    to_delete = ["ExampleFiles", "PreProcess", "Documentation", "DataSets", "PostProcess", "WeatherData"]
                    for fdir in to_delete:
                        try:
                            shutil.rmtree(os.path.join(ep_plat_dir, fdir))
                        except:
                            pass

            if ep_source == "user":
                pass
                #   - Main part is creating user notice to download from link, and provide file path with path selector
                #   - Can use the same code as download, but without download step


        # Psutil excluded for now
        # TODO: include without zip and check for binaries

        # # Unzip psutil as per platform
        # if not os.path.exists(os.path.join(self.eppy_dir, "psutil")):
        #     if self.system == "windows" and platform.machine() == "AMD64":
        #         psutil_zipfile = os.path.join(self.eppy_dir, "psutil_win-64.zip")
        #     elif self.system == "darwin" and platform.machine() == "x86_64":
        #         psutil_zipfile = os.path.join(self.eppy_dir, "psutil_osx-64.zip")
        #     else:
        #         print("Only Windows and macOS x86-64 support psutil. "
        #               f"System: {self.system}-{platform.machine()}.")
        #         psutil_zipfile = None
            
        #     # Only extract if system is supported
        #     if psutil_zipfile is not None:
        #         print("    Extracting psutil...")
        #         with ZipFile(psutil_zipfile, "r") as fp:
        #             fp.extractall(self.eppy_dir)

        #         # Delete all psutil zipfiles
        #         [os.remove(f) for f in os.scandir(self.eppy_dir) if f.name[-4:]==".zip"]
        

        # Module tests
        print("Pandas version: ", pd.__version__)
        try:
            import eppy
            print("Eppy version: ", eppy.__version__)
        except ImportError:
            self.initial_tests.append("Eppy failed to load.")
            
        try:
            import shapely
            print("Shapely version: ", shapely.__version__)
        except ImportError:
            self.initial_tests.append("Shapely failed to load.")
            
        # try:
        #     import psutil
        #     print("Psutil version: ", psutil.__version__)
        # except:
        #     # Do not fail if psutil is not imported, since it is not essential
        #     print("Psutil failed to load - not critical.")
            
        # Test Python script
        test_python = os.path.join(self.plugin_dir, "test_python.py")
        

        # Mac OS specific checks
        if self.system == "darwin":
            # Make E+ application executable
            try:
                chmod_cmd = subprocess.run(["chmod", "+x", self.energyplusexe], check=True)
            except subprocess.CalledProcessError:
                self.initial_tests.append("Chmod command failed on EnergyPlus.")

            # Same for ReadVarsESO
            try:
                chmod_cmd = subprocess.run(["chmod", "+x", self.readvarseso], check=True)
            except subprocess.CalledProcessError:
                self.initial_tests.append("Chmod command failed on ReadVarsESO.")

            # Same for sh script
            mac_verify_ep = os.path.join(self.plugin_dir, "mac_verify_ep.sh")
            try:
                chmod_cmd = subprocess.run(["chmod", "+x", mac_verify_ep], check=True)
            except subprocess.CalledProcessError:
                self.initial_tests.append("Chmod command failed on sh script.")
            
            # Call the sh script to bypass all the security warnings that occur when running E+
            mac_verify_ep_result = subprocess.run(["bash", mac_verify_ep], capture_output=True)
            if mac_verify_ep_result.returncode != 0:
                msg = ("Mac verify EnergyPlus sh script failed, but if",
                      "EnergyPlus runs correctly then this is not a problem.",
                      "It is possible that the initial setup was already run before.")
                stderr = mac_verify_ep_result.stderr.decode("utf-8")
                print(msg)
                self.initial_tests_warnings.append(stderr)

            # If Mac uses ARM architecture, check that user has installed Rosetta
            if platform.processor().casefold() == "arm":
                try:
                    if not os.path.exists("/usr/libexec/rosetta"):
                        raise Warning
                    if len(os.listdir("/usr/libexec/rosetta")) == 0:
                        raise Warning
                except Warning:
                    self.initial_tests_warnings.append("This appears to be a Silicon Mac with an ARM processor.\n"
                                                    "Please ensure Rosetta is installed to allow EnergyPlus to run.")
                

        # Run a test to see if E+ works
        shoebox_output = os.path.join(self.plugin_dir, "shoebox-output")
        if os.path.exists(shoebox_output):
            shutil.rmtree(shoebox_output)
        epw_file = os.path.join(self.plugin_dir, "testing.epw")
        run_ep_test = subprocess.run([self.energyplusexe, '-r','-d', shoebox_output, '-w', epw_file, "shoebox.idf"],
                                     cwd=self.plugin_dir)
        if not os.path.exists(os.path.join(shoebox_output, "eplusout.err")):
            self.initial_tests.append("EnergyPlus could not run.")
        else:
            print("EnergyPlus test completed successfully")
        

        # Run a test to see if ReadVarsESO works
        try:
            subprocess.run([self.readvarseso], cwd=shoebox_output, check=True)
            print("ReadVarsESO test completed successfully")
        except subprocess.CalledProcessError:
            self.initial_tests.append("ReadVarsESO failed to run.")
        

        # Test that the QGIS Python works via subprocess
        run_python_test = subprocess.run([self.qgis_python_location, test_python],
                                         capture_output=True, text=True)
        if run_python_test.stdout != "success\n":
            self.initial_tests.append("Python could not be run.")
        else:
            print("Python test completed successfully")
        

        # Check if any tests failed and report these if necessary
        if len(self.initial_tests) != 0:
            # TODO: print errors in QGIS console message if possible
            qgis.utils.iface.messageBar().pushMessage("Initial setup failed",
                                                      "Some errors have occured - please check the Python console outputs.",
                                                      level=Qgis.Critical,
                                                      duration=5)
            self.initial_setup_worked = False

            # Print errors to Python console
            print("\nERRORS:")
            [print(error) for error in self.initial_tests]

            # Print warnings to Python console (if any)
            if len(self.initial_tests_warnings) != 0:
                print("\nWARNINGS:")
                [print(warning) for warning in self.initial_tests_warnings]

        else:
            self.initial_setup_worked = True
            qgis.utils.iface.messageBar().pushMessage("Initial setup complete",
                                                      "Initial setup completed successfully. Please restart QGIS.",
                                                      level=Qgis.Success)
            print("\nInitial setup completed successfully. Please restart QGIS.")

            # Delete test files
            if os.path.exists(shoebox_output):
                shutil.rmtree(shoebox_output)



    def ask_permission(self):
        from .Simstock_QGIS_dialog import PermissionDialog
        self.dlg2 = PermissionDialog()
        self.dlg2.permissionBox.accepted.connect(self.permission_yes)
        self.dlg2.permissionBox.rejected.connect(self.permission_no)
        self.dlg2.exec_()


    def permission_yes(self):
        self.permission = "y"
        #self.dlg2.done(0)
        self.dlg2.close()

    def permission_no(self):
        self.permission = "n"
        self.dlg2.close()



    def add_fields(self):
        """Allows results_mode=False to be passed to add_new_layer()."""
        
        # Check if Add Fields has already been used
        if not self.added_fields:
            self.added_fields = True
            self.add_new_layer(results_mode=False)
        else:
            print("Please reload the plugin if 'Add Fields' needs to be used again.")
            


    def run_ep(self, idf_path):
        # TODO: move to mptest.py
        idf_fname = os.path.basename(idf_path)
        output_dir = idf_fname[:-4]
        out = subprocess.run([self.energyplusexe, '-d', output_dir, '-w', self.epw_file, idf_fname],
                              cwd=self.idf_dir, capture_output=True, text=True) #no readvarseso
        if out.returncode != 0:
            raise RuntimeError(out.stderr+"\nCheck the err file for %s" % idf_fname)
        
        # For debugging
        #with open(os.path.join(self.plugin_dir, "append1.txt"), "a") as f:
        #    f.write(str(out))# + "\n")



    def run_plugin(self):
        # Check if initial setup worked
        if self.initial_setup_worked is not None:
            if not self.initial_setup_worked:
                print("Warning: Initial setup previously failed - Simstock may not function correctly.")
        
        # Check if user cwd has been set
        if not self.cwd_set:
            raise RuntimeError("Please set the working directory first!")

        # Setup basic settings idf from database materials/constructions
        self.setup_basic_settings()
        
        if self.simulation_started:
            print("\nTo re-run the simulations, please close any open plugin windows and refresh the plugin using the reloader.")
            
        else:
            qgis.utils.iface.messageBar().pushMessage("Simstock running...", "Simstock is currently running. Please wait...", level=Qgis.Info, duration=5)
            self.simulation_started = True

            ### EXTRACT DATA
            # Get layer, check exists and extract features
            self.selectedLayer = self.dlg.mMapLayerComboBox.currentLayer()
            if self.selectedLayer is None:
                raise RuntimeError("Layer does not exist.")
            if not isinstance(self.selectedLayer, QgsVectorLayer):
                raise TypeError("Simstock expects a Vector Layer as input.")
            if not self.selectedLayer.isSpatial():
                raise TypeError("Layer has no geometry.")
            
            self.features = [feature for feature in self.selectedLayer.getFeatures()]
            
            # Path to qgz file
            path_to_file = QgsProject.instance().absoluteFilePath()
            
            # Extract geometry data from layer as polygons
            polygon = [feature.geometry().asWkt() for feature in self.features]
            
            # Extract all other required Simstock data from layer
            headings = [heading.split("-")[0] for heading in self.headings]
            dfdict = {}
            dfdict[headings[0]] = polygon
            for heading in headings[1:]:
                try:
                    dfdict[heading] = [feature[heading] for feature in self.features]
                except KeyError:
                    raise Exception("Attribute '%s' was not found in the attribute table. Check that it is present and spelled correctly and try again." % heading)

            # Data checks
            # Check values which are required for all polygons
            for y, value in enumerate(dfdict["shading"]):
                if isinstance(value, str) and value.lower() not in ["false", "true"]:
                    raise ValueError("Values in the 'shading' field should be 'true' or 'false'.\n Received: '%s' for %s." % (value, dfdict["UID"][y]))
                if isinstance(value, QVariant):
                    raise ValueError("Values in the 'shading' field should be 'true' or 'false'.\n Check value for %s." % dfdict["UID"][y])
                if isinstance(dfdict["height"][y], QVariant):
                    raise ValueError("Check 'height' value for %s." % dfdict["UID"][y])
                if dfdict["height"][y] == 0:
                    raise ValueError("Height value for %s is zero." % dfdict["UID"][y])
                if dfdict["UID"][y] == "":
                    raise ValueError("UID(s) missing! Do not edit the UID column.\nTo regenerate these, delete the entire column and use 'Add Fields' again.")
            
            # TODO: change shading field type to bool?
            if len(set(dfdict["shading"])) == 1 and list(set(dfdict["shading"]))[0] == "true":
                raise ValueError("Polygons cannot all be shading! Ensure that some are set to 'false'.")

            # Check values which are required for only non-shading polygons
            for y, value in enumerate(dfdict["shading"]):
                if str(value).lower() == "false":
                    if isinstance(dfdict["wwr"][y], QVariant):
                        raise ValueError("Check 'wwr' value for %s" % dfdict["UID"][y])
                    if dfdict["construction"][y] == "":
                        raise ValueError("Check 'construction' value for %s" % dfdict["UID"][y])
                    if isinstance(dfdict["ventilation_rate"][y], QVariant):
                        raise ValueError("Check 'ventilation_rate' value for %s" % dfdict["UID"][y])
                    if isinstance(dfdict["infiltration_rate"][y], QVariant):
                        raise ValueError("Check 'infiltration_rate' value for %s" % dfdict["UID"][y])
                    if isinstance(dfdict["nofloors"][y], QVariant):
                        raise ValueError("Check 'nofloors' value for %s" % dfdict["UID"][y])
                    if dfdict["nofloors"][y] == 0:
                        raise ValueError("Polygon %s has zero value for 'nofloors'." % dfdict["UID"][y])
            
            # Extract floor-specific attributes
            max_floors = max(dfdict["nofloors"])
            for x in range(max_floors):
                heading = "FLOOR_{}: use".format(x+1)
                try:
                    dfdict[heading] = [feature[heading] for feature in self.features]
                except KeyError:
                    print("""Could not find 'use' column(s). Assuming all zones to be 'Dwell'.\n
                          To add the 'use' columns, fill out the 'nofloors' column and then use 
                          'Add Fields' afterwards.""")

            # Save data as csv for Simstock to read
            data = pd.DataFrame(dfdict)
            data = data.rename(columns={"UID":"osgb"})
            data.to_csv(os.path.join(self.plugin_dir, "sa_data.csv"))
            
            
            
            ### SIMSTOCK
            # Import and run Simstock
            import simstockone as first
            import simstocktwo as second
            first.main()
            self.preprocessed_df = pd.read_csv(os.path.join(self.plugin_dir, "sa_preprocessed.csv"))
            second.main(idf_dir = self.idf_dir)
            

            
            ### SIMULATION
            def run_simulation(multiprocessing = True):
                """
                Run E+ simulation, generate .rvi files and run ReadVarsESO
                Outputs: a list of directories containing the results for each idf.
                """
                #qgis.utils.iface.messageBar().pushMessage("Running simulation", "EnergyPlus simulation has started...", level=Qgis.Info, duration=3)

                # Weather file
                self.epw_file = os.path.join(self.user_cwd, self.config["epw"])
                if not os.path.exists(self.epw_file):
                    raise FileNotFoundError(f"Weather epw_file '{self.epw_file}'\nnot found! Check "
                    "that it exists in the plugin's base directory and that is spelled correctly in "
                    "the 'config.json' file.")

                # List of output directory names
                idf_result_dirs = []
                for idf_path in self.idf_files:
                    idf_result_dirs.append(idf_path[:-4])
                
                # Single core
                if not multiprocessing:
                    print("Running EnergyPlus simulation on a single core...")
                    for i, idf_file in enumerate(self.idf_files):
                        print(f"Starting simulation {i+1} of {len(self.idf_files)}")
                        self.run_ep(idf_file)
                    generate_rvis(idf_result_dirs)
                    run_readvarseso(idf_result_dirs)
                
                # Parallel processing
                else:
                    print("Running EnergyPlus simulation on multiple cores...")
                    multiprocessingscript = os.path.join(self.plugin_dir, "mptest.py")
                    t1 = time.time()
                    out = subprocess.run([self.qgis_python_location, multiprocessingscript, self.user_cwd], capture_output=True, text=True)
                    if out.returncode != 0:
                        raise RuntimeError(out.stderr)
                    print(f"Simulation completed: took {round(time.time()-t1, 2)}s")
                    
                    # For debugging
                    #with open(os.path.join(self.plugin_dir, "append1.txt"), "a") as f:
                    #    f.write(str(out))# + "\n")
                
                #qgis.utils.iface.messageBar().pushMessage("EnergyPlus finished", "EnergyPlus simulation has completed successfully.", level=Qgis.Success)
                return idf_result_dirs
            
            def run_readvarseso(idf_result_dirs):
                """Calls ReadVarsESO after rvi files have been generated to
                produce the csv result files."""
                for dir in idf_result_dirs:
                    subprocess.run([self.readvarseso, "results-rvi.rvi", "unlimited"], cwd=dir)
            
            def generate_rvis(idf_result_dirs):
                """
                Generates .rvi files within each simulation directory
                to be used by ReadVarsESO

                TODO: combine with mp function so that only one needs to be changed
                """
                for dir in idf_result_dirs:
                    with open (os.path.join(dir, "results-rvi.rvi"), "w") as f:
                        f.write("eplusout.eso\neplusout.csv\n0")

            # Run E+ simulation, generate .rvi files and run ReadVarsESO
            #self.idf_files = [file.path for file in os.scandir(self.idf_dir) if file.name[-4:] == ".idf"]
            self.idf_files = [os.path.join(self.idf_dir, f"{bi}.idf") for bi in self.preprocessed_df[self.preprocessed_df["shading"]==False]["bi"].unique()]
            self.idf_result_dirs = run_simulation(multiprocessing = self.dlg.cbMulti.isChecked()) #check if mp checkbox is ticked

            # Push the results to a new QGIS layer
            self.add_new_layer(results_mode=True)
            qgis.utils.iface.messageBar().pushMessage("Simstock completed", "Simstock has completed successfully.", level=Qgis.Success)

            
    ### RESULTS HANDLING
    def add_new_layer(self, results_mode=True):
        """
        This currently has the whole results procedure including:
            - Creating a new layer in memory
            - Retrieving the results of interest by thermal zone
            - Pushing the results back to the new layer
            - Pushing the new layer to the QGIS console
        This is also now called by the "Add Fields" button with results_mode=False
        TODO: move result fetching fns elsewhere.
        """
        # Set up Eppy
        from eppy.modeleditor import IDF, IDDAlreadySetError
        if self.system in ['windows', 'linux', 'darwin']:
            iddfile = os.path.join(self.EP_DIR, 'ep8.9_{}/Energy+.idd'.format(self.system))
        try:
            IDF.setiddname(iddfile)
        except IDDAlreadySetError:
            pass

        def getzones(idf):
            """Finds thermal zones in idf and outputs numpy array."""
            all_zones = np.array([zone.Name for zone in idf.idfobjects["ZONE"]])
            return all_zones

        def make_allresults_dict():
            """Returns a dict where the key is the name of the thermal
            zone and the value is a df containing all results."""
            all_results = {}
            for dir in self.idf_result_dirs:
                try:
                    df = pd.read_csv(os.path.join(dir, "eplusout.csv"))
                except pd.errors.EmptyDataError:
                    raise RuntimeError("Results for '%s' not found. Check EnergyPlus .err file" % os.path.basename(dir))
                idf = IDF(dir + ".idf")
                zonelist = getzones(idf)

                for zone in zonelist:
                    zonename = zone.upper() #E+ outputs zone names in caps in results
                    zonecols = [col for col in df.columns if zonename in col]
                    all_results[zone] = df[zonecols]

            return all_results

        def extract_results(all_results):
            """
            Extracts the results of interest from the individual dfs. Returns 
            a dict where the key is the zone name and the value is the results.

            Needs generalising
            """

            def get_result_val(output_name, df):
                """Looks into zone result df for a given output. Returns 
                the whole series of values."""
                # Find column(s) containing the specified output name
                value_col = [col for col in df.columns if output_name in col]

                # Raise error if no column is found
                if len(value_col) == 0:
                    raise RuntimeError("Cannot find %s value for zone '%s' in results." % (output_name, zone))
                
                # If more than one column is found
                if len(value_col) > 1:
                    print("Found two values for %s for zone '%s' in results." % (output_name, zone))
                
                # Return the column of interest
                series = df[value_col[0]] #should only be one col
                return series
            
            # Set up empty dict and get post-processing values
            extracted_results = {}
            #cooling_COP = float(self.config["Cooling COP"])
            #grid_factor = float(self.config["Grid factor - kgCO2/kWh"])
            #elec_cost   = float(self.config["Electricity cost - currency/kWh"])

            # Loop over each zone's results df
            for zone, df in all_results.items():
                # Output definition
                # Get operative temperature and use thresholds to get hours above/below
                operative_series = get_result_val("Zone Operative Temperature", df)
                below = operative_series[operative_series < self.low_temp_threshold].count()
                above = operative_series[operative_series > self.high_temp_threshold].count()
                
                # Get electricity consumption
                elec = get_result_val("Electricity", df).sum()

                # Get hypothetical heating/cooling loads
                heating_load = get_result_val("Heating", df).sum()
                cooling_load = get_result_val("Cooling", df).sum()
                #cooling_demand = cooling_load / cooling_COP #apply COP factor

                # Convert to kWh
                elec = round(elec / (3.6E6), 2)
                heating_load = round(heating_load / (3.6E6), 2)
                cooling_load = round(cooling_load / (3.6E6), 2)

                # # Combine to get total electricity demand
                # energy = elec + heating_load + cooling_demand
                # energy = round(energy / (3.6E6), 2) #convert to kWh

                # # Apply grid factor to get associated CO2 emissions in kg
                # co2_emissions = round(energy * grid_factor, 2)

                # # Apply cost of electricity to get total cost
                # total_cost = round(energy * elec_cost, 2)

                # Combine extracted results into list
                lst = [below, above, elec, heating_load, cooling_load] #TODO: this needs to be same order as attr_types, change to dict?
                lst = list(map(float, lst)) #change type from np float to float
                extracted_results[zone] = lst

            return extracted_results

        def new_attrs_all_floors(max_floors, attr_types, results_mode):
            """
            Creates a result field for each result type up to the max number of floors.
            
            Needs generalising - specifically using float for all result fields
            """
            new_attrs = []

            if results_mode:
                # Add the built island ref as a result
                new_attrs.append(QgsField('bi_ref', QVariant.String))

            if max_floors is not None:
                # Must add the same number of fields to each feature
                for i in range(max_floors):

                    # Loop over base result types
                    for attr_type in attr_types:

                        # Prepend floor number to result base name
                        attr_name_floor = "FLOOR_" + str(i+1) + ": " + attr_type

                        if results_mode:
                            # Using "Double" type (float) for all results fields
                            new_attrs.append(QgsField(attr_name_floor, QVariant.Double))
                        else:
                            # Using "String" type for all fields (should only be 'use')
                            new_attrs.append(QgsField(attr_name_floor, QVariant.String))

            # Get the names of each newly created attribute
            #attr_names = [attr.name() for attr in new_attrs]
            return new_attrs#, attr_names

        def add_results_to_features(fields, results_mode, extracted_results=None):
            """Adds the new attributes to the features and populates their values."""
            # Loop through each feature (polygon)
            for i in range(len(self.features)):

                # Update the feature to gain the new fields object
                self.features[i].setFields(fields, initAttributes=False)
                
                # Grab the attributes from this feature
                feature_attrs = self.features[i].attributes()
                
                if results_mode:
                    # Get the unique id for this feature
                    osgb = self.features[i].attribute("UID")
                    print(f"Retrieving results for '{osgb}'...")

                    # Find the BI ref
                    bi_ref = self.preprocessed_df.loc[self.preprocessed_df["osgb"] == osgb, "bi"].values[0]

                    # Collate the new results
                    result_vals = [bi_ref]
                    # Get all the thermal zones belonging to this feature (multifloors)
                    thermal_zones = [zone for zone in extracted_results.keys() if osgb in zone]
                    if len(thermal_zones) != 0: #ignore shading blocks

                        # Loop through the thermal zones belonging to the feature
                        for j, zone in enumerate(thermal_zones):

                            # Check the order is correct
                            print(f"    Found results for floor {j+1}: '{zone}'")
                            if zone[-1] != str(j+1):
                                print(f"Floor results are in the wrong order for zone '{zone}'.")

                            # Collect the results for the thermal zone
                            result_vals.extend(extracted_results[zone])
                    
                    # Put the results with the rest of the attributes ready for adding
                    feature_attrs.extend(result_vals)
                
                elif not results_mode:
                    # Check if UID values already exist
                    done = False
                    for feature in feature_attrs:
                        # This checks the values themselves rather than the col name
                        # TODO: This is not rigorous enough as it could incorrectly flag, but UID
                        #       field has already been added by this point. Could append to list...
                        if "UID" in str(feature):
                            done = True

                    # Add the UID values if not present
                    if not done:
                        # Note: This relies on the UID column being the first to be added which is
                        #       always true. The above check also avoids adding these values if they
                        #       already exist. If it is necessary to add other values, can use
                        #       [f.name() for f in self.features[i].fields()] to get fields in order.
                        feature_attrs.append(self.unique_ids[i])

                    # Debugging feature to add default values to a newly created layer
                    # Note: this fails if some of the columns already exist, but is only for testing
                    debugging = False #TODO: add these default vals to a list to make it easier
                    if debugging:
                        feature_attrs.append("false")
                        feature_attrs.append(3.0)
                        feature_attrs.append(12.0)
                        feature_attrs.append(int(1))
                        feature_attrs.append("uninsulated")
                        feature_attrs.append("uninsulated_glazing")
                        feature_attrs.append(0.7)
                        feature_attrs.append(2.0)

                # Set the feature's attributes
                self.features[i].setAttributes(feature_attrs)


        # Change some of the existing attributes if necessary (probably not)
        #self.features[0].setAttribute(1, "text")

        # Set name for new layer to be created
        if results_mode:
            if self.HeatCool.lower() == "false":
                new_layer_name = self.selectedLayer.name() + "_Simstock-results_HC-Off"
            elif self.HeatCool.lower() == "true":
                new_layer_name = self.selectedLayer.name() + "_Simstock-results_HC-On"
        else:
            self.selectedLayer = self.dlg.mMapLayerComboBox.currentLayer()
            new_layer_name = self.selectedLayer.name() + "_1"

        # Create new layer in memory for the results
        #TODO: Can CRS be sourced from project somehow?
        crs = self.config["CRS"] #get CRS from config file
        #crs = "epsg:4326" #or 32718
        mem_layer = QgsVectorLayer(f"Polygon?crs={crs}", new_layer_name, "memory")
        mem_layer_data = mem_layer.dataProvider()

        # Get attributes and fields from original layer
        layer_attrs = self.selectedLayer.dataProvider().fields().toList() # QgsField type
        layer_fields = self.selectedLayer.fields() # QgsFields type
        new_attrs = []

        if results_mode:
            # Needs generalising
            # Extract the results from the csvs by thermal zone
            all_results = make_allresults_dict()
            self.low_temp_threshold = float(self.config["Low temperature threshold"])
            self.high_temp_threshold = float(self.config["High temperature threshold"])
            #currency = self.config["Currency"]
            extracted_results = extract_results(all_results)

            # Output definition
            # The base names of the results fields to be added (floor number will be appended to these)
            attr_types = [f"Hours/yr below {self.low_temp_threshold}C operative temperature",
                          f"Hours/yr above {self.high_temp_threshold}C operative temperature",
                           "Electricity consumption (kWh/yr)",
                           "Heating load (kWh/yr)",
                           "Cooling load (kWh/yr)"]
            max_floors = int(self.preprocessed_df['nofloors'].max())

        else:
            attr_types = ["use"]
            self.features = [feature for feature in self.selectedLayer.getFeatures()]

            # Create unique IDs for each feature and ensure they are the same length
            padding = len(str(len(self.features)))
            self.unique_ids = [f"UID{str(i).zfill(padding)}" for i in range(len(self.features))]
            
            # Add fields which are not floor-specific
            for field in self.headings:
                heading, QVtype = field.split("-")
                if heading != "polygon":
                    exec(f"new_attrs.append(QgsField('{heading}', QVariant.{QVtype}))", globals(), locals())

            try:
                # This will only work the 2nd time when the nofloors field is present
                nofloors = [feature["nofloors"] for feature in self.features]
                max_floors = max(nofloors)

                # If the field exists but the values have not yet been filled out
                # then skip creating the floor-specific fields
                if isinstance(max_floors, QVariant):
                    print("Field 'nofloors' detected but no values inputted.")
                    max_floors = None

            except KeyError:
                # In the first instance, the layer won't have the nofloors field
                # So ignore the floor-specific fields for now
                max_floors = None
        
        # Add new attribute types for the results for all floors
        new_attrs.extend(new_attrs_all_floors(max_floors, attr_types, results_mode))

        # TODO: is this intermediate step necessary?
        for new_attr in new_attrs:
            layer_fields.append(new_attr)
        layer_attrs.extend(new_attrs)
        
        # Add the actual result values and push to features
        if results_mode:
            add_results_to_features(layer_fields, results_mode, extracted_results)
        else:
            add_results_to_features(layer_fields, results_mode)
        
        # Add the attributes into the new layer and push it to QGIS
        mem_layer_data.addAttributes(layer_attrs)
        mem_layer.updateFields()
        mem_layer_data.addFeatures(self.features)
        QgsProject.instance().addMapLayer(mem_layer)

        # Check the capabilities of the layer
        #caps = mem_layer.dataProvider().capabilities()
        #if caps & QgsVectorDataProvider.AddFeatures:
        #    print("can")
        
        # Refresh the map if necessary
        if qgis.utils.iface.mapCanvas().isCachingEnabled():
            self.selectedLayer.triggerRepaint()
        else:
            qgis.utils.iface.mapCanvas().refresh()



    def load_database(self, file_exists):
        def csv_to_gpkg(database_csvs, database_layer_names):
            """
            Converts a provided list of csvs to a single geopackage file that can 
            be loaded and edited. May be better to remove this from the plugin 
            and start from the gpkg point.
            """

            # These are necessary arguments for writing gpkg layers
            context = QgsCoordinateTransformContext()
            o_save_options = QgsVectorFileWriter.SaveVectorOptions()

            for i, file in enumerate(database_csvs):

                # Load the csv as a vector layer
                uri = "file:///" + file.path + "?delimiter={}".format(",")
                vlayer = QgsVectorLayer(uri, database_layer_names[i], "delimitedtext")

                # First layer addition to gpkg must be done differently to the rest
                if i == 0:
                    o_save_options.layerName = database_layer_names[i]
                    writer = QgsVectorFileWriter.writeAsVectorFormatV3(vlayer, self.gpkg_path[:-5], context, o_save_options)

                # Add the remaining layers
                else: 
                    o_save_options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer 
                    o_save_options.EditionCapability = QgsVectorFileWriter.CanAddNewLayer
                    o_save_options.layerName = database_layer_names[i]
                    writer = QgsVectorFileWriter.writeAsVectorFormatV3(vlayer, self.gpkg_path[:-5], context, o_save_options)

        def load_all_layers_from_gpkg(gpkg, layer_names):
            """Loads all layers from a gpkg file when given their names as a list."""
            for layer in layer_names:
                qgis.utils.iface.addVectorLayer(gpkg + "|layername=" + layer, layer, 'ogr')

        # Find database csvs which contain the default setup idf objects
        self.database_dir = os.path.join(self.plugin_dir, "Database")
        database_csvs = [file for file in os.scandir(self.database_dir) if file.name[-4:] == ".csv" if file.name[:len(self.database_tag)] == self.database_tag]
        database_layer_names = [file.name[:-4] for file in database_csvs] #TODO: remove and use .name method in place

        # If the database gpkg doesn't exit, create it from the csvs
        if not file_exists:
            csv_to_gpkg(database_csvs, database_layer_names)
        
        # Add the database layers
        load_all_layers_from_gpkg(self.gpkg_path, database_layer_names)
    


    def set_cwd(self):
        """
        Sets the input path as the cwd. Used for outputting idfs and database files.
        The cwd will be checked for an existing database file. This will be loaded
        if it exists and a new one created if not.
        """
        # User specified directory for output
        self.user_cwd = self.dlg.mQgsFileWidget.filePath()

        # Check path provided
        if self.user_cwd == "":
            raise FileNotFoundError("Please enter a directory.")
        if not os.path.exists(self.user_cwd):
            raise FileNotFoundError("The selected directory does not exist - please create it if necessary.")
        
        # Set abspath after checks
        self.user_cwd = os.path.abspath(self.user_cwd)

        # Set idf path
        self.idf_dir = os.path.join(self.user_cwd, "idf_files")
        
        print("Loading database...")

        # First check for existing database layers and remove them
        layers = QgsProject.instance().mapLayers()
        database_layer_ids = []
        for _, layer in layers.items():
            if layer.name()[:len(self.database_tag)] == self.database_tag: #find database layers
                database_layer_ids.append(layer.id())
        QgsProject.instance().removeMapLayers(database_layer_ids)

        # Either load or create the gpkg file
        self.gpkg_name = "Simstock-Database.gpkg"
        self.gpkg_path = os.path.join(self.user_cwd, self.gpkg_name)
        if os.path.exists(self.gpkg_path):
            print("Found existing database file. Loading into workspace...")
            self.load_database(file_exists=True)
        else:
            print("Database file not found. Creating from defaults...")
            self.load_database(file_exists=False)
        
        self.cwd_set = True
        print("Current working directory set to: ", self.user_cwd)



    def setup_basic_settings(self):
        """
        Adds materials and constructions to the basic settings idf based on 
        what is in the database files. This is run when the plugin simulation 
        button is pressed.
        """

        def create_obj_dicts(df, dfname=None):
            """
            Converts database dataframe to a list of dictionaries, ignoring 
            both null and notes fields.
            
            Each dict within the returned list corresponds to a standalone idf object. 
            The keys represent the field names of the idf object.
            """
            dict_list = []

            # Each row represents a standalone idf object, so loop over each
            for _, row in df.iterrows():
                dictionary = {}

                # Loop over elements within the row (i.e. fields of the idf object)
                for i, content in enumerate(row):
                    # Get the row header (i.e. field name)
                    label = row.index[i]

                    # Only add the field if it has content and is not notes
                    if not content == qgis_null and not content == "" and not label == 'Notes': #using qgis nulltype instead of pd here

                        # Next check avoids QGIS bug where it appends "_1" to numbered fields
                        # when importing csv (e.g. "Field_1" becomes "Field_1_1")
                        if dfname is not None and "schedule" in dfname.lower() and label[:5] == "Field":
                            label = label[:-2]

                        # Add to dict    
                        dictionary[label] = content
                
                # Append individual dict corresponding to the idf object
                dict_list.append(dictionary)
            return dict_list

        def get_required_materials(const):
            """
            Looks in constructions and extracts the names of all required materials 
            (i.e. the materials which are used in any of the constructions).
            """
            materials_list = []
            for item in const:
                materials = list(item.values())[1:] #ignore construction names
                materials_list = [y for x in [materials_list, materials] for y in x]

            dict_list=[]
            for b in set(materials_list): #get unique
                d = {}
                d['Name'] = b 
                dict_list.append(d)
            return dict_list

        def add_dict_objs_to_idf(obj_dict, class_name):
            """Adds objects to the idf."""
            for obj in obj_dict:
                idf.newidfobject(class_name,**obj)

        def add_materials(key, material_df, used_materials):
            """Checks against required materials list, and adds those required to the idf."""
            materials = create_obj_dicts(material_df)
            material_type = key.split("-")[-1].replace("_", ":") #change names to match energyplus fields
            
            # Add the material properties to the specific instance of the material
            new_list=[]    
            for mat in materials:
                for item in used_materials:
                    if item['Name']==mat['MatName']: #check if required
                        new_item=dict(item,**mat)
                        new_list.append(new_item) #add if so
                        
            for el in new_list:
                del(el['MatName'])#remove the temporary matname field
                idf.newidfobject(material_type,**el) #expand each dictionary as a new material of the relevant type
            
        def attributes_to_dfs(layers):
            """Converts layer's attribute table into dataframe and appends to a dictionary"""
            dict = {}
            for layer in layers:
                cols = [field.name() for field in layer.fields()]
                datagen = ([f[col] for col in cols] for f in layer.getFeatures())
                df = pd.DataFrame.from_records(data=datagen, columns=cols)
                df.drop(columns=['fid'], inplace=True)
                dict[layer.name()] = df
            return dict
        
        def bool_quick_fix(dfs):
            """To delete"""
            # TODO: work out a permanent fix for this
            # One option is by using the csvt method that is in use for the schedules
            try:
                dfs["DB-Fabric-WINDOWMATERIAL_GLAZING"]["Solar_Diffusing"] = "No"
            except KeyError:
                pass
            try:
                dfs["DB-Loads-PEOPLE"]["Enable_ASHRAE_55_Comfort_Warnings"] = "No"
            except KeyError:
                pass


        # Set up Eppy
        from eppy.modeleditor import IDF, IDDAlreadySetError
        if self.system in ['windows', 'linux', 'darwin']:
            iddfile = os.path.join(self.EP_DIR, 'ep8.9_{}/Energy+.idd'.format(self.system))
        try:
            IDF.setiddname(iddfile)
        except IDDAlreadySetError:
            pass

        # Initialise base idf which will be added to and become the basic_settings.idf for Simstock
        idf = IDF(os.path.join(self.plugin_dir, 'base.idf'))
        
        # Find database layers in current project
        layers = QgsProject.instance().mapLayers()
        database_layers = []
        for _, layer in layers.items():
            if layer.name()[:len(self.database_tag)] == self.database_tag:
                database_layers.append(layer)

        # Convert database attribute tables to dataframes
        dfs = attributes_to_dfs(database_layers)
        # TODO: check for blank, missing, incorrect fields

        # This is a temporary quick fix to avoid certain fields being incorrectly
        # identified as bool type meaning that "No" is changed to "0"
        bool_quick_fix(dfs)

        # Add non-material objects to idf
        for key,df in dfs.items():
            if not "MATERIAL" in key and not "HeatingCooling" in key:
                df = dfs[key]
                class_name = key.split("-")[-1].replace("_", ":") #change names to match energyplus fields
                obj_dict = create_obj_dicts(df, key)
                add_dict_objs_to_idf(obj_dict, class_name)

                # Get materials used in constructions
                if "CONSTRUCTION" in key:
                    used_materials = get_required_materials(obj_dict)

        # Add materials to idf
        for key, df in dfs.items():
            if "MATERIAL" in key:
                add_materials(key, df, used_materials)

        # Check whether heating and cooling setpoints are to be included
        self.HeatCool = str(dfs["DB-HeatingCooling-OnOff"].iloc[0,0])
        if not isinstance(self.HeatCool, str):
            print("type ", type(self.HeatCool), self.HeatCool)
            raise NotImplementedError(f"self.HeatCool is {type(self.HeatCool)} type")

        # Choose heating & cooling setpoint schedules according to check
        if self.HeatCool.lower() == "false":
            print("Heating and cooling are not activated.")
            thermostats = idf.idfobjects["ThermostatSetpoint:DualSetpoint"]
            for thermostat in thermostats:
                # Swap the names
                thermostat.Heating_Setpoint_Temperature_Schedule_Name = "Dwell_Heat_Off"
                thermostat.Cooling_Setpoint_Temperature_Schedule_Name = "Dwell_Cool_Off"

        elif self.HeatCool.lower() == "true":
            # Schedules already have the correct names in this case
            print("Heating and cooling are activated.")

        # Save idf
        idf.saveas(os.path.join(self.plugin_dir, 'basic_settings.idf'))
        
"""
# For creating the csvt files:
lst = ['"String"']*103
s = ",".join(lst)
with open("test.csv", 'w') as f:
    f.write(s)
"""