# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SimpleETL
                                 A QGIS plugin
 Simple ETL for spatial data
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-07-07
        git sha              : $Format:%H$
        copyright            : (C) 2022 by OR2C
        email                : or2c@univ-nantes.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QThread
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QTableWidgetItem, QHeaderView

from qgis.core import Qgis, QgsProject, QgsMessageLog, QgsExpression
from qgis.gui import QgsFieldExpressionWidget

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .simple_etl_dialog import SimpleETLDialog
# Import worker
from .simple_etl_worker import ETLWorker
# Import helpers
from .schema_transformer_helper import SchemaTransformerHelper
from .logging_handler import LoggingHandler

import os
import logging

logger = logging.getLogger(__name__)
logging.basicConfig(level=os.environ.get("LOGLEVEL", "INFO"))

class SimpleETL:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'SimpleETL_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Simple ETL')

        # Save reference to the QDialog object
        self.dlg = None

        # Create Worker and Thread
        self.worker = ETLWorker()
        self.thread = QThread()

        # Initialize transformer helper
        self.transformer_helper = SchemaTransformerHelper()
        # Initialize logger helper
        self.logger_handler = LoggingHandler()

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('SimpleETL', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/simple_etl/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Simple ETL'),
            callback=self.run,
            parent=self.iface.mainWindow()
        )

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Simple ETL'),
                action)
            self.iface.removeToolBarIcon(action)

    def on_pb_start(self):
        self.dlg.progressBar.setValue(0)
        source_layer = self.dlg.sourceMapLayerComboBox.currentLayer()
        target_layer = self.dlg.targetMapLayerComboBox.currentLayer()
        if source_layer.isValid and target_layer.isValid:            
            ##################### Main Process #########################
            self.processETL()
        else:
            # TODO use log to print out message
            print("Input/Output layer is not valid")

    def on_pb_cancel(self):
        self.thread.quit()
        logger.info('ETL process cancelled')

    def on_pb_save(self):
        etl_parameters = {
            'use_geometry': self.dlg.checkBox_useGeometry.isChecked(),
            'update_data': self.dlg.checkBox_updateData.isChecked(),
            'id_field': self.dlg.comboBox_ID_field.currentIndex(),
            'all_features': self.dlg.radioButton_allFeatures.isChecked(),
        }
        transformation_def = self.generate_transformation_definition()
        self.transformer_helper.save_rules_to_file(etl_parameters, transformation_def)

        logger.info('Configuration file with transformation rules successfully saved')
    
    def on_pb_load(self):
        '''
        Load transformation rules and additional parameters from JSON file using helper.
        Populate table widgets with expressions
        '''
        try:
            configuration = self.transformer_helper.load_rules_from_file()
        except Exception as e:
            logger.error("Error loading config file. Exception: " + str(type(e).__name__) + ": " + str(e))
        else:
            if configuration:
                transformation_rules = configuration['transformation_rules']
                parameters = configuration['parameters']
                try:
                    # Apply loaded parameters
                    self.dlg.comboBox_ID_field.setCurrentIndex(int(parameters['id_field']))
                    self.dlg.checkBox_updateData.setChecked(bool(parameters['update_data']))
                    self.dlg.checkBox_useGeometry.setChecked(bool(parameters['use_geometry']))
                    self.dlg.radioButton_allFeatures.setChecked(bool(parameters['all_features']))
                    # Apply loaded transformation rules
                    for field_number, rule in transformation_rules:
                        exp_widget = self.dlg.tableWidget_targetFields.cellWidget(field_number, 1)
                        if exp_widget:
                            exp_widget.setExpression(rule) # setExpression expects a QString object and not QgsExpression
                except Exception as err:
                    logger.error("Error loading config file: {}, {}".format(type(err).__name__, str(err)))
                else:
                    logger.info('Transformation rules successfully loaded')
                    logger.info(f'Transformation rules : {transformation_rules}')

    def on_progress(self, i):
        self.dlg.progressBar.setValue(i)

    def add_log_message(self, message):
        widget_item = message
        self.dlg.listWidgetLog.addItem(widget_item)

    def on_layer_cb_change(self):
        # Get selected layer
        selected_source_layer = self.dlg.sourceMapLayerComboBox.currentLayer()
        self.iface.source_layer = selected_source_layer
        selected_target_layer = self.dlg.targetMapLayerComboBox.currentLayer()
        self.iface.target_layer = selected_target_layer

        # Check if data is not empty
        if selected_source_layer and selected_target_layer:
            # Clear existing data in ID field combobox then add selected layer fields
            self.dlg.comboBox_ID_field.clear()
            target_field_list = selected_target_layer.dataProvider().fields()
            self.dlg.comboBox_ID_field.addItems([field.name() for field in target_field_list])

            # Populate up the tableWidgets
            self.populate_layer_tables()

            # Enable start button
            self.dlg.pushButton_Start.setEnabled(True)
            self.dlg.pushButton_loadRules.setEnabled(True)

            logger.info('Layers have been selected. Define new rules or load previously saved config file.')
        else:
            logger.error("No data available. You must first load layers to your QGis project.")

    def populate_layer_tables(self):
        '''
        Fill/Update tables with layers' info 
        '''
        self.dlg.tableWidget_sourceFields.clearContents()
        self.dlg.tableWidget_targetFields.clearContents()
        
        source_provider = self.iface.source_layer.dataProvider()
        source_field_list = source_provider.fields()
        QgsMessageLog.logMessage('Source fields : ' + str(len(list(source_field_list))) , 'Simple-ETL', level=Qgis.Info)
        srcEncoding = source_provider.encoding()
        QgsMessageLog.logMessage('Source encoding : ' + str(srcEncoding), 'Simple-ETL', level=Qgis.Info)

        self.dlg.tableWidget_sourceFields.setRowCount(len(list(source_field_list)))
        self.dlg.tableWidget_sourceFields.setColumnCount(1)

        target_provider = self.iface.target_layer.dataProvider()
        # Remove duplicates and score fields from target layer if they exist
        target_field_list = [field for field in target_provider.fields() if field.name() not in ['duplicates', 'score']]

        self.dlg.tableWidget_targetFields.setRowCount(len(list(target_field_list)))
        self.dlg.tableWidget_targetFields.setColumnCount(2)

        for idx, field in enumerate(source_field_list):
            item = QTableWidgetItem(field.name() + '(' + field.typeName() + ')')
            self.dlg.tableWidget_sourceFields.setItem(idx, 0, item)
        
        for idx, field in enumerate(target_field_list):
            item = QTableWidgetItem(field.name() + '(' + field.typeName() + ')')
            self.dlg.tableWidget_targetFields.setItem(idx, 0, item)
            expItem = QgsFieldExpressionWidget()
            expItem.setLayer(self.iface.source_layer)
            self.dlg.tableWidget_targetFields.setCellWidget(idx, 1, expItem)

        # Resize tables
        target_header = self.dlg.tableWidget_targetFields.horizontalHeader()
        source_header = self.dlg.tableWidget_sourceFields.horizontalHeader()
        self.dlg.tableWidget_sourceFields.resizeColumnsToContents()
        self.dlg.tableWidget_sourceFields.setDragEnabled(True)
        self.dlg.tableWidget_targetFields.resizeColumnsToContents()
        self.dlg.tableWidget_targetFields.setDragEnabled(True)
        # Stretch last columns to entire space available
        target_header.setSectionResizeMode(1, QHeaderView.Stretch) # Second column (aka expression field)
        source_header.setSectionResizeMode(0, QHeaderView.Stretch)

    def generate_transformation_definition(self):
        '''
        Method that stores the transformation definition in 2 dimensional list (targetFieldID, QgsExpression)
        '''

        # Get the attribute transformation rules
        layer_transform_def = []
        target_fields = [f for f in self.iface.target_layer.fields() if f.name() not in ['duplicates', 'score']]
        try:
            for idx, field in enumerate(target_fields):
                exp_widget = self.dlg.tableWidget_targetFields.cellWidget(idx, 1)
                value, is_expression, isValid = exp_widget.currentField()
                if is_expression or len(value) > 0:
                    exp = QgsExpression(value)
                    layer_transform_def.append([idx, exp])
                else:
                    # Field is emty
                    pass
        except UnicodeEncodeError as err:
            logger.error("Encoding error : {}".format(err))
        return layer_transform_def
    
    def processETL(self):
        """
        Executes the main ETL process
        """
        # Check ETL parameters
        all_features_selected = self.dlg.radioButton_allFeatures.isChecked()
        use_geometry = self.dlg.checkBox_useGeometry.isChecked()
        update_data = self.dlg.checkBox_updateData.isChecked()
        source_layer = self.iface.source_layer
        target_layer = self.iface.target_layer
        # Get transformation definition
        transformation_def = self.generate_transformation_definition()
        
        # Set Worker object attributes
        self.worker.set_dialog(self.dlg)
        self.worker.set_source_layer(source_layer)
        self.worker.set_target_layer(target_layer)
        self.worker.set_feature_selection(all_features_selected)
        self.worker.set_transformation_definition(transformation_def)
        self.worker.set_use_geometry(use_geometry)
        self.worker.set_update_data(update_data)

        # Start Thread
        self.thread.start()
        #self.thread.finished.connect(app.exit) # * - Thread finished signal will close the app if you want!

    def cancel(self):
        self.thread.quit()
        logger.info('ETL process cancelled.')

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if not self.dlg:
            self.dlg = SimpleETLDialog()
            
            # 
            # NOTE Connect signals here so that they are only created once
            # Otherwise, they would be created each time the plugin is started (run() method is called)
            #
            
            # Initialize Worker object and QThread by connecting signals
            self.worker.progress.connect(self.on_progress) # Connect Worker's signals to plugin slots
            self.worker.moveToThread(self.thread) # **IMPORTANT** Move the Worker object to the Thread object.
            self.worker.finished.connect(self.thread.quit) # Connect Worker Signals to the Thread slots
            self.worker.cancelled.connect(self.thread.quit)
            self.worker.set_dialog(self.dlg)
            self.thread.started.connect(self.worker.run) # Connect Thread started signal to Worker operational slot method

            # Connect click event for Start and Cancel process buttons
            self.dlg.pushButton_Start.clicked.connect(self.on_pb_start)
            self.dlg.pushButton_Cancel.clicked.connect(self.on_pb_cancel)
            # Connect change event for source and target layers comboboxes
            self.dlg.sourceMapLayerComboBox.currentIndexChanged.connect(self.on_layer_cb_change)
            self.dlg.targetMapLayerComboBox.currentIndexChanged.connect(self.on_layer_cb_change)
            # Connect click event for Save and Load definition buttons
            self.dlg.pushButton_saveRules.clicked.connect(self.on_pb_save)
            self.dlg.pushButton_loadRules.clicked.connect(self.on_pb_load)
            # Connect logger output to logging widget
            self.logger_handler.log_signal.connect(self.add_log_message)
            logger.addHandler(self.logger_handler)
            self.worker.logger.addHandler(self.logger_handler) # Add same handler to worker logger

            # Disable buttons connected to actions that can't be called at this stage
            self.dlg.pushButton_Start.setEnabled(False)
            self.dlg.pushButton_loadRules.setEnabled(False)

        # Clear log messages
        self.dlg.listWidgetLog.clear()

        # Initialize values
        self.dlg.progressBar.setValue(0)
        self.dlg.radioButton_allFeatures.setChecked(True)
        self.dlg.listWidgetLog.setWordWrap(True) # Wrap log messages if their length exceeds widget width
        self.dlg.listWidgetLog.setAutoScroll(True) # Add scroll to QListWidget

        # Show the dialog
        self.dlg.show()

        # Fetch the currently loaded layers
        layers = QgsProject.instance().mapLayers().values()
        QgsMessageLog.logMessage('List of available layers: ' + str([layer.name() for layer in layers]), 'Simple-ETL', level=Qgis.Info)




