# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ETLWorker
                                 A QGIS plugin
 Simple ETL for spatial data
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-07-07
        git sha              : $Format:%H$
        copyright            : (C) 2022 by OR2C
        email                : or2c@univ-nantes.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QObject, QVariant, pyqtSignal, pyqtSlot
from qgis.core import QgsFeature, QgsExpressionContext, QgsExpressionContextUtils, QgsField

import os
import logging

# Import helpers
from .schema_transformer_helper import SchemaTransformerHelper

logger = logging.getLogger(__name__)
logging.basicConfig(level=os.environ.get("LOGLEVEL", "INFO"))

class ETLWorker(QObject):
    """
    Class to manage main ETL pipeline
    """

    finished = pyqtSignal()
    cancelled = pyqtSignal()
    progress = pyqtSignal(int)

    def __init__(self, *args, **kwargs):
        super(ETLWorker, self).__init__()
        # Worker parameters
        self.dialog = None
        self.source_layer = None
        self.target_layer = None
        self.all_features_selected = None
        self.transformation_definition = None
        self.logger = logger
        self.use_geometry = True
        self.update_data = False

    @pyqtSlot()
    def run(self):
        self.logger.info('ETL process started.')
        # Enable edits on target layer
        self.target_layer.startEditing()
        src_layer_provider = self.source_layer.dataProvider()
        trg_layer_provider = self.target_layer.dataProvider()

        # Get feature list depending on user selection
        if self.all_features_selected:
            features = src_layer_provider.getFeatures() # Returns Iterator that can be accessed only once (not reusable) !
        else:
            features = self.source_layer.getSelectedFeatures()
        target_fields = trg_layer_provider.fields()
        id_field = self.dialog.comboBox_ID_field.currentIndex()

        # Add new field to store possible duplicates
        duplicate_field = "duplicates"
        duplicate_score = "score"
        if duplicate_field not in [field.name() for field in target_fields]:
            trg_layer_provider.addAttributes([QgsField(duplicate_field, QVariant.String)])
        if duplicate_score not in [field.name() for field in target_fields]: 
            trg_layer_provider.addAttributes([QgsField(duplicate_score, QVariant.Double, "double", 10, 3)])
        self.target_layer.updateFields()

        new_features = []
        updated_features = 0
        skipped_features = 0
        duplicate_detected = 0
        progress_count = 0
        # Define increment to emit appropriate value to the Qt ProgressBar
        progress_increment = 100 / src_layer_provider.featureCount()
        for idx, source_feature in enumerate(features):
            # Create new feature
            target_feature = QgsFeature(target_fields)
            # Apply transformation
            target_feature = self.apply_feature_transformation(source_feature, target_feature)
            # Check unicity constraint in target database based on ID field
            if target_feature.attribute(id_field):
                existing_feature = SchemaTransformerHelper.check_existing_feature(target_feature, trg_layer_provider, id_field)
                if not existing_feature:
                    if target_feature.attribute(duplicate_field):
                        duplicate_detected += 1
                    new_features.append(target_feature)
                else:
                    # Feature already exists in database
                    # TODO decide what to do in that case : e.g update fields, fill only missing data or open dialog to get user input
                    # For instance, just overwrite or skip adding the feature depending on ETL config
                    if self.update_data:
                        # Update existing feature : simply overwrite
                        feature_to_update = next( iter([feature for feature in trg_layer_provider.getFeatures() if str(feature.attribute(id_field)) == str(target_feature.attribute(id_field))]) )
                        # Update feature in the target layer
                        # NOTE QgsFeature::attributeMap() is implemented in QGIS > 3.22.2 so manually create the map instead
                        target_feature_map = { i : value for i, value in enumerate(target_feature.attributes())}
                        feature_to_update_map = { i : value for i, value in enumerate(feature_to_update.attributes())}
                        self.target_layer.changeAttributeValues(feature_to_update.id(), target_feature_map, feature_to_update_map) # Update attribute values
                        self.target_layer.changeGeometry(feature_to_update.id(), target_feature.geometry()) # Update geometry
                        updated_features += 1
                    else:
                        # We don't want to update data, so just skip this feature
                        skipped_features += 1
                        self.logger.info(f"Feature n°{idx} could not be added because ID {target_feature.attribute(id_field)} already exists in target layer.")
            else:
                self.logger.info(f"Feature n°{idx} could not be added. ID could not be created. Check input data and transformation rules for possible null values.")

            self.progress.emit(int(progress_count))
            progress_count += progress_increment
        if duplicate_detected:
            self.logger.info(f'{duplicate_detected} duplicate detected. Check output to clean them.')
        # Write to target layer
        if new_features:
            self.target_layer.addFeatures(new_features)
        self.target_layer.commitChanges()
        self.finished.emit()
        self.progress.emit(100)
        self.logger.info(f'ETL process completed, {len(new_features)} new features added, {updated_features} features updated, {skipped_features} features skipped.')

    def apply_feature_transformation(self, source_feature: QgsFeature, target_feature: QgsFeature) -> QgsFeature:
        '''
        Apply transformation rules defined by user to integrate source feature to target layer
        '''

        # Instantiate expression context from source layer
        exp_context = QgsExpressionContext(QgsExpressionContextUtils.globalProjectLayerScopes(self.source_layer))

        # Get the user defined ID field in target database
        id_field = self.dialog.comboBox_ID_field.currentIndex()
        # Get corresponding ID for source feature
        feature_id = source_feature.attribute(id_field)

        # For every field, get the expression and apply transformation.
        # NOTE: QGis expression can be just a field name from source layer, or a more complex transformation rule
        for rule in self.transformation_definition:
            rule_id = rule[0]
            rule_expression = rule[1]
            rule_expression.prepare(exp_context)

            # Check if expression has evaluation or parsing errors
            # If so, just stop the process
            if rule_expression.hasEvalError():
                self.logger.error(f'Error during expression evaluation for field #{rule_id} on feature {feature_id}: {rule_expression.evalErrorString()}')
                self.cancelled.emit()
                raise Exception()
            elif rule_expression.hasParserError():
                self.logger.error(f'Error parsing expression for field #{rule_id} on feature {feature_id}: {rule_expression.parserErrorString()}')
                self.cancelled.emit()
                raise Exception()
            else:
                # Check if some expression has been defined for this field
                if len(rule_expression.expression()) > 1:
                    try:
                        exp_context.setFeature(source_feature)
                        target_feature[rule_id] = rule_expression.evaluate(exp_context)
                        if str(target_feature[rule_id]) == 'NULL':
                            #no Value
                            target_feature[rule_id] = None
                    except Exception as err:
                        self.logger.info(f'Error during integration of feature #{feature_id}. Error message: {err}')
                else:
                    # If no expression has been defined, just leave the field as blank
                    target_feature[rule_id] = None

        # Check if geometry should be considered. If so, add it to the target feature and check for duplicates
        if self.use_geometry:
            # Add geometry from source feature
            source_geometry = source_feature.geometry()
            # Check for possible duplicates
            possible_duplicate = []
            possible_subgeometry_duplicate = []
            min_distance = 9999
            orientation_max = 20 # degree from North
            duplicate_score = 0
            #self.logger.error('Begin duplicate detection')
            for feat in self.target_layer.dataProvider().getFeatures():
                # Skip if the feature has the same ID as feature being added (possibly updating)
                if feat.attribute(id_field) != target_feature.attribute(id_field):
                    # Initialize score to 0
                    score = 0
                    # Check "exact" duplicates based on Hausdorff distance
                    distance = SchemaTransformerHelper.check_geometry_duplicate(source_geometry, feat.geometry())
                    # Check "close" duplicates based on buffer intersection and bbox orientation
                    buffer_intersection, intersection_area_max_ratio, intersection_area_min_ratio = SchemaTransformerHelper.check_buffer_intersection(source_geometry, feat.geometry(), 10)
                    #self.logger.error(f'Distance and buffer computed {distance} # {buffer_intersection} # {intersection_area_max_ratio} # {intersection_area_min_ratio}')
                    if distance < 10 and distance < min_distance:
                        #self.logger.error('Hausdorff threshold')
                        possible_duplicate = [source_feature, feat, distance]
                        min_distance = distance
                        score = 1
                    elif buffer_intersection:
                        #self.logger.error('Buffers intersect, check orientation and intersection area')
                        # If geometries intersect, check bounding boxes orientation
                        orientation_diff = SchemaTransformerHelper.check_geometry_orientation(source_geometry, feat.geometry())
                        if orientation_diff <= orientation_max:
                            # If orientation threshold passed, compute score based on orientation difference, intersection area ratio and minimal buffer coverage
                            possible_subgeometry_duplicate = [source_feature, feat]
                            score = round(intersection_area_min_ratio * intersection_area_max_ratio * (1 - (orientation_diff / (2 * orientation_max))), 1) # intersection_area_ratio score x orientation score (orientation max should score 0.5, null orientation should score 1)
                            #self.logger.error(f'Compute score {score}')
                        else:
                            #self.logger.error(f'Low proba of duplicate. Skip to next one')
                            pass
                    else:
                        pass
                    # Stop the loop if duplicate_score is 1
                    if score == 1:
                        duplicate_score = score
                        break
                    elif score > duplicate_score:
                        duplicate_score = score
                    else:
                        pass
                else:
                    pass
                #self.logger.error('Done checking : {}'.format(score))
            # Get the duplicate candidate ID and add it to the list
            if len(possible_duplicate) > 0:
                #self.logger.error('Duplicates found')
                #print("{}, {}, {}, {}".format(possible_duplicate[0].attribute(id_field), possible_duplicate[1].attribute(id_field), possible_duplicate[2], duplicate_score))
                # Add the list of possible duplicates into the "duplicates" field
                target_feature.setAttribute("duplicates", str(possible_duplicate[1].attribute(id_field)))
                target_feature.setAttribute("score", duplicate_score)
            elif len(possible_subgeometry_duplicate) > 0:
                #self.logger.error('Duplicates found on subgeometry')
                #print("Subgeometry similarity")
                print(f"{possible_subgeometry_duplicate[0].attribute(id_field)}, {possible_subgeometry_duplicate[1].attribute(id_field)}, {duplicate_score}")
                # Add the list of possible duplicates into the "duplicates" field
                target_feature.setAttribute("duplicates", str(possible_subgeometry_duplicate[1].attribute(id_field)))
                target_feature.setAttribute("score", duplicate_score)
            else:
                pass
            target_feature.setGeometry(source_geometry)

        return target_feature

    def set_dialog(self, dlg):
        self.dialog = dlg
    
    def set_source_layer(self, layer):
        self.source_layer = layer

    def set_target_layer(self, layer):
        self.target_layer = layer
    
    def set_feature_selection(self, all_features_selected):
        self.all_features_selected = all_features_selected
    
    def set_transformation_definition(self, transformation_definition):
        self.transformation_definition = transformation_definition

    def set_use_geometry(self, use_geometry):
        self.use_geometry = use_geometry
    
    def set_update_data(self, update_data):
        self.update_data = update_data