# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SimpleETL
                                 A QGIS plugin
 Simple ETL for spatial data
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-07-07
        git sha              : $Format:%H$
        copyright            : (C) 2022 by OR2C
        email                : or2c@univ-nantes.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from xmlrpc.client import boolean
from qgis.PyQt.QtCore import QObject, pyqtSignal, pyqtSlot
from qgis.PyQt.QtWidgets import QFileDialog
from qgis.core import QgsGeometry, QgsFeature, QgsDataProvider

import json

class SchemaTransformerHelper(QObject):
    def __init__(self):
        super(SchemaTransformerHelper, self).__init__()
        
    def save_rules_to_file(self, parameters: dict, transform_definition: list) -> None:
        '''
        Save transformation rules to JSON file
        '''
        file_dialog_window = QFileDialog()                                              
        file_dialog_window.setAcceptMode(1) # Means save File
        file_dialog_window.setDefaultSuffix("json")
        file_dialog_window.setNameFilters(["JSON files (*.json)"])
        if file_dialog_window.exec_() == 0:
            return
        filename = file_dialog_window.selectedFiles()[0]
        if filename:
            with open(filename,'w') as file:
                file_content = {}
                # Save ETL parameters
                file_content['parameters'] = {}
                file_content['parameters']['use_geometry'] = parameters['use_geometry']
                file_content['parameters']['id_field'] = parameters['id_field']
                file_content['parameters']['update_data'] = parameters['update_data']
                file_content['parameters']['all_features'] = parameters['all_features']

                # Save transformation rules
                file_content['rules'] = {}
                for rule in transform_definition:
                    field_number = rule[0]
                    expression = rule[1].dump() # Returns an expression string because QgsExpression is not JSON serializable
                    file_content['rules'][field_number] = expression
                
                # Save data to JSON file
                json.dump(file_content, file, indent = 4)

    def load_rules_from_file(self) -> dict:
        '''
        Load transformation rules from input JSON file.
        Return list of tuples (field_number: int, expression: str)
        '''
        transformation_rules = []
        file_dialog_window = QFileDialog()                                              
        file_dialog_window.setAcceptMode(0) # Means read File
        file_dialog_window.setDefaultSuffix("json")
        file_dialog_window.setNameFilters(["JSON files (*.json)"])
        if file_dialog_window.exec_() == 0:
            return
        filename = file_dialog_window.selectedFiles()[0]
        with open(filename, 'r') as json_file:
            data = json.load(json_file)
            # Load ETL parameters
            parameters = data['parameters']

            # Load transformation rules
            rules = data['rules']
            for field_number, rule in rules.items():
                rule_definition = (int(field_number), rule) # Return expression rule as String instead of QgsExpression
                transformation_rules.append(rule_definition)
        return { 
            'transformation_rules': transformation_rules,
            'parameters': parameters
        }

    @staticmethod
    def check_geometry_duplicate(ref_geometry: QgsGeometry, cur_geometry: QgsGeometry) -> float:
        '''
        Check for duplicates based on geometry.
        For sake of simplicity and computation speed, compare only bounding boxes using Hausdorff distance.
        '''
        return ref_geometry.hausdorffDistance(cur_geometry)

    @staticmethod
    def check_subgeometry_inclusion(ref_geometry: QgsGeometry, cur_geometry: QgsGeometry, buffer_size: float) -> bool:
        '''
        Check for duplicates based on subgeometries.
        Creates buffer from both input geometries and check whether one contains the other
        '''
        ref_buffer = ref_geometry.buffer(buffer_size, 5)
        cur_buffer = cur_geometry.buffer(buffer_size, 5)
        if ref_buffer.contains(cur_geometry) or cur_buffer.contains(ref_geometry):
            return True
        else:
            return False

    @staticmethod
    def check_buffer_intersection(ref_geometry: QgsGeometry, cur_geometry: QgsGeometry, buffer_size: float) -> (bool, float):
        '''
        Creates buffer from both input geometries and check whether they intersect
        Returns boolean and area ratio between intersection and input geometries area (max and min) if they do intersect
        '''
        ref_buffer = ref_geometry.buffer(buffer_size, 5)
        cur_buffer = cur_geometry.buffer(buffer_size, 5)
        if ref_buffer.intersects(cur_buffer):
            try:
                return (True, ref_buffer.intersection(cur_buffer).area() / max(ref_buffer.area(), cur_buffer.area()), ref_buffer.intersection(cur_buffer).area() / min(ref_buffer.area(), cur_buffer.area()))
            except:
                print(f'One of computed buffers has null area. Skip')
                return (False, None, None)
        else:
            return (False, None, None)

    @staticmethod
    def check_geometry_orientation(ref_geometry: QgsGeometry, cur_geometry: QgsGeometry) -> float:
        '''
        Compute orientation of input geometries and returns angle difference between them
        '''
        _, ref_geom_bbox_area, ref_geom_orientation, _, _ = ref_geometry.orientedMinimumBoundingBox()
        _, cur_geom_bbox_area, cur_geom_orientation, _, _ = cur_geometry.orientedMinimumBoundingBox()
        result = abs(ref_geom_orientation - cur_geom_orientation)
        return result

    @staticmethod
    def check_existing_feature(feature: QgsFeature, target_layer_provider: QgsDataProvider, id_field: int) -> bool:
        '''
        Check if feature already exists in layer based on ID field
        '''

        # Get features list from target layer (assuming it already contains some data)
        target_features = target_layer_provider.getFeatures()
        # Compute the list of IDs for every feature in target layer
        id_list = [str(feat.attribute(id_field)) for feat in target_features] # Use String type as the most common type to operate checking

        return str(feature.attribute(id_field)) in id_list

