# -*- coding: utf-8 -*-

"""
***************************************************************************
    EditScriptDialog.py
    ---------------------
    Date                 : December 2012
    Copyright            : (C) 2012 by Alexander Bruy
    Email                : alexander dot bruy at gmail dot com
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

__author__ = "Alexander Bruy"
__date__ = "December 2012"
__copyright__ = "(C) 2012, Alexander Bruy"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"

import codecs
import os
import warnings

from processing.gui.AlgorithmDialog import AlgorithmDialog
from processing.script import ScriptUtils
from qgis.core import QgsApplication, QgsError, QgsProcessingAlgorithm, QgsProcessingFeatureBasedAlgorithm, QgsSettings
from qgis.gui import QgsErrorDialog, QgsGui
from qgis.PyQt import uic
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QCursor
from qgis.PyQt.QtWidgets import QFileDialog, QMessageBox
from qgis.utils import OverrideCursor, iface

from .gui_utils import GuiUtils
from .algorithm import RAlgorithm
from ..Utils import ParamsR

# from qgis.processing import alg as algfactory


pluginPath = os.path.split(os.path.dirname(__file__))[0]

with warnings.catch_warnings():
    warnings.filterwarnings("ignore", category=DeprecationWarning)
    path = os.path.join(os.path.dirname(__file__),"DlgScriptEditor.ui")
    WIDGET, BASE = uic.loadUiType(path)


# This class is ported from the QGIS core Processing script editor.
# Unfortunately generalising the core editor to allow everything we want in an R editor
# isn't feasible... so lots of duplicate code here :(
# Try to keep the diff between the two as small as possible, to allow porting fixes from QGIS core


class ScriptEditorDialog(BASE, WIDGET):
    hasChanged = False

    def __init__(self, filePath=None, parent=None):
        super(ScriptEditorDialog, self).__init__(parent)
        self.setupUi(self)

        QgsGui.instance().enableAutoGeometryRestore(self)

        self.editor.initLexer()
        self.editor.setText("# Voici quelques conseils et éléments importants pour la création de vos scripts R\n"
                            "# Vous pouvez conserver les éléments qui vous intéressent et supprimer ceux qui ne vous servent pas)\n" \
                            "\n" \
                            "# I. Groupe et nom du script\n" \
                            "#ex.:\n" \
                            "#  Nom du groupe=group\n" \
                            "#  Nom du script=name\n" \
                            "\n" \
                            "# 2. Valeurs et couches en entrée du script.\n" 
                            "# Voici différents éléments possibles :\n" \
                            "#  Une couche vecteur (avec filtre) : ##Couche_vecteur=vector (point, polyline, polygon)\n" \
                            "#  Une couche raster : ##Couche_raster=raster\n" \
                            "#  Une couche optionelle : ##Couche_optionnelle=optional vector\n" \
                            "#  Le champ d'une couche : ##Nom_champ=Field Couche_vecteur\n" \
                            "#  Une valeur numérique avec une valeur par défaut : ##Nombre=number 0\n" \
                            "#  Une entrée de texte et sa valeur par défaut: ##Texte=string Test\n" \
                            "#  Un choix entre plusieurs valeurs numériques : ##Choix_nombre=enum literal 0;1;2;3;4 ;\n" \
                            "# Un choix entre plusieurs valeurs de texte : ##Choix_texte=selection Option 1; Option 2; Option 3 ;\n" \
                            "# Attention, si vos créez une liste avec des choix que vous voulez ensuite utiliser pour trouver une valeur spécifique, voici une fonction à rajouter pouvant vous aider :\n" \
                            "#  #val = liste de valeurs déclarées, string_map = Traduction des valeurs pour votre fonction\n" \
                            "#  get_string_value <- function(val, string_map) {\n" \
                            "#  string_map[val + 1] # +1 because R indexing starts from 1\n" \
                            "#   }\n" \
                            "\n" \
                            "# 3. Vaeurs et couches de sortie du script\n" \
                            "#  Sortie d'une couche vecteur : ##Sortie_vecteur=output vector\n" \
                            "#  Sortie d'une couche raster : ##Sortie_vecteur=output raster\n" \
                            "#  Sortie d'un nombre : ##Sortie_vecteur=output number\n" \
                            "#  Sortie d'un texte : ##Sortie_vecteur=output string\n" \
                            "#  Rajouter ceci en plus pour récupérer les différents étapes de l'analyse :\n" \
                            "#  ##output_plots_to_html\n" \
                            "#  ##showplots\n" \
                            "\n" \
                            "# 4. Fonctions du script\n" \
                            "# Il est bien entendu possible d'utiliser tous les outils de base de R... :\n" \
                            "# ex. :\n" \
                            "#  Min <- min(Couche_vecteur[[Nom_champ]])\n" \
                            "#  Max <- max(Couche_vecteur[[Nom_champ]])\n" \
                            "#  Summary <- paste(Min, 'to', Max, sep = ' ')\n" \
                            "# ... ou d'utiliser des librairies installées ou non. Attention, voici un segment de code pouvant vous aider :\n" \
                            "#   packages <- c('sp','sf','raster')\n" \
                            "#   installed_packages <- packages %in% rownames(installed.packages())\n" \
                            "#   if (any(installed_packages == FALSE)) {\n" \
                            "#       install.packages(packages[!installed_packages])\n" \
                            "#   }\n" \
                            "#   invisible(lapply(packages, library, character.only = TRUE))\n" \
                            "\n" \
                            "#  library(sp)\n" \
                            "#  library(sf)\n" \
                            "#  library(raster)"
                            )
        self.searchWidget.setVisible(False)

        if iface is not None:
            self.toolBar.setIconSize(iface.iconSize())
            self.setStyleSheet(iface.mainWindow().styleSheet())

        self.actionOpenScript.setIcon(QgsApplication.getThemeIcon("/mActionScriptOpen.svg"))
        self.actionSaveScript.setIcon(QgsApplication.getThemeIcon("/mActionFileSave.svg"))
        self.actionSaveScriptAs.setIcon(QgsApplication.getThemeIcon("/mActionFileSaveAs.svg"))
        self.actionRunScript.setIcon(QgsApplication.getThemeIcon("/mActionStart.svg"))
        self.actionCut.setIcon(QgsApplication.getThemeIcon("/mActionEditCut.svg"))
        self.actionCopy.setIcon(QgsApplication.getThemeIcon("/mActionEditCopy.svg"))
        self.actionPaste.setIcon(QgsApplication.getThemeIcon("/mActionEditPaste.svg"))
        self.actionUndo.setIcon(QgsApplication.getThemeIcon("/mActionUndo.svg"))
        self.actionRedo.setIcon(QgsApplication.getThemeIcon("/mActionRedo.svg"))
        self.actionFindReplace.setIcon(QgsApplication.getThemeIcon("/mActionFindReplace.svg"))
        self.actionIncreaseFontSize.setIcon(QgsApplication.getThemeIcon("/mActionIncreaseFont.svg"))
        self.actionDecreaseFontSize.setIcon(QgsApplication.getThemeIcon("/mActionDecreaseFont.svg"))

        # Connect signals and slots
        self.actionOpenScript.triggered.connect(self.openScript)
        self.actionSaveScript.triggered.connect(self.save)
        self.actionSaveScriptAs.triggered.connect(self.saveAs)
        self.actionRunScript.triggered.connect(self.runAlgorithm)
        self.actionCut.triggered.connect(self.editor.cut)
        self.actionCopy.triggered.connect(self.editor.copy)
        self.actionPaste.triggered.connect(self.editor.paste)
        self.actionUndo.triggered.connect(self.editor.undo)
        self.actionRedo.triggered.connect(self.editor.redo)
        self.actionFindReplace.toggled.connect(self.toggleSearchBox)
        self.actionIncreaseFontSize.triggered.connect(self.editor.zoomIn)
        self.actionDecreaseFontSize.triggered.connect(self.editor.zoomOut)
        self.editor.textChanged.connect(lambda: self.setHasChanged(True))

        self.leFindText.returnPressed.connect(self.find)
        self.btnFind.clicked.connect(self.find)
        self.btnReplace.clicked.connect(self.replace)
        self.lastSearch = None

        self.filePath = None
        if filePath is not None:
            self._loadFile(filePath)

        self.setHasChanged(False)

    def update_dialog_title(self):
        """
        Updates the script editor dialog title
        """
        if self.filePath:
            path, file_name = os.path.split(self.filePath)
        else:
            file_name = self.tr("Untitled Script")

        if self.hasChanged:
            file_name = "*" + file_name

        self.setWindowTitle(self.tr("{} - R Script Editor").format(file_name))

    def closeEvent(self, event):
        settings = QgsSettings()
        settings.setValue("/Processing/stateScriptEditor", self.saveState())
        settings.setValue("/Processing/geometryScriptEditor", self.saveGeometry())

        if self.hasChanged:
            ret = QMessageBox.question(
                self,
                self.tr("Save Script?"),
                self.tr("There are unsaved changes in this script. Do you want to keep those?"),
                QMessageBox.StandardButton.Save | QMessageBox.StandardButton.Cancel | QMessageBox.StandardButton.Discard,
                QMessageBox.StandardButton.Cancel,
            )

            if ret == QMessageBox.StandardButton.Save:
                self.saveScript(False)
                event.accept()
            elif ret == QMessageBox.StandardButton.Discard:
                event.accept()
            else:
                event.ignore()
        else:
            event.accept()

    def openScript(self):
        if self.hasChanged:
            ret = QMessageBox.warning(
                self,
                self.tr("Unsaved changes"),
                self.tr("There are unsaved changes in the script. Continue?"),
                QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No,
                QMessageBox.StandardButton.No,
            )
            if ret == QMessageBox.StandardButton.No:
                return

        scriptDir = ParamsR.default_scripts_folder()
        fileName, _ = QFileDialog.getOpenFileName(
            self, self.tr("Open script"), scriptDir, self.tr("R scripts (*.rsx *.RSX)")
        )

        if fileName == "":
            return

        with OverrideCursor(Qt.CursorShape.WaitCursor):
            self._loadFile(fileName)

    def save(self):
        self.saveScript(False)

    def saveAs(self):
        self.saveScript(True)

    def saveScript(self, saveAs):
        newPath = None
        if self.filePath is None or saveAs:
            scriptDir = ParamsR.default_scripts_folder()
            newPath, _ = QFileDialog.getSaveFileName(
                self, self.tr("Save script"), scriptDir, self.tr("R scripts (*.rsx *.RSX)")
            )

            if newPath:
                if not newPath.lower().endswith(".rsx"):
                    newPath += ".rsx"

                self.filePath = newPath

        if self.filePath:
            text = self.editor.text()
            try:
                with codecs.open(self.filePath, "w", encoding="utf-8") as f:
                    f.write(text)
            except IOError as e:
                QMessageBox.warning(self, self.tr("I/O error"), self.tr("Unable to save edits:\n{}").format(str(e)))
                return

            self.setHasChanged(False)

        QgsApplication.processingRegistry().providerById("siliciter").refreshAlgorithms()

    def setHasChanged(self, hasChanged):
        self.hasChanged = hasChanged
        self.actionSaveScript.setEnabled(hasChanged)
        self.update_dialog_title()

    def runAlgorithm(self):
        alg = RAlgorithm(description_file=None, script=self.editor.text())
        if alg.error is not None:
            error = QgsError(alg.error, "R")
            QgsErrorDialog.show(error, self.tr("Execution error"))
            return

        alg.setProvider(QgsApplication.processingRegistry().providerById("siliciter"))
        alg.initAlgorithm()

        dlg = alg.createCustomParametersWidget(iface.mainWindow())
        if not dlg:
            dlg = AlgorithmDialog(alg, parent=iface.mainWindow())

        canvas = iface.mapCanvas()
        prevMapTool = canvas.mapTool()

        dlg.show()

        if canvas.mapTool() != prevMapTool:
            if canvas.mapTool():
                canvas.mapTool().reset()
            canvas.setMapTool(prevMapTool)

    def find(self):
        textToFind = self.leFindText.text()
        caseSensitive = self.chkCaseSensitive.isChecked()
        wholeWord = self.chkWholeWord.isChecked()
        if self.lastSearch is None or textToFind != self.lastSearch:
            self.editor.findFirst(textToFind, False, caseSensitive, wholeWord, True)
        else:
            self.editor.findNext()

    def replace(self):
        textToReplace = self.leReplaceText.text()
        self.editor.replaceSelectedText(textToReplace)

    def toggleSearchBox(self, checked):
        self.searchWidget.setVisible(checked)
        if checked:
            self.leFindText.setFocus()

    def _loadFile(self, filePath):
        with codecs.open(filePath, "r", encoding="utf-8") as f:
            txt = f.read()

        self.editor.setText(txt)
        self.hasChanged = False
        self.editor.setModified(False)
        self.editor.recolor()

        self.filePath = filePath
        self.update_dialog_title()
