# -*- coding: utf-8 -*-
"""Utility functions for scaling numeric values to a common range.

These helpers are intended for use across Sidewalk Priority Toolkit
algorithms so that all scores share consistent 0–100 semantics.
"""

from typing import Union

Number = Union[int, float]


def scale_value(value: Number, old_min: Number, old_max: Number,
                new_min: Number, new_max: Number) -> int:
    """Linearly scale *value* from [old_min, old_max] to [new_min, new_max],
    returning an integer.

    The result is rounded to the nearest integer and clamped to the
    [new_min, new_max] range (respecting whether that range is ascending
    or descending). This matches the toolkit's convention that scores are
    0–100 integers rather than high-precision floats.

    If ``old_max <= old_min`` the function returns ``int(round(new_min))`` so
    that a degenerate input range (all values equal) does not cause
    division by zero. Callers are expected to handle special cases like
    "all values are zero" at a higher level if they need custom
    behaviour.
    """
    old_min_f = float(old_min)
    old_max_f = float(old_max)
    new_min_f = float(new_min)
    new_max_f = float(new_max)

    if old_max_f <= old_min_f:
        return int(round(new_min_f))

    ratio = (float(value) - old_min_f) / (old_max_f - old_min_f)
    raw = new_min_f + ratio * (new_max_f - new_min_f)

    # Clamp to the target range.
    if new_min_f < new_max_f:
        clamped = max(new_min_f, min(raw, new_max_f))
    else:
        clamped = max(new_max_f, min(raw, new_min_f))

    return int(round(clamped))
