# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Sidewalk Priority Toolkit - A QGIS plugin
 Processing provider for sidewalk priority analysis
                              -------------------
        begin                : 2025-10-13
        copyright            : (C) 2025 by Mark Stosberg
        email                : mark@stosberg.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
from qgis.PyQt.QtGui import QIcon
from qgis.core import QgsProcessingProvider

from .algorithms.walk_potential import (
    WalkPotentialFromGridAlgorithm,
)
from .algorithms.grid_priority_points import GridPriorityFromPointsAlgorithm
from .algorithms.grid_priority_lines import (
    GridPriorityFromLineAttributesAlgorithm,
    GridPriorityFromLineLengthAlgorithm,
)
from .algorithms.grid_priority_polygon_coverage import (
    GridPriorityFromPolygonCoverageAlgorithm,
)
from .algorithms.grid_priority_polygon_center import (
    GridPriorityFromPolygonCenterAlgorithm,
)
from .algorithms.grid_priority_proximity import GridPriorityProximityAlgorithm
from .algorithms.create_hex_grid import CreateHexGridAlgorithm
from .algorithms.create_missing_sidewalk_map import (
    CreateMissingSidewalkMapAlgorithm,
)
from .algorithms.grid_priority_apply_default_heatmap_style import (
    GridPriorityApplyDefaultHeatmapStyleAlgorithm,
)

class SidewalkPriorityToolkitProvider(QgsProcessingProvider):
    """
    Processing provider for Sidewalk Priority Toolkit algorithms.
    """

    def __init__(self):
        super().__init__()

    def id(self):
        """
        Returns the unique provider id, used for identifying the provider.
        """
        return 'sidewalkprioritytoolkit'

    def name(self):
        """
        Returns the provider name, which is used to describe the provider
        within the GUI.
        """
        return 'Sidewalk Priority Toolkit'

    def icon(self):
        """
        Returns an icon for the provider.
        """
        # You can return a custom icon here if you have one
        icon_path = os.path.join(os.path.dirname(__file__), 'icon.png')
        if os.path.exists(icon_path):
            return QIcon(icon_path)
        return QgsProcessingProvider.icon(self)

    def loadAlgorithms(self):
        """Loads all algorithms belonging to this provider."""
        self.addAlgorithm(WalkPotentialFromGridAlgorithm())
        self.addAlgorithm(GridPriorityFromPointsAlgorithm())
        self.addAlgorithm(GridPriorityFromLineAttributesAlgorithm())
        self.addAlgorithm(GridPriorityFromLineLengthAlgorithm())
        self.addAlgorithm(GridPriorityFromPolygonCoverageAlgorithm())
        self.addAlgorithm(GridPriorityFromPolygonCenterAlgorithm())
        self.addAlgorithm(GridPriorityProximityAlgorithm())
        self.addAlgorithm(CreateHexGridAlgorithm())
        self.addAlgorithm(CreateMissingSidewalkMapAlgorithm())
        self.addAlgorithm(GridPriorityApplyDefaultHeatmapStyleAlgorithm())

    def longName(self):
        """
        Returns the longer version of the provider name.
        """
        return 'Algorithms to calculation location priority, particularly for sidewalks'
