# -*- coding: utf-8 -*-
"""
Module for fetching isochrones from routing services
"""

import json
import threading
from typing import Dict, Optional
from qgis.PyQt.QtNetwork import QNetworkAccessManager, QNetworkRequest
from qgis.PyQt.QtCore import QUrl, QEventLoop


class IsochroneError(Exception):
    """Exception raised when isochrone generation fails"""
    pass


class IsochroneFetcher:
    """Class to fetch isochrones from a routing service"""
    
    # Thread-local storage for network managers
    _thread_local = threading.local()
    
    def __init__(self, url_template: str, max_retries: int = 3, progress_callback=None):
        """
        Initialize the isochrone fetcher
        
        :param url_template: URL template with {{lat}} and {{lon}} placeholders
        :param max_retries: Maximum number of retry attempts for failed requests
        :param progress_callback: Optional callback for progress/debug messages
        """
        self.url_template = url_template
        self.max_retries = max_retries
        self.progress_callback = progress_callback or (lambda x: None)
    
    @classmethod
    def _get_network_manager(cls):
        """Get or create a network manager for the current thread"""
        if not hasattr(cls._thread_local, 'network_manager'):
            cls._thread_local.network_manager = QNetworkAccessManager()
        return cls._thread_local.network_manager
    
    def get_isochrone_url(self, lat: float, lon: float) -> str:
        """
        Generate isochrone URL by replacing placeholders with coordinates
        
        :param lat: Latitude
        :param lon: Longitude
        :return: Complete URL string
        """
        url = self.url_template
        url = url.replace('{{lat}}', str(lat))
        url = url.replace('{{lon}}', str(lon))
        return url
    
    def fetch_isochrone(self, lon: float, lat: float) -> Optional[Dict]:
        """
        Fetch an isochrone for a given point
        
        :param lon: Longitude
        :param lat: Latitude
        :return: GeoJSON feature representing the isochrone, or None if failed
        :raises IsochroneError: If the request fails after all retries
        """
        url = self.get_isochrone_url(lat, lon)
        network_manager = self._get_network_manager()
        
        for attempt in range(self.max_retries):
            try:
                # Create request
                qurl = QUrl(url)
                request = QNetworkRequest(qurl)
                request.setHeader(request.UserAgentHeader, 'Walk Potential QGIS Plugin')
                request.setHeader(request.ContentTypeHeader, 'application/json')
                
                # Perform synchronous request using event loop
                reply = network_manager.get(request)
                event_loop = QEventLoop()
                reply.finished.connect(event_loop.quit)
                event_loop.exec_()
                
                # Debug: always log the full URL for testing
                if attempt == 0:
                    self.progress_callback(f"DEBUG: Full isochrone URL: {url}")
                    self.progress_callback(f"DEBUG: Response status: {reply.attribute(QNetworkRequest.HttpStatusCodeAttribute)}")
                
                # Check for network errors first
                if reply.error() != reply.NoError and not reply.attribute(QNetworkRequest.HttpStatusCodeAttribute):
                    error = reply.errorString()
                    if attempt < self.max_retries - 1:
                        import time
                        time.sleep(1)  # Wait 1 second before retry
                        continue
                    else:
                        raise IsochroneError(
                            f"Network error connecting to isochrone service: {error}. "
                            f"Please verify the service is running and accessible."
                        )
                
                status_code = reply.attribute(QNetworkRequest.HttpStatusCodeAttribute)
                
                if status_code and 200 <= status_code < 300:
                    # Success
                    data_bytes = reply.readAll()
                    data_str = bytes(data_bytes).decode('utf-8')
                    data = json.loads(data_str)
                    
                    # Debug: log response type
                    if attempt == 0:
                        self.progress_callback(f"DEBUG: Response type: {data.get('type', 'unknown')}")
                        self.progress_callback(f"DEBUG: Response keys: {list(data.keys())}")
                    
                    # Handle different response formats
                    # Some services return a FeatureCollection
                    if data.get('type') == 'FeatureCollection':
                        features = data.get('features', [])
                        if features:
                            return features[0]
                    # Some return a single Feature
                    elif data.get('type') == 'Feature':
                        return data
                    # Some might return just geometry
                    elif 'geometry' in data:
                        return {
                            'type': 'Feature',
                            'geometry': data['geometry'],
                            'properties': {}
                        }
                    
                    raise IsochroneError(f"Unexpected response format from isochrone service")
                
                elif status_code == 400:
                    # Bad request - probably no coverage or not near a road
                    return None
                elif status_code == 404:
                    # Not found - no coverage
                    return None
                else:
                    # Other error - might be temporary, retry
                    error = reply.errorString()
                    if attempt < self.max_retries - 1:
                        continue
                    else:
                        raise IsochroneError(
                            f"Isochrone request failed with status {status_code}: {error}"
                        )
            
            except json.JSONDecodeError as e:
                raise IsochroneError(f"Failed to parse isochrone response: {str(e)}")
            except Exception as e:
                if attempt < self.max_retries - 1:
                    continue
                else:
                    raise IsochroneError(f"Network error fetching isochrone: {str(e)}")
        
        return None
