# -*- coding: utf-8 -*-
"""This module provides two algorithms:

1. GridPriorityFromLineAttributesAlgorithm

   - Aggregates a numeric attribute from line features intersecting each
     grid cell (currently using the maximum value per cell).
   - Writes ``min_<metric_name>``, ``max_<metric_name>`` and
     ``<metric_name>_score`` attributes.

   Useful for "max speed", "max width", etc of streets intersecting the cell.

2. GridPriorityFromLineLengthAlgorithm
   - Sums the total length of line features intersecting each grid cell.
   - Writes ``<metric_name>_length`` and ``<metric_name>_score`` attributes.

  Useful for "Length of missing sidewalks in this cell".

Both algorithms scale scores to a 0–100 range so they can be combined
with other metrics in the Sidewalk Priority Toolkit.
"""

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsFeature,
    QgsField,
    QgsFields,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterField,
    QgsProcessingParameterString,
    QgsSpatialIndex,
    QgsFeatureSink,
    QgsUnitTypes,
)

from ..utils.scaling import scale_value


class GridPriorityFromLineAttributesAlgorithm(QgsProcessingAlgorithm):
    """Grid Priority (Values from Intersecting Lines).

    For each polygon cell in the input grid layer, this algorithm looks at
    all line features which intersect the cell, finds the minimum and
    maximum values of a chosen numeric attribute, and writes:

    - ``min_<metric_name>`` – minimum attribute value in intersecting lines
    - ``max_<metric_name>`` – maximum attribute value in intersecting lines
    - ``<metric_name>_score`` – 0–100 score based on the maximum value

    The score is scaled so it can be combined with other 0–100 metrics.
    Whether higher attribute values increase or decrease the score is
    controlled by a boolean parameter.
    """

    # Parameter names
    INPUT_GRID = "INPUT_GRID"
    INPUT_LINES = "INPUT_LINES"
    LINE_FIELD = "LINE_FIELD"
    METRIC_NAME = "METRIC_NAME"
    HIGHER_VALUES_INCREASE_SCORE = "HIGHER_VALUES_INCREASE_SCORE"
    OUTPUT = "OUTPUT"

    def tr(self, string):
        """Returns a translatable string with the self.tr() function."""
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return GridPriorityFromLineAttributesAlgorithm()

    def name(self):
        return "grid_priority_from_line_attributes"

    def displayName(self):
        return self.tr("Grid Priority (Values from Intersecting Lines)")

    def group(self):
        return ""

    def groupId(self):
        return ""

    def shortHelpString(self):
        return self.tr(
            "Prioritizes grid cells based on a numeric attribute from "
            "intersecting line features. For each grid cell, the minimum "
            "and maximum values of the chosen line attribute are recorded, "
            "and a 0–100 score is calculated from the maximum value."
        )

    def initAlgorithm(self, config=None):
        # Input grid layer (polygons)
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_GRID,
                self.tr("Grid Layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # Input line layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_LINES,
                self.tr("Line Layer"),
                [QgsProcessing.TypeVectorLine],
            )
        )

        # Field from line layer to use for the metric
        self.addParameter(
            QgsProcessingParameterField(
                self.LINE_FIELD,
                self.tr("Line attribute field"),
                parentLayerParameterName=self.INPUT_LINES,
                type=QgsProcessingParameterField.Numeric,
            )
        )

        # Metric name used as base for output attributes
        self.addParameter(
            QgsProcessingParameterString(
                self.METRIC_NAME,
                self.tr("Metric name (base attribute name)"),
                defaultValue="metric",
            )
        )

        # Whether higher attribute values should increase the score
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.HIGHER_VALUES_INCREASE_SCORE,
                self.tr("Higher values increase the score"),
                defaultValue=True,
            )
        )

        # Output grid layer
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr("Grid Priority (Lines) Output"),
            )
        )

    def flags(self):
        return super().flags() | QgsProcessingAlgorithm.FlagNoThreading

    def processAlgorithm(self, parameters, context, feedback):
        grid_source = self.parameterAsSource(parameters, self.INPUT_GRID, context)
        if grid_source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT_GRID)
            )

        line_source = self.parameterAsSource(parameters, self.INPUT_LINES, context)
        if line_source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT_LINES)
            )

        field_name = self.parameterAsString(parameters, self.LINE_FIELD, context)
        if not field_name:
            raise QgsProcessingException(
                self.tr("Line attribute field must be specified.")
            )

        metric_name = self.parameterAsString(parameters, self.METRIC_NAME, context).strip()
        if not metric_name:
            raise QgsProcessingException(
                self.tr("Metric name (base attribute name) must not be empty.")
            )

        higher_values_increase_score = self.parameterAsBoolean(
            parameters, self.HIGHER_VALUES_INCREASE_SCORE, context
        )

        min_field_name = f"min_{metric_name}"
        max_field_name = f"max_{metric_name}"
        score_field_name = f"{metric_name}_score"

        # Build spatial index for lines and store features by id
        feedback.pushInfo(self.tr("Building spatial index for line layer..."))
        line_index = QgsSpatialIndex()
        lines_by_id = {}

        for line_feat in line_source.getFeatures():
            if feedback.isCanceled():
                raise QgsProcessingException(self.tr("Calculation cancelled by user"))
            if not line_feat.hasGeometry():
                continue
            line_index.addFeature(line_feat)
            lines_by_id[line_feat.id()] = QgsFeature(line_feat)

        # First pass: compute per-cell min/max and max-based aggregate
        feedback.pushInfo(self.tr("Aggregating attribute values per grid cell..."))
        min_by_fid = {}
        max_by_fid = {}
        agg_by_fid = {}
        agg_values = []

        for grid_feat in grid_source.getFeatures():
            if feedback.isCanceled():
                raise QgsProcessingException(self.tr("Calculation cancelled by user"))

            geom = grid_feat.geometry()
            if not geom or geom.isEmpty():
                cell_min = 0.0
                cell_max = 0.0
            else:
                bbox = geom.boundingBox()
                candidate_ids = line_index.intersects(bbox)
                cell_min = None
                cell_max = None

                for lid in candidate_ids:
                    line_feat = lines_by_id.get(lid)
                    if not line_feat:
                        continue
                    line_geom = line_feat.geometry()
                    if not line_geom or line_geom.isEmpty():
                        continue
                    if not geom.intersects(line_geom):
                        continue

                    val = line_feat[field_name]
                    if val is None:
                        continue
                    try:
                        num_val = float(val)
                    except (TypeError, ValueError):
                        continue

                    if cell_min is None or num_val < cell_min:
                        cell_min = num_val
                    if cell_max is None or num_val > cell_max:
                        cell_max = num_val

                if cell_min is None:
                    cell_min = 0.0
                if cell_max is None:
                    cell_max = 0.0

            fid = grid_feat.id()
            min_by_fid[fid] = cell_min
            max_by_fid[fid] = cell_max

            # Use max value as the aggregate driving the score
            agg_val = cell_max
            agg_by_fid[fid] = agg_val
            agg_values.append(agg_val)

        # Compute scaling range from non-zero aggregates
        positive_values = [v for v in agg_values if v > 0]
        if positive_values:
            min_val = min(positive_values)
            max_val = max(positive_values)
        else:
            min_val = 0.0
            max_val = 0.0

        feedback.pushInfo(
            self.tr("Aggregate values range from {min_val} to {max_val}.").format(
                min_val=min_val, max_val=max_val
            )
        )

        # Prepare output fields
        orig_fields = grid_source.fields()
        fields = QgsFields(orig_fields)

        min_idx = fields.indexOf(min_field_name)
        if min_idx == -1:
            fields.append(QgsField(min_field_name, QVariant.Double))
            min_idx = fields.indexOf(min_field_name)

        max_idx = fields.indexOf(max_field_name)
        if max_idx == -1:
            fields.append(QgsField(max_field_name, QVariant.Double))
            max_idx = fields.indexOf(max_field_name)

        score_idx = fields.indexOf(score_field_name)
        if score_idx == -1:
            # Scores are 0–100 integers; store them as Int for clarity.
            fields.append(QgsField(score_field_name, QVariant.Int))
            score_idx = fields.indexOf(score_field_name)

        # Create output sink
        sink, dest_id = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            fields,
            grid_source.wkbType(),
            grid_source.sourceCrs(),
        )

        if sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))

        # Second pass: compute scores and write features
        feedback.pushInfo(self.tr("Calculating scores and writing output..."))

        if not positive_values or max_val <= min_val:
            # No positive values, or degenerate range: everything gets 0 or 100
            default_score = 0 if higher_values_increase_score else 100

            for grid_feat in grid_source.getFeatures():
                if feedback.isCanceled():
                    raise QgsProcessingException(self.tr("Calculation cancelled by user"))

                fid = grid_feat.id()
                cell_min = float(min_by_fid.get(fid, 0.0))
                cell_max = float(max_by_fid.get(fid, 0.0))

                # Create a new feature with the full field structure
                out_feat = QgsFeature()
                out_feat.setFields(fields)
                out_feat.setGeometry(grid_feat.geometry())
                out_feat.initAttributes(fields.count())

                # Copy original attributes by index (may be fewer than fields)
                orig_attrs = grid_feat.attributes()
                for i, val in enumerate(orig_attrs):
                    if i < fields.count():
                        out_feat.setAttribute(i, val)

                # Set our metric-specific attributes
                out_feat.setAttribute(min_idx, cell_min)
                out_feat.setAttribute(max_idx, cell_max)
                out_feat.setAttribute(score_idx, int(default_score))
                sink.addFeature(out_feat, QgsFeatureSink.FastInsert)
        else:
            for grid_feat in grid_source.getFeatures():
                if feedback.isCanceled():
                    raise QgsProcessingException(self.tr("Calculation cancelled by user"))

                fid = grid_feat.id()
                cell_min = float(min_by_fid.get(fid, 0.0))
                cell_max = float(max_by_fid.get(fid, 0.0))
                agg_val = float(agg_by_fid.get(fid, 0.0))

                if agg_val <= 0:
                    score = 0 if higher_values_increase_score else 100
                else:
                    if higher_values_increase_score:
                        score = scale_value(agg_val, min_val, max_val, 1, 100)
                    else:
                        score = scale_value(agg_val, min_val, max_val, 100, 0)

                # Create a new feature with the full field structure
                out_feat = QgsFeature()
                out_feat.setFields(fields)
                out_feat.setGeometry(grid_feat.geometry())
                out_feat.initAttributes(fields.count())

                # Copy original attributes by index
                orig_attrs = grid_feat.attributes()
                for i, val in enumerate(orig_attrs):
                    if i < fields.count():
                        out_feat.setAttribute(i, val)

                # Set our metric-specific attributes
                out_feat.setAttribute(min_idx, cell_min)
                out_feat.setAttribute(max_idx, cell_max)
                out_feat.setAttribute(score_idx, int(score))
                sink.addFeature(out_feat, QgsFeatureSink.FastInsert)

        feedback.pushInfo(self.tr("Grid priority calculation from line attributes complete."))
        return {self.OUTPUT: dest_id}


class GridPriorityFromLineLengthAlgorithm(QgsProcessingAlgorithm):
    """Grid Priority (Length of Intersecting Lines).

    For each polygon cell in the input grid layer, this algorithm
    computes the total length of all line features which intersect the
    cell and writes two attributes:

    - ``<metric_name>_<unit>`` – total line length within the cell, where
      ``<unit>`` reflects the line layer's CRS units (e.g. ``_m`` or
      ``_km``)
    - ``<metric_name>_score`` – 0–100 score derived from those lengths

    By default, greater total length results in a higher score, but this
    can be inverted via a boolean parameter. Lengths are measured in the
    units of the line layer's CRS.
    """

    # Parameter names
    INPUT_GRID = "INPUT_GRID"
    INPUT_LINES = "INPUT_LINES"
    METRIC_NAME = "METRIC_NAME"
    MORE_LENGTH_INCREASES_SCORE = "MORE_LENGTH_INCREASES_SCORE"
    OUTPUT = "OUTPUT"

    def tr(self, string):
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return GridPriorityFromLineLengthAlgorithm()

    def name(self):
        return "grid_priority_from_line_length"

    def displayName(self):
        return self.tr("Grid Priority (Length of Intersecting Lines)")

    def group(self):
        return ""

    def groupId(self):
        return ""

    def shortHelpString(self):
        return self.tr(
            "Prioritizes grid cells based on the total length of intersecting "
            "line features, such as missing sidewalk segments or specific "
            "street types. Lengths are summed per cell and converted to a "
            "0–100 score."
        )

    def initAlgorithm(self, config=None):
        # Input grid layer (polygons)
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_GRID,
                self.tr("Grid Layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # Input line layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_LINES,
                self.tr("Line Layer"),
                [QgsProcessing.TypeVectorLine],
            )
        )

        # Metric name used as base for output attributes
        self.addParameter(
            QgsProcessingParameterString(
                self.METRIC_NAME,
                self.tr("Metric name (base attribute name)"),
                defaultValue="metric",
            )
        )

        # Whether more total length should result in a higher score
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.MORE_LENGTH_INCREASES_SCORE,
                self.tr("More length increases the score"),
                defaultValue=True,
            )
        )

        # Output grid layer
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr("Grid Priority (Lines) Output"),
            )
        )

    def flags(self):
        return super().flags() | QgsProcessingAlgorithm.FlagNoThreading

    def processAlgorithm(self, parameters, context, feedback):
        grid_source = self.parameterAsSource(parameters, self.INPUT_GRID, context)
        if grid_source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT_GRID)
            )

        line_source = self.parameterAsSource(parameters, self.INPUT_LINES, context)
        if line_source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT_LINES)
            )

        metric_name = self.parameterAsString(parameters, self.METRIC_NAME, context).strip()
        if not metric_name:
            raise QgsProcessingException(
                self.tr("Metric name (base attribute name) must not be empty.")
            )

        more_length_increases_score = self.parameterAsBoolean(
            parameters, self.MORE_LENGTH_INCREASES_SCORE, context
        )

        # Choose a unit-specific suffix based on the line layer's CRS units,
        # so the attribute name clearly indicates the measurement units.
        line_crs = line_source.sourceCrs()
        units = line_crs.mapUnits()
        if units == QgsUnitTypes.DistanceMeters:
            unit_suffix = "m"
        elif units == QgsUnitTypes.DistanceKilometers:
            unit_suffix = "km"
        elif units == QgsUnitTypes.DistanceFeet:
            unit_suffix = "ft"
        elif units == QgsUnitTypes.DistanceMiles:
            unit_suffix = "mi"
        else:
            # Fallback for degrees/unknown; keep generic but explicit.
            unit_suffix = "length"

        length_field_name = f"{metric_name}_{unit_suffix}"
        score_field_name = f"{metric_name}_score"

        # Build spatial index for lines and store features by id
        feedback.pushInfo(self.tr("Building spatial index for line layer..."))
        line_index = QgsSpatialIndex()
        lines_by_id = {}

        for line_feat in line_source.getFeatures():
            if feedback.isCanceled():
                raise QgsProcessingException(self.tr("Calculation cancelled by user"))
            if not line_feat.hasGeometry():
                continue
            line_index.addFeature(line_feat)
            lines_by_id[line_feat.id()] = QgsFeature(line_feat)

        # First pass: compute per-cell total length
        feedback.pushInfo(self.tr("Summing line lengths per grid cell..."))
        length_by_fid = {}
        length_values = []

        for grid_feat in grid_source.getFeatures():
            if feedback.isCanceled():
                raise QgsProcessingException(self.tr("Calculation cancelled by user"))

            geom = grid_feat.geometry()
            if not geom or geom.isEmpty():
                total_length = 0.0
            else:
                bbox = geom.boundingBox()
                candidate_ids = line_index.intersects(bbox)
                total_length = 0.0

                for lid in candidate_ids:
                    line_feat = lines_by_id.get(lid)
                    if not line_feat:
                        continue
                    line_geom = line_feat.geometry()
                    if not line_geom or line_geom.isEmpty():
                        continue
                    if not geom.intersects(line_geom):
                        continue

                    # For simplicity, count the length of the portion which
                    # intersects the grid cell. This may double-count
                    # segments which cross multiple cells, but keeps the
                    # behaviour predictable and generic.
                    inter_geom = line_geom.intersection(geom)
                    if inter_geom and not inter_geom.isEmpty():
                        total_length += inter_geom.length()

            fid = grid_feat.id()
            length_by_fid[fid] = total_length
            length_values.append(total_length)

        positive_values = [v for v in length_values if v > 0]
        if positive_values:
            min_len = min(positive_values)
            max_len = max(positive_values)
        else:
            min_len = 0.0
            max_len = 0.0

        feedback.pushInfo(
            self.tr("Total lengths range from {min_len} to {max_len}.").format(
                min_len=min_len, max_len=max_len
            )
        )

        # Prepare output fields
        orig_fields = grid_source.fields()
        fields = QgsFields(orig_fields)

        length_idx = fields.indexOf(length_field_name)
        if length_idx == -1:
            fields.append(QgsField(length_field_name, QVariant.Double))
            length_idx = fields.indexOf(length_field_name)

        score_idx = fields.indexOf(score_field_name)
        if score_idx == -1:
            # Scores are 0–100 integers; store them as Int for clarity.
            fields.append(QgsField(score_field_name, QVariant.Int))
            score_idx = fields.indexOf(score_field_name)

        # Create output sink
        sink, dest_id = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            fields,
            grid_source.wkbType(),
            grid_source.sourceCrs(),
        )

        if sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))

        # Second pass: compute scores and write features
        feedback.pushInfo(self.tr("Calculating scores and writing output..."))

        if not positive_values or max_len <= min_len:
            # No positive lengths, or degenerate range
            default_score = 0 if more_length_increases_score else 100

            for grid_feat in grid_source.getFeatures():
                if feedback.isCanceled():
                    raise QgsProcessingException(self.tr("Calculation cancelled by user"))

                fid = grid_feat.id()
                total_length = float(length_by_fid.get(fid, 0.0))

                # Create a new feature with the full field structure
                out_feat = QgsFeature()
                out_feat.setFields(fields)
                out_feat.setGeometry(grid_feat.geometry())
                out_feat.initAttributes(fields.count())

                # Copy original attributes by index (may be fewer than fields)
                orig_attrs = grid_feat.attributes()
                for i, val in enumerate(orig_attrs):
                    if i < fields.count():
                        out_feat.setAttribute(i, val)

                # Set our metric-specific attributes
                out_feat.setAttribute(length_idx, total_length)
                out_feat.setAttribute(score_idx, int(default_score))
                sink.addFeature(out_feat, QgsFeatureSink.FastInsert)
        else:
            for grid_feat in grid_source.getFeatures():
                if feedback.isCanceled():
                    raise QgsProcessingException(self.tr("Calculation cancelled by user"))

                fid = grid_feat.id()
                total_length = float(length_by_fid.get(fid, 0.0))

                if total_length <= 0:
                    score = 0 if more_length_increases_score else 100
                else:
                    if more_length_increases_score:
                        score = scale_value(total_length, min_len, max_len, 1, 100)
                    else:
                        score = scale_value(total_length, min_len, max_len, 100, 0)

                # Create a new feature with the full field structure
                out_feat = QgsFeature()
                out_feat.setFields(fields)
                out_feat.setGeometry(grid_feat.geometry())
                out_feat.initAttributes(fields.count())

                # Copy original attributes by index
                orig_attrs = grid_feat.attributes()
                for i, val in enumerate(orig_attrs):
                    if i < fields.count():
                        out_feat.setAttribute(i, val)

                # Set our metric-specific attributes
                out_feat.setAttribute(length_idx, total_length)
                out_feat.setAttribute(score_idx, int(score))
                sink.addFeature(out_feat, QgsFeatureSink.FastInsert)

        feedback.pushInfo(self.tr("Grid priority calculation from line length complete."))
        return {self.OUTPUT: dest_id}
