# -*- coding: utf-8 -*-
"""Processing algorithm to derive a missing sidewalk map from road and
sidewalk layers.

The algorithm:

1. Builds a cleaned sidewalk "coverage" polygon by buffering sidewalk
   features.
2. Computes road segments without sidewalks via a Difference
   (roads minus sidewalk coverage).
3. Snaps and de-duplicates the resulting geometries.

All distance parameters are interpreted in layer units. This algorithm
expects that the project and input layers use a projected CRS with
meters as map units (for example, a local UTM zone). It will check for
this and raise an error if the project CRS is geographic or not in
meters.
"""

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsApplication,
    QgsFeatureRequest,
    QgsFeatureSink,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterNumber,
    QgsProject,
    QgsUnitTypes,
    QgsWkbTypes,
)
from qgis import processing


class CreateMissingSidewalkMapAlgorithm(QgsProcessingAlgorithm):
    """Create Missing Sidewalk Map.

    This algorithm takes a road network layer and a sidewalk network
    layer and returns a layer representing road segments without
    sidewalks. Sidewalks are converted to a buffered coverage polygon
    which is then subtracted from the roads. The remaining road
    geometries are snapped and de-duplicated for cleaner results.
    """

    # Parameter names
    INPUT_ROADS = "INPUT_ROADS"
    INPUT_SIDEWALKS = "INPUT_SIDEWALKS"
    SIDEWALK_BUFFER_METERS = "SIDEWALK_BUFFER_METERS"
    SIMPLIFY_TOLERANCE_METERS = "SIMPLIFY_TOLERANCE_METERS"
    SNAP_TOLERANCE_METERS = "SNAP_TOLERANCE_METERS"
    OUTPUT = "OUTPUT"

    def tr(self, string: str) -> str:
        """Returns a translatable string with the self.tr() function."""

        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        """Required by QGIS processing framework."""

        return CreateMissingSidewalkMapAlgorithm()

    def name(self) -> str:
        """Algorithm id used internally."""

        return "create_missing_sidewalk_map"

    def displayName(self) -> str:
        """Human‑readable name shown in the toolbox."""

        return self.tr("Create Missing Sidewalk Map")

    def group(self) -> str:
        return ""

    def groupId(self) -> str:
        return ""

    def shortHelpString(self) -> str:
        return self.tr(
            "Creates a layer of road segments which are not covered by the "
            "sidewalk network."
            ""
            "The algorithm buffers the sidewalk layer into a coverage "
            "polygon, simplifies the result, and subtracts it from the road "
            "network to derive candidate road segments without sidewalks. "
            "The output is then snapped and duplicate geometries are "
            "removed to clean up small gaps and overlaps."
            ""
            "All distances are interpreted in layer units; the project is "
            "expected to use a projected CRS in meters"
            "(for example, a local UTM zone)."
        )

    def initAlgorithm(self, config=None) -> None:
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ROADS,
                self.tr("Road layer"),
                [
                    QgsProcessing.TypeVectorLine,
                    QgsProcessing.TypeVectorPolygon,
                ],
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_SIDEWALKS,
                self.tr("Sidewalk layer"),
                [
                    QgsProcessing.TypeVectorLine,
                    QgsProcessing.TypeVectorPolygon,
                ],
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.SIDEWALK_BUFFER_METERS,
                self.tr("Sidewalk buffer distance (m)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=15.0,
                minValue=0.0,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.SIMPLIFY_TOLERANCE_METERS,
                self.tr("Simplify tolerance (m)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.5,
                minValue=0.0,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.SNAP_TOLERANCE_METERS,
                self.tr("Snap tolerance (m)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.5,
                minValue=0.0,
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr("Missing sidewalk segments"),
            )
        )

    def flags(self):
        return super().flags() | QgsProcessingAlgorithm.FlagNoThreading

    def _ensure_metre_crs(self, feedback) -> None:
        """Check that the project CRS uses meters as map units.

        Raises a processing exception if the project CRS is geographic or
        not in meters. This mirrors the expectation that the project is
        set to a local UTM (or similar) CRS so that distance parameters
        given in meters behave as expected.
        """

        project = QgsProject.instance()
        crs = project.crs()

        if not crs.isValid():
            raise QgsProcessingException(
                self.tr(
                    "Project CRS is not set or invalid. Please set the project "
                    "CRS to a projected CRS in meters (e.g. a local UTM zone) "
                    "before running this algorithm."
                )
            )

        if crs.isGeographic():
            raise QgsProcessingException(
                self.tr(
                    "Project CRS is geographic (degrees). Please set a "
                    "projected CRS in meters (e.g. a local UTM zone) before "
                    "running this algorithm."
                )
            )

        if crs.mapUnits() != QgsUnitTypes.DistanceMeters:
            raise QgsProcessingException(
                self.tr(
                    "Project CRS does not use meters as map units. Please set "
                    "the project CRS to a projected CRS in meters (e.g. a "
                    "local UTM zone) before running this algorithm."
                )
            )

        feedback.pushInfo(
            self.tr(
                "Project CRS '{authid}' uses meters as map units; proceeding "
                "with metre-based distances."
            ).format(authid=crs.authid())
        )

    def _reproject_if_needed(self, layer, context, feedback):
        """Reproject a layer to the project CRS if needed.

        Assumes the project CRS has already been validated by
        _ensure_metre_crs() to be a projected CRS in meters.
        """

        if not layer or not layer.isValid():
            raise QgsProcessingException(
                self.tr("Input layer is invalid and cannot be reprojected.")
            )

        project_crs = QgsProject.instance().crs()
        layer_crs = layer.crs()

        # If CRS already matches the project CRS and uses meters, keep as-is.
        if layer_crs == project_crs and project_crs.mapUnits() == QgsUnitTypes.DistanceMeters:
            return layer

        feedback.pushInfo(
            self.tr(
                "Reprojecting layer from {src} to project CRS {dst} for meter-based processing..."
            ).format(src=layer_crs.authid(), dst=project_crs.authid())
        )

        params = {
            "INPUT": layer,
            "TARGET_CRS": project_crs,
            "OUTPUT": "memory:",
        }
        result = processing.run(
            "native:reprojectlayer",
            params,
            context=context,
            feedback=feedback,
        )
        return result["OUTPUT"]

    def processAlgorithm(self, parameters, context, feedback):
        self._ensure_meter_crs(feedback)

        # Get sources
        road_source = self.parameterAsSource(parameters, self.INPUT_ROADS, context)
        if road_source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT_ROADS)
            )

        sidewalk_source = self.parameterAsSource(
            parameters, self.INPUT_SIDEWALKS, context
        )
        if sidewalk_source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT_SIDEWALKS)
            )

        buffer_distance = self.parameterAsDouble(
            parameters, self.SIDEWALK_BUFFER_METERS, context
        )
        simplify_tolerance = self.parameterAsDouble(
            parameters, self.SIMPLIFY_TOLERANCE_METERS, context
        )
        snap_tolerance = self.parameterAsDouble(
            parameters, self.SNAP_TOLERANCE_METERS, context
        )

        # Materialize sources into temporary vector layers
        road_layer = road_source.materialize(QgsFeatureRequest(), feedback)
        if road_layer is None:
            raise QgsProcessingException(
                self.tr("Could not materialize road layer")
            )

        sidewalk_layer = sidewalk_source.materialize(QgsFeatureRequest(), feedback)
        if sidewalk_layer is None:
            raise QgsProcessingException(
                self.tr("Could not materialize sidewalk layer")
            )

        # Reproject layers into the project CRS (with should support meters) if required so all
        # distance-based operations behave consistently.
        road_layer = self._reproject_if_needed(road_layer, context, feedback)
        sidewalk_layer = self._reproject_if_needed(sidewalk_layer, context, feedback)

        feedback.pushInfo(
            self.tr(
                "Input roads: {roads} features, CRS {roads_crs}; sidewalks: "
                "{sw} features, CRS {sw_crs}."
            ).format(
                roads=road_layer.featureCount(),
                roads_crs=road_layer.crs().authid(),
                sw=sidewalk_layer.featureCount(),
                sw_crs=sidewalk_layer.crs().authid(),
            )
        )

        # Step 1: Create sidewalk coverage polygon
        feedback.pushInfo(self.tr("Building sidewalk coverage polygon..."))

        # If sidewalks are already polygons, we can simplify directly.
        is_sidewalk_polygon = sidewalk_layer.geometryType() == QgsWkbTypes.PolygonGeometry

        if is_sidewalk_polygon:
            coverage_input = sidewalk_layer
        else:
            # Buffer sidewalks (lines) into polygons
            buffer_params = {
                "INPUT": sidewalk_layer,
                "DISTANCE": buffer_distance,
                "SEGMENTS": 5,
                "END_CAP_STYLE": 0,  # Round
                "JOIN_STYLE": 0,  # Round
                "MITER_LIMIT": 2.0,
                "DISSOLVE": True,
                "OUTPUT": "memory:",
            }
            buffer_result = processing.run(
                "native:buffer",
                buffer_params,
                context=context,
                feedback=feedback,
            )
            coverage_input = buffer_result["OUTPUT"]

        # Simplify / clean the coverage polygon
        simplify_params = {
            "INPUT": coverage_input,
            "METHOD": 0,  # Distance tolerance
            "TOLERANCE": simplify_tolerance,
            "OUTPUT": "memory:",
        }
        simplify_result = processing.run(
            "native:simplifygeometries",
            simplify_params,
            context=context,
            feedback=feedback,
        )
        sidewalk_coverage = simplify_result["OUTPUT"]

        # Optionally run GRASS v.clean (rmarea) to remove slivers / fix
        # topology if the provider is available. Any failure here is
        # silently ignored so the algorithm still runs without GRASS.
        # We also first check that the grass7 provider and v.clean
        # algorithm are actually available to avoid noisy errors when
        # GRASS is not installed.
        try:
            registry = QgsApplication.processingRegistry()
            alg = registry.algorithmById("grass7:v.clean")
            if not alg:
                raise RuntimeError("GRASS v.clean algorithm not available")

            vclean_params = {
                "input": sidewalk_coverage,
                # In the QGIS/GRASS wrapper, tools are selected by numeric
                # index. The index 7 corresponds to rmarea in current
                # GRASS v.clean wrappers.
                "tool": [7],
                # Use simplify_tolerance as a rough threshold; if GRASS
                # is not available or the parameters are incompatible,
                # this whole block is skipped.
                "threshold": [max(simplify_tolerance, 0.0)],
                "output": "memory:",
                "-b": False,
                "-c": False,
                "-d": False,
                "-e": False,
                "-f": False,
                "-g": False,
                "-r": False,
                "-t": False,
                "GRASS_REGION_PARAMETER": None,
                "GRASS_SNAP_TOLERANCE_PARAMETER": -1,
                "GRASS_MIN_AREA_PARAMETER": 0.0001,
                "GRASS_OUTPUT_TYPE_PARAMETER": 0,
                "GRASS_REGION_CELLSIZE_PARAMETER": 0,
            }
            vclean_result = processing.run(
                "grass7:v.clean",
                vclean_params,
                context=context,
                feedback=feedback,
            )
            sidewalk_coverage = vclean_result.get("output", sidewalk_coverage)
        except Exception:
            # GRASS not available or v.clean call failed – fall back to
            # the simplified coverage without bothering the user.
            pass

        feedback.pushInfo(
            self.tr(
                "Sidewalk coverage layer has {n} feature(s)."
            ).format(n=sidewalk_coverage.featureCount())
        )

        # Convert polygon roads to lines so the output is always a line
        # network, regardless of whether the input roads are stored as
        # polygons or centreline geometries.
        if road_layer.geometryType() == QgsWkbTypes.PolygonGeometry:
            poly_to_lines_params = {
                "INPUT": road_layer,
                "OUTPUT": "memory:",
            }
            poly_to_lines_result = processing.run(
                "native:polygonstolines",
                poly_to_lines_params,
                context=context,
                feedback=feedback,
            )
            road_lines = poly_to_lines_result["OUTPUT"]
        else:
            road_lines = road_layer

        # Step 2: Roads without sidewalks via Difference
        feedback.pushInfo(self.tr("Computing road segments without sidewalks (Difference)..."))

        difference_params = {
            "INPUT": road_lines,
            "OVERLAY": sidewalk_coverage,
            "OUTPUT": "memory:",
        }
        difference_result = processing.run(
            "native:difference",
            difference_params,
            context=context,
            feedback=feedback,
        )
        roads_without_sidewalks = difference_result["OUTPUT"]

        feedback.pushInfo(
            self.tr(
                "Initial missing-sidewalk segments: {n} feature(s)."
            ).format(n=roads_without_sidewalks.featureCount())
        )

        # Step 3: Snap and clean the result
        feedback.pushInfo(self.tr("Snapping geometries and removing duplicates..."))

        # Snap geometries to the (line) road network to close tiny gaps
        snap_params = {
            "INPUT": roads_without_sidewalks,
            "REFERENCE_LAYER": road_lines,
            "TOLERANCE": snap_tolerance,
            "BEHAVIOR": 0,  # Prefer closest vertex/segment (default behaviour)
            "OUTPUT": "memory:",
        }
        snap_result = processing.run(
            "native:snapgeometries",
            snap_params,
            context=context,
            feedback=feedback,
        )
        snapped = snap_result["OUTPUT"]

        # Delete duplicate geometries which can appear after Difference/Snap
        dedupe_params = {
            "INPUT": snapped,
            "OUTPUT": "memory:",
        }
        dedupe_result = processing.run(
            "native:deleteduplicategeometries",
            dedupe_params,
            context=context,
            feedback=feedback,
        )
        cleaned = dedupe_result["OUTPUT"]

        feedback.pushInfo(
            self.tr(
                "Cleaned missing-sidewalk layer has {n} feature(s)."
            ).format(n=cleaned.featureCount())
        )

        # Create output sink and copy features
        sink, dest_id = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            cleaned.fields(),
            cleaned.wkbType(),
            cleaned.crs(),
        )

        if sink is None:
            raise QgsProcessingException(
                self.invalidSinkError(parameters, self.OUTPUT)
            )

        for feat in cleaned.getFeatures():
            if feedback.isCanceled():
                raise QgsProcessingException(
                    self.tr("Calculation cancelled by user")
                )
            sink.addFeature(feat, QgsFeatureSink.FastInsert)

        feedback.pushInfo(
            self.tr("Missing sidewalk map creation complete.")
        )

        return {self.OUTPUT: dest_id}
