# -*- coding: utf-8 -*-
"""Processing algorithm to create a hex grid covering a boundary layer."""

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsFeature,
    QgsFeatureRequest,
    QgsFields,
    QgsField,
    QgsGeometry,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterNumber,
    QgsFeatureSink,
    QgsWkbTypes,
    QgsCoordinateReferenceSystem,
)

from ..utils.geometry import create_hex_grid


class CreateHexGridAlgorithm(QgsProcessingAlgorithm):
    """Create Hex Grid Covering Boundary.

    This algorithm generates a hexagonal grid covering the extent of an
    input boundary layer.
    """

    INPUT_BOUNDARY = "INPUT_BOUNDARY"
    HEX_SIDE_KM = "HEX_SIDE_KM"
    OUTPUT = "OUTPUT"

    def tr(self, string: str) -> str:
        return QCoreApplication.translate("Processing", string)

    def name(self) -> str:
        return "create_hex_grid_covering_boundary"

    def displayName(self) -> str:
        return self.tr("Create Hex Grid Covering Boundary")

    def group(self) -> str:
        return ""

    def groupId(self) -> str:
        return ""

    def shortHelpString(self) -> str:
        return self.tr(
            "Creates a hexagonal grid which covers the extent of a boundary "
            "layer. The side length of the hexes is specified in kilometres "
            "and defaults to 0.05 km (about 50 metres). This uses the same "
            "grid generation logic as the Walk Potential algorithm."
        )

    def createInstance(self):
        return CreateHexGridAlgorithm()

    def initAlgorithm(self, config=None) -> None:
        # Boundary layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_BOUNDARY,
                self.tr("Boundary Layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # Hex side length
        self.addParameter(
            QgsProcessingParameterNumber(
                self.HEX_SIDE_KM,
                self.tr("Hex side length (km)"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.05,
                minValue=0.0001,
            )
        )

        # Output grid
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr("Hex Grid"),
            )
        )

    def flags(self):
        return super().flags() | QgsProcessingAlgorithm.FlagNoThreading

    def processAlgorithm(self, parameters, context, feedback):
        boundary_source = self.parameterAsSource(
            parameters, self.INPUT_BOUNDARY, context
        )
        if boundary_source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT_BOUNDARY)
            )

        hex_side_km = self.parameterAsDouble(
            parameters, self.HEX_SIDE_KM, context
        )

        # Materialize boundary layer
        boundary_layer = boundary_source.materialize(
            QgsFeatureRequest(), feedback
        )
        if boundary_layer is None:
            raise QgsProcessingException("Could not materialize boundary layer")

        # Combine boundary geometries
        boundary_geom = None
        for feat in boundary_layer.getFeatures():
            geom = feat.geometry()
            if not geom or geom.isEmpty():
                continue
            boundary_geom = geom if boundary_geom is None else boundary_geom.combine(geom)

        if not boundary_geom or boundary_geom.isEmpty():
            raise QgsProcessingException(
                "Boundary layer contains no valid geometries"
            )

        feedback.pushInfo(
            self.tr("Generating hex grid with side length {side_km} km...").format(
                side_km=hex_side_km
            )
        )

        hex_features = create_hex_grid(boundary_geom, hex_side_km)

        if not hex_features:
            feedback.pushInfo(
                self.tr("Hex grid generation produced no cells (boundary may be too small).")
            )

        # Prepare output fields: simple id field
        fields = QgsFields()
        fields.append(QgsField("id", QVariant.Int))

        # Determine geometry type and CRS from the hex features or fallback
        if hex_features:
            wkb_type = hex_features[0].geometry().wkbType()
            crs = QgsCoordinateReferenceSystem("EPSG:4326")
        else:
            wkb_type = QgsWkbTypes.Polygon
            crs = boundary_layer.sourceCrs() or QgsCoordinateReferenceSystem("EPSG:4326")

        sink, dest_id = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            fields,
            wkb_type,
            crs,
        )

        if sink is None:
            raise QgsProcessingException(
                self.invalidSinkError(parameters, self.OUTPUT)
            )

        for idx, feat in enumerate(hex_features):
            if feedback.isCanceled():
                raise QgsProcessingException("Calculation cancelled by user")

            # Create a new feature with the full field structure
            out_feat = QgsFeature()
            out_feat.setFields(fields)
            out_feat.setGeometry(feat.geometry())
            out_feat.initAttributes(fields.count())

            # Set id attribute (index 0)
            out_feat.setAttribute(0, idx)
            sink.addFeature(out_feat, QgsFeatureSink.FastInsert)

        feedback.pushInfo(
            self.tr("Created {count} hex cells.").format(count=len(hex_features))
        )

        return {self.OUTPUT: dest_id}
