# -*- coding: utf-8 -*-
"""
Module for querying OpenStreetMap data via Overpass API
"""

import json
import threading
from typing import List, Dict, Tuple
from qgis.PyQt.QtNetwork import QNetworkAccessManager, QNetworkRequest
from qgis.PyQt.QtCore import QUrl, QEventLoop, QByteArray


class OverpassQueryError(Exception):
    """Exception raised for Overpass API query errors"""
    pass


class OverpassQuery:
    """Class to handle Overpass API queries"""
    
    # Thread-local storage for network managers
    _thread_local = threading.local()
    
    def __init__(self, endpoint='https://overpass-api.de/api/interpreter'):
        self.endpoint = endpoint
    
    @classmethod
    def _get_network_manager(cls):
        """Get or create a network manager for the current thread"""
        if not hasattr(cls._thread_local, 'network_manager'):
            cls._thread_local.network_manager = QNetworkAccessManager()
        return cls._thread_local.network_manager
    
    def bbox_query_to_geojson(self, query: str, bbox: Tuple[float, float, float, float], 
                               progress_callback=None) -> List[Dict]:
        """
        Query Overpass API with a bounding box and return GeoJSON features
        
        :param query: Overpass query string (e.g., "nwr[amenity=bank];nwr[amenity=atm];")
        :param bbox: Tuple of (minlon, minlat, maxlon, maxlat)
        :param progress_callback: Optional callback function for progress updates
        :return: List of GeoJSON feature dictionaries
        """
        minlon, minlat, maxlon, maxlat = bbox
        
        # Build the Overpass QL query
        # Using the bbox format: (south, west, north, east)
        bbox_str = f"{minlat},{minlon},{maxlat},{maxlon}"
        
        # Parse multiple query parts separated by semicolons
        query_parts = [q.strip() for q in query.split(';') if q.strip()]
        
        # Build complete Overpass QL query
        overpass_query = "[out:json][timeout:180];\n(\n"
        for part in query_parts:
            overpass_query += f"  {part}({bbox_str});\n"
        overpass_query += ");\nout geom;"
        
        if progress_callback:
            progress_callback(f"Querying Overpass API...")
        
        try:
            network_manager = self._get_network_manager()
            
            # Create request
            qurl = QUrl(self.endpoint)
            request = QNetworkRequest(qurl)
            request.setHeader(request.UserAgentHeader, 'Walk Potential QGIS Plugin')
            request.setHeader(request.ContentTypeHeader, 'application/x-www-form-urlencoded')
            
            # Prepare POST data
            post_data = QByteArray(f"data={overpass_query}".encode('utf-8'))
            
            # Perform synchronous request using event loop
            reply = network_manager.post(request, post_data)
            event_loop = QEventLoop()
            reply.finished.connect(event_loop.quit)
            event_loop.exec_()
            
            status_code = reply.attribute(QNetworkRequest.HttpStatusCodeAttribute)
            
            if status_code and 200 <= status_code < 300:
                # Success
                data_bytes = reply.readAll()
                data_str = bytes(data_bytes).decode('utf-8')
                data = json.loads(data_str)
                features = self._convert_to_geojson_features(data.get('elements', []))
                
                if progress_callback:
                    progress_callback(f"Found {len(features)} features")
                
                return features
            else:
                error = reply.errorString()
                raise OverpassQueryError(f"Overpass API query failed with status {status_code}: {error}")
            
        except json.JSONDecodeError as e:
            raise OverpassQueryError(f"Failed to parse Overpass API response: {str(e)}")
        except Exception as e:
            raise OverpassQueryError(f"Overpass API query failed: {str(e)}")
    
    def _convert_to_geojson_features(self, elements: List[Dict]) -> List[Dict]:
        """
        Convert Overpass API elements to GeoJSON features
        
        :param elements: List of Overpass API element objects
        :return: List of GeoJSON feature dictionaries
        """
        features = []
        
        for element in elements:
            feature = self._element_to_geojson(element)
            if feature:
                features.append(feature)
        
        return features
    
    def _element_to_geojson(self, element: Dict) -> Dict:
        """
        Convert a single Overpass element to a GeoJSON feature
        
        :param element: Overpass API element object
        :return: GeoJSON feature dictionary or None if conversion fails
        """
        elem_type = element.get('type')
        properties = element.get('tags', {})
        properties['osm_id'] = element.get('id')
        properties['osm_type'] = elem_type
        
        try:
            if elem_type == 'node':
                return {
                    'type': 'Feature',
                    'geometry': {
                        'type': 'Point',
                        'coordinates': [element['lon'], element['lat']]
                    },
                    'properties': properties
                }
            
            elif elem_type == 'way':
                if 'geometry' in element:
                    coords = [[node['lon'], node['lat']] for node in element['geometry']]
                    
                    # Check if it's a closed way (polygon)
                    is_closed = (coords[0] == coords[-1]) if len(coords) > 1 else False
                    
                    # Determine if it should be a Polygon or LineString
                    # If it has area tags and is closed, make it a polygon
                    area_tags = {'building', 'landuse', 'leisure', 'amenity', 'natural'}
                    has_area_tag = any(tag in properties for tag in area_tags)
                    
                    if is_closed and has_area_tag:
                        geom_type = 'Polygon'
                        coordinates = [coords]
                    else:
                        geom_type = 'LineString'
                        coordinates = coords
                    
                    return {
                        'type': 'Feature',
                        'geometry': {
                            'type': geom_type,
                            'coordinates': coordinates
                        },
                        'properties': properties
                    }
            
            elif elem_type == 'relation':
                # Simplified relation handling - would need more work for complex multipolygons
                if 'members' in element and 'geometry' in element:
                    # Try to construct a geometry from the relation
                    # This is simplified and may not work for all relations
                    return {
                        'type': 'Feature',
                        'geometry': {
                            'type': 'GeometryCollection',
                            'geometries': []
                        },
                        'properties': properties
                    }
        
        except (KeyError, IndexError) as e:
            # Skip elements that can't be converted
            pass
        
        return None
