# -*- coding: utf-8 -*-
"""Processing algorithm to apply the default Sidewalk Priority heatmap style.

This algorithm takes a grid layer (typically a hex grid or other polygon
layer) and a numeric attribute containing 0–100 scores, and applies a
"stop-light" graduated color style using a 30% transparent heatmap:

- Red (0–20) – Very low score
- Orange (20–40) – Low
- Yellow (40–60) – Medium
- Light green (60–80) – High
- Dark green (80–100) – Very high

The algorithm does not modify feature geometry or attributes; it only
updates the layer's renderer in the current QGIS project. It is intended
for use in models where you build custom grid-priority workflows and
want a consistent example heatmap styling applied at the end.
"""

from qgis.PyQt.QtCore import QCoreApplication, Qt
from qgis.PyQt.QtGui import QColor
from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingParameterVectorLayer,
    QgsProcessingParameterField,
    QgsVectorLayer,
    QgsGraduatedSymbolRenderer,
    QgsRendererRange,
    QgsSymbol,
)


class GridPriorityApplyDefaultHeatmapStyleAlgorithm(QgsProcessingAlgorithm):
    """Grid Priority (Apply Default Heatmap Style).

    Applies the Sidewalk Priority Toolkit's default 30% transparent
    stop-light style to a grid layer using a chosen score field. The
    field is expected to contain values in the range 0–100.
    """

    INPUT_GRID = "INPUT_GRID"
    SCORE_FIELD = "SCORE_FIELD"

    def tr(self, string: str) -> str:
        """Returns a translatable string with the self.tr() function."""

        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        """Returns a new instance of the algorithm."""

        return GridPriorityApplyDefaultHeatmapStyleAlgorithm()

    def name(self) -> str:
        """Algorithm id used internally."""

        return "grid_priority_apply_default_heatmap_style"

    def displayName(self) -> str:
        """Human-readable name shown in the toolbox."""

        return self.tr("Grid Priority (Apply Default Heatmap Style)")

    def group(self) -> str:
        return ""

    def groupId(self) -> str:
        return ""

    def shortHelpString(self) -> str:
        return self.tr(
            "Applies a default stop-light heatmap style to a grid layer using "
            "a chosen score field. The field is expected to contain values in "
            "the 0–100 range. "
            "The style uses five classes from 0 to 100, with 30% transparency: "
            "red for the lowest scores, then orange, yellow, light green, and "
            "dark green for the highest scores. This provides an example "
            "heatmap styling suitable for use in models which combine "
            "multiple grid-priority metrics."
        )

    def initAlgorithm(self, config=None) -> None:
        # Input grid layer (typically polygons)
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT_GRID,
                self.tr("Grid Layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # Score field (0–100) to drive the heatmap
        self.addParameter(
            QgsProcessingParameterField(
                self.SCORE_FIELD,
                self.tr("Score field (0–100)"),
                parentLayerParameterName=self.INPUT_GRID,
                type=QgsProcessingParameterField.Numeric,
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        # Get the target layer as a QgsVectorLayer so we can modify its style
        layer = self.parameterAsVectorLayer(parameters, self.INPUT_GRID, context)
        if not isinstance(layer, QgsVectorLayer) or not layer.isValid():
            raise QgsProcessingException(self.tr("Input layer is invalid."))

        field_name = self.parameterAsString(parameters, self.SCORE_FIELD, context)
        if not field_name:
            raise QgsProcessingException(self.tr("Score field must be specified."))

        if layer.fields().indexOf(field_name) == -1:
            raise QgsProcessingException(
                self.tr("Field '{name}' does not exist on the input layer.").format(
                    name=field_name
                )
            )

        feedback.pushInfo(
            self.tr("Applying default heatmap style using field '{name}'.").format(
                name=field_name
            )
        )

        # Define 5 ranges with colors from red (low) to green (high)
        ranges = [
            (0, 20, "#d73027", self.tr("Very Low")),      # Red
            (20, 40, "#fc8d59", self.tr("Low")),          # Orange
            (40, 60, "#fee08b", self.tr("Medium")),       # Yellow
            (60, 80, "#91cf60", self.tr("High")),         # Light green
            (80, 100, "#1a9850", self.tr("Very High")),   # Dark green
        ]

        renderer_ranges: list[QgsRendererRange] = []
        for lower, upper, color, label in ranges:
            symbol = QgsSymbol.defaultSymbol(layer.geometryType())
            symbol.setColor(QColor(color))
            # 0.7 opacity ≈ 30% transparent
            symbol.setOpacity(0.7)

            # Remove outline for cleaner look
            symbol.symbolLayer(0).setStrokeStyle(Qt.NoPen)

            renderer_range = QgsRendererRange(lower, upper, symbol, label)
            renderer_ranges.append(renderer_range)

        renderer = QgsGraduatedSymbolRenderer(field_name, renderer_ranges)
        renderer.setMode(QgsGraduatedSymbolRenderer.Custom)
        layer.setRenderer(renderer)
        layer.triggerRepaint()

        feedback.pushInfo(self.tr("Default heatmap style applied."))

        # This algorithm only changes the style of an existing layer and
        # does not produce a new layer, so we return an empty result dict.
        return {}
