# -*- coding: utf-8 -*-
"""
/***************************************************************************
 shortpath
                                 A QGIS plugin
 Find shortest path between points based on line (network) layer.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-02-21
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Revaz Gviniashvili
        email                : rewaza@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import *
from qgis.PyQt.QtGui import *
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from .resources import *
from .short_path_dialog import shortpathDialog
import os.path
from qgis.core import *
from qgis.analysis import *
from qgis.utils import *
import processing


class shortpath:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'shortpath_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Short Path')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('shortpath', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/short_path/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Short Path'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Short Path'),
                action)
            self.iface.removeToolBarIcon(action)
            
    def drawline_d(self, networkLayer, scoordP, lcoordP, sname, ename, lwd, lcol):
        director = QgsVectorLayerDirector(networkLayer, -1, '', '', '', QgsVectorLayerDirector.DirectionBoth)
        strategy = QgsNetworkDistanceStrategy()
        director.addStrategy(strategy)
        builder = QgsGraphBuilder(networkLayer.sourceCrs())
        startPoint = QgsPointXY(scoordP)
        endPoint = QgsPointXY(lcoordP)
        tiedPoints = director.makeGraph(builder, [startPoint, endPoint])
        tStart, tStop = tiedPoints
        graph = builder.graph()
        idxStart = graph.findVertex(tStart)
        idxEnd = graph.findVertex(tStop)
        (tree, costs) = QgsGraphAnalyzer.dijkstra(graph, idxStart, 0)
        if tree[idxEnd] == -1:
            raise Exception('No route!')
        cost = costs[idxEnd]
        route = [graph.vertex(idxEnd).point()]
        while idxEnd != idxStart:
            idxEnd = graph.edge(tree[idxEnd]).fromVertex()
            route.insert(0, graph.vertex(idxEnd).point())
        n_layer = QgsVectorLayer('LineString', 'shortest path between ' + sname + "-" + ename, 'memory')
        pr = n_layer.dataProvider()
        seg = QgsFeature()
        seg.setGeometry(QgsGeometry.fromPolylineXY(route))
        pr.addFeatures([seg])
        QgsProject.instance().addMapLayers([n_layer])

        fts = n_layer.getFeatures()
        polyPath = QgsVectorLayer('Polygon', 'Buffer', "memory")
        for f in fts:
            buff = f.geometry().buffer(0.00001, 4)
            pr = polyPath.dataProvider()
            b = QgsFeature()
            b.setGeometry(buff)
            pr.addFeatures([b])
        polyPath.updateExtents()
        polyPath.setOpacity(0.2)
        QgsProject.instance().addMapLayers([polyPath])
        var = processing.run("native:clip", {'INPUT': networkLayer,
                                             'OVERLAY': polyPath,
                                             'OUTPUT': 'memory: Shortest path' + " " + sname + "-"+ ename})['OUTPUT']
        QgsProject.instance().addMapLayer(var)
        QgsProject.instance().removeMapLayer(polyPath)
        QgsProject.instance().removeMapLayer(n_layer)

        features = var.getFeatures()
        for f in features:
            var.dataProvider().addAttributes([QgsField("full_length", QVariant.Double, 'double', 10, 2)])
            var.updateFields()
            provider = var.dataProvider()
            updateMap = {}
            fieldIdx = provider.fields().indexFromName("full_length")
            features = provider.getFeatures()
            for feature in features:
                updateMap[feature.id()] = {fieldIdx: cost}
            provider.changeAttributeValues(updateMap)

        nfeatures = var.getFeatures()
        for nlen in nfeatures:
            var.dataProvider().addAttributes([QgsField("segment_length", QVariant.Double, 'double', 10, 2)])
            var.updateFields()
            nprovider = var.dataProvider()
            feat_map = {}
            field_index = nprovider.fields().indexFromName("segment_length")
            nfeatures = nprovider.getFeatures()
            for nfeature in nfeatures:
                n_length = nfeature.geometry().length()
                fn_length = n_length * 100000
                feat_map[nfeature.id()] = {field_index: fn_length}
                var.updateFeature(nfeature)
            nprovider.changeAttributeValues(feat_map)
            var.updateFeature(nlen)

        single_symbol_renderer = var.renderer()
        symbol = single_symbol_renderer.symbol()
        symbol.setColor(QColor(lcol)) #rc,gc,bc
        symbol.setWidth(float(lwd)) #float(lwd)
        var.triggerRepaint()
        iface.layerTreeView().refreshLayerSymbology(var.id())
        iface.zoomToActiveLayer()

    def drawstright(self,startP, endP, nameSt, nameEn):
        start_point = startP
        end_point = endP
        v_layer = QgsVectorLayer('LineString', 'logical connection' + ' ' + nameSt + '-' + nameEn, 'memory')
        pr = v_layer.dataProvider()
        seg = QgsFeature()
        seg.setGeometry(QgsGeometry.fromPolyline([start_point, end_point]))
        pr.addFeatures([seg])
        QgsProject.instance().addMapLayers([v_layer])
        iface.zoomToActiveLayer()


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = shortpathDialog()

        l_weight = self.dlg.doubleSpinBox.value()
        l_color = self.dlg.mColorButton.color().name()
        self.dlg.comboBox.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.dlg.comboBox.setShowCrs(True)
        self.dlg.comboBox_2.setFilters(QgsMapLayerProxyModel.PointLayer)
        self.dlg.comboBox_2.setShowCrs(True)
        self.dlg.comboBox_3.setFilters(QgsMapLayerProxyModel.PointLayer)
        self.dlg.comboBox_3.setShowCrs(True)
        self.dlg.show()
        result = self.dlg.exec_()
        if result:
            selectedLayer = self.dlg.comboBox.currentLayer()
            selectedLayer_2 = self.dlg.comboBox_2.currentLayer()
            selectedLayer_3 = self.dlg.comboBox_3.currentLayer()
            mg = [g[1] for g in selectedLayer_2.getFeatures()]
            mh = [h[2] for h in selectedLayer_3.getFeatures()]
            res = [mg.index(i) for i in mh]
            for r,c in enumerate(res):
                sn = [g[1] for g in selectedLayer_2.getFeatures()]
                sx = [g[2] for g in selectedLayer_2.getFeatures()]
                sy = [g[3] for g in selectedLayer_2.getFeatures()]
                scoord = QgsPoint(float(sx[c]), float(sy[c]))
                nlns = str(sn[c])
                ln = [h[1] for h in selectedLayer_3.getFeatures()]
                lx = [h[3] for h in selectedLayer_3.getFeatures()]
                ly = [h[4] for h in selectedLayer_3.getFeatures()]
                lcoord = QgsPoint(float(lx[r]), float(ly[r]))
                nlne = str(ln[r])
                self.drawline_d(selectedLayer, scoord, lcoord, nlns, nlne, l_weight, l_color)
                if self.dlg.checkBox.isChecked():
                    self.drawstright(scoord, lcoord, nlne, nlns)
                    self.iface.messageBar().pushMessage("Logical Connections Added")
                else:
                    self.iface.messageBar().pushMessage("Logical Connections Ignored")
                try:
                    self.iface.messageBar().pushMessage("Short path between " + nlns + " and " + nlne + " added")
                except:
                    self.iface.messageBar().pushMessage("route not found")
            self.iface.messageBar().pushMessage(
                "Success", "Shortest Paths added",
                level=Qgis.Success, duration=3)

