# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Animove
                                 A QGIS plugin
 AniMove for QGIS
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-07-19
        copyright            : (C) 2021 by Matteo Ghetta (Faunalia)
        email                : matteo.ghetta@faunalia.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Matteo Ghetta (Faunalia)'
__date__ = '2021-07-19'
__copyright__ = '(C) 2021 by Matteo Ghetta (Faunalia)'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon

from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterNumber,
                       QgsWkbTypes,
                       QgsFields,
                       QgsField,
                       QgsPointXY,
                       QgsDistanceArea,
                       QgsGeometry,
                       QgsFeature,
                       QgsProcessingException,
                       QgsSpatialIndex,
                       QgsFeatureRequest,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingParameterRange,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink)
import random
import math
import codecs
import os


class AnimoveRandomHRPath(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # INPUTS
    PATHS_LAYER = 'PATHS_LAYER'
    BOUND_LAYER = 'BOUND_LAYER'
    ANGLE_RANGE = 'ANGLE_RANGE'
    ITERATIONS = 'ITERATIONS'
    OVERLAY_LAYER = 'OVERLAY_LAYER'
    POINTS_LAYER = 'POINTS_LAYER'
    KEEP_START_POINTS = 'KEEP_START_POINTS'
    KEEP_ANGLES = 'KEEP_ANGLES'
    RANDOM_PATHS = 'RANDOM_PATHS'
    SUMMARY = 'SUMMARY'

    def initAlgorithm(self, config):
        
        ### INPUTS ###

        # line paths layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                name=self.PATHS_LAYER,
                description=self.tr('Input paths layer'),
                types=[QgsProcessing.TypeVectorLine]
            )
        )

        # study area layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                name=self.BOUND_LAYER,
                description=self.tr('Study area layer'),
                types=[QgsProcessing.TypeVectorPolygon]
            )
        )

        # angle range
        self.addParameter(
            QgsProcessingParameterRange(
                name=self.ANGLE_RANGE,
                description=self.tr("Range for the random angles"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=[0, 360]
            )
        )

        # iterations
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.ITERATIONS,
                description=self.tr("Number of iterations"),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=10,
                minValue=0,
                maxValue=999
            )
        )

        # overlayer line layer
        # self.addParameter(
        #     QgsProcessingParameterFeatureSource(
        #         name=self.OVERLAY_LAYER,
        #         description=self.tr("Overlay layer"),
        #         types=[QgsProcessing.TypeVectorPolygon],
        #         optional=True
        #     )
        # )

        # # boolean start point
        # self.addParameter(
        #     QgsProcessingParameterBoolean(
        #         name=self.KEEP_ANGLES,
        #         description=self.tr("Use start points from original path"),
        #         defaultValue=False
        #     )
        # )

        # boolean between points
        # self.addParameter(
        #     QgsProcessingParameterBoolean(
        #         name=self.KEEP_ANGLES,
        #         description=self.tr("Keep angles between points same as in original path"),
        #         defaultValue=False
        #     )
        # )

        # # points layer
        # self.addParameter(
        #     QgsProcessingParameterFeatureSource(
        #         name=self.POINTS_LAYER,
        #         description=self.tr("Points layer"),
        #         types=[QgsProcessing.TypeVectorPoint],
        #         optional=True
        #     )
        # )


        ### OUTPUTS ###

        # random paths layer
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                name=self.RANDOM_PATHS,
                description=self.tr("Random paths")
            )
        )

        # summary statistics file
        self.addParameter(
            QgsProcessingParameterFileDestination(
                name=self.SUMMARY,
                description=self.tr("Summary"),
                fileFilter=self.tr('HTML files (*.html')
            )
        )



    def processAlgorithm(self, parameters, context, feedback):
        
        pathsLayer = self.parameterAsSource(
            parameters,
            self.PATHS_LAYER,
            context
        )
        
        boundLayer = self.parameterAsSource(
            parameters,
            self.BOUND_LAYER,
            context
        )
        
        overlayLayer = self.parameterAsSource(
            parameters,
            self.OVERLAY_LAYER,
            context
        )
        
        pointsLayer = self.parameterAsSource(
            parameters,
            self.POINTS_LAYER,
            context
        )

        angles = self.parameterAsRange(
            parameters,
            self.ANGLE_RANGE,
            context
        )

        iterations = self.parameterAsInt(
            parameters,
            self.ITERATIONS,
            context
        )

        keepStart = self.parameterAsBool(
            parameters,
            self.KEEP_START_POINTS,
            context
        )

        keepAngles = self.parameterAsBool(
            parameters,
            self.KEEP_ANGLES,
            context
        )

        summaryFile = self.parameterAsFileOutput(
            parameters,
            self.SUMMARY,
            context
        )

        # get the angles ranges as single values
        minAngle = float(angles[0])
        maxAngle = float(angles[1])


        fields = QgsFields()
        fields.append(QgsField("id", QVariant.Int))
        fields.append(QgsField("intersect", QVariant.Int))

        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.RANDOM_PATHS,
            context,
            fields,
            QgsWkbTypes.MultiLineString,
            pathsLayer.sourceCrs()
        )

        f = QgsFeature()
        f.initAttributes(2)
        f.setFields(fields)

        feedback.pushInfo(self.tr("Analyze input data..."))

        if boundLayer.featureCount() != 1:
            raise QgsProcessingException(
                'The study area layer should contain exactly one polygon or '
                'multipolygon.')


        startPoints = []
        if not keepStart and pointsLayer is not None:
            for feature in pointsLayer.getFeatures():
                geom = QgsGeometry(feature.geometry())
                startPoints.append(geom.asPoint())


        txt = '<h1>QGIS Random Paths summary</h1>\n'
        txt += f'<p>Paths layer: {pathsLayer.sourceName()}</p>\n'
        txt += f'<p>Number of paths: {pathsLayer.featureCount()}</p>\n'
        txt += f'<p>Frame layer: {boundLayer.sourceName()}</p>'
        if overlayLayer is not None:
            txt += f'<p>Overlay layer: {overlayLayer.sourceName()}</p>\n'
        txt += f'<p>Number of iterations: {iterations}</p>\n'
        # txt += f'</p>iteration' + ';' + 'feature id' + ';' + 'intersects</p>'

        if overlayLayer is not None:
            index = QgsSpatialIndex(flags=QgsSpatialIndex.FlagStoreFeatureGeometries)
            index.addFeatures(overlayLayer.getFeatures(), QgsFeatureSink.FastInsert)


        bbox = boundLayer.sourceExtent()
        extent = QgsGeometry().fromRect(bbox)

        request = QgsFeatureRequest()

        total = 100.0 / iterations


        pid = 0
        p = None
        output = []
        for i in range(iterations):
            for feature in pathsLayer.getFeatures():

                if not keepStart and pointsLayer is not None:
                    if pid > len(startPoints) - 1:
                        pid = 0
                    p = startPoints[pid]

                geom = feature.geometry()
                if geom.isMultipart():
                    lines = geom.asMultiPolyline()
                    for points in lines:
                        if keepStart:
                            p = points[0]
                        output.append(self._randomPath(p, points, bbox, extent, minAngle, maxAngle, keepAngles, feedback, context))
                    geom = QgsGeometry.fromMultiPolylineXY(output)
                else:
                    points = geom.asPolyline()
                    if keepStart:
                        p = points[0]
                    output = self._randomPath(p, points, bbox, extent, minAngle, maxAngle, keepAngles, feedback, context)
                    geom = QgsGeometry.fromPolyline(output)

                intersects = 0
                if overlayLayer is not None:
                    rect = geom.boundingBox()
                    ids = index.intersects(rect)
                    for i in ids:
                        ft = next(overlayLayer.getFeatures(request.setFilterFid(i)))
                        tmpGeom = ft.geometry()
                        if geom.intersects(tmpGeom):
                            intersects = 1

                f.setGeometry(geom)
                f.setAttribute('id', feature.id())
                f.setAttribute('intersect', intersects)
                sink.addFeature(f, QgsFeatureSink.FastInsert)
                output[:] = []
                pid += 1

                # txt += f'iteration {i};{feature.id()};{intersects} \n'

            feedback.setProgress(int((i + 1) * total))
        
        results = {self.RANDOM_PATHS: dest_id}
    
        with codecs.open(summaryFile, 'w', encoding='utf-8') as f:
            f.write('<html><head>')
            f.write('<meta http-equiv="Content-Type" content="text/html; charset=utf-8" /></head><body>')
            f.write(txt)
            f.write('</body></html>')
        
        results[self.SUMMARY] = summaryFile

        return results


    def _randomPath(self, p, points, bbox, extent, minAngle, maxAngle, keepAngles, feedback, context):
        random.seed()
        output = []
        rx = bbox.xMinimum() + bbox.width() * random.random()
        ry = bbox.yMinimum() + bbox.height() * random.random()
        pnt = QgsPointXY(rx, ry)

        if p is not None:
            output.append(p)
        else:
            output.append(pnt)

        nIterations = 0
        nPoints = len(points)

        da = QgsDistanceArea()
        for i in range(len(points) - 1):
            p0 = output[-1]
            p1 = points[i]
            p2 = points[i + 1]
            distance = da.measureLine(p1, p2)

            if keepAngles:
                angle = math.radians(p1.azimuth(p2))
                zen = math.radians(90)
                d = distance * math.sin(zen)
                x = p0.x() + d * math.sin(angle)
                y = p0.y() + d * math.cos(angle)

                pnt = QgsPointXY(x, y)
                geom = QgsGeometry.fromPoint(pnt)
                output.append(pnt)
            else:
                while True:
                    angle = minAngle + maxAngle * random.random()

                    # correction for angles outside of 0 - 360
                    while (angle > 360.0):
                        angle = angle - 360.0
                    while (angle < 0.0):
                        angle = angle + 360.0

                    angle = math.radians(angle)
                    zen = math.radians(90)
                    d = distance * math.sin(zen)
                    x = p0.x() + d * math.sin(angle)
                    y = p0.y() + d * math.cos(angle)

                    pnt = QgsPointXY(x, y)
                    geom = QgsGeometry.fromPointXY(pnt)
                    if geom.within(extent):
                        output.append(pnt)
                        break

        return output

    def name(self):
        return 'Random HR Path'

    def displayName(self):
        return self.tr(self.name())

    def group(self):
        return self.tr(self.groupId())

    def groupId(self):
        return 'Animove Algorithms'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def icon(self):
        icon_path = os.path.join(
            os.path.dirname(__file__),
            'icons',
            'mcp.png'
        )
        return QIcon(icon_path)

    def tags(self):
        return self.tr('random,home range,animal').split(',')

    def createInstance(self):
        return AnimoveRandomHRPath()
