# -*- coding: utf-8 -*-
"""
/***************************************************************************
 serviciosIDEE2QGIS
                                 A QGIS plugin
 Permite cargar a QGIS los servicios de la IDEE a partir del Catálogo CSW
                              -------------------
        begin                : 2025-10-15
        git sha              : $Format:%H$
        copyright            : (C) 2025
        author               : ingenieroGeomatico
        email                : aurearagon@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QTableWidgetItem, QMessageBox
from qgis.PyQt.QtCore import QThread, pyqtSignal
from qgis.PyQt.QtWidgets import QProgressDialog

from qgis.core import QgsRasterLayer, QgsVectorLayer, QgsProject, QgsVectorTileLayer


from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QTableWidget, QTableWidgetItem,
    QDialogButtonBox, QHeaderView, QAbstractItemView, QPushButton,
)
from .resources import *
from .serviciosIDEE2QGIS_dialog import serviciosIDEE2QGISDialog

import os
import time
import requests
import xml.etree.ElementTree as ET


URL_Catalogo = "https://www.idee.es/segun-tipo-de-servicio"
URL_WMS_est = (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?"
    "p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&"
    "p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&"
    "_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=supVisWmsEst&"
    "_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"
)
URL_WMS_aut = (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=supVisWmsAut&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"
)
URL_WMS_loc = (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=supVisWmsLoc&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"
)
URL_WMS_pve= (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=supVisWmsPV&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"    
)
URL_WMTS = (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=sup-vis-wmts&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"
)
URL_XYZ= (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=sup-vis-rts&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"
)
URL_VectorTile = (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=sup-vis-vts&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"
)
URL_WFS = (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=sup-des-wfs&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"    
)
URL_WCS = (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=sup-des-wcs&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"
)
URL_OGCAPI = (
    "https://www.idee.es/web/idee/segun-tipo-de-servicio?p_p_id=es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f&p_p_lifecycle=2&p_p_state=normal&p_p_mode=view&p_p_cacheability=cacheLevelPage&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_id=sup-ogc-api&_es_igncnig_dirserv72_DirectorioServiciosPortlet_INSTANCE_YZFuNrhnVi4f_actionName=cargaTablaSrv"
)

# -----------------------------------------------------------------------------
# Clase principal del plugin
# -----------------------------------------------------------------------------

class serviciosIDEE2QGIS:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(self.plugin_dir, 'i18n', f'serviciosIDEE2QGIS_{locale}.qm')
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        self.actions = []
        self.menu = self.tr(u'&serviciosIDEE2QGIS')
        self.first_start = None
        

    def tr(self, message):
        return QCoreApplication.translate('serviciosIDEE2QGIS', message)

    def add_action(self, icon_path, text, callback, parent=None):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        self.iface.addToolBarIcon(action)
        self.iface.addPluginToWebMenu(self.menu, action)
        self.actions.append(action)
        return action

    def initGui(self):
        icon_path = ':/plugins/serviciosIDEE2QGIS/icon.png'
        self.add_action(icon_path, text=self.tr(u'Servicios IDEE'), callback=self.run, parent=self.iface.mainWindow())
        self.first_start = True

    def unload(self):
        for action in self.actions:
            self.iface.removePluginWebMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        if self.first_start:
            self.first_start = False
            self.dlg = serviciosIDEE2QGISDialog()
            # 💡 Aquí conectamos los filtros
            self.dlg.lineEditBuscar_WMS.textChanged.connect(self.filtrar_tabla_wms)
            self.dlg.lineEditBuscar_WMTS.textChanged.connect(self.filtrar_tabla_wmts)
            self.dlg.lineEditBuscar_XYZ.textChanged.connect(self.filtrar_tabla_xyz)
            self.dlg.lineEditBuscar_MVT.textChanged.connect(self.filtrar_tabla_mvt)
            self.dlg.lineEditBuscar_WFS.textChanged.connect(self.filtrar_tabla_wfs)
            self.dlg.lineEditBuscar_WCS.textChanged.connect(self.filtrar_tabla_wcs)
            self.dlg.lineEditBuscar_OGCAPI.textChanged.connect(self.filtrar_tabla_ogcapi)
            self.dlg.show()  # Abrir el diálogo inmediatamente

            # Crear ProgressDialog
            self.progress = QProgressDialog("Cargando servicios IDEE...", "Cancelar", 0, 0, self.dlg)
            self.progress.setWindowModality(Qt.WindowModal)
            self.progress.show()

            # Lanzar threads para obtener listado de servicios
            self.worker_wms = CargarServiciosWorker("WMS", [URL_WMS_est, URL_WMS_aut, URL_WMS_loc, URL_WMS_pve])
            self.worker_wms.resultado.connect(self.rellenar_tabla)
            self.worker_wms.start()

            self.worker_wmts = CargarServiciosWorker("WMTS", [URL_WMTS])
            self.worker_wmts.resultado.connect(self.rellenar_tabla)
            self.worker_wmts.start()

            self.worker_xyz = CargarServiciosWorker("TMSXYZ", [URL_XYZ])
            self.worker_xyz.resultado.connect(self.rellenar_tabla)
            self.worker_xyz.start()

            self.worker_mvt = CargarServiciosWorker("MVT", [URL_VectorTile])
            self.worker_mvt.resultado.connect(self.rellenar_tabla)
            self.worker_mvt.start()

            self.worker_wfs = CargarServiciosWorker("WFS", [URL_WFS])
            self.worker_wfs.resultado.connect(self.rellenar_tabla)
            self.worker_wfs.start()

            self.worker_wcs = CargarServiciosWorker("WCS", [URL_WCS])
            self.worker_wcs.resultado.connect(self.rellenar_tabla)
            self.worker_wcs.start()

            self.worker_ogcapi = CargarServiciosWorker("OGCAPI", [URL_OGCAPI])
            self.worker_ogcapi.resultado.connect(self.rellenar_tabla)
            self.worker_ogcapi.start()

        else:
            self.dlg.show()

    
    def rellenar_tabla(self, tab, datos_json):

        if tab == "WMS":
            tabla = self.dlg.tableWidget_WMS

        elif tab == "WMTS":
            tabla = self.dlg.tableWidget_WMTS

        elif tab == "TMSXYZ":
            tabla = self.dlg.tableWidget_XYZ
        
        elif tab == "MVT":
            tabla = self.dlg.tableWidget_MVT
        
        elif tab == "WFS":
            tabla = self.dlg.tableWidget_WFS
        
        elif tab == "WCS":
            tabla = self.dlg.tableWidget_WCS
        
        elif tab == "OGCAPI":
            tabla = self.dlg.tableWidget_OGCAPI


        else:
            return

        tabla.setRowCount(0)

        for nodo in datos_json:
            for org in nodo:
                nombre_org = org.get("name", "")
                lista_servicios = org.get("listorg", org.get("listserv", []))

                for srv in lista_servicios:
                    lista_servicios_2 = srv.get("listserv", [])
                    lista_Servicios_2_nodos = srv.get("listsuborg", [])
                    capa_2 = srv.get("capa", [])

                    def rellenarFila(item, nomOrg):
                        
                        def obtenerTipo_OGCAPI(obj):
                            """
                            Determina el tipo de OGC API a partir de la URL, nombre o descripción.
                            Devuelve un string como 'Features', 'Maps', 'Coverages', 'Tiles', etc.
                            """
                            url = obj.get("url", "").lower()
                            nombre = obj.get("name", "").lower()

                            tipo = ""

                            # Detectar por la URL
                            if "features" in url:
                                tipo = "Features"
                            elif "coverages" in url:
                                tipo = "Coverages"
                            elif "maps" in url:
                                tipo = "Maps"
                            elif "processes" in url:
                                tipo = "Processes"
                            elif "tiles" in url:
                                tipo = "Tiles"

                            # Si no se detectó, intentar por nombre o descripción
                            if not tipo:
                                if any(k in nombre for k in ["feature", "features"]):
                                    tipo = "Features"
                                elif any(k in nombre for k in ["coverage", "coverages"]):
                                    tipo = "Coverages"
                                elif any(k in nombre for k in ["map", "maps"]):
                                    tipo = "Maps"
                                elif any(k in nombre for k in ["process", "processes"]):
                                    tipo = "Processes"
                                elif any(k in nombre for k in ["tile", "tiles"]):
                                    tipo = "Tiles"

                            # genérico
                            if not tipo:
                                tipo = "Desconocido"

                            return tipo

                        row = tabla.rowCount()
                        tabla.insertRow(row)
                        nombre = item.get("name", "")
                        servicioCap = item.get("capa", "")
                        servicio = item.get("url", "")
                        masInfo = item.get("urlInfo", "")
                        btn = QPushButton("Añadir a mapa")

                        obj={
                            "nombre": nombre,
                            "capa": servicioCap,
                            "url": servicio,
                            "masInfo": masInfo,
                            "tipo": tab
                        }
                        if tab == "OGCAPI":
                            obj["masInfo"]  = obtenerTipo_OGCAPI(obj)
                            masInfo = obj["masInfo"]

                            if masInfo == "Desconocido" or masInfo == "Features":
                                btn.setDisabled(False)
                            else:
                                btn.setDisabled(True)
                            

                        tabla.setCellWidget(row, 0, btn)
                        btn.clicked.connect(lambda checked, s=servicio, t=tab: self.anadir_a_mapa(s, t, obj))
                        tabla.setItem(row, 1, QTableWidgetItem(nomOrg))
                        tabla.setItem(row, 2, QTableWidgetItem(nombre))
                        tabla.setItem(row, 3, QTableWidgetItem(servicioCap))

                        if masInfo:
                            tabla.setItem(row, 4, QTableWidgetItem(masInfo))

                    if capa_2:
                        rellenarFila(srv, nombre_org)
                    elif lista_servicios_2:
                        if srv.get("name"):
                            nombre_org = srv.get("name", "")
                        for srv2 in lista_servicios_2:
                            rellenarFila(srv2, nombre_org)
                    elif lista_Servicios_2_nodos:
                        for srv2 in lista_Servicios_2_nodos:
                            lista_servicios_3 = srv2.get("listserv", [])
                            if srv2.get("name"):
                                nombre_org = srv2.get("name", "")
                            for srv3 in lista_servicios_3:
                                rellenarFila(srv3, nombre_org)

        # Cerrar el progress si ambos workers han terminado
        if ((not self.worker_xyz.isRunning()) and 
            (not self.worker_wms.isRunning()) and 
            (not self.worker_wmts.isRunning()) and
            (not self.worker_mvt.isRunning()) and
            (not self.worker_wfs.isRunning()) and
            (not self.worker_wcs.isRunning()) and
            (not self.worker_ogcapi.isRunning())
            ):
            self.progress.close()


    # -------------------------------------------------------------------------
    # Método para descargar y rellenar la tabla
    # -------------------------------------------------------------------------

    def anadir_a_mapa(self, servicio, tab, obj):
        """
        Función que se ejecuta al pulsar 'Añadir a mapa'.
        Aquí se abre el diálogo de selección de capas WMS y se cargan al mapa.
        """

        if tab == "WMS":
            progress = QProgressDialog("Buscando capas en el WMS...", "Cancelar", 0, 0, self.dlg)
            progress.setWindowModality(Qt.WindowModal)
            progress.show()

            self.worker = CargarCapasWMSWorker(servicio)
            self.worker.terminado.connect(lambda capas: self._on_capas_cargadas_WMS(capas,servicio, progress))
            self.worker.error.connect(lambda e: (progress.close(), QMessageBox.warning(self.dlg, "Error", e)))            
            self.worker.start()

        elif tab == "WMTS":
            progress = QProgressDialog("Buscando capas en el WMTS...", "Cancelar", 0, 0, self.dlg)
            progress.setWindowModality(Qt.WindowModal)
            progress.show()

            self.worker = CargarCapasWMTSWorker(servicio)
            self.worker.terminado.connect(lambda capas: self._on_capas_cargadas_WMTS(capas,servicio, progress))
            self.worker.error.connect(lambda e: (progress.close(), QMessageBox.warning(self.dlg, "Error", e)))            
            self.worker.start()
        
        elif tab == "TMSXYZ":
            self._on_capas_cargadas_XYZ(obj, servicio)
        
        elif tab == "MVT":
            self._on_capas_cargadas_MVT(obj, servicio)
        
        elif tab == "WFS":
            progress = QProgressDialog("Buscando capas en el WFS...", "Cancelar", 0, 0, self.dlg)
            progress.setWindowModality(Qt.WindowModal)
            progress.show()

            self.worker = CargarCapasWFSWorker(servicio)
            self.worker.terminado.connect(lambda capas: self._on_capas_cargadas_WFS(capas,servicio, progress))
            self.worker.error.connect(lambda e: (progress.close(), QMessageBox.warning(self.dlg, "Error", e)))            
            self.worker.start()
        
        elif tab == "WCS":
            progress = QProgressDialog("Buscando capas en el WCS...", "Cancelar", 0, 0, self.dlg)
            progress.setWindowModality(Qt.WindowModal)
            progress.show()

            self.worker = CargarCapasWCSWorker(servicio)
            self.worker.terminado.connect(lambda capas: self._on_capas_cargadas_WCS(capas,servicio, progress))
            self.worker.error.connect(lambda e: (progress.close(), QMessageBox.warning(self.dlg, "Error", e)))            
            self.worker.start()
        
        elif tab == "OGCAPI":
            progress = QProgressDialog("Buscando capas OGCAPI...", "Cancelar", 0, 0, self.dlg)
            progress.setWindowModality(Qt.WindowModal)
            progress.show()

            self.worker = CargarCapasOGCAPIWorker(servicio,obj)
            self.worker.terminado.connect(lambda capas: self._on_capas_cargadas_OGCAPI(capas,servicio, progress))
            self.worker.error.connect(lambda e: (progress.close(), QMessageBox.warning(self.dlg, "Error", e)))            
            self.worker.start()


        else:
            return

    def _on_capas_cargadas_WMS(self, capas, servicio, progress):
        """Se llama cuando el worker ha terminado de obtener las capas WMS"""
        progress.close()  # cerrar el diálogo de espera

        if not capas:
            QMessageBox.warning(self.dlg, "Error", "No se encontraron subcapas en el WMS")
            return

        # Abrir diálogo de selección
        dialog = SeleccionarCapasDialog_WMS(capas, parent=self.dlg)
        if dialog.exec_():
            capas_elegidas = dialog.capas_seleccionadas()
            for capa in capas_elegidas:
                uri = (
                    f"contextualWMSLegend=0&url={servicio}"
                    f"&layers={capa['name']}&styles=&format={capa['format']}&crs=EPSG:3857"
                )
                nombre_capa = capa['name']
                title = capa['title']
                layer = QgsRasterLayer(uri, title, "wms")
                if layer.isValid():
                    QgsProject.instance().addMapLayer(layer)
                else:
                    print(f"No se pudo cargar la capa: {nombre_capa}")

    def filtrar_tabla_wms(self):
        """Filtra la tabla de WMS por texto en las columnas Organismo y Nombre."""
        texto = self.dlg.lineEditBuscar_WMS.text().lower()
        tabla = self.dlg.tableWidget_WMS

        for row in range(tabla.rowCount()):
            item_org = tabla.item(row, 1)   # columna Organismo
            item_nombre = tabla.item(row, 2)  # columna Nombre / Descripción

            texto_org = item_org.text().lower() if item_org else ""
            texto_nombre = item_nombre.text().lower() if item_nombre else ""

            # Mostrar fila si el texto aparece en alguna de las dos columnas
            visible = (texto in texto_org) or (texto in texto_nombre)
            tabla.setRowHidden(row, not visible)
    

    def _on_capas_cargadas_WMTS(self, capas, servicio, progress):
        """Se llama cuando el worker ha terminado de obtener las capas WMTS"""
        progress.close()  # cerrar el diálogo de espera

        if not capas:
            QMessageBox.warning(self.dlg, "Error", "No se encontraron capas en el WMTS")
            return

        # Abrir diálogo de selección
        dialog = SeleccionarCapasDialog_WMTS(capas, parent=self.dlg)
        if dialog.exec_():
            capas_elegidas = dialog.capas_seleccionadas()
            for capa in capas_elegidas:
                url_base = servicio.split("?")[0]  # quitar parámetros GetCapabilities

                uri = (
                    f"crs=EPSG:3857"
                    f"&dpiMode=7"
                    f"&format={capa['format']}"
                    f"&layers={capa['identifier']}"
                    f"&styles=default"
                    f"&tileMatrixSet=GoogleMapsCompatible"
                    f"&tilePixelRatio=0"
                    f"&url={url_base}"
                )

                layer = QgsRasterLayer(uri, f"{capa['title']}", "wms")

                if layer.isValid():
                    QgsProject.instance().addMapLayer(layer)
                else:
                    print(f"No se pudo cargar la capa: {capa}")

    def filtrar_tabla_wmts(self):
        """Filtra la tabla de WMTS por texto en las columnas Organismo y Nombre."""
        texto = self.dlg.lineEditBuscar_WMTS.text().lower()
        tabla = self.dlg.tableWidget_WMTS

        for row in range(tabla.rowCount()):
            item_org = tabla.item(row, 1)   # columna Organismo
            item_nombre = tabla.item(row, 2)  # columna Nombre / Descripción

            texto_org = item_org.text().lower() if item_org else ""
            texto_nombre = item_nombre.text().lower() if item_nombre else ""

            visible = (texto in texto_org) or (texto in texto_nombre)
            tabla.setRowHidden(row, not visible)


    def _on_capas_cargadas_XYZ(self, obj, servicio):
        """
        Se llama cuando el worker ha terminado de obtener las capas XYZ.
        Carga la capa XYZ en QGIS.
        """

        if not obj:
            QMessageBox.warning(self.dlg, "Error", "No se encontraron capas en el servicio XYZ")
            return

        # Asegurarnos de tener la URL base sin parámetros
        url_base = servicio.split("?")[0]

        # La plantilla de URL XYZ debe tener {z}, {x}, {y} o {-y}
        # Si el servicio no la tiene, la añadimos
        if "{z}" not in url_base:
            if not url_base.endswith("/"):
                url_base += "/"
            url_base += "{z}/{x}/{y}.png"

        # Construcción del URI para QGIS
        uri = (
            f"http-header:referer="
            f"&type=xyz"
            f"&url={url_base}"
            f"&zmax=19"
            f"&zmin=0"
        )

        layer_name = obj.get("nombre") or "Capa XYZ"

        layer = QgsRasterLayer(uri, layer_name, "wms")

        if layer.isValid():
            QgsProject.instance().addMapLayer(layer)
        else:
            QMessageBox.warning(self.dlg, "Error", f"No se pudo cargar la capa XYZ:\n{layer_name}")
            print(f"No se pudo cargar la capa: {obj}")

    def filtrar_tabla_xyz(self):
        """Filtra la tabla de TMS por texto en las columnas Organismo y Nombre."""
        texto = self.dlg.lineEditBuscar_XYZ.text().lower()
        tabla = self.dlg.tableWidget_XYZ

        for row in range(tabla.rowCount()):
            item_org = tabla.item(row, 1)   # columna Organismo
            item_nombre = tabla.item(row, 2)  # columna Nombre / Descripción

            texto_org = item_org.text().lower() if item_org else ""
            texto_nombre = item_nombre.text().lower() if item_nombre else ""

            visible = (texto in texto_org) or (texto in texto_nombre)
            tabla.setRowHidden(row, not visible)


    def _on_capas_cargadas_MVT(self, obj, servicio):
        """
        Se llama cuando el worker ha terminado de obtener las capas MVT.
        Carga la capa vectorial (MVT) en QGIS.
        """

        if not obj:
            QMessageBox.warning(self.dlg, "Error", "No se encontraron capas en el servicio MVT")
            return

        # Asegurarnos de tener la URL base sin parámetros
        url_base = servicio.split("?")[0]

        # Asegurarnos de que la URL tenga la plantilla {z}/{y}/{x}
        if "{z}" not in url_base:
            if not url_base.endswith("/"):
                url_base += "/"
            url_base += "{z}/{y}/{x}.pbf"

        # Construcción del URI para QGIS (según formato xyzvectortiles)
        uri = (
            f"type=xyz"
            f"&url={url_base}"
            f"&zmax=18"
            f"&zmin=0"
            f"&http-header:referer="
        )

        layer_name = obj.get("nombre") or "Capa MVT"

        # Cargar como capa de vector tiles
        layer = QgsVectorTileLayer(uri, layer_name)

        if layer.isValid():
            QgsProject.instance().addMapLayer(layer)
            print(f"Capa MVT cargada correctamente: {layer_name}")
        else:
            QMessageBox.warning(self.dlg, "Error", f"No se pudo cargar la capa MVT:\n{layer_name}")
            print(f"No se pudo cargar la capa MVT: {obj}")

    def filtrar_tabla_mvt(self):
        """Filtra la tabla de MVT por texto en las columnas Organismo y Nombre."""
        texto = self.dlg.lineEditBuscar_MVT.text().lower()
        tabla = self.dlg.tableWidget_MVT

        for row in range(tabla.rowCount()):
            item_org = tabla.item(row, 1)   # columna Organismo
            item_nombre = tabla.item(row, 2)  # columna Nombre / Descripción

            texto_org = item_org.text().lower() if item_org else ""
            texto_nombre = item_nombre.text().lower() if item_nombre else ""

            visible = (texto in texto_org) or (texto in texto_nombre)
            tabla.setRowHidden(row, not visible)


    def _on_capas_cargadas_WFS(self, capas, servicio, progress):
        """Se llama cuando el worker ha terminado de obtener las capas WFS"""
        progress.close()

        if not capas:
            QMessageBox.warning(self.dlg, "Error", "No se encontraron capas en el WFS")
            return

        dialog = SeleccionarCapasDialog_WFS(capas, parent=self.dlg)
        if dialog.exec_():
            capas_elegidas = dialog.capas_seleccionadas()
            for capa in capas_elegidas:
                typename = capa['identifier'].strip()
                title = capa['title'].strip()
                if not typename:
                    print(f"Capa sin identifier válido: {capa}")
                    continue

                url_base = servicio.split("?")[0]  # quitar parámetros GetCapabilities

                # URI con formato compatible con QgsVectorLayer WFS
                uri = (
                    f"url='{url_base}' "
                    f"typename='{typename}' "
                    f"restrictToRequestBBOX=1 "
                    f"InvertAxisOrientation=1 "
                )

                layer = QgsVectorLayer(uri, f"{title}", "WFS")

                if layer.isValid():
                    QgsProject.instance().addMapLayer(layer)
                    print(f"Capa WFS añadida: {typename}")
                else:
                    print(f"No se pudo cargar la capa: {typename}")

    def filtrar_tabla_wfs(self):
        """Filtra la tabla de WFS por texto en las columnas Organismo y Nombre."""
        texto = self.dlg.lineEditBuscar_WFS.text().lower()
        tabla = self.dlg.tableWidget_WFS

        for row in range(tabla.rowCount()):
            item_org = tabla.item(row, 1)   # columna Organismo
            item_nombre = tabla.item(row, 2)  # columna Nombre / Descripción

            texto_org = item_org.text().lower() if item_org else ""
            texto_nombre = item_nombre.text().lower() if item_nombre else ""

            visible = (texto in texto_org) or (texto in texto_nombre)
            tabla.setRowHidden(row, not visible)


    def _on_capas_cargadas_WCS(self, capas, servicio, progress):
        """Se llama cuando el worker ha terminado de obtener las capas WCS"""
        progress.close()

        if not capas:
            QMessageBox.warning(self.dlg, "Error", "No se encontraron capas en el WCS")
            return

        dialog = SeleccionarCapasDialog_WCS(capas, parent=self.dlg)
        if dialog.exec_():
            capas_elegidas = dialog.capas_seleccionadas()
            for capa in capas_elegidas:
                identifier = capa['identifier'].strip()
                if not identifier:
                    print(f"Capa sin identifier válido: {capa}")
                    continue

                url_base = servicio.split("?")[0]  # quitar parámetros GetCapabilities

                # URI con formato con '&' para QgsRasterLayer WCS
                uri = (
                    f"dpiMode=7"
                    f"&identifier={identifier}"
                    f"&tilePixelRatio=0"
                    f"&url={url_base}"
                )

                layer = QgsRasterLayer(uri, f"{identifier}", "wcs")

                if layer.isValid():
                    QgsProject.instance().addMapLayer(layer)
                    print(f"Capa WCS añadida: {identifier}")
                else:
                    print(f"No se pudo cargar la capa: {identifier}")
                    print("URI usada:", uri)

    def filtrar_tabla_wcs(self):
        """Filtra la tabla de WCS por texto en las columnas Organismo y Nombre."""
        texto = self.dlg.lineEditBuscar_WCS.text().lower()
        tabla = self.dlg.tableWidget_WCS

        for row in range(tabla.rowCount()):
            item_org = tabla.item(row, 1)   # columna Organismo
            item_nombre = tabla.item(row, 2)  # columna Nombre / Descripción

            texto_org = item_org.text().lower() if item_org else ""
            texto_nombre = item_nombre.text().lower() if item_nombre else ""

            visible = (texto in texto_org) or (texto in texto_nombre)
            tabla.setRowHidden(row, not visible)


    def _on_capas_cargadas_OGCAPI(self, capas, servicio, progress):
        """Se llama cuando el worker ha terminado de obtener las capas OGCAPI Features"""
        progress.close()

        if not capas:
            QMessageBox.warning(self.dlg, "Error", "No se encontraron capas en el OGC API Features")
            return

        dialog = SeleccionarCapasDialog_OGCAPI(capas, parent=self.dlg)
        if dialog.exec_():
            capas_elegidas = dialog.capas_seleccionadas()
            for capa in capas_elegidas:
                typename = capa['identifier'].strip()
                title = capa['title'].strip()
                if not typename:
                    print(f"Capa sin identifier válido: {capa}")
                    continue

                # Quitar barra final y "/collections" si viene incluida
                url_base = servicio.rstrip("/")
                if url_base.lower().endswith("/collections"):
                    url_base = url_base[: -len("/collections")]

                # URI para el proveedor 'oapif'
                uri = (
                    f"url={url_base} "
                    f"typename={typename} "
                    f"restrictToRequestBBOX=1 "
                    f"pageSize=1000 "
                    f"maxNumFeatures=10000"
                )

                # Crear layer con proveedor OGC API Features
                layer = QgsVectorLayer(uri, title, "oapif")

                if layer.isValid():
                    QgsProject.instance().addMapLayer(layer)
                    print(f"Capa OGCAPI Features añadida: {typename}")
                else:
                    print(f"No se pudo cargar la capa: {typename}")
                    print("URI usada:", uri)

    def filtrar_tabla_ogcapi(self):
        """Filtra la tabla de OGCAPI por texto en las columnas Organismo y Nombre."""
        texto = self.dlg.lineEditBuscar_OGCAPI.text().lower()
        tabla = self.dlg.tableWidget_OGCAPI

        for row in range(tabla.rowCount()):
            item_org = tabla.item(row, 1)   # columna Organismo
            item_nombre = tabla.item(row, 2)  # columna Nombre / Descripción

            texto_org = item_org.text().lower() if item_org else ""
            texto_nombre = item_nombre.text().lower() if item_nombre else ""

            visible = (texto in texto_org) or (texto in texto_nombre)
            tabla.setRowHidden(row, not visible)



class CargarServiciosWorker(QThread):
    resultado = pyqtSignal(str, object)  # tab, datos_json

    def __init__(self, tab, urls):
        super().__init__()
        self.tab = tab
        self.urls = urls  # lista de URLs a descargar

    def run(self):
        datos_completos = []
        for url in self.urls:
            try:
                response = requests.get(url)
                response.raise_for_status()
                datos_json_r = response.json()
                # if self.tab == "WMS":
                #     datos_tab = [
                #         datos_json_r["datos"].get("est", []),
                #         datos_json_r["datos"].get("aut", []),
                #         datos_json_r["datos"].get("loc", []),
                #         datos_json_r["datos"].get("pve", []),
                #     ]
                # elif self.tab == "WMTS":
                #     datos_tab = [
                #         datos_json_r["datos"].get("est", []),
                #         datos_json_r["datos"].get("aut", []),
                #         datos_json_r["datos"].get("loc", []),
                #         datos_json_r["datos"].get("pve", []),
                #     ]
                # elif self.tab == "TMSXYZ":
                #     datos_tab = [
                #         datos_json_r["datos"].get("est", []),
                #         datos_json_r["datos"].get("aut", []),
                #         datos_json_r["datos"].get("loc", []),
                #         datos_json_r["datos"].get("pve", []),
                #     ]
                # elif self.tab == "MVT":
                #     datos_tab = [
                #         datos_json_r["datos"].get("est", []),
                #         datos_json_r["datos"].get("aut", []),
                #         datos_json_r["datos"].get("loc", []),
                #         datos_json_r["datos"].get("pve", []),
                #     ]
                # else:
                #     pass
                datos_tab = [
                        datos_json_r["datos"].get("est", []),
                        datos_json_r["datos"].get("aut", []),
                        datos_json_r["datos"].get("loc", []),
                        datos_json_r["datos"].get("pve", []),
                ]
                datos_completos.extend(datos_tab)

            except Exception as e:
                print(f"Error descargando {url}: {e}")

        self.resultado.emit(self.tab, datos_completos)


class SeleccionarCapasDialog_WMS(QDialog):
    def __init__(self, capas, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Seleccionar capas WMS")
        self.setMinimumWidth(700)
        self.setMinimumHeight(400)

        layout = QVBoxLayout()
        self.setLayout(layout)

        # Tabla de capas
        self.table = QTableWidget()
        self.table.setColumnCount(4)
        self.table.setHorizontalHeaderLabels(["Identificador", "Título", "formato", "Descripción"])
        self.table.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.table.setSelectionMode(QAbstractItemView.MultiSelection)
        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.table.verticalHeader().setVisible(False)
        self.table.setColumnWidth(0, 200)
        self.table.setColumnWidth(1, 250)
        self.table.setColumnWidth(2, 200)
        self.table.setColumnWidth(3, 300)

        # Llenar tabla
        self.table.setRowCount(len(capas))
        for row, capa in enumerate(capas):
            nombre = capa.get("name", "")
            titulo = capa.get("title", "")
            formato = capa.get("format", "")
            descripcion = capa.get("abstract", "")
            self.table.setItem(row, 0, QTableWidgetItem(nombre))
            self.table.setItem(row, 1, QTableWidgetItem(titulo))
            self.table.setItem(row, 2, QTableWidgetItem(formato))
            self.table.setItem(row, 3, QTableWidgetItem(descripcion))

        layout.addWidget(self.table)

        # Botones OK / Cancelar
        self.buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)
        layout.addWidget(self.buttonBox)

    def capas_seleccionadas(self):
        """Devuelve lista con los 'name' de las capas seleccionadas"""
        capas = []
        for item in self.table.selectionModel().selectedRows():
            row = item.row()
            nombre = self.table.item(row, 0).text()
            title = self.table.item(row, 1).text()
            formato = self.table.item(row, 2).text()
            obj = {
                "name": nombre,
                "format": formato,
                "title": title
            }
            capas.append(obj)
        return capas

class CargarCapasWMSWorker(QThread):
    terminado = pyqtSignal(list)
    error = pyqtSignal(str)

    def __init__(self, servicio):
        super().__init__()
        self.servicio = servicio

    def run(self):
        try:
            capas = self.obtener_capas_wms(self.servicio)
            self.terminado.emit(capas)
        except Exception as e:
            self.error.emit(str(e))
    
    def obtener_capas_wms(self, servicio):
        """
        Obtiene las capas disponibles en un servicio WMS mediante su GetCapabilities.
        Devuelve una lista de diccionarios:
        [{'name':..., 'title':..., 'abstract':..., 'format':...}, ...]
        Cada capa se repite si tiene varios formatos de salida.
        """
        url = servicio
        if not url.lower().startswith("http"):
            return []

        params = {"SERVICE": "WMS", "REQUEST": "GetCapabilities"}
        response = requests.get(url, params=params, timeout=10)
        response.raise_for_status()

        root = ET.fromstring(response.content)
        ns = {
            "wms": "http://www.opengis.net/wms"
        }

        # Obtener los formatos disponibles a nivel global (GetMap)
        global_formats = root.findall(".//wms:Capability/wms:Request/wms:GetMap/wms:Format", ns)
        formatos_globales = [f.text for f in global_formats if f.text] or ["image/png"]

        capas = []
        for layer in root.findall(".//wms:Layer", ns):
            name_el = layer.find("wms:Name", ns)
            title_el = layer.find("wms:Title", ns)
            abstract_el = layer.find("wms:Abstract", ns)

            if name_el is not None:
                # Buscar formatos específicos de la capa (si existen)
                layer_formats = layer.findall("wms:Format", ns)
                formatos = [f.text for f in layer_formats if f.text] or formatos_globales

                # Crear un registro por cada formato
                for fmt in formatos:
                    capas.append({
                        "name": name_el.text or "",
                        "title": title_el.text if title_el is not None else "",
                        "abstract": abstract_el.text if abstract_el is not None else "",
                        "format": fmt
                    })

        return capas


class SeleccionarCapasDialog_WMTS(QDialog):
    def __init__(self, capas, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Seleccionar capas WMTS")
        self.setMinimumWidth(700)
        self.setMinimumHeight(400)

        layout = QVBoxLayout()
        self.setLayout(layout)

        # Tabla de capas
        self.table = QTableWidget()
        self.table.setColumnCount(4)
        self.table.setHorizontalHeaderLabels(["Identificador", "Título", "formato", "Descripción"])
        self.table.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.table.setSelectionMode(QAbstractItemView.MultiSelection)
        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.table.verticalHeader().setVisible(False)
        self.table.setColumnWidth(0, 200)
        self.table.setColumnWidth(1, 250)
        self.table.setColumnWidth(2, 200)
        self.table.setColumnWidth(3, 300)

        # Llenar tabla
        self.table.setRowCount(len(capas))
        for row, capa in enumerate(capas):
            identifier = capa.get("identifier", "")
            titulo = capa.get("title", "")
            format = capa.get("format", "")
            descripcion = capa.get("abstract", "")
            self.table.setItem(row, 0, QTableWidgetItem(identifier))
            self.table.setItem(row, 1, QTableWidgetItem(titulo))
            self.table.setItem(row, 2, QTableWidgetItem(format))
            self.table.setItem(row, 3, QTableWidgetItem(descripcion))

        layout.addWidget(self.table)

        # Botones OK / Cancelar
        self.buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)
        layout.addWidget(self.buttonBox)

    def capas_seleccionadas(self):
        """Devuelve lista con los 'identifier' de las capas seleccionadas"""
        capas = []
        for item in self.table.selectionModel().selectedRows():
            row = item.row()
            identifier = self.table.item(row, 0).text()
            title = self.table.item(row, 1).text()
            format = self.table.item(row, 2).text()
            obj={
                "identifier": identifier,
                "format":format,
                "title":title
            }
            capas.append(obj)
        return capas

class CargarCapasWMTSWorker(QThread):
    terminado = pyqtSignal(list)
    error = pyqtSignal(str)

    def __init__(self, servicio):
        super().__init__()
        self.servicio = servicio

    def run(self):
        try:
            capas = self.obtener_capas_wmts(self.servicio)
            self.terminado.emit(capas)
        except Exception as e:
            self.error.emit(str(e))
    
    def obtener_capas_wmts(self, servicio):
        """
        Obtiene las capas disponibles en un servicio WMTS mediante su GetCapabilities.
        Devuelve una lista de diccionarios: [{'identifier':..., 'title':..., 'abstract':..., 'format':...}, ...]
        Cada capa se repite si tiene varios formatos de salida.
        """
        url = servicio
        if not url.lower().startswith("http"):
            return []

        params = {"SERVICE": "WMTS", "REQUEST": "GetCapabilities"}
        try:
            response = requests.get(url, params=params, timeout=10)
            response.raise_for_status()
        except requests.RequestException as e:
            QMessageBox.warning(self.dlg, "Error", f"No se pudo acceder al WMTS:\n{e}")
            return []

        root = ET.fromstring(response.content)
        ns = {
            "wmts": "http://www.opengis.net/wmts/1.0",
            "ows": "http://www.opengis.net/ows/1.1",
        }

        capas = []
        for layer in root.findall(".//wmts:Layer", ns):
            identifier_el = layer.find("ows:Identifier", ns)
            title_el = layer.find("ows:Title", ns)
            abstract_el = layer.find("ows:Abstract", ns)

            if identifier_el is not None:
                # Buscar formatos disponibles
                formats = layer.findall("wmts:Format", ns)
                if not formats:
                    # Si no hay formatos, añadir un registro vacío
                    capas.append({
                        "identifier": identifier_el.text or "",
                        "title": title_el.text if title_el is not None else "",
                        "abstract": abstract_el.text if abstract_el is not None else "",
                        "format": ""
                    })
                else:
                    for fmt in formats:
                        capas.append({
                            "identifier": identifier_el.text or "",
                            "title": title_el.text if title_el is not None else "",
                            "abstract": abstract_el.text if abstract_el is not None else "",
                            "format": fmt.text or ""
                        })
        return capas


class SeleccionarCapasDialog_WFS(QDialog):
    def __init__(self, capas, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Seleccionar capas WFS")
        self.setMinimumWidth(700)
        self.setMinimumHeight(400)

        layout = QVBoxLayout()
        self.setLayout(layout)

        # Tabla de capas
        self.table = QTableWidget()
        self.table.setColumnCount(4)
        self.table.setHorizontalHeaderLabels(["Identificador", "Título", "formato", "Descripción"])
        self.table.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.table.setSelectionMode(QAbstractItemView.MultiSelection)
        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.table.verticalHeader().setVisible(False)
        self.table.setColumnWidth(0, 200)
        self.table.setColumnWidth(1, 250)
        self.table.setColumnWidth(2, 200)
        self.table.setColumnWidth(3, 300)

        # Llenar tabla
        self.table.setRowCount(len(capas))
        for row, capa in enumerate(capas):
            identifier = capa.get("identifier", "")
            titulo = capa.get("title", "")
            format = capa.get("format", "")
            descripcion = capa.get("abstract", "")
            self.table.setItem(row, 0, QTableWidgetItem(identifier))
            self.table.setItem(row, 1, QTableWidgetItem(titulo))
            self.table.setItem(row, 2, QTableWidgetItem(format))
            self.table.setItem(row, 3, QTableWidgetItem(descripcion))

        layout.addWidget(self.table)

        # Botones OK / Cancelar
        self.buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)
        layout.addWidget(self.buttonBox)

    def capas_seleccionadas(self):
        """Devuelve lista con los 'identifier' de las capas seleccionadas"""
        capas = []
        for item in self.table.selectionModel().selectedRows():
            row = item.row()
            identifier = self.table.item(row, 0).text()
            title = self.table.item(row, 1).text()
            format = self.table.item(row, 2).text()
            obj={
                "identifier": identifier,
                "format":format,
                "title":title
            }
            capas.append(obj)
        return capas

class CargarCapasWFSWorker(QThread):
    terminado = pyqtSignal(list)
    error = pyqtSignal(str)

    def __init__(self, servicio):
        super().__init__()
        self.servicio = servicio

    def run(self):
        try:
            capas = self.obtener_capas_wfs(self.servicio)
            self.terminado.emit(capas)
        except Exception as e:
            self.error.emit(str(e))
        
    def obtener_capas_wfs(self, servicio):
        
        def obtener_output_formats(root):
            """
            Devuelve una lista de formatos de salida soportados por el WFS.
            Intenta capturar OutputFormats/Format y ows:Parameter/AllowedValues/Value.
            """
            output_formats = set()  # set para evitar duplicados

            # 1. Formatos dentro de OutputFormats/Format
            for of in root.iter():
                if of.tag.endswith('OutputFormats'):
                    for fmt in of:
                        if fmt.tag.endswith('Format') and fmt.text:
                            output_formats.add(fmt.text.strip())

            # 2. Formatos dentro de ows:Parameter/AllowedValues/Value
            for param in root.iter():
                if param.tag.endswith('Parameter') and param.attrib.get('name') == 'outputFormat':
                    for val in param.iter():
                        if val.tag.endswith('Value') and val.text:
                            output_formats.add(val.text.strip())

            if not output_formats:
                # fallback
                output_formats = {"GML2", "GML3", "GeoJSON"}

            return list(output_formats)


        if not servicio.lower().startswith("http"):
            return []

        params = {"SERVICE": "WFS", "REQUEST": "GetCapabilities"}
        try:
            response = requests.get(servicio, params=params, timeout=10)
            response.raise_for_status()
        except requests.RequestException as e:
            QMessageBox.warning(None, "Error", f"No se pudo acceder al WFS:\n{e}")
            return []

        try:
            root = ET.fromstring(response.content)
        except ET.ParseError as e:
            QMessageBox.warning(None, "Error", f"No se pudo parsear el XML de GetCapabilities:\n{e}")
            return []

        # Buscar FeatureType
        feature_types = [ft for ft in root.iter() if ft.tag.endswith('FeatureType')]
        if not feature_types:
            QMessageBox.warning(None, "Error", "No se encontraron capas en el WFS.")
            return []

        # Obtener formatos
        output_formats = obtener_output_formats(root)

        capas = []
        for ft in feature_types:
            name_el = title_el = abstract_el = None
            for child in ft:
                if child.tag.endswith('Name'):
                    name_el = child
                elif child.tag.endswith('Title'):
                    title_el = child
                elif child.tag.endswith('Abstract'):
                    abstract_el = child

            identifier = name_el.text.strip() if name_el is not None and name_el.text else ""
            title = title_el.text.strip() if title_el is not None and title_el.text else ""
            abstract = abstract_el.text.strip() if abstract_el is not None and abstract_el.text else ""

            for fmt in output_formats:
                capas.append({
                    "identifier": identifier,
                    "title": title,
                    "abstract": abstract,
                    "format": fmt
                })

        return capas


class SeleccionarCapasDialog_WCS(QDialog):
    def __init__(self, capas, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Seleccionar capas WCS")
        self.setMinimumWidth(700)
        self.setMinimumHeight(400)

        layout = QVBoxLayout()
        self.setLayout(layout)

        # Tabla de capas
        self.table = QTableWidget()
        self.table.setColumnCount(4)
        self.table.setHorizontalHeaderLabels(["Identificador", "Título", "formato", "Descripción"])
        self.table.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.table.setSelectionMode(QAbstractItemView.MultiSelection)
        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.table.verticalHeader().setVisible(False)
        self.table.setColumnWidth(0, 200)
        self.table.setColumnWidth(1, 250)
        self.table.setColumnWidth(2, 200)
        self.table.setColumnWidth(3, 300)

        # Llenar tabla
        self.table.setRowCount(len(capas))
        for row, capa in enumerate(capas):
            identifier = capa.get("identifier", "")
            titulo = capa.get("title", "")
            format = capa.get("format", "")
            descripcion = capa.get("abstract", "")
            self.table.setItem(row, 0, QTableWidgetItem(identifier))
            self.table.setItem(row, 1, QTableWidgetItem(titulo))
            self.table.setItem(row, 2, QTableWidgetItem(format))
            self.table.setItem(row, 3, QTableWidgetItem(descripcion))

        layout.addWidget(self.table)

        # Botones OK / Cancelar
        self.buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)
        layout.addWidget(self.buttonBox)

    def capas_seleccionadas(self):
        """Devuelve lista con los 'identifier' de las capas seleccionadas"""
        capas = []
        for item in self.table.selectionModel().selectedRows():
            row = item.row()
            identifier = self.table.item(row, 0).text()
            format = self.table.item(row, 2).text()
            obj={
                "identifier": identifier,
                "format":format
            }
            capas.append(obj)
        return capas

class CargarCapasWCSWorker(QThread):
    terminado = pyqtSignal(list)
    error = pyqtSignal(str)

    def __init__(self, servicio):
        super().__init__()
        self.servicio = servicio

    def run(self):
        try:
            capas = self.obtener_capas_wcs(self.servicio)
            self.terminado.emit(capas)
        except Exception as e:
            self.error.emit(str(e))
        
    def obtener_capas_wcs(self, servicio):


        def obtener_output_formats(root):
            output_formats = set()

            for sf in root.iter():
                if sf.tag.endswith('SupportedFormats'):
                    for fmt in sf:
                        if fmt.tag.endswith('Format') and fmt.text:
                            output_formats.add(fmt.text.strip())

            for param in root.iter():
                if param.tag.endswith('Parameter') and param.attrib.get('name') == 'outputFormat':
                    for val in param.iter():
                        if val.tag.endswith('Value') and val.text:
                            output_formats.add(val.text.strip())

            if not output_formats:
                output_formats = {"GeoTIFF", "NetCDF", "image/tiff"}

            return list(output_formats)

        if not servicio.lower().startswith("http"):
            return []

        params = {"SERVICE": "WCS", "REQUEST": "GetCapabilities"}
        try:
            response = requests.get(servicio, params=params, timeout=10)
            response.raise_for_status()
        except requests.RequestException as e:
            QMessageBox.warning(None, "Error", f"No se pudo acceder al WCS:\n{e}")
            return []

        try:
            root = ET.fromstring(response.content)
        except ET.ParseError as e:
            QMessageBox.warning(None, "Error", f"No se pudo parsear el XML de GetCapabilities:\n{e}")
            return []

        coverages = []
        for elem in root.iter():
            if elem.tag.endswith('CoverageSummary') or elem.tag.endswith('CoverageDescription'):
                coverages.append(elem)

        if not coverages:
            QMessageBox.warning(None, "Error", "No se encontraron coberturas en el WCS.")
            return []

        output_formats = obtener_output_formats(root)
        capas = []

        for cov in coverages:
            identifier_el = title_el = abstract_el = None

            for child in cov:
                if child.tag.endswith('CoverageId') or child.tag.endswith('Identifier'):
                    identifier_el = child
                elif child.tag.endswith('Title'):
                    title_el = child
                elif child.tag.endswith('Abstract'):
                    abstract_el = child

            identifier = identifier_el.text.strip() if identifier_el is not None and identifier_el.text else ""
            title = title_el.text.strip() if title_el is not None and title_el.text else identifier
            abstract = abstract_el.text.strip() if abstract_el is not None and abstract_el.text else ""

            for fmt in output_formats:
                capas.append({
                    "identifier": identifier,
                    "title": title,
                    "abstract": abstract,
                    "format": fmt
                })

        return capas


class SeleccionarCapasDialog_OGCAPI(QDialog):
    def __init__(self, capas, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Seleccionar capas OGCAPI")
        self.setMinimumWidth(700)
        self.setMinimumHeight(400)

        layout = QVBoxLayout()
        self.setLayout(layout)

        # Tabla de capas
        self.table = QTableWidget()
        self.table.setColumnCount(4)
        self.table.setHorizontalHeaderLabels(["Identificador", "Título",  "Descripción"])
        self.table.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.table.setSelectionMode(QAbstractItemView.MultiSelection)
        self.table.horizontalHeader().setSectionResizeMode(QHeaderView.Interactive)
        self.table.verticalHeader().setVisible(False)
        self.table.setColumnWidth(0, 200)
        self.table.setColumnWidth(1, 250)
        self.table.setColumnWidth(2, 300)

        # Llenar tabla
        self.table.setRowCount(len(capas))
        for row, capa in enumerate(capas):
            identifier = capa.get("identifier", "")
            titulo = capa.get("title", "")
            descripcion = capa.get("abstract", "")
            self.table.setItem(row, 0, QTableWidgetItem(identifier))
            self.table.setItem(row, 1, QTableWidgetItem(titulo))
            self.table.setItem(row, 2, QTableWidgetItem(descripcion))

        layout.addWidget(self.table)

        # Botones OK / Cancelar
        self.buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)
        layout.addWidget(self.buttonBox)

    def capas_seleccionadas(self):
        """Devuelve lista con los 'identifier' de las capas seleccionadas"""
        capas = []
        for item in self.table.selectionModel().selectedRows():
            row = item.row()
            identifier = self.table.item(row, 0).text()
            title = self.table.item(row, 1).text()
            format = self.table.item(row, 2).text()
            obj={
                "identifier": identifier,
                "title":title,
                "format":format
            }
            capas.append(obj)
        return capas

class CargarCapasOGCAPIWorker(QThread):
    terminado = pyqtSignal(list)
    error = pyqtSignal(str)

    def __init__(self, servicio, obj):
        super().__init__()
        self.servicio = servicio
        self.obj = obj

    def run(self):
        try:
            if self.obj["masInfo"] == "Features" or self.obj["masInfo"] == "Desconocido":
                capas = self.obtener_capas_ogcapi_features(self.servicio)
                self.terminado.emit(capas)
            else:
                self.terminado.emit({})
        except Exception as e:
            self.error.emit(str(e))
        
    def obtener_capas_ogcapi_features(self, servicio):
        """
        Obtiene las colecciones (capas) de un servicio OGC API Features en JSON.
        Devuelve una lista de diccionarios:
        [{'identifier':..., 'title':..., 'abstract':..., 'format':...}, ...]
        """

        if not servicio.lower().startswith("http"):
            return []

        # Normalizar URL: debe terminar sin barra y apuntar a la raíz del OGCAPI Features
        if servicio.lower().endswith("/collections"):
            url_collections = servicio
        else:
            url_collections = f"{servicio.rstrip('/')}/collections"


        try:
            response = requests.get(url_collections, timeout=10)
            response.raise_for_status()
            data = response.json()
        except requests.RequestException as e:
            QMessageBox.warning(None, "Error", f"No se pudo acceder al OGC API Features:\n{e}")
            return []
        except ValueError as e:
            QMessageBox.warning(None, "Error", f"No se pudo parsear la respuesta JSON:\n{e}")
            return []

        capas = []

        # Recorrer las colecciones
        for collection in data.get("collections", []):
            identifier = collection.get("id", "")
            title = collection.get("title", identifier)
            abstract = collection.get("description", "")

            capas.append({
                "identifier": identifier,
                "title": title,
                "abstract": abstract,
            })

        if not capas:
            QMessageBox.warning(None, "Error", "No se encontraron colecciones en el OGC API Features.")

        return capas
